// (c) Copyright 1994, SOFTIMAGE Inc.
//
//
// This example uses DKit v1.1 functions and structures to load,
// modify, and save a SOFTIMAGE scene file.
//
// The (C++) code loads the latest version of a scene called "scene" from the
// default database (ie, the first entry in the DatabaseDir.rsrc file in
// $SI_DBDIR).  It also assumes the path "/usr/softimage/rsrc".
//
// The following miscellaneous parameters are modified:
//    - fcurve slope management flags
//    - model X scale fcurve
//    - materials
//    - playback start frame, end frame, current frame, step, and rate.
//
// The resulting scene is then saved back to the default database as
// "new_scene".
// 
//
// VL.
//


#include <iostream.h>           // for cerr
#include <string.h>             // for strdup()
#include <assert.h>             // for assert()

extern "C"
{
#include "tinySceneToken.h"
#include "tinyScene.h"          // for scene structure
#include "libdbdk.h"            // for database access, DK_db* functions
#include "tinybyteswap.h"
}


void main(int, char *argv[])
{

#ifdef _WIN32
   DK_streamInitialize();
#endif

   // ------------
   // load a scene

   ::DK_setVerbose(1);  // get more feedback from libdbdk.a functions

   ::DK_dbInit("/usr/softimage/rsrc", 1);  // 1: do use DatabaseDir.rsrc

   DK_Database db = ::DK_dbGet("");  // get default database

   if (db == 0)
   {
      cerr << "DK_dbGet() returned 0 - exit." << endl;
      ::exit(1);
   }

   // get scene name without the .dsc extension
   // this returns the scene name with the latest version number,
   // (and with substitution number 0).
   char* scene_name = ::DK_dbGetLatestSceneVersion("scene", db);

   if (scene_name == 0)
   {
      cerr << "DK_dbGetLatestSceneVersion() returned 0 - exit." << endl;
      ::exit(1);
   }

   DK_Scene* sc = ::DK_dbSceneLoad(scene_name, db);

   if (sc == 0)
   {
      cerr << "DK_dbSceneLoad() returned 0 - exit." << endl;
      ::exit(1);
   }

   sc->name = ::strdup("new_scene");  // output to "new_scene"




   // ------------------------------------------------------------------
   // print out the values of any function curves attached to the camera,
   // at each frame, starting from the first key, and ending at the last key.

   int n = DK_FCV_CAM_NUM; 

   DK_Fcurve* fc = 0;
   float t, e, dt = 1.0/sc->rate;

   while (n--)
   {
      if ((fc = sc->camera.fcv[n]) == 0 || fc->size < 1)
      {
	 continue;
      }

      cerr << "camera fcurve #" << n << ":" << endl;

      t = fc->keys[0].time;
      e = fc->keys[fc->size - 1].time;

      int f = int(t * sc->rate);

      while (t <= e)
      {
	 cerr << "   frame " << f++ << ": " << ::DK_fcurveValue(fc, t) << endl;
	 t += dt;
      }
   }



   // ----------------------------------------------------------------------
   // modify the camera's Y explicit translation fcurve so that the function
   // curve has automatic (system), double (left & right are broken) slopes,
   // and the keys have user, single (left & right slopes are unified) slopes.

   if ((fc = sc->camera.fcv[DK_FCV_CAM_EPOSY]) != 0 && 0 < fc->size)
   {
      fc->slopeFlag = 1;

      // (first bit value of 0 = system/automatic, 1 = user slope,
      //  second bit value of 0 = single/unified, 1 = double/broken slope).

      n = fc->size;

      while (n--)
      {
	 fc->keys[n].slopeFlag = 2;
      }
   }



   // -----------------------------------------------------------------
   // modify the X scale function curve of the first model, so that its
   // scale in X grows from 1 at time = 1, to 2 at time = 2 seconds.

   if (sc->model != 0)
   {
      DK_Fcurve* fc = ::DK_fcurveAllocate();
      assert(fc);

      fc->type = DK_FCV_TYPE_MDL_SCL_X;
      fc->size = 2;
      fc->interp = DK_LINEAR_INTERP;
      fc->keys = ::DK_fcvkeyAllocate(2);

      fc->keys[0].time = 1;
      fc->keys[0].value = 1;
      fc->keys[0].keyType = DK_KEYFRAME;
      fc->keys[0].interp = DK_LINEAR_INTERP;

      fc->keys[1].time = 2;
      fc->keys[1].value = 2;
      fc->keys[1].keyType = DK_KEYFRAME;
      fc->keys[1].interp = DK_LINEAR_INTERP;

      if (sc->model->fcv[DK_FCV_MDL_SCLX] != 0)
      {
	 ::DK_fcurveDispose(&(sc->model->fcv[DK_FCV_MDL_SCLX]));
      }

      sc->model->fcv[DK_FCV_MDL_SCLX] = fc;
   }


   // ---------------------------------------------------
   // modify the material of the first model in the scene

   if (sc->model != 0)
   {
      if (sc->model->materials)
      {
	 // dispose all materials associated with this model
	 ::DK_materialDispose(&(sc->model->materials));
      }

      sc->model->materials = ::DK_materialAllocate();
      assert(sc->model->materials);

      sc->model->nbMaterials = 1;
      sc->model->materials->name = ::strdup("my_material");
      sc->model->materials->ambient.r = 1.;
      sc->model->materials->ambient.g = 0.;
      sc->model->materials->ambient.b = 0.;
      sc->model->materials->diffuse.r = 0.;
      sc->model->materials->diffuse.g = .5;
      sc->model->materials->diffuse.b = .5;
   }


   // -------------------------------------------------------------------
   // modify the scene playback rate, step, and start/end/current frames.

   sc->rate /= 2;
   sc->step = 3;
   sc->start = 9;
   sc->end = 123;
   sc->current = 99;


   // -----------------------------------------------------------------------
   // save the scene, FALSE == do not overwrite - use next available version.

   ::DK_dbSceneSave(sc, db, FALSE);

   ::DK_sceneDispose(&sc);
}
