
/*******************************************************************************
********************************************************************************

   sceneDumpMatrix.c

   Written by: Dominique Boisvert, November '91.

   - Dumps the global transformation matrix (scaling, rotation, translation)
     of each model (and sub-models) of the given scene in standard output.

     Usage: sceneDump <ascii scene file name>

   (c) Copyright 1991 Softimage Inc.

********************************************************************************
*******************************************************************************/

#include "tinyModel.h"
#include "tinySceneToken.h"
#include "tinyScene.h"
#include "DK_filter.h"

/*******************************************************************************

   vctXmat

   - Multiply the vector `v' with the matrix `m'.  The result is in vector `r'.

   Returned Value: None.

*******************************************************************************/

static void vctXmat
   (
      Vector *r,	/* resulting vector (out) */
      Vector v,		/* input vector (in) */
      Matrix m		/* input matrix (in) */
   )
{
   r->x = v.x * m[0][0] + v.y * m[1][0] + v.z * m[2][0] + m[3][0];
   r->y = v.x * m[0][1] + v.y * m[1][1] + v.z * m[2][1] + m[3][1];
   r->z = v.x * m[0][2] + v.y * m[1][2] + v.z * m[2][2] + m[3][2];
}


/*******************************************************************************

   matrixInit

   - Initialize the matrix `mat'.

   Returned Value: None.

*******************************************************************************/

static void matrixInit
   (
      Matrix mat	/* matrix to initialize (in-out) */
   )
{
   register int i,j;

   for ( i = 0; i < 4; i++ )
      for ( j = 0; j < 4; j++ )
	 mat[i][j] = ( i == j ? 1.0 : 0.0 );
}


/*******************************************************************************

   matrixMult

   - Multiply two matrices `mat1' and `mat2'.  The result is in `mat3'.

   Returned Value: None.

*******************************************************************************/

static void matrixMult
   (
      Matrix mat1,	/* first matrix (in) */
      Matrix mat2,	/* second matrix (in) */
      Matrix mat3	/* mat1 x mat2 (out) */
   )
{
   float  *c, *a, *b;

   a = &mat1[0][0];
   b = &mat2[0][0];
   c = &mat3[0][0];

   *c = a[0]*b[0] + a[1]*b[4] + a[2]*b[8] + a[3]*b[12];   c++;
   *c = a[0]*b[1] + a[1]*b[5] + a[2]*b[9] + a[3]*b[13];   c++;
   *c = a[0]*b[2] + a[1]*b[6] + a[2]*b[10] + a[3]*b[14];   c++;
   *c = a[0]*b[3] + a[1]*b[7] + a[2]*b[11] + a[3]*b[15];   c++;

   *c = a[4]*b[0] + a[5]*b[4] + a[6]*b[8] + a[7]*b[12];   c++;
   *c = a[4]*b[1] + a[5]*b[5] + a[6]*b[9] + a[7]*b[13];   c++;
   *c = a[4]*b[2] + a[5]*b[6] + a[6]*b[10] + a[7]*b[14];   c++;
   *c = a[4]*b[3] + a[5]*b[7] + a[6]*b[11] + a[7]*b[15];   c++;

   *c = a[8]*b[0] + a[9]*b[4] + a[10]*b[8] + a[11]*b[12];   c++;
   *c = a[8]*b[1] + a[9]*b[5] + a[10]*b[9] + a[11]*b[13];   c++;
   *c = a[8]*b[2] + a[9]*b[6] + a[10]*b[10] + a[11]*b[14];   c++;
   *c = a[8]*b[3] + a[9]*b[7] + a[10]*b[11] + a[11]*b[15];   c++;

   *c = a[12]*b[0] + a[13]*b[4] + a[14]*b[8] + a[15]*b[12];   c++;
   *c = a[12]*b[1] + a[13]*b[5] + a[14]*b[9] + a[15]*b[13];   c++;
   *c = a[12]*b[2] + a[13]*b[6] + a[14]*b[10] + a[15]*b[14];   c++;
   *c = a[12]*b[3] + a[13]*b[7] + a[14]*b[11] + a[15]*b[15];
}


/*******************************************************************************

   matrixRotation

   - Builds the rotation matrix `mat' given the `vrot' rotation vector.

   Returned Value: None.

*******************************************************************************/

static void matrixRotation
   (
      Matrix mat,	/* resulting rotation matrix (out) */
      Vector *vrot	/* rotation vector (in) */
   )
{
   float sx, cx, sy, cy, sz, cz, tmp;

   sx = sin( vrot->x );   cx = cos( vrot->x );
   sy = sin( vrot->y );   cy = cos( vrot->y );
   sz = sin( vrot->z );   cz = cos( vrot->z );

   mat[0][0] = cy * cz;
   mat[0][1] = cy * sz;
   mat[0][2] = -sy;

   tmp = sx * sy;
   mat[1][0] = tmp * cz - cx * sz;
   mat[1][1] = tmp * sz + cx * cz;
   mat[1][2] = sx * cy;

   tmp = cx * sy;
   mat[2][0] = tmp * cz + sx * sz;
   mat[2][1] = tmp * sz - sx * cz;
   mat[2][2] = cx * cy;

   mat[0][3] = mat[1][3] = mat[2][3] = 0.0;
   mat[3][0] = mat[3][1] = mat[3][2] = 0.0;
   mat[3][3] = 1.0;
}


/*******************************************************************************

   modelDump

   - Dumps the global matrix of the `mdl' model.  This function is recursive.

   Returned Value: None.

*******************************************************************************/

static void modelDump
   (
      FILE *file,	/* output file (in) */
      Model *mdl,	/* model to process (in) */
      Matrix hmat,	/* herited global matrix (in) */
      Matrix hrot,	/* herited global rotation matrix (in) */
      Vector hscl	/* herited global scaling vector (in) */
   )
{
   Matrix mrot, grot, gmat;
   Vector gscl, gtrs;
   register int i;

   if ( mdl == NULL )
      return;

   /*-------------------------------------------------------------------------
      Here we compute the model's global rotation matrix `grot'.
      If there's a rotation constraint on the effector, we don't use the
      herited global rotation.
    ------------------------------------------------------------------------*/
   if ( mdl->algotype == AIKLEAF && mdl->parent->ikvar->type == 1 )
      matrixRotation( grot, &mdl->rotation );
   else
   {
      matrixRotation( mrot, &mdl->rotation );
      matrixMult( mrot, hrot, grot );
   }

   /*-------------------------------------------------------------------------
      Here we compute the model's global translation & scaling vectors:
      `gtrs' and `gscl'.
    ------------------------------------------------------------------------*/
   if ( mdl->parent == NULL )
   {
      gtrs = mdl->translation;
      gscl = mdl->scaling;
   }
   else
   {
      vctXmat( &gtrs, mdl->translation, hmat );
      gscl.x = mdl->scaling.x * hscl.x;
      gscl.y = mdl->scaling.y * hscl.y;
      gscl.z = mdl->scaling.z * hscl.z;
   }

   /*-------------------------------------------------------------------------
      Here we compute the model's global transformation matrix `gmat'.
    ------------------------------------------------------------------------*/
   gmat[0][0] = grot[0][0] * gscl.x;
   gmat[0][1] = grot[0][1] * gscl.x;
   gmat[0][2] = grot[0][2] * gscl.x;
   gmat[0][3] = 0.0;

   gmat[1][0] = grot[1][0] * gscl.y;
   gmat[1][1] = grot[1][1] * gscl.y;
   gmat[1][2] = grot[1][2] * gscl.y;
   gmat[1][3] = 0.0;

   gmat[2][0] = grot[2][0] * gscl.z;
   gmat[2][1] = grot[2][1] * gscl.z;
   gmat[2][2] = grot[2][2] * gscl.z;
   gmat[2][3] = 0.0;

   gmat[3][0] = gtrs.x;
   gmat[3][1] = gtrs.y;
   gmat[3][2] = gtrs.z;
   gmat[3][3] = 1.0;

   /*-------------------------------------------------------------------------
      Print the `gmat' matrix.
    ------------------------------------------------------------------------*/
   fprintf(file, "Model: %s\n", mdl->name );
   for ( i = 0; i < 4; i++ )
      fprintf(file, "   %10.5f %10.5f %10.5f %10.5f\n",
	 gmat[i][0], gmat[i][1], gmat[i][2], gmat[i][3]);
   fprintf(file, "\n");

   /*-------------------------------------------------------------------------
      Call `modelDump' function recursively in depth-first order (son then
      brother).
    ------------------------------------------------------------------------*/
   modelDump( file, mdl->child, gmat, grot, gscl );
   modelDump( file, mdl->sibling, hmat, hrot, hscl );
}


/*******************************************************************************

   sceneDump

   - Dumps the global matrix of each model in the given scene.

   Returned Value: None.

*******************************************************************************/

static void sceneDump
   (
      FILE *file,	/* output file (in) */
      Scene *scn	/* scene's pointer (in) */
   )
{
   Model *mdl;
   Matrix gmat, grot;
   Vector gscl;

   /*-------------------------------------------------------------------------
      Initialize variables used by `modelDump' function.
    ------------------------------------------------------------------------*/
   matrixInit( gmat );
   matrixInit( grot );
   gscl.x = gscl.y = gscl.z = 1.0;

   /*-------------------------------------------------------------------------
      Call `modelDump' for each hierarchy in the scene.
    ------------------------------------------------------------------------*/
   for ( mdl = scn->model; mdl != NULL; mdl = mdl->next )
      modelDump( file, mdl, gmat, grot, gscl );
}


main(int argc, char *argv[] )
{
   Scene *scn;
   float version;

#ifdef _WIN32
   DK_streamInitialize();
#endif

   if (argc != 2)
   {
      fprintf(stderr, "Usage: sceneDump <ascii scene file>\n");
      exit(0);
   }

   scn = sceneAllocate();
   scn->name = argv[1];
   if (sceneReadAsciiAnimation (&version, scn) == TRUE)
      sceneDump( stdout, scn );
   sceneDispose(&scn);
}

