/**************************************************************/
/* This program merges the meshes found at the same hierarchy */
/* level. The following assumptions are made :                */
/*                                                            */
/* - Hierarchy files (.hrc) generated by Mediagramme's        */
/*   translators are used.                                    */
/* - There is no scaling and rotation applied on the models   */
/*   containing the meshes.                                   */
/* - A mesh model node has a child pointer equal to NULL.     */
/**************************************************************/

#include <math.h>
#ifdef _WIN32
#include <irixMath.h>
#endif
#include "tinyModel.h"
#include "tinyMesh.h"

#define EPS 0.0001

typedef struct new_edge {
   long vrt[ 2 ];
   long pol[ 2 ][ 10 ]; 

   int  nb_pol[ 2 ];
} NewEdge;

static void add_polygon(NewEdge *, int, int);


/**************************************************************/
/* Apply the model's transformations to each point of a mesh. */
/**************************************************************/

void apply_transformation( 
   DK_Model *mdl
)
{
   int        i;
   DK_Mesh   *msh = (DK_Mesh *) mdl->definition;
   DK_Vertex *v   = msh->vertices;

   for ( i=0; i<msh->nbVertices; i++, v++ ) {
      v->position.x += mdl->translation.x;
      v->position.y += mdl->translation.y;
      v->position.z += mdl->translation.z;
   }

   mdl->translation.x = 0.0;
   mdl->translation.y = 0.0;
   mdl->translation.z = 0.0;
}

/**************************************************************/
/* Locates a particular vertex in the list of vertices of a   */
/* mesh. When found, the vertex's position is returned.       */
/* Otherwise, a -1 value is returned.                         */
/**************************************************************/

long find_vertex( 
   DK_Mesh   *msh, 
   DK_Vertex  vrt
)
{
   long       i  = msh->nbVertices;
   DK_Vertex *pv = &msh->vertices[ i-1 ];

   for ( ; i>0; i--, pv-- ) {
      if ( (fabsf( pv->position.x - vrt.position.x ) < EPS) && 
	   (fabsf( pv->position.y - vrt.position.y ) < EPS) && 
	   (fabsf( pv->position.z - vrt.position.z ) < EPS) ) return i-1;
   }

   return -1;
}

/**************************************************************/
/* Determine if two polygons are the same regardless of their */
/* orientations.                                              */
/**************************************************************/

int are_the_same( 
   DK_Mesh *msh, 
   long     p1, 
   long     p2 
)
{
   DK_Polygon *pol1 = &msh->polygons[ p1 ],
	      *pol2 = &msh->polygons[ p2 ];
   int         i,
	       j;

   if ( pol1->nbNodes != pol2->nbNodes ) return 0;

   for ( i=0; i<pol1->nbNodes; i++ ) {
      int v1    = pol1->nodes[ i ].vertex,
	  found = 0;

      for ( j=0; j<pol2->nbNodes; j++ ) {
	 int v2 = pol2->nodes[ j ].vertex;

	 if ( v1 == v2 ) {
	    found = 1;
	    break;
	 }
      }

      if ( ! found ) return 0;
   }

   /* Create a more complex test. */

   return 1; 
}

int find_edge( 
   NewEdge *edg, 
   int      nb_edg, 
   long     v0, 
   long     v1,
   int     *dir
) 
{
   NewEdge *n = edg;
   int      i;

   for ( i=0; i<nb_edg; i++, n++ ) {
      if ( n->vrt[ 0 ] == v0 && n->vrt[ 1 ] == v1 ) {
	 *dir = 0;
	 return i+1;

      } else if ( n->vrt[ 0 ] == v1 && n->vrt[ 1 ] == v0 ) {
	 *dir = 1;
	 return i+1;
      }
   }

   return 0;
}

NewEdge *build_edge_table(
   DK_Mesh *msh,
   int     *nb_edg 
)
{
   int      i,
            max_edg = 0;
   NewEdge *edg;

   *nb_edg = 0;

   for ( i=0; i<msh->nbPolygons; i++ ) {
      max_edg += msh->polygons[ i ].nbNodes;
   }

   edg = (NewEdge *) malloc( max_edg * sizeof( NewEdge ) );

   for ( i=0; i<msh->nbPolygons; i++ ) {
      DK_Polygon *p  = &msh->polygons[ i ];
      int         nb = p->nbNodes,
	          j;

      for ( j=0; j<p->nbNodes; j++ ) {
	 int v0 = p->nodes[ j ].vertex,
	     v1 = p->nodes[ (j+1) % nb ].vertex,
	     pos,
	     dir;

         if ( pos = find_edge( edg, *nb_edg, v0, v1, &dir ) ) {
	    add_polygon( &edg[ pos-1 ], dir, i );

	 } else {
	    NewEdge *n = &edg[ *nb_edg ];

	    n->vrt[ 0 ]      = v0;
	    n->vrt[ 1 ]      = v1;
	    n->nb_pol[ 0 ]   = 1;
	    n->pol[ 0 ][ 0 ] = i;
	    n->nb_pol[ 1 ]   = 0;

	    (*nb_edg)++;
	 }
      }
   }

   return edg;
}

/**************************************************************/
/* Merges together two meshes. The resulting mesh replaces    */
/* the mesh found as the first parameter of this function.    */
/**************************************************************/

void merge( 
   DK_Model *mdl_mesh,
   DK_Model *mesh_to_add
)
{
   DK_Mesh *msh1    = (DK_Mesh *) mdl_mesh->definition;
   DK_Mesh *msh2    = (DK_Mesh *) mesh_to_add->definition;
   DK_Mesh *new_msh = DK_meshAllocate();

   DK_Vertex *nv;
   DK_Vertex *ov;

   DK_Polygon *np;
   DK_Polygon *op;

   int i;
   int same = 0;
   int nb   = msh1->nbVertices;

   /* Allocate spaces for both structures. */

   new_msh->nbVertices = msh1->nbVertices + msh2->nbVertices;
   new_msh->vertices   = DK_vertexAllocate( new_msh->nbVertices );

   new_msh->nbPolygons = msh1->nbPolygons + msh2->nbPolygons;
   new_msh->polygons   = DK_polygonAllocate( new_msh->nbPolygons );

   /* Copy the vertices. */

   new_msh->nbVertices = nb;

   nv = new_msh->vertices;
   ov = msh1->vertices;

   for ( i=0; i<msh1->nbVertices; i++, nv++, ov++ ) {
      *nv = *ov;
   }

   /* Copy the polygons. */

   np = new_msh->polygons;
   op = msh1->polygons;

   for ( i=0; i<msh1->nbPolygons; i++, np++, op++ ) {
      *np = *op;

      op->nbNodes = 0;
      op->nodes   = NULL;
   }

   op = msh2->polygons;

   for ( i=0; i<msh2->nbPolygons; i++, np++, op++ ) {
      int j;

      *np = *op;

      op->nbNodes = 0;
      op->nodes   = NULL;

      for ( j=0; j<np->nbNodes; j++ ) {
	 long v = np->nodes[ j ].vertex;

	 if ( v != -1 ) {
	    long pos = find_vertex( new_msh, msh2->vertices[ v ] );

	    if ( pos == -1 ) {
	       pos = new_msh->nbVertices;

	       *nv = msh2->vertices[ v ]; 

	       nv++;
	       new_msh->nbVertices++;

	    } else {
	       same++;
	    }

	    np->nodes[ j ].vertex = pos;
	 }
      }
   }

   nb = new_msh->nbVertices;

   DK_meshDispose( &msh1 );
   DK_meshDispose( &msh2 );

   mesh_to_add->definition = NULL;

   if ( same ) {
      DK_Mesh *msh = DK_meshAllocate();
      int      s,
	       t,
	       sss = 0;

      for ( t=0; t<new_msh->nbPolygons-1; t++ ) {
	 for ( s=t+1; s<new_msh->nbPolygons; s++ ) {
	    if ( are_the_same( new_msh, s, t ) ) {
	       DK_Polygon *p = &new_msh->polygons[ s ];

	       free( p->nodes );

	       p->nodes   = NULL;
	       p->nbNodes = 0;

	       p = &new_msh->polygons[ t ];

	       free( p->nodes );

	       p->nodes   = NULL;
	       p->nbNodes = 0;

	       sss += 2;
	    }
	 }
      }

      /* Allocate spaces for both structures.                 */
      /*                                                      */
      /* Note : No all the allocated space in the new mesh is */
      /*        used.                                         */

      msh->nbVertices = nb; 
      msh->vertices   = DK_vertexAllocate( msh->nbVertices );

      msh->nbPolygons = new_msh->nbPolygons - sss;
      msh->polygons   = DK_polygonAllocate( msh->nbPolygons );

      /* Copy the vertices. */

      nv = msh->vertices;
      ov = new_msh->vertices;

      for ( i=0; i<msh->nbVertices; i++, nv++, ov++ ) {
	 *nv = *ov;
      }

      /* Copy the polygons. */

      np = msh->polygons;
      op = new_msh->polygons;

      for ( i=0; i<new_msh->nbPolygons; i++, op++ ) {
	 if ( op->nbNodes ) { 
	    *np = *op;

	    np++;

	    op->nbNodes = 0;
	    op->nodes   = NULL;
	 }
      }

      DK_meshDispose( &new_msh );

      new_msh = msh;
   }

   mdl_mesh->definition = new_msh;
}

DK_Mesh *clean_mesh(
   DK_Mesh *msh,
   int      id
)
{
   int         i,
	       unused_vrt = 0,
	       unused_pol = 0,
	       last_map,
	      *map;
   DK_Vertex  *v       = msh->vertices;
   DK_Polygon *p       = msh->polygons;
   DK_Mesh    *new_msh = DK_meshAllocate();
   DK_Vertex  *nv;
   DK_Vertex  *ov;
   DK_Polygon *np;
   DK_Polygon *op;

   for ( i=0; i<msh->nbVertices; i++, v++ ) {
      v->flag = 0;
   }

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      if ( p->nbNodes && p->flag == id ) {
	 int j;

	 for ( j=0; j<p->nbNodes; j++ ) {
	    if ( p->nodes[ j ].vertex != -1 ) {
	       msh->vertices[ p->nodes[ j ].vertex ].flag = 1;
	    }
	 }
      } else {
	 unused_pol++;
      }
   }

   for ( v=msh->vertices, i=0; i<msh->nbVertices; i++, v++ ) {
      if ( ! v->flag ) unused_vrt++;
   }

   last_map = 0;
   map      = (int *) malloc( msh->nbVertices * sizeof( int ) );

   /* Allocate spaces for both structures.                 */

   new_msh->nbVertices = msh->nbVertices - unused_vrt; 
   new_msh->vertices   = DK_vertexAllocate( new_msh->nbVertices );

   new_msh->nbPolygons = msh->nbPolygons - unused_pol;
   new_msh->polygons   = DK_polygonAllocate( new_msh->nbPolygons );

   /* Copy the vertices. */

   nv = new_msh->vertices;
   ov = msh->vertices;

   for ( i=0; i<msh->nbVertices; i++, ov++ ) {
      if ( ov->flag ) {
	 *nv = *ov;

	 map[ i ] = last_map;
	 last_map++;
	 nv++;
      }
   }

   /* Copy the polygons. */

   np = new_msh->polygons;
   op = msh->polygons;

   for ( i=0; i<msh->nbPolygons; i++, op++ ) {
      if ( op->nbNodes && op->flag == id ) { 
	 int j;

	 *np = *op;

	 for ( j=0; j<np->nbNodes; j++ ) {
	    if ( np->nodes[ j ].vertex != -1 ) {
	       np->nodes[ j ].vertex = map[ np->nodes[ j ].vertex ];
	    }
	 }

	 np->flag = 0;

	 np++;

	 op->nbNodes = 0;
	 op->nodes   = NULL;
      }
   }

   free( map );

   return new_msh;
}

DK_Mesh *copy_mesh(
   DK_Mesh *msh
)
{
   int         i,
	       unused_vrt = 0,
	       unused_pol = 0,
	       last_map,
	      *map;
   DK_Vertex  *v = msh->vertices;
   DK_Polygon *p = msh->polygons;
   DK_Mesh    *new_msh = DK_meshAllocate();
   DK_Vertex  *nv;
   DK_Vertex  *ov;
   DK_Polygon *np;
   DK_Polygon *op;

   for ( i=0; i<msh->nbVertices; i++, v++ ) {
      v->flag = 0;
   }

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      if ( p->nbNodes ) {
	 int j;

	 for ( j=0; j<p->nbNodes; j++ ) {
	    if ( p->nodes[ j ].vertex != -1 ) {
	       msh->vertices[ p->nodes[ j ].vertex ].flag = 1;
	    }
	 }
      } else {
	 unused_pol++;
      }
   }

   for ( v=msh->vertices, i=0; i<msh->nbVertices; i++, v++ ) {
      if ( ! v->flag ) unused_vrt++;
   }

   last_map = 0;
   map      = (int *) malloc( msh->nbVertices * sizeof( int ) );

   /* Allocate spaces for both structures.                 */

   new_msh->nbVertices = msh->nbVertices - unused_vrt; 
   new_msh->vertices   = DK_vertexAllocate( new_msh->nbVertices );

   new_msh->nbPolygons = msh->nbPolygons - unused_pol;
   new_msh->polygons   = DK_polygonAllocate( new_msh->nbPolygons );

   /* Copy the vertices. */

   nv = new_msh->vertices;
   ov = msh->vertices;

   for ( i=0; i<msh->nbVertices; i++, ov++ ) {
      if ( ov->flag ) {
	 *nv = *ov;

	 map[ i ] = last_map;
	 last_map++;
	 nv++;
      }
   }

   /* Copy the polygons. */

   np = new_msh->polygons;
   op = msh->polygons;

   for ( i=0; i<msh->nbPolygons; i++, op++ ) {
      if ( op->nbNodes ) { 
	 int j;


	 np->nbNodes = op->nbNodes;
	 np->nodes   = (DK_PolyNode *) malloc(
	    np->nbNodes * sizeof( DK_PolyNode )
	 );

	 for ( j=0; j<op->nbNodes; j++ ) {
	    if ( op->nodes[ j ].vertex != -1 ) {
	       np->nodes[ j ].vertex = map[ op->nodes[ j ].vertex ];
	    }
	 }

	 np->flag = 0;

	 np++;
      }
   }

   free( map );

   return new_msh;
}

void dump_edge(
   NewEdge *edg
)
{
   int j;

   printf( "v0 = %d - v1 = %d\n", edg->vrt[0], edg->vrt[1] );

   for ( j=0; j<2; j++ ) {
      int k;
      
      printf( "Direction %d\n", j );
      printf( "nb polygons = %d\n", edg->nb_pol[j] );

      for ( k=0; k<edg->nb_pol[j]; k++ ) {
	 printf( "%d ", edg->pol[j][k] );
      }

      printf( "\n" );
   }
   
   printf( "\n" );
}

void dump_edge_table(
   NewEdge *edg,
   int      nbe
)
{
   int i;

   for ( i=0; i<nbe; i++ ) {
      dump_edge( &edg[ i ] );
   }
}

static void add_polygon( 
   NewEdge *ne, 
   int      dir, 
   int      id
)
{
   ne->pol[ dir ][ ne->nb_pol[ dir ] ] = id;
   ne->nb_pol[ dir ]++;
}

void remove_polygon( 
   NewEdge *ne, 
   int      dir, 
   int      id
)
{
   int i;

   for ( i=0; i<ne->nb_pol[ dir ]; i++ ) {
      if ( ne->pol[ dir ][ i ] == id ) {
	 int j;

	 for ( j=i; j<ne->nb_pol[ dir ]-1; j++ ) {
	    ne->pol[ dir ][ j ] = ne->pol[ dir ][ j+1 ];
	 }

	 ne->nb_pol[ dir ]--;

	 return;
      }
   }
}

void inverse_polygon( 
   int      id,
   DK_Mesh *msh,
   NewEdge *edg, 
   int      nbe 
)
{
   DK_Polygon *p = &msh->polygons[ id ];
   int         i,
	       nb = p->nbNodes;

   for ( i=0; i<nb; i++ ) {
      int  v0 = p->nodes[ i ].vertex,
	   v1 = p->nodes[ (i+1) % nb ].vertex,
	   pos,
	   dir;

      if ( pos = find_edge( edg, nbe, v0, v1, &dir ) ) {
	 NewEdge *ne = &edg[ pos-1 ];

	 remove_polygon( ne, dir, id );
	 add_polygon( ne, 1 - dir, id );
      }
   }

   for ( i=0; i<nb/2; i++ ) {
      int tmp = p->nodes[ i ].vertex;

      p->nodes[ i ].vertex          = p->nodes[ nb - i - 1 ].vertex;
      p->nodes[ nb - i - 1 ].vertex = tmp;
   }
}

void erase_polygon( 
   int      id,
   DK_Mesh *msh,
   NewEdge *edg, 
   int      nbe
)
{
   DK_Polygon *p = &msh->polygons[ id ];
   int         i,
	       nb = p->nbNodes;

   for ( i=0; i<nb; i++ ) {
      int  v0 = p->nodes[ i ].vertex,
	   v1 = p->nodes[ (i+1) % nb ].vertex,
	   pos,
	   dir;

      if ( pos = find_edge( edg, nbe, v0, v1, &dir ) ) {
	 remove_polygon( &edg[ pos-1 ], dir, id );
      }
   }
}

void dump_polygon(
   DK_Polygon *pol
)
{
   int i;

   printf( "Polygon nodes : " );

   for ( i=0; i<pol->nbNodes; i++ ) {
      printf( "%d ", pol->nodes[ i ].vertex );
   }

   printf( "\n" );
}

void process_polygon(
   int      id,
   DK_Mesh *msh,
   NewEdge *edg,
   int      nbe,
   DK_Mesh *reject
)
{
   DK_Polygon *pol = &msh->polygons[ id ];
   int         i,
               nb  = pol->nbNodes;

   if ( pol->flag & DK_POL_SELECTED ) return;

   pol->flag = DK_POL_SELECTED;

   for ( i=0; i<nb; i++ ) {
      int v0 = pol->nodes[ i ].vertex,
	  v1 = pol->nodes[ (i+1) % nb ].vertex,
	  pos,
	  dir,
	  nb_pol[ 2 ];

      if ( pos = find_edge( edg, nbe, v0, v1, &dir ) ) {
	 NewEdge *cur; 
	 int      cur_pol = -1;
	 int      j,
		  max;

         cur = &edg[ pos-1 ];

	 /* remove current polygon id from the list of this edge. */

	 remove_polygon( cur, dir, id );

	 /* Look in the opposite direction. */

	 if ( cur->nb_pol[ 1 - dir ] ) {
	    /* find one that was processed. */

	    max = cur->nb_pol[ 1 - dir ];

	    for ( j=0; j<max; j++ ) {
	       int cpol = cur->pol[ 1 - dir ][ j ];

	       if ( msh->polygons[ cpol ].flag & DK_POL_SELECTED ) {
		  cur_pol = cpol;
		  break;
	       }
	    }

	    if ( cur_pol != -1 ) {
	       remove_polygon( cur, 1 - dir, cur_pol );
	    
	    } else {
	       cur_pol = cur->pol[ 1-dir ][ max-1 ];
	       cur->nb_pol[ 1 - dir ]--;
	    }
	 
	 /* Check the same orientation which means a flip. */

	 } else if ( cur->nb_pol[ dir ] ) {
	    int ind;

	    /* find one that was processed. */

	    max = cur->nb_pol[ dir ];

	    for ( j=0; j<max; j++ ) {
	       int cpol = cur->pol[ dir ][ j ];

	       if ( msh->polygons[ cpol ].flag & DK_POL_SELECTED ) {
		  printf( "bad ...\n" );

		  erase_polygon( id, msh, edg, nbe );

		  msh->polygons[ id ].nbNodes = 0;

		  if ( msh->polygons[ id ].nodes ) {
		     free( msh->polygons[ id ].nodes );
		     msh->polygons[ id ].nodes = NULL;
		  }

		  return;

	       } else {
		  cur_pol = cpol;
		  ind     = j;
	       }
	    }

	    if ( cur_pol != -1 ) {
	       inverse_polygon( cur_pol, msh, edg, nbe );

	       remove_polygon( cur, 1 - dir, cur_pol );
	    }
	 }

	 nb_pol[ 0 ] = cur->nb_pol[ 0 ];
	 nb_pol[ 1 ] = cur->nb_pol[ 1 ];

         for ( j=0; j<2; j++ ) {
	    int k;
	    
	    for ( k=0; k<nb_pol[ j ]; k++ ) {
	       int p = cur->pol[ j ][ k ];

	       if ( msh->polygons[ p ].flag & DK_POL_SELECTED ) {
		  if ( cur_pol != -1 ) {
		     add_polygon( cur, 1 - dir, cur_pol );
		  }

		  erase_polygon( id, msh, edg, nbe );

		  msh->polygons[ id ].nbNodes = 0;

		  if ( msh->polygons[ id ].nodes ) {
		     free( msh->polygons[ id ].nodes );
		     msh->polygons[ id ].nodes = NULL;
		  }

		  return;
	       }
	    }
	 }

	 /* The information remaining in the edge table is for */
	 /* polygons in extra. */

	 cur->nb_pol[ 0 ] = 0;
	 cur->nb_pol[ 1 ] = 0;

         for ( j=0; j<2; j++ ) {
	    int k;
	    
	    for ( k=0; k<nb_pol[ j ]; k++ ) {
	       int p = cur->pol[ j ][ k ];

	       erase_polygon( p, msh, edg, nbe );

	       msh->polygons[ p ].nbNodes = 0;

	       if ( msh->polygons[ p ].nodes ) {
		  free( msh->polygons[ p ].nodes );
		  msh->polygons[ p ].nodes = NULL;
	       }
	    }
	 }

	 /* Reset the edge information. */

	 cur->nb_pol[ dir ] = 1;
	 cur->pol[ dir ][ 0 ] = id;

	 reject->polygons[ id ].nbNodes = 0;
	 reject->polygons[ id ].nodes   = NULL;

	 if ( cur_pol != -1 ) {
	    cur->nb_pol[ 1 - dir ]   = 1;
	    cur->pol[ 1 - dir ][ 0 ] = cur_pol;

	    reject->polygons[ cur_pol ].nbNodes = 0;
	    reject->polygons[ cur_pol ].nodes   = NULL;

	    process_polygon( cur_pol, msh, edg, nbe, reject );

	 } else {
	    cur->nb_pol[ 1 - dir ] = 0;
	 }
      }
   }
}

void reset_polygons_flag(
   DK_Mesh *msh
)
{
   int         i;
   DK_Polygon *p = msh->polygons;

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      p->flag = 0;
   }
}

void reset_vertices_flag(
   DK_Mesh *msh
)
{
   int     i;
   DK_Vertex *v = msh->vertices;

   for ( i=0; i<msh->nbVertices; i++, v++ ) {
      v->flag = 0;
   }
}

void order_mesh(
   DK_Mesh *msh,
   DK_Mesh *reject,
   NewEdge *edg,
   int      nbe
)
{
   int      i;
   DK_Polygon *p = msh->polygons;

   reset_polygons_flag( msh );

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      p->flag = 0;
   }

   p = msh->polygons;

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      if ( ! (p->flag & DK_POL_SELECTED) && p->nbNodes ) {
	 process_polygon( i, msh, edg, nbe, reject );
      }
   }
}


void mark_polygon(
   int      mark, 
   int      id,
   DK_Mesh *msh,
   NewEdge *edg,
   int      nbe
)
{
   DK_Polygon *pol = &msh->polygons[ id ];
   int         i,
               nb  = pol->nbNodes;

   if ( pol->flag ) return;

   pol->flag = mark;

   for ( i=0; i<nb; i++ ) {
      int v0 = pol->nodes[ i ].vertex,
	  v1 = pol->nodes[ (i+1) % nb ].vertex,
	  pos,
	  dir;

      if ( pos = find_edge( edg, nbe, v0, v1, &dir ) ) {
         NewEdge *ne = &edg[ pos - 1 ];

	 if ( ne->nb_pol[ 1 - dir ] ) {
	    mark_polygon( mark, ne->pol[1-dir][0], msh, edg, nbe );
	 }
      }
   }
}

int number_of_shells( 
   DK_Mesh *msh, 
   NewEdge *edg, 
   int      nbe 
)
{
   int         i;
   int         shell = 0;
   DK_Polygon *p = msh->polygons;

   reset_polygons_flag( msh );

   for ( i=0; i<msh->nbPolygons; i++, p++ ) {
      if ( ! p->flag && p->nbNodes ) { 
	 shell++;
	 mark_polygon( shell, i, msh, edg, nbe );
      }
   }
    
   return shell;
}

/**************************************************************/
/* Processes a hierarchy of models and merges together all    */
/* the meshes found at the same level. This function will     */
/* also remove unnecessary models.                            */
/**************************************************************/

void process_model( 
   DK_Model *mdl
)
{
   DK_Model *p        = mdl->child;
   DK_Model *last     = NULL;
   DK_Model *mesh_mdl = NULL;

   while ( p ) {
      if ( p->type == DK_MDL_MESH ) {
	 apply_transformation( p );  

	 if ( mesh_mdl ) {
	    merge( mesh_mdl, p );

	 } else {
	    mesh_mdl = p;
	 }
      }

      if ( p->child ) {
	 process_model( p );
      }

      /* Only keep useful nodes. */

      if ( ! p->definition && ! p->child ) {
	 if ( last ) {
	    last->sibling = p->sibling;

	 } else {
	    mdl->child = p->sibling;
	 }

      } else {
	 last = p;
      }

      p = p->sibling;
   }

   if ( mesh_mdl && mesh_mdl->definition ) {
      DK_Mesh    *msh = (DK_Mesh *) mesh_mdl->definition;
      int      i,
	       nbe;
      NewEdge *n;
      int      nb_shells;
      DK_Mesh *rej;
      int      nb_pol = msh->nbPolygons;
      int      cnt = 1;
      
      mesh_mdl->definition = NULL;

      while ( nb_pol ) {
	 nbe = 0;
	 n = build_edge_table( msh, &nbe );

	 rej = copy_mesh( msh );
	 order_mesh( msh, rej, n, nbe );

	 nb_pol = 0;

	 for ( i=0; i<rej->nbPolygons; i++ ) {
	    if ( rej->polygons[ i ].nbNodes ) nb_pol++;
	 }

	 nb_shells = number_of_shells( msh, n, nbe );

	 for ( i=0; i<nb_shells; i++ ) {
	    DK_Mesh *new_mesh = clean_mesh( msh, i+1 );

	    if ( mesh_mdl->definition == NULL ) {
	       mesh_mdl->definition = new_mesh;
	    
	    } else {
	       DK_Model *m = DK_modelAllocate();
	       int    lg;

	       lg = strlen( mesh_mdl->name ) + 5 + 1;

	       m->name = (char *) malloc( lg );

	       sprintf( m->name, "%s_%04d", mesh_mdl->name, cnt );
	       cnt++;

	       m->type       = DK_MDL_MESH;
	       m->definition = new_mesh;
	       m->sibling    = mesh_mdl->sibling;
	       m->parent     = mesh_mdl->parent;

	       mesh_mdl->sibling = m;
	    }
	 }

	 DK_meshDispose( &msh );

	 reset_polygons_flag( rej );
	 reset_vertices_flag( rej );

	 msh = clean_mesh( rej, 0 );

	 DK_meshDispose( &rej );
      }
   }
}

/* Main program. */

main( 
   int   argc, 
   char *argv[] 
)
{
   DK_Model      *mdl;
   int	          i;
   DK_String      inname = NULL;
   DK_String      outname = NULL;
   DK_FileFormat  fmt     = DK_FIL_BINARY;

#ifdef _WIN32
   DK_streamInitialize();
#endif

   for ( i = 1; i < argc; i++ ) {
      if ( ! inname ) {
         inname = argv[i];

      } else if ( ! outname ) {
         outname = argv[i];

      } else {
	 fprintf( stderr, "usage: merge_msh <infile> <outfile> \n" );
	 exit( -1 );
      }
   }
   
   if ( outname == NULL ) {
      printf( "usage: merge_msh <infile> <outfile> \n" );
      exit( -1 );
   }

   if ( mdl = DK_modelReadFile( inname ) ) {
      process_model( mdl );

      DK_NORMAL = 0;

      if ( ! DK_modelWriteFile( outname, mdl, fmt ) ) {
        fprintf( 
	   stderr, 
	   "ERROR: Cannot write output file: %s(.hrc)\n", outname 
	);
      }

   } else {
     fprintf( 
	stderr, 
	"ERROR: Cannot read input file: %s(.hrc)\n", inname 
     );
   }
}
