
/*******************************************************************************
********************************************************************************

   camera.c
   Written by: Colin Hui

   This program reads a raw camera data file with the following format:

   rate [px py pz ix iy iz roll fov]

   where 
       [] denotes zero or more camera data entries
       rate is the scene frame rate (integer)
       px, py, pz is the camera's position (floats)
       ix, iy, iz is the camera's interest (floats)
       roll is the camera's roll (float)
       fov is the camera's field of view (float)

   A ascii scene file is written out.
   The default fcurve interpolation type is linear.

   (c) Copyright 1992 Softimage Inc.

********************************************************************************
*******************************************************************************/

static char copyright[] = "(c) Copyright 1992 Softimage Inc.";

#include "tinySoftType.h"
#include "tinyFile.h"
#include "tinyFcurve.h"
#include "tinyScene.h"
#include "DK_filter.h"

#define KEY_CHUNK 4096

static void 	getFcurve ( Fcurve **, int, int);
static void 	addKey ( Fcurve *, float, float, int);
static int 	cameraRead ( char *, Scene *, int);
static int 	cameraWrite ( char *, Scene *);

main(argc, argv)
int argc;
char *argv[];
{
    Scene *scn;
    float version = 2.5;
    int i, sceneToRaw = FALSE, interp = LINEAR_INTERP;

#ifdef _WIN32
   DK_streamInitialize();
#endif

    if (argc > 2)
    {
	for (i = 1; i < argc; i++)
	{
	    if (argv[i][0] == '-')
	    {
		switch (argv[i][1])
		{
		    case 'r':
			sceneToRaw = TRUE;
			break;
		}
	    }
	}

        scn = sceneAllocate();
	if (scn != NULL)
        {
	    if (sceneToRaw == TRUE)
	    {
	        scn->name = strdup(argv[1]);
	        if (sceneReadAsciiAnimation ( &version,  scn) == TRUE)
		    cameraWrite(argv[2], scn);
	    }
	    else if (cameraRead(argv[1], scn, interp) == TRUE)
	    {
	        scn->name = strdup(argv[2]);
	        sceneWriteAsciiAnimation ( ASCII_FILE_HEADER, version, scn);
	    }
	    sceneDispose(&scn);
	}
    }
    else
    {
	printf("Usage: camera <infile> <outfile> [-r]\n");
	printf("This program reads in a raw camera file and writes out a ascii scene file.\n\n");
	printf("options: -r	outputs a raw camera file from an ascii scene file\n\n");
    }
}

/* allocate and initialize a fcurve */

static void getFcurve
    (
	Fcurve **fcv,	/* fcurve pointer */
	int type,	/* fcurve type */
	int interp	/* fcurve default interpolation type */
    )
{
    *fcv = fcurveAllocate();
    (*fcv)->interp = interp;
    (*fcv)->type = type;
    (*fcv)->size = 0;
}

/* add a key to the fcurve.
   The keys are allocated in chunks of KEY_CHUNK and are re-allocated
   if another KEY_CHUNK is needed 
*/

static void addKey
    (
	Fcurve *fcv,	/* fcurve pointer */
	float time,	/* key time */
	float value,	/* key value */
	int interp	/* key interpolation type */
    )
{
    Fcvkey *key;

    /* allocate key data structures */
    if (fcv->size == 0)
    {
	fcv->keys = _MALLOC( struct fcvkey, sizeof (struct fcvkey) *
		KEY_CHUNK, -1);
    }
    else if ((fcv->size % KEY_CHUNK) == 0)
	fcv->keys = (struct fcvkey *) realloc(fcv->keys,
		sizeof(struct fcvkey) * (fcv->size / KEY_CHUNK + 1) *
		KEY_CHUNK);

    /* assign key values */
    key = &fcv->keys[fcv->size];
    key->time = time;
    key->value = value;
    key->interp = interp;
    fcv->size++;
}

/* read in the raw camera data */

static int cameraRead
    (
	char *filename,		/* raw camera data file */
	Scene *scn,		/* scene pointer */
	int interp
    )
{
    FILE *fp;
    float x, y, z, roll, fov, time, dtime;
    Fcurve *fcv;
    int rate;

    if (filename == NULL || (fp = fopen(filename, "r")) == NULL)
    {
        printf("Unable to open file '%s'.\n",filename);
	return(FALSE);
    }

    printf("Reading camera file '%s'.\n",filename);

    /* allocate camera fcurves */
    getFcurve(&scn->camera.fcv[FCV_CAM_EPOSX], FCV_TYPE_CAM_EPOSX, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_EPOSY], FCV_TYPE_CAM_EPOSY, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_EPOSZ], FCV_TYPE_CAM_EPOSZ, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_EINTX], FCV_TYPE_CAM_EINTX, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_EINTY], FCV_TYPE_CAM_EINTY, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_EINTZ], FCV_TYPE_CAM_EINTZ, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_ROLL], FCV_TYPE_CAM_ROLL, interp);
    getFcurve(&scn->camera.fcv[FCV_CAM_FOV], FCV_TYPE_CAM_FOV, interp);

    /* get frame rate */
    fscanf(fp,"%d",&rate);
    scn->rate = rate;
    dtime = 1.0 / rate;

    time = 0.0;
    /* get camera position */
    while (fscanf(fp,"%f%f%f",&x, &y, &z) > 0)
    {
	addKey(scn->camera.fcv[FCV_CAM_EPOSX], time, x, interp);
	addKey(scn->camera.fcv[FCV_CAM_EPOSY], time, y, interp);
	addKey(scn->camera.fcv[FCV_CAM_EPOSZ], time, z, interp);

	/* get camera interest */
	fscanf(fp,"%f%f%f",&x, &y, &z);
	addKey(scn->camera.fcv[FCV_CAM_EINTX], time, x, interp);
	addKey(scn->camera.fcv[FCV_CAM_EINTY], time, y, interp);
	addKey(scn->camera.fcv[FCV_CAM_EINTZ], time, z, interp);
	 
	/* get camera roll */
	fscanf(fp,"%f",&roll);
	addKey(scn->camera.fcv[FCV_CAM_ROLL], time, roll, interp);

	/* get camera field of view */
	fscanf(fp,"%f",&fov);
	addKey(scn->camera.fcv[FCV_CAM_FOV], time, fov, interp);

	time += dtime;
    }
    fclose(fp);
    return(TRUE);
}

static cameraWrite
    (
	char *filename,		/* raw camera data file */
	Scene *scn		/* scene pointer */
    )
{
    FILE *fp;
    Fcvkey *px, *py, *pz, *ix, *iy, *iz, *roll, *fov;
    int i, size;

    if (filename == NULL || (fp = fopen(filename, "w")) == NULL)
    {
        printf("Unable to open file '%s'.\n",filename);
	return(FALSE);
    }
    
    printf("Writing camera file '%s'.\n",filename);

    /* write frame rate */
    fprintf(fp,"%d\n\n",scn->rate);

    px = scn->camera.fcv[FCV_CAM_EPOSX]->keys;
    py = scn->camera.fcv[FCV_CAM_EPOSY]->keys;
    pz = scn->camera.fcv[FCV_CAM_EPOSZ]->keys;

    ix = scn->camera.fcv[FCV_CAM_EINTX]->keys;
    iy = scn->camera.fcv[FCV_CAM_EINTY]->keys;
    iz = scn->camera.fcv[FCV_CAM_EINTZ]->keys;

    roll = scn->camera.fcv[FCV_CAM_ROLL]->keys;
    fov = scn->camera.fcv[FCV_CAM_FOV]->keys;

    size = scn->camera.fcv[FCV_CAM_EPOSX]->size;
    for (i = 0; i < size; i++)
    {
	fprintf(fp,"%g %g %g\n",px[i].value, py[i].value, pz[i].value);
	fprintf(fp,"%g %g %g\n",ix[i].value, iy[i].value, iz[i].value);
	fprintf(fp,"%g\n",roll[i].value);
	fprintf(fp,"%g\n",fov[i].value);
	fprintf(fp,"\n");
    }
}
