/*
--------------------------------------------------------------------------------

Custom Effect Twist_Anim

File: twist_anim.c 
Associated files: Twist_Anim.cus

April 1994

This custom effect example allows the user to animate the twisting of a mesh
model around its Y-axis by specifying the twist factor at key frames.

DKit (v1.1) functions and structures are used to read and write the SOFTIMAGE
models to and from Creative Environment, and to access the vertices of the mesh.

(c) Copyright 1994, SOFTIMAGE Inc.

--------------------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#ifdef _WIN32
#include <irixMath.h>
#endif

#include "tinyModel.h"
#include "DK_filter.h"


#define DEG2RAD(x)	((x) * M_PI / 180.0)

static void	Read_Data( FILE * );
static void	Read_Icon( void );
static void	Read_Model( char * );
static void	Process_Mesh( void );
static void	Write_Model( char * );

static float	factor;
static Model	*mdl;
static DK_Mesh	*mesh;
static float	IcnSx,IcnSy,IcnSz,IcnRx,IcnRy,IcnRz,IcnTx,IcnTy,IcnTz;

void main(int argc, char *argv[])
{
   FILE 	*fd_data;
   char		input_mdl[256];
   char		output_mdl[256];

#ifdef _WIN32
   DK_streamInitialize();
#endif

   if (argc < 3)
   {
      fprintf(stderr,"Insufficient arguments\n");
      exit(-1);
   }

   if (strncmp(argv[1],"-",1) == 0)
   {
      fd_data = stdin;
   }
   else	
   {
      if ((fd_data = fopen(argv[1], "r")) == NULL)
      {
	 printf("Unable to open file '%s'.\n",argv[1]);
	 exit( -1 );
      }
   }

   Read_Data(fd_data);
   Read_Icon();
   Read_Model(argv[2]);
   Process_Mesh();

   /* DK_NORMAL flag: TRUE means SOFTIMAGE will recompute the normals. */
   DK_NORMAL = TRUE;

   Write_Model(argv[3]);
   DK_modelDispose(&mdl);

   while(1)
   {
      Read_Data(fd_data);
      Read_Icon();
      Read_Model(argv[2]);
      Process_Mesh();

      /* DK_NORMAL flag: TRUE means SOFTIMAGE will recompute the normals. */
      DK_NORMAL = TRUE;

      Write_Model(argv[3]);
      DK_modelDispose(&mdl);
   }
}

/*
--------------------------------------------------------------------------------
Read in the parameters specified by the user in the dialog window.
The (ascii) format is:
_DATA
FACTOR float
_END
--------------------------------------------------------------------------------
*/
static void Read_Data( FILE *fd_data )
{
   char 	buffer[256];
   int		ret;

   strcpy(buffer,"");
   while(1)
   {
      if ((ret = fscanf(fd_data,"%s",buffer)) != 1)
	 exit(0);

      if (strncmp(buffer,"_END",4) == 0)
      {
	 fscanf(fd_data,"\n");
	 break;
      }

      if (strncmp(buffer,"_DATA",5) == 0)
	 fscanf(fd_data,"\n");

      if (strncmp(buffer,"FACTOR",6) == 0)
	 fscanf(fd_data,"%f\n",&factor);
   }
}

/*
--------------------------------------------------------------------------------
Read in the scale, rotation, and translation values for the Animation Icon.
--------------------------------------------------------------------------------
*/
static void Read_Icon( void )
{
   fscanf(stdin,"%f %f %f\n",&IcnSx,&IcnSy,&IcnSz);
   fscanf(stdin,"%f %f %f\n",&IcnRx,&IcnRy,&IcnRz);
   fscanf(stdin,"%f %f %f\n",&IcnTx,&IcnTy,&IcnTz);
}


/*
--------------------------------------------------------------------------------
Read in the model specified by the user.
If the first argument was a '-', then read from stdin (pipes are being used.)
Else, read in the model from the .hrc file specified as arg 1.
--------------------------------------------------------------------------------
*/
static void Read_Model( char *modelname )
{ 
   if (strncmp(modelname,"-",1) == 0)
   {
      if(!( mdl = DK_fdmodelReadBinaryFile( stdin ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read model from stdin.\n");
      }
   }
   else
   {
      if(!( mdl = DK_modelReadFile( modelname ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read input file: %s(.hrc)\n", modelname );
      }
   }

   mesh = (DK_Mesh *) (mdl->definition);
}

/*
--------------------------------------------------------------------------------
Do the twist.

The transformation that is applied to the model's vertices is:

x = x*cos(theta) + z*sin(theta)
y = y
z = -x*sin(theta) + z*cos(theta)
where theta = factor*y*scale
where factor = float provided by user
where scale = Icon's X-axis scale value.

NOTE: The center of the twist is located at the x,y,z of the Icon.  
Therefore, as added flexiblity, the Icon can be animated, to affect
the twisting factor and the center of the twist.
--------------------------------------------------------------------------------
*/
static void Process_Mesh( void )
{
   long   NumVertices;
   float  x,y,z,newX,newZ;
   double theta;
   int    i;
   DK_Vertex *My_Vertex;
   DK_Point  *My_Point;

   NumVertices = (mesh->nbVertices);
   for (i = 0; i < NumVertices; i++)
   {
      My_Vertex = &(mesh->vertices[i]);
      My_Point = &(My_Vertex->position);

      x = (My_Point->x) - IcnTx;
      y = (My_Point->y) - IcnTy;
      z = (My_Point->z) - IcnTz;

      theta = DEG2RAD(factor * y * IcnSx);

      newX = (float) (x * cos(theta)) + (z * sin(theta));
      newZ = (float) ((-x) * sin(theta)) + (z * cos(theta));

      (My_Point->x) = newX + IcnTx;
      (My_Point->z) = newZ + IcnTz;
   }
}

/*
--------------------------------------------------------------------------------
Write the twisted model out.
If the 2nd argument was '-', write it out to stdout.
Otherwise write it out to the specified file.
--------------------------------------------------------------------------------
*/
static void Write_Model( char *outname )
{
   /* Write model file */
   if (strncmp(outname,"-",1) == 0)
   {
      DK_fdmodelWriteBinaryFile(stdout, mdl);
   }
   else
   {
      DK_modelWriteFile(outname, mdl, FIL_BINARY );
   }
}

