/*
Custom Effect Twist

File: twist.c 
Associated files: Twist.cus and Twist_Pipe.cus

April 1994

This custom effect example allows the user to twist a mesh model.
The user specifies a twist factor and selects a mesh. The custom
effect creates a new mesh twisted around its Y-axis.

DKit (v1.1) functions and structures are used to read and write the SOFTIMAGE
models to and from Creative Environment, and to access the vertices of the mesh.

The twist executable may be run as a standalone on the command line, or as
a custom effect from within Creative Environment.

Usage: twist <data_file> <input_mdl> <output_mdl>

When '-' is specified for <data_file> or <input_mdl>, the (dialog) data
or model is read from stdin, instead of a file.

When '-' is specified for <output_mdl>, the twisted model gets written to
stdout, instead of a .hrc file.

Twist_Pipe.cus, for example, uses stdin and stdout, and calls twist with '-'
as arguments, whereas Twist.cus reads and writes to files.

(c) Copyright 1994, SOFTIMAGE Inc.
--------------------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#ifdef _WIN32
#include <irixMath.h>
#endif
#include "tinyModel.h"
#include "DK_filter.h"

#define DEG2RAD(x)	((x) * M_PI / 180.0)

static void	Read_Data( FILE * );
static void	Read_Model( char * );
static void	Process_Mesh( void );
static void	Write_Model( char * );

static float	factor;
static Model	*mdl;
static DK_Mesh	*mesh;

void main(int argc, char *argv[])
{
   FILE 	*fd_data;

#ifdef _WIN32
   DK_streamInitialize();
#endif

   if (argc < 3)
   {
      fprintf(stderr,"Insufficient arguments\n");
      exit(-1);
   }

   if (strncmp(argv[1],"-",1) == 0)
   {
      fd_data = stdin;
   }
   else
   {
      if ((fd_data = fopen(argv[1], "r")) == NULL)
      {
	 printf("Unable to open file '%s'.\n",argv[1]);
	 exit( -1 );
      }
   }

   Read_Data(fd_data);
   Read_Model(argv[2]);
   Process_Mesh();

   /* DK_NORMAL flag: TRUE means SOFTIMAGE will recompute the normals. */
   DK_NORMAL = FALSE;

   Write_Model(argv[3]);
}

/*
--------------------------------------------------------------------------------
Read in the parameters specified by the user in the dialog window.
The (ascii) format is:
_DATA
FACTOR float
_END
--------------------------------------------------------------------------------
*/
static void Read_Data( FILE *fd_data )
{
   char 	buffer[256];

   strcpy(buffer,"");
   while(1)
   {
      fscanf(fd_data,"%s",buffer);

      if (strncmp(buffer,"_END",4) == 0)
      {
	 fscanf(fd_data,"\n");
	 break;
      }

      if (strncmp(buffer,"_DATA",5) == 0)
	 fscanf(fd_data,"\n");

      if (strncmp(buffer,"FACTOR",6) == 0)
	 fscanf(fd_data,"%f\n",&factor);
   }
}

/*
--------------------------------------------------------------------------------
Read in the model specified by the user.
If the first argument was a '-', then read from stdin (pipes are being used.)
Else, read in the model from the .hrc file specified as arg 1.
--------------------------------------------------------------------------------
*/
static void Read_Model( char *modelname )
{ 
   if (strncmp(modelname,"-",1) == 0)
   {
      if(!( mdl = DK_fdmodelReadBinaryFile( stdin ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read model from stdin.\n");
      }
   }
   else	
   {
      if(!( mdl = DK_modelReadFile( modelname ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read input file: %s(.hrc)\n", modelname );
      }
   }

   mesh = (DK_Mesh *) (mdl->definition);
}

/*
--------------------------------------------------------------------------------
Do the twist.

The transformation that is applied to the model's vertices is:

x = x*cos(theta) + z*sin(theta)
y = y
z = -x*sin(theta) + z*cos(theta)

where theta = factor*y and factor = float provided by user.
--------------------------------------------------------------------------------
*/
static void Process_Mesh( void )
{
   long   NumVertices;
   float  x,y,z,newX,newZ;
   double theta;
   int    i;
   DK_Vertex *My_Vertex;
   DK_Point  *My_Point;

   NumVertices = (mesh->nbVertices);
   for (i = 0; i < NumVertices; i++)
   {
      My_Vertex = &(mesh->vertices[i]);
      My_Point = &(My_Vertex->position);
      x = (My_Point->x);
      y = (My_Point->y);
      z = (My_Point->z);
      theta = DEG2RAD(factor * y);
      newX = (float) (x * cos(theta)) + (z * sin(theta));
      newZ = (float) ((-x) * sin(theta)) + (z * cos(theta));

      (My_Point->x) = newX;
      (My_Point->z) = newZ;
   }
}

/*
--------------------------------------------------------------------------------
Write the twisted model out.
If the 2nd argument was '-', write it out to stdout.
Otherwise write it out to the specified file.
--------------------------------------------------------------------------------
*/

static void Write_Model( char *outname )
{
   FILE	*fd_out;

   /* Write model file */
   if (strncmp(outname,"-",1) == 0)
   {
      fd_out = stdout;
      DK_fdmodelWriteBinaryFile(fd_out, mdl);
   }
   else
   {
      DK_modelWriteFile(outname, mdl, FIL_BINARY );
   }
}

