/*
Custom Effect Orbit

File: orbit.c 
Associated files: Orbit.cus

April 1994

This custom effect example allows the user to specify a tag point, a radius,
a plane (X, Y, or Z), and a model, and orbits the model around this point,
parallel to the specified plane.

DKit (v1.1) functions and structures are used to read and write the SOFTIMAGE
models to and from Creative Environment, and to access the vertices of the
tagged mesh.

(c) Copyright 1994, SOFTIMAGE Inc.
--------------------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#ifdef _WIN32
#include <irixMath.h>
#endif

#include "tinyModel.h"
#include "DK_filter.h"

#define DEG2RAD(x)      ((x) * M_PI / 180.0)
#define RATE		30

static void	Read_Data( FILE * );
static void	Read_Tag( void );
static void	Read_TagModel( char *modelname );
static void	Read_Srt( void );
static void	Read_Time( void );
static void	Process_Orbit( void );
static void	Write_Srt( void );

static float	radius,duration;
static int	xy,xz,yz;
static Model	*mdl;
static DK_Mesh	*mesh;
static char	*tag;
static float	Sx,Sy,Sz,Rx,Ry,Rz,Tx,Ty,Tz;
static float	time;

void main(int argc, char *argv[])
{
   FILE 	*fd_data;

#ifdef _WIN32
   DK_streamInitialize();
#endif

   if (argc < 5)
   {
      fprintf(stderr,"Insufficient arguments\n");
      exit(-1);
   }

   /* Read in the parameters specified by the user in the dialog window */
   Read_Data(stdin);

   Read_Time();

   /* Read in the orbiting model's scale, rotation,and translation */
   Read_Srt();

   /* Read in the user-selected model around which to orbit */
   Read_TagModel(argv[3]);

   /* Read in its tag points information */
   Read_Tag();

   Process_Orbit();

   /* Write out the orbiting model's scale, rotation,and translation */
   Write_Srt();

   DK_modelDispose(&mdl);

   while(1)
   {
      Read_Data(stdin);
      Read_Time();
      Read_Srt();
      Read_TagModel(argv[3]);
      Process_Orbit();
      Write_Srt();
      DK_modelDispose(&mdl);
   }
}

/*
--------------------------------------------------------------------------------
Read in the parameters specified by the user in the dialog window.
The (ascii) format is:

_DATA
RADIUS float 		(radius of revolution)
DURATION float 		(frames per revolution)
XY integer 		(0 or 1)
XZ integer 		(0 or 1)
YZ integer 		(0 or 1)
_END
--------------------------------------------------------------------------------
*/
static void Read_Data( FILE *fd_data )
{
   char 	buffer[256];
   int		ret;

   strcpy(buffer,"");
   while(1)
   {
      if ((ret = fscanf(fd_data,"%s",buffer)) != 1)
	 exit(0);

      if (strncmp(buffer,"_END",4) == 0)
      {
	 fscanf(fd_data,"\n");
	 break;
      }

      if (strncmp(buffer,"_DATA",5) == 0)
	 fscanf(fd_data,"\n");

      if (strncmp(buffer,"RADIUS",6) == 0)
	 fscanf(fd_data,"%f\n",&radius);

      if (strncmp(buffer,"DURATION",8) == 0)
	 fscanf(fd_data,"%f\n",&duration);

      if (strncmp(buffer,"XY",2) == 0)
	 fscanf(fd_data,"%d\n",&xy);

      if (strncmp(buffer,"XZ",2) == 0)
	 fscanf(fd_data,"%d\n",&xz);

      if (strncmp(buffer,"YZ",2) == 0)
	 fscanf(fd_data,"%d\n",&yz);
   }
}

/*
--------------------------------------------------------------------------------
Read in the tag points information for the model around which to orbit.
The format of this information is as follows:

The first 4 bytes give the number of vertices in the object.
The rest of the list is a list of bytes representing a 1 if the vertex
is tagged, and 0 if it is not. 

For example:  For a cube with the first 4 points tagged, the binary
list would be: 8,1,1,1,1,0,0,0,0
--------------------------------------------------------------------------------
*/
static void Read_Tag( void )
{ 
   int	num;

   free(tag);
   fread(&num,4,1,stdin);
   tag = (char *) malloc(num * sizeof(char));
   fread(tag,num,1,stdin);
}


/*
--------------------------------------------------------------------------------
Read in the model around which to orbit.
If '-' was specified instead of a file name, then read from stdin (pipe).
Else, read in the model from the specified .hrc file.
--------------------------------------------------------------------------------
*/
static void Read_TagModel( char *modelname )
{ 
   if (strncmp(modelname,"-",1) == 0)
   {
      if( !( mdl = DK_fdmodelReadBinaryFile( stdin ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read model from stdin.\n");
      }
   }
   else
   {
      if( !( mdl = DK_modelReadFile( modelname ) ) )
      {
	 fprintf(stderr, "ERROR: Cannot read input file: %s(.hrc)\n", modelname );
      }
   }

   mesh = (DK_Mesh *) (mdl->definition);
}


/*
--------------------------------------------------------------------------------
Read in the orbiting model's scale, rotation, and translation values.
--------------------------------------------------------------------------------
*/
static void Read_Srt( void )
{ 
   fscanf(stdin,"%f %f %f\n",&Sx,&Sy,&Sz);
   fscanf(stdin,"%f %f %f\n",&Rx,&Ry,&Rz);
   fscanf(stdin,"%f %f %f\n",&Tx,&Ty,&Tz);
}

/*
--------------------------------------------------------------------------------
Read in the time.
--------------------------------------------------------------------------------
*/
static void Read_Time( void )
{ 
   fscanf(stdin,"%f\n",&time);
}


/*
--------------------------------------------------------------------------------
Calculate the orbit.

If the Tag model is scaled or rotated or translated, the Orbit model will
follow those changes, maintaining the same original orbit plane.

(So it really represents a shift in the orbit plane when the Tag model
undergoes any changes in scale, rotation, or translation.)
--------------------------------------------------------------------------------
*/
static void Process_Orbit( void )
{
   long   NumVertices;
   float  x,y,z;
   double theta,Rtheta;
   float  NewX,NewY,NewZ;
   float  TSX,TSY,TSZ,TRX,TRY,TRZ,TTX,TTY,TTZ;
   float  RX,RY,RZ;
   float  tmp1, tmp2, tmp3;
   int    i;
   DK_Vertex *My_Vertex;
   DK_Point  *My_Point;

   TTX = (float)(mdl->translation.x);
   TTY = (float)(mdl->translation.y);
   TTZ = (float)(mdl->translation.z);
   TSX = (float)(mdl->scaling.x);
   TSY = (float)(mdl->scaling.y);
   TSZ = (float)(mdl->scaling.z);
   TRX = (float)(mdl->rotation.x);
   TRY = (float)(mdl->rotation.y);
   TRZ = (float)(mdl->rotation.z);

   NumVertices = (mesh->nbVertices);
   for (i = 0; i < NumVertices; i++)
   {
      if (tag[i])
      {
	 My_Vertex = &(mesh->vertices[i]);
	 My_Point  = &(My_Vertex->position);

	 x = (My_Point->x) * TSX;
	 y = (My_Point->y) * TSY;
	 z = (My_Point->z) * TSZ;

	 RX = x;
	 RY = y;
	 RZ = z;

	 /* Rotation in Y-Z plane */
	 Rtheta = TRX;
	 tmp1 = RX;
	 tmp2 = (float) (RY * cos(Rtheta)) - (RZ * sin(Rtheta));
	 tmp3 = (float) (RY * sin(Rtheta)) + (RZ * cos(Rtheta));
	 RX = tmp1;
	 RY = tmp2;
	 RZ = tmp3;

	 /* Rotation in X-Z plane */
	 Rtheta = TRY;
	 tmp1 = (float) (RX * cos(Rtheta)) - (RZ * sin(Rtheta));
	 tmp2 = RY;
	 tmp3 = (float) (RX * sin(Rtheta)) + (RZ * cos(Rtheta));
	 RX = tmp1;
	 RY = tmp2;
	 RZ = tmp3;

	 /* Rotation in X-Y plane */
	 Rtheta = TRZ;
	 tmp1 = (float) (RX * cos(Rtheta)) - (RY * sin(Rtheta));
	 tmp2 = (float) (RX * sin(Rtheta)) + (RY * cos(Rtheta));
	 tmp3 = RZ;
	 RX = tmp1;
	 RY = tmp2;
	 RZ = tmp3;

	 theta = DEG2RAD((360 * (time * RATE)) / duration);

	 if (xz)
	 {
	    NewX = (float) radius * cos(theta);
	    NewY = 0;
	    NewZ = (float) radius * sin(theta);
	 }

	 if (xy)
	 {
	    NewX = (float) radius * cos(theta);
	    NewY = (float) radius * sin(theta);
	    NewZ = 0;
	 }

	 if (yz)
	 {
	    NewX = 0;
	    NewY = (float) radius * cos(theta);
	    NewZ = (float) radius * sin(theta);
	 }

	 if (!xy && !xz && !yz)
	 {
	    NewX = (float) radius * cos(theta);
	    NewY = 0;
	    NewZ = (float) radius * sin(theta);
	 }

	 Tx = NewX + RX + TTX;
	 Ty = NewY + RY + TTY;
	 Tz = NewZ + RZ + TTZ;

	 break;
      }
   }
}

/*
--------------------------------------------------------------------------------
Write out the orbiting model's scale, rotation, and translation values.
--------------------------------------------------------------------------------
*/
static void Write_Srt( void )
{
   fprintf(stdout,"%f %f %f\n",Sx,Sy,Sz);
   fprintf(stdout,"%f %f %f\n",Rx,Ry,Rz);
   fprintf(stdout,"%f %f %f\n",Tx,Ty,Tz);
   fflush(stdout);
}

