{************************************************}
{                                                }
{   Turbo Pascal 7.0                             }
{   Graphic Vision Demo                          }
{   Copyright (c) 1995 Jason G Burgon            }
{                                                }
{************************************************}

{ Mouse option dialog used by TVDEMO.PAS and TVRDEMO.PAS }

unit MouseDlg;

{$S-,X+,I-}
{$IFNDEF DPMI}
{$F+,O+}
{$ENDIF}
{$IFNDEF DEBUG}
{$L-,D-}
{$ENDIF}

interface

uses GObjects, ScrnDriv, GDrivers, GViews, GWindows, GDialogs;

const
  CClickTester = #31#32;

type
  { TClickTester }

  {Palette layout}
  { 1 = Unclicked }
  { 2 = Clicked }

  PClickTester = ^TClickTester;
  TClickTester = object(TView)
    Clicked: Boolean;
    Text   : String[10];
    constructor Init(var Bounds: TRect; AText: String);
    constructor Load(var S: TStream);
    procedure Store(var S: TStream);
    procedure Draw; virtual;
    function  GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  { TDoubleScrollBar }

  PDoubleScrollBar = ^TDoubleScrollBar;
  TDoubleScrollBar = object(TGroup)
    BigBar    : PScrollBar;
    TopBar    : PScrollBar;
    BottomBar : PScrollBar;
    LastValue : integer;
    constructor Init(var Bounds: TRect; AOptions: word);
    constructor Load(var S: TStream);
    procedure HandleEvent(var Event : TEvent); virtual;
    procedure Store(var S: TStream);
  end;

  PMouseSense = ^TMouseSense;
  TMouseSense = object(TDoubleScrollBar)
    DoubleSpeed : word;
    constructor Init(var Bounds: TRect);
    constructor Load(var S: TStream);
    procedure GetSensitivity(var Hor, Vert, Double : word);
    procedure HandleEvent(var Event : TEvent); virtual;
    procedure SetSensitivity(Hor, Vert, Double : word);
   private
    procedure SetControls;
  end;

  { TMouseDialog }

  PMouseDialog = ^TMouseDialog;
  TMouseDialog = object(TDialog)
    Acceleration   : PRadioButtons;
    DoubleClickBar : PScrollBar;
    Sensitivity    : PMouseSense;
    SwapButtons    : PCheckBoxes;
    constructor Init;
    constructor Load(var S: TStream);
    function  GetAcceleration : word;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure SaveSettings;
    procedure SetAcceleration(Curve : word);
    procedure ReverseButtons(Swap : Boolean);
    procedure Store(var S: TStream);
   private
    OldHor         : word;
    OldVert        : word;
    OldDoubleSpeed : word;
    OldDoubleDelay : word;
    OldAccelCurve  : word;
    OldReverse     : WordRec;
  end;

{ Stream registration records }

const
  RClickTester: TStreamRec = (
     ObjType: 210;
     VmtLink: Ofs(TypeOf(TClickTester)^);
     Load:    @TClickTester.Load;
     Store:   @TClickTester.Store
  );
const
  RDoubleScrollBar: TStreamRec = (
     ObjType: 211;
     VmtLink: Ofs(TypeOf(TDoubleScrollBar)^);
     Load:    @TDoubleScrollBar.Load;
     Store:   @TDoubleScrollBar.Store
  );
const
  RMouseSense: TStreamRec = (
     ObjType: 212;
     VmtLink: Ofs(TypeOf(TMouseSense)^);
     Load:    @TMouseSense.Load;
     Store:   @TMouseSense.Store
  );
const
  RMouseDialog: TStreamRec = (
     ObjType: 213;
     VmtLink: Ofs(TypeOf(TMouseDialog)^);
     Load:    @TMouseDialog.Load;
     Store:   @TMouseDialog.Store
  );

procedure RegisterMouseDlg;

implementation

 { TDoubleScrollBar }

constructor TDoubleScrollBar.init(var Bounds: TRect; AOptions: word);
var
  R,S: TRect;
begin
  inherited Init(Bounds);
  ID := 211;
  Options := Options or ofFirstClick;
  GetExtent(S);
  New(BigBar, Init(S, AOptions));           { This is the Big Bar behind    }
  BigBar^.ButtonSize.X := CharSize.X shl 1; { the two smaller scrollbars    }
  BigBar^.SetParams(15000, 0, 32000, 1, 1);
  LastValue := 15000;
  Insert(BigBar);

  R.A.X := S.A.X +2; R.A.Y := S.A.Y;
  R.B.X := S.B.X -2; R.B.Y := S.B.Y shr 1;
  New(TopBar, Init(R, AOptions));
  Insert(TopBar);
  { TopBar^.Hide; }

  Inc(R.A.Y); Inc(R.B.Y, R.B.Y);
  New(BottomBar, Init(R, AOptions));
  Insert(BottomBar);
  { BottomBar^.Hide; }
end;

constructor TDoubleScrollBar.Load(var S: TStream);
begin
  inherited Load(S);
  GetSubViewPtr(S, BigBar);
  GetSubViewPtr(S, TopBar);
  GetSubViewPtr(S, BottomBar);
  LastValue := 15000;
  BigBar^.Value :=15000;
end;

procedure TDoubleScrollBar.Store(var S : TStream);
begin
  inherited Store(S);
  PutSubViewPtr(S, BigBar);
  PutSubViewPtr(S, TopBar);
  PutSubViewPtr(S, BottomBar);
end;

procedure TDoubleScrollBar.HandleEvent(var Event : TEvent);
var
  part: integer;
begin
  inherited HandleEvent(Event);
  if (Event.What = evBroadCast) AND (Event.InfoPtr = BigBar) then
   begin
    if (Event.Command = cmScrollBarChanged) then
     begin
      if BigBar^.Value < LastValue
       then part := sbLeftArrow
       else part := sbRightArrow;
      TopBar^.SetValue(TopBar^.Value + TopBar^.ScrollStep(part));
      BottomBar^.SetValue(BottomBar^.Value + BottomBar^.ScrollStep(part));
      LastValue := BigBar^.Value;
     end;
    ClearEvent(Event);
   end;
end;

{ TMouseSense }

constructor TMouseSense.Init(var Bounds: TRect);
begin
  inherited Init(Bounds, sbHorizontal);
  ID := 212;
  SetControls;
end;

constructor TMouseSense.Load(var S: TStream);
begin
  inherited Load(S);
  SetControls;
end;

procedure TMouseSense.HandleEvent(Var Event : TEvent);
begin
  inherited HandleEvent(Event);
  with Event do
  if (What = evBroadcast) and (Command = cmScrollBarChanged) then
   if ((InfoPtr = BottomBar) or (Event.InfoPtr = TopBar)) then
    begin
     SetSensitivity(BottomBar^.Value, TopBar^.Value, DoubleSpeed);
     ClearEvent(Event);
    end else if InfoPtr = BigBar
     then ClearEvent(Event);
end;

procedure TMouseSense.GetSensitivity(var Hor, Vert, Double: word); assembler;
asm
            call  GetMouseSense;
            les   si,[hor]
            mov   [es:si],bx
            les   si,[Vert]
            mov   [es:si],cx
            les   si,[Double]
            mov   [es:si],dx
end;

procedure TMouseSense.SetSensitivity(Hor, Vert, Double: word); assembler;
asm
            mov   bx,[Hor]
            mov   cx,[Vert]
            mov   dx,[Double]
            call  SetMouseSense
end;

procedure TMouseSense.SetControls;                    { Private method      }
var
  HorSense, VertSense: word;
begin
  GetSensitivity(HorSense, VertSense, DoubleSpeed);   { Set the sensitivity }
  TopBar^.SetParams(VertSense, 1, 100, 10, 2);        { controls values to  }
  BottomBar^.SetParams(HorSense, 1, 100, 10, 2);      { current mouse state }
end;

{ TClickTester }

constructor TClickTester.Init(var Bounds: TRect; AText: String);
begin
  inherited Init(Bounds);
  ID := 210;
  SizeAdj.B.Y := CharSize.Y shr 1;
  Options := Options or (ofSelectable + ofFirstClick);
  Text := AText;
  Clicked := False;
end;

constructor TClickTester.Load(var S : TStream);
begin
  inherited Load(S);
  S.Read(Text, SizeOf(Text));
  Clicked := False;
end;

procedure TClickTester.Store(var S : TStream);
begin
  inherited Store(S);
  S.Write(Text, SizeOf(Text));
end;

function TClickTester.GetPalette: PPalette;
const
  P: String[Length(CClickTester)] = CClickTester;
begin
  GetPalette := @P;
end;

procedure TClickTester.HandleEvent(var Event: TEvent);
begin
  TView.HandleEvent(Event);
  if (Event.What = evMouseDown) then
  begin
    if Event.Double then
    begin
      Clicked := not Clicked;
      DrawView;
    end;
    ClearEvent(Event);
  end;
end;

procedure TClickTester.Draw;
var
  R    : TRect;
  Color: Longint;
begin
  GetPixExtent(R);
  if Clicked
   then Color := MapColor(2)
   else Color := MapColor(1);
  DrawButton(R, LongRec(Color).Lo, Text, 1, 1, False);
end;

{ TMouseDialog }

constructor TMouseDialog.Init;
var
  S: integer;
  R: TRect;
  P: PView;

 procedure MakeSimpleFrame;
  begin
    P:= New(PFramer, Init(R, EmptyFill));
    P^.SizeAdj.A.Y := CharSize.Y shr 1;
    P^.Options := P^.Options or ofFramed;
    Insert(P);
  end;

begin
  R.Assign(0, 0, 44, 20);
  TDialog.Init(R, 'Mouse options');
  ID := 213;
  Options := Options or ofCentered;
  if Frame <> nil
   then Frame^.Interior := EmptyFill;

  R.Assign(3, 3, 30, 7);
  New(Acceleration, Init(R,
    NewSItem('~S~low',
    NewSItem('~M~oderate',
    NewSItem('~F~ast',
    NewSItem('~U~naccelerated', nil))))));
  Insert(Acceleration);
  Dec(R.A.Y); R.B.Y := R.A.Y+1;
  Insert(New(PLabel, Init(R, '~A~cceleration', Acceleration)));

  Dec(R.A.X); Inc(R.B.X,12);
  Inc(R.A.Y, 6); R.B.Y := R.A.Y + 3;    { Define Bounding Rectangle for the }
  MakeSimpleFrame;                      { Double-Click area                 }

  Inc(R.A.X); R.B.X := 31; Inc(R.A.Y); R.B.Y := R.A.Y+1;
  New(DoubleClickBar, Init(R, sbHorizontal));
  with DoubleClickBar^ do
   begin
    SizeAdj.A.X := -1;
    Options := Options or (ofSelectable + ofFirstClick);
    SetParams(1, 1, 20, 20, 1);
   end;
  Insert(DoubleClickBar);
  R.B.X := R.A.X + 18; Dec(R.A.Y); Dec(R.B.Y);
  Insert(New(PLabel, Init(R, '~D~ouble Click Speed', DoubleClickBar)));

  Inc(R.A.Y, 2); R.B.X := R.A.X + 28; R.B.Y := R.A.Y+1;
  Insert(New(PLabel, Init(R, 'Fast        Medium      Slow', nil)));
  R.A.X := 32; R.B.X := R.A.X + 9; Dec(R.A.Y);
  Insert(New(PClickTester, Init(R, 'TEST')));

  R.A.X := 3; R.B.X := 30; Inc(R.A.Y,3); Inc(R.B.Y, 2);
  New(SwapButtons, Init(R, NewSItem('~R~everse mouse buttons', nil)));
  Insert(SwapButtons);

  Inc(R.A.Y, 2); Inc(R.B.Y, 6);        { Define Bounding Rectangle for the }
  Dec(R.A.X); Inc(R.B.X, 12);          { Mouse Sensitivity controls        }
  MakeSimpleFrame;

  Inc(R.A.X); Inc(R.A.Y, 2); R.B.X := 41; R.B.Y := R.A.Y+2;
  Sensitivity := New(PMouseSense, Init(R));
  Insert(Sensitivity);
  R.B.X := R.A.X + 11; Dec(R.A.Y,2); R.B.Y := R.A.Y+1;
  Insert(New(PLabel, Init(R, '~S~ensitivity', Sensitivity)));

  Inc(R.A.Y); Inc(R.B.Y); R.A.X := 17; R.B.X := 26;
  Insert(New(PLabel, init(R, 'Vertical', Sensitivity^.TopBar)));

  Inc(R.A.Y,3); Inc(R.B.Y,3); R.A.X := 3; R.B.X := 41;
  Insert(New(PLabel, init(R, 'Slow          Horizontal          Fast',
                          Sensitivity^.BottomBar)));

  S := (CharSize.Y shr 2) + 1;
  R.Assign(32, 3, 42, 5);
  P := New(PButton, Init(R, 'O~K~', cmOk, bfDefault));
  Dec(P^.SizeAdj.A.Y, 1);
  Inc(P^.SizeAdj.B.Y, 1);
  Insert(P);
  Inc(R.A.Y, 2); Inc(R.B.Y, 2);
  P := New(PButton, Init(R, 'Cancel', cmCancel, bfNormal));
  Inc(P^.SizeAdj.A.Y, S);
  Dec(P^.SizeAdj.B.Y, S);
  Insert(P);
  Inc(R.A.Y, 7); Inc(R.B.Y, 7);
  P := New(PButton, Init(R, '~H~elp', cmHelp, bfNormal));
  Dec(P^.SizeAdj.A.Y, 1);
  Inc(P^.SizeAdj.B.Y, 1);
  Insert(P);

  SelectNext(False);
  SaveSettings;
end;

constructor TMouseDialog.Load(var S: TStream);
begin
  TDialog.Load(S);
  GetSubViewPtr(S, Acceleration);
  GetSubViewPtr(S, DoubleClickBar);
  GetSubViewPtr(S, Sensitivity);
  GetSubViewPtr(S, SwapButtons);
  SaveSettings;
end;

procedure TMouseDialog.Store(var S: TStream);
begin
  TDialog.Store(S);
  PutSubViewPtr(S, Acceleration);
  PutSubViewPtr(S, DoubleClickBar);
  PutSubViewPtr(S, Sensitivity);
  PutSubViewPtr(S, SwapButtons);
end;

function TMouseDialog.GetAcceleration: word; assembler;
asm
            mov   bx,-1
            call  GDrivers.GetSetAccelCurve
            mov   ax,bx
            dec   ax
end;

procedure TMouseDialog.SetAcceleration(Curve : word); assembler;
asm
            mov   bx,[Curve]
            inc   bx
            call  GDrivers.GetSetAccelCurve
end;

procedure TMouseDialog.SaveSettings;
begin
  OldReverse.Lo := Byte(MouseReverse);
  SwapButtons^.SetData(OldReverse);
  Sensitivity^.GetSensitivity(OldHor, OldVert, OldDoubleSpeed);
  OldDoubleDelay := DoubleDelay;
  DoubleClickBar^.SetValue(DoubleDelay);
  OldAccelCurve := GetAcceleration;
  Acceleration^.SetData(OldAccelCurve);
end;

procedure TMouseDialog.ReverseButtons(Swap: Boolean);
begin
  MouseReverse := Swap;
end;

procedure TMouseDialog.HandleEvent(var Event: TEvent);
begin
 if (Event.What = evCommand) AND (Event.Command = cmCancel) then
   begin
    MouseReverse := Boolean(OldReverse.Lo);
    Sensitivity^.SetSensitivity(OldHor, OldVert, OldDoubleSpeed);
    DoubleDelay := OldDoubleDelay;
    SetAcceleration(OldAccelCurve);
   end;
  inherited HandleEvent(Event);
  if Event.What = evBroadCast then
    begin
     case Event.Command of
      cmScrollBarChanged : DoubleDelay := DoubleClickBar^.Value;
      cmClusterChanged   : if Event.InfoPtr = Acceleration
                            then SetAcceleration(Acceleration^.Value)
                            else MouseReverse := boolean(SwapButtons^.Value);
      else exit;
      ClearEvent(Event);
     end;
    end;
end;

procedure RegisterMouseDlg;
begin
  RegisterType(RClickTester);
  RegisterType(RDoubleScrollBar);
  RegisterType(RMouseSense);
  RegisterType(RMouseDialog);
end;

end.
