
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphic Vision Unit                             }
{                                                       }
{       Copyright (c) 1995 J G Burgon                   }
{                                                       }
{       Video Mode Selector Dialog (14:10:94)           }
{       Version : 1.1 (31:10:94)                        }
{                                                       }
{*******************************************************}

unit ModeSlct;

{$S-,X+,I-,V-}
{$IFNDEF DPMI}
{$F+,O+}
{$ENDIF}
{$IFNDEF DEBUG}
{$L-,D-}
{$ENDIF}

interface
uses GObjects, ScrnDriv, GDrivers, GViews, GWindows, GDialogs;

const
 cmTestVideoMode = 900;

type
 TModeRes = record                { Video Mode information record type      }
  Mode       : word;              { IBM or VESA video mode number           }
  Resolution : TPoint;            { Screen X and Y resolution in pixels     }
  ColorDepth : Longint;           { Number of colours                       }
 end;

 PModeResArray = ^TModeResArray;
 TModeResArray = Array[0..99] of TModeRes;  { This is 1105 bytes big        }

{ TModeList Video mode list viewer }

 PModeList = ^TModeList;          { Displays list of supported Video Modes  }
 TModeList = object(TListViewer)
  ModeInfo : PModeResArray;       { Array of mode details records           }
  CurMode  : Integer;             { Ordinal pos of current mode in mode list}
  ColorsMin: Longint;             { Minimum acceptable Number of colours    }
  ColorsMax: Longint;             { Maximum acceptable number of colours    }
  ResMin   : TPoint;              { Minimum screen resolution in pixels     }
  ResMax   : TPoint;              { Maximum screen resolution in pixels     }
  constructor Init(var Bounds: TRect; ANumCols: Word; AScrollBar: PScrollBar);
  constructor Load(var S: TStream);
  destructor  Done; virtual;
  procedure GetData(var Rec); virtual;
  procedure GetModeList;
  function  GetText(Item: Integer; MaxLen: Integer): String; virtual;
  function  IsSelected(Item: Integer): Boolean; virtual;
  function  IsValidMode(const Mode: TModeRes): boolean; virtual;
  procedure SetFilters(AColorsMin, AColorsMax: Longint;
                       ResMinX, ResMinY, ResMaxX, ResMaxY: word);
  procedure Store(var S: TStream); virtual;
 end;

{ TVideoModeDlg }

 PVideoModeDlg = ^TVideoModeDlg;
 TVideoModeDlg = object(TDialog)
  ModeList : PModeList;                 { Pointer to video Mode List box    }
  constructor Init;
  constructor Load(var S: TStream);
  procedure HandleEvent(var Event : TEvent); virtual;
  procedure Store(var S: TStream);
 end;

{ Stream Registration }

procedure RegisterModeSlct;

const
  RModeList: TStreamRec = (
     ObjType: 200;
     VmtLink: Ofs(TypeOf(TModeList)^);
     Load   : @TModeList.Load;
     Store  : @TModeList.Store
  );
const
  RVideoModeDlg: TStreamRec = (
     ObjType: 201;
     VmtLink: Ofs(TypeOf(TVideoModeDlg)^);
     Load   : @TVideoModeDlg.Load;
     Store  : @TVideoModeDlg.Store
  );

implementation
uses Gapp, GMsgBox;

{******************************* TModeList *********************************}

constructor TModeList.Init(var Bounds: TRect; ANumCols: Word;
                           AScrollBar: PScrollBar);
begin
  inherited Init(Bounds, ANumCols, nil, AScrollBar);
  ID := 200;
  Options := Options or ofFramed;
  New(ModeInfo);
  SetFilters(256, 256, 640, 400, 1600, 1200);
end;

constructor TModeList.Load(var S: TStream);
begin
  inherited Load(S);
  New(ModeInfo);
  S.Read(ColorsMin, SizeOf(Longint) * 2 + SizeOf(TPoint) * 2);
  GetModeList;
end;

destructor TModeList.Done;
begin
  Dispose(ModeInfo);
  inherited Done;
end;

procedure TModeList.GetModeList;
var
  ModeNum  : word;
  I,C,Index: word;
  Info     : PModeInfo;
  ModeList : PScrnModeList;
begin
  GetScrnModes(C, ModeList);           { Get list of supported video modes  }
  CurMode:= 0;
  Range := 0;
  Index := 0;
  for i := 0 to C-1 do                 { Loop for each mode in the mode list}
   begin
     ModeNum := ModeList^[i];
     Info := GetModeInfo(ModeNum);     { Info @ VESA-Style mode info record }
     if Info <> nil then               { This should never fail             }
      with ModeInfo^[Index] do         { Define the next ModeInfo record    }
      begin
       Mode := ModeNum;                { VESA/IBM Video Mode                }
       Resolution := Info^.ScrnRes;    { Screen Resolution in pixels        }
       ColorDepth := 1 shl Info^.BitsPerPixel;  { Number of colours         }
       if IsValidMode(ModeInfo^[Index]) then    { Filter the mode           }
        begin
          if Mode = ScreenMode         { Check to see if mode is the current}
            then CurMode := Index;     { active video mode                  }
          inc(Index);                  { Mode got through the filter        }
        end;
      end;
   end;                                { Next video mode in the mode list   }
 SetRange(Index);                      { Index = number of filtered modes   }
 FocusItem(CurMode);
end;

function TModeList.GetText(Item: Integer; MaxLen: Integer): String;
var
  S: String[28];
  P: Array[0..3] of Longint;
begin
  with ModeInfo^[Item] do
   begin
    P[0] := Mode;
    P[1] := Resolution.X;
    P[2] := Resolution.Y;
    P[3] := ColorDepth;
   end;
  FormatStr(S, ' %4x   %4d x %-4d  %-8d', P);
  GetText := Copy(S, 1, MaxLen);
end;

procedure TModeList.GetData(var Rec);
begin
  Word(Rec) := ModeInfo^[Focused].Mode;
end;

function TModeList.IsSelected(Item: Integer): Boolean;
begin
{ if (State and (sfActive + sfSelected)) = sfActive + sfSelected }
{  then } IsSelected :=  (Item = CurMode) or
{  else  IsSelected := } (Item = Focused);
end;

procedure TModeList.SetFilters(AColorsMin, AColorsMax: Longint;
                               ResMinX, ResMinY, ResMaxX, ResMaxY: word);
begin
  ColorsMin:= AColorsMin; ColorsMax := AColorsMax;
  ResMin.X := ResMinX; ResMin.Y := ResMinY;
  ResMax.X := ResMaxX; ResMax.Y := ResMaxY;
  GetModeList;
end;

function TModeList.IsValidMode(const Mode: TModeRes): boolean;
begin
  with Mode do
  IsValidMode:=(Resolution.X >= ResMin.X) and (Resolution.X <= ResMax.X) and
               (Resolution.Y >= ResMin.Y) and (Resolution.Y <= ResMax.Y) and
               (ColorDepth >= ColorsMin)  and (ColorDepth <= ColorsMax);
end;

procedure TModeList.Store(var S: TStream);
begin
  inherited Store(S);
  New(ModeInfo);
  S.Write(ColorsMin, SizeOf(Longint) * 2 + SizeOf(TPoint) * 2);
end;


{**************************** TVideoModeDlg *****************************}

constructor TVideoModeDlg.Init;
var
 R : TRect;
 P : PView;

 procedure InsertPanel(x1, y1, x2, y2 : integer);
  begin
   R.Assign(x1,y1,x2,y2);
   P := New(PPanel, Init(R, 2, 0, true));
   P^.SizeAdj.A.Y := -(CharSize.Y shr 2);
   Insert(P);
  end;

begin
  R.Assign(0,0,46,16);
  inherited Init(R, 'Change the Video Mode');
  ID := 201;
  Options := Options or ofCentered;

  InsertPanel(2,2,31,15);

  R.Assign(28,3,30,14);
  P := New(PScrollBar, Init(R, sbVertical+sbFramed+sbUpdateOnTrack));
  R.Assign(3,3,28,14);
  New(ModeList, Init(R, 1, PScrollBar(P)));
  Insert(ModeList);
  Insert(P);
  Dec(R.A.Y); R.B.Y := R.A.Y + 1; Inc(R.B.X);
  Insert(New(PLabel, Init(R, ' Mode   Resolution  Colors', nil)));

  InsertPanel(32, 2, 44, 15);

  R.Assign(33,3,43,5);
  Insert(New(PButton, Init(R, '~O~K', cmOK, bfDefault)));
  R.Move(0,3);
  Insert(New(PButton, Init(R, 'Cancel', cmCancel, bfNormal)));
  R.Move(0,3);
  Insert(New(PButton, Init(R, '~T~est', cmTestVideoMode,
         bfNormal +bfBroadCast)));
  R.Move(0, 3);
  Insert(New(PButton, Init(R, '~H~elp', cmHelp, bfNormal)));
  SelectNext(False);
end;

constructor TVideoModeDlg.Load(var S: TStream);
begin
  inherited Load(S);
  GetSubViewPtr(S, ModeList);
end;

procedure TVideoModeDlg.Store(var S: TStream);
begin
  inherited Store(S);
  PutSubViewPtr(S, ModeList);
end;

procedure TVideoModeDlg.HandleEvent(var Event : TEvent);
const
 PressAny  : string[54] = 'Press any key or click the mouse '+
                          'to restore the screen';
var
 ModeNum : LongRec;

function CentreX(L : integer): integer; near;
begin
  CentreX := ((ScTxtSize.X - L) shr 1) * CharSize.X;
end;

function DoMessage : boolean;
begin
  DoMessage := false;
  ModeList^.GetData(ModeNum.Lo);
  ModeNum.Hi := 0;
  if ModeNum.Lo = ScreenMode then
     MessageBox(#3'Mode $%x is the current video mode', @ModeNum,
                  mfError + mfOKButton)
   else DoMessage:= MessageBox(
              #3+'About to set video mode $%x.'+ #13 + #3+ PressAny,
               @ModeNum, mfInformation + mfOKCancel) = cmOK;
end;

procedure DoTest;
const
 WhiteRGB: TColorRGB = (Red:255; Green:255; Blue:255);
var
  OldMode : Word;
  BarSize : Word;
  i,X,Y   : Integer;
  Colors  : Longint;
  ResY    : Longint;
  ResX    : LongInt;
  Mode    : LongRec;
  PalRGB  : PPalette256RGB;
  S       : String[40];

procedure ColorSquares(Colors: Word; X, Y: integer); near;
var
  Columns: integer;
  Size   : Integer;
  Color  : word;
  c,r    : word;
  Origin : TPoint;
begin
  Origin.X  := X; Origin.Y := Y;
  if Colors = 16
   then Columns := 4
   else Columns := 16;
  Size := 256 div Columns;
  Color:= 0;
  for r := 0 to Columns -1 do
   begin
    for c := 0 to Columns -1 do
     begin
      SetBkColor(Color);
      Inc(Color);
      Bar(X, Y, X + Size, Y + Size);
      Inc(X, Size);
     end;
    X := Origin.X;
    Inc(Y, Size);
   end;
  SetColor(MaxColor);
  X := Origin.X;
  Y := Origin.Y;
  Rectangle(Origin.X, Origin.Y, Origin.X + 256, Origin.Y + 256);
  for r := 0 to Columns -2 do
   begin
    Inc(X, Size);
    Line(X, Y, X, Y+255);
   end;
  X := Origin.X;
  for c := 0 to Columns -2 do
   begin
    Inc(Y, Size);
    Line(X, Y, X+255, Y);
   end;
end;

begin { DoTest }
  if not MonoCard then
   begin
    New(PalRGB);
    if PalRGB = nil then
     begin
       MessageBox('Not enough memory for this operation', nil,
                  mfError + mfOKButton);
       exit;
     end;
   end;
  GetRGBPalette(0, MaxColor+1, PalRGB);   { Save the current VGA palette    }
  OldMode := ScreenMode;
  SetVideoMode(ModeNum.Lo);
  if ModeNum.Lo <> ScreenMode
   then MessageBox(#3'Unable to set the requested video Mode ', nil,
                   mfOkButton + mfError) else
   begin
    SetRGBPalette(MaxColor, 1, @WhiteRGB);  { For when there is no mouse      }
    SetColor(MaxColor);
    HideMouse;
    Rectangle(0,0, ScrnRes.X-1, ScrnRes.Y-1);
    if not MonoCard
     then ColorSquares(16, 16, 16);
    if MaxColor = 255
     then ColorSquares(256, ScrnRes.X-272, 16);
    SetColor(MaxColor);
    ResX := ScrnRes.X; ResY := ScrnRes.Y;
    Longint(Mode) := ScreenMode; Colors := MaxColor + 1;
    FormatStr(S,'Mode %-4x (%d x %d :- %-3d colours)', Mode);
    Y := ScrnRes.Y - (CharSize.Y shl 2);
    OutTextXY(CentreX(Length(S)), Y, S);
    Inc(Y, CharSize.Y shl 1);
    OutTextXY(CentreX(Length(PressAny)), Y, PressAny);
    ShowMouse;
    repeat
     GetMouseEvent(Event);
     if Event.What <> evMouseDown
      then GetKeyEvent(Event);
    until Event.What and (evMouseDown + evKeyDown) <> 0;
    Application^.SetScreenMode(OldMode);
   end;
  if not MonoCard then
   begin
    SetRGBPalette(0, MaxColor+1, PalRGB);   { Restore the old VGA palette   }
    Dispose(PalRGB);
   end;
end;

begin
  inherited HandleEvent(Event);
  if (Event.What = evBroadCast) AND (Event.Command = cmTestVideoMode) then
   begin
    if DoMessage
     then DoTest;
    ClearEvent(Event);
   end;
end;

procedure RegisterModeSlct;
begin
  RegisterType(RModeList);
  RegisterType(RVideoModeDlg);
end;

end.
