{***************************************************}
{                                                   }
{   Graphic Vision Demo                             }
{   Copyright (c) 1992 by Borland International and }
{   Copyright (c) 1996 Jason G Burgon               }
{                                                   }
{***************************************************}

program GVDemo;

{$M 16384,102400,655360}
{$X+,S-,I-,G+}
{$S 65535}
{$IFNDEF DPMI}
{$F+}
{$ENDIF}

{ Graphic Vision demo program. This has all the functionality of the TVDemo
  program that comes with BP7, and has the additional enhancements:

  The Window|CloseAll command is enabled, and the CloseAll method added to
  close all the windows on the desktop (except the ClipBoard window).  This
  is made possible by the better TGroup.ForEach procedure.

  The Mouse dialog allows much more comprehensive mouse control.

  The video mode can be changed via the Options|Video Mode menu, using the
  objects from the ModeSlct unit.

  The TCalendarView, TImageBackground and About dialog show how easy it is to
  use bitmap views from the BitMaps unit. The calendar uses two MS Windows
  compatible Device Independant Bitmap files (NAUGHTY.BMP and NICE.BMP). The
  calendars also show Graphic Vision's hardware palette management system in
  action. You can see this working when you have at least one "naughty" and
  one "nice" calendar displayed.

  This program also uses many of the Graphic Vision standard and demo units,
  including:

    GStdDlg   - Open file browser, change directory tree.
    GMsgBox   - Simple dialog to display messages.
    GColors   - Color customization.
    Gadgets   - Shows system time and available heap space.
    AsciiTab  - ASCII table.
    Calendar  - View a month at a time, choice of two pictures
    GCalc     - Desktop calculator.
    HelpFile  - Context sensitive help.
    MouseDlg  - Comprehensive mouse control dialog.
    ModeSlct  - Video mode change dialog.
    Puzzle    - Simple brain puzzle.
    GEditors  - Text Editor object.

  And of course this program includes many standard Graphic Vision
  objects and behaviors (menubar, desktop, status line, dialog boxes,
  mouse support, window resize/move/tile/cascade).
}

uses {$IFNDEF DPMI} Overlay, {$ENDIF}
  Dos, GObjects, ScrnDriv, Font8x16, GDrivers, GMemory, GViews, GWindows,
  GMenus, BitMaps, GDialogs, GStdDlg, GHistLst, GMsgBox, GApp, GColors,
  MouseDlg, ModeSlct, DemoCmds, Gadgets, Puzzle, Calendar, AsciiTab, GCalc,
  HelpFile, DemoHelp, GEditors {$IFDEF TESTPAL} ,PalView {$ENDIF};

{ If you get a FILE NOT FOUND error when compiling this program from a
  DOS IDE, change to the \BP\GV\DEMOS directory (use File|Change dir),
  and don't forget to add \BP\GV\UNITS; in front of \BP\UNITS in the Units
  directory (use Options|Directories).
}

{$G Font8x16}           { Always put linked-in fonts into their own segment }
{$G GObjects, GDrivers} { These two units are Fixed, Preload, Permanent     }
{$G ScrnDriv }          { Put ScrnDriv in its own segment for speed.        }
{$G GViews, GWindows, GMenus, Gapp, Gadgets, GVDemo}  { Group "hot" units.  }
{$O GDialogs } {$G GDialogs }
{$O GStdDlg  } {$G GStdDlg  }
{$O GMsgBox  } {$G GMsgBox  }
{$O GCalc    } {$G GCalc    }
{$O Calendar } {$G Calendar }
{$O GEditors } {$G GEditors }
{$O GColors  } {$G GColors  }
{$O MouseDlg } {$G MouseDlg }
{$O HelpFile } {$G HelpFile }
{$O ModeSlct } {$G ModeSlct }

const
  ColorsHistoryID = 1;                  { "Load colour palette" history ID  }
  HeapSize = 100 * 1024 div 16;         { Save 100k for normal heap memory. }

  { Desktop file signature information }

  SignatureLen = 21;
  DSKSignature : string[SignatureLen] = 'GV Demo Desktop File'#26;

  { Desktop image filename }

  DesktopImageFile = 'GVDSKTOP.BMP';

var
  ClipWindow: PEditWindow;

type

 PImageBackground = ^TImageBackground;
 TImageBackground = object(TBitMapViewDIB)
   constructor Init(Bounds: TRect; const AFileName: PathStr);
 end;

 PNewDesktop = ^TNewDesktop;
 TNewDesktop = object(TDesktop)
   procedure HandleEvent(var Event: TEvent); virtual;
   procedure InitBackground; virtual;
end;

 { TGVDemo }

  PGVDemo = ^TGVDemo;
  TGVDemo = object(TApplication)
    Clock: PClockView;
    Heap : PHeapView;
{$IFDEF TESTPAL}
    Palette: PPaletteWindow;
{$ENDIF}
    constructor Init;
    function  CloseAll: boolean;
    procedure FileOpen(WildCard: PathStr);
    function  OpenEditor(FileName: FNameStr; Visible: Boolean;
                         Number: Word): PEditWindow;
    procedure GetEvent(var Event: TEvent); virtual;
    function  GetPalette: PPalette; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Idle; virtual;
    procedure InitDesktop; virtual;
    procedure InitMenuBar; virtual;
    procedure InitStatusLine; virtual;
    procedure LoadDesktop(var S: TStream);
    procedure OutOfMemory; virtual;
    procedure StoreDesktop(var S: TStream);
  end;

{ TImageBackground }

constructor TImageBackground.Init(Bounds: TRect; const AFileName: PathStr);
var
  Adj: TRect;
begin
  Adj := ZR;
  inherited Init(Bounds, Adj, bmcClipCentre + bmcTileImages, bmoCacheImage,
                              bmpRoundToDac + bmpLoadPalette +
                              bmpRemoveDups + bmpSavePalette,
                              bmuSaveImage, AFileName);
  GrowMode := gfGrowHiX + gfGrowHiY;
  EventMask := 0;
end;

{ TNewDeskTop }

procedure TNewDeskTop.HandleEvent(var Event: TEvent);

function ExecPopUp (At:TPoint; PopupMenu:PMenuPopup): Integer;
var
  Bounds : TRect;              { shows the "PopupMenu" at the position "At" }
  Adj    : Integer;
begin
  ExecPopUp := cmCancel;
  if Application^.ValidView(PopupMenu) <> nil then
  begin
    Application^.MakeLocal(At, At);
    Inc(At.Y);
    Application^.GetBounds(Bounds);
    Adj := Bounds.B.Y - PopupMenu^.Size.Y - At.Y;
    if Adj < 0
      then Inc(At.Y, Adj);
    Adj := Bounds.B.X - PopupMenu^.Size.X - At.X;
    if Adj < 0
      then Inc(At.X, Adj);
    PopupMenu^.Origin := At;
    ExecPopUp := Application^.ExecView(PopupMenu);
  end;
end;

var
  Menu : PMenu;
  PopUp: PMenuPopUp;
  R    : TRect;
  Res  : Word;
begin
  if (Event.What = evMouseDown) and (Event.Buttons = mbRightButton) and
     (TypeOf(DeskTop^.Current^) = TypeOf(TEditWindow)) then
   begin
     Longint(R.A) := 0;
     LongInt(R.B) := 0;
     Menu := NewMenu(
       StdEditMenuItems(
       NewLine(
       NewItem('~S~how clipboard', '', kbNoKey, cmShowClip, hcShowClip,
       nil))));
     New(PopUp, Init(R, Menu));
     Res := ExecPopUp(Event.Where, PopUp);
     DisposeMenu(Menu);
     Dispose(PopUp, Done);
     ClearEvent(Event);
   end
   else inherited HandleEvent(Event);
end;

procedure TNewDeskTop.InitBackground;
var
  R: TRect;
begin
  GetExtent(R);
  Background := PBackground(New(PImageBackground,
                                Init(R, {ExePath+}DesktopImageFile)));
end;

{ TGVDemo }

constructor TGVDemo.Init;
var
  R: TRect;
  Box, RowCol: TPoint;
  I: Integer;
  Event: TEvent;
  FileName: PathStr;
begin
  DialogInterior := BorDlgFill;
  ScreenMode := $101;       { All VGA monitors will support 640 x 480 x 256 }
  RegisterFont(@Sys_8x16);  { Use the linked-in internal font               }
{$IFNDEF DPMI}              { Allocate all available memory to editor buffrs}
  MaxHeapSize := HeapSize;  { except small amount for normal heap allocation}
{$ENDIF}                    { and the critical error handler.               }
  inherited Init;
  if MemAvail div 16 < LowMemSize then
   begin
      inherited Done;
      PrintStr('Not enough memory to run the application'#13+
               'Program terminated'#13#13);
      Halt(1);
    end;
  RegisterObjects;
  RegisterViews;            { RegisterViews only registers TView & TGroup,  }
  RegisterWindows;          { so you must call RegisterWindows as well.     }
  RegisterMenus;
  RegisterDialogs;
  RegisterApp;
  RegisterHelpFile;
  RegisterPuzzle;
  RegisterCalendar;
  RegisterAsciiTab;
  RegisterCalc;
  RegisterEditors;
  RegisterBitMaps;

  { Initialize demo gadgets }

  GetExtent(R);
  R.B.Y := R.A.Y + 1;
  Clock := New(PClockView, Init(R));
  Insert(Clock);

  GetExtent(R);
  Dec(R.B.X);
  R.A.X := R.B.X - 9; R.A.Y := R.B.Y - 1;
  Heap := New(PHeapView, Init(R));
  Insert(Heap);
{$IFDEF TESTPAL}
  Box.X := 10;
  Box.Y := 10;
  RowCol.X := 32;
  RowCol.Y := 8;
  Palette := New(PPaletteWindow, Init(R, Box, RowCol));
  Insert(Palette);
{$ENDIF}
  DisableCommands([cmSave, cmSaveAs, cmCut, cmCopy, cmPaste, cmClear,
    cmUndo, cmFind, cmReplace, cmSearchAgain]);
  EditorDialog := StdEditorDialog;
  ClipWindow := OpenEditor('', False, wnNoNumber);
  if ClipWindow <> nil then
  begin
    Clipboard := ClipWindow^.Editor;
    Clipboard^.CanUndo := False;
  end;

  if ParamCount > 0 then
   for I := 1 to ParamCount do
    begin
      FileName := ParamStr(I);
      if FileName[Length(FileName)] = '\'
        then FileName := FileName + '*.*';
      if (Pos('?', FileName) = 0) and (Pos('*', FileName) = 0)
        then OpenEditor(FExpand(FileName), True, wnNoNumber+1)
        else FileOpen(FileName);
    end else
    begin
      Event.What := evCommand;
      Event.Command := cmAbout;
      PutEvent(Event);
    end;
end;

procedure TGVDemo.InitDeskTop;
var
  R: TRect;
begin
  GetExtent(R);
  Inc(R.A.Y);
  Dec(R.B.Y);
  DeskTop := New(PNewDesktop, Init(R));
end;

function TGVDemo.OpenEditor(FileName: FNameStr; Visible: Boolean;
                            Number: Word): PEditWindow;
var
  P: PWindow;
  R: TRect;
begin
  DeskTop^.GetExtent(R);
  P := New(PEditWindow, Init(R, FileName, Number));
  if Assigned(P) then
   begin
    if not Visible
     then P^.Hide;
    P^.HelpCtx := hcViewer;
   end;
  OpenEditor := PEditWindow(InsertWindow(P));
end;

function TGVDemo.CloseAll: boolean;    { Graphic Vision can close all the   }
                                       { windows on the desktop (unlike     }
 procedure CloseView(P: PView); far;   { Turbo Vision) because of the better}
 begin                                 { TGroup.ForEach function            }
   Message(P, evCommand, cmClose, nil);
 end;

begin { TGVDemo.CloseAll }
 if Desktop^.Valid(cmClose) then
  begin
   DeskTop^.ForEach(@CloseView);
   CloseAll := true;
  end else
   CloseAll := false;
end;

procedure TGVDemo.FileOpen(WildCard: PathStr);
var
  FileName: FNameStr;
begin
  FileName := '*.*';
  if ExecuteDialog(New(PFileDialog, Init(WildCard, 'Open a file',
    '~N~ame', fdOpenButton + fdHelpButton, 100)), @FileName) <> cmCancel
    then OpenEditor(FileName, True, wnNoNumber+1);
end;

procedure TGVDemo.GetEvent(var Event: TEvent);
var
  W: PWindow;
  HFile: PHelpFile;
  HelpStrm: PDosStream;
const
  HelpInUse: Boolean = False;
begin
  inherited GetEvent(Event);
  case Event.What of
    evCommand:
      if (Event.Command = cmHelp) and not HelpInUse then
      begin
        HelpInUse := True;
        HelpStrm := New(PDosStream, Init(ExePath + 'GVDEMO.HLP', stOpenRead));
        HFile := New(PHelpFile, Init(HelpStrm));
        if HelpStrm^.Status <> stOk then
        begin
          MessageBox('Could not open help file.', nil, mfError + mfOkButton);
          Dispose(HFile, Done);
        end
        else
        begin
          W := New(PHelpWindow,Init(HFile, GetHelpCtx));
          ExecuteDialog(PDialog(W), nil);
          ClearEvent(Event);
        end;
        HelpInUse := False;
      end;
  end;
end;

function TGVDemo.GetPalette: PPalette;
const
  CNewColor = CColor + CHelpColor;
  CNewBlackWhite = CBlackWhite + CHelpBlackWhite;
  CNewMonochrome = CMonochrome + CHelpMonochrome;
  P: array[apColor..apMonochrome] of string[Length(CNewColor)] =
    (CNewColor, CNewBlackWhite, CNewMonochrome);
begin
  GetPalette := @P[AppPalette];
end;

procedure TGVDemo.HandleEvent(var Event: TEvent);

procedure ChangeDir;
var
  D: PChDirDialog;
begin
  D := New(PChDirDialog, Init(cdNormal + cdHelpButton, 101));
  D^.HelpCtx := hcFCChDirDBox;
  ExecuteDialog(D, nil);
end;

procedure Puzzle;
var
  P: PPuzzleWindow;
begin
  P := New(PPuzzleWindow, Init);
  P^.HelpCtx := hcPuzzle;
  InsertWindow(P);
end;

procedure Calendar;
var
  P: PCalendarWindow;
begin
  P := New(PCalendarWindow, Init);
  P^.HelpCtx := hcCalendar;
  if InsertWindow(P) = nil
    then MessageBox('Not enough memory for this operation', nil,
                    mfError + mfOKButton);
end;

procedure About;
var
  D: PDialog;
  Control: PView;
  R: TRect;
begin
  R.Assign(0, 0, 45, 13);
  D := New(PDialog, Init(R, 'About'));
  with D^ do
  begin
    Options := Options or ofCentered;
    Inc(SizeAdj.B.Y, CharSize.Y shr 1);          { Move the bottom of the   }
    Inc(Frame^.SizeAdj.B.Y, CharSize.Y shr 1);   { dialog box up by half a  }
    R.Grow(-2, -2);                              { character                }
    Dec(R.B.Y, 1);
    Control := New(PPanel, Init(R, 2, 0, true)); { Insert a sunken panel    }
    with Control^.SizeAdj do
     begin
       Dec(A.Y, CharSize.Y shr 1);               { adjust its (pixel) size  }
       Inc(B.Y, CharSize.Y shr 2);               { so it looks pretty       }
      end;
    Insert(Control);
    R.Move(1, 0);
    Control := New(PBitMapViewDIB, Init(R, ZR, 0, 0,
                                        bmpLoadPalette + bmpSavePalette +
                                        bmpNoUseSys + bmpRemoveDups +
                                        bmpRoundToDac, 0,
                                        ExePath + 'ABOUT.BMP'));
    if not Control^.Valid(cmValid) then
     begin
       Dispose(Control, Done);
       Dec(R.B.X, 2); Dec(R.B.Y);
       Insert(New(PStaticText, Init(R,           { Insert the copyright     }
      ^C'~Unicorn Software Presents~'#13#13 +
      ^C'Graphic Vision'#13 +                    { static text message view }
      ^C'Copyright (c) 1995'#13  +               { just in case the bitmap  }
      ^C'Jason G Burgon'#13#13 +                 { version fails to load.   }
      ^C'gvision@jayman.demon.co.uk')));
     end else Insert(Control);
    R.Assign(17, 10, 27, 12);                     { Insert an OK button      }
    Control := New(PButton, Init(R, 'O~K~', cmOk, bfDefault));
    with Control^.SizeAdj do
     begin
       Inc(A.Y, 2);                              { adjust its (pixel) size  }
       Dec(B.Y, 2);                              { so it looks pretty       }
      end;
    Insert(Control);
  end;
  ExecuteDialog(D, nil);
end;

procedure AsciiTab;
var
  P: PAsciiChart;
begin
  P := New(PAsciiChart, Init);
  P^.HelpCtx := hcAsciiTable;
  InsertWindow(P);
end;

procedure Calculator;
var
  P: PCalculator;
begin
  P := New(PCalculator, Init);
  P^.HelpCtx := hcCalculator;
  InsertWindow(P);
end;

procedure Colors;
var
  D        : PColorDialog;
  AppPal   : PPalette;
  OldAppPal: TPalette;
  OldRGBPal: TPalette16RGB;
begin
  AppPal    := Application^.GetPalette;     { Indexes Application's Palette }
  OldAppPal := AppPal^;                     { Save the Current App Palette  }
  GetRGBPalette(0, 16, @OldRGBPal);         { Save the current RGB Palette  }
  D := New(PColorDialog, Init(AppPal,
    ColorGroup('Desktop', DesktopColorItems(nil),
    ColorGroup('Menu/Statusline',MenuColorItems(nil),
    ColorGroup('Dialogs & Calc', DialogColorItems(nil),
    ColorGroup('File Editor',    WindowColorItems(wpBlueWindow, nil),
    ColorGroup('Calendar',       CalendarColorItems(wpCyanWindow),
    ColorGroup('Ascii table',    WindowColorItems(wpGrayWindow, nil),
    ColorGroup('Help Window',    HelpWindowColorItems(nil),
    nil))))))), ColorsHistoryID));

  D^.HelpCtx := hcOCColorsDBox;
  if ExecuteDialog(D, @AppPal) = cmCancel then
    begin
      AppPal^ := OldAppPal;                 { Restore previous App Palette  }
      SetRGBPalette(0, 16, @OldRGBPal);     { Restore Video Card Palette    }
      ReDraw;                               { Redraw app with old palette   }
    end else
    begin
      GetRGBPalette(0, 16, @OldRGBPal);     { Read updated VGA colour regs. }
      ChangeSysColors(@OldRGBPal);          { Inform the palette management }
    end;                                    { System of the changes.        }
end;

procedure Mouse;
var
  D: PDialog;
begin
  D := New(PMouseDialog, Init);
  D^.HelpCtx := hcOMMouseDBox;
  ExecuteDialog(D, nil);
end;

procedure RetrieveDesktop;
var
  Mode: word;
  S: TBufStream;
  AppPal: TPalette;
  VgaPal: TPalette16RGB;
  Signature: string[SignatureLen];
begin
  S.Init('GVDEMO.DSK', stOpenRead, 1024);
  if LowMemory
    then OutOfMemory
    else if S.Status <> stOk then
      MessageBox('Could not open desktop file', nil, mfOkButton + mfError)
  else
  begin
    Signature[0] := Char(SignatureLen);
    S.Read(Signature[1], SignatureLen);
    if Signature = DSKSignature then
    begin
      S.Read(Mode, SizeOf(word));
      SetScreenMode(Mode);
      if ReadPalette(S, @AppPal, @VgaPal) = stOk then
       begin
         GetPalette^ := AppPal;
         LoadDesktop(S);
         LoadIndexes(S);
         LoadHistory(S);
         if S.Status <> stOk then
          MessageBox('Error reading desktop file', nil, mfOkButton + mfError);
       end;
    end else
      MessageBox('Error: Invalid Desktop file.', nil, mfOkButton + mfError);
  end;
  S.Done;
end;

procedure SaveDesktop;
var
  S: TBufStream;
  F: File;
begin
  S.Init('GVDEMO.DSK', stCreate, 1024);
  if not LowMemory and (S.Status = stOk) then
  begin
    S.Write(DSKSignature[1], SignatureLen);
    S.Write(ScreenMode, SizeOf(Word));
    StorePalette(S, Application^.GetPalette);
    StoreDesktop(S);
    StoreIndexes(S);
    StoreHistory(S);
    if S.Status <> stOk then
    begin
      MessageBox('Could not create GVDEMO.DSK.', nil, mfOkButton + mfError);
      {$I-}
      S.Done;
      Assign(F, 'GVDEMO.DSK');
      Erase(F);
      Exit;
    end;
  end;
  S.Done;
end;

procedure ChangeMode;
var
  OldRGBPalette: PPalette256RGB;
  OldMaxColor  : Word;
  OldMode      : Word;
  Mode         : Word;
  ModePtr      : PWord;
  P            : PDialog;
begin
  ModePtr := @Mode;
  OldMode := ScreenMode;               { Save the current Video Mode        }
  P := New(PVideoModeDlg, Init);
  P^.HelpCtx := hcOVideoModeDBox;
  if (ExecuteDialog(New(PVideoModeDlg, Init), ModePtr) <> cmCancel) and
     (Mode <> OldMode) then
    begin                              { A new video mode has been selected }
      OldMaxColor := GetMaxColor;
      if OldMaxColor > 2 then          { If the current mode has got a      }
        begin                          { hadware palette then save it       }
          New(OldRGBPalette);
          GetRGBPalette(0, OldMaxColor+1, OldRGBPalette);
        end else OldRGBPalette := nil;
      SetScreenMode(Mode);
      if (OldRGBPalette <> nil) and (GetMaxColor = OldMaxColor){ restore    }
       then SetRGBPalette(0, OldMaxColor+1, OldRGBPalette);    { the palette}
      if OldRGBPalette <> nil
       then Dispose(OldRGBPalette);
    end;
end;

{$IFDEF DPMI}
procedure FastPageTest;
const
  sFastPageTest: String[28] = 'Fast Page Switch Support: %s';
  sYes : String[3] = 'Yes';
  sNo  : String[2] = 'No';
var
  P    : PString;
begin
  P := @sYes;
  if ScrnDriv.SwitchTest
   then P := @sNo;
  MessageBox(sFastPageTest, @P, mfInformation + mfOKButton);
end;
{$ENDIF}

procedure FileNew;
begin
  OpenEditor('', True, wnNoNumber+1);
end;

procedure ShowClip;
begin
  ClipWindow^.Select;
  ClipWindow^.Show;
end;

begin
  inherited HandleEvent(Event);
  case Event.What of
    evCommand:
      begin
        case Event.Command of
          cmOpen: FileOpen('*.*');
          cmNew: FileNew;
          cmCloseAll: CloseAll;
          cmShowClip: ShowClip;
          cmChangeDir: ChangeDir;
          cmAbout: About;
          cmPuzzle: Puzzle;
          cmCalendar: Calendar;
          cmAsciiTab: AsciiTab;
          cmCalculator: Calculator;
          cmColors: Colors;
          cmMouse: Mouse;
          cmSaveDesktop: SaveDesktop;
          cmRetrieveDesktop: RetrieveDesktop;
          cmChangeMode: ChangeMode;
          {$IFDEF DPMI}
          cmFastPageTest: FastPageTest
          {$ENDIF}
        else
          Exit;
        end;
        ClearEvent(Event);
      end;
  end;
end;

procedure TGVDemo.Idle;

function IsTileable(P: PView): Boolean; far;
begin
  IsTileable := (P^.Options and ofTileable <> 0) and
    (P^.State and sfVisible <> 0);
end;

begin
  inherited Idle;
  Clock^.Update;
  Heap^.Update;
{$IFDEF TESTPAL}
  Palette^.PalView^.Update;
{$ENDIF}
  if Desktop^.FirstThat(@IsTileable) <> nil then
    EnableCommands([cmTile, cmCascade])
  else
    DisableCommands([cmTile, cmCascade]);
end;

procedure TGVDemo.InitMenuBar;
var
  R: TRect;
begin
  GetExtent(R);
  R.B.Y := R.A.Y+1;
  MenuBar := New(PMenuBar, Init(R, NewMenu(
    NewSubMenu('~'#240'~', hcSystem, NewMenu(
      NewItem('~A~bout', '', kbNoKey, cmAbout, hcSAbout,
      NewLine(
      NewItem('~P~uzzle', '', kbNoKey, cmPuzzle, hcSPuzzle,
      NewItem('Ca~l~endar', '', kbNoKey, cmCalendar, hcSCalendar,
      NewItem('Ascii ~t~able', '', kbNoKey, cmAsciiTab, hcSAsciiTable,
      NewItem('~C~alculator', '', kbNoKey, cmCalculator, hcCalculator, nil))))))),
    NewSubMenu('~F~ile', hcFile, NewMenu(
      StdFileMenuItems(nil)),
    NewSubMenu('~E~dit', hcEdit, NewMenu(
      StdEditMenuItems(
      NewLine(
      NewItem('~S~how clipboard', '', kbNoKey, cmShowClip, hcShowClip,
      nil)))),
    NewSubMenu('~S~earch', hcSearch, NewMenu(
      NewItem('~F~ind...', '', kbNoKey, cmFind, hcFind,
      NewItem('~R~eplace...', '', kbNoKey, cmReplace, hcReplace,
      NewItem('~S~earch again', '', kbNoKey, cmSearchAgain, hcSearchAgain,
      nil)))),
    NewSubMenu('~W~indow', hcWindows, NewMenu(
      StdWindowMenuItems(nil)),
    NewSubMenu('~O~ptions', hcOptions, NewMenu(
      NewItem('~M~ouse...', '', kbNoKey, cmMouse, hcOMouse,
      NewItem('~C~olors...', '', kbNoKey, cmColors, hcOColors,
      NewItem('~V~ideo Mode...','',kbNoKey, cmChangeMode, hcNoContext,
      NewLine(
      NewItem('~S~ave desktop', '', kbNoKey, cmSaveDesktop, hcOSaveDesktop,
      NewItem('~R~etrieve desktop', '', kbNoKey, cmRetrieveDesktop, hcORestoreDesktop,
      {$IFDEF DPMI}
      NewLine(
      NewItem('~F~ast Switch Test...', '', kbNoKey, cmFastPageTest, hcNoContext,
      nil)) {$ELSE} nil {$ENDIF} ))))))),
    nil)))))))));
end;

procedure TGVDemo.InitStatusLine;
var
  R: TRect;
begin
  GetExtent(R);
  R.A.Y := R.B.Y - 1;
  StatusLine := New(PStatusLine, Init(R,
    NewStatusDef(0, $FFFF,
      NewStatusKey('~Alt-X~ Exit', kbAltX, cmQuit,
      NewStatusKey('~F1~ Help', kbF1, cmHelp,
      NewStatusKey('~F3~ Open', kbF3, cmOpen,
      NewStatusKey('~Alt-F3~ Close', kbAltF3, cmClose,
      NewStatusKey('~F5~ Zoom', kbF5, cmZoom,
      NewStatusKey('', kbF2, cmSave,
      NewStatusKey('', kbF6, cmNext,
      NewStatusKey('', kbShiftF6, cmPrev,
      NewStatusKey('', kbF10, cmMenu,
      NewStatusKey('', kbCtrlF5, cmResize,
      nil)))))))))),
    nil)));
end;

procedure TGVDemo.OutOfMemory;
begin
  MessageBox('Not enough memory available to complete operation.',
    nil, mfError + mfOkButton);
end;

{ Since the safety pool is only large enough to guarantee that allocating
  a window will not run out of memory, loading the entire desktop without
  checking LowMemory could cause a heap error.  This means that each
  window should be read individually, instead of using Desktop's Load.
}

procedure TGVDemo.LoadDesktop(var S: TStream);
var
  P,T   : PView;

 procedure CloseView(P: PView); far;
 begin
   Message(P, evCommand, cmClose, nil);
 end;

begin
  if Desktop^.Valid(cmClose) then
   with Desktop^ do
     begin
       ForEach(@CloseView); { Clear the desktop }
       repeat
         P := PView(S.Get);
         Insert(ValidView(P));
       until P = nil;
     end;
end;

procedure TGVDemo.StoreDesktop(var S: TStream);
var
  P: PView;
begin
  with Desktop^ do
   begin
     P := Last;
     repeat
       P := P^.PrevView;
       if (P = nil) or (P <> PView(ClipWindow))
         then S.Put(P);
     until P = nil;
   end;
end;

{$IFNDEF DPMI}
procedure OvrMessage;
const
  OvrMessage : array[ovrNoEMSMemory..ovrError] of String[53] = (
  'No EMS available for Overlays',
  'No EMS driver for Overlays',
  'Overlay file I/O error',
  'Not enough memory for overlay buffer',
  'Overlay file not found in .EXE file or .EXE directory',
  'Overlay manager error');
begin
  if (OvrResult > ovrError) or (OvrResult < ovrNoEMSMemory)
    then OvrResult := ovrError;
  PrintStr(OvrMessage[ovrResult] + #13#10);
  PrintStr('Unable to run program' + #13#10);
  Halt(1);
end;
{$ENDIF}

var
  Demo: TGVDemo;

begin
{$IFNDEF DPMI}
  OvrInit(ExePath + 'GVDEMO.EXE');     { initialize overlay system          }
  if OvrResult <> ovrOk then
    OvrInit(ExePath + 'GVDEMO.OVR');   { initialize overlay system          }
  if OvrResult = ovrOk then
   begin
     OvrInitEMS;
     OvrSetBuf(OvrGetBuf  + (1024 * 12));
     OvrSetRetry(OvrGetBuf div 3);
   end else OvrMessage;
{$ELSE}
{$IFDEF DEBUG}
  HeapLimit := 0;
{$ENDIF}
{$ENDIF}
  Demo.Init;
  Demo.Run;
  Demo.Done;
end.
