{************************************************}
{                                                }
{   Turbo Pascal 6.0                             }
{   Graphic Vision Demo                          }
{   Copyright (c) 1995 Jason G Burgon            }
{                                                }
{************************************************}

unit Calendar;

{$S-,X+,I-}
{$IFNDEF DPMI}
{$F+,O+}
{$ENDIF}
{$IFNDEF DEBUG}
{$L-,D-}
{$ENDIF}
{
  Calendar object for viewing a month at a time. See TVDEMO.PAS
  for an example program that uses this unit.
}

interface

uses Dos, GObjects, ScrnDriv, GDrivers, GViews, GWindows, GMenus, BitMaps,
     GDialogs, GApp, GColors;

const
   cmToggleCalendar = 1500;

   CCalendarView = #5#4;

   DaysInMonth: array[1..12] of Byte =
     (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

   MonthStr: array[1..12] of string[10] =
     ('January   ',
      'February  ',
      'March     ',
      'April     ',
      'May       ',
      'June      ',
      'July      ',
      'August    ',
      'September ',
      'October   ',
      'November  ',
      'December  ');

type

 PCalendarView = ^TCalendarView;
 TCalendarView = object(TBitMapViewDIB)
   Year, Month, Days: Word;
   CurYear, CurMonth, CurDay : Word;
   constructor Init(var Bounds: TRect; const AFileName: PathStr);
   constructor Load(var S: TStream);
   function  GetPalette: PPalette; virtual;
   procedure HandleEvent(var Event: TEvent); virtual;
   procedure Draw; virtual;
   procedure Store(var S: TStream);
 end;

 TPicture = (Nice, Naughty);

 PCalendarWindow = ^TCalendarWindow;
 TCalendarWindow = object(TWindow)
   Picture : TPicture;
   Calendar: PCalendarView;
   constructor Init;
   constructor Load(var S: TStream);
   function  GetCloseMenu: PMenu; virtual;
   procedure HandleEvent(var Event: TEvent); virtual;
   procedure SizeLimits(var Min, Max: TPoint); virtual;
   procedure Store(var S: TStream);
 end;

const
  PicFileName: array[TPicture] of String[11] = (
   'NICE.BMP', 'NAUGHTY.BMP');
  PicMenuName: array[TPicture] of TMenuStr = (
   'Naughty ~P~icture',
   'Nice ~P~icture');
const

  RCalendarView: TStreamRec = (
     ObjType: 10020;
     VmtLink: Ofs(TypeOf(TCalendarView)^);
     Load:    @TCalendarView.Load;
     Store:   @TCalendarView.Store
  );
  RCalendarWindow: TStreamRec = (
     ObjType: 10021;
     VmtLink: Ofs(TypeOf(TCalendarWindow)^);
     Load:    @TCalendarWindow.Load;
     Store:   @TCalendarWindow.Store
  );

function CalendarColorItems(WindowPalette: word): PColorItem;
procedure RegisterCalendar;

implementation

{******************************* TCalendarWindow ***************************}

constructor TCalendarWindow.Init;
var
  Max: TPoint;
  Min: TPoint;
  R: TRect absolute Min;
begin
  Longint(R.A) := 0;
  R.B.X := ScTxtSize.X;
  R.B.Y := ScTxtSize.Y-2;
  inherited Init(R, 'Calendar', 0);
  ID := 10021;
  Flags := Flags and not (wfZoom + wfGrow);    { Not resizeable }
  GrowMode :=0;
  Palette := wpCyanWindow;
  Inc(R.A.Y);                          { Take account of the titlebar.      }
  Calendar := New(PCalendarView, Init(R, ExePath + PicFileName[Picture]));
  Calendar^.GetExtent(R);              { Get extent of calendar view and add}
  Inc(R.B.Y);                          { 1 cell for the frame.              }
  R.Move(Origin.X, Origin.Y);          { to the current desktop position.   }
  Locate(R);
  Insert(Calendar);
end;

constructor TCalendarWindow.Load(var S: TStream);
begin
  inherited Load(S);
  GetSubViewPtr(S, Calendar);
  S.Read(Picture, SizeOf(TPicture));
end;

function TCalendarWindow.GetCloseMenu: PMenu;
var
  M: PMenu;
  I: PMenuItem;
  P: TPicture;
begin
  M := inherited GetCloseMenu;
  I := M^.Items;
  while I^.Next <> nil do
   I := I^.Next;
  I^.Next :=
    NewLine(
    NewItem(PicMenuName[Picture], '', kbNoKey, cmToggleCalendar, hcNoContext,
    nil));
  GetCloseMenu := M;
end;

procedure TCalendarWindow.HandleEvent(var Event: TEvent);
var
  R : TRect;
begin
  inherited HandleEvent(Event);
  if (Event.What = evCommand) and (Event.Command = cmToggleCalendar) then
   begin
     Lock;
     Byte(Picture) := Byte(Picture) xor 1;
     Dispose(Calendar, Done);
     Owner^.GetExtent(R);              { Get Desktop extent and add 1 cell  }
     Inc(R.A.Y);                       { account for the titlebar.          }
     Calendar := New(PCalendarView, Init(R, ExePath + PicFileName[Picture]));
     Calendar^.GetExtent(R);           { Get extent of calendar view and add}
     Inc(R.B.Y);                       { 1 cell for the frame.              }
     R.Move(Origin.X, Origin.Y);       { to the current desktop position.   }
     Locate(R);
     Insert(Calendar);
     UnLock;
   end
end;

procedure TCalendarWindow.SizeLimits(var Min, Max: TPoint);
var
  SelfMin, SelfMax: TPoint;
begin
  Calendar^.SizeLimits(Min, Max);
  Inc(Max.Y);
  inherited SizeLimits(SelfMin, SelfMax);
  if SelfMin.X > Min.X
    then Min.X := SelfMin.X;
  if SelfMin.Y > Min.Y
    then Min.Y := SelfMin.Y;
  if SelfMax.X < Max.X
    then Max.X := SelfMax.X;
  if SelfMax.Y < Max.Y
    then Max.Y := SelfMax.Y;
end;

procedure TCalendarWindow.Store(var S: TStream);
begin
  inherited Store(S);
  PutSubViewPtr(S, Calendar);
  S.Write(Picture, SizeOf(TPicture));
end;

{****************************** TCalendarView ******************************}

constructor TCalendarView.Init(var Bounds: TRect; const AFileName: PathStr);
var
  H  : Word;
  Adj: TRect;
begin
  Adj.A.X := FrameSize;                { Extend left to the TFrame's edge   }
  Adj.A.Y := -CharSize.Y + FrameSize + TitleBarSize; { Probably equals 0    }
  Adj.B.X := Adj.A.X;                  { Extend right to the frame edge     }
  Adj.B.Y := FrameSize;                { Extend right to the TFRames' edge  }
  inherited Init(Bounds, Adj, bmcClipCentre, bmoCacheImage,
                 bmpNoUseSys + bmpRoundToDac + bmpLoadPalette +
                 bmpRemoveDups + bmpSavePalette, 0, AFileName);
  ID := 10020;
  Options := Options or ofSelectable;
  EventMask := EventMask or evMouseAuto;
  Locate(Bounds);
  if SizeAdj.B.X > FrameSize
    then Dec(Size.X);
  if SizeAdj.B.Y > FrameSize
    then Dec(Size.Y);
  SizeAdj.B.X := FrameSize;
  SizeAdj.B.Y := FrameSize;
  GetDate(CurYear, CurMonth, CurDay, H);
  Year := CurYear;
  Month := CurMonth;
end;

constructor TCalendarView.Load(var S: TStream);
var
  H: Word;
begin
  inherited Load(S);
  GetDate(CurYear, CurMonth, CurDay, H);
  S.Read(Year, SizeOf(Year));
  S.Read(Month, SizeOf(Month));
end;

function DayOfWeek(Day, Month, Year: Integer): Integer; near;
var
  century, yr, dw: Integer;
begin
  if Month < 3 then
  begin
    Inc(Month, 10);
    Dec(Year);
  end
  else
     Dec(Month, 2);
  century := Year div 100;
  yr := year mod 100;
  dw := (((26 * month - 2) div 10) + day + yr + (yr div 4) +
    (century div 4) - (2 * century)) mod 7;
  if dw < 0 then DayOfWeek := dw + 7
  else DayOfWeek := dw;
end;

procedure TCalendarView.Draw;
const
  Width = 21;
var
  i, j, k, DayOf, CurDays: Integer;
  S, B: String[Width+2];
  Color : LongRec;
  SpecialColor: Word;

function Num2Str(I: Integer): String; near;
var
  S: String[3];
begin
  Str(i:2, S);
  Num2Str := S;
end;

begin
  inherited Draw;
  LongInt(Color) := MapColor($0201);
  SetColor(WordRec(Color.Hi).Hi);      { Use frame title active paper colour}
  DayOf := DayOfWeek(1, Month, Year);  { for the calendar text              }
  Days := DaysInMonth[Month] + Byte((Year mod 4 = 0) and (Month = 2));
  Str(Year:4, S);
  k := (CharSize.X * 3) div 2;
  OutTextXY(k, 0, MonthStr[Month] + S + '  '#30'  '#31);
  OutTextXY(k, CharSize.Y * 8, 'Su Mo Tu We Th Fr Sa');
  CurDays := 1 - DayOf;
  for i := 1 to 6 do
  begin
    S := '';
    for j := 1 to 7 do
    begin
      if (CurDays < 1) or (CurDays > Days) then
        S := S + '   '
      else
        { if it is the current day }
        if (Year = CurYear) and (Month = CurMonth) and (CurDays = CurDay)
          then S := S + '~' + Num2Str(CurDays) + '~ '
        else
          S := S + Num2Str(CurDays) + ' ';
      Inc(CurDays);
    end;
    UnderLineXY(k, (i + 8) * CharSize.Y, S);
  end;
end;

function TCalendarView.GetPalette: PPalette;
const
  P: string[Length(CCalendarView)] = CCalendarView;
begin
  GetPalette := @P;
end;

procedure TCalendarView.HandleEvent(var Event: TEvent);
var
  Point    : TPoint;
  SelectDay: Word;
  k        : Integer;
  R        : TRect;
begin
  k := (CharSize.X * 3) div 2;
  TView.HandleEvent(Event);
  if (State and sfSelected <> 0) then
  begin
    if Event.What and (evMouseDown + evMouseAuto) <> 0 then
    begin
      MakePixLocal(Event.Where, Point);
      R.A.X := (16 * CharSize.X) + k;
      R.B.X := R.A.X + CharSize.X;
      R.A.Y := 0; R.B.Y := CharSize.Y;
      if R.Contains(Point) then
      begin
        Inc(Month);
        if Month > 12 then
        begin
          Inc(Year);
          Month := 1;
        end;
        DrawView;
      end;
      R.Move(CharSize.X * 3, 0);
      if R.Contains(Point) then
      begin
        Dec(Month);
        if Month < 1 then
        begin
          Dec(Year);
          Month := 12;
        end;
        DrawView;
      end;
    end
    else if Event.What = evKeyDown then
    begin
      if (Lo(Event.KeyCode) = byte('+')) or (Event.KeyCode = kbDown) then
      begin
        Inc(Month);
        if Month > 12 then
        begin
          Inc(Year);
          Month := 1;
        end;
      end;
      if (Lo(Event.KeyCode) = Byte('-')) or (Event.KeyCode = kbUp) then
      begin
        Dec(Month);
        if Month < 1 then
        begin
          Dec(Year);
          Month := 12;
        end;
      end;
      DrawView;
    end;
  end;
end;

procedure TCalendarView.Store(var S: TStream);
begin
  inherited Store(S);
  S.Write(Year, SizeOf(Year));
  S.Write(Month, SizeOf(Month));
end;

function CalendarColorItems(WindowPalette: word): PColorItem;
const
  Offset = 20;
begin
  CalendarColorItems :=
  ColorItem('Frame passive',         Offset + 0,
  ColorItem('Title passive',         Offset + 1,
  ColorItem('Frame active',          Offset + 2,
  ColorItem('Title Active and Text', Offset + 3,
  ColorItem('Frame Internal',        Offset + 4,
  ColorItem('Frame icons',           Offset + 5,
  nil))))));
end;

procedure RegisterCalendar;
begin
  RegisterType(RCalendarView);
  RegisterType(RCalendarWindow);
end;

end.
