unit ExtractU;

interface
USES Classes, IcoTypes;

TYPE
  DataBuffer = ARRAY[0..6000] OF Byte;
  TFileDataType = (fdtNormal, fdtICO, fdtPE, fdtUnknown);
  TFileData = Class(TObject)
    CONSTRUCTOR Create(Filename : String);
    private
      fdName : String;      {file name}
      fdDate,               {file date/time (for verification)}
      fdSize,               {file size (for verification)}
      fdStartIco,           {offset of icon resource table}
      fdStartImg,           {offset of image resource table}
      fdAFactor,            {align factor}
      fdNumIco,             {number of icons}
      fdNumImg : LongInt;   {number of images}
      fdType : TFileDataType;
      {read access methods for indexed properties}
      function GetSizIco(Index: Integer) : LongInt;
      function GetSizImgs(Index: Integer) : LongInt;
      function GetNumImgs(Index: Integer) : LongInt;
      {private methods}
      function InitDataICO : Integer;
      function InitDataEXE : Integer;
      FUNCTION PullIconDirectory(IconNum    : Integer;
        VAR IDSize : Word; VAR Buffer : DataBuffer;
        ForIcon: Boolean) : Integer;
      FUNCTION PullIconDirectoryICO(VAR IDSize : Word;
        VAR Buffer : DataBuffer; ForIcon: Boolean) : Integer;
      FUNCTION PullIconImage(ImageNum   : Integer;
        VAR IDSize : Word; VAR Buffer : DataBuffer) : Integer;
      FUNCTION PullIconImageIco(ImgNum : Word; VAR IDSize : Word;
        VAR Buffer : DataBuffer) : Integer;
      FUNCTION IcoFileFromExe(IFileName : String;
        IconNum   : Integer) : Integer;
    public
      {read-only properties}
      property FileDataType : TFileDataType read fdType;
      property StartIco : LongInt read fdStartIco;
      property StartImg : LongInt read fdStartImg;
      property AFactor  : LongInt read fdAFactor;
      property NumIco   : LongInt read fdNumIco;
      property NumImg   : LongInt read fdNumImg;
      property SizIco[Index: Integer] : LongInt read GetSizIco;
      property SizImgs[Index: Integer] : LongInt read GetSizImgs;
      property NumImgs[Index: Integer] : LongInt read GetNumImgs;
      {public methods}
      function Verify : Integer;
      function ExtractIconFromExe(IFilename : String;
        IcoNum : Integer) : Integer;
      function GetIconDirectory(IcoNum : Integer;
        VAR Buffer : DataBuffer) : Integer;
      function GetIconImage(IcoNum, ImgNum : Integer;
        VAR Buffer : DataBuffer) : Integer;
      FUNCTION ErrorString(ErrNum : Integer) : String;
  END;

CONST
  {Error constants for ExtractU functions}
  EXU_OK          = 0;
  EXU_DOSHEAD     = -1;
  EXU_NOTNE       = -2;
  EXU_ISPE        = -3;
  EXU_NORESOURCES = -4;
  EXU_NOICONS     = -5;
  EXU_FILEERROR   = -6;
  EXU_BADINDEX    = -7;
  EXU_BADIMGINDEX = -8;
  EXU_CHANGED     = -9;
  EXU_MISSING     = -10;
  EXU_INVALIDICON = -11;
  EXU_WRONGTYPE   = -12;
  EXU_PEMONO      = -13;
  EXU_WINDOWS2    = -14;

(*********************************************)
implementation
uses SysUtils, WinTypes;

PROCEDURE OpenForReadOnly(VAR F : File; FName : String);
{The calling routine must handle any exception caused
 by the attempt to open the file}
VAR OldFileMode : Byte;
BEGIN
  AssignFile(F, FName);
  OldFileMode := FileMode;
  FileMode := 0; {read-only}
  Reset(F,1);
  FileMode := OldFileMode;
END;

CONSTRUCTOR TFileData.Create(Filename : String);
VAR Rslt : Integer;
BEGIN
  Inherited Create;
  fdName := Filename;
  IF UpperCase(ExtractFileExt(Filename)) = '.ICO' THEN
    fdType := fdtICO
  ELSE fdType := fdtNormal;
  fdStartImg := 0;
  fdStartIco := 0;
  fdNumIco   := 1;
  fdAFactor  := 1;
  CASE fdType OF
    fdtNormal : BEGIN
                  Rslt := InitDataEXE;
                  IF Rslt = EXU_ISPE THEN
                   fdType := fdtPE
                  ELSE IF Rslt <= 0 THEN
                    fdType := fdtUnknown;
                END;
    fdtICO    : IF InitDataICO <> EXU_OK THEN
                fdType := fdtUnknown;
  END;
END;

FUNCTION TFileData.InitDataICO : Integer;
VAR
  F  : File;
  ID : IconDir;
BEGIN
  Result := EXU_OK;
  try
    OpenForReadOnly(F, fdName);
    try
      fdDate := FileGetDate(TFileRec(F).Handle);
      fdSize := FileSize(F);
      BlockRead(F, ID, SizeOf(ID));
      IF (ID.idReserved <> 0) OR
         (ID.idType <> 1) THEN
        Result := EXU_INVALIDICON
      ELSE fdNumImg := ID.idCount;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.InitDataEXE : Integer;
{Get location of resource tables, number of icons and
 images, etc for .EXE and .DLL files}
VAR
  W, N          : Word;
  F             : File;
  DH            : TDOSHeader;
  NEOffset      : Word; {Offset of NE header}
  NE            : TNEHeader;
  RTI           : ResTypeInfo;
BEGIN
  try
    OpenForReadOnly(F, fdName);
    try
      fdDate := FileGetDate(TFileRec(F).Handle);
      fdSize := FileSize(F);
      BlockRead(F, DH, SizeOf(DH));
      Move(DH[0], N, 2);
      Move(DH[$18], W, 2);
      IF (N <> $5A4D{MZ}) OR (W < $40) THEN
        BEGIN
          Result := EXU_DOSHEAD;
          Exit;
        END;
      Move(DH[$3C], NEOffset, 2);
      Seek(F, NEOffset);
      BlockRead(F, NE, SizeOf(NE));
      IF NE.Signature <> $454E{NE} THEN
        BEGIN
          IF NE.Signature = $4550{PE} THEN
            Result := EXU_ISPE
          ELSE Result := EXU_NOTNE;
          Exit;
        END;
      IF NE.ExpectedWindowsVersion < $300 THEN
        BEGIN
          Result := EXU_WINDOWS2;
          Exit;
        END;
      IF NE.ResourceTableRelOffset = NE.ResidentNameTableRelOffset THEN
        BEGIN
          Result := EXU_NORESOURCES;
          Exit;
        END;
      Seek(F, NE.ResourceTableRelOffset + NEOffset);
      BlockRead(F, W, 2);
      fdAFactor := 1;
      FOR N := 1 TO W DO
        fdAFactor := fdAFactor * 2;
      BlockRead(F, RTI, SizeOf(RTI));
      fdStartIco := 0;
      fdStartImg := 0;
      WHILE (RTI.rtTypeID <> 0) AND (RTI.rtReserved = 0) DO
        BEGIN
          IF RTI.rtTypeID AND $7FFF = LongInt(RT_ICON) THEN
            BEGIN
              fdStartImg := FilePos(F);
              fdNumImg := RTI.rtResourceCount;
            END
          ELSE IF RTI.rtTypeID AND $7FFF = LongInt(RT_GROUP_ICON) THEN
            BEGIN
              fdStartIco := FilePos(F);
              fdNumIco := RTI.rtResourceCount;
              Result := RTI.rtResourceCount;
            END;
          Seek(F, FilePos(F) + LongInt(RTI.rtResourceCount)*SizeOf(NameInfo));
          BlockRead(F, RTI, SizeOf(RTI));
        END;
      IF (fdStartIco=0) OR (fdStartImg=0) THEN
        Result := EXU_NOICONS;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.PullIconDirectory(
{pull the icon directory from an .EXE or .DLL file}
  IconNum    : Integer;     {Index of desired icon within the file}
  VAR IDSize : Word;        {Size of resulting data}
  VAR Buffer : DataBuffer;  {buffer to hold resulting data}
  ForIcon: Boolean)         {true if extracting for icon}
  : Integer;
VAR
  F       : File;
  NI      : NameInfo;
  ID      : IconDir;
  IDE     : IconDirectoryEntry;
  NImage,
  DataPos : Word;
BEGIN
  Result := EXU_OK;
  try
    OpenForReadOnly(F, fdName);
    try
      Seek(F, fdStartIco + IconNum*SizeOf(NameInfo));
      BlockRead(F, NI, SizeOf(NI));
      Seek(F, NI.niOffset*fdAFactor);
      BlockRead(F, ID, SizeOf(ID));
      Move(ID, Buffer[0], SizeOf(ID));
      DataPos := SizeOf(ID);
      FOR NImage := 0 TO ID.idCount-1 DO
        BEGIN
          IDE.dwImageOffset := 0;
          BlockRead(F, IDE, SizeOf(IDE)-2);
          IF ForIcon THEN
            BEGIN
              Move(IDE, Buffer[DataPos], SizeOf(IDE));
              DataPos := DataPos + SizeOf(IDE);
            END
          ELSE
            BEGIN
              Move(IDE, Buffer[DataPos], SizeOf(IDE)-2);
              DataPos := DataPos + SizeOf(IDE)-2;
            END;
        END;
      IDSize := DataPos;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.PullIconDirectoryICO(
{pull the icon directory from an .ICO}
  VAR IDSize : Word;        {Size of resulting data}
  VAR Buffer : DataBuffer;  {buffer to hold resulting data}
  ForIcon: Boolean)         {true if extracting for icon}
  : Integer;
VAR
  F         : File;
  ID        : IconDir;
  IDE       : IconDirectoryEntry;
  NImage, W,
  DataPos   : Word;
begin
  try
    OpenForReadOnly(F, fdName);
    try
      BlockRead(F, ID, SizeOf(ID));
      Move(ID, Buffer[0], SizeOF(ID));
      DataPos := SizeOf(ID);
      FOR NImage := 0 TO ID.idCount-1 DO
        BEGIN
          BlockRead(F, IDE, SizeOf(IDE)-2);
          Move(IDE, Buffer[DataPos], SizeOf(IDE)-2);
          BlockRead(F, W, 2);
          DataPos := DataPos + SizeOf(IDE)-2;
        END;
      IDSize := DataPos;
      Result := EXU_OK;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.PullIconImage(
{pull one icon image from an .EXE file}
  ImageNum   : Integer;     {Index of desired icon within the file}
  VAR IDSize : Word;        {Size of resulting data}
  VAR Buffer : DataBuffer)  {buffer to hold resulting data}
  : Integer;
VAR
  F      : File;
  NI     : NameInfo;
  Actual : Word;
BEGIN
  Result := EXU_OK;
  try
    OpenForReadOnly(F, fdName);
    try
      Seek(F, fdStartImg + (ImageNum-1)*SizeOf(NameInfo));
      BlockRead(F, NI, SizeOf(NI));
      Seek(F, NI.niOffset*fdAFactor);
      BlockRead(F, Buffer, LongInt(NI.niLength)*fdAFactor, Actual);
      IDSize := Actual;
    finally
      CloseFile(F);
    end;
  except
    ON Exception DO Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.PullIconImageIco(ImgNum : Word; VAR IDSize : Word;
  VAR Buffer : DataBuffer) : Integer;
VAR
  F : File;
  ID : IconDir;
  IDE : IconDirectoryEntry;
begin
  try
    OpenForReadOnly(F, fdName);
    try
      Seek(F, SizeOf(ID) + ImgNum*SizeOf(IDE));
      BlockRead(F, IDE, SizeOf(IDE));
      Seek(F, IDE.dwImageOffset);
      BlockRead(F, Buffer, IDE.dwBytesInRes);
      Result := IDE.dwBytesInRes;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_FILEERROR;
  end;
end;

FUNCTION TFileData.IcoFileFromExe(
{extract an .ICO file from an .EXE/.DLL file}
  IFileName : String;  {.ICO file name to create}
  IconNum   : Integer) {index of icon to extract}
  : Integer;
VAR
  Ico_F       : File;
  N           : Word;
  IcoBuffer   : DataBuffer;
  IcoHead     : IconDirICO ABSOLUTE IcoBuffer;
  IcoBuffSize : Word;
  ImgBuffer   : DataBuffer;
  ImgBuffSize : Word;
BEGIN
  try
    Assign(Ico_F, IFilename);
    Rewrite(Ico_F, 1);
    try
      Result := PullIconDirectory(IconNum, IcoBuffSize, IcoBuffer, True);
      IF Result <> EXU_OK THEN Exit;
      IcoBuffSize := IcoHead.idCount * SizeOf(IconDirectoryEntry) + 6;
      BlockWrite(Ico_F, IcoBuffer, IcoBuffSize);
      FOR N := 0 TO IcoHead.idCount-1 DO
        BEGIN
          Result := PullIconImage(IcoHead.idEntries[N].dwImageOffset,
            ImgBuffSize, ImgBuffer);
          IF Result <> EXU_OK THEN Exit;
          IcoHead.idEntries[N].dwImageOffset := FilePos(Ico_F);
          BlockWrite(Ico_F, ImgBuffer, IcoHead.idEntries[N].dwBytesInRes);
        END;
      Seek(Ico_F, 0);
      BlockWrite(Ico_F, IcoBuffer, IcoBuffSize);
    finally
      Close(Ico_F);
    end;
  except ON EInOutError DO
    Result := EXU_FILEERROR;
  end;
END;

FUNCTION TFileData.Verify : Integer;
VAR F : File;
BEGIN
  try
    OpenForReadOnly(F, fdName);
    try
      IF (fdDate = FileGetDate(TFileRec(F).Handle)) AND
         (fdSize = FileSize(F)) THEN
        Result := EXU_OK
      ELSE Result := EXU_CHANGED;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_MISSING;
  end;
END;

function TFileData.GetSizIco(Index: Integer) : LongInt;
{read access function for indexed property SizIco}
VAR
  F  : File;
  NI : NameInfo;
  N  : Word;
  ID : IconDir;
begin
  IF fdType = fdtPE THEN
    BEGIN
      Result := 22; {size for standard single-image icon}
      Exit;
    END;
  IF Index >= fdNumIco THEN
    BEGIN
      Result := EXU_BADINDEX;
      Exit;
    END;
  try
    OpenForReadOnly(F, fdName);
    try
      CASE fdType OF
        fdtNormal : BEGIN
          Seek(F, fdStartIco + Index*SizeOf(NameInfo));
          BlockRead(F, NI, SizeOf(NameInfo));
          Result := fdAFactor * NI.niLength;
        END;
        fdtIco    : BEGIN
          BlockRead(F, ID, SizeOf(ID));
          Result := SizeOf(ID) + ID.idCount * SizeOf(IconDirectoryEntry);
        END;
      END;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_FILEERROR;
  end;
end;

function TFileData.GetSizImgs(Index: Integer) : LongInt;
{read access function for indexed property SizImgs}
VAR
  F         : File;
  NI        : NameInfo;
  NImage    : Word;
  ID        : IconDir;
  IDE       : IconDirectoryEntry;
  IDEs      : IconDirEntryShort;
  ImageNums : ARRAY[0..11] OF Word;
begin
  IF fdType = fdtPE THEN
    BEGIN
      Result := 744; {image size for standard single-image icon}
      Exit;
    END;
  IF Index >= fdNumIco THEN
    BEGIN
      Result := EXU_BADINDEX;
      Exit;
    END;
  try
    OpenForReadOnly(F, fdName);
    try
      CASE fdType OF
        fdtNormal : BEGIN
          Seek(F, fdStartIco + Index*SizeOf(NameInfo));
          BlockRead(F, NI, SizeOf(NameInfo));
          Seek(F, AFactor*NI.niOffset);
          BlockRead(F, ID, SizeOf(ID));
          FOR NImage := 0 TO ID.idCount-1 DO
            BEGIN
              BlockRead(F, IDEs, SizeOf(IDEs));
              ImageNums[NImage] := IDEs.wID;
              IF ImageNums[NImage] > fdNumImg THEN
                BEGIN
                  Result := EXU_BADIMGINDEX;
                  Exit;
                END;
            END;
          Result := 0;
          FOR NImage := 0 TO ID.idCount-1 DO
            BEGIN
              Seek(F, fdStartImg + (ImageNums[NImage]-1)*SizeOf(NI));
              BlockRead(F, NI, SizeOf(NI));
              Result := Result + AFactor * NI.niLength;
            END;
        END;
        fdtICO    : BEGIN
          BlockRead(F, ID, SizeOf(ID));
          Result := 0;
          FOR NImage := 0 TO ID.idCount-1 DO
            BEGIN
              BlockRead(F, IDE, SizeOf(IDE));
              Result := Result + IDE.dwBytesInRes;
            END;
        END;
      END;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_FILEERROR;
  end;
end;

function TFileData.GetNumImgs(Index: Integer) : LongInt;
{read access function for indexed property NumImgs}
VAR
  F  : File;
  NI : NameInfo;
  ID : IconDir;
begin
  IF fdType = fdtPE THEN
    BEGIN
      Result := 1;
      Exit;
    END;
  IF Index >= fdNumIco THEN
    BEGIN
      Result := EXU_BADINDEX;
      Exit;
    END;
  try
    OpenForReadOnly(F, fdName);
    try
      CASE fdType OF
        fdtNormal : BEGIN
          Seek(F, fdStartIco + Index*SizeOf(NameInfo));
          BlockRead(F, NI, SizeOf(NameInfo));
          Seek(F, AFactor*NI.niOffset);
          BlockRead(F, ID, SizeOf(ID));
          Result := ID.idCount;
        END;
        fdtICO    : BEGIN
          BlockRead(F, ID, SizeOf(ID));
          Result := ID.idCount;
        END;
      END;
    finally
      CloseFile(F);
    end;
  except
    ON EInOutError DO
      Result := EXU_FILEERROR;
  end;
end;

function TFileData.ExtractIconFromExe(IFilename : String;
  IcoNum : Integer) : Integer;
BEGIN
  IF fdType = fdtNormal THEN
    BEGIN
      Result := Verify;
      IF Result = EXU_OK THEN
        Result := IcoFileFromExe(IFilename, IcoNum);
    END
  ELSE Result := EXU_WRONGTYPE;
END;

function TFileData.GetIconDirectory(IcoNum : Integer;
        VAR Buffer : DataBuffer) : Integer;
VAR IDSize   : Word;
BEGIN
  CASE fdType OF
    fdtNormal : Result := PullIconDirectory(IcoNum, IDSize,
                  Buffer, False);
    fdtICO    : Result := PullIconDirectoryICO(IDSize,
                  Buffer, False);
    ELSE Result := EXU_WRONGTYPE;
  END;
  IF Result = EXU_OK THEN Result := IDSize;
END;

function TFileData.GetIconImage(IcoNum, ImgNum : Integer;
        VAR Buffer : DataBuffer) : Integer;
VAR
  IcoHead : IconDirEXE ABSOLUTE Buffer;
  IDSize  : Word;
BEGIN
  CASE fdType OF
    fdtNormal : BEGIN
      Result := PullIconDirectory(IcoNum, IDSize, Buffer, False);
      IF Result = EXU_OK THEN
        Result := PullIconImage(IcoHead.idEntries[ImgNum].wID,
          IDSize, Buffer);
      IF Result = EXU_OK THEN
        Result := idSize;
    END;
    fdtICO  : Result := PullIconImageICO(ImgNum, IDSize, Buffer);
    ELSE Result := EXU_WRONGTYPE;
  END;
END;

FUNCTION TFileData.ErrorString(ErrNum : Integer) : String;
{Returns an explanatory string for errors associated with
 TFileData}
BEGIN
  CASE ErrNum OF
    EXU_DOSHEAD     : Result := 'File appears to be a DOS executable';
    EXU_NOTNE       : Result := 'File is not a Windows executable';
    EXU_ISPE        : Result := 'File is a 32-bit Portable Executable';
    EXU_NORESOURCES : Result := 'File contains no resources';
    EXU_NOICONS     : Result := 'File contains no icons';
    EXU_FILEERROR   : Result := 'File I/O error';
    EXU_OK          : Result := 'No error';
    EXU_BADINDEX    : Result := 'Invalid icon index';
    EXU_BADIMGINDEX : Result := 'Invalid image index (file corrupt)';
    EXU_CHANGED     : Result := 'File has been modified';
    EXU_MISSING     : Result := 'File missing';
    EXU_INVALIDICON : Result := 'Invalid .ICO file';
    EXU_WRONGTYPE   : Result := 'Wrong file type';
    EXU_PEMONO      : Result := 'Cannot handle 2-color icon from PE file';
    EXU_WINDOWS2    : Result := 'Windows 2.0 file';
    ELSE              Result := 'UNKNOWN error';
  END;
END;
end.

