UNIT vtSetup;

{$I DEFINES.INC}

INTERFACE

Uses crt,fileio,comio,vidio;

procedure TTSetup;             { Communications setup function }
procedure VidSetup;            { Video setup function }
procedure KeySetup;            { Keyboard setup function }
procedure VT100Setup;          { VT emulation setup function }
procedure FileSetup;           { File system setup function }

IMPLEMENTATION

uses keyio;

CONST
  F1   = $3B00;             { Scancode for function key F1  }
  F2   = $3C00;             { Scancode for function key F2  }
  F3   = $3D00;             { Scancode for function key F3  }
  F4   = $3E00;             { Scancode for function key F4  }
  F5   = $3F00;             { Scancode for function key F5  }
  F6   = $4000;             { Scancode for function key F6  }
  F7   = $4100;             { Scancode for function key F7  }
  F8   = $4200;             { Scancode for function key F8  }
  F9   = $4300;             { Scancode for function key F9  }
  F10  = $4400;             { Scancode for function key F10 }
  ESC  = $011b;             { Scancode for ESCape key       }

  vcolors : array[0..15] of string[12] = (        { Array of color names }
    {123456789012    123456789012    123456789012}
    'BLACK       ', 'BLUE        ', 'GREEN       ',
    'CYAN        ', 'RED         ', 'MAGENTA     ',
    'BROWN       ', 'LIGHTGRAY   ', 'DARKGRAY    ',
    'LIGHTBLUE   ', 'LIGHTGREEN  ', 'LIGHTCYAN   ',
    'LIGHTRED    ', 'LIGHTMAGENTA', 'YELLOW      ',
    'WHITE       ');

  vbauds : array[0..13] of word = ( {Array of valid baud rate settings }
       50, 75, 110, 150, 300,
       600, 1200, 1800, 2000,
       2400, 4800, 9600, 19200,
       38400);
  vparitys : array[0..2] of string[4] = (  { Array of valid parity settings }
       'NONE', 'EVEN', 'ODD ');
  vdatabits : array[0..3] of integer = (  { Array of valid databit settings }
       5,6,7,8);
  OffOn : array[boolean] of string[3] = ('OFF','ON ');

VAR
  holdbackground : boolean;  { Temporary holder for background setting }
  saved          : boolean;  { Flag indication setup already saved }
  vbdx : integer;
  vpdx : integer;
  vddx : integer;

{***************************************************************************}
{***************************************************************************}

function ltos(l : longint; w,v : integer) : string;
var
  s : string;
begin
  str(l:w,s);
  while length(s) < v do s := s + ' ';
  ltos := s;
end;


{ S T A R T S E T U P -- Begin a setup mode screen }

procedure StartSetup(title : string);
begin
   SaveScreen;
   ClearScreen;
   vtprintf(0,0,true,'                                                                                ');
   vtprintf(0,0,true,title+'  Setup Mode,   Use the function keys to toggle settings');
   vtprintf(24,2,false,'ESC) or F10)   Exit '+title+' Setup');
end;


{ E N D S E T U P -- End the setup mode screen }

procedure EndSetup;
begin
   RestoreScreen;               { Restore the pre-setup screen }
end;



{  D I S P L A Y P O R T -- Display COM port setting }

procedure DisplayPort; begin
   vtprintf(5,2,false, ' F1)  COM port  = '+ltos(comport,1,1));
end;


{  D I S P L A Y B A U D -- Display baud rate setting }

procedure DisplayBaud; begin
   vtprintf(7,2,false, ' F2)  Baud rate = '+ltos(speed,1,5));
end;


{  D I S P L A Y P A R I T Y -- Display parity setting }

procedure DisplayParity; begin
   vtprintf(9,2,false, ' F3)  Parity    = '+parity);
end;


{  D I S P L A Y D A T A B I T S -- Display databits setting }

procedure DisplayDataBits; begin
   vtprintf(11,2,false,' F4)  Data bits = '+ltos(databits,1,1));
end;


{  D I S P L A Y S T O P B I T S -- Display stopbits setting }

procedure DisplayStopBits; begin
   vtprintf(13,2,false,' F5)  Stop bits = '+ltos(stopbits,1,1));
end;



{  T O G G L E P O R T -- toggle the COM port setting }

procedure TogglePort;
begin
  comport := 3-comport;
end;


{  T O G G L E B A U D -- Toggle the baud rate setting }

procedure ToggleBaud;
begin
  vbdx := succ(vbdx) mod 14;
  speed := vbauds[vbdx];
end;


{  T O G G L E P A R I T Y -- Toggle the parity setting }

procedure ToggleParity;
begin
  vpdx := succ(vpdx) mod 3;
  parity := vparitys[vpdx];
end;


{  T O G G L E D A T A B I T S -- Toggle databits setting }

procedure ToggleDataBits;
begin
  vddx := succ(vddx) mod 4;
  databits := vdatabits[vddx];
end;


{  T O G G L E S T O P B I T S -- Toggle StopBits setting }

procedure ToggleStopBits;
begin
  stopbits := 3 - stopbits;
end;

{  T T S E T U P -- Enter Communications Setup Mode }

procedure TTSetup;
var
  thru, changes : boolean;
begin
   thru := false;
   changes := false;
   StartSetup('Communications');  { Begin this Setup Screen }
   DisplayPort;                   { Display initial current port setting }
   DisplayBaud;                   { Display initial current baud rate setting }
   DisplayParity;                 { Display initial current parity setting }
   DisplayDataBits;               { Display initial current data bits setting }
   DisplayStopBits;               { Display initial current stop bits setting }
   vbdx := 13;
   while (vbdx > 0) and (speed <> vbauds[vbdx]) do dec(vbdx);
   vpdx := 2;
   while (vpdx > 0) and (parity <> vparitys[vpdx]) do dec(vpdx);
   vddx := 3;
   while (vddx > 0) and (databits <> vdatabits[vddx]) do dec(vddx);
   while not thru do case getkey of
     F1 : { Toggle the COM port setting }
          begin
            TogglePort;
            DisplayPort;
            changes := true;
          end;
     F2 : { Toggle the baud rate setting }
          begin
            ToggleBaud;
            DisplayBaud;
            changes := true;
          end;
     F3 : { Toggle the parity setting }
          begin
            ToggleParity;
            DisplayParity;
            changes := true;
          end;
     F4 : { Toggle the data bits setting }
          begin
            ToggleDataBits;
            DisplayDataBits;
            changes := true;
          end;
     F5 : { Toggle the stop bits setting }
          begin
            ToggleStopBits;
            DisplayStopBits;
            changes := true;
          end;
     F10,
     ESC :
          thru := true;
   end;
   RestoreScreen;               { Restore the initial emulation screen }
   if (changes) then begin      { If changes have been made }
     if ttclose <> 0 then         { Close the communications port }
      badexit('could not close com port');
     if ttopen <> 0 then          { reopen com port to update }
       badexit('could not open com port');
   end;
end;

{***************************************************************************}


{  D I S P L A Y R E T R A C E -- Display video retrace mode setting }

procedure DisplayRetrace;

begin
  vtprintf(5,2,false,' F1)  Video Snow Retrace Wait = '+OffOn[retracemode]);
end;


{  D I S P L A Y F O R E C O L O R -- Display foreground color }

procedure DisplayForeColor;
begin
  vtprintf(7,2,false,' F2)  Foreground Color = '+vcolors[forecolor]);
end;


{  D I S P L A Y B A C K C O L O R -- Display background color }

procedure DisplayBackColor;
begin
   vtprintf(9,2,false,' F3)  Background Color = '+vcolors[backcolor]);
end;



{  T O G G L E R E T R A C E -- Toggle Video Retrace mode setting }

procedure ToggleRetrace;
begin
   if (vidmode <> 7) then        { Don't allow toggling on a monochrome }
     retracemode := not retracemode
   else begin
     retracemode := false;
     sound(324);
     delay(400);
     nosound;
   end;
end;

{  T O G G L E F O R E C O L O R -- Toggle the foreground color }

procedure ToggleForeColor;
begin
  forecolor := succ(forecolor) mod 16;
end;

{  T O G G L E B A C K C O L O R -- Toggle the background color }

procedure ToggleBackColor;
begin
  backcolor := succ(backcolor) mod 16;
end;


{  V I D S E T U P -- Enter Video Setup Mode }

procedure VidSetup;
var
  thru, colorchange : boolean;
begin
  thru := false;
  colorchange := false;

  StartSetup('Video');           { Begin this Setup Screen }
  DisplayRetrace;              { Display initial retrace mode setting }
  DisplayForeColor;            { Display initial foreground color }
  DisplayBackColor;            { Display initial background color }

  while not thru do case GetKey of
    F1 : { Toggle the Retrace mode setting }
         begin
           ToggleRetrace;
           DisplayRetrace;
         end;
    F2 : { Toggle the Foreground color }
         begin
           ToggleForeColor;
           DisplayForeColor;
           colorchange := true;
         end;
    F3 : { Toggle the Background color }
         begin
           ToggleBackColor;
           DisplayBackColor;
           colorchange := true;
         end;
    F10,
    ESC : thru := true;
   end;
   RestoreScreen;               { Restore the initial emulation screen }
   if (colorchange) then SetColor;
end;

{***************************************************************************}

{  D I S P L A Y B A C K S P A C E -- Display backspace interpretation }

procedure DisplayBackSpace;
const
  BackDel : array[boolean] of string[9] = ('DELETE   ','BACKSPACE');
begin
  vtprintf(5,2,false,' F1)  BackSpace Interpretation = '+BackDel[backspace]);
end;


{  D I S P L A Y K E Y C L I C K -- Display keyclick mode }

procedure DisplayKeyClick;
begin
  vtprintf(7,2,false,' F2)  KeyClick is '+OffOn[keyclick]);
end;

{  D I S P L A Y K E Y P A D M O D E -- Display keypad mode }

procedure DisplayKeyPadMode;
const
  NumApp : array[boolean] of string[11] = ('NUMERIC    ','APPLICATION');
begin
  vtprintf(9,2,false,' F3)  Key Pad is set to '+NumApp[applkeypad]);
end;


{  T O G G L E B A C K S P A C E -- Toggle Backspace interpretation }

procedure ToggleBackSpace;
begin
  backspace := not backspace;
end;

{  T O G G L E K E Y C L I C K -- Toggle Keyclick mode }

procedure ToggleKeyClick;
begin
   keyclick := not keyclick;
end;

{  T O G G L E K E Y P A D M O D E -- Toggle KeyPad mode }

procedure ToggleKeyPadMode;
begin
  applkeypad := not applkeypad;
end;



{  K E Y S E T U P -- Enter Keyboard Setup Mode }

procedure KeySetup;
var
  thru : boolean;
begin
  thru := false;
  StartSetup('KeyBoard');        { Begin this Setup Screen }

  DisplayBackSpace;            { Display initial retrace mode setting }
  DisplayKeyClick;             { Display initial Key Click mode setting }
  DisplayKeyPadMode;           { Display initial Key Pad mode setting }

  while( not thru ) do case getkey of
    F1 : { Toggle the Backspace key interpretation}
         begin
           ToggleBackSpace;
           DisplayBackSpace;
         end;
    F2 : { Toggle the KeyClick Setting }
         begin
           ToggleKeyClick;
           DisplayKeyClick;
         end;
    F3 : { Toggle the KeyPadMode Setting }
         begin
           ToggleKeyPadMode;
           DisplayKeyPadMode;
         end;
    F10,
    ESC : thru := true;
   end;
   RestoreScreen;               { Restore the initial emulation screen }
end;

{***************************************************************************}



{  D I S P L A Y S A V E S E T U P -- Display video retrace mode setting }

procedure DisplaySaveSetup;
const
  Done : array[boolean] of string[7] = ('       ',' *DONE*');
begin
  vtprintf(5,2,false,' F1)  Save Setup to '+setupfile+Done[saved]);
end;

{  D I S P L A Y L O G M O D E -- Display log file status }

procedure DisplayLogMode;
begin
  if (log) then
    vtprintf(7,2,false,' F2)  Logging incoming characters to '+logfile)
  else
    vtprintf(7,2,false,' F2)  Logging of incoming characters is OFF            ');
end;


{  T O G G L E L O G M O D E -- Toggle the log file status }

procedure ToggleLogMode;
begin
  log := not log;
end;

{  F I L E S E T U P -- Enter File Setup Mode }

procedure FileSetup;
var
  thru : boolean;
begin
  thru := false;
  StartSetup('File');           { Begin this Setup Screen }
  saved := false;               { Start with information not saved }
  DisplaySaveSetup;           { Display initially unsaved setup }
  DisplayLogMode;             { Display status of Log File }

  while( not thru ) do case getkey of
     F1 : { Save the setup information to disk }
          begin
            SaveSetup;
            saved := true;
            DisplaySaveSetup;
         end;
     F2 : { Set/reset logging to disk }
          begin
            ToggleLogMode;
            DisplayLogMode;
            if (log) then
              OpenLogFile
            else
              CloseLogFile;
          end;
     F10,
     ESC : thru := true;
   end;
   RestoreScreen;               { Restore the initial emulation screen }
end;


{***************************************************************************}

{  D I S P L A Y O R I G I N M O D E -- Display the origin mode }

procedure DisplayOriginMode;
const
  OrMode : array[boolean] of string[8] = ('ABSOLUTE', 'RELATIVE');
begin
  vtprintf(7,2,false,' F1)  Origin Mode is '+OrMode[originmode]);
end;

{  D I S P L A Y I N S E R T M O D E -- Display insert/replace mode }

procedure DisplayInsertMode;
const
  InsMode  : array[boolean] of string[7] = ('REPLACE', 'INSERT  ');
begin
  vtprintf(9,2,false,' F2)  Insert/Replace mode is set to '+InsMode[insertmode]);
end;


{  D I S P L A Y A U T O W R A P -- Display the autowrap mode }

procedure DisplayAutoWrapMode;
begin
  vtprintf(11,2,false,' F3)  Auto Wrap mode is '+OffOn[autowrap]);
end;


{  D I S P L A Y N E W L I N E -- Display the newline mode }

procedure DisplayNewLine;
begin
  vtprintf(13,2,false,' F4)  New Line mode is '+OffOn[newline]);
end;


{  D I S P L A Y C U R S O R V I S I B L E -- Display the cursor visibility }

procedure DisplayCursorVisible;
const
  VisMode : array[boolean] of string[7] = ('HIDDEN ','VISIBLE');
begin
  vtprintf(15,2,false,' F5)  Cursor is '+VisMode[cursorvisible]);
end;


{  D I S P L A Y R E V E R S E B A C K G R O U N D -- Display background }

procedure DisplayReverseBackGround;
const
  BackMode : array[boolean] of string[8] = ('NORMAL  ','REVERSED');
begin
  vtprintf(17,2,false,' F6)  Background is '+BackMode[holdbackground]);
end;

{  D I S P L A Y S C R E E N W I D -- Display the screen width }

procedure DisplayScreenWid;
begin
  vtprintf(19,2,false,' F7)  Logical Screen width is '+ltos(screenwid,1,1)+' columns ');
end;

{  T O G G L E O R I G I N M O D E -- Toggle the origin mode }

procedure ToggleOriginMode;
begin
   originmode := not originmode;
end;

{  T O G G L E I N S E R T M O D E -- Toggle insert/replace mode }

procedure ToggleInsertMode;
begin
   insertmode := not insertmode;
end;


{  T O G G L E A U T O W R A P -- Toggle the autowrap mode }

procedure ToggleAutoWrapMode;
begin
   autowrap := not autowrap;
end;


{  T O G G L E N E W L I N E -- Toggle the newline mode }

procedure ToggleNewLine;
begin
   newline := not newline;
end;


{  T O G G L E C U R S O R V I S I B L E -- Toggle the cursor visibility }

procedure ToggleCursorVisible;
begin
   cursorvisible := not cursorvisible;
end;


{  T O G G L E R E V E R S E B A C K G R O U N D -- Toggle background }

procedure ToggleReverseBackGround;
begin
   holdbackground := not holdbackground;
end;

{  T O G G L E S C R E E N W I D -- Toggle the screen width }

procedure ToggleScreenWid;
begin
  if (screenwid = 80) then
    screenwid := 132
  else
    screenwid := 80;
end;




{  V T S E T U P -- Enter Emulation Setup Mode }

procedure VT100setup;
var
   thru : boolean;
   cursorchange : boolean;
   backgroundchange : boolean;
   screenwidchange : boolean;
begin
  thru := false;
  cursorchange := false;
  backgroundchange := false;
  screenwidchange := false;
  holdbackground := reversebackground; { Get value of background setting }

  StartSetup('Emulation');       { Begin this Setup Screen }

  DisplayOriginMode;           { Display initial origin mode setting }
  DisplayInsertMode;           { Display initial insert mode setting }
  DisplayAutoWrapMode;         { Display initial autowrap mode setting }
  DisplayNewLine;              { Display initial newline mode setting }
  DisplayCursorVisible;        { Display initial cursorvisible setting }
  DisplayReverseBackGround;    { Display initial setting for backgroun }
  DisplayScreenWid;            { Display initial logical screen width }

  while (not thru) do case getkey of
    F1 : { Toggle origin mode setting }
         begin
           ToggleOriginMode;
           DisplayOriginMode;
         end;
    F2 : { Toggle insert mode setting }
         begin
           ToggleInsertMode;
           DisplayInsertMode;
         end;
    F3 : { Toggle autowrap mode setting }
         begin
           ToggleAutoWrapMode;
           DisplayAutoWrapMode;
         end;
    F4 : { Toggle newline mode setting }
         begin
           ToggleNewLine;
           DisplayNewLine;
         end;
    F5 : { Toggle cursor visibility }
         begin
           ToggleCursorVisible;
           DisplayCursorVisible;
           cursorchange := true;
         end;
    F6 : { Toggle background appearance }
         begin
           ToggleReverseBackGround;
           DisplayReverseBackGround;
           backgroundchange := true;
         end;
    F7 : { Toggle screen width }
         begin
           ToggleScreenWid;
           DisplayScreenWid;
           screenwidchange := true;
        end;

    F10,
    ESC : thru := true;
  end;

  RestoreScreen;                { Restore the initial emulation screen }
  if (cursorchange) then        { Change the cursor type if needed }
    SetCursorVisibility(cursorvisible);
  if (backgroundchange) then    { Change the background if needed }
    SetBackGround(holdbackground);
  if (screenwidchange) then     { Change the screenwidth if needed }
    SetScreenWidth(screenwid);
end;

end.
 {***************************************************************************}





