UNIT Vidio;

{$I DEFINES.INC}
{$R-}  { must have range checking off! }

INTERFACE

USES dos,fileio;


procedure VidInit;                  { Initialize the video system }
procedure SetVattr(attr : byte );   { Set the video attribute }
procedure AddVattr(attr : byte );   { Add attribute to current video attribute}
procedure SubVattr(attr : byte );   { Sub attribute from current vid attribute}
procedure BrkAtt(attr : byte );     { Break attribute into extra and base     }
function  AddAtt : byte;            { Add extra and base attributes to get    }
                                    {  a resulting displayable video attribute}
procedure ChrWrite(c : char );      { Write character to the screen }
procedure SetScroll(top, bottom : integer);        { Set the scrolling region }
procedure ScrollDown;          { Move down a row scrolling if necessary }
procedure ScrollUp;            { Move up a row scrolling if necessary }
procedure SetCurs(col,row : integer); { Set the cursor to absolute coordinates}
procedure SetRelCurs(col,row : integer);
                                      { Set the cursor to relative coordinates}
procedure PosCurs;             { Position the cursor to cursx,cursy }
procedure ClearScreen;         { Clear the terminal screen }
procedure ClearEOS;            { Clear from cursor to end of screen }
procedure ClearBOS;            { Clear from cursor to top of screen }
procedure ClearEOL;            { Clear from cursor to end of line }
procedure ClearBOL;            { Clear from cursor to start of line }
procedure ClearBox(left, top, right, bottom, attr : byte);
                               { Clear a box on the video screen }

procedure MapCharSet(charset : integer );  { Map a character set }
procedure vtprintf(row, col : integer; reverse : boolean; s : string);

procedure SetCharSet(gset : integer; cs : char);  { Set a character set }
procedure SaveCursor;          { Save the cursor description }
procedure RestoreCursor;       { Restore the cursor description }
procedure SetCursorVisibility(mode : boolean);  { Set the cursor visibility mode }

procedure SetBackGround(reverse : boolean); { Set background video attribute }
procedure SetColor;            { Set the screen colors }
procedure InitTabs;            { Initialize the tab settings }
procedure DoTab;               { Perform a tab }
procedure SetTabStop;          { Set a tab stop at cursor position }
procedure ClearTabStop;        { Clear a tab stop at the cursor position}
procedure ClearAllTabs;        { Clear all the defined tab stops }
procedure SetScreenWidth(width : integer);  { Set the logical width of the screen }
procedure SaveScreen;          { Save contents of video memory }
procedure RestoreScreen;       { Restore contents of video memory }


IMPLEMENTATION

TYPE
  BytePtr = ^byte;

CONST
  VIDINT     = $10 ;    { BIOS video interrupt number }
  RETRACE    = $3da;    { Video Retrace port address for CGA }
  ASCII      = 0   ;    { ASCII character set }
  UK         = 1   ;    { UK character set }
  SPECIAL    = 2   ;    { Special character set, graphics chars }

VAR
  screentop : byte;   { Absolute top of screen }
  screenbot : byte;   { Absolute bottom of screen }
  scrolltop : byte;   { Top row of scrolling region }
  scrollbot : byte;   { Bottom row of scrolling region }

  cursx : integer;                 { X cursor position }
  cursy : integer;                 { Y cursor position }

  scroff   : word;        { Screen memory offset }
  scrseg   : word;        { Screen memory segment }
  scrchars : word;        { Number of chars written to video memory }
  tvmode   : boolean;     { Flag indicates presence of control program: WINDOWS, DESQVIEW...}

  video_state : byte;     { State of video, reversed or normal }
  scbattr     : byte;     { Video attribute of empty video cell }
  curattr     : byte;     { Video attribute of displayable chars }
  baseattr    : byte;     { Base attribute for video attributes }
  extrattr    : byte;     { Extra attribute for video attributes }

  att_reverse : byte;     { Reverse attribute bits }
  att_normal  : byte;     { Normal attribute bits }

CONST
  att_low_mask  : byte = $6;     { Low attribute mask }
  att_underline : byte = $1;     { Underlined attribute bit }
  att_intensity : byte = $8;     { Bold attribute bit }
  att_blink     : byte = $80;    { Blinking attribute bit }
  G0            : byte = ASCII;  { Character set G0 }
  G1            : byte = ASCII;  { Character set G1 }


VAR
  GL : BytePtr;      { Pointer to current mapped character set}
  columns : integer; { Columns on logical terminal screen }
  lines   : integer; { Lines on logical terminal screen }

  tabs    : array[0..131] of boolean;
  deftabs : array[0..131] of boolean;    { default tab stops, 9,17,26 .... }

CONST
  special_chars : array[0..31] of byte = (
    32,4,176,9,12,13,10,248,241,18,11,217,191,218,192,197,
    196,196,196,196,196,195,180,193,194,179,243,242,227,216,156,7);


TYPE
  SaveCursorStruct = record          { Structure to save cursor description }
    cursx, cursy : integer;          { X cursor position, column }
    GL   : BytePtr;                  { pointer to mapped character set }
    G0   : byte;                     { character set G0 }
    G1   : byte;                     { character set G1 }
    mode : boolean;                  { origin mode }
  end;

CONST
  save : SaveCursorStruct =
    (cursx : 1; cursy : 1; GL : NIL; G0 : ASCII; G1 : ASCII; mode : false);

                                { Pointer to memory allocated to hold }
  screen_save : pointer =  NIL; { the contents of video memory }

{**************************************************************************}
{**************************************************************************}

{ S T A R T S C R E E N -- Start a direct screen addressing "transaction" }

procedure StartScreen(row,col : integer);
var
  vscroff, vscrseg : word;
  r : registers;
begin
  if ( vidmode = 7 ) then         { If in Monochrome video mode     }
    scrseg := $b000               { Assume video segment of B000 }
  else                            { If other mode then assumme }
    scrseg := $b800;              { video memory segment is B800 }

  scroff :=((row*columns)+col)*2; { Calculate offset from beginning of }
                                  { screen memory }
  scrchars := 0;                  { Initialize count of characters written }

                                  { Query the address of video memory for }
  r.es := scrseg;                 { this process under Windows, DesQview, ... }
  r.di := 0;                      { ES:DI := assumed screen address }
  r.ah := $FE;                    { Issue Video interrupt function FE }
  intr(VIDINT,r);                 {  in order to check for control program }
  vscroff := r.di;                { ES:DI now holds actual address of screen}
  vscrseg := r.es;
  tvmode  := false;               { Assume no control program }
  if (vscroff <> 0) or (vscrseg <> scrseg) then begin
    scrseg := vscrseg;            { If address of screen is different from}
    scroff := scroff + vscroff;   { assumed address then a control program}
    tvmode := true;               { is present and has remapped the screen}
  end
  else if retracemode then begin
    repeat until (port[RETRACE] and $8 = 0  ); { Wait until refreshing      }
    repeat until (port[RETRACE] and $8 <> 0 ); { Wait for end of refresh    }
    port[$3d8] := $25;                         { Turn off the screen refresh}
  end;
end;
 

{ E N D S C R E E N -- End a direct screen addressing "transaction" }
 
procedure EndScreen;
const
  modeset : array[0..7] of byte = ($2C,$28,$2D,$29,$2A,$2E,$1E,$29);
            { Values to turn the screen back on }
            { after video refresh is turned off }
var
  r : registers;
begin
  if tvmode and (scrchars > 0) then begin
    r.es := scrseg;             { Point ES:DI to beginning of video memory}
    r.di := scroff;             { to update }
    r.cx := scrchars;           { CX holds the number of chars to update }
    r.ah := $FF;                { Use function FF of BIOS video interrupt }
    intr(VIDINT,r);             {  to copy video buffer to screen }
  end
  else if retracemode then { Else if screen is off turn it back on }
    port[$3d8] := modeset[vidmode];
end;


{ W R I T E O N E C H A R -- writes on character to video memory      }
{ NOTE: this function does not shut down video refresh in order       }
{ to aprocedure snow on CGA, so it is *much* faster than using StartScreen }
{ and an EndScreen transaction for one character screen writes.       }

procedure WriteOneChar(c : char; row, col : integer);
var
  vscroff, vscrseg, vc : word;
  r : registers;
begin
  if ( vidmode = 7 ) then   { If in Monochrome video mode     }
    scrseg := $b000         { Assume video segment of B000 }
  else                      { If other mode then assumme }
    scrseg := $b800;        { video memory segment is B800 }

  vc := ord(c) or (curattr shl 8);  { char/attribute to write to screen  }
  scroff :=((row*columns)+col)*2;   { Calculate offset from beginning of }
                                    { screen memory }
                                    { Query the address of video memory for }
  r.es := scrseg;                   { this process under Windows, DesQview, ... }
  r.di := 0;                        { ES:DI := assumed screen address }
  r.ah := $FE;                      { Issue Video interrupt function FE }
  intr(VIDINT,r);                   {  in order to check for control program }
  vscroff := r.di;                  { ES:DI now holds actual address of screen}
  vscrseg := r.es;
  tvmode  := false;                 { Assume no control program }
  if (vscroff <> 0 ) or (vscrseg <> scrseg) then begin
    scrseg := vscrseg;           { If address of screen is different from}
    scroff := scroff + vscroff;  { assumed address then a control program}
    tvmode := true;              { is present and has remapped the screen}
  end
  else if retracemode then begin
    repeat until (port[RETRACE] and $8 = 0  ); { Wait until refreshing      }
    repeat until (port[RETRACE] and $8 <> 0 ); { Wait for end of refresh    }
  end;
  memw[scrseg:scroff] := vc;
  if tvmode then begin         { If control program video update required}
    r.es := scrseg;            { Point ES:DI to beginning of video memory}
    r.di := scroff;            {  to update }
    r.cx := 1;                 { CX holds the number of chars to update }
    r.ah := $FF;               { Use function FF of BIOS video interrupt }
    intr(VIDINT,r);            { to copy video buffer to screen }
  end;
end;


{ V T P R I N T F -- print a string to the video screen }

procedure vtprintf(row, col : integer; reverse : boolean; s : string);
var
  attr,i,so : word;
begin
  if reverse then               { If reversed attribute specified }
    attr := att_reverse
  else                          { Else use normal attribute }
    attr := att_normal;
  attr := attr shl 8;
  StartScreen(row,col);         { Start a screen update }
  so := scroff;
  for i := 1 to length(s) do begin
    memw[scrseg:so] := attr or ord(s[i]);
    inc(so,2);
  end;
  inc(scrchars,length(s));
  EndScreen;                  { End the screen update }
end;

{ V I D I N I T  -- Initialize the video system }

procedure VidInit;
var
  r : registers;
begin
  GL := @G0;
  save.GL := @G0;
  r.ah := $0F;                  { Use function F of interrupt 10 }
  intr(VIDINT,r);               { Issue BIOS video interrupt }
  vidmode := r.al;              { Save the video mode }
  columns := r.ah;              { Save the number of columns }
  lines := 25;                  { Lines = 25, (sorry no 43,50 lines }
  screenbot := pred(lines);     { Bottom of screen is 24 }
  screentop := 1;               { Top of screen is line 1 }
  tvmode := false;              { Assume no control program present }

  if not InitFileFound then begin     { If no video parameters are saved }
                                      { then determine default values }
    { Assume video adapter is snowy if it is not a MDA/HGC
    }
    if (vidmode <> 7) then retracemode := true;
    r.ax := $1A00;
    intr(VIDINT,r);
    if (r.al = $1A) then       { If VGA is detected }
      retracemode := false     { No snow protection needed }
    else begin
      r.cl := $C;              { Test the Video BIOS to see if }
      r.bx := $FF10;           {  an EGA can be detected }
      r.ax := $1200;           {  EGA's don't have "snow" problem either}
      intr(VIDINT,r);          { Issue BIOS video interrupt }
      { If EGA is detected, check BIOS data to see if its active }
      retracemode := (r.cl < $C) and ( mem[$40:$87] and $8 = 0);
      { No snow protection required }
    end;
    r.ah := $8;         { Issue function 8 on interrupt 10 }
    r.bh := $0;         { for page 0 }
    intr(VIDINT,r);     { Get video attribute at cursor pos }
    scbattr := r.ah;     { Save this attribute }
    forecolor := scbattr and $f;  { Save values for determined colors }
    backcolor := scbattr shr  4;
  end
  else begin { If saved values were available use saved color values }
    scbattr := ( backcolor shl 4 ) or forecolor;
    { Do not let retracemode be set on if video is a MDA/HGC }
    if (vidmode = 7) then retracemode := false;
  end;
  att_normal := scbattr;
  BrkAtt(scbattr);            { Break the attribute into base,extra }
                              { Reverse the foreground and background}
  baseattr := ( baseattr shr 4) or (baseattr shl 4);
  att_reverse := AddAtt;       { Put the attributes back together }
                               { in order to get reverse attribute }

                               { Clear screen to established attribute}
  r.ax := $600;
  r.bh := scbattr;
  r.cx := 0;
  r.dh := lines;
  r.dl := pred(columns);
  intr(VIDINT,r);
  vtprintf(0,0,true,modestr);     { Display the mode line in reverse }
  if (screen_save = nil) then begin    { If first time to be initialized }
                                        { Attempt to allocate screen mem }
    getmem(screen_save,lines*columns*2);
    if screen_save = nil then
       badexit('No memory available for video screen save buffer');
  end;
end;


{ S E T V A T T R  --  Set the video attribute }

procedure SetVattr(attr : byte);
begin
  video_state := 0;             { Reset the video state }
  BrkAtt(scbattr);              { Break apart the default screen attribute}
  case attr of                  { See what video attribute is requested }
    BLINK : { Blinking characters }
      extrattr  := att_blink;
    REVERSE : { Reversed video characters }
      begin
        video_state  := 1;
        baseattr  := ( baseattr shr 4) or (baseattr shl 4);
      end;
    UNDERLINED : { Underlined characters }
      if (vidmode = $7) then  { Monochrome can underline }
        extrattr  := att_underline
      else begin                { others can't use reverse video }
        video_state  := 1;
        baseattr  := ( baseattr shr 4) or (baseattr shl 4);
      end;
    BOLD : { High intensity, bold, characters }
      extrattr  := att_intensity;
  else if attr <> NORMAL then
    extrattr := 0;
  end;
  curattr := AddAtt;           { Put the video attributes back together }
end;



{ A D D V A T T R  --  Add an attribute bit to the current video attribute }

procedure AddVattr(attr : byte);
begin
  BrkAtt(curattr);     { Break apart the current video attribute}
  case attr of         { See what attribute wants to be added }
    BLINK : { Blinking attribute }
      extrattr := extrattr or att_blink;
    BOLD :  { High intensity, bold, attribute }
      extrattr := extrattr or att_intensity;
    REVERSE : { Reversed attribute }
      if (video_state = 0) then begin
        video_state := 1;
        baseattr := ( baseattr shr 4) or (baseattr shl 4);
      end;
    UNDERLINED:  { Underlined characters }
      if (vidmode = $7) then  { Monochrom can underline }
        extrattr := att_underline
      else if (video_state = 0) then begin
        { others cant use reversed video }
        video_state := 1;
        baseattr := ( baseattr shr 4 ) or (baseattr shl 4);
      end;
  end;
  curattr := AddAtt;           { Put the video attributes back together }
end;

{ S U B V A T T R  --  Remove attribute bit to the current video attribute }

procedure SubVattr(attr : byte);
begin
  BrkAtt(curattr);           { Break apart the current video attribute}
  case attr of               { See what video attribute to remove }
    BLINK : { Remove the blinking attribute }
      extrattr := extrattr and not att_blink;
    BOLD : { Remove the high intensity, bold }
      extrattr := extrattr and not att_intensity;
    REVERSE : { Remove reversed attribute }
      if (video_state = 1) then begin
        video_state := 0;
        baseattr := ( baseattr shr 4) or ( baseattr shl 4);
      end;
    UNDERLINED : { Remove underlined attribute }
      if (vidmode = $7) then   { Monochrome could have underlined }
        extrattr := extrattr and not att_underline
      else if (video_state = 1) then begin
        { others couldn't remove reverse attribute}
        video_state := 0;
        baseattr := ( baseattr shr 4) or (baseattr shl 4);
      end;
  end;
  curattr := AddAtt;  { Put the video attributes back together }
end;



{ B R K A T T R -- Break an attribute into its video components }

procedure BrkAtt(attr : byte);
begin
  extrattr := 0;             { Clear extra attributes }
  baseattr := attr;          { Start specified base attribute }
  if (vidmode = $7) then begin          { If a Monochrome monitor }
     if (attr and att_low_mask <> 0) then { Any Low mask attributes on? }
        baseattr := baseattr or att_normal     { if yes then set normal bits on }
     else begin                         { else check other attributes }
        if (attr and att_underline <> 0 ) then begin { Underline attribute ? }
           extrattr := extrattr or att_underline;   { yes then set underline bit }
           if (attr and $70 <> 0) then       { Reverse video ? }
              baseattr := baseattr and not att_underline  { If yes then clear underline }
           else                            { monochrome can't do both }
              baseattr := baseattr or att_normal;      { Else set normal bits on }
        end;
     end;
  end;

  if (baseattr and att_intensity <> 0) then  { If bold attribute is on }
     extrattr := extrattr or att_intensity;  { then set intensity bit }

  if (baseattr and att_blink <> 0) then      { If blink attribute is on }
     extrattr := extrattr or att_blink;      { then set blink bit }

                                 { Turn off blink,bold in base attribute }
  baseattr := baseattr and not (att_intensity or att_blink);
end;


{ A D D A T R -- Build video attribute from base and extra attributes }

function AddAtt : byte;
begin
  if (extrattr and att_underline <> 0 ) then  { If underline is requested }
    baseattr := baseattr and not att_low_mask;    { Clear low mask }
  AddAtt := baseattr or extrattr;     { return the or'ed attributes }
end;


{ C H R W R I T E  -- Write a character to a row and column of the screen }
 
procedure ChrWrite(c : char);
var
  ch, attr : array[0..1] of byte;
  row, ws : integer;
begin
  { Check character set being used
  { if regular ASCII then char is OK
  }
  if (GL^ = SPECIAL) and (c > #94) and (c < #128) then
    c := chr(special_chars[ord(c) - 95]) { translate graphics characters }
  else if (GL^ = UK) and (c = '#') then
    c := '';                             { translate British pound }

  { NOTE:  Inserting a character using this technique is *very* slow
  { for snowy CGA systems
  }
  if (insertmode) then begin     { If insert mode, scoot rest of line over }
    StartScreen(cursy,cursx-1);  { Start direct video memory access        }
    move(mem[scrseg:scroff],mem[scrseg:scroff+2],(columns-cursx)*2);
    inc(scrchars,columns-cursx);
    EndScreen;                   { Update screen in control programs      }
  end;

  if (cursx > screenwid) then begin  { If trying to go beyond the screen width }
   if autowrap then begin             { when autowrap is on }
     ScrollUp;                { scroll the screen up }
     SetCurs(1,0);            { set cursor to column 1 of next line }
   end
   else
     cursx := screenwid;      { else put the cursor on right margin }
  end;
  WriteOneChar(c, cursy, cursx - 1);
  inc(cursx);                   { Increment the cursor X position }
  PosCurs;                      { Move the cursor to the new position }
end;


{ S E T S C R O L L  -- Establish the scrolling boundaries }

procedure SetScroll(top, bottom : integer);
begin
  if (top = 0) then      { If the top scroll boundary is 0 }
    top := 1;            { interpret this as the top screen row }
  if (bottom = 0) then   { If the bottom scroll boundary is 0 }
    bottom := screenbot; { interpret this as bottom screen row }
  if (top > 0) and (top <= screenbot)
  and ( bottom >= top) and (bottom <= screenbot) then begin
    scrolltop := top;            { save top boundary }
    scrollbot := bottom;         { save bottom boundary }
    SetCurs(1,1);               { this also homes the cursor }
  end;
end;

{ I N D E X D O W N  -- Scroll the screen down }

procedure IndexDown;
var
  r : registers;
begin
  r.ax := $0701;           { Call the BIOS to scroll down 1 line }
  r.bh := scbattr;         { Get the attribute for new line }
  r.ch := scrolltop;       { upper left corner }
  r.cl := 0;
  r.dh := scrollbot;       { lower right corner }
  r.dl := pred(columns);
  intr(VIDINT,r);
  PosCurs;                 { Position the cursor }
end;

{ I N D E X U P  -- Scroll the screen up }

procedure IndexUp;
var
  r : registers;
begin
  r.ax := $0601;           { Call the BIOS to scroll up 1 line }
  r.bh := scbattr;         { Get the attribute for new line }
  r.ch := scrolltop;       { upper left corner }
  r.cl := 0;
  r.dh := scrollbot;       { lower right corner }
  r.dl := pred(columns);
  intr(VIDINT,r);
  PosCurs;                 { Position the cursor }
end;


{ S C R O L L D O W N  -- Move up a row scrolling if necessary }

procedure ScrollDown;
begin
 if (cursy = scrolltop) then  { If on the top of the scrolling region }
   IndexDown                  { scroll the rest of the region down }
 else begin                   { Else }
   dec(cursy);                { just decrement cursor Y position }
   PosCurs;                   { and request the reposition }
 end;
end;

 
{ S C R O L L U P  -- Move down a row scrolling if necessary }

procedure ScrollUp;
begin
  if (cursy = scrollbot) then  { If on the bottom of the scrolling region}
    IndexUp                    { scroll the rest of the region down }
  else begin                   { Else }
    inc(cursy);                { just increment the cursor Y position }
    PosCurs;                   { and request the reposition }
  end;
end;


{ S E T C U R S -- Set absolute cursor position on the logical screen }

procedure SetCurs(col, row : integer);
begin
  if (col = 0) then             { If called with X coordinate := zero }
    col := cursx;               {  then default to current coordinate }
  if (row = 0) then             { If called with Y coordinate := zero }
    row := cursy;               {  then default to current coordinate }

  if (originmode) then begin      { If origin mode is relative }
    inc(row,pred(scrolltop));     { adjust the row }
    if (row < scrolltop) or (row > scrollbot) then
      exit;                      { Can not position cursor out of scroll }
  end;                           { region in relative cursor mode }
                                 { Can only position the cursor if it lies }
                                 { within the logical screen limits }
  if (col <= screenwid) and (row <= screenbot) then begin
    cursx := col;               { Set the X cursor coordinate, column }
    cursy := row;               { Set the Y cursor coordinate, row }
    PosCurs;                    { Request the physical positioning }
  end;
end;


{ S E T R E L C U R S -- Set relative curs pos on the logical screen }

procedure SetRelCurs(col,row : integer);
begin
  if (col = 0) then           { If called with X coordinate := zero }
     col := cursx             { then default to current X coordinate }
   else                       { Else }
     inc(col,cursx);          { add col value to X cursor position }
                              { Note:  col can be negative }

  if (row = 0) then           { If called with Y coordinate := zero }
    row := cursy              { then default to current Y coordinate }
  else                        { Else }
    inc(row,cursy);           { add row value to Y cursor position }
                              { Note:  row can be negative }

  if (originmode) then begin  { If origin mode is relative }
    inc(row,pred(scrolltop)); { adjust the row }
    if (row < scrolltop) or (row > scrollbot) then
      exit;                   { Can not position cursor out of scroll }
   end;                       { region in relative cursor mode }

                                   { Can only position the cursor if it lies }
                                   { within the logical screen limits }
   if (col > 0) and (col <= screenwid)
   and (row > 0) and (row <= screenbot) then begin
     cursy := row;                { Set the X cursor coordinate, column }
     cursx := col;                { Set the Y cursor coordinate, row }
     PosCurs;                     { Request the physical positioning }
   end;
end;

{ P O S C U R S -- Position the cursor on the physical screen }

procedure PosCurs;
var
  col, row : integer;
  r : registers;
begin
  col := cursx;
  row := cursy;
  if (col > columns) then          { Check validity of requested column }
    col := columns;                { ok if column is within screen bounds }
                                   { else put cursor on the right bound }

  if (row > lines) then            { Check validity of requested row }
    row := lines;                  { ok if row is within screen bounds }
                                   { else put cursor on the bottom }

  if (cursorvisible) then begin    { Only position the cursor if its visible }
    r.ah := 2;
    r.bh := 0;
    r.dh := row;
    r.dl := pred(col);
    intr(VIDINT,r);
  end;
end;


{ C L E A R B O X -- Clear a window on the screen with the specified attr }

procedure ClearBox(left, top, right, bottom, attr : byte);
var
  r : registers;
begin
  { Use BIOS scroll window function to clear}
  r.ax := $0600;           { Call the BIOS to scroll entire window }
  r.bh := scbattr;         { Get the attribute for new line }
  r.ch := top;             { upper left corner }
  r.cl := pred(left);
  r.dh := bottom;          { lower right corner }
  r.dl := pred(right);
  intr(VIDINT,r);
end;


{ C L E A R S C R E E N -- Clear the screen setting it to a normal attr }

procedure ClearScreen;
begin
  ClearBox(1, screentop, columns, screenbot, scbattr);
end;
 
{ C L E A R E O S -- Clear from the cursor to the end of screen }

procedure ClearEOS;
begin
  ClearEOL;                   { First clear to the End of the Line }
  if (cursy < screenbot) then { Then clear every line below it }
    clearbox(1,cursy + 1,columns, screenbot, scbattr);
end;
 
{ C L E A R B O S -- Clear from the cursor to the beggining of screen }

procedure ClearBOS;
begin
  ClearBOL;                     { First clear to the Beginning of the Line }
  if (cursy > screentop) then   { Then clear every line above it }
    ClearBox(1,screentop,columns, cursy - 1, scbattr);
end;
 
{ C L E A R E O L -- Clear to the end of the current line }

procedure ClearEOL;
begin
  ClearBox(cursx, cursy, columns, cursy, scbattr );
end;


{ C L E A R B O L -- Clear to the beginning of the current line }

procedure ClearBOL;
begin
  ClearBOX(1, cursy, cursx, cursy, scbattr );
end;




{ M A P C H A R S E T -- Map an established character set to current }

procedure MapCharSet(charset : integer);
begin
  if (charset = 0) then        { If mapping G0 character set }
    GL := @G0                  { Point the current char set,GL to G0 }
  else if (charset = 1) then   { If mapping G1 character set }
    GL := @G1;                 { Point the current char set,GL, to G1 }
end;


{ S E T C H A R S E T -- Establish a character set }

procedure SetCharSet(gset : integer; cs : char);
var
  charset : ^byte;
begin
  if (gset = 0) then                { Check to see what character set is }
    charset := @G0                 { going to be set }
  else if (gset = 1) then
    charset := @G1
  else
    exit;                         { If not valid set then return }
  case cs of
    'B' : { 'B' maps the character set to ASCII }
         charset^ := ASCII;   { this is the normal character set }
    'A' : { 'A' maps the character set to UK }
         charset^ := UK;      { only difference between UK and ASCII is  }
    '0' : { '0' maps the character set to SPECIAL }
         charset^ := SPECIAL; { this character set is the 'graphics' }
                              { character set used for line drawing }
  end;
end;

{ S A V E C U R S O R  --  Save the cursor description into memory }

procedure SaveCursor;
begin
   save.cursx := cursx;              { Save the X cursor position }
   save.cursy := cursy;              { Save the Y cursor position }
   save.GL := GL;                    { Save the current mapped character set }
   save.G0 := G0;                    { Save G0 character set }
   save.G1 := G1;                    { Save G1 character set }
   save.mode := originmode;          { Also save the origin mode }
end;


{ R E S T O R E C U R S O R  --  Restore the cursor description from memory }

procedure RestoreCursor;
begin
   cursx := save.cursx;              { Restore the saved X cursor position }
   cursy := save.cursy;              { Restore the saved Y cursor position }
   GL := save.GL;                    { Restore the saved mapped character set }
   G0 := save.G0;                    { Restore the saved G0 character set }
   G1 := save.G1;                    { Restore the saved G1 character set }
   originmode := save.mode;          { Also restore the saved origin mode }
   PosCurs;                          { Then reposition the cursor }
end;


{ S E T C U R S O R V I S I B I L I T Y -- Show/Hide the cursor }

procedure SetCursorVisibility(mode : boolean);
var
  r : registers;
begin
  cursorvisible := mode;
  if (mode) then    { If visible cursor is specified, then the }
    SetCurs(0,0)    { cursor will be shown at the current position } 
   else begin       { Else the cursor will not appear on the screen }
     r.ah := 2;
     r.bh := 0;
     r.dh := lines;
     r.dl := 0;
     intr(VIDINT,r);
   end;
end;


{ S E T B A C K G R O U N D -- Set the background attribute }

procedure SetBackGround(reverse : boolean);
var
  i,so : integer;
begin
  { only do reversal of screen attributes only if
  { reverse = true and reversebackground = false
  { or
  { reverse = false and reversebackground = true
  }
  if reverse <> reversebackground then begin
    StartScreen(0,0);
    so := scroff+1;  {point to first attribute}
    scrchars := lines*columns;
    for i := 1 to scrchars do begin
      BrkAtt(mem[scrseg:so]);
      baseattr := ( baseattr shr 4) or (baseattr shl 4);
      mem[scrseg:so] := AddAtt;
      inc(so,2);
    end;
    EndScreen;
    BrkAtt(scbattr);              { reverse the default character attr }
    baseattr := ( baseattr shr 4) or (baseattr shl 4);
    scbattr := AddAtt;
    BrkAtt(curattr);              { reverse the current character attr }
    baseattr := ( baseattr shr 4) or (baseattr shl 4);
    curattr := AddAtt;
  end;
  reversebackground := reverse;
end;



{ S E T C O L O R -- Set the screen color }

procedure SetColor;
var
  i : integer;
  attr : byte;
  so : word;
begin
  BrkAtt(att_normal);            { Break apart the current screen color }
  attr := baseattr;               { Save this attribute }

  { Create the new screen attribute }
  scbattr := (backcolor shl 4) or forecolor;
  att_normal := scbattr;
  curattr    := scbattr;
  BrkAtt(att_normal);        { and the new reverse attribute }
  baseattr   := ( baseattr shr 4) or (baseattr shl 4);
  att_reverse := AddAtt;

  StartScreen(0,0);
  so := scroff+1;  {point to first attribute}
  scrchars := lines*columns;
  for i := 1 to scrchars do begin
    BrkAtt(mem[scrseg:so]);
    if (baseattr = attr) then   { If this chars base attributes are the }
      baseattr := att_normal    { old screen color then this is a normal }
    else                        { character so set it to new normal attr }
      baseattr := att_reverse;  { Else set this character to new reverse }
    mem[scrseg:so] := AddAtt;
    inc(so,2);
  end;
  EndScreen;
end;


{ I N I T T A B S -- Initialize Tab stops to default settings }

procedure InitTabs;
var
  i : integer;
begin
  fillchar(deftabs,sizeof(deftabs),0);  { Zero indicates no tab here }
  i := 8;
  while i <= 131 do begin
    deftabs[i+1] := true;     { 1 indicates tab here, 9, 17, 25 ... }
    inc(i,8);
  end;
end;


{ D O T A B -- Perform a tab }
 
procedure DoTab;
var
  i : integer;
begin
  i := succ(cursx);
  while (i <= screenwid) and (not tabs[i]) do inc(i);
  if i <= screenwid then SetCurs(i,cursy);
end;

{ S E T T A B S T O P  -- set a tab stop at the current cursor position }
 
procedure SetTabStop;
begin
  tabs[cursx] := true;   { Mark current cursor position as tab stop}
end;

{ C L E A R T A B S T O P  -- clear a tab stop at the current curs position }

procedure ClearTabStop;
begin
  tabs[cursx] := false;  { Clear current cursor position tab stop }
end;

{ C L E A R A L L T A B S  -- clear all tab stops }

procedure ClearAllTabs;
begin
  fillchar(tabs,sizeof(deftabs),0);  { Zero indicates no tab here }
end;


{ S E T S C R E E N W I D T H -- set the screen width }

procedure SetScreenWidth(width : integer);
begin
   if (width = 132) then      { When the screen is set to 132 columns }
      screenwid := 132        { set the logical right boundary }
   else if (width = 80) then  { Else if the screen width is 80 }
      screenwid := 80;        { set the logical right boundary }

                                 { Setting the screen width also }
   ClearScreen;                  { Clears the screen }
   originmode := false;          { Sets the origin mode to absolute }
   SetScroll(0,0);               { Resets the scrolling region }
   SetCurs(1,1);                 { Sets the cursor to the home position }
end;




{ S A V E S C R E E N -- Save the contents of the video screen }

procedure SaveScreen;
var
  r : registers;
begin
  StartScreen(0,0);           { Start the video direct access }
  move(mem[scrseg:scroff],screen_save^,lines*columns*2);
  scrchars := 0;              { Zero characters were written to screen }
  EndScreen;                  { End the screen access }
  r.ah := 2;                  { Hide the cursor }
  r.bh := 0;
  r.dh := lines;
  r.dl := 0;
  intr(VIDINT,r);
end;

{ R E S T O R E S C R E E N -- Restore contents of the video screen }

procedure RestoreScreen;
begin
  StartScreen(0,0);           { Start the video direct access }
  move(screen_save^,mem[scrseg:scroff],lines*columns*2);
  scrchars := lines*columns;  { number of characters written to screen }
  EndScreen;                  { End the screen access }
  PosCurs;                    { Reposition the cursor }
end;

end.

