UNIT COMIO;

{$I DEFINES.INC}

INTERFACE

uses dos,fileio;

procedure ttinit;                 { Initialize the communications system    }
function ttopen : integer;        { Open a port for communications          }
function ttclose : integer;       { Close the communications port           }
function ttchk : integer;         { Return count of received characters     }
procedure ttoc(c : char);         { Output a character to the com port      }
function ttinc : char;            { Input a character from circular buffer  }

IMPLEMENTATION

CONST
  MDMDAT1 = $03F8;           { Address of modem port 1 data               }
  MDMSTS1 = $03FD;           { Address of modem port 1 status             }
  MDMCOM1 = $03FB;           { Address of modem port 1 command            }
  MDMDAT2 = $02F8;           { Address of modem port 2 data               }
  MDMSTS2 = $02FD;           { Address of modem port 2 status             }
  MDMCOM2 = $02FB;           { Address of modem port 2 command            }
  MDMINTV = $000C;           { Com 1 interrupt vector                     }
  MDINTV2 = $000B;           { Com 2 interrupt vector                     }
  MDMINTO = $00EF;           { Mask to enable IRQ3 for port 1             }
  MDINTO2 = $00F7;           { Mask to enable IRQ4 for port 2             }
  MDMINTC = $0010;           { Mask to Disable IRQ4 for port 1            }
  MDINTC2 = $0008;           { Mask to Disable IRQ3 for port 2            }
  INTCONT = $0021;           { 8259 interrupt controller ICW2-3           }
  INTCON1 = $0020;           { Address of 8259 ICW1                       }

  COM_BUFF_SIZE = 1024;            { Communications port buffer size      }
  XOFFPT  = COM_BUFF_SIZE*3 div 4; { chars in buff before sending XOFF    }
  XONPT   = COM_BUFF_SIZE*1 div 4; { chars in buff to send XON after XOFF }
  XOFF    = #$13;                  { XOFF value                           }
  XON     = #$11;                  { XON value                            }


VAR
  buffer : array[0..COM_BUFF_SIZE-1] of char;          { Circular buffer }
  inptr  : integer;           { Pointer to input point of circular buff  }
  outptr : integer;           { Pointer to output point of circular buff }
  count  : integer;           { Number of characters in buffer           }

  modem : record              { struct to hold current com port info }
    mddat  : word;            { 8250 data register }
    mdstat : word;            { 8250 line-status register }
    mdcom  : word;            { 8250 line-control register }
    mden   : byte;            { 8259 IRQ enable mask }
    mddis  : byte;            { 8259 IRQ disable mask }
    mdintv : byte;            { Interrupt for selected com port }
 end;

 oldvec : pointer;

CONST
  portin : boolean = false;   { Flag to indicate com port is open }
  xofsnt : boolean = false;   { Flag to indicate an XOFF transmitted }
  xofrcv : boolean = false;   { Flag to indicate an XOFF received }

{***************************************************************************}

{ M A C R O S }

procedure DisableInterrupts; inline($FA {cli} );
procedure EnableInterrupts;  inline($FB {sti} );


{  S E R I N T  -- Serial interrupt handler, recieves incoming characters }

procedure serint; interrupt;
begin
  buffer[inptr] := chr(port[modem.mddat]); { Quickly read arriving character }
  inc(inptr);
  inc(count);                         { Increment received count }
  if (count > XOFFPT) and not xofsnt then begin { buffer almost full }
    ttoc(XOFF);                                 { send an XOFF }
    xofsnt := true;                { and save XOFF sent status }
  end;
  disableInterrupts;              { NO interrupts are allowed while }
                                  { new input pointer is stored }
  if (inptr = COM_BUFF_SIZE) then { At end of circular buff? }
    inptr := 0;                   { if so then save new output point }

  enableInterrupts;               { Interrupts ok now }
  port[$20] := $20;               { send End Of Interrupt to 8259 }
end;

{ D O B A U D  --  Set the baud rate for the current port }

function dobaud(baudrate : word) : integer;
var
  portval, blo, bhi : byte;
begin
  case baudrate of  { Get 8250 baud rate divisor values }
     50:     begin bhi := $9;  blo := $00;  end;
     75:     begin bhi := $6;  blo := $00;  end;
     110:    begin bhi := $4;  blo := $17;  end;
     150:    begin bhi := $3;  blo := $00;  end;
     300:    begin bhi := $1;  blo := $80;  end;
     600:    begin bhi := $0;  blo := $C0;  end;
     1200:   begin bhi := $0;  blo := $60;  end;
     1800:   begin bhi := $0;  blo := $40;  end;
     2000:   begin bhi := $0;  blo := $3A;  end;
     2400:   begin bhi := $0;  blo := $30;  end;
     4800:   begin bhi := $0;  blo := $18;  end;
     9600:   begin bhi := $0;  blo := $0C;  end;
     19200:  begin bhi := $0;  blo := $06;  end;
     38400:  begin bhi := $0;  blo := $03;  end;
  else begin
    dobaud := -1;
    exit;
  end;
  end;
  portval := port[modem.mdcom];   { Save current value of command register }
                                 { In order to set the baud rate the      }
                                 { high bit of command data register is   }
  port[modem.mdcom] := portval or $80;   { set before sending baud data   }

  port[modem.mddat] := blo;      { Set LSB Baud-Rate divisor for baud }
  port[modem.mddat + 1] := bhi;  { Set MSB Baud-Rate divisor for baud }

  port[modem.mdcom] := portval;  { Reset original command register value }

  dobaud := 0;                   { Return success }
end;


{  C O M S  --  Set up the modem structure for the specified com port }

procedure coms(portid : byte);
begin
  if (portid = 1) then begin     { Port data for COM 1 }
    modem.mddat := MDMDAT1;       { Port 1 Data register }
    modem.mdstat := MDMSTS1;      { Port 1 Status register }
    modem.mdcom := MDMCOM1;       { Port 1 Command register }
    modem.mddis := MDMINTC;       { Port 1 8259 IRQ4 disable mask }
    modem.mden := MDMINTO;        { Port 1 8259 IRQ4 enable mask }
    modem.mdintv := MDMINTV;      { Port 1 interrupt number }
  end
  else if (portid = 2) then begin { Port data for COM 2 }
    modem.mddat := MDMDAT2;       { Port 2 Data register }
    modem.mdstat := MDMSTS2;      { Port 2 Status register }
    modem.mdcom := MDMCOM2;       { Port 2 Command register }
    modem.mddis := MDINTC2;       { Port 2 8259 IRQ4 disable mask }
    modem.mden := MDINTO2;        { Port 2 8259 IRQ4 enable mask }
    modem.mdintv := MDINTV2;      { Port 2 interrupt number }
  end;
end;


{ S E R I N I  -- initialize the serial port for interrupts }

procedure serini;
var
  portval : byte;
begin
  if not portin then begin       { Ignore if already open }
    portin := true;               { save port open status }
    inptr := 0;
    outptr := 0;                  { set circular buffer pointers }
    count := 0;                   { indicate no characters received }
    getintvec(modem.mdintv,oldvec);  { save old com interrupt }
    setintvec(modem.mdintv,@serint); { set SERINT as communications ISR }

    portval := 0;             { Byte value to output to the Line }
                              { Control Register (LCR) to set the }
                              { Parity, Stopbits, Databits }
                              { Start out with all bits zero }

    if (parity = 'EVEN') then
      portval := portval or $8       { Set bit 3 on for odd parity }
    else if (parity = 'ODD') then
      portval := portval or $18;     { Set bits 3 and 4 on for even parity }
                                     { Leave bits 3 and 4 off for no parity }
    if (stopbits = 2) then           { Set bit 2 on if 2 Stopbits are used }
       portval := portval or $4;     { Leave bit 2 off for 1 Stopbit }

    if (databits = 6) then           { Set bit 0 on for 6 data bits }
       portval := portval or $1
    else if (databits = 7) then      { Set bit 1 on for 7 data bits }
       portval := portval or $2
    else if (databits = 8) then      { Set bits 0 and 1 on for 8 data bits }
       portval := portval or $3;
                                  { Leave bits 0 and 1 off for 5 data bits }

    port[modem.mdcom] := portval;  { Output the settings to the LCR }

    port[modem.mdcom + 1] := $b;   { Assert OUT2, RTS, DTR }

    portval := port[modem.mddat];  { Clear any left over characters }
    port[modem.mddat+1] := $1;     { Enable receiver interrupts }

    portval := port[INTCONT];       { Read 8259 interrupt enable mask }
    port[INTCONT] := modem.mden and portval;  {Set bit on for com IRQ }
  end;
end;


{ S E R R S T -- Reset serial port interrupts }

procedure serrst;
var
  portval : byte;
begin
  if portin then begin           { Ignore if interrupts already disabled }
    portin := false;              { save port closed status }
    portval := port[INTCONT];     { Read 8259 interrupt enable mask }
    port[INTCONT] := modem.mddis or portval;     {Set bit off for com IRQ }
    setintvec(modem.mdintv,oldvec);  { return original interrupt vector }
  end;
end;


{  T T I N I T  -- Initialize the communications system }

procedure ttinit;
begin
  if not InitFileFound then begin   { If no saved values are available }
     comport   := 1;                { then set default values }
     speed     := 2400;
     parity    := 'NONE';
     databits  := 8;
     stopbits  := 1;
  end;
end;



{  T T O P E N  -- Open the communications port }

function ttopen : integer;
         {
         { ttopen = 0 if ok, -1 if error
         }
begin
  ttopen := 0;       { return success, ignoring call if already open }
  if not portin then begin
    if (comport <> 1) and (comport <> 2) then
      ttopen := -1
    else begin
      coms(comport);
      ttopen := dobaud(speed);   { Set baud rate }
      serini;          { enable interrupt handler }
    end;
  end;
end;


{  T T C L O S E --  Close the communications port  }

function ttclose : integer; {always returns 0 for success}
begin
  ttclose := 0;
  if portin then serrst;     { Ignore if port is already closed }
end;                         { otherwise disable interrupts }


{ T T C H K  --  Return a count of characters at the serial port }

function ttchk : integer;
begin
  ttchk := count;
end;


{ T T O C -- Output a character to the current serial port }

procedure ttoc(c : char);
begin
  repeat until port[modem.mdstat] and $20 <> 0; { wait until ready }
  port[modem.mddat] := ord(c);         { then output the character }
end;



{ T T F L U I  --  Clear the input buffer of characters }


procedure ttflui;
begin
  if (xofsnt) then begin         { Check if XON should be sent after XOFF }
    xofsnt := false;             { if so then reset XOFF sent status }
    ttoc(XON);                   { and send the XON }
  end;
  disableInterrupts;             { NO interrupts allowed now }
  inptr  := 0;
  outptr := 0;                   { Reset input out output pointers }
  count  := 0;                   { Set received characters count to 0 }
  enableInterrupts;              { Now interrupts are ok }
end;


{ T T I N C  -- Read a character from serial ports circular buffer }

function ttinc : char;
begin
  { Check if XON should be sent after XOFF
  }
  if (count < XONPT) and xofsnt then begin
    xofsnt := false;                  { if so then reset XOFF sent status }
    ttoc(XON);                   { and send the XON }
  end;
  repeat until count > 0;
  ttinc := buffer[outptr];        { Get this character and increment ptr }
  inc(outptr);
  if outptr = COM_BUFF_SIZE then  { See if circular buff should be wrapped }
    outptr := 0;                  { if so then save new output point }
  disableInterrupts;              { NO interrupts allowed now }
  dec(count);                     { Decrement count of received characters }
  enableInterrupts;               { Interrupts can continue now }
end;

var
  ExitSave : pointer;

procedure MyExitProc; far;
begin
  if ttClose <> 0 then ;
  ExitProc := ExitSave;
end;

begin
  ExitSave := ExitProc;
  ExitProc := @MyExitProc;
end.
