Program Mandelbrot;
{ This program will show the mandelbrot set, and then zoom in on it.       }
{ It will run for hours before it starts showing uninteresting pictures.   }
{ Pressing a numeric key will generate a new palette with n colour-ranges, }
{ where n is the key pressed. Press [Esc] to quit.                         }

{$DEFINE SWAP}    { Don't define this symbol to see the set being drawn }

Uses
  EGOF,
  VgaU,
  VesaU,
  FractalU,
  PalU,
  CRT;

Const
  Mode  = Mode320x240;       { Try substituting other ModeAAAxBBB constants }
  Esc   = #27;
  Enter = #13;

  NoParams = 2;
  Params :Array [1..NoParams] Of Record
                            Mr,Mi,
                            Jr,Ji :Double;
                          End = ((Mr:-1.4796629362739300 ; Mi: 0.0009341553491329 ;
                                  Jr: 0.0000000000000000 ; Ji: 0.0000000000000000),
                                 (Mr:-0.5254033722785989 ; Mi:+0.5224319800134103 ;
                                  Jr:-0.1997819692855134 ; Ji:-0.0791884279314515));

Var
  {$IFDEF SWAP}
  Scr :ScrMpMapP;        { Video Graphics Array Multi Page Map Pointer! }
  {$ELSE}
  Scr :ScrMapP;          { Video Graphics Array Map Pointer! }
  {$ENDIF}
  Man :MandelbrotO;
  Jul :JuliaP;
  Pal :VgaPalO;
  A,V :Byte;
  Key :Byte;
  Mr,Mi,
  Jr,Ji :Double;


Procedure ECheck (Graphic:Boolean);
{ Chck for errors and if any are found display an error message and halt    }
{ Graphic tells us weather we are in graphics mode. If so we must change to }
{ text mode before destroying the RS object                                 }
Var
  EC :ShortInt;
Begin
  If IsEgofError Then Begin
    If Graphic Then
      Scr^.Done;
    Dispose (Scr);
    EC := GetEgofError;

    { Write Error message }
    WriteLn ('Error #',EC,' : ',EgofErrorStr[EC],'.');
    WriteLn;

    { Write Error message in Icelandic }
    WriteLn ('Villa ',EC,'  : ',EgofErrorIce[EC],'.');
    Repeat Until ReadKey<>#0;
    Halt;
  End;
End;


Function Pow (A,B :Double) :Double;
Begin
  Pow := Exp(Ln(A)*B);
End;


Procedure RandomCol (NoR :Byte);
{ Create [NoR] random colour-ranges, but keep colour 0 black }
Var
  R1,G1,B1,
  Rl,Gl,Bl,
  R,G,B  :Byte;
  Ra     :Byte;
Begin
  R1 := Random (63);
  G1 := Random (63);
  B1 := Random (63);

  Rl := R1;
  Gl := G1;
  Bl := B1;

  For Ra := 1 To NoR-1 Do Begin
    R := Random (63);
    G := Random (63);
    B := Random (63);

    Pal.Range (((Ra-1)*256 Div NoR), Rl,Gl,Bl, (Ra*256 Div NoR), R,G,B);

    Rl := R;
    Gl := G;
    Bl := B;
  End;

  Pal.Range (((NoR-1)*256 Div NoR), Rl,Gl,Bl, 255, R1,G1,B1);

  Pal.SetCol (0,0,0,0);
End;


Procedure ZoomMap (F :FracMapP; I1 :Word; IM, ZM :Double);
{ F  = Fractal to zoom
  I1 = Starting Iterations
  IM = Iteration Multiplier
  ZM = Zoom Multiplier }
Var
  K   :Char;
  I   :Double;
Begin
  I := I1;
  K := #0;                                                { No key yet    }
  Repeat
    {$IFDEF SWAP}
    Scr^.SetActive (A);                        { SetActivePage              }
    Scr^.SetVisual (V);                        { SetVisualPage              }
    {$ENDIF}

    F^.SetIter (Round(I));
    F^.ShowOn (Scr,0,0);                       { Draw the mandelbrot set    }

    F^.Zoom (0.5, 0.5, ZM);                    { Zoom a bit further in      }
    I := I*IM;                                 { We need more iterations    }

    {$IFDEF SWAP}
    If Scr^.PageNr>1 Then Begin                { Swap pages?                }
      V := A;                                  { Show newly painted page    }
      A := A Xor $1;                           { New active page            }
    End;
    {$ENDIF}

    If KeyPressed Then Repeat                  { Did you say something ?    }
      K := ReadKey;
      If K In ['1'..'9'] Then
        RandomCol (Byte(K)-Byte('0'));         { New colours                }
    Until (K=Enter) Or (K=Esc);
  Until K=Esc;
End;


Function ReadNum (P:String) :Double;
Var
  C :Integer;
  S :String;
  N :Double;
Begin
  Write (P);
  GoToXY (Length(P)+2,WhereY);
  Repeat
    ClrEOL;
    ReadLn (S);
    For C := 1 to Byte(S[0]) Do          { Substitute kommas with points }
      If S[C]=',' Then
        S[C] := '.';
    Val (S,N,C);
    GoToXY (Length(P)+2,WhereY-1);
  Until (C=0);
  WriteLn;
  ReadNum := N;
End;

Begin
  WriteLn ('Pick parameter set. [1-',NoParams,']');
  WriteLn ('Hit ''0'' to enter your own parameters');
  WriteLn ('or ''Q'' to quit');

  Repeat
    Key := Byte(UpCase(ReadKey))-Byte('0');
    If Key = Byte('Q')-Byte('0') Then
      Halt;
  Until Key in [0..NoParams];

  If Key=0 Then Begin
    WriteLn;
    WriteLn ('Mandelbrot set parameters');
    Mr := ReadNum ('Real part     :');
    Mi := ReadNum ('Imaginary part:');
    WriteLn;
    WriteLn ('Julia set parameters');
    Jr := ReadNum ('Real part     :');
    Ji := ReadNum ('Imaginary part:');
  End
  Else Begin
    Mr := Params[Key].Mr;
    Mi := Params[Key].Mi;
    Jr := Params[Key].Jr;
    Ji := Params[Key].Ji;
  End;

  {$IFDEF SWAP}
  Case Mode Of
    Mode320x200,ModeNsMin..ModeNsMax :
      Scr := New (VgaNsP,Init(Mode));
    ModeVesaMin..ModeVesaMax :
      Scr := New (VesaP,Init(Mode));
  End;
  {$ELSE}
  Case Mode Of
    Mode320x200 :
      Scr := New (VgaP,Init);
    ModeNsMin..ModeNsMax :
      Scr := New (VgaNsP,Init(Mode));
    ModeVesaMin..ModeVesaMax :
      Scr := New (VesaP,Init(Mode));
    Else
      Halt;
  End;
  {$ENDIF}
  ECheck (False);

  Pal.Init;              { Initialise palette }
  Randomize;             { We're using random colours }
  RandomCol(5);

  A := 0;   { We'll start with page 0 both active and visible }
  V := 0;

  Man.Init (Scr^.XRes,Scr^.YRes,10);           { Make full screen mandelbrot }
  Man.SetPixRatio (Scr^.XRes/Scr^.YRes * 3/4); { Set pixel ratio to get circular circles  }
  Man.SetPos (Mr,Mi, 0.5);                     { Aim at an interesting spot  }
  ZoomMap (@Man, 128, 1.1, 1.5);               { Show slide of blow-up }

  Jul := Man.MakeJulia (Scr^.XRes,Scr^.YRes,100);
  Jul^.SetPos (Jr,Ji, 0.5);                    { Aim at an interesting spot }
  ZoomMap (Jul, 256, 1.08, 1.5);
  Repeat Until (ReadKey<>#0);                  { Give us a chance to see... }

  Dispose (Jul,Done);
  Man.Done;                                    { Discard of Mandelbrot set  }
  Pal.Done;                                    { Discard of palette         }
  Dispose(Scr,Done);                           { Discard of display         }
End.