/***
 *  Copyright (C) 1993   Marc Stern  (internet: stern@mble.philips.be)
 *
 *  File	: trace.h
 *
 *  Description	: General macros for tracing
 *                Print all the begining and ending of functions
 *                (indented) with return values.
 *
 *  OS/Compiler : All
 *
 *  Usage       : - Add the line BEGIN( function_name, return_format );
 *                  just after all the declaration of the function.
 *                  return_format is one of the C standard format
 *                  ( "%d", "%s", "%f", ... )
 *
 *                - To trace a value call the function 'trace( (...) );'
 *                  where (...) will be given as is to printf.
 *
 *                - Define 'TRACE' to actually perform tracing,
 *                  do not define it to suppress tracing.
 *
 ***/


#ifndef _TRACE_H
#define _TRACE_H

#include <stdio.h>
#include <stdlib.h>


#ifdef TRACE	/*  For debugging  */

extern int	G_nesting;

#define main \
	extern int G_nesting = 0; \
	main

#define trace( args ) \
	printf( "%*d : ", G_nesting, G_nesting );	\
	printf args

#define BEGIN( function, ret_format ) \
	char _FUNCTION_[] = #function;				\
/*	char _RET_FORMAT_[] = #ret_format;*/  			\
	char _RET_FORMAT_[] = ret_format;  			\
	G_nesting ++;	                                        \
	printf( "\n" );			    		        \
	trace( ("<<< Begin of function " #function "\n" ) );	\
	trace( (" << in file " __FILE__ "\n") );		\
	trace( (" << at line %d\n\n", __LINE__) )

#define exit( status ) \
	{ printf( "\n" );						\
	  trace( (">>> Exit of function %s\n", _FUNCTION_) );		\
	  trace( (" >> in file " __FILE__ "\n") );			\
	  trace( (" >> at line %d\n\n", __LINE__) );			\
	  trace( (" >> exit code : '" #status "' = %d", status) );	\
	  G_nesting --;						        \
	  exit( status );						\
	}

#define _exit( status ) \
	{ printf( "\n" );						\
	  trace( (">>> Exit of function %s\n", _FUNCTION_) );		\
	  trace( (" >> in file " __FILE__ "\n") );			\
	  trace( (" >> at line %d\n\n", __LINE__) );			\
	  trace( (" >> exit code : '" #status "' = %d", status) );	\
	  G_nesting --;						        \
	  _exit( status );						\
	}

#define return( value ) \
	{ char line[256];							\
	  printf( "\n" );							\
	  trace( (">>> Return from function %s\n", _FUNCTION_) );		\
	  trace( (" >> in file " __FILE__ "\n") );				\
	  trace( (" >> at line %d\n\n", __LINE__) );				\
	  sprintf( line, " >> return code : '" #value "' = %s\n\n", _RET_FORMAT_ ); \
	  trace( (line, value) );						\
	  G_nesting --;							        \
	  return( value );							\
	}

#define syserror( error ) \
	{ printf( "\n!!! %s\n", strerror(errno) );		\
	  printf( " !! in file     : " __FILE__ "\n" );	        \
	  printf( " !! in function : %s\n", _FUNCTION_ );	\
	  printf( " !! at line     : %d\n", __LINE__ );	        \
	  printf( " !! instruction : " #error "\n\n" );	        \
	}

#else

#define BEGIN( function, ret_format )

#define trace( args )

#define syserror( error ) \
	{ printf( "\n!!! %s\n", strerror(errno) );		\
	  printf( " !! in file     : " __FILE__ "\n" );	        \
	  printf( " !! at line     : %d\n", __LINE__ );	        \
	  printf( " !! instruction : " #error "\n\n" );	        \
	}

#endif


#endif
