/******************************************************************************
    upgrade.sql - Upgrade the Image Server's database
 
    Copyright (c) 1994-1995 Watermark Software, Inc.
 
*******************************************************************************/

declare @curError smallint

set nocount on
select @curError=0
declare @curVer char(64)
    /* currentVersion marks the state of the service:
        0.95    Image Server rev 1.0/1.1
        0.95+   schema conversion to 2.0 started, not completed
        0.95++  schema parially converted
        0.95+++ schema converted, data in database not converted
        2.00    Image Server rev 2.0
    */
select @curVer = currentVersion from service

       
if (@curVer = '0.95' or @curVer = '0.95+') 
begin
    print "upgrading Image Server's database to rev 2.0 format"
    update service set currentVersion='0.95+'



        /* documents now have version numbers and families */  
    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'version' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='document')
        alter table document
            add version int null

    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'targetID' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='document')
        execute sp_rename 'document.targetID', familyID

    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'childSecuritySeg' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='document')
        alter table document
            add childSecuritySeg int null

    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'format' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='document')
        alter table document
            add format char(32) null


        /* in the past, securityDescriptors were incorrectly called accessControlLists */
    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'accessControlList' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='document')
        execute sp_rename 'document.accessControlList', securitySegment

    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'accessControlList' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='immigrant')
        execute sp_rename 'immigrant.accessControlList', securitySegment


        /* in the past, 'cabinet' had a typographical error */
    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'cabnetLocation' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='volume')
        execute sp_rename 'volume.cabnetLocation', cabinetLocation

        /* Add the new Assimilate State field to the volume table */
    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'assimState' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='volume')
        alter table volume
            add assimState binary(1) null

        /* change the passport field "crowd" to the more descriptive name "schedVname" */
    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'crowd' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='passport')
        execute sp_rename 'passport.crowd', schedVname

        /* The immigrant table field "aliasFlag" has never been used, and is */
        /* zero in all records.  Reuse the field, renaming it to "readOnly". */
    if exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'aliasFlag' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='immigrant')
        execute sp_rename 'immigrant.aliasFlag', readOnly

        /* adding volset to the immigrant allows the Image Server's gooming
           queries to be done by traversing an index rather than creating 
           a large temporary table.                                         */
        /* spliting the old 'state' into a 'state' and 'dirty' allows all
           grooming queries to specify 'state=xx' rather than 'state!=xx'.
           The SQL Server query optimizer does a better job with equality
           than with inequality                                           */
    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'volset' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='immigrant')
    begin        
        alter table immigrant
            add volset char(32) null
        alter table immigrant
            add dirty binary(1) null
    end

        /* Drop the existing volume index and rebuild it to include volset name */
    if exists (select * from sysindexes where name = 'volumeByName')
        drop index volume.volumeByName 
    create unique clustered index volumeByName on volume(volumeSet,name)
    if @@error <> 0 goto oops1

        /* schema has been changed, mark the database */
    update service set currentVersion='0.95++'

end 
else if (@curVer = '0.95++')
    print "resuming incomplete conversion"
else if (@curVer != '2.00')
    print "unknown schema version"

goto done1

oops1:
    print "error happened"
done1:


go


    /* the upgrade script is large enough that if it is run as a single procedure,
        SQL Server will sometimes, but not always, report "insufficient memory".
        therefore, the upgrade is run as two procedures                         */
set nocount on
declare @curError smallint
select @curError=0
declare @curVer char(64)
select @curVer = currentVersion from service

       
if (@curVer = '0.95++' or @curVer = '0.95+++') 
begin
    update service set currentVersion='0.95+++'
 
        /* upon startup, the Image Server searches the database for all documents 
            that have been schduled for deletion but not yet completely deleted.
            This index makes that search faster.                                    */
    if not exists (select * from sysindexes where name = 'docByDeleteInProgress')
    begin
        create index docByDeleteInProgress on document(deleteInProgress)
        if @@error <> 0 goto oops2
    end
    
        /* this index is required for decent performance of queues */
    if not exists (select * from sysindexes where name = 'docByParentAndTLM')
    begin
        create index docByParentAndTLM on document(parent,dateMostRecentUpdate,targetClass)
        if @@error <> 0 goto oops2
    end



           /*  table of segments of potentially long values, e.g., securityDescriptors */
    if not exists (select * from sysobjects where name = 'segment' and type = 'U')
    begin
        create table segment (     segmentID int
                                             not null,
                                        size int)   /* total size of segment */
        execute sp_primarykey segment, segmentID
        create unique clustered index segmentByID on segment(segmentID)
        create index segmentBySize on segment(size)
    end

    if not exists (select * from sysobjects where name = 'segmentChunk' and type = 'U')
    begin
        create table segmentChunk(segmentID int,
                             sequenceNumber int,
                                      value varbinary(64))
        execute sp_primarykey segmentChunk, segmentID, sequenceNumber
        create unique clustered index segmentChunkBySequence on
                       segmentChunk(segmentID,sequenceNumber)
        execute sp_foreignkey segmentChunk, segment, segmentID
    end



        /* passports now must be long-lived, so they are stored in the database     */
        /* First, drop passport indexes which have changed during development.      */
    if exists (select * from sysindexes where name = 'passportByCrowd')
        drop index passport.passportByCrowd 
    if exists (select * from sysindexes where name = 'passportByName')
        drop index passport.passportByName 
    if exists (select * from sysindexes where name = 'passportBySchedVol')
        drop index passport.passportBySchedVol 

    if not exists (select * from sysobjects where name = 'passport' and type = 'U')
    begin
            /* passports remember scheduled migrations */
        create table passport(immigrantUniqueID binary(16)
                                                not null,
                                   sourceVolSet char(32)
                                                not null,
                                    sourceVname char(32)
                                                not null,
                                sourceShortName int
                                                not null,
                                       function int
                                                not null,
                                          state binary(1)
                                                not null,
                                    schedVolSet char(32)
                                                null,
                                     schedVname char(32)
                                                null,
                              destinationVolSet char(32)
                                                null,
                               destinationVname char(32)
                                                null,
                           destinationShortName int
                                                null,
                                       deadline binary(8)
                                                not null,
                                       priority int
                                                not null)
    end

        /*  Since volume names are no longer unique, add the source and scheduled
            volset name to the passport table, if they aren't already there.            */
    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'sourceVolSet' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='passport')
        alter table passport
            add sourceVolSet char(32) null  /* can't alter if "not null" */
    if not exists (select * from syscolumns col, sysobjects obj 
                   where col.name = 'schedVolSet' and col.id = obj.id and 
                         obj.type = 'U' and obj.name='passport')
        alter table passport
            add schedVolSet char(32) null

        /* Create the passport indexes */
    if not exists (select * from sysindexes where name = 'passportByName')
        create unique clustered index passportByName on 
           passport(sourceShortName,sourceVname,sourceVolSet,function,destinationVolSet)
    if not exists (select * from sysindexes where name = 'passportBySchedVol')
        create index passportBySchedVol on passport(schedVname,schedVolSet)
    if not exists (select * from sysindexes where name = 'passportByDeadline')
        create index passportByDeadline on passport(deadline)
    if not exists (select * from sysindexes where name = 'passportByUniqueID')
        create index passportByUniqueID on passport(immigrantUniqueID)


       /* this table tracks the resolutions of queue elements */
    if not exists (select * from sysobjects where name = 'resolution' and type = 'U')
    begin
        create table resolution(documentUniqueID binary(16)
                                                 not null,
                                    resolutionID binary(16)
                                                 not null)
        create unique clustered index resolutionByDocumentID on 
                resolution(documentUniqueID, resolutionID)
        create index resolutionByResolutionID on resolution(resolutionID)
    end


        /* The SQL Server Query optimizer uses only the first colume in an
           index when selecting which index to use. Thus, we rearrange the
           order of columns in the user-field index to make the most useful
           column appear first.                                             */
    if exists (select * from sysindexes where name = 'userFieldByValue')
        drop index userField.userFieldByValue 

    if not exists (select * from sysindexes where name = 'userFieldByName')
        create index userFieldByName on userField(name,value)



        /* schema has been changed, mark the database */
    update service set currentVersion='0.95+++'
end 
else if (@curVer = '0.95+++')
    print "resuming incomplete conversion"
else if (@curVer != '2.00')
    print "unknown schema version"

goto done2

oops2:
    print "error happened"
done2:
go




    /* To avoid exteremely lengthy conversions, we create a temporary
        indexes to aid conversion, and drop it conversion is complete */
declare @curVer char(64)
declare @retCode int

select @curVer = currentVersion from service
       
if (@curVer = '0.95+++') 
begin
    if not exists (select * from sysindexes where name = 'nullImmigrants')
        create index nullImmigrants on immigrant(dirty)
    select @retCode=65536+indid from sysindexes where name = 'nullImmigrants'
end
else select @retCode=65536+250

EXIT(select @retCode)
go

