; SCRIPT:        DOFEED.SC
; Version:      6.0a
; Date:          09 April, 1995  
;
; The DOFEED script contains the procedures used to generate stock buy and
; sell activity (done at random). When DOFEED first begins, it attempts to
; initialize the PRICE table -- if the PRICE table has already been
; initialized, it is assumed that another copy of FEED is running elsewhere
; on the network, and the information in the fields manipulated by Rnd is not
; zeroed out.  If the PRICE table is not yet initialized, then information
; about the latest stock trading is zeroed out (High and Low set to the
; Price, Last Activity zeroed, and so on).

; The DoFeed procedure creates random stock trading information and displays
; it as it is generated.

PROC CLOSED DoFeed()
  Usevars Curline,FirstStockrow,LastStockrow,LeftMargin,IsColorMonitor
  @ 0,0
  ?? FORMAT("w80,ac",
          "SIMULATED PRICE FEED IS RUNNING - PRESS ANY KEY TO ESCAPE")
  R=FirstStockRow - 1                           ; display trades on starting line
  COEDIT "PRICE"                                ; get the PRICE table on workspace
  WHILE NOT Charwaiting()                       ; For as long as user doesn't press key
    Zip=INT(Rand()*Nrecords(Table()))           ; Generate random record #       in table
    MOVETO RECORD Zip + 1                       ; And move to that record
    Retry = FALSE                               ; Don't want to retry just yet
    LOCKRECORD                                  ; Can we get the record?
    WHILE NOT retval                            ; no, so we must
      Retry = TRUE                              ;   keep reminder that we need to retry
      SLEEP 200                                 ;   pause a moment before retrying
      LOCKRECORD                                ;   and try to get the record again
      PutMessage("*")                           ;   display an asterisk to show we're retrying
    ENDWHILE

; If we get here, we locked record successfully

    IF (Retry) THEN ClearMessage() ENDIF        ; If we had to retry, erase asterisk
    while(true)
       ChangeDecimal = .125 * Int(Rand()*8)     ; Make the changes to be 1/8 of the dollar
       ChangeInt = Int(Round((Rand()-.5)*[Price],3))       ; Generate a random price
       if ChangeInt  < 0 then
         Change = ChangeInt - ChangeDecimal
       else
         Change = ChangeInt + ChangeDecimal
       endif
       Newprice = [Price]+Change                ; Calculate new stock price
       If Newprice > 200 THEN                   ; Maximum of new price should be less then $200
         LOOP
       ELSE
         QUITLOOP
       ENDIF
    ENDWHILE
    Volume = Int(Rand() * 5000) + 5
    [High] = Max( [High], Newprice)              ; Calculate new session High price
    [Low]  = Min( [Low], Newprice)               ; Calculate new session Low price
    [Volume] = [Volume] + Volume                 ; Update the stock trading volume
    [Chg] = Newprice - ( [Price] - [Chg])        ; Stick in the change in price
    [Price]= NewPrice                            ; And update the Price
    [LAST]=Time()                                ; Finally, stick in the time of the trade

; Time to display what we've done

   Lastr=R                          ; Lastr is previous row we displayed trade on
   R=R+1                            ; Increment our row counter to get new trade row
   IF R>LastStockRow THEN R=FirstStockRow ENDIF    ; Wrap to start line when we get to bottom
   @ Lastr,LeftMargin ?? "            " ; Clear the LAST TRADE: marker from previous trade
   ?? Curline                       ; and display previous trade
   IF (IsColorMonitor)
     THEN STYLE ATTRIBUTE 116       ; Red on Light Grey for Color monitors
     ELSE STYLE REVERSE             ; everyone else gets inverse
   ENDIF

   @ R,LeftMargin ?? "LAST TRADE: "         ; And update the trade information
   Curline = Format("W6",[TICKER]) +
             "  " + FORMAT("W9.3",[PRICE]) +
             " " + FORMAT("W6",VOLUME) + " SHARES " +
             " AT " + [LAST] + "  "
   SWITCH
     CASE Change > .001 :
        Curline = Curline + "+" + FORMAT("W8.3",Change)
     CASE Change < -.001 :
        Curline = Curline + "-" + FORMAT("W8.3",-Change)
     OTHERWISE:
        Curline = Curline + "UNCHANGED"
   ENDSWITCH
   ?? Curline                       ; Display the trade information
   STYLE ATTRIBUTE 31               ; Back to High Intensity or White on Blue
   UNLOCKRECORD
ENDWHILE

; If we drop out of the above loop, user must have pressed a key

  x = Getchar()                     ; Gobble up the keystroke
  DO_IT!                            ; Update the table!
  CLEARALL                          ; Clear the workspace
  DoneFeed()                        ; Update Semaphor to tell world we're done
ENDPROC ; DoFeed

; TryToInitialize attempts to reinitialize fields in the PRICE table.
; If we are the first ones there, we perform the initialization by
; zeroing out the fields that will be generated by Rnd. If we aren't the
; first ones there, then someone else is already generating stock activity
; and we shouldn't reinitialize the table.

PROC TryToInitialize()
  IF (NOT TestInitialize())         ; Can we perform an initialization?
    THEN                            ; No, so
      IF (TestWaitForInitialize())  ;   Do we need to wait for someone else to
                                    ;     complete an initialization?
        THEN                        ;   Yes, so
          PutMessage(
     "Session Initialization being performed by another user. Please wait...")
          WHILE (TestWaitForInitialize())  ; keep checking
            SLEEP 500                      ; after a short pause
          ENDWHILE
          ClearMessage()            ;   erase the message, since we can go on
      ENDIF
      RETURN                        ;   move on
  ENDIF

; If we get here, we must have to initialize the market

  PutMessage("Initializing Stock Transaction Session...")
  WHILE (TRUE)                      ; wait for any query to finish
    LOCK "Price" PWL                ;    since queries do a WL on table
    IF (RETVAL) THEN QUITLOOP ENDIF
    SLEEP 200
  ENDWHILE
  COEDIT "Price"                    ; get the table on workspace
  SCAN                              ; and for each record,
    [VOLUME] = 0                    ;   reset the daily volume
    [High]   = [Price]              ;   Today's High price is equal to previous close
    [Low]    = [Price]              ;   Today's Low price is equal to previous close
    [Chg]    = 0                    ;   No change in price yet
  ENDSCAN
  DO_IT!                            ; And save the new table
  UNLOCK "Price" PWL
  DoneInitialize()                  ; Release the Initialization semaphor
  ClearMessage()                    ; Erase the Init message
ENDPROC

PROC ClearBackground(Attr,RestAttr,FirstLine,LastLine)
  PRIVATE s,i
  STYLE ATTRIBUTE Attr
  s = SPACES(80)
  FOR i FROM FirstLine TO LastLine
    @ i,0 ?? s
  ENDFOR
  STYLE ATTRIBUTE RestAttr
ENDPROC

PROC PutMessage(s)             ; define a message window ala Paradox
  IF (IsColorMonitor)          ; are we on a color monitor?
    THEN STYLE ATTRIBUTE 116   ; yes, so use red on light grey
    ELSE STYLE REVERSE         ; no, everyone else gets inverse
  ENDIF
  @ 24,80 - LEN(s) ?? s        ; display s in MESSAGE window
  STYLE ATTRIBUTE 31           ; return style to normal
ENDPROC

PROC ClearMessage()            ; clear the message window we defined
  ClearBackground(31,31,24,24)
ENDPROC

PROC StartUp()                 ; Get the feed started
  RESET                        ; Start with a known system state
  Curline = ""
  FirstStockRow = 8            ; where will start displaying stock info?
  LastStockRow = 20            ; and where will end disply of stock info?
  LeftMargin = 6
  CURSOR OFF                   ; don't need the cursor to display info
  IsColorMonitor = (Monitor()="Color")  ; find out what kind of monitor we're on
  ClearBackground(31,31,0,24)  ; and initialize the screen to blank
  TryToInitialize()            ; see if we need to initialize Price table
  @ FirstStockRow-2,LeftMargin ; finally, display headings for stock listings
  ?? "           " +
     FORMAT("W6,AL,","Symbol")+
     FORMAT("W11,AR","Price")+
     FORMAT("W20,AC","Shared Traded")+
     FORMAT("W8,AC","Time")+
     FORMAT("W9,AR","Change")

  CLEARALL                     ; Clear all tables on workspace
ENDPROC

PROC TestInitialize()
  LOCK "Semaphor" WL, "Semaphor" PWL
  RETURN retval
ENDPROC

PROC TestWaitForInitialize()
  LOCK "Semaphor" WL
  RETURN NOT retval
ENDPROC

PROC DoneInitialize()
  UNLOCK "Semaphor" PWL
ENDPROC

PROC DoneFeed()
  UNLOCK "Semaphor" WL
ENDPROC
