//***************************************************************************
//*                                                                         *
//* Interactive 3D Tool Kit I3D v2.1                                        *
//*                                                                         *
//* I3D API                                                                 *
//*                                                                         *
//* (c) 1993-95 Jim O'Keane                                                 *
//* All Rights Reserved Worldwide                                           *
//*                                                                         *
//***************************************************************************

#ifndef _I3DKIT_H

#define _I3DKIT_H

// Simple macros to facilitate global vars in .h files
// Define MAIN in only one file before #includes.
#ifndef MAIN
#define GLOBAL extern
#else
#define GLOBAL
#endif

// MS Windows-like types
#ifndef _WINDOWS_
typedef unsigned char     BYTE;
typedef unsigned short    WORD;
typedef unsigned long     DWORD;
typedef int               BOOL;

#define TRUE  1
#define FALSE 0

typedef struct
{
	short x;
	short y;
} POINT;

#endif // windows

// Maps are always 128 by 128 blocks. (128 x 128 x 4 = 64K Bytes)
// If you need a smaller map, just don't use whole array.
// The map edges "wrap" so if you walk off one edge you walk onto the other.
// The map contains indexes into the block definition array.

// coords (128 * 256) will still fit in a short int)
#define MAP_WIDTH  128
#define MAP_HEIGHT 128

// cheap way to keep map coords "in bounds"
#define MAP_MASK   127

// Automapping flags

// map cell has been seen by the viewer (automap)
#define AUTOMAP_SEEN   0x0001

// Map array type, allows 32767 different TYPES of blocks
typedef short MAPTYPE[MAP_WIDTH][MAP_HEIGHT];

// the VGA palette data structure RED=0,GREEN=1,BLUE=2
typedef BYTE DACPAL256[256][3];

// size of map cell 256x256x256 units (high byte of world coords is map coord)
#define BLOCK_SIZE     256
// mask is used to find position inside a block
#define BLOCK_MASK     255
// shift is used to find the map coords from world coords
#define BLOCK_SHIFT    8

// Block shape types
#define BLOCK_OPEN     0  // block is just floor & ceiling
#define BLOCK_CUBE     1  // block is a cube
#define BLOCK_HORZ     2  // block is a horizontal (EW) divider
#define BLOCK_VERT     3  // block is a vertical (NS) divider
#define BLOCK_ACTOR    4  // block is an actor or prop
#define BLOCK_DIAG1    5  // block is a diagonal wall
#define BLOCK_DIAG2    6  // block is a diagonal wall

// Bit flags used by I3D Engine
#define BLOCK_TRANS      1   // block is transparent
#define BLOCK_WALL       2   // block is an impassable
#define BLOCK_BI         4   // Thing is bilaterally symmetric, saves views
#define BLOCK_DECAL      8   // block is transparent decal

// definition of a block
typedef struct tag_block
{
	long shape;     // type of shape
	long flags;     // bit flags

	long recolor;   // palette redirection 0 = normal
	long light;     // "light" level 0 = normal, < 0 "darker", > 0 "lighter"

	long twidth;    // texture map width (32,64,128)
	long twidth_shift; // texture map width in shift (5,6,7)
	long theight;   // texture map height
	long tsize;     // texture map size in bytes

	long bottom;    // bottom of block (z coord) normally = 0
	long top;       // top of block (z coord)    normally = 256

	long x_offset;  // positional offset (+/- 1/2 BLOCK_SIZE)
	long y_offset;  // used to animate blocks and things

	long n_wall;    // wall panel to show (-1 = none)
	long e_wall;    // wall panel to show (-1 = none)
	long s_wall;    // wall panel to show (-1 = none)
	long w_wall;    // wall panel to show (-1 = none)

	long ceil;      // ceiling panel to show - must be (twidth x twidth)
	long floor;     // floor panel to show  - must be (twidth x twidth)

	long user1;     // user defined
	long user2;
} BLOCK;

// if a block is an actor or prop, cast the block struct to this:
typedef struct tag_thing
{
	long shape;     // type of shape
	long flags;     // bit flags

	long recolor;   // palette redirection 0 = normal
	long light;     // "light" level 0 = normal, < 0 darker, > 0 lighter

	long twidth;    // texture map width (32,64,128)
	long twidth_shift; // texture map width in shift (5,6,7)
	long theight;   // texture map height
	long tsize;     // texture map size in bytes

	long bottom;    // bottom of block (z coord) normally = 0
	long top;       // top of block (z coord)    normally = 256

	long x_offset;  // positional offset +/- 1/2 BLOCK_SIZE
	long y_offset;  // used to animate blocks and things

	long panel;     // base panel to show (-1 = none)
	long block;     // what background block is at this location
	long views;     // how many directions give different views?
	long heading;   // direction facing

	long res1;      // reserved
	long res2;

	short user1;     // user defined
	short user2;
} THING;

// standard image operator macros
#define IMAGE_SIZE256(width,height) ((long)(width)*(height))

// calc offset into image byte array for a pixel at (x,y)
#define PIXPOS(x,y,width) (((y)*(width))+(x))

// handy macros
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define SIGN(x)  (((x) < 0) ? -1 : (((x) > 0) ? 1 : 0))
#define ABS(x) ((x)<0 ? -(x):(x))

// Prototypes ////////////////////////////////////////////////////

#if defined ( __cplusplus )
  extern "C" {
#endif

// Returns a psuedo random number between 0 and num-1
long Random(long __num);

// Quicky integer math functions. High speed, low accuracy.
long q_sin(long angle);
long q_cos(long angle);

// rotate point (x,y) about center (cx,cy) by angle
void rotate(long *x, long *y, long cx, long cy, long angle);

// I3D functions ////////////////////////////

// Tell i3d about the map.
//
// Tell the i3d engine what map to use. i3d keeps a copy of
// this pointer, so you can change the map on the fly.
//
// Must call before asking i3d to render.
//
// Parameters:      map_ptr     - pointer to map
// 									automap_ptr - pointer to automap
//
// Returns:         NONE
void i3d_set_map(MAPTYPE *map_ptr,MAPTYPE *automap_ptr);

// Tell i3d about the texture panels.
//
// Tell the i3d engine about the texture map panels. i3d keeps a copy of
// the pointer to the panel list, so you can change the texture maps on
// the fly.
//
// Must call before asking i3d to render.
//
// Parameters:      num_panels - how many panels
//                  panel_list - pointer to an array of pointers to textures
//
// Returns:         NONE
void i3d_set_panels(long num_panels, BYTE **panel_list);

// Tell i3d engine about the list of block definitions.
//
// Tell the i3d engine about you block definition list. i3d keeps a copy
// of the pointer to the block list, so you can change the block definitions
// on the fly.
//
// Must call before asking i3d to render.
//
// Parameters:      num_blocks - number of blocks in list
//                  block_list - pointer to first block in list
//
// Returns:         NONE
void i3d_set_blocks(long num_blocks, BLOCK *block_list);

// Set the floor and ceiling colors
//
// Tell the i3d engine to use a solid color for floors and ceilings. If set
// to zero, uses texture mapped floors and/or ceilings. If set to -1, then
// no drawing is done, caller is expected to fill POV buffer with backdrop.
//
// Parameters:      floor_col - palette index of color to show
//                  ceil_col  - palette index of color to show
//
// Returns:         NONE
void i3d_set_floor_ceil(long floor_col, long ceil_col);

// Setup the backdrop bitmap for "outdoors" scenes
//
// Parameters:      image  - pointer to the texture map for backdrop
//                  width  - width of texture map
//                  height - height of texture map
//                  horizon- horizon level in texture map
//                  scale  - scale factor for heading
//
// Returns:         NONE
void i3d_set_backdrop(BYTE *image,long width,long height,
											long horizon,long scale);

// Setup the fake lighting for i3d.
//
// Tell the i3d engine to use a set of palette indirection tables.
// i3d keeps a copy of the pointer to the paltable list, so you can change
// the actual tables on the fly.
//
// max_levels is the table number which lighting effects fade to at the
// far end. Shift is how far to right shift the distance to determine
// the light level.
//
// For example, I want each light level to be 1 block wide (256) so my shift
// is 8. I set max_levels to be 16, and I can see at most 16 blocks
// away before it is all darkness (or solid color).
//
// Set paltables_ptr to NULL and max_levels to 0 for no lighting effects.
// (default)
//
// Parameters:      paltables_ptr - Ptr to palette indirection tables
//                  max_levels    - maximum number of "light" levels
//                  shift         - shift factor for converting distances
//									to light levels
//
// Returns:         NONE
void i3d_set_lighting(BYTE **paltables_ptr,
                            long max_levels, long shift);

// Calculate color fade / lighting palette indirection tables.
//
// Calculate the fade tables from 0% to 100% over a series of steps
//
// Parameters:      paltable_ptr - ptr to tables computed
//                  pal          - the input palette
//                  max          - maximum "light" levels
//                  color        - palette index of destination color
//
// Returns:         NONE
void i3d_calc_paltables(BYTE **paltable_ptr,DACPAL256 *pal,
                              long max, long color);

// Tell i3d about the POV window buffer to draw into.
//
// Parameters:      width      - width of the buffer in pixels
//                  height     - height of buffer in pixels
//                  line_width - width of the entire screen buf in pixels
//                  buf        - pointer to upper left corner of buffer
//									invert		 - does POV buffer need to be inverted? (DIB?)
//
// Returns:         NONE
void i3d_set_window_buffer(long width, long height,
                                 long line_width,
                                 BYTE *buf,BOOL invert);

// Shutdown the i3d engine, free memory buffers.
//
// Parameters:      NONE
//
// Returns:         NONE
void i3d_shutdown(void);

// Ask i3d to render the current view in the POV window.
//
// Parameters:      s_x         - location of viewer in world space
//                  s_y
//                  horizon     - horizon level in POV window
//                                (makes head tilt up/down)
//                  eye_level   - eye level (1 to BLOCK_SIZE-1)
//                  scan_width  - aspect ratio vars
//                  scan_height
//                  heading     - direction viewer is facing (0-255)
//
// Returns:         NONE
void i3d_view_scan256(long s_x, long s_y,
                            long horizon, long eye_level,
                            long scan_width, long scan_height,
                            long heading);

// Scan the POV to see what block is under click_x,click_y
//
// Parameters:      s_x         - location of viewer in world coords
//                  s_y
//                  horizon     - level of horizon in POV buffer
//                  eye_level   - height level of eye (1 to BLOCK_SIZE-1)
//                  scan_width  - used to set aspect ratio of blocks
//                  scan_height
//                  heading     - what direction the viewer is looking (0-255)
//                  click_x     - point in POV window to check for hit/pick
//                  click_y
//                  h_x         - point in world space that was hit
//                  h_y
//                  h_z
//                  h_d         - depth (distance) point is from viewer
//                  block_id    - which type of block was hit
//                  panel_u     - where in the texture did we hit?
//                  panel_v
//                  panel_n     - which texture panel did we hit?
//
// Returns:         NONE
void i3d_hit_scan256(long s_x, long s_y,
                     long horizon, long eye_level,
                     long scan_width, long scan_height,
                     long heading,
                     long click_x, long click_y,
                     long *h_x, long *h_y, long *h_z, long *h_d,
                     long *block_id,
 						 			   long *panel_u, long *panel_v, long *panel_n);


#if defined ( __cplusplus )
  }
#endif


#endif // I3DKIT.H
