/******************************************************************************/
/*
        dlink.c
        user authentication and print spooling RPC server
*/
/******************************************************************************/

#include <stdio.h>
#include <pwd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <rpc/rpc.h>

#define DLINK_PROG     150001L
#define DLINK_VERS     1L
#define DLINK_AUTH     1L
#define DLINK_INIT     2L
#define DLINK_PRNT     3L

#define MAX_PATH_LENGTH 1024

/******************************************************************************/
/*
        result codes
*/
/******************************************************************************/

/* authentication status */
#define AUTH_OK                 0
#define AUTH_FAKE               1
#define AUTH_FAIL               2

/* initialization status */
#define INIT_OK                 0
#define INIT_NO_SUCH_PRINTER    1
#define INIT_FAIL               2

/* print status */
#define PRNT_OK                 0
#define PRNT_ALREADY            1
#define PRNT_NULL               2
#define PRNT_NO_FILE            3
#define PRNT_FAIL               4

/******************************************************************************/
/*
        RPC call structures
*/
/******************************************************************************/

typedef struct _auth_arg {
    char *user;
    char *pwrd;
} auth_arg;

typedef struct _auth_res {
    int status;
    long uid;
    long gid;
} auth_res;

typedef struct _init_arg {
    char *client;
    char *printer;
} init_arg;

typedef struct _init_res {
    int status;
    char *dir;
} init_res;

typedef struct _prnt_arg {
    char *client;
    char *printer;
    char *user;
    char *file;
    char *options;
} prnt_arg;

typedef struct _prnt_res {
    int status;
} prnt_res;

/******************************************************************************/

struct stat stats;
char pathname [MAX_PATH_LENGTH];
char new_pathname [MAX_PATH_LENGTH];
char spool_dir [MAX_PATH_LENGTH];

/******************************************************************************/
/*
        decode RPC string arguments
*/
/******************************************************************************/

int decode (dst, src)
char *dst;
char *src;

{
    while (*src)
    {
        *dst++ = (*src ^ 0x5b) & 0x7f;
        src++;
    }
    *dst = 0;
}

/******************************************************************************/
/*
        child termination signal handler
*/
/******************************************************************************/

int free_child ()

{
    int id, status;

    id = wait (&status);
}

/******************************************************************************/
/*
        XDR routines
*/
/******************************************************************************/

bool_t xdr_auth_arg (xdr, p)
XDR *xdr;
auth_arg *p;

{
    if (xdr_string (xdr, &p->user, 32))
        return (xdr_string (xdr, &p->pwrd, 64));
    return (FALSE);
}

/******************************************************************************/

bool_t xdr_auth_res (xdr, p)
XDR *xdr;
auth_res *p;

{
    if (xdr_int (xdr, &p->status))
        if (xdr_long (xdr, &p->uid))
            return (xdr_long (xdr, &p->gid));
    return (FALSE);
}

/******************************************************************************/

bool_t xdr_init_arg (xdr, p)
XDR *xdr;
init_arg *p;

{
    if (xdr_string (xdr, &p->client, 64))
        return (xdr_string (xdr, &p->printer, 64));
    return (FALSE);
}

/******************************************************************************/

bool_t xdr_init_res (xdr, p)
XDR *xdr;
init_res *p;

{
    if (xdr_int (xdr, &p->status))
        return (xdr_string (xdr, &p->dir, 255));
    return (FALSE);
}

/******************************************************************************/

bool_t xdr_prnt_arg (xdr, p)
XDR *xdr;
prnt_arg *p;

{
    if (xdr_string (xdr, &p->client, 64))
        if (xdr_string (xdr, &p->printer, 64))
            if (xdr_string (xdr, &p->user, 64))
                if (xdr_string (xdr, &p->file, 64))
                    return (xdr_string (xdr, &p->options, 64));
    return (FALSE);
}

/******************************************************************************/

bool_t xdr_prnt_res (xdr, p)
XDR *xdr;
prnt_res *p;

{
    return (xdr_int (xdr, &p->status));
}

/******************************************************************************/
/*
        RPC server routines
*/
/******************************************************************************/

char *auth_proc (a)
auth_arg *a;

{
    static auth_res r;
    char username [32];
    char password [64];
    int c1, c2;
    struct passwd *p;

    /* assume failure */
    r.status = AUTH_FAIL;
    decode (username, a->user);
    decode (password, a->pwrd);
    if ((p = getpwnam (username)) == NULL) return ((char *) &r);

    /* verify password */
    c1 = strlen (password);
    c2 = strlen (p->pw_passwd);
    if ((c1 && !c2) || (c2 && !c1))
        return ((char *) &r);
    if (strcmp (p->pw_passwd, crypt (password, p->pw_passwd)))
        return ((char *) &r);

    /* success */
    r.status = AUTH_OK;
    r.uid = p->pw_uid;
    r.gid = p->pw_gid;
    return ((char *) &r);
}

/******************************************************************************/

char *pr_init_proc (a)
init_arg *a;

{
    static init_res r;

    /* build spool directory for host */
    strcpy (pathname, spool_dir);
    strcat (pathname, "/");
    strcat (pathname, a->client);
    mkdir (pathname);

    /* check if directory exists */
    if (stat (pathname, &stats) || !(stats.st_mode & S_IFDIR))
    {
        fprintf (stderr, "DLINK: Cannot create spool directory %s\r\n",
            pathname);
        pathname [0] = 0;
        r.status = INIT_FAIL;
    }
    else
    {
        chmod (pathname, 0777);
        r.status = INIT_OK;
    }
    r.dir = &pathname [0];
    return ((char *) &r);
}

/******************************************************************************/

char *pr_start_proc (a)
prnt_arg *a;

{
    static prnt_res r;
    int pid, i;
    char prnt_opt [64];
    char user_opt [64];
    char clnt_opt [64];
    struct passwd *p;
    long rnum;
    char snum [20];

    /* set up option strings */
    strcpy (prnt_opt, "-P"); /* printer name */
    strcat (prnt_opt, a->printer);
    strcpy (user_opt, "-J"); /* user name */
    strcat (user_opt, a->user);
    strcpy (clnt_opt, "-C"); /* client machine */
    strcat (clnt_opt, a->client);

    /* catch signal when child terminates */
    signal (SIGCHLD, free_child);

    /* build complete path of file to print */
    strcpy (pathname, spool_dir);
    strcat (pathname, "/");
    strcat (pathname, a->client);
    strcat (pathname, "/");
    strcat (pathname, a->file);

    /* check if file actually exists */
    if (stat (pathname, &stats))
    {
        /* print may be in progress */
        strcat (pathname, ".spl");
        if (stat (pathname, &stats))
        {
            /* file does not exist */
            r.status = PRNT_NO_FILE;
            return ((char *) &r);
        }
        r.status = PRNT_ALREADY;
        return ((char *) &r);
    }

    /* check if file is empty */
    if (stats.st_size == 0)
    {
        unlink (pathname);
        r.status = PRNT_NULL;
        return ((char *) &r);
    }

    /* append extension to existing file */
    strcpy (new_pathname, pathname);
    strcat (new_pathname, ".spl");

    /* use three digit extension if file exists */
    for (i = 0; i < 100; i++)
    {
        if (stat (new_pathname, &stats)) break;

        /* build a new name with random number */
        strcpy (new_pathname, pathname);
        sprintf (snum, "%ld", random ());
        strncat (new_pathname, snum, 3);
        strcat (new_pathname, ".spl");
    }

    /* rename will only fail if all 100 files exist */
    if (rename (pathname, new_pathname))
    {
        fprintf (stderr, "DLINK: Could not rename spool file %s to %s\r\n",
            pathname, new_pathname);
        r.status = PRNT_FAIL;
        return ((char *) &r);
    }

    /* execute print process */
    pid = fork ();
    if (pid == 0)
    {
        execlp ("/usr/ucb/lpr", "lpr", "-s", "-r", prnt_opt, user_opt,
            clnt_opt, new_pathname, 0);
        perror ("DLINK: Unable to execute print process");
        exit (0);
    }
    else if (pid == -1)
    {
        perror ("DLINK: Fork failed");
        r.status = PRNT_FAIL;
    }
    else
    {
        r.status = PRNT_OK;
    }
        return ((char *) &r);
}

/******************************************************************************/
/*
        program entry point
*/
/******************************************************************************/

int main (argc, argv)
int argc;
char *argv [];

{
    if (fork () == 0)
    {
        /* verify print spool directory */
        strcpy (spool_dir, (argc < 2) ? "/usr/spool" : argv [1]);
        if (stat (spool_dir, &stats) || !(stats.st_mode & S_IFDIR))
        {
            fprintf (stderr, "DLINK: Invalid spool directory %s\r\n",
                spool_dir);
            exit (1);
        }

        /* make RPC services known */
        registerrpc (DLINK_PROG, DLINK_VERS, DLINK_AUTH, auth_proc,
            xdr_auth_arg, xdr_auth_res);
        registerrpc (DLINK_PROG, DLINK_VERS, DLINK_INIT, pr_init_proc,
            xdr_init_arg, xdr_init_res);
        registerrpc (DLINK_PROG, DLINK_VERS, DLINK_PRNT, pr_start_proc,
            xdr_prnt_arg, xdr_prnt_res);

        /* should never return */
        svc_run ();
        fprintf (stderr, "DLINK: svc_run () returned\r\n");
        exit (1);
    }
}

/******************************************************************************/
