/****************************************************************************

	WINOUT.C

		Code for output functions of the Windows PipeTool Client.

	PURPOSE:

		This is the windows client side of a simple pipe tool used to test
		the basic functionality of pipe IO calls. All basic forms of IO are
		available (read, write, transact, and call), virtually all pipe
		parameters can be set, and data is checked for corruption when read.
		The catch is that a pipe is used only once (i.e. it is closed after
		each requested transaction is completed.)

	COMMENTS:

		Windows can have several copies of this application running at the
		same time.  The variable hInst keeps track of which instance this
		application is so that processing will be to the correct window.
		Also in order to not shut out user input to windows while this
		program is running it is chopped up into pieces (states) and then
		that state machine is traversed with each step triggered by a timer
		or by clicking on the "step!" menu item when in single step mode.

		This program must be used with server program spt.exe which runs on
		OS/2.  These programs make no assumption that the parameters you
		specify are correct. That is, it was designed to try out situations
		that would not work.  See pipetool.doc for more information regarding
		this.

	FUNCTIONS:

		This file:

			ttyOut() - Output function for results
			UpdateLine() - Output function when header info is changed

		In wincpt.c:

			WinMain() - application and instance init, main message loop
			MainWndProc() - processes messages for main window

		In dialogs.c:

			About() - processes messages for "About" dialog box
			PipeName() - processes messages for "Pipe Name" dialog box
			DataSIzes() - processes messages for "Data Sizes" dialog box
			OpenMode() - processes messages for "Open Mode" dialog box
			ResetOMButtons() - Checks radio buttons based on OM value
			PipeMode() - processes messages for "Pipe Mode" dialog box
			ResetPMButtons() - Checks radio buttons based on PM value
			SaveToFile() - processes messages for "Save" dialog box

		In dostate.c

			DoState() - Does the state machine traversal stuff.
			StartTransaction() - gets a selected transaction rolling
			ResetTransaction() - cleans up aborted or finished transaction
			WaitAsync() - Callback function for async pipe calls

	Copyright (C) 1992 Microsoft Corporation.

	This code sample is provided for demonstration purposes only.
	Microsoft makes no warranty, either express or implied,
	as to its usability in any given situation.

****************************************************************************/

#include <windows.h>		// Windows Includes

#include <string.h>		// C Runtime Includes
#include <stdio.h>
#include <stdarg.h>

#include "wincpt.h"		// Include specifically for windows PipeTool

// ..........................................................................
//
//	FUNCTION: ttyOut( char * )
//
//	COMMENTS:
//
//		displays the specified string on the last available line in tty fashion.
//
// ..........................................................................

VOID ttyOut( char * fmt, ... )
{
	va_list marker;
	DWORD ulSizes = 0;

	va_start( marker, fmt );
	vsprintf( Result[ iResultNext ].acBuff, fmt, marker );
	va_end( marker );

	Result[ iResultNext ].iLen = strlen( Result[ iResultNext ].acBuff );

	if ( Result[ iResultNext ].iLen > nMaxWidth )
	{
		nMaxWidth = Result[ iResultNext ].iLen;

		ulSizes = ((ulSizes | cyClient) << 16) | cxClient;
		SendMessage( hWnd, WM_SIZE, 0 , ulSizes );
	}

	iResultNext++;

	if ( iResultNext == MAXNUM_RESULT_LINE )
	{
		iResultNext = 0;

		fResultWrapped = TRUE;

		nNumLines = MAXNUM_RESULT_LINE + NUM_HEADER_LINE;
	}

	if ( !fResultWrapped )
	{
		nNumLines++;

		if ( nVscrollMax = max( 0, nNumLines - cyClient / cyChar ) )
		{
			SetScrollRange( hWnd, SB_VERT, 0, nVscrollMax, FALSE );
		}	
	}

	UpdateLine( nNumLines - 1 );

	if( nVscrollPos == nVscrollMax - 1)
	{
		SendMessage( hWnd, WM_VSCROLL, SB_LINEDOWN, 0L );
	}

} // End: ttyOut

// ..........................................................................
//
//	FUNCTION: UpdateLine( int  )
//
//	COMMENTS:
//
//		Tries to update the information in the header (if it's visible).
//
// ..........................................................................

VOID UpdateLine( int Line )
{
	RECT  rect;	

	rect.left = 0;
	rect.right = cxClient;
	rect.top = max( 0, (Line - nVscrollPos ) * cyChar );
	rect.bottom = min( (Line - nVscrollPos + 1) * cyChar, cyClient );

	InvalidateRect( hWnd, &rect, TRUE );

} // End: UpdateHeader


