/****************************************************************************

	WINCPT.C

		Main code module for the PipeTool Windows client. Includes main and main
		window procedure as well as overall description of wincpt.exe.

	PURPOSE:

		This is the windows client side of a simple pipe tool used to test
		the basic functionality of pipe IO calls. All basic forms of IO are
		available (read, write, transact, and call), virtually all pipe
		parameters can be set, and data is checked for corruption when read.
		The catch is that a pipe is used only once (i.e. it is closed after
		each requested transaction is completed.)

	COMMENTS:

		Windows can have several copies of this application running at the
		same time.  The variable hInst keeps track of which instance this
		application is so that processing will be to the correct window.
		Also in order to not shut out user input to windows while this
		program is running it is chopped up into pieces (states) and then
		that state machine is traversed with each step triggered by a timer
		or by clicking on the "step!" menu item when in single step mode.

		This program must be used with server program spt.exe which runs on
		OS/2.  These programs make no assumption that the parameters you
		specify are correct. That is, it was designed to try out situations
		that would not work.  See pipetool.doc for more information regarding
		this.

	FUNCTIONS:

		This file:
	
			WinMain() - application and instance init, main message loop
			MainWndProc() - processes messages for main window

		In dialogs.c:

			About() - processes messages for "About" dialog box
			PipeName() - processes messages for "Pipe Name" dialog box
			DataSIzes() - processes messages for "Data Sizes" dialog box
			OpenMode() - processes messages for "Open Mode" dialog box
			ResetOMButtons() - Checks radio buttons based on OM value
			PipeMode() - processes messages for "Pipe Mode" dialog box
			ResetPMButtons() - Checks radio buttons based on PM value
			SaveToFile() - processes messages for "Save" dialog box

		In winout.c:

			ttyOut() - Output function for results
			UpdateLine() - Output function when header info is changed

		In dostate.c

			DoState() - Does the state machine traversal stuff.
			StartTransaction() - gets a selected transaction rolling
			ResetTransaction() - cleans up aborted or finished transaction
			WaitAsync() - Callback function for async pipe calls

	Copyright (C) 1992 Microsoft Corporation.

	This code sample is provided for demonstration purposes only.
	Microsoft makes no warranty, either express or implied,
	as to its usability in any given situation.

****************************************************************************/

#include <windows.h>		// Windows Includes

#include <stdio.h>		// C Runtime Includes
#include <stdlib.h>
#include <string.h>

#define INCL_NETNMPIPE
#include <lan.h>			// LanMan Includes

#include "wincpt.h"		// Include specifically for windows PipeTool


/*** Declaration of global data ***/

HANDLE			hInst;									// Current instance
HMENU				hMenu;									// Needed to manipulate menu
HWND				hWnd;										// Main window handle.

char				pszSaveFileName[FILENAME_LEN];	// Filename/path for saved results

HeaderLineType	Header[ NUM_HEADER_LINE ];			// Array for header lines
ResultLineType	Result[ MAXNUM_RESULT_LINE ];		// Circular result array

char				pszPipeName[PIPENAME_LEN];			// Pipe control parameters passed
unsigned short	usDataSize 	 = DEF_DATASIZE;		//   to the pipe and file IO api.
unsigned short	usWriteSize  = DEF_DATASIZE;		//   All have default values that
unsigned short	usReadSize   = DEF_DATASIZE;		//   are defined in wincpt.h and
unsigned short	usOpenMode 	 = DEF_OPENMODE;		//   are changed by parameter
unsigned short	usPipeMode	 = DEF_PIPEMODE;		//   dialog boxes.
BOOL				fAsyncRead   = DEF_AREADFLAG;		// Control parameters set by
BOOL				fAsyncWrite  = DEF_AWRITEFLAG;	//   checking a menu item

int				hPipe    = NULL;						// Pipe handle
HANDLE			hInBuff  = NULL;						// Handles to globally allocated
HANDLE			hOutBuff = NULL;						//   data buffers

BOOL				fAsyncComplete;						// Async callback simply sets this

char				pszAsyncRead[10];						// Strings that async flag values
char				pszAsyncWrite[10];					//   translate to.

short				cxChar;									// Average text width (pixel)
short				cyChar;									// Total text height (pixel)

short				cxClient;								// Horizontal size of client area (pixel)
short				cyClient;								// Vertical size of client area (pixel)

short				nMaxWidth;								// Width of longest line (char)
short				nNumLines;								// Number of lines (char)

int				iResultNext    = 0;					// Index of next line to use
BOOL				fResultWrapped = FALSE;				// Has Result wrapped around?

short				nVscrollPos = 0;						// Position of vertical and
short				nHscrollPos = 0;						//   horizontal scrollbar thumb.

short				nVscrollMax;							// Max position of scrollbar thumbs based
short				nHscrollMax;							//   on window displayed vs. used.

WORD				wTransType;								// ie. Call, Transact, etc.
unsigned short	uCurState = WAIT_STATE;				// Place in state machine.

BOOL				fStep     = FALSE;					// Single step option toggle

BOOL				fModeSet    = FALSE;					// Has Pipe mode been set yet?
BOOL				fPipeOpened	= FALSE;					// Some things disabled after this


//...........................................................................
//
//	FUNCTION: WinMain( HANDLE, HANDLE, LPSTR, int )
//
//	PURPOSE: initialization, message loop
//
//	COMMENTS:
//
//		Windows recognizes this function by name as the initial entry point
//		for the program.  It will register the window class if no other instance
//		of the program is running, create the window, set up the timer, display
//		the window and start the message retrieval and dispatch loop that is the
//		top-level control structure for the remainder of execution.  The loop is
//		terminated when a WM_QUIT message is received, at which time this function
//		exits the application instance by returning the value passed by PostQuitMessage().
//
//		If this function must abort before entering the message loop, it
//		returns the conventional value, NULL.
//
//...........................................................................

int PASCAL WinMain( HANDLE	hInstance,		// current instance
                    HANDLE	hPrevInstance,	// previous instance
                    LPSTR	lpCmdLine,		// command line
                    int		nCmdShow )		// show-window type (open/icon)
{
	MSG       msg;
	WNDCLASS  wc;

	if ( !hPrevInstance )
	{
		wc.style = CS_HREDRAW | CS_VREDRAW;               // Class style(s).	
		wc.lpfnWndProc = MainWndProc;                     // Function to retrieve messages for
		                                                  //   windows of this class.
		wc.cbClsExtra = 0;                                // No per-class extra data.
		wc.cbWndExtra = 0;                                // No per-window extra data.
		wc.hInstance = hInstance;                         // Application that owns the class.
		wc.hIcon = LoadIcon( hInstance, "WinCPTIcon" );   // Uses its own super cool icon
		wc.hCursor = LoadCursor( NULL, IDC_ARROW );       // Uses default arrow cursor
		wc.hbrBackground = GetStockObject( WHITE_BRUSH ); // Background always white
		wc.lpszMenuName =  "WinCPTMenu";                  // Name of menu resource in .RC file.
		wc.lpszClassName = "WinCPTWClass";                // Name used in call to CreateWindow.
	
		if ( !RegisterClass( &wc ) )
			return( FALSE );
	}

	hInst = hInstance;

	if ( !( hWnd = CreateWindow( "WinCPTWClass",       // See RegisterClass() call.
	                             "WinCPT",             // Text for window title bar.
	                             WS_OVERLAPPEDWINDOW | // Window style.
	                             WS_VSCROLL |
	                             WS_HSCROLL,
	                             CW_USEDEFAULT,        // Default horizontal position.
	                             CW_USEDEFAULT,        // Default vertical position.
	                             CW_USEDEFAULT,        // Default width.
	                             CW_USEDEFAULT,        // Default height.
	                             NULL,                 // Overlapped windows have no parent.
	                             NULL,                 // Use the window class menu.
	                             hInstance,            // This instance owns this window.
	                             NULL ) ) )            // Pointer not needed.
	{
		return ( FALSE );
	}

	if ( !SetTimer( hWnd, TIMER_ID, TIMER_DELAY, NULL ) )
	{
		MessageBox( hWnd,
		            "Too many clocks or timers!",
		            "WinCPT",
		            MB_ICONEXCLAMATION | MB_OK );

		return( FALSE );
	}

	ShowWindow( hWnd, nCmdShow ); // Show the window
	UpdateWindow( hWnd );         // Sends WM_PAINT message

	// Acquire and dispatch messages until a WM_QUIT message is received.
	// The NULL's mean all messages for all windows in this program.

	while ( GetMessage( &msg,    // message structure
	                    NULL,    // handle of window to get messeges for
	                    NULL,    // lowest message to get
	                    NULL ) ) // highest message to get
	{
		TranslateMessage( &msg );  // Translates virtual key codes
		DispatchMessage(  &msg );  // Dispatches message to window
	}

	return ( msg.wParam );  // Returns the value from PostQuitMessage

} // End: WinMain


//...........................................................................
//
//	FUNCTION: MainWndProc( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for main window
//
//	MESSAGES:
//
//		WM_CREATE - Create window, get text size metrics.
//		WM_SIZE - Handle re-sizing of the window.
//		WM_VSCROLL - Vertical scroll bar events.
//		WM_HCCROLL - Horizontal scroll bar events.
//		WM_KEYDOWN - Process keys that can control scrolling.
//		WM_COMMAND - Process application menu selections.
//		WM_TIMER - Signal step to next state in program flow. This is done so
//			this program doesn't lock out windows and thus user input (ie. to
//			reset or abort test).
//		WM_PAINT - Paint client area.
//		WM_DESTROY - Destroy window.
//
//...........................................................................

long FAR PASCAL MainWndProc( HWND hWnd,
                             unsigned message,
                             WORD wParam,
                             LONG lParam )
{
	HDC				hdc;					// Needed for output, and getting test metrics
	PAINTSTRUCT		ps;					// Needed for Paint calls
	TEXTMETRIC		tm;					// dc info struct, to get text sizes

	short				nVscrollInc;		// Increment variables used in response to
	short				nHscrollInc;		//    scrolling messages

	FARPROC			lpDialogProc;		// Pointer to modal dialog box unction

	short				nPaintBeg;			// Boundary indexes to lines that need to
	short				nPaintEnd;			//    be redrawn on paint msg

	short				i, j;					// Indexes for to lines being displayed
	short				x, y;					// Position of text for TextOut func (pixel)

	unsigned			uRC;					// Return code for named pipe calls
	
	int				hSaveFile;			// Handle for Result file
	char				acLineBuf[512];	// Buffer to hold lines to be written
	int 				iLineBufLen;		// Length of line to write
	
	switch( message )
	{
		case WM_CREATE:	// Create window

			// get menu handle to be able to gray and check stuff

			hMenu = GetMenu( hWnd );

			// Get text size metrics

			hdc = GetDC( hWnd );
			GetTextMetrics( hdc, &tm );
			cxChar = tm.tmAveCharWidth;
			cyChar = tm.tmHeight + tm.tmExternalLeading;
			ReleaseDC( hWnd, hdc );

			// Set Default Pipe control variable strings

			strcpy( pszPipeName, DEF_PIPENAME );
												
			if ( fAsyncRead )
				strcpy( pszAsyncRead, "are" );
			else
				strcpy( pszAsyncRead, "are not" );

			if ( fAsyncWrite )
				strcpy( pszAsyncWrite, "are" );
			else
				strcpy( pszAsyncWrite, "are not" );

			// Set up memory for header portion of stuff to display and
			// initialize it.
			// NOTE: memory is allocated as fixed so there is no need to
			//    lock it. The handle returned is a valid near pointer

			Header[0].hBuff = LocalAlloc( LPTR, 300 );
			sprintf( (char *)Header[0].hBuff, "Pipe Name:  %s", pszPipeName );
			Header[0].iLen = strlen( (char *)Header[0].hBuff );

			Header[1].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[1].hBuff, "Data Size:  %d", usDataSize );
			Header[1].iLen = strlen( (char *)Header[1].hBuff );

			Header[2].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[2].hBuff, "Write Size:  %d", usWriteSize );
			Header[2].iLen = strlen( (char *)Header[2].hBuff );

			Header[3].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[3].hBuff, "Writes %s Async", pszAsyncWrite );
			Header[3].iLen = strlen( (char *)Header[3].hBuff );

			Header[4].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[4].hBuff, "Read Size:  %d", usReadSize );
			Header[4].iLen = strlen( (char *)Header[4].hBuff );

			Header[5].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[5].hBuff, "Reads %s Async", pszAsyncRead );
			Header[5].iLen = strlen( (char *)Header[5].hBuff );

			Header[6].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[6].hBuff, "Open Mode:  0x%x", usOpenMode );
			Header[6].iLen = strlen( (char *)Header[6].hBuff );

			Header[7].hBuff = LocalAlloc( LPTR, 25  );
			sprintf( (char *)Header[7].hBuff, "Pipe Mode:  0x%x", usPipeMode );
			Header[7].iLen = strlen( (char *)Header[7].hBuff );

			Header[8].hBuff = LocalAlloc( LPTR, 50  );
			sprintf( (char *)Header[8].hBuff,
			         "Results (only the last %d lines are displayed):",
			         MAXNUM_RESULT_LINE );
			Header[8].iLen = strlen( (char *)Header[8].hBuff );

			// Set line count and length of longest line

			nNumLines = NUM_HEADER_LINE;

			for ( i = 0, nMaxWidth = 0;
			      i < nNumLines;
			      i++ )
			{
				nMaxWidth = max( Header[i].iLen, nMaxWidth );
			}

			break;

		case WM_SIZE: // Window size setup or change

			// Get window size metrics

			cyClient = HIWORD( lParam );
			cxClient = LOWORD( lParam );

			// Setup scrolling control variables
			// NOTE: max values are the last position that will allow all the rest
			//    of the test to be displayed

			nVscrollMax = max( 0, nNumLines - cyClient / cyChar );
			nVscrollPos = min( nVscrollPos, nVscrollMax );

			SetScrollRange( hWnd, SB_VERT, 0, nVscrollMax, FALSE );
			SetScrollPos( hWnd, SB_VERT, nVscrollPos, TRUE );

			nHscrollMax = max( 0, nMaxWidth - cxClient / cxChar );
			nHscrollPos = min( nHscrollPos, nHscrollMax );

			SetScrollRange( hWnd, SB_HORZ, 0, nHscrollMax, FALSE );
			SetScrollPos( hWnd, SB_HORZ, nHscrollPos, TRUE );

			break;

		case WM_PAINT: // Restore the portion of the client area that's changed

			hdc = BeginPaint( hWnd, &ps );

			// Find the boundaries for the line numbers that need to be redrawn

			nPaintBeg = max( 0, nVscrollPos + ps.rcPaint.top / cyChar );
			nPaintEnd = min( nNumLines, nVscrollPos + ps.rcPaint.bottom / cyChar );

			// Redraw only those lines for efficiency

			for ( i = nPaintBeg; i <= nPaintEnd; i++ )
			{
				x = cxChar * ( 1 - nHscrollPos );
				y = cyChar * ( i - nVscrollPos );

				if ( i < NUM_HEADER_LINE )
				{
					// Display header line

					TextOut( hdc, x, y,
					         (char *)Header[ i ].hBuff,
					         Header[ i ].iLen );
				}
				else
				{
					// Display Line from circular result array

					// Adjust line number to array index
					j = i - NUM_HEADER_LINE;
					if ( fResultWrapped )
					{
						j = ( j + iResultNext ) % MAXNUM_RESULT_LINE;
					}

					TextOut( hdc, x, y,
					         Result[ j ].acBuff,
					         Result[ j ].iLen );
				}
			}

			EndPaint( hWnd, &ps );

			break;

		case WM_VSCROLL: // Vertical display window position change

			switch( wParam )
			{
				case SB_TOP:

					nVscrollInc = -nVscrollPos;
					break;

				case SB_BOTTOM:

					nVscrollInc = nVscrollMax - nVscrollPos;
					break;

				case SB_LINEUP:

					nVscrollInc = -1;
					break;

				case SB_LINEDOWN:

					nVscrollInc = 1;
					break;

				case SB_PAGEUP:

					nVscrollInc = min( -1, -cyClient / cyChar );
					break;

				case SB_PAGEDOWN:

					nVscrollInc = max( 1, cyClient / cyChar );
					break;

				case SB_THUMBTRACK:

					nVscrollInc = LOWORD( lParam ) - nVscrollPos;
					break;

				default:

					nVscrollInc = 0;
			}

			// Clip increment so position will stay between 0 and nVscrollMax

			if ( nVscrollInc = max( -nVscrollPos,
			                   min( nVscrollInc, nVscrollMax - nVscrollPos ) ) )
			{
				nVscrollPos += nVscrollInc;
				ScrollWindow( hWnd, 0, -cyChar * nVscrollInc, NULL, NULL );
				SetScrollPos( hWnd, SB_VERT, nVscrollPos, TRUE );
				UpdateWindow( hWnd );
			}

			break;

		case WM_HSCROLL: // Horizontal display window position change

			switch( wParam )
			{
				case SB_LINEUP:

					nHscrollInc = -1;
					break;

				case SB_LINEDOWN:

					nHscrollInc = 1;
					break;

				case SB_PAGEUP:

					nHscrollInc = -8;
					break;

				case SB_PAGEDOWN:

					nHscrollInc = 8;
					break;

				case SB_THUMBPOSITION:

					nHscrollInc = LOWORD( lParam ) - nHscrollPos;
					break;

				default:

					nHscrollInc = 0;
			}

			// Clip increment so position will stay between 0 and nHscrollMax

			if ( nHscrollInc = max( -nHscrollPos,
			                   min( nHscrollInc, nHscrollMax - nHscrollPos ) ) )
			{
				nHscrollPos += nHscrollInc;
				ScrollWindow( hWnd, -cxChar * nHscrollInc, 0, NULL, NULL );
				SetScrollPos( hWnd, SB_HORZ, nHscrollPos, TRUE );
			}

			break;

		case WM_KEYDOWN: // Proccess cursor movement keys for scrollbar control

			switch( wParam )
			{
				case VK_HOME:

					SendMessage( hWnd, WM_VSCROLL, SB_TOP, 0L );
					break;

				case VK_END:

					SendMessage( hWnd, WM_VSCROLL, SB_BOTTOM, 0L);
					break;

				case VK_PRIOR:

					SendMessage( hWnd, WM_VSCROLL, SB_PAGEUP, 0L );
					break;

				case VK_NEXT:

					SendMessage( hWnd, WM_VSCROLL, SB_PAGEDOWN, 0L );
					break;

				case VK_UP:

					SendMessage( hWnd, WM_VSCROLL, SB_LINEUP, 0L );
					break;

				case VK_DOWN:

					SendMessage( hWnd, WM_VSCROLL, SB_LINEDOWN, 0L );
					break;

				case VK_LEFT:

					SendMessage( hWnd, WM_HSCROLL, SB_PAGEUP, 0L );
					break;

				case VK_RIGHT:

					SendMessage( hWnd, WM_HSCROLL, SB_PAGEDOWN, 0L );
					break;
			}

			break;

		case WM_COMMAND: // Proccess selections from application menu

			switch( wParam )
			{
				case IDM_CALL: // Do DosCallNmPipe type transaction

					wTransType = IDM_CALL;
					StartTransaction();
					break;

				case IDM_TRANSACT: // Do DosTransactNmPipe type transaction

					wTransType = IDM_TRANSACT;
					StartTransaction();
					break;

				 case IDM_READ: // Do DosRead type transaction

					wTransType = IDM_READ;
					StartTransaction();
					break;

				case IDM_WRITE:	// Do DosWrite type transaction

					wTransType = IDM_WRITE;
					StartTransaction();
					break;

				case IDM_READWRITE: // Do DosRead/DosWrite type transaction

					wTransType = IDM_READWRITE;
					StartTransaction();
					break;

				case IDM_ABOUT: // Display About WinCPT dialog box

					lpDialogProc = MakeProcInstance( About, hInst );

					DialogBox( hInst,				
					           "AboutBox",		
					           hWnd,				
					           lpDialogProc );	

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_EXIT: // Exit program

					ResetTransaction();

					DestroyWindow( hWnd );

					break;

				case IDM_PIPENAME: // Display PipeName dialog box

					lpDialogProc = MakeProcInstance( PipeName, hInst );
															
					if ( DialogBox( hInst,			 	
					                "PipeNameBox",
					                hWnd,			  	
					                lpDialogProc ) )
					{
						sprintf( (char *)Header[0].hBuff, "Pipe Name:  %s", pszPipeName );
						Header[0].iLen = strlen( (char *)Header[0].hBuff );

						UpdateLine( 0 );
					}

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_DATASIZES: // Display DataSizes dialog box

					lpDialogProc = MakeProcInstance( DataSizes, hInst );

					if( DialogBox( hInst,
					               "DataSizesBox",
					               hWnd,
					               lpDialogProc ) )
					{
						sprintf( (char *)Header[1].hBuff, "Data Size:  %d", usDataSize );
						Header[1].iLen = strlen( (char *)Header[1].hBuff );

						UpdateLine( 1 );

						sprintf( (char *)Header[2].hBuff, "Write Size:  %d", usWriteSize );
						Header[2].iLen = strlen( (char *)Header[2].hBuff );

						UpdateLine( 2 );

						sprintf( (char *)Header[4].hBuff, "Read Size:  %d", usReadSize );
						Header[4].iLen = strlen( (char *)Header[4].hBuff );

						UpdateLine( 4 );
					}

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_OPENMODE: // Display OpenMode dialog box

					lpDialogProc = MakeProcInstance( OpenMode, hInst );

					if( DialogBox( hInst,
					               "OpenModeBox",
					               hWnd,
										lpDialogProc ) )
					{
						sprintf( (char *)Header[6].hBuff, "Open Mode:  0x%x", usOpenMode );
						Header[6].iLen = strlen( (char *)Header[6].hBuff );

						UpdateLine( 6 );
					}

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_PIPEMODE: // Display PipeMode dialog box

					lpDialogProc = MakeProcInstance( PipeMode, hInst );

					if( DialogBox( hInst,
					               "PipeModeBox",
					               hWnd,
					               lpDialogProc ) )
					{
						sprintf( (char *)Header[7].hBuff, "Pipe Mode:  0x%x", usPipeMode );
						Header[7].iLen = strlen( (char *)Header[7].hBuff );

						UpdateLine( 7 );
						
						if ( fModeSet ) // Pipe mode already set, need to reset it
						{
							ttyOut( "Setting pipe mode..." );

							if ( uRC = DosSetNmPHandState( hPipe,
																	 usPipeMode ) )
							{
								ttyOut( "...ERROR - DosSetNmPHandState() puked, returned %u with pipe mode 0x%x.", uRC, usPipeMode );
							}
							else
							{
								ttyOut( "...DosSetNmPHandState() completed successfully with pipe mode 0x%x.", usPipeMode );

								fModeSet = TRUE;
							}
						}
					}

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_SAVE: // Display SaveToFile dialog box

					lpDialogProc = MakeProcInstance( SaveToFile, hInst );

					if( DialogBox( hInst,
					               "SaveBox",
					               hWnd,
					               lpDialogProc ) )
					{
						hSaveFile = _lcreat( pszSaveFileName, 0);
						if (hSaveFile == -1)
						{
							MessageBox( hWnd, "Can't Open File.", NULL, MB_OK | MB_ICONEXCLAMATION );
						}
						else
						{
							for ( i = 0; i < NUM_HEADER_LINE; i++)
							{
								iLineBufLen = sprintf(acLineBuf, "%s\r\n", Header[ i ].hBuff );
								_lwrite( hSaveFile, acLineBuf, iLineBufLen );
							}

							for ( i = NUM_HEADER_LINE; i < nNumLines; i++ )
							{
								// Adjust line number to array index
								j = i - NUM_HEADER_LINE;
								if ( fResultWrapped )
								{
									j = ( j + iResultNext ) % MAXNUM_RESULT_LINE;
								}

								iLineBufLen = sprintf(acLineBuf, "%s\r\n", Result[ j ].acBuff );
								_lwrite( hSaveFile, acLineBuf, iLineBufLen );
							}

							_lclose( hSaveFile );
						}
					}				

					FreeProcInstance( lpDialogProc );

					break;

				case IDM_READASYNC: // Toggle async read status

					if ( fAsyncRead )
					{
						CheckMenuItem( hMenu, IDM_READASYNC, MF_UNCHECKED );
						fAsyncRead = FALSE;
						strcpy( pszAsyncRead, "are not" );
					}
					else
					{
						CheckMenuItem( hMenu, IDM_READASYNC, MF_CHECKED );
						fAsyncRead = TRUE;
						strcpy( pszAsyncRead, "are" );
					}

					sprintf( (char *)Header[5].hBuff, "Reads %s Async", pszAsyncRead );
					Header[5].iLen = strlen( (char *)Header[5].hBuff );

					UpdateLine( 5 );

					DrawMenuBar( hWnd );

					break;

				case IDM_WRITEASYNC: // Toggle async write status

					if ( fAsyncWrite )
					{
						CheckMenuItem( hMenu, IDM_WRITEASYNC, MF_UNCHECKED );
						fAsyncWrite = FALSE;
						strcpy( pszAsyncWrite, "are not" );
					}
					else
					{
						CheckMenuItem( hMenu, IDM_WRITEASYNC, MF_CHECKED );
						fAsyncWrite = TRUE;
						strcpy( pszAsyncWrite, "are" );
					}

					sprintf( (char *)Header[3].hBuff, "Writes %s Async", pszAsyncWrite );
					Header[3].iLen = strlen( (char *)Header[3].hBuff );

					UpdateLine( 3 );

					DrawMenuBar( hWnd );

					break;

				case IDM_TOGGLESTEP: // Toggle single step status

					if ( fStep )
					{
						CheckMenuItem( hMenu, IDM_TOGGLESTEP, MF_UNCHECKED );
						EnableMenuItem( hMenu, IDM_STEP, MF_GRAYED );
						fStep = FALSE;
					}
					else
					{
						CheckMenuItem( hMenu, IDM_TOGGLESTEP, MF_CHECKED );
						EnableMenuItem( hMenu, IDM_STEP, MF_ENABLED );
						fStep = TRUE;
					}

					DrawMenuBar( hWnd );

					break;

				case IDM_RESET:	// Abort transaction and reset with existing options

					ResetTransaction();

					break;

				case IDM_STEP: // Step to next state when in single step mode

					if ( fStep )
					{
						DoState();
					}

					break;

				default: // Let Windows process it

					return( DefWindowProc( hWnd, message, wParam, lParam ) );

			} // End: switch( wParam ) for WM_COMMAND

			break;

		case WM_TIMER: // Step to next state when not in single step mode
			
			if ( !fStep )
			{
				DoState();
			}

			break;

		case WM_DESTROY: // Window being destroyed

			KillTimer( hWnd, TIMER_ID );

			PostQuitMessage( 0 );

			break;

		default: // Pass message on to windows if unprocessed

			return( DefWindowProc( hWnd, message, wParam, lParam ));
	}

	return (NULL);

} // End: MainWndProc

