/****************************************************************************

	DOSTATE.C

		Code for the PipeTool Windows client that does transaction or the states
		that it is broken into.

	PURPOSE:

		This is the windows client side of a simple pipe tool used to test
		the basic functionality of pipe IO calls. All basic forms of IO are
		available (read, write, transact, and call), virtually all pipe
		parameters can be set, and data is checked for corruption when read.
		The catch is that a pipe is used only once (i.e. it is closed after
		each requested transaction is completed.)

	COMMENTS:

		Windows can have several copies of this application running at the
		same time.  The variable hInst keeps track of which instance this
		application is so that processing will be to the correct window.
		Also in order to not shut out user input to windows while this
		program is running it is chopped up into pieces (states) and then
		that state machine is traversed with each step triggered by a timer
		or by clicking on the "step!" menu item when in single step mode.

		This program must be used with server program spt.exe which runs on
		OS/2.  These programs make no assumption that the parameters you
		specify are correct. That is, it was designed to try out situations
		that would not work.  See pipetool.doc for more information regarding
		this.

	FUNCTIONS:

		This file:

			DoState() - Does the state machine traversal stuff.
			StartTransaction() - gets a selected transaction rolling
			ResetTransaction() - cleans up aborted or finished transaction
			WaitAsync() - Callback function for async pipe calls

		In wincpt.c:

			WinMain() - application and instance init, main message loop
			MainWndProc() - processes messages for main window

		In dialogs.c:

			About() - processes messages for "About" dialog box
			PipeName() - processes messages for "Pipe Name" dialog box
			DataSIzes() - processes messages for "Data Sizes" dialog box
			OpenMode() - processes messages for "Open Mode" dialog box
			ResetOMButtons() - Checks radio buttons based on OM value
			PipeMode() - processes messages for "Pipe Mode" dialog box
			ResetPMButtons() - Checks radio buttons based on PM value
			SaveToFile() - processes messages for "Save" dialog box

		In winout.c:

			ttyOut() - Output function for results
			UpdateLine() - Output function when header info is changed

	Copyright (C) 1992 Microsoft Corporation.

	This code sample is provided for demonstration purposes only.
	Microsoft makes no warranty, either express or implied,
	as to its usability in any given situation.

****************************************************************************/

#include <windows.h>		// Windows Includes

#include <stdio.h>      // C Runtime Includes
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#define INCL_NETNMPIPE
#include <lan.h>        // LanMan Includes

#include "wincpt.h"		// Include specifically for windows PipeTool

// ..........................................................................
//
//  FUNCTION: DoState( VOID )
//
//  COMMENTS:
//
//		Does whatever state is the current state and updates current state.
//		Called when state change is triggered by timer or "step!" menu item.
//
// ..........................................................................

VOID DoState( VOID )
{
	LPSTR							pszInBuff;				// locked ptr to input buffer
	LPSTR							pszOutBuff;				// Locked ptr to output buffer

	static unsigned short	usBytesRead;   		// Static for async case, one state
	static unsigned short	usBytesWritten;		//   calls, another reports results

	static unsigned short	usTotalRead;			// Total data read or written so
	static unsigned short   usTotalWritten;		//   far

	unsigned short				usRWSize;            // Size of each read or wirte
	
	static unsigned short	usAsyncRC;				// Return code from completed async call
	unsigned 					uRC;						// immediate return code

	static API_CALLBACK		lpfnWaitAsync;			// Stores MakeProcInstance for FreeProcInstance
	
	unsigned short		 		usCurPM;					// Q'ed PipeMode

	BOOL 							error = FALSE;
	int							exterr;

	unsigned short  			usInfoBufSize = 300;	// buffer for pipe info
	char            			cbInfoBuf[ 300 ];

	unsigned short far *		pusTemp;					// used to fill OutBuff with data
	unsigned short far *		pusTemp2;				//   and to compare it with InBuff
	unsigned short				usIndex;


	switch( uCurState )
	{
		case WAIT_STATE: // Do Nothing just Wait for selection to be made or reset

			break;

		case START_STATE:

			ttyOut( "Allocating input/output data buffer(s)..." );

			if ( wTransType != IDM_WRITE )
			{
				if ( !(hInBuff = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT,
				                              usDataSize )) )
				{
					error = TRUE;
				}
			}

			if ( !(hOutBuff = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT,
			                               usDataSize )) )
			{
				error = TRUE;
			}
			else
			{
			   // Set up data in buffer going out.

				pszOutBuff = GlobalLock( hOutBuff );

				for ( pusTemp = (unsigned short far *)pszOutBuff, usIndex = 0;
				      usIndex < usDataSize / 2;
				      pusTemp++, usIndex++ )
				{
					*pusTemp = usIndex;
				}

				GlobalUnlock( hOutBuff );
			}

			// Check for error and switch to next state.

			if ( error )
			{
				ttyOut( "...ERROR - A GlobalAlloc() failed." );
			
				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "...All GlobalAlloc() completed successfully." );

				if ( wTransType == IDM_CALL )
				{
					uCurState = CALL_STATE;
				}
				else
				{
					uCurState = OPEN_STATE;
				}
			}

			// Reset static read/written counts

			usBytesRead = 0;
			usBytesWritten = 0;
			usTotalRead = 0;
			usTotalWritten = 0;

			break;

		case CALL_STATE:

			ttyOut( "Calling pipe with/for %u bytes...", usDataSize );

			pszInBuff = GlobalLock( hInBuff );
			pszOutBuff = GlobalLock( hOutBuff );

			if ( uRC = DosCallNmPipe( pszPipeName,
			                          pszOutBuff,
			                          usDataSize,
			                          pszInBuff,
			                          usDataSize,
			                          &usTotalRead,
			                          0 ) )	// timeout
			{
				ttyOut( "...ERROR - DosCallNmPipe() failed, returned %u.", uRC );

				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "...DosCallNmPipe() completed successfully, %u bytes read.", usTotalRead );

				uCurState = CHECKDATA_STATE;
			}

			GlobalUnlock( hInBuff );
			GlobalUnlock( hOutBuff );

			break;

		case OPEN_STATE:

			ttyOut( "Opening pipe..." );

			if ( (hPipe = _lopen( pszPipeName,
			                      (int)usOpenMode )) == -1 )
			{
				ttyOut( "...ERROR - _lopen( %s, %x ) failed.", pszPipeName, usOpenMode );
				hPipe = NULL;

				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "..._lopen( %s, 0x%x ) completed successfully.", pszPipeName, usOpenMode );

				uCurState = QPIPEHANDSTATE_STATE;

				EnableMenuItem( hMenu, IDM_PIPENAME, MF_GRAYED );

				EnableMenuItem( hMenu, IDM_OPENMODE, MF_GRAYED );

				fPipeOpened = TRUE;
			}

			break;

		case QPIPEHANDSTATE_STATE:

			ttyOut( "Querying pipe mode..." );

			if ( (uRC = DosQNmPHandState( hPipe,
			                              &usCurPM )) )
			{
				ttyOut( "...ERROR - DosQNmPHandState() failed, returned %u.", uRC );

				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "...DosQNmPHandState() completed successfully, pipe mode = 0x%x.", usCurPM );

				uCurState = SETPIPEHANDSTATE_STATE;
			}

			break;

		case SETPIPEHANDSTATE_STATE:

			ttyOut( "Setting pipe mode..." );

			if ( uRC = DosSetNmPHandState( hPipe,
			                               usPipeMode ) )
			{
				ttyOut( "...ERROR - DosSetNmPHandState() failed, returned %u with pipe mode 0x%x.", uRC, usPipeMode );

				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "...DosSetNmPHandState() completed successfully with pipe mode 0x%x.", usPipeMode );

				uCurState = QPIPEINFO_STATE;

				fModeSet = TRUE;
			}

			break;

		case QPIPEINFO_STATE:

			ttyOut( "Getting pipe info..." );

			if ( uRC = DosQNmPipeInfo( hPipe,
			                           1,
			                           cbInfoBuf,
			                           usInfoBufSize ) )
			{
				ttyOut( "...ERROR - DosQNmPipeInfo() failed, returned %u.", uRC );

				uCurState = END_STATE;
			}
			else
			{
				ttyOut( "...DosQNmPipeInfo() completed successfully:" );
				ttyOut( "...Outgoing buff size: %u", ((PIPEINFO *)cbInfoBuf)->cbOut );
				ttyOut( "...Incoming buff size: %u", ((PIPEINFO *)cbInfoBuf)->cbIn );
				ttyOut( "...Maximum instances:  %u", ((PIPEINFO *)cbInfoBuf)->cbMaxInst );
				ttyOut( "...Current instances:  %u", ((PIPEINFO *)cbInfoBuf)->cbCurInst );
				ttyOut( "...Length of name:     %u", ((PIPEINFO *)cbInfoBuf)->cbName );
				ttyOut( "...Pipe name:          %s", ((PIPEINFO *)cbInfoBuf)->szName );

				switch ( wTransType )
				{
					case IDM_TRANSACT:

						uCurState = TRANSACT_STATE;
						break;

					case IDM_READ:

						uCurState = READ_STATE;
						break;

					case IDM_WRITE:
					case IDM_READWRITE:

						uCurState = WRITE_STATE;
				}
			}

			break;

		case TRANSACT_STATE:

			ttyOut( "Transacting pipe with/for %u bytes...", usDataSize );

			pszInBuff = GlobalLock( hInBuff );
			pszOutBuff = GlobalLock( hOutBuff );

			if ( uRC = DosTransactNmPipe( hPipe,
			                              pszOutBuff,
			                              usDataSize,
			                              pszInBuff,
			                              usDataSize,
			                              &usTotalRead ) )
			{
				ttyOut( "...ERROR - DosTransactNmPipe() failed, returned %u.", uRC );

				uCurState = END_STATE;
			}
			else
			{	
				ttyOut( "...DosTransactNmPipe() completed successfully, %u bytes read.", usTotalRead );

				uCurState = CHECKDATA_STATE;
			}

			GlobalUnlock( hInBuff );
			GlobalUnlock( hOutBuff );		
									
			break;					

		case WRITE_STATE:

			if ( usTotalWritten < usDataSize )
			{
				usRWSize = min( usDataSize - usTotalWritten, usWriteSize );
				
				ttyOut( "Writing %u Bytes...", usRWSize );

				pszOutBuff = GlobalLock( hOutBuff );

				if ( fAsyncWrite )
				{
					fAsyncComplete = FALSE;

					lpfnWaitAsync = MakeProcInstance( WaitAsync, hInst );
					
					if ( uRC = DosWriteAsyncNmPipe( hPipe,
					                                lpfnWaitAsync,
					                                &usAsyncRC,
					                                pszOutBuff + usTotalWritten,
					                                usRWSize,
					                                &usBytesWritten ) )
					{
						ttyOut( "...ERROR - DosWriteAsyncNmPipe() failed, returned %u.", uRC );

						GlobalUnlock( hOutBuff );

						FreeProcInstance( lpfnWaitAsync );

						uCurState = END_STATE;
					}
					else
					{
						ttyOut( "...DosWriteAsyncNmPipe() called successfully..." );

						uCurState = WRITEASYNC_WAIT_STATE;
					}
				}
				else // !fAsyncWrite
				{
					usBytesWritten = _lwrite( hPipe,
					                          pszOutBuff + usTotalWritten,
					                          usRWSize );

					exterr = dosexterr( NULL );

					if ( usBytesWritten == -1 )
					{
						ttyOut( "...ERROR - _lwrite() failed, extended error %d.", exterr );

						GlobalUnlock( hOutBuff );

						uCurState = END_STATE;
					}
					else
					{
						ttyOut( "..._lwrite() completed successfully, %u bytes written, extended error %d.",
						        usBytesWritten, exterr );

						usTotalWritten += usBytesWritten;
					}
				}
			}
			else
			{
				GlobalUnlock( hOutBuff );

				if (wTransType != IDM_READWRITE)
				{
					uCurState = END_STATE;
				}
				else
				{
					uCurState = READ_STATE;
				}
			}

			break;

		case WRITEASYNC_WAIT_STATE:

			if ( fAsyncComplete )
			{
				ttyOut( "...Completed, returned %u, wrote %u.", usAsyncRC, usBytesWritten );

				usTotalWritten += usBytesWritten;

				FreeProcInstance( lpfnWaitAsync );

				uCurState = WRITE_STATE;
			}
			else
			{
				ttyOut( "...Waiting..." );
			}

			break;

		case READ_STATE:

			if ( usTotalRead < usDataSize )
			{
				usRWSize = min( usDataSize - usTotalRead, usReadSize );

				ttyOut( "Reading %u Bytes...", usRWSize );

				pszInBuff = GlobalLock( hInBuff );

				if ( fAsyncRead )
				{
					fAsyncComplete = FALSE;

					lpfnWaitAsync = MakeProcInstance( WaitAsync, hInst );
					
					if ( uRC = DosReadAsyncNmPipe( hPipe,
					                               lpfnWaitAsync,
					                               &usAsyncRC,
					                               pszInBuff + usTotalRead,
					                               usRWSize,
					                               &usBytesRead ) )
					{
						ttyOut( "...ERROR - DosReadAsyncNmPipe() failed, returned %u.", uRC );

						GlobalUnlock( hInBuff );

						FreeProcInstance( lpfnWaitAsync );

						uCurState = END_STATE;
					}
					else
					{
						ttyOut( "...DosReadAsyncNmPipe() called successfully..." );

						uCurState = READASYNC_WAIT_STATE;
					}
				}
				else // !fAsyncRead
				{
					usBytesRead = _lread( hPipe,
					                      pszInBuff + usTotalRead,
					                      usRWSize );

					exterr = dosexterr( NULL );

					if ( usBytesRead == -1 )
					{
						ttyOut( "...ERROR - _lread() failed, extended error %d.", exterr );

						GlobalUnlock( hInBuff );

						uCurState = END_STATE;
					}
					else
					{
						ttyOut( "..._lread() completed successfully, %u bytes read, extended error %d.",
						        usBytesRead, exterr );

						usTotalRead += usBytesRead;
					}
				}
			}
			else
			{
				GlobalUnlock( hInBuff );

				uCurState = CHECKDATA_STATE;
			}

			break;

		case READASYNC_WAIT_STATE:

			if ( fAsyncComplete )
			{
				ttyOut( "...Completed, returned %u, read %u.", usAsyncRC, usBytesRead );

				usTotalRead += usBytesRead;

				FreeProcInstance( lpfnWaitAsync );

				uCurState = READ_STATE;
			}
			else
			{
				ttyOut( "...Waiting..." );
			}

			break;

		case CHECKDATA_STATE:

			ttyOut( "Checking data, %u Bytes...", usDataSize );

			pszOutBuff = GlobalLock( hOutBuff );
			pszInBuff = GlobalLock( hInBuff );

			for ( pusTemp = (unsigned short far *)pszOutBuff,
			      pusTemp2 = (unsigned short far *)pszInBuff,
			      usIndex = 0;
			      usIndex < usDataSize / 2;
			      pusTemp++, pusTemp2++, usIndex++ )
			{
				if ( *pusTemp != *pusTemp2 )
				{
					ttyOut( "...Corruption found at word %u.", usIndex );

					error = TRUE;

					break;
				}
			}

			if ( !error )
			{
				ttyOut( "...Completed successfully." );
			}

			GlobalUnlock( hOutBuff );
			GlobalUnlock( hInBuff );

			uCurState = END_STATE;

			break;

		case END_STATE:

			ttyOut( "Processing completed, reset or exit." );

			uCurState = WAIT_STATE;

	} // End: switch( uCurState )

} // End: DoState()

//...........................................................................
//
//	FUNCTION: StartTransaction( VOID )
//
//	PURPOSE:  Starts Transaction selected from Execute menu
//
//	COMMENTS:
//
//		This function grays out transaction options, and switches state to
//		START_STATE so processing can start on next timer click or signal
//		from "Step" menu selection depending on single step mode.
//
//...........................................................................

VOID StartTransaction( VOID )
{
	WORD wTransOption;

	// Disable transaction menu options

	for ( wTransOption = TT_LOW;
	      wTransOption <= TT_HIGH;
	      wTransOption++ )
	{
		EnableMenuItem( hMenu, wTransOption, MF_GRAYED );
	}

	// Enable reset menu option

	EnableMenuItem( hMenu, IDM_RESET, MF_ENABLED );

	// Get things rolling

	uCurState = START_STATE;

} // End: StartTransaction

//...........................................................................
//
//	FUNCTION: ResetTransaction( VOID )
//
//	PURPOSE:  Cleans up transaction so that a new one can be started
//
//	COMMENTS:
//
//		This fuction Re-enables transaction options after it cleans up
//		the one that's currently running, and switches state back to
//		WAIT_STATE.
//
//...........................................................................

VOID ResetTransaction( VOID )
{
	WORD  wTransOption;
	int   i;
	DWORD ulSizes = 0;

	// Enable transaction menu items

	for ( wTransOption = TT_LOW;
	      wTransOption <= TT_HIGH;
	      wTransOption++ )
	{
		EnableMenuItem( hMenu, wTransOption, MF_ENABLED );
	}

	// Disable reset menu item

	EnableMenuItem( hMenu, IDM_RESET, MF_GRAYED );

	// Make things spin in wait state

	uCurState = WAIT_STATE;

	// Close pipe if open

	if ( hPipe )
	{
		_lclose( hPipe );
		hPipe = NULL;
	}

	// Free data buffers if allocated

	if ( hInBuff )
	{
		GlobalFree( hInBuff );
		hInBuff = NULL;
	}

	if ( hOutBuff )
	{
		GlobalFree( hOutBuff );
		hOutBuff = NULL;
	}

	// Re-enable options that were disabled	if pipe was opened

	if ( fPipeOpened )
	{
		EnableMenuItem( hMenu, IDM_PIPENAME, MF_ENABLED );

		EnableMenuItem( hMenu, IDM_OPENMODE, MF_ENABLED );

		fPipeOpened = FALSE;
	}

	// Clear screen, reset display defaults

	for ( i=0; i < MAXNUM_RESULT_LINE; i++ )
	{
		Result[ i ].iLen = 0;
	}

	nNumLines = NUM_HEADER_LINE;

	for ( i = 0, nMaxWidth = 0;
	      i < nNumLines;
	      i++ )
	{
		nMaxWidth = max( Header[i].iLen, nMaxWidth );
	}

	SendMessage( hWnd, WM_VSCROLL, SB_TOP, 0 );

	InvalidateRect( hWnd, NULL, TRUE );

	UpdateWindow( hWnd );

	ulSizes = ((ulSizes | cyClient) << 16) | cxClient;

	SendMessage( hWnd, WM_SIZE, 0 , ulSizes );

	iResultNext    = 0;
	fResultWrapped = FALSE;

} // End: ResetTransaction


//...........................................................................
//
//	FUNCTION: WaitAsync()
//
//	PURPOSE:  Callback function passed to Dos???AsyncNmPipe calls
//
//  COMMENTS:
//
//		Simply sets a global flag.
//
//..........................................................................

VOID FAR PASCAL WaitAsync( unsigned long ulJunk )
{
	fAsyncComplete = TRUE;
}
	
