/****************************************************************************

    DIALOGS.C

		Code for dialog box routines of the PipeTool Windows client.

	PURPOSE:

		This is the windows client side of a simple pipe tool used to test
		the basic functionality of pipe IO calls. All basic forms of IO are
		available (read, write, transact, and call), virtually all pipe
		parameters can be set, and data is checked for corruption when read.
		The catch is that a pipe is used only once (i.e. it is closed after
		each requested transaction is completed.)

	COMMENTS:

		Windows can have several copies of this application running at the
		same time.  The variable hInst keeps track of which instance this
		application is so that processing will be to the correct window.
		Also in order to not shut out user input to windows while this
		program is running it is chopped up into pieces (states) and then
		that state machine is traversed with each step triggered by a timer
		or by clicking on the "step!" menu item when in single step mode.

		This program must be used with server program spt.exe which runs on
		OS/2.  These programs make no assumption that the parameters you
		specify are correct. That is, it was designed to try out situations
		that would not work.  See pipetool.doc for more information regarding
		this.

	FUNCTIONS:

		This file:
	
			About() - processes messages for "About" dialog box
			PipeName() - processes messages for "Pipe Name" dialog box
			DataSIzes() - processes messages for "Data Sizes" dialog box
			OpenMode() - processes messages for "Open Mode" dialog box
			ResetOMButtons() - Checks radio buttons based on OM value
			PipeMode() - processes messages for "Pipe Mode" dialog box
			ResetPMButtons() - Checks radio buttons based on PM value
			SaveToFile() - processes messages for "Save" dialog box

		In wincpt.c

			WinMain() - application and instance init, main message loop
			MainWndProc() - processes messages for main window

		In winout.c:																

			ttyOut() - Output function for results
			UpdateLine() - Output function when header info is changed

		In dostate.c

			DoState() - Does the state machine traversal stuff.
			StartTransaction() - gets a selected transaction rolling
			ResetTransaction() - cleans up aborted or finished transaction
			WaitAsync() - Callback function for async pipe calls

	Copyright (C) 1992 Microsoft Corporation.

	This code sample is provided for demonstration purposes only.
	Microsoft makes no warranty, either express or implied,
	as to its usability in any given situation.

****************************************************************************/

#include <windows.h>		// Windows Includes

#include <stdio.h>      // C Runtime Includes
#include <stdlib.h>
#include <string.h>

#define INCL_NETNMPIPE
#include <lan.h>        // LanMan Includes

#include "wincpt.h"		// Include specifically for windows PipeTool

//...........................................................................
//
//	FUNCTION: About( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "About" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		No initialization is needed for this particular dialog box, but TRUE
//		must be returned to Windows.
//
//		Wait for user to click on "Ok" button, then close the dialog box.
//
//...........................................................................

BOOL FAR PASCAL About( HWND     hDlg,
                       unsigned message,
                       WORD     wParam,
                       LONG     lParam )
{
	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			return( TRUE );

		case WM_COMMAND: // Process messages from controls

			if ( wParam == IDOK    ||
			     wParam == IDCANCEL )
			{
				EndDialog( hDlg, TRUE );

				return( TRUE );
			}
	}

	return ( FALSE );	// Didn't process a message

} // End: About


//...........................................................................
//
//	FUNCTION: PipeName( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "Pipe Name" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		Must initialize control to current value of pipename parameter.
//		Cancel button causes nothing to be changed in the main window. Selecting
//		OK changes global variables and causes header to be repainted on main
//		window if necessary. After the test is past the open state ( or the
//		equivalent ) this dialog will be unavailable.
//
//...........................................................................

BOOL FAR PASCAL PipeName( HWND     hDlg,
                          unsigned message,
                          WORD     wParam,
                          LONG     lParam )
{
	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			SetDlgItemText( hDlg, IDD_PIPENAME, pszPipeName );

			return( TRUE );
	
		case WM_COMMAND: // Process actions on controls

			switch ( wParam )
			{
				case IDCANCEL: // Exit with nothing changed

					EndDialog( hDlg, FALSE );

					return( TRUE );

				case IDOK: // Save changes made

					GetDlgItemText( hDlg, IDD_PIPENAME, pszPipeName, PIPENAME_LEN );

					EndDialog( hDlg, TRUE );

					return( TRUE );
			}
	}

	return( FALSE );

} // End: PipeName


//...........................................................................
//
//	FUNCTION: DataSizes( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "Data Sizes" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		Must initialize controls to current value of data size parameters.
//		Cancel button causes nothing to be changed in the main window. Selecting
//		OK changes global variables and causes header to be repainted on main
//		window if necessary. When the test is past the Open state (or the
//		equivalent) the "Data SIze" control needs to be disabled.
//
//...........................................................................

BOOL FAR PASCAL DataSizes( HWND     hDlg,
                           unsigned message,
                           WORD     wParam,
                           LONG     lParam )
{
	BOOL retry = FALSE;
	BOOL ok;
	WORD temp;

	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			SetDlgItemInt( hDlg, IDD_DATASIZE, usDataSize, 0 );

			SetDlgItemInt( hDlg, IDD_READSIZE, usReadSize, 0 );

			SetDlgItemInt( hDlg, IDD_WRITESIZE, usWriteSize, 0 );

			return ( TRUE );

		case WM_COMMAND: // Process actions on controls

			switch ( wParam )
			{
				case IDCANCEL: // Exit with nothing changed

					EndDialog( hDlg, FALSE );

					return( TRUE );

				case IDOK: // Save changes made

					temp = GetDlgItemInt( hDlg, IDD_DATASIZE, &ok, 0 );

					if ( !ok )
					{
						MessageBox( hDlg, "Invalid Data Size!", NULL, MB_OK | MB_ICONEXCLAMATION );

						retry = TRUE;
					}
					else
					{
						if ( (uCurState > START_STATE) && (temp != usDataSize) )
						{
							MessageBox( hDlg, "Can't change data size now that its allocated.", NULL, MB_OK | MB_ICONEXCLAMATION );
						}
						else
						{
							usDataSize = temp;
						}	
					}

					temp = GetDlgItemInt( hDlg, IDD_READSIZE, &ok, 0 );

					if ( !ok )
					{
						MessageBox( hDlg, "Invalid Read Size!", NULL, MB_OK | MB_ICONEXCLAMATION );

						retry = TRUE;
					}
					else
					{
						if ( (uCurState > READASYNC_WAIT_STATE) && (temp != usReadSize) )
						{
							MessageBox( hDlg, "Can't change read size, reading has completed.", NULL, MB_OK | MB_ICONEXCLAMATION );
						}
						else
						{
							usReadSize = temp;
						}	
					}

					temp = GetDlgItemInt( hDlg, IDD_WRITESIZE, &ok, 0 );

					if ( !ok )
					{
						MessageBox( hDlg, "Invalid Write Size!", NULL, MB_OK | MB_ICONEXCLAMATION );

						retry = TRUE;
					}
					else
					{
						if ( (uCurState > WRITEASYNC_WAIT_STATE) && (temp != usWriteSize) )
						{
							MessageBox( hDlg, "Can't change write size, writing has completed.", NULL, MB_OK | MB_ICONEXCLAMATION );
						}
						else
						{
							usWriteSize = temp;
						}	
					}

					if ( !retry )
					{
						EndDialog( hDlg, TRUE );
					}

					return( TRUE );

				default:

					return( FALSE ); // Didn't process a message
			}
	
			break;
	}

	return ( FALSE ); // Didn't process a message

} // End: DataSizes


//...........................................................................
//
//	FUNCTION: Openode( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "Open Mode" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		Must initialize controls to current value of open mode parameters.
//		Cancel button causes nothing to be changed in the main window. Selecting
//		OK changes global variables and causes header to be repainted on main
//		window if necessary.	Also actions on the value in the edit control
//		will if possible show what modes they relate to in the radio buttons
//		and vise versa.
//
//...........................................................................

BOOL FAR PASCAL OpenMode( HWND     hDlg,
                          unsigned message,
                          WORD     wParam,
                          LONG     lParam )
{
	char pszBuff[10]; // For conversion of integer to string values

	static unsigned short usTemp; // Temporary storage of usOpenMode

	int MsgRet; // MessageBox used in IDOK processing return value

	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			usTemp = usOpenMode;

			sprintf( pszBuff, "0x%x", usOpenMode );

			SetDlgItemText( hDlg, IDD_OPENMODE, pszBuff );

			return ( TRUE );

		case WM_COMMAND: // Process actions on controls

			if ( (wParam >= IDD_DENYCOMPAT) && (wParam <= IDD_DENYREADWRITE) )
			{
				usTemp &= OM_SHARE_MASK;

				switch ( wParam )
				{
					case IDD_DENYNONE:

						usTemp |= OF_SHARE_DENY_NONE;

						break;
				
					case IDD_DENYREAD:

						usTemp |= OF_SHARE_DENY_READ;

						break;

					case IDD_DENYWRITE:

						usTemp |= OF_SHARE_DENY_WRITE;

						break;

					case IDD_DENYREADWRITE:

						usTemp |= OF_SHARE_EXCLUSIVE;

						break;

					case IDD_DENYCOMPAT:

						usTemp |= OF_SHARE_COMPAT;

						break;
				}

				sprintf( pszBuff, "0x%x", usTemp );
	
				SetDlgItemText( hDlg, IDD_OPENMODE, pszBuff );

				return ( TRUE );
			}

			if ( (wParam >= IDD_WRITEONLY) && (wParam <= IDD_READWRITE) )
			{
				usTemp &= OM_ACCESS_MASK;

				switch ( wParam )
				{
					case IDD_READONLY:

						usTemp |= OF_READ;

						break;

					case IDD_WRITEONLY:

						usTemp |= OF_WRITE;

						break;

					case IDD_READWRITE:

   						usTemp |= OF_READWRITE;

						break;
				}

				sprintf( pszBuff, "0x%x", usTemp );
	
				SetDlgItemText( hDlg, IDD_OPENMODE, pszBuff );

				return ( TRUE );
			}

			switch( wParam )
			{
				case IDD_OPENMODE:
					
					if ( HIWORD( lParam ) == EN_CHANGE )
					{
						GetDlgItemText( hDlg, IDD_OPENMODE, pszBuff, 10 );
					
						sscanf( pszBuff, "%x", &usTemp );

						ResetOMButtons( hDlg, usTemp );

						return( TRUE );
					}

					break;

				case IDCANCEL: // Exit with nothing changed

					EndDialog( hDlg, FALSE );

					return( TRUE );

				case IDOK: // Save changes made

					GetDlgItemText( hDlg, IDD_OPENMODE, pszBuff, 10 );
					
					if ( !sscanf( pszBuff, "%x", &usTemp ) )
					{
						usTemp = usOpenMode;

						MsgRet = MessageBox( hDlg,
						                     "Wrong! Not a valid hex value, resetting to original.",
									         NULL,
									         MB_RETRYCANCEL | MB_ICONEXCLAMATION );
						switch( MsgRet )
						{
							case IDRETRY:
				
								sprintf( pszBuff, "0x%x", usTemp );
	
								SetDlgItemText( hDlg, IDD_OPENMODE, pszBuff );
						
								break;

							case IDCANCEL:
							
								EndDialog( hDlg, FALSE );
						}
					}
					else
					{
						usOpenMode = usTemp;

						EndDialog( hDlg, TRUE );
					}

					return( TRUE );
			}
	}
	
	return ( FALSE ); // Didn't process a message

} // End: OpenMode

// ..........................................................................
//
//  FUNCTION: ResetOMButtons( HWND, unsigned short )
//
//  COMMENTS:
//
//		Sets up radio buttons when open mode dialog is initialized and when
//		OpenMode edit control is modified directly.
//
// ..........................................................................

VOID ResetOMButtons( HWND hDlg, unsigned short usNewOpenMode )
{
	switch( usNewOpenMode & ~OM_SHARE_MASK ) // mask all but share mode bits
	{
		case OF_SHARE_DENY_NONE:

			CheckRadioButton( hDlg, IDD_DENYCOMPAT, IDD_DENYREADWRITE, IDD_DENYNONE );

			break;

		case OF_SHARE_DENY_READ:

			CheckRadioButton( hDlg, IDD_DENYCOMPAT, IDD_DENYREADWRITE, IDD_DENYREAD );

			break;

		case OF_SHARE_DENY_WRITE:

			CheckRadioButton( hDlg, IDD_DENYCOMPAT, IDD_DENYREADWRITE, IDD_DENYWRITE );

			break;

		case OF_SHARE_EXCLUSIVE:

			CheckRadioButton( hDlg, IDD_DENYCOMPAT, IDD_DENYREADWRITE, IDD_DENYREADWRITE );

			break;

		case OF_SHARE_COMPAT:

			CheckRadioButton( hDlg, IDD_DENYCOMPAT, IDD_DENYREADWRITE, IDD_DENYCOMPAT );
	}

	switch( usNewOpenMode & ~OM_ACCESS_MASK ) // mask all but access mode bits
	{
		case OF_READ:

			CheckRadioButton( hDlg, IDD_WRITEONLY, IDD_READWRITE, IDD_READONLY );

			break;

		case OF_WRITE:

			CheckRadioButton( hDlg, IDD_WRITEONLY, IDD_READWRITE, IDD_WRITEONLY );

			break;

		case OF_READWRITE:

			CheckRadioButton( hDlg, IDD_WRITEONLY, IDD_READWRITE, IDD_READWRITE );
	}

} // End: ResetOMButtons


//...........................................................................
//
//	FUNCTION: PipeMode( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "Pipe Mode" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		Must initialize controls to current value of pipe mode parameters.
//		Cancel button causes nothing to be changed in the main window. Selecting
//		OK changes global variables and causes header to be repainted on main
//		window if necessary.	Also actions on the value in the edit control
//		will if possible show what modes they relate to on the radio buttons
//		and vise versa.
//
//.............................................................................

BOOL FAR PASCAL PipeMode( HWND     hDlg,
                          unsigned message,
                          WORD     wParam,
                          LONG     lParam )
{
	char pszBuff[10]; // For conversion of integer to string values

	static unsigned short usTemp; // Temporary storage of us PipeMode

	int MsgRet; // MessageBox used in IDOK processing return value

	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			usTemp = usPipeMode;

			sprintf( pszBuff, "0x%x", usPipeMode );

			SetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff );

			return ( TRUE );

		case WM_COMMAND: // Process actions on controls

			if ( (wParam == IDD_WAIT) || (wParam == IDD_NOWAIT) )
			{
				usTemp &= PM_WAIT_MASK;

				switch ( wParam )
				{
					case IDD_WAIT:

						usTemp |= NP_WAIT;

						break;
				
					case IDD_NOWAIT:

						usTemp |= NP_NOWAIT;
				}

				sprintf( pszBuff, "0x%x", usTemp );
	
				SetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff );

				return ( TRUE );
			}

			if ( (wParam == IDD_MESSAGE) || (wParam == IDD_BYTE) )
			{
				usTemp &= PM_READ_MASK;

				switch ( wParam )
				{
					case IDD_MESSAGE:

						usTemp |= NP_READMODE_MESSAGE;

						break;

					case IDD_BYTE:

						usTemp |= NP_READMODE_BYTE;
				}

				sprintf( pszBuff, "0x%x", usTemp );
	
				SetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff );

				return ( TRUE );
			}

			switch( wParam )
			{
				case IDD_PIPEMODE:
					
					if ( HIWORD( lParam ) == EN_CHANGE )
					{
						GetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff, 10 );
					
						sscanf( pszBuff, "%x", &usTemp );

						ResetPMButtons( hDlg, usTemp );

						return( TRUE );
					}

					break;

				case IDCANCEL: // Exit with nothing changed

					EndDialog( hDlg, FALSE );

					return( TRUE );

				case IDOK: // Save changes made

					GetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff, 10 );
					
					if ( !sscanf( pszBuff, "%x", &usTemp ) )
					{
						usTemp = usPipeMode;

						MsgRet = MessageBox( hDlg,
						                     "Wrong! Not a valid hex value, resetting to original.",
									         NULL,
									         MB_RETRYCANCEL | MB_ICONEXCLAMATION );
						switch( MsgRet )
						{
							case IDRETRY:

								sprintf( pszBuff, "0x%x", usTemp );
	
								SetDlgItemText( hDlg, IDD_PIPEMODE, pszBuff );
						
								break;

							case IDCANCEL:
							
								EndDialog( hDlg, FALSE );
						}
					}
					else
					{
						usPipeMode = usTemp;

						EndDialog( hDlg, TRUE );
					}

					return( TRUE );
			}
	}
	
	return ( FALSE ); // Didn't process a message

} // End: PipeMode

// ..........................................................................
//
//  FUNCTION: ResetPMButtons( HWND, unsigned short )
//
//  COMMENTS:
//
//		Sets up radio buttons when pipe mode dialog is initialized and when
//		PipeMode edit control is modified directly.
//
// ..........................................................................

VOID ResetPMButtons( HWND hDlg, unsigned short usNewPipeMode )
{
	switch( usNewPipeMode & ~PM_WAIT_MASK ) // mask all but blocking mode bit
	{
		case NP_WAIT:

			CheckRadioButton( hDlg, IDD_WAIT, IDD_NOWAIT, IDD_WAIT );

			break;

		case NP_NOWAIT:

			CheckRadioButton( hDlg, IDD_WAIT, IDD_NOWAIT, IDD_NOWAIT );
	}

	switch( usNewPipeMode & ~PM_READ_MASK ) // mask all but read mode bit
	{
		case NP_READMODE_MESSAGE:

			CheckRadioButton( hDlg, IDD_MESSAGE, IDD_BYTE, IDD_MESSAGE );

			break;

		case NP_READMODE_BYTE:

			CheckRadioButton( hDlg, IDD_MESSAGE, IDD_BYTE, IDD_BYTE );
	}

} // End: ResetPMButtons


//...........................................................................
//
//	FUNCTION: SaveToFile( HWND, unsigned, WORD, LONG )
//
//	PURPOSE:  Processes messages for "Save" dialog box
//
//	MESSAGES:
//
//		WM_INITDIALOG - initialize dialog box
//		WM_COMMAND - Input received
//
//	COMMENTS:
//
//		No need to initialize anything, Just get filename.
//
//...........................................................................

BOOL FAR PASCAL SaveToFile( HWND     hDlg,
                            unsigned message,
                            WORD     wParam,
                            LONG     lParam )
{
	switch ( message )
	{
		case WM_INITDIALOG: // Initialize dialog box

			return ( TRUE );

		case WM_COMMAND: // Process actions on controls

			switch ( wParam )
			{
				case IDCANCEL: // Exit with nothing changed

					EndDialog( hDlg, FALSE );

					return( TRUE );

				case IDOK: // Save changes made

					GetDlgItemText( hDlg, IDD_FILENAME, pszSaveFileName, FILENAME_LEN );

					EndDialog( hDlg, TRUE );

					return( TRUE );
			}
	}

	return ( FALSE ); // Didn't process a message

} // End: SaveToFile


