/*
   NBSTAT.C -- a sample program demonstrating the NetBIOS API functions

      This application demonstrates a program which can submit NCBs
      through the MSDOS int 0x5C method, the NetBIOSSubmit API, or the
      NetBIOS API.

   Compile Instructions:

      Microsoft C 6.0:
         CL /C NBSTAT.C
         LINK /FARCALL NBSTAT,,,OS2.LIB+SLIBCEP.LIB+LAN.LIB+SAMPLES.LIB;

         If the program is going to be run under MSDOS, it may be bound with
         the following command line:

         BIND NBSTAT.EXE LAN.LIB DOSLAN.LIB /n @NBSTAT.BND
      

      usage: nbstat [-s] [-m] [-x] [-n netname | -d lana]
                    [-a address | -r remotename]
         where -s       = pad the remotename to 16 bytes using spaces rather
                          than nulls
               -m       = remote name is a LAN Manager computer name
               -x       = Do not submit a reset before submitting NCBs
               -n       = Use NetBIOSSubmit() on net "netname"
               -d       = Use NetBIOS() on the specified adapter number
               -r       = Remote name from which to get status
               -a       = Adapter address from which to get status

   API              Used to...
   ============     ================================================
   NetBiosOpen	    Open the adapter
   NetBiosSubmit    Submit an NCB to the network
   NetBiosClose     Close the adapter
   NetBios	    Submit NCBs to the network in conformance with NetBIOS 3.0

   This code sample is provided for demonstration purposes only.
   Microsoft makes no warranty, either express or implied, 
   as to its usability in any given situation.
*/


#define INCL_DOS
#define INCL_DOSMODULEMGR
#include <os2.h>			// OS2 header files

#define INCL_NETBIOS
#define INCL_NETERRORS
#include <lan.h>			// LAN Manager header files

#include <stdio.h>			// C run-time header files
#include <stdlib.h>
#include <string.h>

#include "samples.h"			// Internal routine header files

#define OS2		0		// Indicators for OS version
#define DOS		1

#define UNINITIALIZED   0		// States of the remote name buffer
#define AADDRESS        1
#define NAME            2

#define	NBAPI		0		// Use NetBios()
#define NBSUBMIT	1		// Use NetBiosSubmit()
#define INT5C		2		// Use int5c() (DOS only)

#define RESET		0		// Submit an NCB.RESET before starting
#define NORESET		1		// Do not submit an NCB.RESET

#define MAXNAMES	20		// Maximum names to can display

#define BUFSIZE sizeof(struct _astatbuf) + MAXNAMES*sizeof (struct _namebuf)

/*
   Forward declarations
*/

USHORT MyNetBios (void far *);
USHORT int5c (void far * pNCB);
void Usage(char *pszString);

/*
   Global variable holding the Operating System type
*/

USHORT fOSType;

void main (int argc, char *argv[]) 
{
   NCB	      sNcb;			// Our NCB structure
   PASTATBUF  pAstatBuf;		// Returned adapter status buffer
   USHORT     i, j;
   BYTE       bFiller = 0;		// default filler byte = 0
   BYTE	      bLana = 0;		// default to adapter 0
   BYTE       bLastByte = bFiller;

   PNAMEBUF   pNameBuf;

   HMODULE    hmod;
   UCHAR      szFailName[CCHMAXPATH];
   USHORT     rc;
   UCHAR     *pszNetName = "NET1";
   UCHAR     *pszAdapterAddress;
   UCHAR     *pszHexString = "0123456789ABCDEF";
   UCHAR     *pszMatchString = "X";

   BYTE       bHexval;

   USHORT     usDosVersion;
   
   BYTE       fNameFlag = UNINITIALIZED;
   BYTE       fResetFlag = RESET;	// Default to submitting an NCB.RESET

   USHORT     hNBOpen;			// NetBiosOpen handle
   USHORT     usSubmitType = NBAPI;	// Default to NetBios()
   UCHAR      stRemoteName[NCBNAMSZ];

   memset (&stRemoteName, bFiller, NCBNAMSZ);	// default to local status
   stRemoteName[0] = '*';

/*
   Process command line switches
*/

   for (i = 1; i < (USHORT) argc; i++)
   {
      if ((*argv[i] == '-') || (*argv[i] == '/'))
      {
         switch (tolower(*(argv[i]+1)))
         {
	    case 'a':
               if (fNameFlag != UNINITIALIZED) {
                  Usage(argv[0]);
               }
               pszAdapterAddress = SafeMalloc (NCBNAMSZ);
               strcpy (pszAdapterAddress, strupr(argv[++i]));
               memset (&stRemoteName, 0, NCBNAMSZ);
               for (i = 0; i < 12; i++) {
                  pszMatchString[0] = (char) *pszAdapterAddress++;
                  bHexval = LOBYTE(strcspn(pszHexString,pszMatchString) * 16);
                  pszMatchString[0] = (char) *pszAdapterAddress++;
                  bHexval += strcspn(pszHexString,pszMatchString);
                  stRemoteName[i+10] = bHexval;
               }
               fNameFlag = AADDRESS;
               break;
            case 'x':
               fResetFlag = NORESET;
               break;
            case 'n':                        // -n network name
               pszNetName = argv[++i];
               usSubmitType = NBSUBMIT;	    // implies netbiossubmit()
               break;
            case 'd':
               bLana = (BYTE) atoi(argv[++i]);
               usSubmitType = NBAPI;
               break;
            case 'r':
               if (fNameFlag != UNINITIALIZED)
                  Usage(argv[0]);
               strncpy (stRemoteName, &(*argv[++i]), (NCBNAMSZ-1>strlen(&(*argv[i]))?strlen(&(*argv[i+1])):NCBNAMSZ-1));
	       break;
            case 's':
               if (i != 1)
                  Usage(argv[0]);
               bFiller = ' ';	// filler is spaces
               bLastByte = bFiller;
               memset (&stRemoteName, bFiller, NCBNAMSZ);
               break;
            case 'm':
               bLastByte = 0x00;	// last byte is LM machine name marker
               break;
            default:
               Usage(argv[0]);
         }
      }
      else
         Usage(argv[0]);
   }

/*
   Determine the version of the operating system.
*/

   fOSType = OS2;
   DosGetVersion (&usDosVersion);
   if (LOBYTE(usDosVersion) < 10) {
      if (usSubmitType == NBSUBMIT) {
         printf ("The -n switch is valid only under OS/2.\n");
         exit (1);
      }
      fOSType = DOS;
      usSubmitType = INT5C;
   }


/*
   Setup the remote name to submit a status for.
*/

   if (fNameFlag != AADDRESS)
      stRemoteName[NCBNAMSZ-1]=bLastByte;

/*
   Since this program is bindable, use DosLoadModule and DosGetProcAddr to
   get the pNetBios entrypoint.  If this fails, default to trying to use
   NetBiosSubmit().
*/

   if (usSubmitType == NBAPI) {
      DosLoadModule (szFailName, sizeof(szFailName), "ACSNETB", &hmod);
      rc = DosGetProcAddr (hmod, "NETBIOS", &pNetBios);

      if (rc) {
         printf ("Cannot use NetBIOS() submission, " \
                 "DosGetProcAddr rc= 0x%x (%u)\n",rc, rc);
         printf ("Attempting to use NetBiosSubmit() interface on %s\n\n",
                 pszNetName);
         usSubmitType = NBSUBMIT;
      }
   }

/*
   If the submission method is NetBios, first initialize the adapter by
   submitting an NCB.RESET command.  Under DOS, this will completely reset
   the adapter, and terminate all current network connections.  Under OS2,
   an NCB.RESET must be submitted by all applications prior to submitting
   NCBs on that adapter.

   To ensure that the NETBIOS.OS2 driver is loaded, submit a test hangup NCB
   and check the return code.
*/

   if ( ((usSubmitType == NBAPI) | (usSubmitType == INT5C)) &&
        (fResetFlag == RESET)) {
      sNcb.ncb_command = NCBRESET;
      sNcb.ncb_lsn = 0;			// Don't free resources
      sNcb.ncb_lana_num = bLana;	// Adapter to reset
      memset (&(sNcb.ncb_callname), 0, NCBNAMSZ);  // don't use bFiller here!

      MyNetBios (&sNcb);

      if (sNcb.ncb_retcode != NRC_GOODRET) {
         if (sNcb.ncb_retcode == NRC_BRIDGE) {
            printf ("Error: Invalid adapter number; NCB retcode = 0x%x\n",
                    sNcb.ncb_retcode);
            exit(sNcb.ncb_retcode);
         }
         if (sNcb.ncb_retcode >= NRC_SYSTEM) {
            printf ("Fatal error during reset.  Retcode = 0x%x\n",
                    sNcb.ncb_retcode);
            exit (sNcb.ncb_retcode);
         }
         printf ("Warning: Reset returned with 0x%x (%u).  Status may fail\n",
                 sNcb.ncb_retcode, sNcb.ncb_retcode);
      }

      sNcb.ncb_command = NCBHANGUP;
      sNcb.ncb_lsn = 0;			// invalid LSN number
      sNcb.ncb_lana_num = bLana;

      MyNetBios (&sNcb);

      if (sNcb.ncb_retcode >= NRC_ENVNOTDEF) {
         printf ("The NETBIOS.OS2 driver does not appear to be loaded.\n");
         printf ("  Attempting to use NetBiosSubmit() interface on %s.\n\n",
                 pszNetName);
         usSubmitType = NBSUBMIT;
      }
   }

/*
   If the submission method is NetBiosSubmit, first initialize the adapter
   through a NetBiosOpen call.
*/

   if (usSubmitType == NBSUBMIT) {
      rc = NetBiosOpen (pszNetName,0, NB_REGULAR, &hNBOpen);

      if (rc) {
         printf ("NetBiosOpen failed: 0x%x (%u)\n", rc, rc);
         exit (rc);
      }
   }

/*
   Allocate memory for the adapter status buffer and the names to be returned.
*/

   pAstatBuf = (PASTATBUF) SafeMalloc(BUFSIZE);

/* 
   Prepare to submit the adapter status request
*/

   sNcb.ncb_command = NCBASTAT;
   sNcb.ncb_buffer = (char far *) pAstatBuf;
   sNcb.ncb_length = BUFSIZE;
   sNcb.ncb_lana_num = bLana;

   memset (&(sNcb.ncb_name), bFiller, NCBNAMSZ);
   memset (&(sNcb.ncb_callname), bFiller, NCBNAMSZ);
   memcpy (&(sNcb.ncb_callname), &stRemoteName, NCBNAMSZ);

   if (fNameFlag != AADDRESS)
      sNcb.ncb_callname[NCBNAMSZ-1] = bLastByte;

   rc = (usSubmitType == NBSUBMIT)?NetBiosSubmit(hNBOpen, 0, &sNcb):
                                   MyNetBios(&sNcb);

   if ( (sNcb.ncb_retcode != 0) && (sNcb.ncb_retcode != 6)) {
      if ( (sNcb.ncb_retcode >= NRC_SYSTEM) || 
           (sNcb.ncb_retcode == NRC_ENVNOTDEF)) {
         printf ("Fatal error from Ncb: %#x (%u)\nCannot perform status.\n",
                 sNcb.ncb_retcode, sNcb.ncb_retcode);
         exit (sNcb.ncb_retcode);
      } else {
         printf ("*** WARNING ***\n");
         printf ("NCB returned with error 0x%x (%u); data may be garbage\n\n",
                 sNcb.ncb_retcode, sNcb.ncb_retcode);
      }
   }

/*
   Display the adapter status information.
*/

   printf ("Burned in adapter address:  0x");
   for (i = 0; i < 6; i++)
      if (pAstatBuf->adapter_address[i] < 0x10) {
        printf ("0%1x", pAstatBuf->adapter_address[i]);
      } else {
         printf ("%2x", pAstatBuf->adapter_address[i]);
      }
   printf ("\n");
   
   printf ("NetBios Rev level: %u.%u,  ", pAstatBuf->rev_major,
                                          pAstatBuf->rev_minor);
   printf ("adapter type: 0x%x\n", pAstatBuf->adapter_type);
   printf ("Duration of reporting period: %u\n", pAstatBuf->duration);
   printf ("Reserved1 field: %#4x:%#4x (%15lu)\n",
           HIUSHORT(pAstatBuf->reserved1),
           LOUSHORT(pAstatBuf->reserved1), pAstatBuf->reserved1);
   printf ("%lu successful packets sent, ", pAstatBuf->xmit_success);
   printf ("%lu successful packets recieved\n", pAstatBuf->recv_success);
   printf ("%u transmissions aborted, %u transmissions retried\n",
           pAstatBuf->xmit_aborts, pAstatBuf->iframe_xmit_err);
   printf ("%u packets recieved incorrectly\n", pAstatBuf->iframe_recv_err);
   printf ("%u FRMR sent, %u FRMR recieved\n", pAstatBuf->frmr_xmit,
           pAstatBuf->frmr_recv);
   printf ("%u t1 timeouts, %u ti timeouts\n", pAstatBuf->t1_timeouts,
           pAstatBuf->ti_timeouts);
   printf ("%u times out of recieve buffers, %u times out of xmit buffers\n",
           pAstatBuf->recv_buff_unavail, pAstatBuf->xmit_buf_unavail);
   printf ("%u Ncbs free, out of %u Ncbs configured (%u max configurable)\n",
           pAstatBuf->free_ncbs, pAstatBuf->max_cfg_ncbs, pAstatBuf->max_ncbs);
   printf ("Maximum datagram size: %u, Maximum session data packet size: %u\n",
           pAstatBuf->max_dgram_size, pAstatBuf->max_sess_pkt_size);
   printf ("%u pending sessions, out of %u sessions available "
           "(%u max configurable)\n", pAstatBuf->pending_sess,
           pAstatBuf->max_cfg_sess, pAstatBuf->max_sess);
   printf ("Registered names: %u\n", pAstatBuf->name_count);

/*
   Check that name_count is reasonable.  This is to handle the case where
   NCB.ASTAT returned with an error code, but the information is still
   reported.
*/

   if (pAstatBuf->name_count <= 255) {
      printf ("\n ## :  Netbios Name\n");

                                          // Find start of names
      pNameBuf = (PNAMEBUF) ASTAT_NAME_BUFFER(pAstatBuf);

      for (i = 0; (i < MAXNAMES) && (i < pAstatBuf->name_count); i++) {
         printf (" %2xh: ", pNameBuf->name_num);
                                          // Names are not zero-terminated
                                          // so print them by character
         for (j = 0; (j < NCBNAMSZ); j++) {
            printf ("%c", *(pNameBuf->name + j));
         }
         printf (", last byte: 0x%2x", *(pNameBuf->name + NCBNAMSZ-1));
         printf (", flags 0x%2x (", pNameBuf->name_flags & NAME_FLAGS_MASK);
   
         switch (pNameBuf->name_flags & (NAME_FLAGS_MASK - GROUP_NAME)) {
            case (REGISTERING):
               printf ("Registering");
               break;
            case (REGISTERED):
               printf ("Registered");
               break;
            case (DEREGISTERED):
               printf ("Deregistered");
               break;
            case (DUPLICATE):
               printf ("Duplicate");
               break;
            case (DUPLICATE_DEREG):
               printf ("Dup - degistering");
               break;
         }
         if (pNameBuf->name_flags & GROUP_NAME) {
            printf (" GROUP");
         }
         printf (")\n");
         pNameBuf += 1;
      }
   }

   free (pAstatBuf);
   exit (0);
}


/*
   The following routine is for NCB submission in an MSDOS environment.
*/

USHORT int5c (void far *pNCB)
{
   USHORT rc;

   _asm {
      push     es
      push     bx

      les      bx, pNCB
      int      5ch

      sub      ah, ah

      pop      bx
      pop      es
      mov      rc, ax
   }

   return (rc);
}


/*
   The following routine is for submissions other than NetBiosSubmit.
*/

USHORT MyNetBios (void far *pNCB)
{
   return((fOSType==OS2)?pNetBios(pNCB):int5c(pNCB));
}


void Usage(char *pszString)
{
   fprintf(stderr, "Usage:  %s [-s]", pszString);
   fprintf(stderr, " [-m] [-x] [-n netname | -d lana_num]\n");
   fprintf(stderr, "   [-a aaddress | -r remotename]\n");
   fprintf(stderr, "\n-s: pad remotename with spaces rather than nulls\n");
   fprintf(stderr, "    If included, the '-s' parameter must come first.\n");
   fprintf(stderr, "-m: remote name is LAN Manager machine name\n");
   fprintf(stderr, "-x: do not submit an NCB.RESET before submitting NCBs.\n");
   fprintf(stderr, "-n: use NetBiosSubmit() virtual net names\n");
   fprintf(stderr, "-d: use NetBios() lana numbers\n");
   fprintf(stderr, "-a: use remote adapter address (hex - must be 12 chars)\n");
   fprintf(stderr, "-r: get status from remote name\n");
   fprintf(stderr, "    ** Names are case sensitive **\n");
   exit(1);
}
