/* Copyright (c) Oracle Corporation 1994.  All Rights Reserved */

/*
    This source code is provided as a debugging aid for developers
    who have purchased Oracle Objects for OLE    .  Please see the
    online help for documentation of these classes.
*/

/*
    Oracle Objects for OLE     C++ Classes
    
    This file is the header for the basic classes.  That is, all the
    classes except OBound and OBinder (which use the obound.h header)
                           
    CREATED    ********   11/22/94

*/

#ifndef ORACL_ORACLE
#define ORACL_ORACLE

// compiler specific defines
#ifdef _MSC_VER
#define OEXPORT _export
#define OHUGESP
#endif

#ifdef __BORLANDC__
#define OEXPORT _export
#define OHUGESP
#endif

#ifndef OEXPORT
#define OEXPORT
#define OHUGESP
#endif
        

// handy boolean type
typedef int oboolean;
// simple success return value
typedef int oresult;

// oresults can have one of the following values
#define OSUCCESS 0
#define OFAILURE 101

// error numbers returned by the ErrorNumber method.

// ----- error numbers
#define OERROR_NONE 0  // there isn't an error

#define OERROR_NOINTER 11  // we couldn't get a needed interface
#define OERROR_MEMORY 12  // memory allocation problem
#define OERROR_BADERR 13  // error in error handling (!)
#define OERROR_INVPARENT 14 // parent object on open is invalid (not open)
#define OERROR_SYSTEM 15     // some system error
#define OERROR_NOTOPEN 16   // attempt to use unopened object
#define OERROR_BADARG 17    // bad argument to routine
#define OERROR_INVRECORD 18 // the current record is invalid

#define OERROR_ADVISEULINK 4096  // not an advisory connection
#define OERROR_DBCONNECT 4097  // connection not made
#define OERROR_POSITION 4098  // invalid database position
#define OERROR_NOFIELDNAME 4099  // field not found
#define OERROR_NOFIELDINDEX 4100  // invalid field index
#define OERROR_TRANSIP 4101  // transaction already in progress
#define OERROR_SCHEMAERR 4102  // error retreiving table definition
#define OERROR_ORLONERR 4103  // unable to make connection
#define OERROR_TRANSNIPC 4104  // commit when no transaction 
#define OERROR_TRANSNIPR 4105 // rollback when no transaction
#define OERROR_NODSET 4106  // no such dynaset attached to connection
#define OERROR_INVROWNUM 4108  // invalid row reference
#define OERROR_TEMPFILE 4109  // error creating temporary file
#define OERROR_DUPSESSION 4110  // duplicate session name
#define OERROR_NOSESSION 4111   // no such session on detach
#define OERROR_NOOBJECTN 4112  // no object with specified name
#define OERROR_DUPCONN 4113  // duplicate connection
#define OERROR_NOCONN 4114  // no such connection on detach
#define OERROR_BFINDEX 4115  // invalid field index
#define OERROR_CURNREADY 4116  // cursor not ready
#define OERROR_NOUPDATES 4117  // updates not allowed
#define OERROR_NOTEDITING 4118  // not currently editing
#define OERROR_DATACHANGE 4119  // data has changed since last read
#define OERROR_NOBUFMEM 4120  // no memory for binding buffers
#define OERROR_INVBKMRK 4121  // invalid bookmark
#define OERROR_BNDVNOEN 4122  // bind variable not enabled
#define OERROR_DUPPARAM 4123  // duplicate parameter name
#define OERROR_INVARGVAL 4124  // invalid argument value
#define OERROR_INVFLDTYPE 4125  // invalid field type
#define OERROR_NOTIMPL 4126  // operation not implemented
#define OERROR_TRANSFORUP 4127  // For Update detected, no transaction
#define OERROR_NOTUPFORUP 4128  // For Update detected, not updatable

#define OERROR_ECURSOR 8193  // cannot create cursor
#define OERROR_FETCHERR 8194  // error fetching field
#define OERROR_BINDERR 8195  // output data binding error
#define OERROR_SQLERR 8196  // error in SQL statement
#define OERROR_ESQLEXEC 8197  // SQL execution error
#define OERROR_COMERR 8198  // error during commit
#define OERROR_ROLERR 8199  // error during rollback
#define OERROR_OPTERR 8200  // error setting options
#define OERROR_CONNERR 8201  // unable to make connection

// server data types
#define OTYPE_VARCHAR2 1
#define OTYPE_NUMBER 2
#define OTYPE_LONG 8
#define OTYPE_ROWID 11
#define OTYPE_DATE 12
#define OTYPE_RAW 23
#define OTYPE_LONGRAW 24
#define OTYPE_CHAR 96
#define OTYPE_MSLABEL 106

// edit modes for dynaset
#define ODYNASET_EDIT_NOEDIT 0
#define ODYNASET_EDIT_EDITING 1
#define ODYNASET_EDIT_NEWRECORD 2

// parameter io types
#define OPARAMETER_INVAR 1
#define OPARAMETER_OUTVAR 2
#define OPARAMETER_INOUTVAR 3

// parameter status flags
#define OPARAMETER_STATUS_IN 1
#define OPARAMETER_STATUS_OUT 2
#define OPARAMETER_STATUS_AUTOENABLED 4
#define OPARAMETER_STATUS_ENABLED 8

// Options for creating database
#define ODATABASE_DEFAULT 0
#define ODATABASE_PARTIAL_INSERT 1
#define ODATABASE_EDIT_NOWAIT 2

// Options for creating dynaset
#define ODYNASET_DEFAULT 0
#define ODYNASET_NOBIND 1
#define ODYNASET_KEEP_BLANKS 2
#define ODYNASET_READONLY 4
#define ODYNASET_NOCACHE 8

// definitions of actions used in the callback routines
#define OADVISE_MOVE_FIRST 1
#define OADVISE_MOVE_PREV 2
#define OADVISE_MOVE_NEXT 3
#define OADVISE_MOVE_LAST 4
#define OADVISE_MOVE_TOMARK 5
#define OADVISE_REFRESH 11
#define OADVISE_DELETE 21
#define OADVISE_ADDNEW 22
#define OADVISE_UPDATE 24
#define OADVISE_ROLLBACK 29
#define OADVISE_FOUNDLAST 151
#define OADVISE_OTHER 99


// forward references
class OEXPORT OSession;
class OEXPORT OSessionCollection;
class OEXPORT OClient;
class OEXPORT OConnection;
class OEXPORT OConnectionCollection;
class OEXPORT ODatabase;
class OEXPORT ODynaset;
class OEXPORT ODynasetMark;
class OEXPORT OField;
class OEXPORT OFieldCollection;
class OEXPORT OAdvise;
class OEXPORT OParameter;
class OEXPORT OParameterCollection;
class OOLEvar;

// ------------------------------------------------------------
// functions that are not class methods

// routine to initialize library.  Should be called once at startup 
//    returns TRUE if successful, FALSE if not
oboolean OEXPORT OStartup(void);

// routine to close down the library.  Should be called once
//   at application shutdown
void OEXPORT OShutdown(void);

// ----- OOracleObject -----------------------------------------------
// base object class
//   This class is the base for the OO4W classes.  By itself
//   it provides the error reporting interface and helps with
//   the various copying mechanisms

class OEXPORT OOracleObject
{
public:
    // constructors & destructors
    OOracleObject(void);
    OOracleObject(const OOracleObject &other);
    virtual ~OOracleObject(void);
    
    // overloaded operators
    OOracleObject &operator=(const OOracleObject &other);
    int operator==(const OOracleObject &other) const;
    int operator!=(const OOracleObject &other) const;
    
    // properties
    virtual oboolean IsOpen(void) const;
    
    // Error handling methods
    long  ErrorNumber(void) const; // return error "number"
    char *LookupErrorText(long errno) const;  // get error text for given error number
    char *GetErrorText(void) const;  // get description of last error

    // set error information
    void  ErrorReset(void) const;  // reset error state to "no error"
    // SetOtherError and SetInternalError are really for internal use
    void  SetOtherError(void *otheri) const;
    void  SetInternalError(long errno) const;
    
    void *Internal(void) const;
    
protected:
    // copy and cleanup routines (used to implement destructor, constructor, =)
    virtual oresult Copy(const OOracleObject &other);
    virtual oresult Cleanup(void);
    
    // access to object interface
    oresult SetObjectInterface(void *obji, oboolean geterror = TRUE);

    oresult ActionStart(void) const;  // start of most of the methods
    oresult ActionGetStart(const OOracleObject *nobj) const;

private:
    void   *m_obji;   // pointer to object interface
    void   *m_erri;   // pointer to error interface of object
    void   *m_errotheri;  // pointer to error interface on other object
    char   *m_lasterrstring;  // last error string we've handled
    
    int     m_errstate;   // where did the error come from (values in implementation)
    long    m_errno;  // error number (see below, or negative for internal values)
};

// ----- OOracleCollection -----------------------------------------------
// general set class
// This is a base class used for collections of sessions, connections and fields
// It has no utility on its own.  Routines to get items from the collection are
// in the subclasses

class OEXPORT OOracleCollection : public OOracleObject
{
public:

    // constructors & destructors
    OOracleCollection(void);
    OOracleCollection(const OOracleCollection &other);
    ~OOracleCollection(void);
    
    // Open & Close
    oresult Close(void);   // close the set
    
    // overloaded operators
    OOracleCollection &operator=(const OOracleCollection &other);
    
    // # of items in the set
    long   GetCount(void) const;

protected:
    oresult OpenSetHelper(void *idisp, void *otheri, unsigned char stype);  // finish the work of opening the set
    void  *GetItem(unsigned char stype, int index) const;

private:
    unsigned char m_settype;
    
    // internal helper routines
    oresult Cleanup(void);
    oresult Copy(const OOracleCollection &other);
};

// ----- OSession -----------------------------------------------

class OEXPORT OSession : public OOracleObject
{
public:
    // construction & destruction
    OSession(void);
    OSession(const OSession &other); // copy constructor
    OSession(const char *sname);     // construct & open (NULL sname means open default session)
    ~OSession(void);
    
    // open the session
    oresult Open(void);               // open the default session
    oresult Open(const char *sname);  // open a new session with specific name
    oresult Close(void);
    
    // Getting other objects
    OConnectionCollection GetConnections(void) const;
    OClient GetClient(void) const; 
    static OSession GetNamedSession(const char *sname);

    // overloaded operators
    OSession &operator=(const OSession &other);
    
    // error handling
    long ServerErrorNumber(void) const;
    const char *GetServerErrorText(void) const;
    oresult ServerErrorReset(void);
    
    // get properties
    const char       *GetName(void) const;   // returns session name 
    const char       *GetVersion(void) const;  // returns version of Oracle Objects
    
    // transaction operations
    oresult  BeginTransaction(void);  // start a transaction
    oresult  Commit(oboolean startnew = FALSE);   // commit (may start new transaction)
    oresult  Rollback(oboolean startnew = FALSE); // rolls back transaction (may start new transaction)
    oresult  ResetTransaction(void);  // unconditionally rollback (no advisories)
    
    // function used by other classes to construct OSession objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);

private:
    char      *m_name;
    char      *m_errtext;
    char      *m_version;
    
    // internal helper routines
    oresult Cleanup(void);
    oresult Copy(const OSession &other);
    
};

// ----- OSessionCollection -----------------------------------------------

class OEXPORT OSessionCollection : public OOracleCollection
{
public:
    OSession GetSession(int index) const;

    // function used by other classes to construct OSessionCollection objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);  // finish the work of opening the set
};

// ----- ODatabase -----------------------------------------------

class OEXPORT ODatabase : public OOracleObject
{
public:
    // construction & destruction
    ODatabase(void);
    ODatabase(const ODatabase &other);
    ~ODatabase(void); 
    
    // construct & open
    ODatabase(const OSession &dbsess, const char *dbname, const char *username, const char *pwd, long options = ODATABASE_DEFAULT);
    ODatabase(const char *dbname, const char *username, const char *pwd, long options = ODATABASE_DEFAULT);
    
    // for the Open calls, if pwd is NULL it is assumed that username contains username/password
    
    // open a database on a new session (implicitly creates a session)
    oresult Open(const char *dbname, const char *username, const char *pwd, long options = ODATABASE_DEFAULT);
    
    // open a database on an existing session
    oresult Open(const OSession &dbsess, const char *dbname, const char *username, const char *pwd, long options = ODATABASE_DEFAULT);
    oresult Close(void);
    
    // getting other objects
    OSession GetSession(void) const;
    OConnection GetConnection(void) const;
    OParameterCollection GetParameters(void) const;
    
    // overloaded operators
    ODatabase &operator=(const ODatabase &other);
    
    // execute an arbitrary SQL statement
    oresult ExecuteSQL(const char *sqlstmt) const;
    
    // properties
    const char *GetName(void) const;     // return database name (from the connection)
    const char *GetConnectString(void) const;  // return connect string (without password)
    long        GetOptions(void) const;
    
    // error handling
    long ServerErrorNumber(void) const;
    const char *GetServerErrorText(void) const;
    oresult ServerErrorReset(void);
    int ServerErrorSQLPos(void) const;
    
    // function used by other classes to construct ODatabase objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);

private:
    char    *m_dbname;
    char    *m_dbconnect;
    char    *m_errtext;
    
    // internal helper routines
    oresult Cleanup(void);
    oresult Copy(const ODatabase &other);
};

// ----- OValue -----------------------------------------------

class OEXPORT OValue
{
public:
    // construction & destruction
    OValue(void);
    OValue(int intval);       // allows OValue val = 3
    OValue(long longval);     // allows OValue val = 99L;
    OValue(double doubleval); // allows initialization with a double
    OValue(const char *tval); // allows initialization with a string
    OValue(const OValue &other); // copy constructor
    ~OValue(void);
    
    OValue &operator=(const OValue &other);
    int operator==(const OValue &other) const;
    int operator!=(const OValue &other) const;
    
    // setting the data & type
    oresult Clear(void);    // clear the values
    oresult SetValue(const OValue &val);
    oresult SetValue(const char *val); // sets string value (copies text)
    oresult SetValue(int val);     // sets to int value
    oresult SetValue(long val);
    oresult SetValue(double dval); // sets to double value 
    // oresult SetValue(const void *longval, long len);  // set to long value (not implemented)
    
    oboolean IsNull(void) const;  // returns TRUE if value is NULL (which includes uninitialized)
    
    // getting data (overloaded cast operators)
    operator int() const;
    operator long() const;
    operator const char *() const; // returns 0 if instance isn't string (no allocation)
    operator double() const;
    
    // conversions to implementation-specific representations (used by OField)
    oresult FromLocalType(void *localv);

    // helper routine for implementation of other classes
    void *Internal(void) const;    
private:
    OOLEvar  *m_value;  // pointer to data representation 
    
    // helper routines
    oresult Copy(const OValue &other);
    oresult Cleanup(void);
};

// ----- ODynaset -----------------------------------------------

class OEXPORT ODynaset : public OOracleObject
{
public:
    // construction & destruction
    ODynaset(void);
    ODynaset(const ODynaset &other);
    ODynaset(const ODatabase &odb, const char *sql_statement,
             long options = ODYNASET_DEFAULT);  // construct & open
    
    ~ODynaset(void);
    
    // Open takes an SQL query as an argument.  This constructs a set of records which can
    //    then be accessed.
    oresult Open(const ODatabase &odb, const char *sql_statement, long options = ODYNASET_DEFAULT);  // creates a dynaset object
    // The clone of a dynaset is looking at the original data, but has a different navigational
    //    position.  Useful if you want to navigate through a set of data without side-effects
    oresult Close(void);
    
    // getting other objects
    ODynaset Clone(void) const;  // clone a dynaset
    ODatabase GetDatabase(void) const;    
    OFieldCollection GetFields(void) const;
    OSession GetSession(void) const;
    OField GetField(int index) const;
    OField GetField(const char *fieldname) const;
    OConnection GetConnection(void) const;
    ODynasetMark GetMark(void) const;  // bookmark at current position
    ODynasetMark GetLastModifiedMark(void) const;  // get bookmark at last modified record 
    
    // overloaded operators
    ODynaset &operator=(const ODynaset &other);
    
    // the sql statement of the dynaset
    oresult SetSQL(const char *sql_statement);  // sets new sql statement for the dynaset
    const char *GetSQL(void) const;  // gets sql statement
    oresult Refresh(void);  // refresh dynaset with current sql statement

    // navigation methods
    oresult MoveFirst(void);   // go to first record in the set
    oresult MoveLast(void);    // go to last record in the set.  Note that this requires that all the records
                            //    in the query be downloaded from the server.  Can be expensive.
    
    // MovePrev and MoveNext take a "gopast" argument.  If it is TRUE we can navigate to an
    //   invalid record before the first or after the last.  If "gopast" is FALSE, then we won't
    //   go before the first or after the last record.  
    oresult MovePrev(oboolean gopast = TRUE);    // go to previous record
    oresult MoveNext(oboolean gopast = TRUE);    // go to next record - the most frequently used
    oresult MoveToMark(const ODynasetMark &odmark);  // repositions to mark (see ODynasetMark class)
    oboolean IsEOF(void) const;  // true if at end of dynaset.  Set to TRUE when you have tried to go
                                //     beyond the end
    oboolean IsBOF(void) const;  // true if at begin of dynaset
    oboolean IsValidRecord(void) const;  // true if current row is valid
    
    // editing
    oboolean CanTransact(void) const;  // returns TRUE if transaction processing is enabled 
    oboolean CanUpdate(void) const;  // returns TRUE if this dynaset is updateable
    int GetEditMode(void) const;  // returns current edit mode: ODYNASET_EDIT_*
    oresult StartEdit(void);  // starts edit operation
    oresult CancelEdit(void); // cancels edit operation
    oresult Update(void);     // finishes editing
    oresult DuplicateRecord(void);  // copy the current record, adds a new record, sets field to copied values
    
    // others
    oboolean CanRefresh(void) const;  // our dynasets can always requery
    oboolean CanScroll(void) const;    // always scrollable
    oboolean CanMark(void) const;
    
    long GetOptions(void) const;
    
    // record manipulation
    oresult AddNewRecord(void);
    oresult DeleteRecord(void);
    
    // record access
    int GetFieldCount(void) const;  // returns # of fields in a record
    long GetRecordCount(void) const;  // (dangerous!) gets total number of records in dynaset
                         // this routine downloads the entire dynaset to the client   
    int GetFieldIndex(const char *fieldname) const;  // gets the index of a field by name
    
    // all the Get and Set methods can refer to a column either by index (column position in the
    //    query) or by fieldname (the name of the column as specified in the query).  GetFieldIndex
    //    can be used to find the index of columns - which is more efficient
    short GetFieldServerType(int index) const;
    short GetFieldServerType(const char *fieldname) const;
    short GetFieldPrecision(int index) const;
    short GetFieldPrecision(const char *fieldname) const;
    short GetFieldScale(int index) const;
    short GetFieldScale(const char *fieldname) const;
    oboolean IsFieldNullOK(int index) const;
    oboolean IsFieldNullOK(const char *fieldname) const;
    
    long GetFieldSize(int index) const;
    long GetFieldSize(const char *fieldname) const;
    long GetFieldServerSize(int index) const;
    long GetFieldServerSize(const char *fieldname) const;
    oboolean IsFieldTruncated(int index) const;
    oboolean IsFieldTruncated(const char *fieldname) const;
    
    // get data from a field, either by name or index
    oresult GetFieldValue(int index, OValue *val) const;
    oresult GetFieldValue(const char *fieldname, OValue *val) const;
    oresult GetFieldValue(int index, int *val) const;
    oresult GetFieldValue(const char *fieldname, int *val) const;
    oresult GetFieldValue(int index, long *val) const;
    oresult GetFieldValue(const char *fieldname, long *val) const;
    oresult GetFieldValue(int index, double *val) const;
    oresult GetFieldValue(const char *fieldname, double *val) const;
    oresult GetFieldValue(int index, char *val, unsigned short maxlen) const;
    oresult GetFieldValue(const char *fieldname, char *val, unsigned short maxlen) const;
    oresult GetFieldValue(int index, void __huge *longval, long len, long *readlen) const;
    oresult GetFieldValue(const char *fieldname, void __huge *longval, long len, long *readlen) const;
    oresult GetFieldChunk(int index, void *chunkp, long offset, unsigned short len) const;
    oresult GetFieldChunk(const char *fieldname, void *chunkp, long offset, unsigned short len) const;
    
    // set data in a field, either by name or index
    oresult SetFieldValue(int index, const OValue &val);
    oresult SetFieldValue(const char *fieldname, const OValue &val);
    oresult SetFieldValue(int index, int val);
    oresult SetFieldValue(const char *fieldname, int val);
    oresult SetFieldValue(int index, long val);
    oresult SetFieldValue(const char *fieldname, long val);
    oresult SetFieldValue(int index, double val);
    oresult SetFieldValue(const char *fieldname, double val);
    oresult SetFieldValue(int index, const char *val);
    oresult SetFieldValue(const char *fieldname, const char *val);
    oresult SetFieldValue(int index, const void __huge *longval, long len);
    oresult SetFieldValue(const char *fieldname, const void __huge *longval, long len);
    oresult AppendFieldChunk(int index, const void *chunkp, unsigned short len);
    oresult AppendFieldChunk(const char *fieldname, const void *chunkp, unsigned short len);

    // function used by other classes to construct ODynaset objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri); 
    
private:
    char *m_sqlstmt;  // the sql statement that creates the dynaset
    
    // internal helper routines
    oresult Copy(const ODynaset &other);
    oresult Cleanup(void);
};

// ----- ODynasetMark -----------------------------------------------
  
class OEXPORT ODynasetMark : public OOracleObject
{
public:
    // construction & destruction
    ODynasetMark(void); 
    ODynasetMark(const ODynasetMark &other);
    ~ODynasetMark(void);
    
    // overloaded operators
    ODynasetMark &operator=(const ODynasetMark &other);
    int operator==(const ODynasetMark &other) const;
    int operator!=(const ODynasetMark &other) const;
                               
    
    oresult Close(void);
    oboolean IsOpen(void) const;
    
    // function used by other classes to construct ODynasetMark objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);
    
    char *Internal(void) const;

private:
    char  *m_markdata;  // holds data for bookmark
    
    oresult Cleanup(void);
};

// ----- OField -----------------------------------------------

class OEXPORT OField : public OOracleObject
{
public:
    // construction & destruction
    OField(void);
    OField(const OField &other);
    ~OField(void);              
    
    oresult Close(void);
    
    // overloaded operators
    OField &operator=(const OField &other);
    
    // other objects
    ODynaset GetDynaset(void) const;
    
    // properties
    const char *GetName(void) const; // returns name of field (user doesn't need to free)
    long GetSize(void) const;  // returns length of field
    long GetServerSize(void) const;  // returns length of a field (in server)
    short GetServerType(void) const;
    short GetPrecision(void) const;
    short GetScale(void) const;
    oboolean IsNullOK(void) const;
    oboolean IsTruncated(void) const;
    
    // field value
    oresult GetValue(OValue *val) const;  // returns value of field in val
    oresult GetValue(int *val) const;
    oresult GetValue(long *val) const;
    oresult GetValue(double *val) const;
    oresult GetValue(const char **val) const;
    oresult GetValue(void __huge *longval, long len, long *readlen) const;
    oresult SetValue(const OValue &val);  // sets the field to new value
    oresult SetValue(int val);
    oresult SetValue(long val);
    oresult SetValue(double val);
    oresult SetValue(const char *val);
    oresult SetValue(const void __huge *longval, long len); 
    
    // for long fields
    oresult AppendChunk(const void *chunkp, unsigned short numbytes);
    oresult GetChunk(const char **chunkp, long offset, unsigned short numbytes) const;                                            
    
    // getting data (overloaded cast operators)
    operator int() const;
    operator long() const;
    operator double() const;
    operator const char *() const;

    // function used by other classes to construct OField objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);

private:
    short  m_index;
    char  *m_name;
    char  *m_data;

    // internal helper routines
    oresult Copy(const OField &other);
    oresult Cleanup(void);   
};

// ----- OFieldCollection -----------------------------------------------

class OEXPORT OFieldCollection : public OOracleCollection
{
public:
    OField GetField(int index) const;

    // function used by other classes to construct OFieldCollection objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);  // finish the work of opening the set
};

// ----- OClient -----------------------------------------------

class OEXPORT OClient : public OOracleObject
{
public:
    // construction & destruction
    OClient(void);
    OClient(const OClient &other);
    ~OClient(void);
    
    // use OSession::GetClient to open a client object
    oresult Close(void);

    // Getting other objects
    OSessionCollection GetSessions(void) const;  // get set of sessions

    // overloaded operators
    OClient &operator=(const OClient &other);
    
    // get properties
    const char        *GetName(void) const;        // returns client name.
    
    // function used by other classes to construct OSession objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);

private:
    char *m_name;

    // internal helper routines
    oresult Copy(const OClient &other);
    oresult Cleanup(void);   
};

// ----- OConnection -----------------------------------------------

class OEXPORT OConnection : public OOracleObject
{
public:
    // construction & destruction
    OConnection(void);
    OConnection(const OConnection &other);
    ~OConnection(void);
    
    oresult Close(void);
    
    // getting other objects
    OSession GetSession(void) const;   

    // overloaded operators
    OConnection &operator=(const OConnection &other);
    
    // properties
    const char  *GetConnectString(void) const;  // returns connect string
    const char  *GetDatabaseName(void) const;  // returns database name for this connection
    
    // function used by other classes to construct OConnection objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);

private:
    char *m_dbname;
    char *m_dbconnect;

    // internal helper routines
    oresult Copy(const OConnection &other);
    oresult Cleanup(void);   
};
 
// ----- OConnectionCollection -----------------------------------------------

class OEXPORT OConnectionCollection : public OOracleCollection
{
public:
    OConnection GetConnection(int index) const;

    // function used by other classes to construct OConnectionCollection objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);  // finish the work of opening the set
};

// ----- OAdvise -----------------------------------------------
 
class OEXPORT OAdvise : public OOracleObject
{
public:
    OAdvise(void);
    OAdvise(const OAdvise &other);
    OAdvise(const ODynaset &odyn);  // construct & open
    virtual ~OAdvise(void);
    
    virtual oresult Open(const ODynaset &odyn);
    virtual oresult Close(void);  // stop the advisory sink
    
    // getting other objects
    ODynaset GetDynaset(void) const;
    
    // overloaded operators
    OAdvise &operator=(const OAdvise &other);
    
    // called before an action takes place
    virtual oboolean ActionRequest(int movekind);  // returns TRUE if action is allowed, FALSE to cancel
    
    // routines called to to notify that action occured
    virtual void ActionNotify(int movekind); 
    
    // routine called on status change
    virtual void StatusChange(int statuskind);
    
private:
    ODynaset  m_oDyn;    // the dynaset we're getting notices from
    
    // private helper routines
    oresult Cleanup(void);
    oresult Copy(const OAdvise &other);
};

// ----- OParameter -----------------------------------------------

class OEXPORT OParameter : public OOracleObject
{
public:
    OParameter(void);
    OParameter(const OParameter &other);
    ~OParameter(void);
    
    oresult Close(void);
    
    // overloaded operators
    OParameter &operator=(const OParameter &other);
    
    // parameter name
    const char *GetName(void) const;
    
    short GetServerType(void) const;
    
    // parameter value
    oresult GetValue(OValue *val) const;
    oresult GetValue(int *val) const;
    oresult GetValue(long *val) const;
    oresult GetValue(double *val) const;
    oresult GetValue(const char **val) const;
    
    oresult Clear(void);
    oresult SetValue(const OValue &val);
    oresult SetValue(int val);
    oresult SetValue(long val);
    oresult SetValue(double val);
    oresult SetValue(const char *val);
    
    // getting data (overloaded cast operators)
    operator int() const;
    operator long() const;
    operator double() const;
    operator const char *() const;

    // other status information
    int GetStatus(void) const;
    
    // enabling
    oresult AutoEnable(oboolean enable);

    // function used by other classes to construct OParameter objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);  // finish the work of opening the object

private:
    char *m_name; 
    char *m_data;

    // internal helper routines
    oresult Copy(const OParameter &other);
    oresult Cleanup(void);
    oresult HelpGetValue(OOLEvar *tempv) const;
};

// ----- OParameterCollection -----------------------------------------------

class OEXPORT OParameterCollection : public OOracleObject
{
public:
    OParameterCollection(void);
    OParameterCollection(const OParameterCollection &other);
    ~OParameterCollection(void);
    
    oresult Close(void);
    
    // overloaded operators
    OParameterCollection &operator=(const OParameterCollection &other);
    
    // getting other objects
    OParameter GetParameter(int index) const;
    OParameter GetParameter(const char *pname) const;
    
    // adding a new parameter
    OParameter Add(const char *name, int value, int iotype, int serverType);
    OParameter Add(const char *name, long value, int iotype, int serverType);
    OParameter Add(const char *name, double value, int iotype, int serverType);
    OParameter Add(const char *name, const char *value, int iotype, int serverType);     
    
    // removing a parameter
    oresult Remove(int index);
    oresult Remove(const char *name);
    
    // getting # of parameters
    long GetCount(void) const;
    
    // function used by other classes to construct OParameterCollection objects (don't call this!)
    oresult OpenHelper(void *idisp, void *otheri);  // finish the work of opening the set

private:
    // internal helper routines
    oresult Copy(const OParameterCollection &other);
    oresult Cleanup(void);   
    OParameter AddHelp(const char *name, const OValue &val, int iotype, int serverType);

};

#endif // ORACL_ORACLE
