/****************************************************/
/*  This file uses the INTERSOLV SQLServer driver.  */
/****************************************************/
/*
    File:       GETCATS.CPP
    
    Revision:   1.0 Release
    
    Date:       17-May-1994

    Author:     Dale Hunscher
    
    Description:

    This program performs each catalog function in turn
    and performs a Prepare() and AutoBind(), then outputs a 
    structure definition that describes the result set as it 
    exists in the cursor instance.

    This is a small piece of code that will be incorporated into
    an incremental code generator we hope to release soon.

    To use this program, run it; enter a filename for the output,
    and select a data source.
       
    /////////////////////////////////////////////////////////////
    ///////////////////// NOTICE ////////////////////////////////
    /////////////////////////////////////////////////////////////
                                                                     
    Copyright (c) 1994 by INTERSOLV, Inc. All rights reserved.

    Information in this document is subject to change without
    notice and does not represent a commitment on the part of
    INTERSOLV, Inc. This software is provided under
    a license agreement or non-disclosure agreement. The software
    may be used and/or copied only in accordance with the terms
    of the governing agreement. It is against the law to copy
    the software on any medium except as specifically allowed
    in the governing agreement. No part of this software may be 
    reproduced or transmitted in any form or by any means, 
    electronic or mechanical, including photocopying, recording,
    or information storage and retrieval systems, for any purpose
    other than the licensee's personal use, without the express
    written permission of INTERSOLV, Inc.
    
    /////////////////////////////////////////////////////////////
*/

#define V2 // turn on version 2 functionality

#include <sql.hpp>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <memory.h>

// get name of C language type.
LPSTR GetCTypeName( SWORD fSqlType )
  {
  switch ( fSqlType )
    {
      case  SQL_C_BINARY      :
      case  SQL_C_CHAR        :
        return "char" ;

      case  SQL_C_LONG        : 
        return "long" ;

      case  SQL_C_SHORT       : 
        return "short" ;

      case  SQL_C_FLOAT       : 
        return "float" ;

      case  SQL_C_DOUBLE      :
        return "double" ;

      case  SQL_C_DATE        :
        return "DATE_STRUCT" ;

      case  SQL_C_TIME        :
        return "TIME_STRUCT" ;

      case  SQL_C_TIMESTAMP   :
        return "TIMESTAMP_STRUCT" ;

      case  SQL_C_TINYINT     :
        return "unsigned char" ;

      case  SQL_C_BIT         :
        return "unsigned char" ;

      default :
        break ;
    } // end switch
  return "?" ;
  }

// get constant for C language type.
LPSTR GetCTypeConstant( SWORD fSqlType )
  {
  switch ( fSqlType )
    {
      case  SQL_C_BINARY      :
        return "SQL_C_BINARY";
        
      case  SQL_C_CHAR        :
        return "SQL_C_CHAR" ;

      case  SQL_C_LONG        : 
        return "SQL_C_LONG" ;

      case  SQL_C_SHORT       : 
        return "SQL_C_SHORT" ;

      case  SQL_C_FLOAT       : 
        return "SQL_C_FLOAT" ;

      case  SQL_C_DOUBLE      :
        return "SQL_C_DOUBLE" ;

      case  SQL_C_DATE        :
        return "SQL_C_DATE" ;

      case  SQL_C_TIME        :
        return "SQL_C_TIME" ;

      case  SQL_C_TIMESTAMP   :
        return "SQL_C_TIMESTAMP" ;

      case  SQL_C_TINYINT     :
        return "SQL_C_TINYINT" ;

      case  SQL_C_BIT         :
        return "SQL_C_BIT" ;

      default :
        break ;
    } // end switch
  return "?" ;
  }

// get C language Hungarian prefix
LPSTR GetCTypeHungarianPrefix( SWORD fSqlType )
  {
  switch ( fSqlType )
    {
      case  SQL_C_BINARY      :
        return "rgch" ;

      case  SQL_C_CHAR        :
        return "sz" ;

      case  SQL_C_LONG        : 
        return "l" ;

      case  SQL_C_SHORT       : 
        return "n" ;

      case  SQL_C_FLOAT       : 
        return "f" ;

      case  SQL_C_DOUBLE      :
        return "d" ;

      case  SQL_C_DATE        :
        return "dt" ;

      case  SQL_C_TIME        :
        return "tm" ;

      case  SQL_C_TIMESTAMP   :
        return "ts" ;

      case  SQL_C_TINYINT     :
        return "u" ;

      case  SQL_C_BIT         :
        return "b" ;

      default :
        break ;
    } // end switch
  return "?" ;
  }

// Return TRUE if this type maps to an array of chars
BOOL CTypeIsArray( SWORD fSqlType )
  {
  switch ( fSqlType )
    {
      case  SQL_C_BINARY      :
      case  SQL_C_CHAR        :
        return TRUE ;

      default :
        break ;
    } // end switch
  return FALSE ;
  }

// Make a const int name from column name. Roughly,
// creates from column 'mycolname' a constant
// name MYCOLNAME_SIZE. Non-alphanumeric characters
// are replaced by underscores.

// NOTE: if the returned string is not to be used
// immediately, make a copy of the result; returns address
// of static buffer whose content is subject to change.

LPSTR ConstNameFromColName( LPUCSTR lpszCol )
    {
      // allow six extra bytes for appended '_SIZE'
      static char szTemp[COLUMN_NAME_MAX+6] ;

      lstrcpyn( szTemp, (LPCSTR)lpszCol, sizeof(szTemp)-6);
      szTemp[sizeof(szTemp)-7] = 0;
      strupr(szTemp);
      for( register int i = 0; i < lstrlen(szTemp); i++)
        if (!isalnum(szTemp[i]))
          szTemp[i] = '_';
      lstrcat( szTemp, "_SIZE" );
      return szTemp;
    }

LPCSTR GetSqlTypeName( SWORD fSqlType )
  {
  switch ( fSqlType )
    {
      case  SQL_CHAR          : //  1
        return "SQL_CHAR" ;

      case  SQL_NUMERIC       : //  2
        return "SQL_NUMERIC" ;

      case  SQL_DECIMAL       : //  3
        return "SQL_DECIMAL" ;

      case  SQL_INTEGER       : //  4
        return "SQL_INTEGER" ;

      case  SQL_SMALLINT      : //  5
        return "SQL_SMALLINT" ;

      case  SQL_FLOAT         : //  6
        return "SQL_FLOAT" ;

      case  SQL_REAL          : //  7
        return "SQL_REAL" ;

      case  SQL_DOUBLE        : //  8
        return "SQL_DOUBLE" ;

      case  SQL_DATE          : //  9
        return "SQL_DATE" ;

      case  SQL_TIME          : // 10
        return "SQL_TIME" ;

      case  SQL_TIMESTAMP     : // 11
        return "SQL_TIMESTAMP" ;

      case  SQL_VARCHAR       : // 12
        return "SQL_VARCHAR" ;

      case  SQL_LONGVARCHAR   : // (-1)
        return "SQL_LONGVARCHAR" ;

      case  SQL_BINARY        : // (-2)
        return "SQL_BINARY" ;

      case  SQL_VARBINARY     : // (-3)
        return "SQL_VARBINARY" ;

      case  SQL_LONGVARBINARY : // (-4)
        return "SQL_LONGVARBINARY" ;

      case  SQL_BIGINT        : // (-5)
        return "SQL_BIGINT" ;

      case  SQL_TINYINT       : // (-6)
        return "SQL_TINYINT" ;

      case  SQL_BIT           : // (-7)
        return "SQL_BIT" ;

      default :
        break ;
    } // end switch
  return "SQL_?" ;
  }

void DoDefinition( odbcCURSOR &cursor, FILE *fd, LPSTR lwrname, LPSTR name_in)
{
    int i, j;
    static char name[128];
    // for character manipulation
    UCHAR c;
    
    sprintf( name, "s%sRESULTSET", name_in);
    
    
    for ( i = 0; i < cursor.NumResultCols(); i++)
    {
      psCOLBIND pCol = cursor.ColResultInfo(i+1);

      // consider the possibility of the impossible
      if (pCol == NULL)
        {
        fprintf(stderr, "\nError: NULL column info.\n");
        break;
        }

      // if an array, write a constant definition for its size
      if (CTypeIsArray(pCol->fCType))
                fprintf( fd, "const int %-32.32s = %ld ;\n",
                         ConstNameFromColName(pCol->szColName),
                         pCol->cbValueMax);
            }
            // print opening of struct
            fprintf(fd,"\n struct %s {\n", name);
            for ( i = 0; i < cursor.NumResultCols(); i++)
              {
              // get address of column info structure
              psCOLBIND pCol = cursor.ColResultInfo(i+1);

              // consider the possibility of the impossible
              if (pCol == NULL)
                {
                fprintf(stderr, "\nError: NULL column info.\n");
                break;
                }

              // output type
              fprintf( fd, "  %-20.20s ", GetCTypeName(pCol->fCType));

              // output prefix
              fprintf( fd, "%s", GetCTypeHungarianPrefix(pCol->fCType));

              // lower-case name with upper-case first letter
              // replace non-alphanumeric characters with underscores
              // first do the first character...
              c = pCol->szColName[0];
              fprintf( fd, "%c",
                (isalnum(c) ? toupper(c) : '_'));

                // now do the remaining characters in the name
              for (j = 1, c = pCol->szColName[j]; j < lstrlen((LPCSTR)(pCol->szColName)); c = pCol->szColName[++j])
                fprintf( fd, "%c", (isalnum(c) ? tolower(c) : '_'));

              // if array of chars, output brackets and size constant
              if (CTypeIsArray(pCol->fCType))
                fprintf( fd, "[ %s ]", ConstNameFromColName(pCol->szColName));
              // output semi-colon and eoln
              fprintf( fd, ";\n");
              }

            // close up the structure definition
            fprintf( fd, "};\n\n"); 
            
            // do sCOLBIND array definition as well
            
            fprintf( fd, "// sCOLBIND array definition\n\n");
            
            fprintf( fd, "static sCOLBIND _SQL%sRSB[] = \n{\n", lwrname);
             
            // loop through column definitions
            
            for ( i = 0; i < cursor.NumResultCols(); i++)
              {
              // get address of column info structure
              psCOLBIND pCol = cursor.ColResultInfo(i+1);

              // consider the possibility of the impossible
              if (pCol == NULL)
                {
                fprintf(stderr, "\nError: NULL column info.\n");
                break;
                }
                                 
              // begin entry
              fprintf( fd, "{ ");
              // col number in query
              fprintf( fd, "\t%d,\n", pCol->iCol);           
              // C data type constant                             
              fprintf( fd, "\t%s,\n", GetCTypeConstant(pCol->fCType));         
              // offset in result struct: use FIELDOFFSET                             
              fprintf( fd, "\tFIELDOFFSET( %s, ", name);
              // output prefix
              fprintf( fd, "%s", GetCTypeHungarianPrefix(pCol->fCType));

              // lower-case name with upper-case first letter
              // replace non-alphanumeric characters with underscores
              // first do the first character...
              c = pCol->szColName[0];
              fprintf( fd, "%c",
                (isalnum(c) ? toupper(c) : '_'));

                // now do the remaining characters in the name
              for (j = 1, c = pCol->szColName[j]; j < lstrlen((LPCSTR)(pCol->szColName)); c = pCol->szColName[++j])
                fprintf( fd, "%c", (isalnum(c) ? tolower(c) : '_'));
              // finish FIELDOFFSET entry
              fprintf( fd, " ),\n");        
              fprintf( fd, "\t%ld,\n", pCol->cbValueMax);    // size of buffer
              fprintf( fd, "\t0,\n");        // size of actual result (output)
              fprintf( fd, "\tFALSE,\n", pCol->fPtr);
              // column name used in auto-binding
              fprintf( fd, "\t\"%s\",\n", pCol->szColName);
#if defined( V2 )                                           
    // new in v2.0                                                                                                      
              // actual SQL type
              fprintf( fd, "\t%s,\n", GetSqlTypeName(pCol->fSqlType));       
              fprintf( fd, "\t%d,\n", pCol->ibScale);        // scale (only used for some types)
              fprintf( fd, "\t%d,\n", pCol->fNullable);      // non-zero if null values are possible
              fprintf( fd, "\tNULL,\n");   // if non-NULL, column constraints
                                           //   to be used in CreateTable()
#endif              
              // output end of column definition
              fprintf( fd, "\n},\n");
              }

            // close up the structure definition
            fprintf( fd, "};\nconst int _SQL%sRSBCount\n"
                            "\t= sizeof(_SQL%sRSB)"
                            " / sizeof(_SQL%sRSB[ 0 ]);\n\n",
                            lwrname,
                            lwrname,
                            lwrname
                            ); 
            
}

void ProcessDefinition(LPSTR name, rodbcCURSOR cursor, FILE *fd)
{
    fprintf( stderr, name );
    fprintf( stderr, "...\n" ) ;
    if (!cursor.sqlsuccess())
        {
        fprintf( stderr, "\nError.\nSQL State: %s\n  Message:%s\n",
        cursor.SqlState(),
        cursor.ErrMsg()
        );
        cursor.Close();
        }
    else if (!cursor.NumResultCols())
        {
        fprintf( stderr, "\nNo result set.\n");
        cursor.Close();
        }
    else
        {
        char *upper = new char[strlen( name ) + 1]; 
        strcpy( upper, name );
        strupr( upper );
        cursor.AutoBind();
        fprintf(fd,"// " );
        fprintf(fd, name);
        fprintf(fd, "\n" );
        DoDefinition( cursor, fd, name, upper );
        delete[] upper ;
        cursor.Close(); 
        }
    fprintf( stderr, "\n" ) ;
    }

#if defined( WIN32 )
// this error routine will be called automatically when 
// an error occurs (see odbcbase.hpp for details).
void CALLBACK PrintErr(
    RETCODE         lastRet,
    UCHAR FAR *     szSqlState,
    SDWORD          fNativeError,
    UCHAR FAR *     szErrorMsg,
    odbcBASE FAR *  pObj
    )
    {
    char buf[ 80 ];

    MessageBeep( MB_ICONEXCLAMATION );
    fprintf( stderr, "Ret: %ld\nMsg: %s\nSQL: %s\n Nat: %ld\n\n"
    			"Press Enter to continue...\n",
                lastRet,
                (LPSTR)szErrorMsg,
                (LPSTR)szSqlState,
                fNativeError);
                
    gets(buf);
    }

#endif


void main(int , char **)
    {
    // instantiate an environment.  This allocates
    // an ODBC environment handle for the app.
    odbcENV env;
	//buffer for input
    char buf[ 80 ], UID[80], PWD[80];
    // for fopen() 
    char buffer[255] ; 
    // for output file
    FILE *fd = stderr ;
    
    fprintf( stderr, "Enter filename for output (Enter for stderr):\n" );

    if (gets(buffer) == NULL )
       {
       fprintf( stderr, "filename get failed, or you pressed ctrl-Z.\n");
       goto afterwards;
       }

    if (lstrlen(buffer) > 0)
        if ((fd = fopen(buffer, "w")) == NULL)
           {
           fprintf(stderr, "open failed.\n");
           goto afterwards;
           }

    env.AutoRetrieve(odbcREPSUCCESSWITHINFO);
    env.AutoReport(odbcREPSUCCESSWITHINFO);
#if !defined( WIN32 )
    env.SetWnd( GetActiveWindow()) ;
#else
	env.SetErrHandler( PrintErr ) ;
#endif
    if (env.sqlsuccess())
        {
        // using environment constructed in WinMain(),
        // instantiate a connection. This does not perform the
        // connection.
        odbcCONNECT connect( &env ) ;

        // use the default error handling mechanism to report
        // errors; ignore success-with-info returns.

        connect.AutoRetrieve(odbcREPERRS);
        connect.AutoReport(odbcREPERRS);
#if !defined( WIN32 )
        connect.SetWnd( GetActiveWindow() );
#endif
         fprintf(stderr, "\nConnecting to target...\n");
            fprintf( stderr, "\nEnter your SQLServer User ID: " );

        	if (gets(UID) == NULL )
           		{
           		fprintf( stderr, "get failed, or you pressed ctrl-Z.\n");
           		return;
           		}

        	fprintf( stderr, "\nEnter your SQLServer password: " );

        	if (gets(PWD) == NULL )
           		{
           		fprintf( stderr, "get failed, or you pressed ctrl-Z.\n");
           		return;
           		}
            connect.Connect(
                   "SQLServer",
                   UID,
                   PWD
                   );
        
        // if we fail, go home.
        if ( !connect.sqlsuccess() )
          {
          fprintf( stderr, "\nConnection failed.\nSQLState: %s\n Message: %s\n",
                   connect.SqlState(), connect.ErrMsg()) ;
          goto afterwards ;
          }

        odbcCURSOR cursor(&connect);
        cursor.AutoRetrieve( odbcREPERRS );

        if ( !cursor.sqlsuccess() )
          {
          fprintf( stderr, "\nCursor construction failed.\nSQLState: %s\n Message: %s\n",
                   cursor.SqlState(), cursor.ErrMsg()) ;
          goto afterwards ;
          }
         
         cursor.ColumnPrivileges(
                "",
                "",
                "",
                ""
                );
                
         ProcessDefinition("ColumnPrivileges", cursor, fd);
                         
         cursor.Procedures(
                "",
                "",
                ""
                );
                
         ProcessDefinition("Procedures", cursor, fd);
         
         cursor.ProcedureColumns(
                "",
                "",
                "",
                ""
                );
                
         ProcessDefinition("ProcedureColumns", cursor, fd);
         
         cursor.ForeignKeys(
                "",
                "",
                "",
                "",
                "",
                ""
                );
                
         ProcessDefinition("ForeignKeys", cursor, fd);
                
         cursor.TablePrivileges(
                "",
                "",
                ""
                );
                
         ProcessDefinition("TablePrivileges", cursor, fd);
         
         cursor.PrimaryKeys(
                "",
                "",
                ""
                );
                
         ProcessDefinition("PrimaryKeys", cursor, fd);
         
      } // env.sqlsuccess()

afterwards:
    if (fd != NULL && fd != stderr)
        fclose(fd);

    fprintf( stderr, "\nExecution complete!\n" );
	gets( buf );

  }
    
