/*
    File:       CURSOR.HPP

    Revision:   2.0 Release

    Date:       12-26-93

    Author:     Dale Hunscher

    Description:

    This file describes the interface to the cursor class. It
    provides a functional interface to SQL statements that return
    result sets (SELECT statements).

    Engineers:
    DAH     Dale A. Hunscher

    Revision History
    ================
    Date        Who     Did What
    -------------------------------------------------------------
    9/16/93     DAH     Added fully automatic binding functions.

    1-26-94     DAH     Fixed bug reported by Martin Grasruck
                        in SetStmt().

    2-22-94     DAH     Added new overloads of ColResultAddr()
                        to support block cursor result access;
                        added ColResultAs<type> functions.

    2-23-94     DAH     Corrected bug in AutoBind(); was not
                        allowing for multiple rows in returned
                        values.  Added data member pcbValues.

    6-1-94      DAH     GetFirst() and GetNext() return types
                        were different typedefs that their
                        prototypes in the include file.

    25-Jun-94   DAH     Was not resetting pSSize to zero after freeing
                        pSet in FreeColBindings().

    25-Jun-94   DAH     Added GetLast() and GetPrev().
    
    02-Dec-94   DAH     Changed declarations of routines that accepted
                        SQL type when C type was the appropriate type:
                        FmtColumnData, FmtBrowserColumnData, ColValueAsInt,
                        SetColValueFromInt, DetermineBrowserColumnLen,
                        DetermineColumnLen.

    /////////////////////////////////////////////////////////////
    ///////////////////// NOTICE ////////////////////////////////
    /////////////////////////////////////////////////////////////

    Copyright (c) 1993-1995 by INTERSOLV, Inc. All rights reserved.

    Information in this document is subject to change without
    notice and does not represent a commitment on the part of
    INTERSOLV, Inc. This software is provided under
    a license agreement or non-disclosure agreement. The software
    may be used and/or copied only in accordance with the terms
    of the governing agreement. It is against the law to copy
    the software on any medium except as specifically allowed
    in the governing agreement. No part of this software may be
    reproduced or transmitted in any form or by any means,
    electronic or mechanical, including photocopying, recording,
    or information storage and retrieval systems, for any purpose
    other than the licensee's personal use, without the express
    written permission of INTERSOLV, Inc.

    /////////////////////////////////////////////////////////////

*/

#if !defined(_CURSOR_HPP)
#define _CURSOR_HPP

///////////////////////////////////////////////////////////
//////////////////////// cursors
///////////////////////////////////////////////////////////

// iterator callback function type used in EnumResults member
// function.

typedef RETCODE (    *pfENUMRESULTS)(podbcCURSOR pCursor, PTR pUser);

class odbcEXPORTED odbcCURSOR : public odbcSTMT {
//#include <odbclslb\cursorwn.hpp>
private:

// no private members

protected:
/**********************************************************
    pColBindings

    Array of sCOLBIND structures representing the column
    bindings.

**********************************************************/

    psCOLBIND       pColBindings;

/**********************************************************
    pcbValues

    Array of SDWORDs representing the returned size of column
    data in extende fetch operations.

**********************************************************/

    SDWORD      *pcbValues;
    
/**********************************************************
    ColCount
    
    Count of array elements in pColBindings.

**********************************************************/

    UWORD           ColCount;
    
/**********************************************************
    pSet
    
 Address of structure to whose members are bound to columns using the 
information in the pColBindings array.
    
**********************************************************/

    PTR             pSet;
    
/**********************************************************
    bFirst
    
    Flag indicating we are ready to perform a GetFirst
    fetch after performing a QueryAndBind call.
    
**********************************************************/

    BOOL            bFirst;
    
/**********************************************************
    bColsBound
    
    Non-zero (TRUE) if columns have been bound.
    
**********************************************************/

    BOOL            bColsBound;
    
/**********************************************************
    bAutoBound
    
    Non-zero (TRUE) if columns have been bound.
    
**********************************************************/

    BOOL            bAutoBound;
    
/**********************************************************
    AutoBindCols
    
    Non-zero (TRUE) if columns should be auto-bound.
    
**********************************************************/

    BOOL            AutoBindCols;
    
/***************************************************

    pSSize
    
    Size of automatically allocated structure 
    for storing bound column output.
***************************************************/

    UWORD pSSize;   
    
/**********************************************************
    iCol
    
    Last column processed in column bindings; useful in
    pinpointing errors when errors occur while binding
    multiple parameters.
    
**********************************************************/

    UWORD           iCol;

/**********************************************************
    bNoExtraByteOnStrings
    
    If TRUE, no extra byte should be allocated when dynamically
    binding columns in AutoBind().
    
**********************************************************/

    BOOL            bNoExtraByteOnStrings;

/***************************************************

    ExtFetchRowCount
    
    Row count for use in supporting automated extended 
    fetch functions.
***************************************************/

    UWORD   ExtFetchRowCount;
    
/***************************************************

    ExtFetchRow
    
    Row variable for supporting automated extended 
    fetch functions.
***************************************************/

    UDWORD  ExtFetchRow;
    
/***************************************************

    ExtFetchRelAbsRow
    
    Relative/absolute row sent in automated extended 
    fetch functions.
***************************************************/

    SDWORD  ExtFetchRelAbsRow;
    
/***************************************************

    ExtFetchType
    
    Extended fetch op type for supporting automated extended 
    fetch functions.
***************************************************/

    UWORD   ExtFetchType;
    
/***************************************************

    ExtFetchStatus
    
    Row status array pointer for supporting automated 
    extended fetch functions.
***************************************************/

    UWORD       *ExtFetchStatus;

public:

/**********************************************************
    bTrimAllTrailingBlanks

    Boolean value; 0 if trailing blanks should not be auto-
    matically trimmed from SQL_CHAR columns in result set,
    1 if the trailing blanks should be trimmed. Inherited
    from the connection.
**********************************************************/

    BOOL           bTrimAllTrailingBlanks;

/**********************************************************
    odbcCURSOR
    
    Constructor. See constructor for odbcSTMT for description
    of arguments passed to odbcSTMT().  Cursor-specific 
    arguments:
    
        BOOL bAutoBind
        
            After execution of statement (during constructor
            or otherwise), call AutoBind to bind columns to
            members of automatically allocated structure.
    
        cpsCOLBIND  pColBindings
        
            If non-NULL, address of array of structures
            defining the binding of columns in the
            SQL statement to members of the structure
            to contain data values after successful
            call to fetch(). Default value is NULL.
            
        UWORD           ColCount
        
            Count of array elements in psColBindings.
            Default value is 0.
            
        PTR             pvColStruct
        
            Address of structure containing slots for
            values. Default value is NULL.
    
    
**********************************************************/


    odbcCURSOR
            (
            podbcCONNECT    pC,
            LPUSTR          szStmt          = NULL,
            BOOL            bPrepare        = FALSE,
            BOOL            bExecute        = FALSE,
            BOOL            bAutoBind       = FALSE,
            psPARMBIND      psParmBindings  = NULL,
            UWORD           uParmCount      = 0,
            PTR             pvParmStruct    = NULL,
            psCOLBIND       pColBindings    = NULL,
            UWORD           ColCount        = 0,
            PTR             pvColStruct     = NULL
            ); 

    odbcCURSOR
            (
            podbcCONNECT    pC,
            LPSTR           szStmt,
            BOOL            bPrepare        = FALSE,
            BOOL            bExecute        = FALSE,
            BOOL            bAutoBind       = FALSE,
            psPARMBIND      psParmBindings  = NULL,
            UWORD           uParmCount      = 0,
            PTR             pvParmStruct    = NULL,
            psCOLBIND       pColBindings    = NULL,
            UWORD           ColCount        = 0,
            PTR             pvColStruct     = NULL
            ); 

/***************************************************

    ~odbcCURSOR
    
    Destructor.
***************************************************/

    virtual ~odbcCURSOR();
    
/***************************************************

    FreeColBindings
    
    Free dynamically allocated column bindings and result set..
***************************************************/

    virtual void FreeColBindings(void);
                
/***************************************************

    SetColBindings
    
    Set value of protected data members that define
    bindings between members of data structure
    and SQL statement result set columns, as defined by the
    array of sCOLBIND structs.
***************************************************/

    virtual void        SetColBindings
            (
            psCOLBIND   psBindings  = NULL,
            UWORD       uCount      = 0,
            PTR         pvStruct    = NULL
            )
        {
        FreeColBindings();
        pColBindings    = psBindings;
        ColCount        = uCount;
        pSet            = pvStruct;
        };
        
/***************************************************

    GetColBindings
    
    Retrieves values of protected data members that define
    bindings between members of data structure and SQL 
    statement result set columns, as defined by the array 
    of sCOLBIND structs..
***************************************************/

    virtual void        GetColBindings
            (
            psCOLBIND       *ppsBindings,
            UWORD           *puCount,
            PTR         *ppvStruct
            )
        {
        *ppsBindings    = pColBindings;
        *puCount        = ColCount;
        *ppvStruct      = pSet;
        };
        
/***************************************************

    SetNoExtraByteOnStrings
    
    Set value of bNoExtraByteOnStrings.
***************************************************/

    virtual BOOL        SetNoExtraByteOnStrings
            (
            BOOL        bNewValue
            )
        {
        BOOL bOldValue = bNoExtraByteOnStrings;
        
        bNoExtraByteOnStrings = bNewValue;
        
        return bOldValue;
        };
        
/***************************************************

    LastColProcessed
    
    Zero-based index of the last sCOLBIND processed
    by BindCol() (useful if an error occurs).
***************************************************/

    virtual UWORD       LastColProcessed(void)
        {
        return iCol;
        };
            
/***************************************************

    SetStmt
    
    Sets the SQL command string associated with 
    the statement. You may also need to call
    ResetParams() if parameter bindings have
    taken place.
***************************************************/

    virtual LPUCSTR         SetStmt(LPUCSTR Stmt)
        {
        // inherit base class behavior
        LPUCSTR Temp = odbcSTMT::SetStmt(Stmt);
        // now set our own flag(s)
        bFirst          = FALSE;
        return Temp;
        };
        
    virtual LPCSTR      SetStmt(LPSTR Stmt)
        {
        // inherit base class behavior
        LPCSTR Temp = (LPCSTR)odbcSTMT::SetStmt((LPUSTR)Stmt);
        // now set our own flag(s)
        bFirst          = FALSE;
        return Temp;
        };
        
    ////////////////////////////////////////////////////////
    ////////////////////// core functions //////////////////
    ////////////////////////////////////////////////////////

/***************************************************

    BindCol
    
    Bind a single column in the result set by invoking
    SQLBindCol.
***************************************************/

    virtual RETCODE BindCol(
            UWORD       icol,
            SWORD       fCType,
            PTR         rgbValue,
            SDWORD      cbValueMax,
            SDWORD     *pcbValue);

/***************************************************

    BindCol
    
    Bind multiple columns as represented by the
    elements of the array of sCOLBIND structures.
***************************************************/

    virtual RETCODE BindCol(
            psCOLBIND   psResultSetBindings,
            UWORD       uCount,
            PTR         pvBuf
            );
            
/***************************************************

    BindCol
    
    Bind multiple columns as represented by the
    elements of the array of sCOLBIND structures
    supplied during construction.
***************************************************/

    virtual RETCODE BindCol(void);
            
/***************************************************

    DescribeCol
    
    Invoke SQLDescribeCol to retrieve attributes
    for a given column in the result set.
***************************************************/

    virtual RETCODE DescribeCol(
        UWORD       icol,
        LPUSTR      szColName,
        SWORD       cbColNameMax,
        SWORD      *pcbColName,
        SWORD      *pfSqlType,
        UDWORD     *pcbColDef,
        SWORD      *pibScale,
        SWORD      *pfNullable);

    virtual RETCODE DescribeCol(
        UWORD       icol,
        LPSTR       szColName,
        SWORD       cbColNameMax,
        SWORD      *pcbColName,
        SWORD      *pfSqlType,
        UDWORD     *pcbColDef,
        SWORD      *pibScale,
        SWORD      *pfNullable)
        {
        return
            DescribeCol(
                icol,
                (LPUSTR)szColName,
                cbColNameMax,
                pcbColName,
                pfSqlType,
                pcbColDef,
                pibScale,
                pfNullable);
        };
        
/***************************************************

    ColAttributes
    
    Invoke SQLColAttributes to retrieve attributes
    for a given column in the result set.
***************************************************/

    virtual RETCODE ColAttributes(
        UWORD        icol,
        UWORD        fDescType,
        PTR          rgbDesc,
        SWORD        cbDescMax,
        SWORD       *pcbDesc,
        SDWORD      *pfDesc);

/***************************************************

    Fetch
    
    Invoke SQLFetch.
***************************************************/

    virtual RETCODE Fetch(void);

/***************************************************

    GetCursorName
    
    Get the cursor name.
***************************************************/

    virtual RETCODE GetCursorName(
            LPUSTR      szCursor,
            SWORD       cbCursorMax,
            SWORD      *pcbCursor = NULL
            );

    virtual RETCODE GetCursorName(
            LPSTR       szCursor,
            SWORD       cbCursorMax,
            SWORD      *pcbCursor = NULL
            )
            {
            return
                GetCursorName
                    (
                    (LPUSTR)szCursor,
                    cbCursorMax,
                    pcbCursor
                    );
            };

/***************************************************

    SetCursorName
    
    Set the cursor name.
***************************************************/

    virtual RETCODE SetCursorName(
        LPUSTR szCursor
        );
    
    virtual RETCODE SetCursorName(
        LPSTR szCursor
        )
        {
        return
            SetCursorName(
                (LPUSTR)szCursor
                );
        };
    
/***************************************************

    NumResultCols
    
    Returns number of columns in the result set. This
    overloaded version returns the result code, and
    the number of columns in the signed word pointed
    to by pccol.
***************************************************/

    virtual RETCODE NumResultCols(
        SWORD      *pccol
        );

/***************************************************

    NumResultCols
    
    Returns number of columns in the result set. This
    overloaded version returns the number directly
***************************************************/

    virtual SWORD NumResultCols(void);

/***************************************************

    ExecDirect
    
    Invoke SQLExecDirect.
***************************************************/

    virtual RETCODE ExecDirect(
        LPUCSTR szSqlStr
        );

    virtual RETCODE ExecDirect(
        LPSTR szSqlStr
        )
        {
        return ExecDirect((LPUCSTR)szSqlStr);
        };

/***************************************************

    Unbind
    
    Invoke SQLFreeStmt with SQL_UNBIND flag. Removes 
    column bindings for result set.
***************************************************/

    virtual RETCODE Unbind(void);

/***************************************************

    Execute
    
    Invoke SQLExecute.
***************************************************/

    virtual RETCODE Execute(void);

/***************************************************

    Prepare
    
    Invoke SQLPrepare; bind parameters if parameter
    bindings need to be done. Overloaded version without
    argument prepares statement stored internally.
***************************************************/

    virtual RETCODE Prepare(LPUCSTR szSqlStr);
    
    virtual RETCODE Prepare(
        LPSTR szSqlStr
        )
        {
        return Prepare((LPUCSTR)szSqlStr);
        };

    virtual RETCODE Prepare(void);
    
    ////////////////////////////////////////////////////////
    ////////////////////// level 1 functions ///////////////
    ////////////////////////////////////////////////////////

/***************************************************

    GetData
    
    Invoke SQLGetData.
***************************************************/

    virtual RETCODE GetData(
            UWORD      icol,
            SWORD      fCType,
            PTR        rgbValue,
            SDWORD     cbValueMax,
            SDWORD     *pcbValue);

/***************************************************

    Columns
    
    Invoke SQLColumns to obtain information on the
    columns in the tables in this data source. Results
    are returned as a result set.
***************************************************/

    virtual RETCODE Columns(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName,
            LPUSTR szColumnName);

    virtual RETCODE Columns(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName,
            LPSTR szColumnName)
            {
            return
                Columns(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName,
                    (LPUSTR)szColumnName
                    );
            };

/***************************************************

    ColumnPrivileges
    
    Invoke SQLColumnPrivileges to obtain information 
    on the columns in the tables in this data source. 
    Results are returned as a result set.
***************************************************/

    virtual RETCODE ColumnPrivileges(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName,
            LPUSTR szColumnName);

    virtual RETCODE ColumnPrivileges(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName,
            LPSTR szColumnName)
            {
            return
                ColumnPrivileges(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName,
                    (LPUSTR)szColumnName
                    );
            };

/***************************************************

    Procedures
    
    Invoke SQLProcedures to obtain information
    on the available procedures in this data source.
    Results are returned as a result set.

***************************************************/

    virtual RETCODE Procedures(
            LPUSTR szProcQualifier,
            LPUSTR szProcOwner,
            LPUSTR szProcName);

    virtual RETCODE Procedures(
            LPSTR szProcQualifier,
            LPSTR szProcOwner,
            LPSTR szProcName)
            {
            return
                Procedures(
                    (LPUSTR)szProcQualifier,
                    (LPUSTR)szProcOwner,
                    (LPUSTR)szProcName
                    );
            };

/***************************************************

    ProcedureColumns
    
    Invoke SQLProcedureColumns to obtain information
    on the parameters and the result set columns for the
    available procedures in this data source. Results
    are returned as a result set.

***************************************************/

    virtual RETCODE ProcedureColumns(
            LPUSTR szProcQualifier,
            LPUSTR szProcOwner,
            LPUSTR szProcName,
            LPUSTR szColumnName);

    virtual RETCODE ProcedureColumns(
            LPSTR szProcQualifier,
            LPSTR szProcOwner,
            LPSTR szProcName,
            LPSTR szColumnName)
            {
            return
                ProcedureColumns(
                    (LPUSTR)szProcQualifier,
                    (LPUSTR)szProcOwner,
                    (LPUSTR)szProcName,
                    (LPUSTR)szColumnName
                    );
            };

/***************************************************

    Statistics
    
    Return statistics on the tables in this data source
    as a result set. Invokes SQLStatistics.
***************************************************/

    virtual RETCODE Statistics(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName,
            UWORD      fUnique,
            UWORD      fAccuracy);

    virtual RETCODE Statistics(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName,
            UWORD      fUnique,
            UWORD      fAccuracy)
            {
            return
                Statistics(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName,
                            fUnique,
                            fAccuracy
                            );
            };

/***************************************************

    GetTypeInfo
    
    Get information concerning supported types.
***************************************************/

    virtual RETCODE GetTypeInfo(SWORD fSqlType);

/***************************************************

    Tables
    
    Retrieve result set containing information about
    the tables in the data source.
***************************************************/

    virtual RETCODE Tables(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName,
            LPUSTR szTableType);

    virtual RETCODE Tables(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName,
            LPSTR szTableType)
            {
            return
                Tables(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName,
                    (LPUSTR)szTableType
                    );
            };

/***************************************************

    ForeignKeys
    
    Retrieve result set containing information about
    foreign keys relating the tables in the data source.
***************************************************/

virtual RETCODE ForeignKeys(
        LPUSTR szPkTableQualifier,
        LPUSTR szPkTableOwner,
        LPUSTR szPkTableName,
        LPUSTR szFkTableQualifier,
        LPUSTR szFkTableOwner,
        LPUSTR szFkTableName);

virtual RETCODE ForeignKeys(
        LPSTR szPkTableQualifier,
        LPSTR szPkTableOwner,
        LPSTR szPkTableName,
        LPSTR szFkTableQualifier,
        LPSTR szFkTableOwner,
        LPSTR szFkTableName)
        {
        return ForeignKeys(
                   (LPUSTR)szPkTableQualifier,
                   (LPUSTR)szPkTableOwner,
                   (LPUSTR)szPkTableName,
                   (LPUSTR)szFkTableQualifier,
                   (LPUSTR)szFkTableOwner,
                   (LPUSTR)szFkTableName);
        };

/***************************************************

    TablePrivileges
    
    Retrieve result set containing information about
    the tables in the data source.
***************************************************/

    virtual RETCODE TablePrivileges(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName);

    virtual RETCODE TablePrivileges(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName)
            {
            return
                TablePrivileges(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName
                    );
            };

/***************************************************

    PrimaryKeys
    
    Retrieve result set containing information about
    the primary keys of the tables in the data source.
***************************************************/

    virtual RETCODE PrimaryKeys(
            LPUSTR szTableQualifier,
            LPUSTR szTableOwner,
            LPUSTR szTableName);

    virtual RETCODE PrimaryKeys(
            LPSTR szTableQualifier,
            LPSTR szTableOwner,
            LPSTR szTableName)
            {
            return
                PrimaryKeys(
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName
                    );
            };

/***************************************************

    SpecialColumns
    
    Retrieve result set containing information about
    special columns, such as the set of column(s) that
    (taken together and in the order presented) form(s) an
    unique identifier for a row in the table.
***************************************************/

    virtual RETCODE SpecialColumns
            (
            UWORD   fColType,
            LPUSTR  szTableQualifier,
            LPUSTR  szTableOwner,
            LPUSTR  szTableName,
            UWORD   fScope,
            UWORD   fNullable
            );
    
    virtual RETCODE SpecialColumns
            (
            UWORD   fColType,
            LPSTR   szTableQualifier,
            LPSTR   szTableOwner,
            LPSTR   szTableName,
            UWORD   fScope,
            UWORD   fNullable
            )
            {
            return
                SpecialColumns
                    (
                    fColType,
                    (LPUSTR)szTableQualifier,
                    (LPUSTR)szTableOwner,
                    (LPUSTR)szTableName,
                    fScope,
                    fNullable
                    );
            };
    
    ////////////////////////////////////////////////////////
    ////////////////////// level 2 functions ///////////////
    ////////////////////////////////////////////////////////

/***************************************************

    SetPos
    
    Set cursor position in a returned result set block.
***************************************************/

    virtual RETCODE SetPos(
            UWORD      irow,
            UWORD      fRefresh,
            UWORD      fLock
            );

/***************************************************

    SetScrollOptions
    
    Set concurrency and other factors for use in ExtendedFetch.
***************************************************/

    virtual RETCODE SetScrollOptions(
            UWORD      fConcurrency,
            SDWORD     crowKeyset,
            UWORD      crowRowset
            );

    ////////////////////////////////////////////////
    ///////////// statement executors //////////////
    ////////////////////////////////////////////////
    
/***************************************************

    DoQuery
    
    Call DoStmt() to execute the SQL statement.
***************************************************/

    virtual BOOL DoQuery(void)
        {
        return DoStmt();
        };
        
/***************************************************

    pResultSet
    
    Return address of result set structure.
***************************************************/

    virtual PTR pResultSet(void)
        {
        return pSet;
        };

/***************************************************

    QueryAndBind
    
    Execute the query and bind result columns.
***************************************************/

    virtual BOOL QueryAndBind(void);
    
/***************************************************

    GetFirst

    Call Fetch() to get first row in the result set
    (or ExtFetchFirst() to get first rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

    virtual BOOL GetFirst(void);

/***************************************************

    GetNext

    Call Fetch() to get next row in the result set
    (or ExtFetchNext() to get next rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

    virtual BOOL GetNext(void);

/***************************************************

    GetLast

    Call Fetch() to get last rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

    virtual BOOL GetLast(void);

/***************************************************

    GetPrev

    Call ExtFetchPrev() to get prior rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

    virtual BOOL GetPrev(void);

/***************************************************

    EnumResults
    
    Using Fetch(), retrieve rows in the result set;
    for each row retrieved, invoke the callback
    unction supplied by the caller, passing the 'this'
    pointer and the user-supplied pointer 'pUser'.
    
    The callback function must conform to the typedef
    pfENUMRESULTS. A sample would be
    
    RETCODE     MyEnumFunc( podbcCURSOR thisObj,
                            PTR         pMyData
                          )
        {
        // you could assign and cast thisObj to a descendant
        // of odbcCURSOR here.
        // you could also assign and cast pMyData to a pointer
        // to a data structure of your own design.
        
        return SQL_SUCCESS; // == zero
        }
                    
        return a non-zero value to terminate enumeration
        of the result set; zero return continues enumeration.

***************************************************/

    virtual RETCODE EnumResults(pfENUMRESULTS pfEnum, PTR pUser = NULL);

/***************************************************

    ExtendedFetch
    
    Invoke SQLExtendedFetch.
***************************************************/

    virtual RETCODE ExtendedFetch(
                UWORD      fFetchType,
                SDWORD     irow,
                UDWORD     *pcrow,
                UWORD      *rgfRowStatus);

/***************************************************

    ExtFetchSetup
    
    Set up for automatic features surrounding
    SQLExtendedFetch.  Calls SetScrollOptions and
    allocates space for the row statuses.
***************************************************/

    virtual RETCODE ExtFetchSetup(
            UWORD      fConcurrency,
            SDWORD     crowKeyset,
            UWORD      crowRowset
            );
            
/***************************************************

    ExtFetchFirst
    
    Invoke SQLExtendedFetch to fetch first batch.
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchFirst(void);

/***************************************************

    ExtFetchNext
    
    Invoke SQLExtendedFetch to fetch next batch.
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchNext(void);

/***************************************************

    ExtFetchLast
    
    Invoke SQLExtendedFetch to fetch last batch.
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchLast(void);

/***************************************************

    ExtFetchPrevious
    
    Invoke SQLExtendedFetch to fetch previous batch.
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchPrevious(void);

/***************************************************

    ExtFetchRelative
    
    Invoke SQLExtendedFetch to fetch batch at relative
    position (n rows forward/backward).
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchRelative(
                SDWORD     irow
                );

/***************************************************

    ExtFetchAbsolute

    Invoke SQLExtendedFetch to fetch batch at absolute
    position (beginning at nth row).
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchAbsolute(
                SDWORD     irow
                );

/***************************************************

    ExtFetchResume
    
    Invoke SQLExtendedFetch to resume fetching batch after
    an error occurred during one of the other extended
    fetch operations.
    
    Returns the number of rows actually fetched.  If this
    number is not equal to ExtFetchRowCount, an error may
    have occurred; use the sqlsuccess() member function
    (and other odbcBASE members) to analyze the situation.
***************************************************/

    virtual UDWORD ExtFetchResume(void);

/***************************************************

    GetExtFetchType
    
    Get op type for supporting automated extended 
    fetch functions.
***************************************************/

    virtual UWORD   GetExtFetchType(void)
        {
        return ExtFetchType;
        };
    
/***************************************************

    GetExtFetchRow
    
    Get row variable for supporting automated extended 
    fetch functions.
***************************************************/

    virtual UDWORD  GetExtFetchRow(void)
        {
        return ExtFetchRow;
        };
    
/***************************************************

    GetExtFetchRelAbsRow
    
    Get relative/absolute row sent in automated extended 
    fetch functions.
***************************************************/

    virtual SDWORD  GetExtFetchRelAbsRow(void)
        {
        return ExtFetchRelAbsRow;
        };
    
/***************************************************

    GetExtFetchRowCount

    Get row count variable for supporting automated extended 
    fetch functions.
***************************************************/

        virtual UWORD   GetExtFetchRowCount(void)
        {
        return ExtFetchRowCount;
        };
    
/***************************************************

    GetExtFetchStatus
    
    Get row status from automated extended fetch functions.
    
    First overloaded version returns the status for a given
    row in the rowset.  iRow is 1-based, not zero-based.

    Second overloaded version returns const pointer to array
    of row statuses.  There are ExtFetchRowCount members
***************************************************/

    virtual UWORD   GetExtFetchStatus(UWORD iRow)
        {
        if (ExtFetchStatus != 0 &&
           iRow > 0 &&
           iRow <= ExtFetchRowCount
           )
                return ExtFetchStatus[iRow-1];
                else 
            return 0;
        };
    
    virtual const UWORD     *GetExtFetchStatus(void)
        {
        return ExtFetchStatus;
        };
    
    
/***************************************************

    MoreResults
    
    Invoke SQLMoreResults().
***************************************************/

    virtual RETCODE MoreResults(void);

/***************************************************

    AutoBind
    
    Retrieve array of column bindings.  Allocate a
    buffer to contain the output data and bind all
    the columns to it.  Accepts two args: 
    1) address of unsigned double-word for output structure
       size, and
    2) address of a PTR to store pointer to allocated
       buffer.
***************************************************/

    virtual RETCODE AutoBind
        (
        UWORD     *pcbStructSize    = NULL,
        PTR     *ppStruct           = NULL
        );

/***************************************************

    ColResultAddr

    Return address of bound column output. Six
    overloaded forms: The first four accept
    the name of the column and the second two
    accept the column number (they start with 1).

    The irow argument can be used with block cursors,
    to get the address of the column data for the
    irow'th row.  Row numbers start with 1.

    In all cases a NULL is returned if either the
    icol does not match a bound column OR the irow
    argument is out of bounds.
***************************************************/

    virtual PTR ColResultAddr(LPUCSTR ColName);

    virtual PTR ColResultAddr(LPCSTR ColName);

    virtual PTR ColResultAddr(UWORD icol);

    virtual PTR ColResultAddr(LPUCSTR ColName, UWORD irow);

    virtual PTR ColResultAddr(LPCSTR ColName, UWORD irow);

    virtual PTR ColResultAddr(UWORD icol, UWORD irow);

/***************************************************

    ColResultAs<type>

    Return value of bound column output. forty-two
    overloaded forms, accepting signed and unsigned
        char strings for column names, column numbers in
        the result set (numbered beginning with 1), and
        row within the rowset if using a block cursor.

        <type> can be Short, Unsigned Short, Long, Unsigned
        Long, Float, Double, or LPSTR.

        ColName is the name of a bound column; can be signed
        or unsigned character string.

        icol is the number of a bound column.

    The irow argument can be used with block cursors,
    to get the address of the column data for the
    irow'th row.  Row numbers start with 1.

    In all cases a NULL is returned if either the
    icol does not match a bound column OR the irow
    argument is out of bounds.
***************************************************/

        virtual SWORD  ColResultAsShort(LPCSTR ColName, UWORD irow);
        virtual SWORD  ColResultAsShort(LPUCSTR ColName, UWORD irow);
        virtual SWORD  ColResultAsShort(UWORD icol, UWORD irow);
        virtual UWORD  ColResultAsUnsignedShort(LPCSTR ColName, UWORD irow);
        virtual UWORD  ColResultAsUnsignedShort(LPUCSTR ColName, UWORD irow);
        virtual UWORD  ColResultAsUnsignedShort(UWORD icol, UWORD irow);
        virtual SDWORD ColResultAsLong(LPCSTR ColName, UWORD irow);
        virtual SDWORD ColResultAsLong(LPUCSTR ColName, UWORD irow);
        virtual SDWORD ColResultAsLong(UWORD icol, UWORD irow);
        virtual UDWORD ColResultAsUnsignedLong(LPCSTR ColName, UWORD irow);
          virtual UDWORD ColResultAsUnsignedLong(LPUCSTR ColName, UWORD irow);
        virtual UDWORD ColResultAsUnsignedLong(UWORD icol, UWORD irow);
        virtual float  ColResultAsFloat(LPCSTR ColName, UWORD irow);
        virtual float  ColResultAsFloat(LPUCSTR ColName, UWORD irow);
        virtual float  ColResultAsFloat(UWORD icol, UWORD irow);
        virtual double ColResultAsDouble(LPCSTR ColName, UWORD irow);
        virtual double ColResultAsDouble(LPUCSTR ColName, UWORD irow);
        virtual double ColResultAsDouble(UWORD icol, UWORD irow);
        virtual LPSTR  ColResultAsLPSTR(LPCSTR ColName, UWORD irow);
        virtual LPSTR  ColResultAsLPSTR(LPUCSTR ColName, UWORD irow);
        virtual LPSTR  ColResultAsLPSTR(UWORD icol, UWORD irow);

        virtual SWORD  ColResultAsShort(LPCSTR ColName);
        virtual SWORD  ColResultAsShort(LPUCSTR ColName);
        virtual SWORD  ColResultAsShort(UWORD icol);
        virtual UWORD  ColResultAsUnsignedShort(LPCSTR ColName);
        virtual UWORD  ColResultAsUnsignedShort(LPUCSTR ColName);
        virtual UWORD  ColResultAsUnsignedShort(UWORD icol);
        virtual SDWORD ColResultAsLong(LPCSTR ColName);
        virtual SDWORD ColResultAsLong(LPUCSTR ColName);
        virtual SDWORD ColResultAsLong(UWORD icol);
        virtual UDWORD ColResultAsUnsignedLong(LPCSTR ColName);
        virtual UDWORD ColResultAsUnsignedLong(LPUCSTR ColName);
        virtual UDWORD ColResultAsUnsignedLong(UWORD icol);
        virtual float  ColResultAsFloat(LPCSTR ColName);
        virtual float  ColResultAsFloat(LPUCSTR ColName);
        virtual float  ColResultAsFloat(UWORD icol);
        virtual double ColResultAsDouble(LPCSTR ColName);
        virtual double ColResultAsDouble(LPUCSTR ColName);
        virtual double ColResultAsDouble(UWORD icol);
        virtual LPSTR  ColResultAsLPSTR(LPCSTR ColName);
        virtual LPSTR  ColResultAsLPSTR(LPUCSTR ColName);
        virtual LPSTR  ColResultAsLPSTR(UWORD icol);
/***************************************************

    ColResultSizeArray

    Return address of column result size array for a
    bound column when extended fetch block cursor in use
    and either form of automatic binding is in use. 
    Two overloaded forms: The first accepts the name of the 
    column and the second accepts the column number (they 
    start with 1).
***************************************************/

        virtual const SDWORD     * ColResultSizeArray(LPUCSTR ColName);
        virtual const SDWORD     * ColResultSizeArray(LPCSTR ColName)
            {
            return ColResultSizeArray((LPUCSTR)ColName);
                        };
        virtual const SDWORD     * ColResultSizeArray(UWORD icol);
/***************************************************

    ColResultInfo
    
    Return address of data structure describing bound column. 
    Two overloaded forms: The first accepts the name of the 
    column and the second accepts the column number (they 
    start with 1).
***************************************************/

    virtual const psCOLBIND ColResultInfo(LPUCSTR ColName); 

    virtual const psCOLBIND ColResultInfo(LPCSTR ColName);

    virtual const psCOLBIND ColResultInfo(UWORD icol);
    
/***************************************************

    StructSize
    
    Return size of automatically allocated structure 
    for storing bound column output.
***************************************************/

    virtual UDWORD StructSize(void) { return pSSize; };

// new in v2.0

/**************************************************

    RowsetRowRecordSize

    Return size of one row's worth of data in the
    automatically allocated structure for storing
    bound column output during extended fetches,
    when the internal data structure consists of
    arrays of column data.

    Used to determine space allocation for a single
    rowset row structure (record structure) for use
    in call to MoveRowsetRowToRecord().
***************************************************/

    virtual size_t RowsetRowRecordSize(void);

/***************************************************

    MoveRowsetRowToRecord

    Return one row's worth of data in the
    automatically allocated structure for storing
    bound column output during extended fetches,
    when the internal data structure consists of
    arrays of column data.

    Use RowsetRowRecordSize() to determine space allocation
    for a single rowset row structure (record structure).
***************************************************/

    virtual RETCODE MoveRowsetRowToRecord( UWORD irow,
                        PTR pRecord,
                        size_t uSize,
                        size_t     *puBytesCopied );

/***************************************************

    MoveRecordToRowsetRow

    Return one row's worth of data from a single-row
    record structure to a rowset row in the internal
    rowset data structure dynamically allocated
    during AutoBind(). The rowset's single-row
    record structure is the same as that returned by
    MoveRowsetRowToRecord.

***************************************************/

    virtual RETCODE MoveRecordToRowsetRow( UWORD irow,
                        PTR pRecord, 
                        size_t uSize, 
                        size_t     *puBytesCopied ) ;
                        
/***************************************************

    GetNoExtraByteOnStrings

    Get value of bNoExtraByteOnStrings.
***************************************************/

    virtual BOOL        GetNoExtraByteOnStrings
            (
            void
            )
        {
        return bNoExtraByteOnStrings;
        };

/***************************************************

    SetParam
    
    Invoke SQLSetParam to bind a parameter.
***************************************************/

        virtual RETCODE SetParam(
                UWORD       ipar,
                SWORD       fCType,
                SWORD       fSqlType,
                UDWORD      cbColDef,
                SWORD       ibScale,
                PTR         rgbValue,
                SDWORD *    pcbValue)
                {
                return odbcSTMT::SetParam(
                            ipar,
                            fCType,
                            fSqlType,
                            cbColDef,
                            ibScale,
                            rgbValue,
                            pcbValue
                            );
                };

/***************************************************

    SetParam

    Bind multiple parameters as represented by the
    array of sPARMBIND structures pointed to by
    psParmBindings. The count of structures is
    passed as the uCount argument.If desired, these
    are bound to the structure pointed to by pvBuf.
    (You can also use the constants SQL_DATA_AT_EXEC
    and SQL_NULL_DATA to indicate run-time parameter
    binding or null data values; see sqlstruc.hpp
    for instructions).

    Unlike its equivalent in odbcSTMT, this one will
    handle BLOBs (the odbcBLOB class constructor requires
    an odbcCURSOR instance, since it can handle reading
    as well as writing BLOB data).

    One disadvantage is that this function binds
    only a single value to each parameter.
***************************************************/

        virtual RETCODE SetParam(
                cpsPARMBIND psParmBindings,
                UWORD       uCount,
                PTR         pvBuf
                );

/***************************************************

    ColResultAsBLOB

    Return value of bound column output where the column
    is of SQL type SQL_LONGVARBINARY or SQL_LONGVARCHAR,
    and was automatically bound as a BLOB (using the odbcBLOB
    class).

        ColName is the name of a bound column; can be signed
        or unsigned character string.

        icol is the number of a bound column.

    The irow argument can be used with block cursors,
    to get the address of the column data for the
    irow'th row.  Row numbers start with 1.

    In all cases a NULL is returned if either the
    icol does not match a bound column OR the irow
    argument is out of bounds.
***************************************************/

        virtual podbcBLOB  ColResultAsBLOB(LPCSTR ColName, UWORD irow);
        virtual podbcBLOB  ColResultAsBLOB(LPUCSTR ColName, UWORD irow);
        virtual podbcBLOB  ColResultAsBLOB(UWORD icol, UWORD irow);
        virtual podbcBLOB  ColResultAsBLOB(LPCSTR ColName);
        virtual podbcBLOB  ColResultAsBLOB(LPUCSTR ColName);
        virtual podbcBLOB  ColResultAsBLOB(UWORD icol);

private:

friend class odbcBLOB; // to get at SetRC()

protected:
/***************************************************

    bBlobsBound

    Flag indicating that BLOBs are bound, so we can
    know to trigger auto-input behavior in Execute()
    and ExecDirect().
***************************************************/
    BOOL bBlobsBound;
    
public:
/***************************************************

    ResetParams
    
    Remove old parameter bindings by calling SQLFreeStmt
    with SQL_RESET_PARAMS flag.
***************************************************/

        virtual RETCODE ResetParams(void)
                        {
                        odbcSTMT::ResetParams();
                        bBlobsBound = FALSE;
                        return lastRC();
                        };

/************************************************************
    SQLSetPos variations for inserting, updating, and deleting
    directly from the rowset. Also for refreshing the rowset,
    setting position, setting lock status, etc.
************************************************************/

    virtual RETCODE PositionToRow( UWORD irow )
            {
            return SetPos( irow, SQL_POSITION, SQL_LOCK_NO_CHANGE );
            };

    virtual RETCODE LockRow( UWORD irow, UWORD fLock )
            {
            return SetPos( irow, SQL_POSITION, fLock );
            };

    virtual RETCODE RefreshRow( UWORD irow, UWORD fLock )
            {
            return SetPos( irow, SQL_REFRESH, fLock );
            };

    virtual RETCODE AddRow( UWORD irow )
            {
            return SetPos( irow, SQL_ADD, SQL_LOCK_NO_CHANGE );
            };

    virtual RETCODE UpdateRow( UWORD irow )
            {
            return SetPos( irow, SQL_UPDATE, SQL_LOCK_NO_CHANGE );
            };

    virtual RETCODE DeleteRow( UWORD irow )
            {
            return SetPos( irow, SQL_DELETE, SQL_LOCK_NO_CHANGE );
            };

// end new in v2.0

// new in v2.1

/*************************************************************

    ClearRowsetBuffer

    Clears contents of rowset buffer.

*************************************************************/
    virtual RETCODE ClearRowsetBuffer( void );

/*************************************************************

    IsColValueNull

    Returns TRUE if column has value of NULL, FALSE otherwise.

*************************************************************/

    virtual BOOL IsColValueNull( LPUCSTR ColName ) ;

    virtual BOOL IsColValueNull(LPUCSTR ColName, UWORD irow) ;

    virtual BOOL IsColValueNull( LPCSTR ColName )
            {
            return IsColValueNull((LPUCSTR)ColName );
            } ;

    virtual BOOL IsColValueNull( LPCSTR ColName, UWORD irow )
            {
            return IsColValueNull((LPUCSTR)ColName, irow );
            } ;

    virtual BOOL IsColValueNull(UWORD icol) ;

    virtual BOOL IsColValueNull(UWORD icol, UWORD irow) ;

/*************************************************************

    SetColValueNull

    Returns TRUE if column can be set to NULL, FALSE otherwise.

*************************************************************/

    virtual BOOL SetColValueNull( LPUCSTR ColName ) ;

    virtual BOOL SetColValueNull(LPUCSTR ColName, UWORD irow) ;

    virtual BOOL SetColValueNull( LPCSTR ColName )
            {
            return SetColValueNull((LPUCSTR)ColName );
            } ;

    virtual BOOL SetColValueNull( LPCSTR ColName, UWORD irow )
            {
            return SetColValueNull((LPUCSTR)ColName, irow );
            } ;

    virtual BOOL SetColValueNull(UWORD icol) ;

    virtual BOOL SetColValueNull(UWORD icol, UWORD irow) ;

// end new in v2.1

};  // end of class declaration: odbcCURSOR

#endif // if !defined(_CURSOR_HPP)
