/*
	File:           QUERITER.CPP
	
	Revision:       1.0 Release
	
	Date:           14-Mar-94

	Author:         Dale Hunscher
	
	Description:
	
	This file is the implementation of the various standard
	cursor classes supporting ODBC library information functions 
	that return result sets.
	
	Each of these is effectively just a cursor; all the specialized
	types do is bind the columns in the result set to an
	appropriate data structure.

	These are useful as examples of how to use an array of sCOLBIND 
	structures to bind columns in a result set to a specialized
	data structure.  While this would be an unwieldy approach to an 
	ad hoc query, it is an effective way to produce reusable queries.
	
	Following the pattern here, you can produce specialized cursor
	classes that "know" how to perform a query and produce output
	from the query in the form of a data structure.
	
    Engineers:
    DAH             Dale A. Hunscher

    Revision History
    ================
    Date            Who         Did What
    -------------------------------------------------------------
    03-Dec-94       DAH         Moved odbcSYSINFOITERATOR::DoQuery
                                here from queriter.hpp.

	/////////////////////////////////////////////////////////////
	///////////////////// NOTICE ////////////////////////////////
	/////////////////////////////////////////////////////////////
									     
	Copyright (c) 1993-1995 by INTERSOLV, Inc. All rights reserved.

	Information in this document is subject to change without
	notice and does not represent a commitment on the part of
	INTERSOLV, Inc. This software is provided under
	a license agreement or non-disclosure agreement. The software
	may be used and/or copied only in accordance with the terms
	of the governing agreement. It is against the law to copy
	the software on any medium except as specifically allowed
	in the governing agreement. No part of this software may be 
	reproduced or transmitted in any form or by any means,
	electronic or mechanical, including photocopying, recording,
	or information storage and retrieval systems, for any purpose
	other than the licensee's personal use, without the express
	written permission of INTERSOLV, Inc.
	
*/

#include <sql.hpp>

    /**********************************************************
	odbcSYSINFOITERATOR

	Constructor for iterator class to iterate result set.
    **********************************************************/

    odbcSYSINFOITERATOR::odbcSYSINFOITERATOR( podbcCONNECT   pC )
                    : odbcCURSOR(pC)
	{
    bCanScroll = FALSE;
	}

/**********************************************************
	~odbcSYSINFOITERATOR

	Destructor.
**********************************************************/

   odbcSYSINFOITERATOR:: ~odbcSYSINFOITERATOR() {}

    /**********************************************************
    DoQuery

    Process query.
    **********************************************************/

BOOL  odbcSYSINFOITERATOR::DoQuery(void)
    {
    if ( pConn->CursorsCanScroll() )
        {
        UWORD bSaveReport = AutoReport( odbcNOREPORT );
        UWORD bSaveRetrieve = AutoRetrieve( odbcNOREPORT );

        UDWORD ScrollOptions = 0;
        pConn->GetInfo( SQL_SCROLL_OPTIONS,
                        &ScrollOptions,
                        sizeof( UDWORD ),
                        NULL );

        SetRC( SQL_SUCCESS );

        // we favor keyset-driven cursors over static...
        if ( ScrollOptions & SQL_SO_KEYSET_DRIVEN )
            {
            // set up for extended fetch operations
            ExtFetchSetup(  SQL_CONCUR_VALUES,
                    SQL_CURSOR_KEYSET_DRIVEN,
                    1
                    ) ;
            }
        // if we didn't succeed with the keyset-driven cursor,
        // or we didn't try because keyset-driven cursors weren't supported,
        // try a static cursor using the ODBC 2.0 constants
        if (
                (!sqlsuccess() ||
                 (!(ScrollOptions & SQL_SO_KEYSET_DRIVEN))
                )
                && (ScrollOptions
                     & SQL_SO_STATIC) )
            {
            ExtFetchSetup(  SQL_CONCUR_VALUES,
                SQL_CURSOR_STATIC,
                1
                ) ;
            }

        // if we didn't succeed with the static or keyset-driven cursor,
        // but keyset-driven cursors are known to be possible,
        // try a keyset-driven cursor using the deprecated (ODBC v1.0)
        // value
        if ( !sqlsuccess() && (ScrollOptions & SQL_SO_KEYSET_DRIVEN) )
            {
            ExtFetchSetup(  SQL_CONCUR_VALUES,
                SQL_SCROLL_KEYSET_DRIVEN,
                1
                ) ;
            }
        else if ( !sqlsuccess() )
            SetRC( SQL_ERROR );
                
        // turn back on the error handling
        AutoReport( bSaveReport );
        AutoRetrieve( bSaveRetrieve );
        bCanScroll = sqlsuccess();
        SetRC( SQL_SUCCESS );
    }
    else
        bCanScroll = FALSE;

    bFirst = TRUE;

    return sqlsuccess();
    };

/***************************************************

    GetFirst

    Call Fetch() to get first row in the result set
    (or ExtFetchFirst() to get first rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

BOOL odbcSYSINFOITERATOR::GetFirst(void)
    {
    if (bError)
	return FALSE;
	
    if (bCanScroll)
	ExtFetchFirst();
    else
	{
	if ( !bFirst )
	    {
	    Close();
	    DoQuery();
	    }
	Fetch();
	bFirst = FALSE;
	}
    return sqlsuccess();
    }

/***************************************************

    GetNext

    Call Fetch() to get next row in the result set
    (or ExtFetchNext() to get next rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

BOOL odbcSYSINFOITERATOR::GetNext(void)
    {
    if (bError)
	return FALSE;

    if (bCanScroll)
	ExtFetchNext();
    else
	Fetch();
    bFirst = FALSE;
    return sqlsuccess();
    }

/***************************************************

    GetLast

    Call Fetch() to get last rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

BOOL odbcSYSINFOITERATOR::GetLast(void)
    {
    if (bError)
	return FALSE;

    if (bCanScroll)
	ExtFetchLast();
    else
	return FALSE;
    return sqlsuccess();
    }

/***************************************************

    GetPrev

    Call ExtFetchPrev() to get prior rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
***************************************************/

BOOL odbcSYSINFOITERATOR::GetPrev(void)
    {
    if (bError)
	return FALSE;
	
    if (bCanScroll)
	ExtFetchPrevious();
    else
	return FALSE;
    return sqlsuccess();
    }

/**********************************************************
	~odbcTableIterator

	Destructor.
**********************************************************/

odbcTableIterator::~odbcTableIterator() {}

/**********************************************************
	odbcTableIterator
	
	Constructor for iterator class to process result set 
	for SQLTables.
**********************************************************/

odbcTableIterator::odbcTableIterator(
		podbcCONNECT    pC,
	LPUSTR                  szTableQualifier,
		LPUSTR                  szTableOwner,
		LPUSTR                  szTableName,
		LPUSTR                  szTableType
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	szType                  = szTableType;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLTablesRSB(),
			pC->SQLTablesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcTableIterator::odbcTableIterator(
		podbcCONNECT    pC,
	LPSTR                   szTableQualifier,
		LPSTR                   szTableOwner,
		LPSTR                   szTableName,
		LPSTR                   szTableType
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	szType                  = (LPUSTR)szTableType;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLTablesRSB(),
			pC->SQLTablesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery
	
	Process query.
**********************************************************/

BOOL odbcTableIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	Tables
		(
		szQualifier,
		szOwner,
		szName,
		szType
		);

	return sqlsuccess();
	}
	
/**********************************************************
	~odbcStatisticsIterator
	
	Destructor.
**********************************************************/

odbcStatisticsIterator::~odbcStatisticsIterator(){}

/**********************************************************
	odbcStatisticsIterator
	
	Constructor for iterator class to process result set 
	for SQLStatistics.
**********************************************************/

odbcStatisticsIterator::odbcStatisticsIterator(
		podbcCONNECT    pC,
	LPUSTR                  szTableQualifier,
		LPUSTR                  szTableOwner,
		LPUSTR                  szTableName,
		UWORD                   fTblUnique,
		UWORD                   fTblAccuracy
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{ 
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	fUnique                 = fTblUnique;
	fAccuracy               = fTblAccuracy;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLStatisticsRSB(), 
			pC->SQLStatisticsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcStatisticsIterator::odbcStatisticsIterator(
		podbcCONNECT    pC,
	LPSTR                   szTableQualifier,
		LPSTR                   szTableOwner,
		LPSTR                   szTableName,
		UWORD                   fTblUnique,
		UWORD                   fTblAccuracy
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{ 
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	fUnique                 = fTblUnique;
	fAccuracy               = fTblAccuracy;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLStatisticsRSB(), 
			pC->SQLStatisticsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery
	
	Process query.
**********************************************************/

BOOL odbcStatisticsIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	Statistics
		(
		szQualifier,
		szOwner,
		szName,
		fUnique,
		fAccuracy
		);

	return sqlsuccess();
	}
	
/**********************************************************
	~odbcColumnIterator
	
	Destructor.
**********************************************************/

odbcColumnIterator::~odbcColumnIterator(){}

/**********************************************************
	odbcColumnIterator
	
	Constructor for iterator class to process result set 
	for SQLColumns.
**********************************************************/

odbcColumnIterator::odbcColumnIterator(
		podbcCONNECT pC,
	LPUSTR szTableQualifier,
		LPUSTR szTableOwner,
		LPUSTR szTableName,
		LPUSTR szColumnName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	szColName               = szColumnName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLColumnsRSB(), 
			pC->SQLColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcColumnIterator::odbcColumnIterator(
		podbcCONNECT pC,
	LPSTR szTableQualifier,
		LPSTR szTableOwner,
		LPSTR szTableName,
		LPSTR szColumnName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	szColName               = (LPUSTR)szColumnName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLColumnsRSB(), 
			pC->SQLColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery
	
	Process query.
**********************************************************/

BOOL odbcColumnIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	Columns
		(
		szQualifier,
		szOwner,
		szName,
		szColName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcSpecialColumnIterator
	
	Destructor.
**********************************************************/

odbcSpecialColumnIterator::~odbcSpecialColumnIterator(){}

/**********************************************************
	odbcSpecialColumnIterator

	Constructor for iterator class to process result set 
	for SQLSpecialColumns.
**********************************************************/

odbcSpecialColumnIterator::odbcSpecialColumnIterator(
			podbcCONNECT    pC,
		UWORD                   fColType,
		LPUSTR                  szTableQualifier,
		LPUSTR                  szTableOwner,
		LPUSTR                  szTableName,
		UWORD                   fScope,
		UWORD                   fNullable
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	Nullable                = fNullable;
	Type                    = fColType;
	Scope                   = fScope;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLSpecialColumnsRSB(), 
			pC->SQLSpecialColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcSpecialColumnIterator::odbcSpecialColumnIterator(
			podbcCONNECT    pC,
		UWORD                   fColType,
		LPSTR                   szTableQualifier,
		LPSTR                   szTableOwner,
		LPSTR                   szTableName,
		UWORD                   fScope,
		UWORD                   fNullable
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	Nullable                = fNullable;
	Type                    = fColType;
	Scope                   = fScope;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLSpecialColumnsRSB(), 
			pC->SQLSpecialColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcSpecialColumnIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	SpecialColumns
		(
		Type,
		szQualifier,
		szOwner,
		szName,
		Scope,
		Nullable
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcTypeInfoIterator
	
	Destructor.
**********************************************************/

odbcTypeInfoIterator::~odbcTypeInfoIterator(){}

/**********************************************************
	odbcTypeInfoIterator
	
	Constructor.
**********************************************************/

odbcTypeInfoIterator::odbcTypeInfoIterator(
			podbcCONNECT    pC
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLTypeInfoRSB(), 
			pC->SQLTypeInfoRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery
	
	Process query.
**********************************************************/

BOOL odbcTypeInfoIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	GetTypeInfo
		(
		SQL_ALL_TYPES
		);

	return sqlsuccess();
	}

// new in v2.0

/**********************************************************
	~odbcColumnPrivilegesIterator
	
	Destructor.
**********************************************************/

odbcColumnPrivilegesIterator::~odbcColumnPrivilegesIterator(){}

/**********************************************************
	odbcColumnPrivilegesIterator
	
	Constructor for iterator class to process result set 
	for SQLColumnPrivileges.
**********************************************************/

odbcColumnPrivilegesIterator::odbcColumnPrivilegesIterator(
		podbcCONNECT pC,
	LPUSTR szTableQualifier,
		LPUSTR szTableOwner,
		LPUSTR szTableName,
		LPUSTR szColumnName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	szColName               = szColumnName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLColumnPrivilegesRSB(), 
			pC->SQLColumnPrivilegesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcColumnPrivilegesIterator::odbcColumnPrivilegesIterator(
		podbcCONNECT pC,
	LPSTR szTableQualifier,
		LPSTR szTableOwner,
		LPSTR szTableName,
		LPSTR szColumnName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	szColName               = (LPUSTR)szColumnName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLColumnPrivilegesRSB(), 
			pC->SQLColumnPrivilegesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcColumnPrivilegesIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	ColumnPrivileges
		(
		szQualifier,
		szOwner,
		szName,
		szColName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcTablePrivilegesIterator

	Destructor.
**********************************************************/

odbcTablePrivilegesIterator::~odbcTablePrivilegesIterator(){}

/**********************************************************
	odbcTablePrivilegesIterator

	Constructor for iterator class to process result set
	for SQLTablePrivileges.
**********************************************************/

odbcTablePrivilegesIterator::odbcTablePrivilegesIterator(
		podbcCONNECT pC,
	LPUSTR szTableQualifier,
		LPUSTR szTableOwner,
		LPUSTR szTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLTablePrivilegesRSB(), 
			pC->SQLTablePrivilegesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcTablePrivilegesIterator::odbcTablePrivilegesIterator(
		podbcCONNECT pC,
	LPSTR szTableQualifier,
		LPSTR szTableOwner,
		LPSTR szTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLTablePrivilegesRSB(), 
			pC->SQLTablePrivilegesRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery
	
	Process query.
**********************************************************/

BOOL odbcTablePrivilegesIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	TablePrivileges
		(
		szQualifier,
		szOwner,
		szName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcProceduresIterator

	Destructor.
**********************************************************/

odbcProceduresIterator::~odbcProceduresIterator(){}

/**********************************************************
	odbcProceduresIterator

	Constructor for iterator class to process result set
	for SQLProcedures.
**********************************************************/

odbcProceduresIterator::odbcProceduresIterator(
		podbcCONNECT pC,
	LPUSTR szProcQualifier,
		LPUSTR szProcOwner,
		LPUSTR szProcName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szProcQualifier;
	szOwner                 = szProcOwner;
	szName                  = szProcName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLProceduresRSB(), 
			pC->SQLProceduresRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcProceduresIterator::odbcProceduresIterator(
		podbcCONNECT pC,
	LPSTR szProcQualifier,
		LPSTR szProcOwner,
		LPSTR szProcName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szProcQualifier;
	szOwner                 = (LPUSTR)szProcOwner;
	szName                  = (LPUSTR)szProcName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLProceduresRSB(), 
			pC->SQLProceduresRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcProceduresIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	Procedures
		(
		szQualifier,
		szOwner,
		szName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcProcedureColumnsIterator

	Destructor.
**********************************************************/

odbcProcedureColumnsIterator::~odbcProcedureColumnsIterator(){}

/**********************************************************
	odbcProcedureColumnsIterator

	Constructor for iterator class to process result set
	for SQLProcedureColumns.
**********************************************************/

odbcProcedureColumnsIterator::odbcProcedureColumnsIterator(
		podbcCONNECT pC,
	LPUSTR szProcQualifier,
		LPUSTR szProcOwner,
		LPUSTR szProcName,
	LPUSTR szProcColumn
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szProcQualifier;
	szOwner                 = szProcOwner;
	szName                  = szProcName;
    szColName       = szProcColumn;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLProcedureColumnsRSB(), 
			pC->SQLProcedureColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcProcedureColumnsIterator::odbcProcedureColumnsIterator(
		podbcCONNECT pC,
	LPSTR szProcQualifier,
		LPSTR szProcOwner,
		LPSTR szProcName,
	LPSTR szProcColumn
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szProcQualifier;
	szOwner                 = (LPUSTR)szProcOwner;
	szName                  = (LPUSTR)szProcName;
    szColName       = (LPUSTR)szProcColumn;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLProcedureColumnsRSB(), 
			pC->SQLProcedureColumnsRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcProcedureColumnsIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	ProcedureColumns
		(
		szQualifier,
		szOwner,
		szName,
	szColName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcPrimaryKeysIterator

	Destructor.
**********************************************************/

odbcPrimaryKeysIterator::~odbcPrimaryKeysIterator(){}

/**********************************************************
	odbcPrimaryKeysIterator

	Constructor for iterator class to process result set
	for SQLPrimaryKeys.
**********************************************************/

odbcPrimaryKeysIterator::odbcPrimaryKeysIterator(
		podbcCONNECT pC,
	LPUSTR szTableQualifier,
		LPUSTR szTableOwner,
		LPUSTR szTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = szTableQualifier;
	szOwner                 = szTableOwner;
	szName                  = szTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLPrimaryKeysRSB(), 
			pC->SQLPrimaryKeysRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcPrimaryKeysIterator::odbcPrimaryKeysIterator(
		podbcCONNECT pC,
	LPSTR szTableQualifier,
		LPSTR szTableOwner,
		LPSTR szTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szQualifier         = (LPUSTR)szTableQualifier;
	szOwner                 = (LPUSTR)szTableOwner;
	szName                  = (LPUSTR)szTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLPrimaryKeysRSB(), 
			pC->SQLPrimaryKeysRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcPrimaryKeysIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	PrimaryKeys
		(
		szQualifier,
		szOwner,
		szName
		);

	return sqlsuccess();
	}

/**********************************************************
	~odbcForeignKeysIterator

	Destructor.
**********************************************************/

odbcForeignKeysIterator::~odbcForeignKeysIterator(){}

/**********************************************************
	odbcForeignKeysIterator

	Constructor for iterator class to process result set
	for SQLForeignKeys.
**********************************************************/

odbcForeignKeysIterator::odbcForeignKeysIterator(
		podbcCONNECT pC,
	LPUSTR szPkTableQualifier,
		LPUSTR szPkTableOwner,
		LPUSTR szPkTableName,
	LPUSTR szFkTableQualifier,
		LPUSTR szFkTableOwner,
		LPUSTR szFkTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szPkQualifier           = szPkTableQualifier;
	szPkOwner                   = szPkTableOwner;
	szPkName                        = szPkTableName;
    szFkQualifier           = szFkTableQualifier;
	szFkOwner                   = szFkTableOwner;
	szFkName                        = szFkTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLForeignKeysRSB(), 
			pC->SQLForeignKeysRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

odbcForeignKeysIterator::odbcForeignKeysIterator(
		podbcCONNECT pC,
	LPSTR szPkTableQualifier,
		LPSTR szPkTableOwner,
		LPSTR szPkTableName,
	LPSTR szFkTableQualifier,
		LPSTR szFkTableOwner,
		LPSTR szFkTableName
		) :
		odbcSYSINFOITERATOR
			(
			pC
			)
	{
    szPkQualifier           = (LPUSTR)szPkTableQualifier;
	szPkOwner                   = (LPUSTR)szPkTableOwner;
	szPkName                        = (LPUSTR)szPkTableName;
    szFkQualifier           = (LPUSTR)szFkTableQualifier;
	szFkOwner                   = (LPUSTR)szFkTableOwner;
	szFkName                        = (LPUSTR)szFkTableName;
	if (sqlsuccess())
		{
		SetColBindings
			(
			pC->SQLForeignKeysRSB(), 
			pC->SQLForeignKeysRSBCount(),
			&Set
			);
		if (!bError)
			{
			QueryAndBind();
			}
		}
	}

/**********************************************************
	DoQuery

	Process query.
**********************************************************/

BOOL odbcForeignKeysIterator::DoQuery(void)
	{
    odbcSYSINFOITERATOR::DoQuery();

	ForeignKeys
		(
		szPkQualifier,
		szPkOwner,
		szPkName,
		szFkQualifier,
		szFkOwner,
		szFkName
		);

	return sqlsuccess();
	}


// end new in v2.0
