/*
	File:		Puzzle.cpp				

	Contains:	Source code for Puzzle, an example/testbed part for:
					Embedding
					Drag & drop
					Splitting
				Instead of scrambling a picture, the user may drag in any
				OpenDoc part. Pretty cool, huh? One may either edit the part in its
				scrambled form, or try and solve the terrifically difficult mind-boggling
				puzzle formed by chopping the part into 15 little rectangles and moving
				them about...
				
				There are some difficulties to take note of here. Parts must be able to handle
				the OpenDoc imaging system and take into account having multiple discountinuous clip
				regions (or clip SHAPES in OpenDoc parlance). Otherwise, these parts may have some
				agregious disagreements with PuzzlePart.

*/

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _ALTPOINT_
#include "AltPoint.h"
#endif

#ifndef SOM_Module_OpenDoc_Commands_defined
#include <CmdDefs.xh>
#endif

#ifndef SOM_ODArbitrator_xh
#include <Arbitrat.xh>
#endif

#ifndef _ISOSTRING_
#include "ISOStr.h"
#endif

#ifndef _ORDCOLL_
#include "OrdColl.h"
#endif

#ifndef SOM_ODDraft_xh
#include <Draft.xh>
#endif

#ifndef SOM_ODDispatcher_xh
#include <Disptch.xh>
#endif

#ifndef _EXCEPT_
#include "Except.h"
#endif

#ifndef _ODMEMORY_
#include "ODMemory.h"
#endif

#ifndef _FOCUSLIB_
#include "FocusLib.h"
#endif

#ifndef SOM_ODFacet_xh
#include <Facet.xh>
#endif

#ifndef SOM_ODFrame_xh
#include <Frame.xh>
#endif

#ifndef SOM_ODShape_xh
#include <Shape.xh>
#endif

#ifndef SOM_ODStorageUnit_xh
#include <StorageU.xh>
#endif

#ifndef SOM_ODStorageUnitView_xh
#include <SUView.xh>
#endif

#ifndef SOM_ODStorageUnitCursor_xh
#include <SUCursor.xh>
#endif

#ifndef SOM_ODDraft_xh
#include <Draft.xh>
#endif

#ifndef SOM_ODDocument_xh
#include <Document.xh>
#endif

#ifndef SOM_ODContainer_xh
#include <ODCtr.xh>
#endif

#ifndef SOM_ODStorageUnit_xh
#include <StorageU.xh>
#endif

#ifndef SOM_ODSession_xh
#include <ODSessn.xh>
#endif

#ifndef _INFOUTIL_
#include <InfoUtil.h>  // $$$$$  Note: This is a private OpenDoc Utility
#endif

#ifndef SOM_ODMenuBar_xh
#include <MenuBar.xh>
#endif

#ifndef SOM_ODWindow_xh
#include <Window.xh>
#endif

#ifndef SOM_ODWindowState_xh
#include <WinStat.xh>
#endif

#ifndef SOM_ODTransform_xh
#include <Trnsform.xh>
#endif

#ifndef _SEUTILS_
#include "SEUtils.h"
#endif

#ifndef SOM_Module_OpenDoc_StdProps_defined
#include <StdProps.xh>
#endif

#ifndef SOM_ODDragAndDrop_xh
#include <DragDrp.xh>
#endif

#ifndef SOM_ODDragItemIterator_xh
#include <DgItmIt.xh>
#endif

#ifndef SOM_ODFrameFacetIterator_xh
#include "FrFaItr.xh"
#endif

#ifndef SOM_ODFacetIterator_xh
#include "FacetItr.xh"
#endif

#ifndef SOM_ODEmbeddedFramesIterator_xh
#include "EmbFrItr.xh"
#endif

#ifndef SOM_Module_OpenDoc_StdTypes_defined
#include <StdTypes.xh>
#endif

#ifndef SOM_Module_OpenDoc_StdDefs_defined
#include <StdDefs.xh>
#endif

#ifndef SOM_ODCanvas_xh
#include <Canvas.xh>
#endif

#ifndef _ODUTILS_
#include <ODUtils.h>
#endif

#ifndef _ODUTILW_
#include <ODUtilW.h>
#endif

#ifndef SOM_ODUndo_xh
#include <Undo.xh>
#endif

#ifndef SOM_Module_OpenDoc_Foci_defined
#include <Foci.xh>
#endif

#ifndef SOM_ODPartWrapper_xh
#include "PartWrap.xh"
#endif

#ifndef __LIMITS__
#include <limits.h>
#endif

#ifndef __STDLIB__
#include <StdLib.h>		// Abs
#endif

#ifndef SOM_ODFocusSet_xh
#include <FocusSet.xh>
#endif

#ifndef SOM_ODClipboard_xh
#include <Clipbd.xh>
#endif

#include	<string.h>


// Puzzle Includes

#ifndef __PUZZLE__
#include "Puzzle.h"
#endif

#define VARIABLE_MACROS													// Wish to use variable macros
//#define METHOD_MACROS													// Wish to use method macros
#define Puzzle_Class_Source												// I am the class source

#ifndef SOM_Puzzle_xih
#include <Puzzle.xih>
#endif

#ifdef TESTING //bt
void ShowRgn(HRGN rgn);
#endif

//==============================================================================
// Global Variable
//==============================================================================

static Proxy*	dadSelection;

//==============================================================================
// Constants
//==============================================================================

//	kill these lines later; just here so stupid linker will stop complaining
/*
//bt const ODType kODPartDrawPart          = "Apple:Kind:TestDraw";
//bt const ODType kODPartClockPart         = "Apple:Kind:TestClock";
//bt const ODType kODPartDragPart          = "Apple:Kind:TestDrag";
*/
//	end kill				PENDING

	const ODSShort kXMPBorderWidth = 4;
	const ODSShort kODHandleLenMultiplier = 1;

	static const COLORREF rgbGray =	 	RGB(127, 127, 127 );
	static const COLORREF rgbRed =		RGB(255, 0, 0 );
	static const COLORREF rgbGreen =	RGB(0, 255, 0 );
	static const COLORREF rgbYellow =	RGB(255, 255, 0 );
	static const COLORREF rgbBlue =	 	RGB(0, 0, 255 );
	static const COLORREF rgbMagenta =	RGB(255, 0, 255 );
	static const COLORREF rgbCyan =	 	RGB(0, 255, 255 );
	static const COLORREF rgbWhite =	RGB(255, 255, 255 );
	static const COLORREF rgbBlack =	RGB(0, 0, 0 );

	const ODSShort kPaletteLeft = 0;
	const ODSShort kPaletteRight = 50;
	const ODSShort kPaletteTop = 0;
	const ODSShort kPaletteBottom = 200;

	enum {
		kGray = 0,
		kRed,
		kGreen,
		kYellow,
		kBlue,
		kMagenta,
		kCyan,
		kWhite,
		kNumColors
	};

	const		ODSShort	kMargin = 10;

const short 	kSuspendResumeMessage = 0x01;	// Resume vs. suspend mask
const short 	kMouseMovedMessage    = 0xFA;	// High byte mouse-moved event message

//==============================================================================
// 'ternalization 
//==============================================================================
		
	const ODPropertyName		kXMPPropFrameInfo = "Puzzle:Property:FrameInfo";
	const ODPropertyName		kXMPPropMouseDownOffset = "Puzzle:Property:MouseDownOffset";

	const ODPropertyName		kXMPPropPuzzleData = "Puzzle:Property:Data";

	const ODValueType			kWinPoint = "Puzzle:Type:Point";
	const ODValueType			kODTPrintRec = "Macintosh:Type:TPrint Record";

	const ODType				kDrawPresNormal =  "Puzzle:Presentation:Normal";
	const ODType				kDrawPresPalette = "Puzzle:Presentation:Palette";

//==============================================================================

SOM_Scope ODWindow*  SOMLINK ODCreateWindow(Puzzle *somSelf, Environment *ev,
		ODFrame* sourceFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	ODPlatformWindow		platformWindow = kODNULL;
	ODWindow*				window = kODNULL;
	
	platformWindow =  _fSession->GetWindowState(ev)->CreatePlatformWindow(ev, kODFalse, WS_CHILD | WS_CLIPSIBLINGS);
	
	window =  _fSession->GetWindowState(ev)->RegisterWindow(ev, platformWindow,
		kODNULL,														
		(sourceFrame==kODNULL),	// Keeps draft open
		kODTrue,	// Is resizable
		kODFalse,	// Is floating
		kODTrue,  // should save
		(ODPart*)somSelf /*_fPartWrapper*/, kODNullTypeToken, kODNullTypeToken, sourceFrame);


	return window;

}

SOM_Scope void  SOMLINK ClipShapeChanged(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

ODUnused(facet);
}

SOM_Scope void  SOMLINK ExternalTransformChanged(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

ODUnused(facet);
}

/*
	check our "mode";
		* if we're in Solve mode, figure out which piece of the puzzle was hit and do the puzzle-shifting-thing
		* if we're in EditInPlace mode, just pass the event along to the embedded facet...
*/
SOM_Scope ODBoolean  SOMLINK HandleMouseDown(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODPoint* where,
		ODEventData* event)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	POINT		mouse = facet->GetWindowContentTransform(ev, kODNULL)->InvertPoint(ev, where).AsWinPoint();


	if (!facet->GetWindow(ev)->IsActive(ev))
		facet->GetWindow(ev)->Select(ev);
	else
		somSelf->ActivateFrame(ev, facet->GetFrame(ev));

	{
		OrderedCollectionIterator i(_fSelection);
		Proxy* p = (Proxy*)i.First();
		
		Point winWhere = where->AsWinPoint();
		
		if (PtInRegion(_fCornerHandleRgn, mouse.x, mouse.y))
			return somSelf->HandleMouseDownCornerResize(ev, facet, p, &mouse);
		else if (PtInRegion(_fEdgeHandleRgn, mouse.x, mouse.y))
			return somSelf->HandleMouseDownEdgeResize(ev, facet, p, &mouse);
		else if (PtInRegion(_fSelectRgn, mouse.x, mouse.y))
			return somSelf->HandleMouseDownDrag(ev, facet, p, event);
		else
			switch (_fSelection->Count())
			{
				case 0:
					break;
			
				case 1:
					somSelf->InvalidateSelection(ev, facet->GetFrame(ev));
					somSelf->EmptySelection(ev);
					somSelf->ClipEmbeddedFrames(ev, facet->GetFrame(ev));
					break;
					
				default:
					// !!! handle multiple selection
					break;
			}
	}

	return kODTrue;
}

SOM_Scope ODBoolean  SOMLINK HandleMouseDownDrag(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		Proxy* selection,
		ODEventData* event)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	ODBoolean 		handled = kODFalse;
	ODPart* 		destPart;
	ODShape* 		scratch;
	ODPoint			translation;
	Point 			mdOffset = {0,0};
	ODPoint			tempPoint = translation;
	ODRgnHandle 	dragRgn, tempRgn;
	ODDragAndDrop*	dad;
	ODStorageUnit*	unit;
	ODDropResult 	dropResult;
	ODTransform* 	wfTrans = kODNULL;


	// Build the drag outline
	ODShape* frameShape = selection->frame->GetFrameShape(ev, kODNULL);
	scratch = frameShape->Copy(ev);
	frameShape->Release(ev); frameShape = kODNULL;
	scratch->Transform(ev, selection->transform);
	dragRgn = scratch->CopyWinRegion(ev);
	wfTrans = facet->GetWindowFrameTransform(ev, kODNULL);
	Point QDtranslation = wfTrans->TransformPoint(ev, &translation).AsWinPoint();
	wfTrans->Release(ev); wfTrans = kODNULL;
	OffsetRgn(dragRgn, QDtranslation.x, QDtranslation.y);
	tempRgn = ODNewRgn();
	CombineRgn (tempRgn, dragRgn, dragRgn, RGN_COPY);
	tempRgn = InflateRGN(tempRgn, -1);
	CombineRgn (dragRgn, dragRgn, tempRgn, RGN_DIFF);
	DeleteObject(tempRgn);
	scratch->Release(ev); scratch = kODNULL;
	
	//  Fill DAD SU with Props n' Vals
	
	dad = _fSession->GetDragAndDrop(ev);
	dad->Clear(ev);
	unit = dad->GetContentStorageUnit(ev);


	//  (optional) save offset between mousedown pt and topLeft pt of selection
	
	ODFrameFacetIterator* faceti = selection->frame->CreateFacetIterator(ev);
	wfTrans = faceti->First(ev)->GetWindowFrameTransform(ev, kODNULL);
	Point topLeft = wfTrans->GetWinOffset(ev);
	wfTrans->Release(ev); wfTrans = kODNULL;

	LONG2POINT(event->lParam, mdOffset);
	mdOffset.x -= topLeft.x;
	mdOffset.y -= topLeft.y;
	  
//bt	unit->AddProperty(ev, kPropMouseDownOffset);
//bt	unit->SetPromiseValue(ev, kWinPoint, 0, sizeof(Point), (ODValue) &mdOffset, somSelf);

	//  (optional) if dragging one frame, save its external transform (not aggregate)
	
	unit->AddProperty(ev, kODPropExternalTransform)->AddValue(ev, kWinPoint);
	unit->SetValue(ev, sizeof(Point), (ODValue) &topLeft);

	// WRite out the frame info in case it is dropped back to this part.
//bt	unit->AddProperty(ev, kPropFrameInfo);
//bt	unit->SetPromiseValue(ev, kODFrameRef, 0, sizeof(Proxy*), &selection, somSelf);
	

	// 
	
	dadSelection = selection;
	
	selection->frame->SetDragging(ev, kODTrue);
	
	dropResult = dad->StartDrag(ev, facet->GetFrame(ev), "RGNH",
			dragRgn, &destPart, event);
	
	dadSelection = kODNULL;
	
	if ( (dropResult == kODDropCopy) || (dropResult == kODDropMove) )
	{
		if (destPart)
		{
			// The frame was dropped in the same document
#ifdef LATER	//bt	
			if (destPart == _fPartWrapper)
#endif
			{
				// If this was a move to the same part
				facet->GetFrame(ev)->Invalidate(ev, kODNULL, kODNULL);
				selection->frame->SetDragging(ev, kODFalse);				
			}
#ifdef LATER	//bt	
			else if ((dropResult == kODDropMove) &&
					(destPart->GetStorageUnit(ev)->GetDraft(ev) == somSelf->GetStorageUnit(ev)->GetDraft(ev)))
			{
				// This was a move to a different part
				ODPart *oldPart = selection->frame->GetPart(ev);
				oldPart->IncrementRefCount(ev);
				facet->GetFrame(ev)->Invalidate(ev, kODNULL, kODNULL);
				somSelf->InvalidateSelection(ev, selection->frame);
				somSelf->EmptySelection(ev);
				
				OrderedCollection* tempFacets = new OrderedCollection;
				
				ODFrameFacetIterator* facets = selection->frame->CreateFacetIterator(ev);
				for (ODFacet* f1 = facets->First(ev);
						facets->IsNotComplete(ev);
						f1 = facets->Next(ev))
				{
					tempFacets->AddLast(f1);
				}
				delete facets;
				
				OrderedCollectionIterator t(tempFacets);
				for (ODFacet* f2 = (ODFacet*)t.First();
						t.IsNotComplete();
						f2 = (ODFacet*)t.Next())
				{
					f2->GetContainingFacet(ev)->RemoveFacet(ev, f2);
					ODDeleteObject(f2);
				}
				delete tempFacets;
				
				somSelf->RemoveEmbeddedFrame(ev, selection->frame);
//				somSelf->GetStorageUnit(ev)->GetDraft(ev)->RemovePart(ev, oldPart);
				somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);

				oldPart->Release(ev);
			}
		}
		else
		{
			// The frame was dropped in a different document
			selection->frame->SetDragging(ev, kODFalse);
#endif

BGNBP
#ifdef LATER
			/* For a move operation, remove ourselves from this container
			 */
			if (dad->GetDragAttributes(ev) & kODDragMove)
			{
				ODPart *oldPart = selection->frame->GetPart(ev);
				oldPart->IncrementRefCount(ev);
				facet->GetFrame(ev)->Invalidate(ev, kODNULL);
				somSelf->InvalidateSelection(ev, selection->frame);
				somSelf->EmptySelection(ev);

				OrderedCollection* tempFacets = new OrderedCollection;

				ODFrameFacetIterator* facets = selection->frame->CreateFacetIterator(ev);
				for (ODFacet* f1 = facets->First(ev);
						facets->IsNotComplete(ev);
						f1 = facets->Next(ev))
				{
					tempFacets->AddLast(f1);
				}
				delete facets;

				OrderedCollectionIterator t(tempFacets);
				for (ODFacet* f2 = (ODFacet*)t.First();
						t.IsNotComplete();
						f2 = (ODFacet*)t.Next())
				{
					f2->GetContainingFacet(ev)->RemoveFacet(ev, f2);
					ODDeleteObject(f2);
				}
				delete tempFacets;

				somSelf->RemoveEmbeddedFrame(ev, selection->frame);
  				somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);

				oldPart->Release(ev);
			}
#endif //LATER
		}
		
		handled = kODTrue;
	}
	else
	{
		selection->frame->SetDragging(ev, kODFalse);
	}
	DeleteObject(dragRgn);

	return handled;
}

SOM_Scope void  SOMLINK InvalEmbedFrameAfterResize(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		Proxy* selection,
		ODShape* oldShape,
		ODShape* newShape,
		Point* transOffset)
{
	 PuzzleData *somThis = PuzzleGetData(somSelf);

	// invalidate old frame shape
	ODShape* oldShapeCopy = oldShape->Copy(ev);
	oldShapeCopy->Transform(ev, selection->transform);
	facet->GetFrame(ev)->Invalidate(ev, oldShapeCopy, kODNULL);
	oldShapeCopy->Release(ev); oldShapeCopy = kODNULL;

	ODShape* newShapeCopy = newShape->Copy(ev);		// save to invalidate later

	selection->frame->ChangeFrameShape(ev, newShape, kODNULL);	// already in frame coords

	// if the left or top side of the frame was adjusted, the transform will be updated
	if ((transOffset->x != 0) || (transOffset->y != 0))
	{
		ODPoint ODtransOffset (*transOffset);
		selection->transform->MoveBy(ev, &ODtransOffset);

		// $$$$$ move facets for selected frame - this is slightly hackish
		ODTransform* newTrans;
		ODFacetIterator* facets =
			facet->GetWindow(ev)->GetRootFacet(ev)->CreateFacetIterator(ev, kODTopDown, kODFrontToBack);
		for (ODFacet* child = facets->First(ev);
				facets->IsNotComplete(ev); child = facets->Next(ev))
		{
			if (child->GetFrame(ev) == selection->frame)
			{
				newTrans = selection->transform->Copy(ev);
				child->ChangeGeometry(ev, kODNULL, newTrans, kODNULL);
				newTrans->Release(ev); newTrans = kODNULL;
				facets->SkipChildren(ev);
			}
		}
//DBGNBP		delete facets;
	}

	// invalidate changed areas
	newShapeCopy->Transform(ev, selection->transform);
	facet->GetFrame(ev)->Invalidate(ev, newShapeCopy, kODNULL);
	newShapeCopy->Release(ev); newShapeCopy = kODNULL;

	somSelf->InvalidateSelection(ev, facet->GetFrame(ev));
	somSelf->UpdateProxyRegion(ev, selection);
	somSelf->CreateSelectionBorder(ev, selection);
	somSelf->ClipEmbeddedFrames(ev, facet->GetFrame(ev));
	somSelf->InvalidateSelection(ev, facet->GetFrame(ev));
	
	somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);
}

SOM_Scope Rect*  SOMLINK GetSelectionRectLocal(Puzzle *somSelf, Environment *ev,
		Proxy* selection)
{
	 PuzzleData *somThis = PuzzleGetData(somSelf);

	Rect shapeRect;
	ODShape* frameShape = selection->frame->GetFrameShape(ev, kODNULL);
	RgnHandle shapeRgn = frameShape->GetWinRegion(ev);
	frameShape->Release(ev); frameShape = kODNULL;
	GetRgnBox (shapeRgn, &shapeRect);
	
	Rect* localRect = new Rect;
	SetRect(localRect, shapeRect.left, shapeRect.top,
						shapeRect.right, shapeRect.bottom);
	
	Point offset = selection->transform->GetWinOffset(ev);
	OffsetRect(localRect, offset.x, offset.y);
	
	return localRect;
}

SOM_Scope ODBoolean  SOMLINK HandleMouseDownEdgeResize(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		Proxy* selection,
		POINT* mouse)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	enum edge {kNone, kTE, kLE, kBE, kRE};
	edge growEdge = kNone;

	Rect* sr = somSelf->GetSelectionRectLocal(ev, selection);

	if		(mouse->y <= (sr->top + (0.1 * sr->top))) growEdge = kTE;
	else if (mouse->x <= (sr->left + (0.1 * sr->left))) growEdge = kLE;
	else if (mouse->y >= (sr->bottom - (0.1 * sr->bottom))) growEdge = kBE;
	else if (mouse->x >= (sr->right -  (0.1 * sr->right))) growEdge = kRE;

	HDC		hdc;
	HWND	hwnd;

	CFocusWindow f(facet, NULL, &hdc, &hwnd);

	Rect growBox = (*sr);
	Rect oldBox = {0,0,0,0};
	
	SetROP2(hdc, R2_NOT);
	SetBkMode (hdc, TRANSPARENT);
	HBRUSH hbrsh = GetStockObject(BLACK_BRUSH);
	
	Point theLoc = {0,0};
	Point lastLoc = {0,0};
	GetCursorPos (&lastLoc);
	ScreenToClient (hwnd, &lastLoc);
	Point origPt = lastLoc;
	MSG msg;

	SetCapture(hwnd);
	int inDrag = kODTrue;
	while (inDrag)
	{
		GetMessage(&msg, hwnd, WM_MOUSEFIRST, WM_MOUSELAST);
		switch(msg.message)
		{
			case WM_MOUSEMOVE:
#ifndef WIN32
				theLoc = MAKEPOINT(msg.lParam);
#else
				LONG2POINT(msg.lParam, theLoc);
#endif
				break;

			case WM_LBUTTONUP:
				inDrag = kODFalse;
				break;
		}

		if ((theLoc.x != lastLoc.x) || (theLoc.y != lastLoc.y))
		{
			switch (growEdge)
			{
				case kTE:
					growBox.top -= (lastLoc.y - theLoc.y);
					break;

				case kLE:
					growBox.left -= (lastLoc.x - theLoc.x);
					break;

				case kBE:
					growBox.bottom -= (lastLoc.y - theLoc.y);
					break;

				case kRE:
					growBox.right -= (lastLoc.x - theLoc.x);
					break;

				default:
					break;
			}

			DrawFocusRect (hdc, &oldBox);
			DrawFocusRect (hdc, &growBox);
		}

		lastLoc = theLoc;
		oldBox = growBox;
	}
	ReleaseCapture();

	DrawFocusRect (hdc, &oldBox);
	
	Point delta;
	delta.x = theLoc.x - origPt.x;
	delta.y = theLoc.y - origPt.y;
	Point transOffset = { 0, 0 };

	ODShape* oldShape = selection->frame->GetFrameShape(ev, kODNULL);
	ODRect newBounds;
	oldShape->GetBoundingBox(ev, &newBounds);

	switch (growEdge)
	{
		case kTE:
			newBounds.bottom -= ff(delta.y);
			transOffset.y += delta.y;
			break;

		case kLE:
			newBounds.right -= ff(delta.x);
			transOffset.x += delta.x;
			break;

		case kBE:
			newBounds.bottom += ff(delta.y);
			break;

		case kRE:
			newBounds.right += ff(delta.x);
			break;

		default:
			break;
	}
	
	ODShape* newShape = oldShape->NewShape(ev);
	newShape->SetRectangle(ev, &newBounds);
	
	somSelf->InvalEmbedFrameAfterResize(ev, facet, selection, oldShape, newShape, &transOffset);
	oldShape->Release(ev); oldShape = kODNULL;
	newShape->Release(ev); newShape = kODNULL;
	
	return kODTrue;
}

SOM_Scope ODBoolean  SOMLINK HandleMouseDownCornerResize(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		Proxy* selection,
		POINT* mouse)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	enum corner {kNone, kTL, kTR, kBL, kBR};
	corner growCorner = kNone;

	Rect* sr = somSelf->GetSelectionRectLocal(ev, selection);
	Point center = { (sr->right + sr->left) / 2, (sr->bottom + sr->top) / 2};		// center of FrameShape
	
	if		((mouse->y <= center.y) && (mouse->x <= center.x)) growCorner = kTL;
	else if ((mouse->y <  center.y) && (mouse->x >  center.x)) growCorner = kTR;
	else if ((mouse->y >  center.y) && (mouse->x <  center.x)) growCorner = kBL;
	else if ((mouse->y >  center.y) && (mouse->x >  center.x)) growCorner = kBR;
		
	HDC		hdc;
	HWND	hwnd;

	CFocusWindow f(facet, NULL, &hdc, &hwnd);

	Rect growBox = (*sr);
	Rect oldBox = {0,0,0,0};
	
	SetROP2(hdc, R2_NOT);
	SetBkMode (hdc, TRANSPARENT);
	HBRUSH hbrsh = GetStockObject(BLACK_BRUSH);

	Point theLoc = {0,0};
	Point lastLoc = {0,0};
	GetCursorPos (&lastLoc);
	ScreenToClient (hwnd, &lastLoc);
	Point origPt = lastLoc;
	MSG msg;
	  
	SetCapture(hwnd);
	int inDrag = kODTrue;
	while (inDrag)
	{
		GetMessage(&msg, hwnd, WM_MOUSEFIRST, WM_MOUSELAST);
		switch(msg.message)
		{
			case WM_MOUSEMOVE:
#ifndef WIN32
				theLoc = MAKEPOINT(msg.lParam);
#else
				LONG2POINT(msg.lParam, theLoc);
#endif
				break;

			case WM_LBUTTONUP:
#ifndef WIN32
				theLoc = MAKEPOINT(msg.lParam);
#else
				LONG2POINT(msg.lParam, theLoc);
#endif
				inDrag = kODFalse;
				break;
		}

		if ((theLoc.x != lastLoc.x) || (theLoc.y != lastLoc.y))
		{
			switch (growCorner)
			{
				case kTL:
					growBox.top -= (lastLoc.y - theLoc.y);
					growBox.left -= (lastLoc.x - theLoc.x);
					break;

				case kTR:
					growBox.top -= (lastLoc.y - theLoc.y);
					growBox.right -= (lastLoc.x - theLoc.x);
					break;

				case kBL:
					growBox.left -= (lastLoc.x - theLoc.x);
					growBox.bottom -= (lastLoc.y - theLoc.y);
					break;

				case kBR:
					growBox.bottom -= (lastLoc.y - theLoc.y);
					growBox.right -= (lastLoc.x - theLoc.x);
					break;

				default:
					break;
			}

			DrawFocusRect (hdc, &oldBox);
			DrawFocusRect (hdc, &growBox);
		}

		lastLoc = theLoc;
		oldBox = growBox;
	}
	ReleaseCapture();

	DrawFocusRect (hdc, &oldBox);

	Point delta;
	delta.x = theLoc.x - origPt.x;
	delta.y = theLoc.y - origPt.y;
	Point transOffset = { 0, 0 };

	ODShape* oldShape = selection->frame->GetFrameShape(ev, kODNULL);
	ODRect newBounds;
	oldShape->GetBoundingBox(ev, &newBounds);
	ODPoint fixDelta = delta;

	switch (growCorner)
	{
		case kTL:
			newBounds.bottom -= fixDelta.y; newBounds.right -= fixDelta.x;
			transOffset.y += delta.y; transOffset.x += delta.x;
			break;

		case kTR:
			newBounds.bottom -= fixDelta.y; newBounds.right += fixDelta.x;
			transOffset.y += delta.y;
			break;

		case kBL:
			newBounds.bottom += fixDelta.y; newBounds.right -= fixDelta.x;
			transOffset.x += delta.x;
			break;

		case kBR:
			newBounds.bottom += fixDelta.y;
			newBounds.right += fixDelta.x;
			break;

		default:
			break;
	}

	ODShape* newShape = oldShape->NewShape(ev);
	newShape->SetRectangle(ev, &newBounds);

	somSelf->InvalEmbedFrameAfterResize(ev, facet, selection, oldShape, newShape, &transOffset);
	oldShape->Release(ev); oldShape = kODNULL;
	newShape->Release(ev); newShape = kODNULL;
	
	return kODTrue;
}

SOM_Scope ODBoolean  SOMLINK HandleMenuEvent(Puzzle *somSelf, Environment *ev,
		ODFrame* focusFrame,
		ODEventData* event)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
 	
	long menuResult = event->message;
	ODBoolean handled = kODFalse;

	// Until such time as we clean this up and register all commands:
	switch (event->wParam)
	{
		case	IDM_ED_IN_PLACE:
			switch (_fMode)
			{
				case	kPuzzleEditInPlace:
					somSelf->FreezeFrames(ev, kODTrue);
					_fMode = kPuzzleSolve;
					break;
				case	kPuzzleSolve:
					somSelf->FreezeFrames(ev, kODFalse);
					_fMode = kPuzzleEditInPlace;
					break;
			}
			break;
		case	IDM_RE_SCRAMBLE : 
			somSelf->ScramblePuzzle(ev);
			break;
		case kODCommandAbout:
			handled = somSelf->DoAbout(ev);
			break;
		
		case kODCommandOpen :
			{
				Proxy* p;
				OrderedCollectionIterator i(_fSelection);
				for (p = (Proxy*) i.First(); i.IsNotComplete(); p = (Proxy*) i.Next())
				{	
					ODFrame* frame = p->frame;
					ODPart* part
							= frame ? frame->GetPart(ev) : (ODPart*)kODNULL;
					if (part)
						part->Open(ev, frame);
					handled = kODTrue;
				}
			}
			break;
			
		case kODCommandViewAsWin :
			somSelf->Open(ev, focusFrame);
			handled = kODTrue;
			break;

		default:
			break;
	}
			
	return handled;

}

//	Handle a click in an embedded facet
//	somePuzzleFacetInfo->fFacet
SOM_Scope ODBoolean  SOMLINK HandleMouseDownInPuzzlePiece(Puzzle *somSelf, Environment *ev,
		ODFacet* containingFacet,
		ODFacet* facetClickedIn)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

//	somSelf->ActivateFrame(ev, container->GetFrame(ev));
	
	PuzzleFacetInfo*				thisPieceInfo = kODNULL;
	POINT							newEmptyLocation;
	OrderedCollectionIterator		tempIterator(_fPuzzleFacets);
	PuzzleFacetInfo					*somePuzzleFacetInfo;
	ODFacet*						tempFacet;
	short							xOffset, yOffset;
	
ODUnused(containingFacet);
	
	//	find facetClickedIn in fPuzzleFacets
	for (somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.First();
			tempIterator.IsNotComplete() && (thisPieceInfo == kODNULL);
			somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.Next())
	{
		if (somePuzzleFacetInfo->fFacet == facetClickedIn)		//	PENDING: this may be a problem with SOM
			thisPieceInfo = somePuzzleFacetInfo;
	}
	
	if(!thisPieceInfo)
	{
		return kODFalse;
	}
	
	newEmptyLocation = thisPieceInfo->fGridLocation;

	//	figure out x & y offset values to give facets that need to move
	if (newEmptyLocation.x == _fEmptyPieceGridLocation.x)
	{
		xOffset = 0;
		if (newEmptyLocation.y < _fEmptyPieceGridLocation.y)
			yOffset = 1;
		else
			yOffset = -1;
	}
	else
	if (newEmptyLocation.y == _fEmptyPieceGridLocation.y)
	{
		yOffset = 0;
		if (newEmptyLocation.x < _fEmptyPieceGridLocation.x)
			xOffset = 1;
		else
			xOffset = -1;
	}
	else
		return kODFalse;

	//	note; if both horizontal and vertical coordinates are the same, we won't get this message
	//	because the click was not in the embedded frame

	//	need to iterate and find all the facets that have the same row/column as the one clicked in
	//	that are also between the clicked facet and the fEmptyPieceGridLocation
	//	move all these facets by xOffset, yOffset
	for (thisPieceInfo = (PuzzleFacetInfo*) tempIterator.First();
			tempIterator.IsNotComplete();
			thisPieceInfo = (PuzzleFacetInfo*) tempIterator.Next())
	{
		tempFacet = thisPieceInfo->fFacet;
		
		if (xOffset == 0)
		{
			if (thisPieceInfo->fGridLocation.x == newEmptyLocation.x)
			{
				if ((yOffset > 0) && (thisPieceInfo->fGridLocation.y >= newEmptyLocation.y)
						&& (thisPieceInfo->fGridLocation.y < _fEmptyPieceGridLocation.y))
					somSelf->MoveThisPuzzlePiece(ev, thisPieceInfo, tempFacet, xOffset, yOffset);
				else
				if ((yOffset < 0) && (thisPieceInfo->fGridLocation.y <= newEmptyLocation.y)
						&& (thisPieceInfo->fGridLocation.y > _fEmptyPieceGridLocation.y))
					somSelf->MoveThisPuzzlePiece(ev, thisPieceInfo, tempFacet, xOffset, yOffset);
			}
		}
		else
		{
			if (thisPieceInfo->fGridLocation.y == newEmptyLocation.y)
			{
				if ((xOffset > 0) && (thisPieceInfo->fGridLocation.x >= newEmptyLocation.x)
						&& (thisPieceInfo->fGridLocation.x < _fEmptyPieceGridLocation.x))
					somSelf->MoveThisPuzzlePiece(ev, thisPieceInfo, tempFacet, xOffset, yOffset);
				else
				if ((xOffset < 0) && (thisPieceInfo->fGridLocation.x <= newEmptyLocation.x)
						&& (thisPieceInfo->fGridLocation.x > _fEmptyPieceGridLocation.x))
					somSelf->MoveThisPuzzlePiece(ev, thisPieceInfo, tempFacet, xOffset, yOffset);
			}
		}
	}
	
	//	set new _fEmptyPieceGridLocation value
	_fEmptyPieceGridLocation = newEmptyLocation;
	
	return kODTrue;
}

SOM_Scope void  SOMLINK DoIdle(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    //	might want to do something...but probably not
}

SOM_Scope void  SOMLINK ActivateFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    ODBoolean		succeeded = kODFalse;
	myPartInfoRec*	pInfo = kODNULL;

	if ( frame != kODNULL)
	{
		pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
		
		if (!(pInfo->fIsActive))
		{
			succeeded = _fSession->GetArbitrator(ev)->RequestFocusSet(ev, _fFocusSet,frame);
					
			if (succeeded)
			{
				somSelf->FocusAcquired(ev, _fSelectionFocus, frame);
				somSelf->FocusAcquired(ev, _fMenuFocus, frame);
				somSelf->FocusAcquired(ev, _fKeyFocus, frame);
			}
		}
	}
}

SOM_Scope void  SOMLINK CommonInitPuzzle(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	_fDisplayFrames = new OrderedCollection;
	_fEmbeddedFrames = new OrderedCollection;
	_fPuzzleFacets = new OrderedCollection;

	_fContents = new OrderedCollection;
	_fSelection = new OrderedCollection;
	
	_fSelectRgn = ODNewRgn();
	_fCornerHandleRgn = ODNewRgn();
	_fEdgeHandleRgn = ODNewRgn();

	_fPuzzleMenu.menu = ::CreatePopupMenu(); 
	strcpy (_fPuzzleMenu.strMenu, "&Puzzle");
	
	_fSession = somSelf->GetStorageUnit(ev)->GetSession(ev);

	::AppendMenu(_fPuzzleMenu.menu,MF_STRING | MF_ENABLED, IDM_RE_SCRAMBLE,   "Re-Scramble");
	::AppendMenu(_fPuzzleMenu.menu,MF_STRING | MF_ENABLED, IDM_ED_IN_PLACE,   "Edit in place");
// later	::AppendMenu(_fPuzzleMenu.menu,MF_STRING | MF_ENABLED, IDM_CLICK_SOUND,   "Puzzle Click Sounds");
	
	_fMenuBar = _fSession->GetWindowState(ev)->CopyBaseMenuBar(ev);

	_fMenuBar->AddMenuLast(ev, (ODMenuID)"Puzzle", &_fPuzzleMenu, _fPartWrapper);

	_fSelectionFocus = _fSession->Tokenize(ev, kODSelectionFocus);
	_fMenuFocus = _fSession->Tokenize(ev, kODMenuFocus);
	_fKeyFocus = _fSession->Tokenize(ev, kODKeyFocus);
	
#if 0
	// The way it should be
	_fFocusSet = _fSession->GetArbitrator(ev)->CreateFocusSet(ev);
#else
	// The way it has to be until Richard implements the above:
	_fFocusSet = new ODFocusSet; _fFocusSet->InitFocusSet(ev);
#endif
	_fFocusSet->Add(ev, _fSelectionFocus);
	_fFocusSet->Add(ev, _fMenuFocus);
	_fFocusSet->Add(ev, _fKeyFocus);
		
	//	tokenize all the standard ViewAs constants so we can easily compare to them
	_fkODViewAsSmallIcon = _fSession->Tokenize(ev, kODViewAsSmallIcon);
	_fkODViewAsLargeIcon = _fSession->Tokenize(ev, kODViewAsLargeIcon);
	_fkODViewAsThumbnail = _fSession->Tokenize(ev, kODViewAsThumbnail);
	_fkODViewAsFrame = _fSession->Tokenize(ev, kODViewAsFrame);
	
}

SOM_Scope void  SOMLINK ActivatingWindow(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	myPartInfoRec* pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
	if (pInfo->fNeedsActivating)
	{
		somSelf->ActivateFrame(ev, frame);
		pInfo->fNeedsActivating = kODFalse;
	}
}

SOM_Scope void  SOMLINK DeActivatingWindow(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	myPartInfoRec* pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
	if (frame == _fSession->GetArbitrator(ev)->GetFocusOwner(ev, _fSelectionFocus))
	{
		pInfo->fNeedsActivating = kODTrue;
	}
	else
		pInfo->fNeedsActivating = kODFalse;
}

SOM_Scope void  SOMLINK InstallMenus(Puzzle *somSelf, Environment *ev,
		ODFrame* aFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if (aFrame && _fMenuBar)
		_fMenuBar->Display(ev);
}

SOM_Scope void  SOMLINK InvalidateSelection(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODShape* selectShape = frame->CreateShape(ev);
	CombineRgn(_fSelectRgn,_fCornerHandleRgn,_fSelectRgn, RGN_OR);
	ODHandle NewRgn = CreateRectRgn(0,0,0,0);
	CombineRgn (NewRgn, (ODHandle)_fSelectRgn, NewRgn, RGN_COPY);
	selectShape->SetWinRegion(ev, (ODRgnHandle)NewRgn);
	frame->Invalidate(ev, selectShape, kODNULL);
	selectShape ->Release(ev);
}

SOM_Scope void  SOMLINK EmptySelection(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	for(ODULong i = _fSelection->Count(); i > 0; i--)
	{
		Proxy* sp = (Proxy*) _fSelection->RemoveFirst();
		ODFrameFacetIterator* facets = sp->frame->CreateFacetIterator(ev);
		for (ODFacet* f1 = facets->First(ev); facets->IsNotComplete(ev); f1 = facets->Next(ev)) {
			if ( f1->IsSelected(ev) ) {
				f1->SetSelected(ev, kODFalse);
				f1->ChangeHighlight(ev, kODNoHighlight);
			}
		}
//DBGNBP		delete facets;
	}
	DeleteObject(_fSelectRgn);
	_fSelectRgn = ODNewRgn();
	DeleteObject(_fCornerHandleRgn);
	_fCornerHandleRgn = ODNewRgn();
	DeleteObject(_fEdgeHandleRgn);
	_fEdgeHandleRgn = ODNewRgn();
}

SOM_Scope void  SOMLINK ClipEmbeddedFrames(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODFrameFacetIterator* facets = frame->CreateFacetIterator(ev);
	for (ODFacet* facet = facets->First(ev); facets->IsNotComplete(ev);
			facet = facets->Next(ev))
		somSelf->ClipEmbeddedFacets(ev, facet);
//DBGNBP	delete facets;
}

SOM_Scope void  SOMLINK ClipEmbeddedFacets(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
}

SOM_Scope Proxy*  SOMLINK ProxyForFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if ( _fEmbeddedFrames->Contains(frame) )
	{
		Proxy* p;
		OrderedCollectionIterator i(_fContents);
		for (p = (Proxy*) i.First(); i.IsNotComplete(); p = (Proxy*) i.Next())
			if (p->frame == frame) return p;
		return kODNULL;
	}
	else
		THROW(kODErrInvalidFrame);

	return kODNULL;
}

SOM_Scope void  SOMLINK CreateSelectionBorder(Puzzle *somSelf, Environment *ev,
		Proxy* theProxy)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	#define		kBorderWidth		3
	#define		kHandleLengthThing	2
	
	// -- selection border region --

	// -- selection border region --

	ODRgnHandle border = ODNewRgn();
	CombineRgn (border, theProxy->region, theProxy->region, RGN_COPY);
	border = InflateRGN (border, kXMPBorderWidth);
	CombineRgn (border, border, theProxy->region, RGN_DIFF);

	RECT borderRect, tempRect;
	ODRgnHandle tempRgn;
	GetRgnBox (border, &borderRect);

	// -- corner region --

ODRgnHandle cornerHdlRgn = ODNewRgn();

	// top left handle
	CopyRect (&tempRect, &borderRect);
	tempRect.right = borderRect.left + (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRect.bottom = borderRect.top + kXMPBorderWidth;
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
	
	CopyRect (&tempRect, &borderRect);
	tempRect.right = borderRect.left + kXMPBorderWidth;
	tempRect.bottom = borderRect.top + (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
	
	// top right handle
	CopyRect (&tempRect, &borderRect);
	tempRect.left = borderRect.right - (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRect.bottom = borderRect.top + kXMPBorderWidth;
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
	
	CopyRect (&tempRect, &borderRect);
	tempRect.left = borderRect.right - kXMPBorderWidth;
	tempRect.bottom = borderRect.top + (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);

	// bottom right handle
	CopyRect (&tempRect, &borderRect);
	tempRect.left = borderRect.right - (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRect.top = borderRect.bottom - kXMPBorderWidth;
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
	
	CopyRect (&tempRect, &borderRect);
	tempRect.left = borderRect.right - kXMPBorderWidth;
	tempRect.top = borderRect.bottom - (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
							
	// bottom left handle
	CopyRect (&tempRect, &borderRect);
	tempRect.right = borderRect.left + (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRect.top = borderRect.bottom - kXMPBorderWidth;
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);
	
	CopyRect (&tempRect, &borderRect);
	tempRect.right = borderRect.left + kXMPBorderWidth;
	tempRect.top = borderRect.bottom - (kODHandleLenMultiplier * kXMPBorderWidth);
	tempRgn = CreateRectRgnIndirect (&tempRect);
	CombineRgn (cornerHdlRgn, cornerHdlRgn, tempRgn, RGN_OR);
	DeleteObject (tempRgn);

	// -- edge region --

	ODRgnHandle edgeHdlRgn = ODNewRgn();

	if ((borderRect.right - borderRect.left) >= (5 * kODHandleLenMultiplier * kXMPBorderWidth)) {
	
		short hcenter = (borderRect.right - borderRect.left) / 2;
		short hdist = (kODHandleLenMultiplier * kXMPBorderWidth) / 2;
		
		// top middle handle
		CopyRect (&tempRect, &borderRect);
		tempRect.left = borderRect.left + hcenter - hdist;
		tempRect.right = borderRect.left + hcenter + hdist;
		tempRect.bottom = borderRect.top + kXMPBorderWidth;
		tempRgn = CreateRectRgnIndirect (&tempRect);
		CombineRgn (edgeHdlRgn, edgeHdlRgn, tempRgn, RGN_OR);
		DeleteObject (tempRgn);
	
		// bottom middle handle
		CopyRect (&tempRect, &borderRect);
		tempRect.left = borderRect.left + hcenter - hdist;
		tempRect.right = borderRect.left + hcenter + hdist;
		tempRect.top = borderRect.bottom - kXMPBorderWidth;
		tempRgn = CreateRectRgnIndirect (&tempRect);
		CombineRgn (edgeHdlRgn, edgeHdlRgn, tempRgn, RGN_OR);
		DeleteObject (tempRgn);

	}
	
	if ((borderRect.bottom - borderRect.top) >= (5 * kODHandleLenMultiplier * kXMPBorderWidth)) {
	
		short vcenter = (borderRect.bottom - borderRect.top) / 2;
		short vdist = (kODHandleLenMultiplier * kXMPBorderWidth) / 2;
		
		// left middle handle
		CopyRect (&tempRect, &borderRect);
		tempRect.right = borderRect.left + kXMPBorderWidth;
		tempRect.top = borderRect.top + vcenter - vdist;
		tempRect.bottom = borderRect.top + vcenter + vdist;
		tempRgn = CreateRectRgnIndirect (&tempRect);
		CombineRgn (edgeHdlRgn, edgeHdlRgn, tempRgn, RGN_OR);
		DeleteObject (tempRgn);
	
		// right middle handle
		CopyRect (&tempRect, &borderRect);
		tempRect.left = borderRect.right - kXMPBorderWidth;
		tempRect.top = borderRect.top + vcenter - vdist;
		tempRect.bottom = borderRect.top + vcenter + vdist;
		tempRgn = CreateRectRgnIndirect (&tempRect);
		CombineRgn (edgeHdlRgn, edgeHdlRgn, tempRgn, RGN_OR);
		DeleteObject (tempRgn);
	}
	

	// clip by obscuring frames

	Proxy* q;
	OrderedCollectionIterator i(_fContents);
	ODBoolean above = TRUE;
	for (q = (Proxy*) i.First(); i.IsNotComplete(); q = (Proxy*) i.Next())
	{	if (theProxy == q) {above = FALSE; continue;}
		if (!above) {continue;}
		CombineRgn (border, border, q->region, RGN_DIFF);
		CombineRgn (cornerHdlRgn, cornerHdlRgn, q->region, RGN_DIFF);
		CombineRgn (edgeHdlRgn, edgeHdlRgn, q->region, RGN_DIFF);
	};


	DeleteObject(_fCornerHandleRgn);
	_fCornerHandleRgn = cornerHdlRgn;
	CombineRgn (border, border, cornerHdlRgn, RGN_OR);

	DeleteObject(_fEdgeHandleRgn);
	_fEdgeHandleRgn = edgeHdlRgn;
	CombineRgn (border, border, edgeHdlRgn, RGN_OR);
	
	DeleteObject(_fSelectRgn);
	_fSelectRgn = border;
	
	// clip path from obscured frames
	somSelf->ClipEmbeddedFrames(ev, theProxy->frame->GetContainingFrame(ev));
}

SOM_Scope void  SOMLINK UpdateProxyRegion(Puzzle *somSelf, Environment *ev,
		Proxy* proxy)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODShape* frameShape = proxy->frame->GetFrameShape(ev, kODNULL);
	ODShape* scratch = frameShape->Copy(ev);
	scratch->Transform(ev, proxy->transform);
	ODRgnHandle scratchRgn = scratch->GetWinRegion(ev);
	CombineRgn (proxy->region, scratchRgn, scratchRgn, RGN_COPY);
	frameShape->Release(ev);
	scratch->Release(ev);
}

SOM_Scope void  SOMLINK DrawPuzzlePartText(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	char				*someText = " Puzzle \r\r Drop part \r Here! ";
	RECT				tempRect;

	ODFrame*			displayFrame = facet->GetFrame(ev);
	ODRgnHandle 		saveClip = ODNewRgn();
	ODShape* 			clipShape = kODNULL;
	ODShape*			tempShape = kODNULL;

	HDC hdc;
	CFocus	f(facet, kODNULL, &hdc);

	
	tempShape = displayFrame->GetFrameShape(ev, kODNULL);
	GetRgnBox (tempShape->GetWinRegion(ev), &tempRect);
	tempShape->Release(ev);

	LOGFONT logfont;
	GetObject (GetStockObject(SYSTEM_FONT), sizeof(LOGFONT), &logfont);
	logfont.lfHeight                = -12;
	logfont.lfWidth                 = 0;
	logfont.lfWeight                = FW_NORMAL;
	lstrcpy(logfont.lfFaceName,"Arial");
	HFONT horigfont, hFont = CreateFontIndirect(&logfont);
	horigfont = SelectObject(hdc,hFont);
	
	::InflateRect(&tempRect, -3, -5);
	::DrawText(hdc, someText, strlen(someText), &tempRect, DT_CENTER);

}

SOM_Scope void  SOMLINK ToggleHilite(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODFrame* 		frame = facet->GetFrame(ev);
	ODRgnHandle 	saveClip = ODNewRgn();
	ODShape* 		frameShape = kODNULL;
	ODRgnHandle		frameArea;
	int				iResult;
	
	HDC hdc;

	CFocus	f(facet, kODNULL, &hdc);
	

	//	get the area to frame
	frameShape = frame->GetFrameShape(ev, kODNULL);
	frameArea = frameShape->CopyWinRegion(ev);
	ODRgnHandle scratchRgn = frameShape->CopyWinRegion(ev);
	scratchRgn = InflateRGN (scratchRgn, 2);
	frameArea = InflateRGN (frameArea, 1);
	CombineRgn(frameArea, scratchRgn, frameArea, RGN_DIFF);
	DeleteObject(scratchRgn);

#ifndef WIN32
			scratchRgn = GetClipRegion (hdc);
#else
			scratchRgn = CreateRectRgn (0,0,1,1);
			iResult = GetClipRgn (hdc, scratchRgn);
#endif //WIN32

	if (iResult)
		SelectClipRgn (hdc, frameArea);

	//	do hilite:
	{
		::FillRgn(hdc, frameArea, GetStockObject (BLACK_BRUSH));
	}
	
	//	restore original region.
	if (scratchRgn)
		SelectClipRgn (hdc, scratchRgn);

	
	frameShape->Release(ev);
}

//	Don't make sure its width is divisible by four pixels, because FrameShapeChanged takes care of that
//	PENDING: kill this routine (but we likely need a KillEmbeddedFacets)
SOM_Scope ODShape*  SOMLINK CalcEmbeddedPartShape(Puzzle *somSelf, Environment *ev,
		ODFrame* parentFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    	
	ODRect		tempRect;
	ODShape*	frameShape = kODNULL;
	ODShape*	newShape = kODNULL;

	//	make newShape the same size as this part's frame less a couple pixels all the way around
	frameShape = parentFrame->GetFrameShape(ev, kODNULL);												// increments frameshape's refcount
	newShape = frameShape->Copy(ev);
	newShape->GetBoundingBox(ev, &tempRect);
	tempRect.Inset(IntToFixed(kPuzzleEmbeededPartBorder), IntToFixed(kPuzzleEmbeededPartBorder));

	newShape->SetRectangle(ev, &tempRect);
	
	frameShape->Release(ev);
	frameShape = kODNULL;


	return newShape;
}

SOM_Scope ODFrame*  SOMLINK CreateAnEmbeddedFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* containingFrame,
		ODPart* embedPart,
		ODTypeToken viewType,
		ODTypeToken presentation,
		ODBoolean isOverlaid)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	somPrintf("In Puzzle::CreateAnEmbeddedFrame\r");
	
	ODShape*		newShape;
	ODFrame*		newFrame;
	ODRgnHandle		tempRgn;
	ODTransform* 	externalTransform;
	
	//	get new shape
	newShape = somSelf->CalcEmbeddedPartShape(ev, containingFrame);
	if (!newShape)
		return(kODNULL);
		
	// create the new frame
	// CreateFrame increments newShape's refcount
	newFrame = somSelf->GetStorageUnit(ev)->GetDraft(ev)->
						CreateFrame(ev,
							kODNULL,								// frame type
							containingFrame, 						// containing frame
							newShape, 								// frame shape
							kODNULL,								// bias canvas
							embedPart,								// part							
							viewType, 								// view type
							presentation, 							// presentation
							kODFalse, 								// is root
							isOverlaid);							
						
	_fEmbeddedFrames->AddFirst(newFrame);
	newFrame->SetFrozen(ev, kODTrue);
	//newFrame->SetPropagateEvents(ev, kODTrue);
	_fMode = kPuzzleSolve;
	
	//	create a proxy to hold the embedded frame
	//	can't use newShape for the below since we'd have to change it
	tempRgn = newShape->CopyWinRegion(ev);
	
	Proxy* p = new Proxy(tempRgn, newFrame, externalTransform);			// Proxy Contructor consumes tempRgn
	
	//	put proxy into contents, possibly adjusting other content
	_fContents->AddFirst(p);

	//	get rid of temporary objects
	newShape->Release(ev);
	newShape = kODNULL;
	
	return newFrame;
}

/*
	Create all the puzzle pieces
	1st, have to split the given frame into 16 rectangles
	Note: bottom right one is left blank, so we only create 15 facets
	inset each rectangle by 1, 1 so we can draw a frame around each puzzle piece
	add 15 facets to the given frame using these clip rectangles and offsets
	Note: randomizing is done in a seperate routine
*/
SOM_Scope void  SOMLINK CreatePuzzleFacets(Puzzle *somSelf, Environment *ev,
		ODFacet* facetDroppedInto,
		ODFrame* newFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODShape				*clipShape, *basePuzzleShape;
	ODShape				*newActiveShape;
	ODTransform			*newExternalXForm;
	POINT				myPoint = {0, 0};
	ODPoint				funkyOffset(myPoint);
	ODRect				bounds;
	RECT				tempRect, basePuzzleRect;
	short				eachPieceWidth, eachPieceHeight;
	short				xIndex, yIndex;
	PuzzleFacetInfo*	thisPieceInfo;
	ODFacet*			thisPieceFacet;
	
	basePuzzleShape = newFrame->GetFrameShape(ev, kODNULL);					// Increments ref count
	basePuzzleShape->GetBoundingBox(ev, &bounds);

	bounds.AsWinRect(basePuzzleRect);

	eachPieceHeight = ((basePuzzleRect.bottom - basePuzzleRect.top) / 4) - 2;
	eachPieceWidth = ((basePuzzleRect.right - basePuzzleRect.left) / 4) - 2;
	::SetRect(&basePuzzleRect, basePuzzleRect.left + 1, basePuzzleRect.top + 1, basePuzzleRect.left + eachPieceWidth, basePuzzleRect.top + eachPieceHeight);

	for (xIndex = 0; xIndex < 4; xIndex++)
	for (yIndex = 0; yIndex < 4; yIndex++)
	{
		if ((yIndex != 3) || (xIndex != 3))	//	don't create the bottom right corner
		{
			tempRect = basePuzzleRect;
			::OffsetRect(&tempRect, xIndex * (eachPieceWidth + 2), yIndex * (eachPieceHeight + 2));
	
			clipShape = basePuzzleShape->Copy(ev);
			bounds = tempRect;
			clipShape->SetRectangle(ev, &bounds);
			
			newExternalXForm = newFrame->CreateTransform(ev);
			newExternalXForm->SetOffset(ev, &funkyOffset);
			thisPieceFacet = facetDroppedInto->CreateEmbeddedFacet(
							ev, 					
							newFrame, 					
							clipShape, 
							newExternalXForm, 
							kODNULL, 							// canvas
							kODNULL, 							// bias canvas
							kODNULL, 							// sibling facet
							kODFrameInFront);					// frame position
			
			newExternalXForm->Release(ev);
			newExternalXForm = kODNULL;
			
			// change active shape to be the shape of the facet not the frame
			newActiveShape = clipShape->Copy(ev);
			thisPieceFacet->ChangeActiveShape(ev, newActiveShape, kODNULL);
			
			clipShape->Release(ev);								// Ref count incremented by CreateEmbeddedFacet
			clipShape = kODNULL;
			
			newActiveShape->Release(ev);						// Ref count incremented by ChangeActiveShape
			newActiveShape = kODNULL;
			
			//	hang my user info off of the frame's fPartInfo
			thisPieceInfo = new PuzzleFacetInfo;
			thisPieceInfo->fGridLocation.x = xIndex;
			thisPieceInfo->fGridLocation.y = yIndex;
			thisPieceInfo->fOriginalGridLocation = thisPieceInfo->fGridLocation;
			thisPieceInfo->fWidth = eachPieceWidth + 2;
			thisPieceInfo->fHeight = eachPieceHeight + 2;
			thisPieceInfo->fFacet = thisPieceFacet;

			_fPuzzleFacets->AddLast((ElementType) thisPieceInfo);
		}
	}

	_fEmptyPieceGridLocation.x = 3;
	_fEmptyPieceGridLocation.y = 3;
	
	basePuzzleShape->Release(ev);
}

SOM_Scope void  SOMLINK FreezeFrames(Puzzle *somSelf, Environment *ev,
		ODBoolean YeahOrNay)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	OrderedCollectionIterator		tempIterator(_fEmbeddedFrames);
	ODFrame*						embeddedFrame;
	
	for (embeddedFrame = (ODFrame*) tempIterator.First();
			tempIterator.IsNotComplete();
			embeddedFrame = (ODFrame*) tempIterator.Next())
	{
		embeddedFrame->SetFrozen(ev, YeahOrNay);
	}
}

//	iterate a hundred times or so & generate random moves...
//	this should effectively scramble the puzzle
SOM_Scope void  SOMLINK ScramblePuzzle(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	PuzzleFacetInfo*				thisPieceInfo = kODNULL;
	OrderedCollectionIterator		tempIterator(_fPuzzleFacets);
	PuzzleFacetInfo					*somePuzzleFacetInfo;
	ODFacet*						tempFacet;
	POINT							gridClickedIn;
	short							numTimesToDoThis = 101;
	
	while (numTimesToDoThis-- > 0)
	{
		//	randomly pick a grid position to click in -- note that given any particular _fEmptyPieceGridLocation,
		//	there are only 6 positions to choose from that are legal moves; 3 vertically and 3 horizontally
		
		gridClickedIn = _fEmptyPieceGridLocation;
		if ((rand() % 2) == 0)	//	50/50 chance...? Does Random return an unsigned?
		{	//	pick a vertically aligned piece
			while(gridClickedIn.y == _fEmptyPieceGridLocation.x)
				gridClickedIn.y = rand() % 4;
			
			if (gridClickedIn.y < 0)
				gridClickedIn.y *= -1;	//	just to make sure rand() is not returning a signed int
		}
		else
		{	//	pick a horizontally aligned piece
			while(gridClickedIn.x == _fEmptyPieceGridLocation.x)
				gridClickedIn.x = rand() % 4;
			
			if (gridClickedIn.x < 0)
				gridClickedIn.x *= -1;	//	just to make sure rand() is not returning a signed int
		}
		
		tempFacet = kODNULL;
		//	find which facet this is and pass it to HandleMouseDownInPuzzlePiece
		for (somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.First();
				tempIterator.IsNotComplete() && (tempFacet == kODNULL);
				somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.Next())
		{
			if ((somePuzzleFacetInfo->fGridLocation.x == gridClickedIn.x) &&
				(somePuzzleFacetInfo->fGridLocation.y == gridClickedIn.y))
				tempFacet = somePuzzleFacetInfo->fFacet;
		}
		
		if (tempFacet != kODNULL)		//	if it does, there's a something wrong somewhere!
		{
			somSelf->HandleMouseDownInPuzzlePiece(ev, kODNULL,  tempFacet);
		}
	}
}

//	check all the puzzlepieces' externaltransforms; if they're all {0,0} then the puzzle is solved!
SOM_Scope ODBoolean  SOMLINK IsPuzzleSolved(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	OrderedCollectionIterator		tempIterator(_fPuzzleFacets);
	PuzzleFacetInfo					*somePuzzleFacetInfo;
	ODTransform						*tempXForm;
	ODPoint							offset(0,0);
	ODBoolean						isSolved = kODTrue;
	
	for (somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.First();
			tempIterator.IsNotComplete() && (isSolved == kODTrue);
			somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.Next())
	{
		tempXForm = somePuzzleFacetInfo->fFacet->GetExternalTransform(ev, kODNULL);				// Increments refcount
		tempXForm->GetOffset(ev, &offset);	
		
		if ((offset.IntX() != 0) || (offset.IntY() != 0))
			isSolved = kODFalse;
			
		tempXForm->Release(ev);
	}

	return(isSolved);
}

//	xAmount and yAmount should be in the range -1..1 -- one at a time!
SOM_Scope void  SOMLINK MoveThisPuzzlePiece(Puzzle *somSelf, Environment *ev,
		PuzzleFacetInfo* thisPieceInfo,
		ODFacet* thisPiece,
		ODSShort xAmount,
		ODSShort yAmount)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODTransform*					tempXForm;
	ODTransform*					externXForm;
	ODPoint							tempPoint;
	POINT							myPoint;
	ODShape*						tempShape;
	ODShape*						clipShape;
	ODRect							tempRect;
	OrderedCollectionIterator		iterator(_fDisplayFrames);
	ODFrame*						frameToInform;
	
	if ((xAmount != 0) && (yAmount != 0))
		return;	//	bad input!

	//	this assumes only one display frame...may be erroneous
	frameToInform = (ODFrame*) iterator.First();	//	thisPiece->GetFrame(ev);
	
	//	invalidate old area; remove when something better gets devised
	clipShape = thisPiece->GetClipShape(ev, kODNULL);	// increments refcount of clipshape
	
	tempShape = clipShape->Copy(ev);
	tempShape->GetBoundingBox(ev, &tempRect);
	tempRect.Inset(IntToFixed(-1), IntToFixed(-1));		//	so bordering rectangles will get redrawn
	clipShape->Release(ev);
	clipShape = kODNULL;

	externXForm = thisPiece->GetExternalTransform(ev, kODNULL);
	tempXForm = externXForm->Copy(ev);

	externXForm->Release(ev);
	externXForm = kODNULL;

	tempXForm->GetOffset(ev, &tempPoint);
	tempRect.Offset(tempPoint);
	tempXForm->Release(ev);
	tempXForm = kODNULL;

	tempShape->SetRectangle(ev, &tempRect);
	frameToInform->Invalidate(ev, tempShape, kODNULL);					// increments refcount of tempShape
	tempShape->Release(ev);
	tempShape = kODNULL;
	//	end of invalidating old area

	//	move facet to new location
	myPoint.x = xAmount * thisPieceInfo->fWidth;
	myPoint.y = yAmount * thisPieceInfo->fHeight;
	tempPoint = myPoint;
	
	externXForm = thisPiece->GetExternalTransform(ev, kODNULL);			// external transform refcount is incremented
	tempXForm = externXForm->Copy(ev);
	externXForm->Release(ev);
	externXForm = kODNULL;

	tempXForm->MoveBy(ev, &tempPoint);
	thisPiece->ChangeGeometry(ev, kODNULL, tempXForm, kODNULL);				//	tempXForm refcount is incremented
	tempXForm->Release(ev);
	tempXForm = kODNULL;
	
	thisPieceInfo->fGridLocation.x += xAmount;
	thisPieceInfo->fGridLocation.y += yAmount;
	//	end of moving to new location
	
	//	update on-screen visual; for now, just invalidate it
	clipShape = thisPiece->GetClipShape(ev, kODNULL);						// increments refcount of clip shape
	tempShape = clipShape->Copy(ev);												
	clipShape->Release(ev);
	clipShape = kODNULL;
	
	tempShape->GetBoundingBox(ev, &tempRect);
	tempRect.Inset(IntToFixed(-1), IntToFixed(-1));							//	so bordering rectangles will get redrawn

	externXForm = thisPiece->GetExternalTransform(ev, kODNULL);				// increments refcount of external transform
	tempXForm = externXForm->Copy(ev);
	externXForm->Release(ev);
	externXForm = kODNULL;

	tempXForm->GetOffset(ev, &tempPoint);
	tempRect.Offset(tempPoint);
	tempXForm->Release(ev);
	tempXForm = kODNULL;

	tempShape->SetRectangle(ev, &tempRect);
	frameToInform->Invalidate(ev, tempShape, kODNULL);						// Increments refcount of invalid shape
	//	end of invalidating new area
	
	tempShape->Release(ev);

}

SOM_Scope void  SOMLINK DoPuzzleSolvedThing(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
}

SOM_Scope void  SOMLINK PlayAPuzzleSound(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
}

SOM_Scope ODBoolean  SOMLINK DoAbout(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	return(kODFalse);
}

SOM_Scope void  SOMLINK somInit(Puzzle *somSelf)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	_fDisplayFrames = kODNULL;
	_fEmbeddedFrames = kODNULL;
	_fPuzzleFacets = kODNULL;
	
	_fFrameGroupIDCounter = 1;
	
	_fContents = kODNULL;
	_fSelection = kODNULL;
	
	_fSelectRgn = kODNULL;
	_fCornerHandleRgn = kODNULL;
	_fEdgeHandleRgn = kODNULL;
	
	_fPuzzleMenu.menu = kODNULL;
	_fMenuBar = kODNULL;
	_fFocusSet = kODNULL;	
	
	_fSession = kODNULL;
	_fPartWrapper = kODNULL;
	
	_fDraggingAcceptableItem = kODFalse;
	_fAlreadyContainAPart = kODFalse;
	_fDoClickSounds = kODFalse; //bt kODTrue;
	
	_fMode = kPuzzleEditInPlace;
}

SOM_Scope void  SOMLINK somUninit(Puzzle *somSelf)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	Environment* ev = somGetGlobalEnvironment();

	if (_fContents != kODNULL)
		delete _fContents;
	if (_fSelection != kODNULL)
		delete _fSelection;
	if (_fSelectRgn) DeleteObject(_fSelectRgn);
	if (_fCornerHandleRgn) DeleteObject(_fCornerHandleRgn);
	if (_fEdgeHandleRgn) DeleteObject(_fEdgeHandleRgn);
	
	if (_fMenuBar != kODNULL) {	
		if (_fMenuBar->GetRefCount(ev) <= 1) 
		{	
			if (_fPuzzleMenu.menu) 
			{
				_fMenuBar->RemoveMenu(ev, IDM_RE_SCRAMBLE);
				_fMenuBar->RemoveMenu(ev, IDM_ED_IN_PLACE);
//				_fMenuBar->RemoveMenu(ev, IDM_CLICK_SOUND);
				DestroyMenu (_fPuzzleMenu.menu);
			}
		}

		_fMenuBar->Release(ev);
	}     
	if (_fMenuBar != kODNULL)
	{
	}
	

	if (_fDisplayFrames != kODNULL)
		delete _fDisplayFrames;		// make sure it's empty first
	if (_fEmbeddedFrames != kODNULL)
		delete _fEmbeddedFrames;
	if (_fPuzzleFacets != kODNULL)
		delete _fPuzzleFacets;

}

//	save our stuff out
//	PENDING: should also externalize weak references to embedded part
//	PENDING: error checking
SOM_Scope void  SOMLINK Externalize(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODStorageUnit*		partSU = somSelf->GetStorageUnit(ev);
	ODStorageUnitRef	aSURef;
	ODStorageUnit*		conSU;
	ODULong				count;
	ODSLong				oldSize;
	ODSLong				newSize;

	Puzzle_parent_ODPart_Externalize(somSelf,ev);
	
	//	write out display frame(s)
	partSU->Focus(ev, kODPropContents, kODPosUndefined, kODKindPuzzle, 0, kODPosUndefined);
	partSU->GetValue(ev, sizeof(ODStorageUnitRef),&aSURef);
	
	conSU = partSU->GetDraft(ev)->GetStorageUnit(ev, partSU->GetIDFromStorageUnitRef(ev, aSURef));

	//	write out display frame(s)
	conSU->Focus(ev, kODPropDisplayFrames,kODPosUndefined,0,1,kODPosFirstSib);

	count = _fDisplayFrames->Count();
	oldSize = conSU->GetSize(ev);
	newSize = (count + 1) * sizeof(ODULong);
	
	if (oldSize != newSize)
		conSU->DeleteValue(ev, oldSize);

	conSU->SetValue(ev, sizeof(count), (ODValue)&count);

	OrderedCollectionIterator		iterator(_fDisplayFrames);
	for (ODFrame* frame = (ODFrame*) iterator.First(); iterator.IsNotComplete(); frame = (ODFrame*) iterator.Next())
	{		
		//	Get a weak reference so the frame won't be cloned when the part is
		aSURef = conSU->GetWeakStorageUnitRef(ev, frame->GetStorageUnit(ev)->GetID(ev));
		conSU->SetValue(ev, sizeof(ODStorageUnitRef), (ODValue)&aSURef);
	}

#ifdef NOT_YET 	// Store embedded Part
	if (_fAlreadyContainAPart && _fContents) {
		OrderedCollectionIterator cIter(_fContents);
		Proxy *proxy = (Proxy*)cIter.First();
		proxy->frame->Externalize(ev);
		conSU->Focus(ev, kODPropContents,kODPosUndefined,0,1,kODPosFirstSib);
		conSU->AddProperty(ev, kODPropContents);
		conSU->AddValue(ev, kODStrongStorageUnitRef);
		aSURef = conSU->GetStrongStorageUnitRef(ev, proxy->frame->GetStorageUnit(ev)->GetID(ev));
		conSU->SetValue(ev, sizeof(ODStorageUnitRef), (ODValue)&aSURef);
	}
#endif

	conSU->Release(ev);
}

SOM_Scope void  SOMLINK Release(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    Puzzle_parent_ODPart_Release(somSelf,ev);
    
	Proxy*		p;
	ODFrame*	frame;

	// NOTE: This has not been fully implemented

	//somPrintf("Release not fully implemented \r");

	if (somSelf->GetRefCount(ev) == 0) {

		frame = (ODFrame*) _fDisplayFrames->First();
		while (frame != kODNULL) {
			_fDisplayFrames->Remove(frame);
			ODReleaseObject(ev,frame);
			frame = (ODFrame*) _fDisplayFrames->First();
		}

		frame = (ODFrame*) _fEmbeddedFrames->First();
		while (frame != kODNULL) {
			p = somSelf->ProxyForFrame(ev, frame);
			_fContents->Remove(p);
			_fEmbeddedFrames->Remove(frame);
			ODReleaseObject(ev,frame);
			frame = (ODFrame*) _fEmbeddedFrames->First();
		}
		somSelf->GetStorageUnit(ev)->GetDraft(ev)->ReleasePart(ev, _fPartWrapper);
	}
}

SOM_Scope void  SOMLINK ReleaseAll(Puzzle *somSelf, Environment *ev)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    Puzzle_parent_ODPart_ReleaseAll(somSelf,ev);
}

SOM_Scope ODBoolean  SOMLINK HasExtension(Puzzle *somSelf, Environment *ev,
		ODType extensionName)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    return (Puzzle_parent_ODPart_HasExtension(somSelf,ev,extensionName));
}

SOM_Scope ODExtension*  SOMLINK GetExtension(Puzzle *somSelf, Environment *ev,
		ODType extensionName)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
    return (Puzzle_parent_ODPart_GetExtension(somSelf,ev,extensionName));
}

SOM_Scope void  SOMLINK FulfillPromise(Puzzle *somSelf, Environment *ev,
		ODStorageUnitView* promiseSUView)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

    Puzzle_parent_ODPart_FulfillPromise(somSelf,ev,promiseSUView);
}

SOM_Scope void  SOMLINK DropCompleted(Puzzle *somSelf, Environment *ev,
		ODPart* destPart,
		ODDropResult dropResult)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(destPart);
ODUnused(dropResult);
}

/*	PuzzlePart will accept any other part, so the function of DragEnter is to :
	* determine if the thing being dragged in is a part
	* make sure a part is not already embedded
	* set the _fDraggingAcceptableItem flag accordingly
	* adorn the frame if item being dragged is droppable
*/
SOM_Scope ODDragResult  SOMLINK DragEnter(Puzzle *somSelf, Environment *ev,
		ODDragItemIterator* dragInfo,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODStorageUnit			*dropSU;
	
ODUnused(dragInfo);
ODUnused(where);
	
//	make sure dragged item is acceptable; if so, give user feedback by hiliting

	dropSU = dragInfo->First(ev);
	if ((dropSU->Exists(ev, kODPropPart, kODISOStr, 0)) && (!_fAlreadyContainAPart))
		_fDraggingAcceptableItem = kODTrue;
	else
		_fDraggingAcceptableItem = kODFalse;

	if (_fDraggingAcceptableItem)
	{
		somSelf->ToggleHilite(ev, facet);
	}
    return kODTrue;
}

SOM_Scope ODDragResult  SOMLINK DragWithin(Puzzle *somSelf, Environment *ev,
		ODDragItemIterator* dragInfo,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(dragInfo);
ODUnused(facet);
ODUnused(where);

    return kODTrue;
}

SOM_Scope void  SOMLINK DragLeave(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(where);

	if (_fDraggingAcceptableItem)	//	if the item wasn't one we'd accept, don't unhilite
	{
		somSelf->ToggleHilite(ev, facet);
	}
}

SOM_Scope ODDropResult  SOMLINK Drop(Puzzle *somSelf, Environment *ev,
		ODDragItemIterator* dropInfo,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	//	kill currently embedded part and everything that goes with it...?
	//	accept 1st new part being dragged
	//	resize it, break it into 16 pieces, scramble them, give part 15 facets
	ODDraft					*theDraft = somSelf->GetStorageUnit(ev)->GetDraft(ev);
	ODULong					attributes;
	ODULong        			keystate;
	short					movePart;
	ODStorageUnit			*dropSU, *newSU;
	ODID					newSUID;
	ODFrame					*frameDroppedIn;
	ODFrame					*newFrame = kODNULL;
	ODPart 					*newPart;
	
	//	check to see if we can handle drop -- if more than one item is being dragged in, ignore extra parts
	dropSU = dropInfo->First(ev);
	if (!(dropSU->Exists(ev, kODPropPart, kODISOStr, 0)))
		return (kODDropFail);
	
	//	first, check to see if we already have an embedded part
	//	if so, tell the user to get rid of the old one first!

	if (_fAlreadyContainAPart)
	{
		return (kODDropFail);
	}

	ODDropResult	dropResult = kODDropMove;

	// Get the attributes for this drag
	attributes = _fSession->GetDragAndDrop(ev)->GetDragAttributes(ev);
	keystate = _fSession->GetDragAndDrop(ev)->GetDropKeyState(ev);
	movePart = (attributes & kODdragIsInSourceFrame) && ((keystate & 0x00008000) == 0);
	dropSU = dropInfo->First(ev);		//	ignore all but the first item dragged
	{
		if (movePart)
		{
			return (kODDropFail);	//	Shouldn't be moving the part around! It needs to be stuck in one spot.
		}
		else
		{
			ODDraft*	fromDraft = dropSU->GetDraft(ev);
			ODID		newFrameID = 0;
			ODDraftKey	key;
			ODCloneKind	cloneKind;
			
			if ((keystate & 0x8000) == 0){
				cloneKind = kODCloneDropMove;
				dropResult = kODDropMove;
			}else{
				cloneKind = kODCloneDropCopy;
				dropResult = kODDropCopy;
			}  

			TRY
				key = dropSU->GetDraft(ev)->BeginClone(ev, theDraft, cloneKind);						// Am I trying to CloneInto theDraft?			
				newSUID = dropSU->GetDraft(ev)->Clone(ev, key, dropSU->GetID(ev), kODNULL, kODNULL);			// was dropSU->CloneTo(ev, key, theDraft, kODNULL)
				if (dropSU->Exists(ev, kODPropContentFrame, kODWeakStorageUnitRef, 0))
				{
					ODStorageUnitRef aSURef;
					dropSU->Focus(ev, kODPropContentFrame, kODPosSame, kODWeakStorageUnitRef, 0, kODPosUndefined);
					dropSU->GetValue(ev, sizeof(ODStorageUnitRef), &aSURef);
					if ( dropSU->IsValidStorageUnitRef(ev, aSURef) )
						newFrameID = fromDraft->Clone(ev, key, dropSU->GetIDFromStorageUnitRef(ev, aSURef), 0, 0);				
				}
				dropSU->GetDraft(ev)->EndClone(ev, key);
				
			CATCH_ALL
				dropSU->GetDraft(ev)->AbortClone(ev, key);
				RERAISE;
			ENDTRY
			
			newSU = theDraft->GetStorageUnit(ev, newSUID);

			newPart = theDraft->GetPart(ev, newSU->GetID(ev));
			newSU->Release(ev);

			frameDroppedIn = facet->GetFrame(ev);

			ODTransform *newExternalXForm = facet->CreateTransform(ev);
			ODShape *newFrameShape = frameDroppedIn->GetFrameShape(ev, kODNULL);
			ODShape *clipShape = newFrameShape->Copy(ev);

			if ( newFrameID == 0 )
			{
				if (dropSU->Exists(ev, kODPropFrameShape, NULL, 0))
				{
					newFrame = somSelf->CreateAnEmbeddedFrame(ev, frameDroppedIn, newPart, kODNullTypeToken, kODNullTypeToken, kODFalse); 
				}
			}else{
				newFrame = theDraft->GetFrame(ev, newFrameID);
				// Resize
				newFrame->RequestFrameShape(ev, newFrameShape, kODNULL);

				newFrame->SetDragging(ev, kODFalse);
				newFrame->SetContainingFrame(ev, frameDroppedIn);
				newFrame->SetDroppable(ev, kODTrue);

				_fEmbeddedFrames->AddFirst(newFrame);
				newFrame->SetFrozen(ev, kODTrue);
				//newFrame->SetPropagateEvents(ev, kODTrue);
				_fMode = kPuzzleSolve;
	
				//	create a proxy to hold the embedded frame
				//	can't use newShape for the below since we'd have to change it
	
				Proxy* p = new Proxy(clipShape->CopyWinRegion(ev), newFrame, newExternalXForm);			// Proxy Contructor consumes tempRgn
	
				//	put proxy into contents, possibly adjusting other content
				_fContents->AddFirst(p);

			}


			// Create facets

			frameDroppedIn->Invalidate(ev, kODNULL, kODNULL);
			ODReleaseObject(ev,newFrameShape);
			ODReleaseObject(ev,clipShape);
			ODReleaseObject(ev,newExternalXForm);
		
			somSelf->CreatePuzzleFacets(ev, facet, newFrame);
			//	PENDING: calculate each piece's offset randomly
			somSelf->ScramblePuzzle(ev);

			//SOM_TraceLevel--;

			frameDroppedIn->Invalidate(ev, kODNULL, kODNULL);
			newPart->Release(ev);
		}
	}
	somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);
	
	_fAlreadyContainAPart = kODTrue;

	return (movePart ? kODDropMove : kODDropCopy);

}

SOM_Scope void  SOMLINK ContainingPartPropertiesChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame,
		ODStorageUnit* propertyUnit)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
ODUnused(propertyUnit);
}

SOM_Scope ODStorageUnit*  SOMLINK GetContainingPartProperties(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
	THROW_IF_ERROR(kODErrCannotEmbed);
	return kODNULL;
}

SOM_Scope ODBoolean  SOMLINK RevealFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* embeddedFrame,
		ODShape* revealShape)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(embeddedFrame);

	THROW_IF_ERROR(kODErrCannotEmbed);

	return kODFalse;
}

SOM_Scope void  SOMLINK EmbeddedFrameSpec(Puzzle *somSelf, Environment *ev,
		ODFrame* embeddedFrame,
		ODObjectSpec* spec)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(embeddedFrame);
ODUnused(spec);

	THROW_IF_ERROR(kODErrCannotEmbed);
}

SOM_Scope ODEmbeddedFramesIterator*  SOMLINK CreateEmbeddedFramesIterator(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODEmbeddedFramesIterator* iter = new ODEmbeddedFramesIterator;
	
	iter->InitEmbeddedFramesIterator(ev, (ODPart*)_fPartWrapper);
	
	return iter;
}

SOM_Scope void  SOMLINK DisplayFrameAdded(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
 
	myPartInfoRec*	pInfo;
	ODTypeToken		viewType;

#ifdef LATER	//bt	
	if (frame->GetPart(ev) == _fPartWrapper)		//	frame belongs to me
#endif
	{
		//	create new part info for this frame
		pInfo = new myPartInfoRec;
		if (frame->IsRoot(ev))
			pInfo->fNeedsActivating = kODTrue;
		
		//	check frame's viewType and presentation and make sure they're OK
		viewType = frame->GetViewType(ev);
		if ((viewType == _fkODViewAsSmallIcon) || (viewType == _fkODViewAsLargeIcon) || (viewType == _fkODViewAsThumbnail) || (viewType == _fkODViewAsFrame))
			pInfo->fFrameType = viewType;
		else
			pInfo->fFrameType = _fkODViewAsFrame;	//	Default to view as frame (could be something exotic we can't handle).
		
		frame->SetDroppable(ev, kODTrue);
		frame->SetPartInfo(ev, (ODInfoType) pInfo);
		_fDisplayFrames->AddLast(frame);			//	add the new frame to the list

		frame->IncrementRefCount(ev);
		
	}
#ifdef LATER	//bt	
	else
		THROW(kODErrInvalidFrame);
#endif
}

SOM_Scope void  SOMLINK DisplayFrameRemoved(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
#ifdef	NOT_NOW	
	if (_fDisplayFrames->Contains(frame))
	{	
		myPartInfoRec* pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
		frame->SetPartInfo(ev, (ODInfoType) kODNULL);
		delete pInfo;

		_fSession->GetArbitrator(ev)->RelinquishFocusSet(ev, _fFocusSet, frame);
		_fDisplayFrames->Remove(frame);

		fSession->GetDispatcher(ev)->UnregisterIdle(ev, _fPartWrapper, frame);
		frame->Release(ev);
	}
#else
	if ( frame != kODNULL) 
	{
		if ( _fDisplayFrames->Contains(frame) )		//	PENDING: clean this mess up!
		{
			OrderedCollection* embeddedFrames = new OrderedCollection;
			{
				OrderedCollectionIterator iter(_fEmbeddedFrames);
				for ( ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
						iter.IsNotComplete(); 
						embeddedFrame = (ODFrame*) iter.Next() )
				{
					embeddedFrames->AddLast(embeddedFrame);
				}
			}
			OrderedCollectionIterator iter(embeddedFrames);
			for ( ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
					iter.IsNotComplete(); 
					embeddedFrame = (ODFrame*) iter.Next() )
			{
				if ( embeddedFrame->GetContainingFrame(ev) == frame )
				{
					somSelf->RemoveEmbeddedFrame(ev, embeddedFrame);
				}
			}
			delete embeddedFrames; // Delete the copy
	
			_fSession->GetArbitrator(ev)->RelinquishFocusSet(ev, _fFocusSet,frame);
	
			myPartInfoRec* pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
			frame->SetPartInfo(ev, (ODInfoType) kODNULL);
			delete pInfo;
			_fDisplayFrames->Remove(frame);
			frame->Release(ev);
			
			OrderedCollectionIterator dIter(_fDisplayFrames);
			ODFrame*	displayFrame = (ODFrame*) dIter.First();

		}
		else
			THROW_IF_ERROR(kODErrInvalidFrame);
	}
#endif
}

SOM_Scope void  SOMLINK DisplayFrameConnected(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	myPartInfoRec*	pInfo;
	ODTypeToken		viewType;
	
#ifdef LATER	//bt	
	if (frame->GetPart(ev) == _fPartWrapper)		//	frame belongs to me
#endif
	{
		//	create new part info for this frame
		pInfo = new myPartInfoRec;
		if (frame->IsRoot(ev))
			pInfo->fNeedsActivating = kODTrue;
		
		//	check frame's viewType and presentation and make sure they're OK
		viewType = frame->GetViewType(ev);
		if ((viewType == _fkODViewAsSmallIcon) || (viewType == _fkODViewAsLargeIcon) || (viewType == _fkODViewAsThumbnail) || (viewType == _fkODViewAsFrame))
			pInfo->fFrameType = viewType;
		else
			pInfo->fFrameType = _fkODViewAsFrame;	//	Default to view as frame (could be something exotic we can't handle).
		
		frame->SetDroppable(ev, kODTrue);
		frame->SetPartInfo(ev, (ODInfoType) pInfo);
		_fDisplayFrames->AddLast(frame);			//	add the new frame to the list

		frame->IncrementRefCount(ev);
		
	}
#ifdef LATER	//bt	
	else
		THROW(kODErrInvalidFrame);
#endif
}

SOM_Scope void  SOMLINK DisplayFrameClosed(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if ( _fDisplayFrames->Contains(frame) )
	{
		OrderedCollection* embeddedFrames = new OrderedCollection;
		{
			OrderedCollectionIterator iter(_fEmbeddedFrames);
			for ( ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
					iter.IsNotComplete(); 
					embeddedFrame = (ODFrame*) iter.Next() )
			{
				embeddedFrames->AddLast(embeddedFrame);
			}
		}
		
		OrderedCollectionIterator iter(embeddedFrames);
		for ( ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
				iter.IsNotComplete(); 
				embeddedFrame = (ODFrame*) iter.Next() )
		{
			if ( embeddedFrame->GetContainingFrame(ev) == frame )
			{
				embeddedFrame->Close(ev);
				_fEmbeddedFrames->Remove(embeddedFrame);
				ODReleaseObject(ev,embeddedFrame);
			}
		}
		delete embeddedFrames; // Delete the copy
		
		_fSession->GetArbitrator(ev)->RelinquishFocusSet(ev, _fFocusSet, frame);

		myPartInfoRec* pInfo = (myPartInfoRec*) frame->GetPartInfo(ev);
		frame->SetPartInfo(ev, (ODInfoType) kODNULL);
		delete pInfo;
		_fDisplayFrames->Remove(frame);
		ODReleaseObject(ev,frame);
	}
	else
		THROW_IF_ERROR(kODErrInvalidFrame);
}

SOM_Scope void  SOMLINK AttachSourceFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* frame,
		ODFrame* sourceFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if ( _fDisplayFrames->Contains(frame)
		 && sourceFrame && _fDisplayFrames->Contains(sourceFrame) )
	{
		
			OrderedCollection* embeddedFrames = new OrderedCollection;
			{
				OrderedCollectionIterator iter(_fEmbeddedFrames);
				for (ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
						iter.IsNotComplete(); 
						embeddedFrame = (ODFrame*) iter.Next())
				{
					embeddedFrames->AddLast(embeddedFrame);
				}
			}
			
			OrderedCollectionIterator iter(embeddedFrames);
			for (ODFrame* embeddedFrame = (ODFrame*) iter.First(); 
					iter.IsNotComplete(); 
					embeddedFrame = (ODFrame*) iter.Next())
			{
				if (embeddedFrame->GetContainingFrame(ev) == sourceFrame)
				{
					ODPart* embeddedPart = embeddedFrame->GetPart(ev);
					ODFrame* newFrame = somSelf->CreateAnEmbeddedFrame(ev, frame, embeddedPart, kODNullTypeToken,
											  kODNullTypeToken, embeddedFrame->IsOverlaid(ev));
				embeddedPart->AttachSourceFrame(ev, newFrame, embeddedFrame);
				}
			}
			delete embeddedFrames; // Delete the copy
		}
	else
		THROW_IF_ERROR(kODErrInvalidFrame);
}

SOM_Scope void  SOMLINK GeometryChanged(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODBoolean clipShapeChanged,
		ODBoolean externalTransformChanged)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	facet->Invalidate(ev, kODNULL, kODNULL);
}

//	need to resize the embedded frame & all embedded facets!
SOM_Scope void  SOMLINK FrameShapeChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

	ODFrame*						frameToResize;
	ODShape*						newShape;
	ODShape*						tempShape = kODNULL;
	ODShape*						tempClip = kODNULL;
	ODRect							bounds;
	RECT							basePuzzleRect, tempRect;
	OrderedCollectionIterator		displayIterator(_fEmbeddedFrames);
	OrderedCollectionIterator		tempIterator(_fPuzzleFacets);
	PuzzleFacetInfo					*somePuzzleFacetInfo;
	short							eachPieceHeight, eachPieceWidth;
	ODTransform*					tempXForm = kODNULL;
	ODPoint							tempPoint;
	POINT							myPoint;

	//somPrintf("PuzzleFrameShapeChanged \r");

	//	get new shape
	if (_fEmbeddedFrames->Count() > 0)
	{
		//	PENDING: call RequestFrameShape with the new shape that ensures embedded frame
		//		will have a width & height such that the puzzle pieces will fit within it nicely
		
		newShape = somSelf->CalcEmbeddedPartShape(ev, frame);
		
		newShape->GetBoundingBox(ev, &bounds);
		bounds.AsWinRect(basePuzzleRect);
		
		//somPrintf("frameShapeRect.l: %d r: %d t: %d b: %d \r", basePuzzleRect.left, basePuzzleRect.right, basePuzzleRect.top, basePuzzleRect.bottom);

		eachPieceHeight = ((basePuzzleRect.bottom - basePuzzleRect.top) / 4) - 2;
		eachPieceWidth = ((basePuzzleRect.right - basePuzzleRect.left) / 4) - 2;

		::SetRect(&basePuzzleRect, basePuzzleRect.left + 1, basePuzzleRect.top + 1, basePuzzleRect.left + eachPieceWidth, basePuzzleRect.top + eachPieceHeight);
		
		//somPrintf("basePuzzleRect.l: %d r: %d t: %d b: %d \r", basePuzzleRect.left, basePuzzleRect.right, basePuzzleRect.top, basePuzzleRect.bottom);
		//somPrintf("eachPieceHeight: %d width: %d \r", eachPieceHeight, eachPieceWidth);

		if (eachPieceHeight < 2 || eachPieceWidth < 2)
		{
			// Puzzle part cannot be resized this small
		
			newShape->Release(ev);
			newShape = kODNULL;
			
			return;
		}
		
		frameToResize = (ODFrame*)displayIterator.First();		//	should only be one; you *could* do the iteration, but it seems safer not to!
		{
			//	make a copy of newShape to pass in
			frameToResize->ChangeFrameShape(ev, newShape, kODNULL);	//	newShape's refcount is incremented
			
			//	now iterate over the puzzle pieces and change their size & facet shape
			for (somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.First();
					tempIterator.IsNotComplete();
					somePuzzleFacetInfo = (PuzzleFacetInfo*) tempIterator.Next())
			{
				somePuzzleFacetInfo->fWidth = eachPieceWidth + 2;
				somePuzzleFacetInfo->fHeight = eachPieceHeight + 2;

				//	calculate new clip shape
				//	use original grid location; this is for the clipshape,
				//	*not* it's temporary scrambled position -- that's handled by the external transform
				tempRect = basePuzzleRect;
				::OffsetRect(&tempRect,
						somePuzzleFacetInfo->fOriginalGridLocation.x * somePuzzleFacetInfo->fWidth,
						somePuzzleFacetInfo->fOriginalGridLocation.y * somePuzzleFacetInfo->fHeight);
				//	convert this rect to an ODRect

				//somPrintf("facetRect.l: %d r: %d t: %d b: %d \r", tempRect.left, tempRect.right, tempRect.top, tempRect.bottom);

				bounds = tempRect;
				
				//tempShape = frameToResize->CreateShape(ev);
				tempClip = somePuzzleFacetInfo->fFacet->GetClipShape(ev, kODNULL);					// clipshape refcount is incremented
				//tempShape = tempClip->Copy(ev);
				tempShape = tempClip->NewShape(ev);
				tempShape->SetRectangle(ev, &bounds);
				
				tempClip->Release(ev);
				tempClip = kODNULL;
				
				//	calculate new offset
				tempXForm = frameToResize->CreateTransform(ev);
				myPoint.x = (somePuzzleFacetInfo->fGridLocation.x - somePuzzleFacetInfo->fOriginalGridLocation.x) * somePuzzleFacetInfo->fWidth;
				myPoint.y = (somePuzzleFacetInfo->fGridLocation.y - somePuzzleFacetInfo->fOriginalGridLocation.y) * somePuzzleFacetInfo->fHeight;
				tempPoint = myPoint;
				tempXForm->SetOffset(ev, &tempPoint);

				//	set the sucker
				somePuzzleFacetInfo->fFacet->ChangeGeometry(ev, tempShape, tempXForm, kODNULL);					//	tempShape and tempXForm refcounts are incremented

				tempXForm->Release(ev);
				tempXForm = kODNULL;

				// Reset activate shape
				ODShape *newActiveShape = tempShape->Copy(ev);
				somePuzzleFacetInfo->fFacet->ChangeActiveShape(ev, newActiveShape, kODNULL);					// newActiveShape refcount is incremented
				newActiveShape->Release(ev);
				newActiveShape = kODNULL;
				
				tempShape->Release(ev);
				tempShape = kODNULL;
			}
		}

		frame->Invalidate(ev, newShape, kODNULL);		//	otherwise only the previously clipped area gets invalidated!
		newShape->Release(ev);
		newShape = kODNULL;
	}

	//somPrintf("Exit PuzzleFrameShapeChanged \r");
}

SOM_Scope void  SOMLINK ViewTypeChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
}

SOM_Scope void  SOMLINK PresentationChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
}

SOM_Scope void  SOMLINK SequenceChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
}

SOM_Scope void  SOMLINK WritePartInfo(Puzzle *somSelf, Environment *ev,
		ODPtr partInfo,
		ODStorageUnitView* storageUnitView)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if (partInfo)
	{
		ODBoolean needsActivating = ((myPartInfoRec*)partInfo)->fNeedsActivating 
									|| ((myPartInfoRec*)partInfo)->fIsActive;
		storageUnitView->SetValue(ev, sizeof(ODBoolean),
									(ODValue)&needsActivating);
	}
}

SOM_Scope ODPtr  SOMLINK ReadPartInfo(Puzzle *somSelf, Environment *ev,
		ODFrame* frame,
		ODStorageUnitView* storageUnitView)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
	if (storageUnitView->GetSize(ev))
	{
		myPartInfoRec* partInfo = new myPartInfoRec;
		
		ODBoolean needsActivating;
		storageUnitView->GetValue(ev, sizeof(ODBoolean),
									(ODValue)&(needsActivating));
		partInfo->fNeedsActivating = needsActivating;
									
		OrderedCollectionIterator d(_fDisplayFrames);
		ODFrame* dispFrame = (ODFrame*) d.First();
	
		return partInfo;
	}
	else
		return ((ODPtr)kODNULL);
}

SOM_Scope ODID  SOMLINK Open(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODWindow*	window = kODNULL;
	ODID		theWindowID = 0;

	if (frame) // Doing a View As Window
	{
		window = _fSession->GetWindowState(ev)->GetWindow(ev, theWindowID);
		if (window)
			window->Select(ev);
		else
		{
			window = somSelf->ODCreateWindow(ev, frame);
			theWindowID = window->GetID(ev);
			window->Open(ev);
			window->Show(ev);
			window->Select(ev);			
		}
	}
	else
	{
		window = somSelf->ODCreateWindow(ev, frame);
		theWindowID = window->GetID(ev);
		window->Open(ev);
		window->Show(ev);
		window->Select(ev);			
	}
	return window->GetID(ev);
}

SOM_Scope ODFrame*  SOMLINK RequestEmbeddedFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* containingFrame,
		ODFrame* baseFrame,
		ODShape* frameShape,
		ODPart* embedPart,
		ODTypeToken viewType,
		ODTypeToken presentation,
		ODBoolean isOverlaid)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODShape*		newShape;
	ODFrame*		newFrame;
	ODShape*		tempShape = containingFrame->CreateShape(ev);
	ODRgnHandle		tempRgn;
	ODTransform* 	externalTransform;
	
	
ODUnused(baseFrame);
ODUnused(frameShape);

	//	get new shape
	newShape = somSelf->CalcEmbeddedPartShape(ev, containingFrame);
	if (!newShape)
		return(kODNULL);
		
	// create the new frame
	newFrame = somSelf->GetStorageUnit(ev)->GetDraft(ev)->
						CreateFrame(ev, kODNULL, containingFrame, newShape, kODNULL, embedPart,
						viewType, presentation, kODFalse, isOverlaid);
						
	_fEmbeddedFrames->AddFirst(newFrame);

	//	create a proxy to hold the embedded frame
	//	can't use newShape for the below since we'd have to change it
	tempShape->CopyFrom(ev, newShape);
	tempRgn = tempShape->CopyWinRegion(ev);
	
	Proxy* p = new Proxy(tempRgn, newFrame, externalTransform);									// contructor consumes tempRgn
	
	//	put proxy into contents, possibly adjusting other content
	_fContents->AddFirst(p);

	//	get rid of temporary objects
	tempShape->Release(ev);
	
	return (newFrame);
}

SOM_Scope void  SOMLINK RemoveEmbeddedFrame(Puzzle *somSelf, Environment *ev,
		ODFrame* embeddedFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if (_fEmbeddedFrames->Contains(embeddedFrame))
	{
		ODFrame* containingFrame = embeddedFrame->GetContainingFrame(ev);
		Proxy* p = somSelf->ProxyForFrame(ev, embeddedFrame);
		if (p)
		{	
			_fContents->Remove(p);
			DeleteObject(p->region);
			delete p;
		}
		_fEmbeddedFrames->Remove(embeddedFrame);
		embeddedFrame->Remove(ev);
		//somSelf->GetStorageUnit(ev)->GetDraft(ev)->RemoveFrame(ev, embeddedFrame);
		
		somSelf->ClipEmbeddedFrames(ev, containingFrame);

	}
	else
		THROW(kODErrInvalidFrame);
}

SOM_Scope ODShape*  SOMLINK RequestFrameShape(Puzzle *somSelf, Environment *ev,
		ODFrame* embeddedFrame,
		ODShape* frameShape)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);

ODUnused(frameShape);
	
	ODShape		*newShape;
	
	newShape = new ODShape;
	newShape->CopyFrom(ev, embeddedFrame->GetFrameShape(ev, kODNULL));

	return(newShape);
}

SOM_Scope void  SOMLINK UsedShapeChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* embeddedFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(embeddedFrame);
}

SOM_Scope ODShape*  SOMLINK AdjustBorderShape(Puzzle *somSelf, Environment *ev,
		ODFacet* embeddedFacet,
		ODShape* shape)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(embeddedFacet);

	return shape;
}

SOM_Scope void  SOMLINK FacetAdded(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
}

SOM_Scope void  SOMLINK FacetRemoved(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODFrame* dispFrame = facet->GetFrame(ev);
	if ( !_fDisplayFrames->Contains(dispFrame) )
		THROW_IF_ERROR(kODErrInvalidFacet);
	
	OrderedCollection* children = new OrderedCollection;
	ODFacetIterator* facets = facet->CreateFacetIterator(ev, kODChildrenOnly, kODFrontToBack);
	for ( ODFacet* childFacet = facets->First(ev);
		facets->IsNotComplete(ev);
		childFacet = facets->Next(ev) )
	{
		children->AddLast(childFacet);
	}
	delete facets;
	
	OrderedCollectionIterator iter(children);
	for ( childFacet = (ODFacet*)iter.First();
			iter.IsNotComplete();
			childFacet = (ODFacet*)iter.Next() )
	{
		facet->RemoveFacet(ev, childFacet);
		delete childFacet;
	}
	delete children;
}

SOM_Scope void  SOMLINK CanvasChanged(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
}

/*
	No intrinsic content, but for friendliness check to make sure some
	content exits. If not, print a meaningful message.
*/
SOM_Scope void  SOMLINK Draw(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODShape* invalidShape)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODFrame* 			frame = facet->GetFrame(ev);
	ODTypeToken			curView = frame->GetViewType(ev);

	ODRgnHandle 		saveClip = ODNewRgn();
	ODShape*			tempClip = kODNULL;
	ODShape*			frameShape = kODNULL;
	
	RECT				tempRect;
	
	myPartInfoRec*		pInfo;
	
	
	pInfo = (myPartInfoRec*)(frame->GetPartInfo(ev));
	
	HDC	hdc;

	CFocus	f(facet, invalidShape, &hdc);

	//	change the origin so drawing calculations are easier
	ODTransform*	localToGlobal = facet->GetContentTransform(ev, kODNULL);
	ODPoint			offset(0,0);
	offset = localToGlobal->TransformPoint(ev, &offset);
	
	localToGlobal->Release(ev);
	localToGlobal = kODNULL;

	frameShape = frame->GetFrameShape(ev, kODNULL);
				

	GetRgnBox (frameShape->GetWinRegion(ev), &tempRect);

	//	do drawing:
	//	check facet's cavas' isDynamic to see if we're drawing into a window...or something else...
	//	check facet's frame's partInfo to see how we should draw
	
	if (pInfo->fFrameType == _fkODViewAsFrame)		//	"Normal" view type
	{
		HBRUSH hbr = CreateSolidBrush (rgbRed);
		::FillRgn(hdc, frameShape->GetWinRegion(ev), hbr);
		DeleteObject (hbr);
		
		//	if there is no embedded part, intrinsic contents is a big X
		//	also draw "No content" message 
		FrameRect(hdc, &tempRect, GetStockObject (BLACK_BRUSH));
		if (!_fAlreadyContainAPart)
		{
			::MoveToEx(hdc, tempRect.left, tempRect.top, kODNULL);
			::LineTo(hdc, tempRect.right, tempRect.bottom);
			::MoveToEx(hdc, tempRect.right, tempRect.top, kODNULL);
			::LineTo(hdc, tempRect.left, tempRect.bottom);
		
			somSelf->DrawPuzzlePartText(ev, facet);
		}
		else
		{
			//	iterate thorugh puzzle pieces and frame them all
			OrderedCollectionIterator		facetIterator(_fPuzzleFacets);
			RECT							bounds;
			PuzzleFacetInfo					*somePuzzleFacetInfo;
			ODRect							odBounds;
			ODTransform						*tempXForm;
			ODPoint							offset(0,0);
			ODShape*						clipShape = kODNULL;
			
			for (somePuzzleFacetInfo = (PuzzleFacetInfo*)facetIterator.First();
					facetIterator.IsNotComplete();
					somePuzzleFacetInfo = (PuzzleFacetInfo*)facetIterator.Next())
			{
				clipShape = somePuzzleFacetInfo->fFacet->GetClipShape(ev, kODNULL);
				
				clipShape->GetBoundingBox(ev, &odBounds);
				
				//	now have to offset this rectangle by the facet's externaltransform
				tempXForm = somePuzzleFacetInfo->fFacet->GetExternalTransform(ev, kODNULL);
				tempXForm->GetOffset(ev, &offset);

				tempXForm->Release(ev);
				tempXForm = kODNULL;
				clipShape->Release(ev);
				clipShape = kODNULL;

				odBounds.AsWinRect(bounds);
				::OffsetRect(&bounds, offset.IntX(), offset.IntY());
				::InflateRect(&bounds, 2, 2);
				::FrameRect(hdc, &bounds, GetStockObject (BLACK_BRUSH));
			}
		}
		
		//	highlight the selection
		if (frame == _fSession->GetArbitrator(ev)->GetFocusOwner(ev, _fSelectionFocus))
		{
			switch (_fSelection->Count())
			{
				case 0:		//	no selection
					break;
				case 1:		//	single item
					{
//bt						FillRgn(_fSelectRgn, &(ODQDGlobals.gray));
//bt						FillRgn(_fCornerHandleRgn, &(ODQDGlobals.white));
//bt						FillRgn(_fEdgeHandleRgn, &(ODQDGlobals.white));
//bt						FrameRgn(_fCornerHandleRgn);
//bt						FrameRgn(_fEdgeHandleRgn);
					};
					break;
		
				default:	//	multiple selection...shouldn't be possible
					break;
			};

		}
	}
	else if (pInfo->fFrameType == _fkODViewAsLargeIcon)
	{
		SetRect(&tempRect, 0, 0, kODLargeIconSize, kODLargeIconSize);
//		PlotIconSuite(&tempRect, atAbsoluteCenter, ttNone, fLargeIcons);
	}
	else if (pInfo->fFrameType == _fkODViewAsSmallIcon)
	{
		SetRect(&tempRect, 0, 0, kODSmallIconSize, kODSmallIconSize);
//		PlotIconSuite(&tempRect, atAbsoluteCenter, ttNone, fSmallIcons);
	}
	else if (pInfo->fFrameType ==  _fkODViewAsThumbnail)
	{
		SetRect(&tempRect, 0, 0, kODSmallIconSize, kODSmallIconSize);
//		PlotIconSuite(&tempRect, atAbsoluteCenter, ttNone, fSmallIcons);
	}
	
//	::SetOrigin(offset.IntX(), offset.IntY());		//	put it back; note the code here is wrong!

	
	frameShape->Release(ev);
	frameShape = kODNULL;
}

SOM_Scope void  SOMLINK CanvasUpdated(Puzzle *somSelf, Environment *ev,
		ODCanvas* canvas)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(canvas);
}

SOM_Scope void  SOMLINK HighlightChanged(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
}

SOM_Scope ODULong  SOMLINK GetPrintResolution(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);

	return 0;
}

SOM_Scope ODLink*  SOMLINK CreateLink(Puzzle *somSelf, Environment *ev,
		ODPtr data,
		ODULong size)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(data);
ODUnused(size);
	return kODNULL;
}

SOM_Scope void  SOMLINK LinkUpdated(Puzzle *somSelf, Environment *ev,
		ODLink* updatedLink,
		ODChangeID id)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(updatedLink);
ODUnused(id);
}

SOM_Scope void  SOMLINK RevealLink(Puzzle *somSelf, Environment *ev,
		ODLinkSource* linkSource)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(linkSource);
}

SOM_Scope void  SOMLINK EmbeddedFrameChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame,
		ODChangeID change)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(change);
ODUnused(frame);
}

SOM_Scope void  SOMLINK LinkStatusChanged(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
}

SOM_Scope ODBoolean  SOMLINK BeginRelinquishFocus(Puzzle *somSelf, Environment *ev,
		ODTypeToken focus,
		ODFrame* ownerFrame,
		ODFrame* proposedFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(focus);
ODUnused(ownerFrame);
ODUnused(proposedFrame);

	return kODTrue;
}

SOM_Scope void  SOMLINK CommitRelinquishFocus(Puzzle *somSelf, Environment *ev,
		ODTypeToken focus,
		ODFrame* ownerFrame,
		ODFrame* proposedFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(proposedFrame);

	somSelf->FocusLost(ev, focus, ownerFrame);
}

SOM_Scope void  SOMLINK AbortRelinquishFocus(Puzzle *somSelf, Environment *ev,
		ODTypeToken focus,
		ODFrame* ownerFrame,
		ODFrame* proposedFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(focus);
ODUnused(ownerFrame);
ODUnused(proposedFrame);
}

SOM_Scope void  SOMLINK FocusAcquired(Puzzle *somSelf, Environment *ev,
		ODTypeToken focus,
		ODFrame* ownerFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	if (focus == _fSelectionFocus) 
	{
		myPartInfoRec* pInfo = (myPartInfoRec*) ownerFrame->GetPartInfo(ev);
		pInfo->fIsActive = kODTrue;
	}
	else if (focus == _fMenuFocus) 
		somSelf->InstallMenus(ev, ownerFrame);
}

SOM_Scope void  SOMLINK FocusLost(Puzzle *somSelf, Environment *ev,
		ODTypeToken focus,
		ODFrame* ownerFrame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	myPartInfoRec*	pInfo = (myPartInfoRec*) ownerFrame->GetPartInfo(ev);
	
	if (focus == _fSelectionFocus)
	{
		pInfo->fIsActive = kODFalse;
		somSelf->InvalidateSelection(ev, ownerFrame);
		somSelf->ClipEmbeddedFrames(ev, ownerFrame);
	}
#if	0
	else if (focus == _fMenuFocus)
		somSelf->RemoveMenus(ev, ownerFrame);
#endif
}

SOM_Scope void  SOMLINK CloneInto(Puzzle *somSelf, Environment *ev,
		ODDraftKey key,
		ODStorageUnit* toSU,
		ODFrame* scope)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODStorageUnit*	su = somSelf->GetStorageUnit(ev);
	ODDraft*		draft = su->GetDraft(ev);

	somSelf->Externalize(ev);
	TRY
	
		ODID scopeFrameID = 0;
		if (scope != kODNULL)
			scopeFrameID = scope->GetStorageUnit(ev)->GetID(ev);
		su->CloneInto(ev, key, toSU, scopeFrameID);

		//su->CloneInto(ev, key, toSU, scope);
		
	CATCH_ALL
	
		draft->AbortClone(ev, key);
		
	ENDTRY
}

SOM_Scope void  SOMLINK ExternalizeKinds(Puzzle *somSelf, Environment *ev,
		ODTypeList* kindset)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(kindset);
	somSelf->Externalize(ev);
}

SOM_Scope void  SOMLINK ChangeKind(Puzzle *somSelf, Environment *ev,
		ODType kind)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(kind);
}

SOM_Scope ODBoolean  SOMLINK HandleEvent(Puzzle *somSelf, Environment *ev,
		ODEventData* event,
		ODFrame* frame,
		ODFacet* facet)
{
	ODBoolean	handled = kODFalse;

#ifndef WIN32
	POINT	p = MAKEPOINT(event->lParam);
#else
	POINT p;
	LONG2POINT(event->lParam, p);
#endif
	ODPoint point = p;
	
	switch (event->message)
	{
		case WM_LBUTTONDOWN:
			handled = somSelf->HandleMouseDown(ev, facet, &point,event);
			break;

		case WM_MOUSEMOVE:
			 if (event->wParam & MK_LBUTTON)
				handled = somSelf->HandleMouseDown(ev, facet, &point,event);
			break;

		case WM_KEYDOWN:
//bt			handled = somSelf->HandleKeyDown(ev, frame, event );
			break;

		case WM_CHAR:
			handled = FALSE;
			break;

		case WM_COMMAND:
			handled = somSelf->HandleMenuEvent(ev, frame,event);
			break;

//bt		case kODEvtMouseDownEmbedded:
//bt			{
//bt				handled = somSelf->HandleMouseDownInEmbeddedFrame(ev, facet,
//bt					(ODFacet*) event->lParam, &point,event);
//bt			}
//bt			break;

		case kODEvtMouseDownBorder:
			{
//bt				handled = somSelf->HandleMouseDownInEmbeddedFrame(ev, facet,
//bt					(ODFacet*) event->lParam, &point,event);
			}
			break;


		case WM_LBUTTONUP:
			break;

		case WM_ACTIVATE:
			handled = kODTrue; // actually ignored by dispatcher
			if (event->wParam  != WA_INACTIVE)
				somSelf->ActivatingWindow(ev, frame);
			else
				somSelf->DeActivatingWindow(ev, frame);
			break;


//bt		case WM_VSCROLL:
//bt			somSelf->SetBGColor1(ev, frame, (ODUShort)((rand()%6)+IDM_RED));
//bt			break;


		default:
			return kODFalse;
	}
	return handled;
			
}

SOM_Scope ODBoolean  SOMLINK HandleEventInEmbedded(Puzzle *somSelf, Environment *ev,
		ODEventData* event,
		ODFrame* frame,
		ODFacet* facet,
		ODFrame* embeddedFrame,
		ODFacet* embeddedFacet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(frame);
ODUnused(embeddedFacet);

	ODBoolean		handled = kODFalse;

#ifndef WIN32
	POINT	p = MAKEPOINT(event->lParam);
#else
	POINT p;
	LONG2POINT(event->lParam, p);
#endif
	ODPoint ODWindowPoint = p;

	switch(event->message)
	{
		case kODEvtMouseDownBorder:
			break;
			
		case kODEvtMouseDownEmbedded:	//	when user clicks in a frozen facet, we get one of these...if the facet is not frozen, we never know about it!
			{
				handled = somSelf->HandleMouseDownInPuzzlePiece(ev, facet, embeddedFacet);
				if (handled)
				{
					if (somSelf->IsPuzzleSolved(ev))
						somSelf->DoPuzzleSolvedThing(ev);	//	do something interesting
					else
						somSelf->PlayAPuzzleSound(ev);
				}
			}
			break;
		
		case kODEvtMouseUpEmbedded:
			break;
						
		default:
			somPrintf("Puzzle HandleEventInEmbedded - event not among handled events\r");
			break;
	}

	return handled;
}

SOM_Scope void  SOMLINK MouseEnter(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
ODUnused(where);

//	::SetCursor(*::GetCursor(1024));
}

SOM_Scope void  SOMLINK MouseWithin(Puzzle *somSelf, Environment *ev,
		ODFacet* facet,
		ODPoint* where)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);
ODUnused(where);
}

SOM_Scope void  SOMLINK MouseLeave(Puzzle *somSelf, Environment *ev,
		ODFacet* facet)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(facet);

//	SetCursor(&ODQDGlobals.arrow);
}

SOM_Scope void  SOMLINK AdjustMenus(Puzzle *somSelf, Environment *ev,
		ODFrame* frame)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	_fMenuBar->EnableCommand(ev, kODCommandOpen, kODTrue);
	_fMenuBar->EnableCommand(ev, IDM_RE_SCRAMBLE,_fAlreadyContainAPart);	
	_fMenuBar->EnableCommand(ev, IDM_ED_IN_PLACE,_fAlreadyContainAPart);	
//	_fMenuBar->EnableCommand(ev, IDM_CLICK_SOUND,_fAlreadyContainAPart);	
	
	//	don't need to do anything with kPuzzleMenuReScramble; it's always enabled
}

SOM_Scope void  SOMLINK UndoAction(Puzzle *somSelf, Environment *ev,
		ODActionData actionState)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(actionState);
}

SOM_Scope void  SOMLINK RedoAction(Puzzle *somSelf, Environment *ev,
		ODActionData actionState)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(actionState);
}

SOM_Scope void  SOMLINK DisposeActionState(Puzzle *somSelf, Environment *ev,
		ODActionData actionState,
		ODDoneState doneState)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(actionState);
ODUnused(doneState);
}

SOM_Scope void  SOMLINK WriteActionState(Puzzle *somSelf, Environment *ev,
		ODPtr actionState,
		ODStorageUnitView* storageUnitView)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(actionState);
ODUnused(storageUnitView);
}

SOM_Scope ODPtr  SOMLINK ReadActionState(Puzzle *somSelf, Environment *ev,
		ODStorageUnitView* storageUnitView)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
ODUnused(storageUnitView);
	return kODNULL;
}

SOM_Scope void  SOMLINK InitPart(Puzzle *somSelf, Environment *ev,
		ODStorageUnit* storageUnit,
		ODPartWrapper* partWrapper)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODStorageUnit*		myStorageUnit;
	ODStorageUnit*		consu;
	ODStorageUnitRef	consuRef;
	
	if (somSelf->IsInitialized(ev))
		return;
	
	somSelf->InitPersistentObject(ev, storageUnit);
	
	_fSession = storageUnit->GetSession(ev);
	_fPartWrapper = partWrapper;
	somSelf->CommonInitPuzzle(ev);

	myStorageUnit = somSelf->GetStorageUnit(ev);
	
	myStorageUnit->AddProperty(ev, kODPropContents)->AddValue(ev, kODKindPuzzle);
	
	consu = myStorageUnit->GetDraft(ev)->CreateStorageUnit(ev);
	consuRef = myStorageUnit->GetStrongStorageUnitRef(ev, consu->GetID(ev));
	myStorageUnit->SetValue(ev, sizeof(ODStorageUnitRef),&consuRef);
	
	consu->AddProperty(ev, kODPropDisplayFrames);
	consu->AddValue(ev, kODWeakStorageUnitRef);
	
	consu->Release(ev);
}

SOM_Scope void  SOMLINK InitPartFromStorage(Puzzle *somSelf, Environment *ev,
		ODStorageUnit* storageUnit,
		ODPartWrapper* partWrapper)
{
    PuzzleData *somThis = PuzzleGetData(somSelf);
    
	ODStorageUnit*		consu;
	ODStorageUnitRef	suRef;

	if (somSelf->IsInitialized(ev))
		return;
	
	somSelf->InitPersistentObjectFromStorage(ev, storageUnit);

	_fSession = storageUnit->GetSession(ev);
	_fPartWrapper = partWrapper;

	somSelf->CommonInitPuzzle(ev);

	storageUnit->Focus(ev, kODPropContents, kODPosUndefined, kODKindPuzzle, 0, kODPosUndefined);
	storageUnit->GetValue(ev, sizeof(ODStorageUnitRef),&suRef);
	
	consu = storageUnit->GetDraft(ev)->GetStorageUnit(ev, storageUnit->GetIDFromStorageUnitRef(ev, suRef));
	
	//	read in display frame
	if (consu->Exists(ev, kODPropDisplayFrames, NULL, 0))
	{
	 	ODStorageUnitRef	aSURef;
		ODFrame*		ODFrame;
		ODULong			i;
	
		consu->Focus(ev, kODPropDisplayFrames,kODPosUndefined,0,1,kODPosFirstSib);
	
		ODULong		count;
		consu->GetValue(ev, sizeof(count), (ODValue)&count);
	
		for (i = 0; i < count; i++)
		{
			consu->GetValue(ev, sizeof(ODStorageUnitRef), (ODValue)&aSURef);
			
			TRY
			
				ODFrame = consu->GetDraft(ev)->GetFrame(ev, consu->GetIDFromStorageUnitRef(ev, aSURef));
				_fDisplayFrames->AddLast((ElementType) ODFrame);
				
			CATCH_ALL
				
				ODFrame = kODNULL;
				
			ENDTRY
		}
	}

	
	consu->Release(ev);
}

#ifdef TESTING //bt
void ShowRgn(HRGN rgn)
{
	HDC hOldDC = GetDC (GetDesktopWindow());
	HDC hDC = CreateCompatibleDC (hOldDC);
	HBITMAP hBitmap, hOldBitmap;
	hBitmap = CreateCompatibleBitmap (hDC, 1024, 768);
	hOldBitmap = SelectObject (hDC, hBitmap);

	RECT rc;
	SetRect (&rc, 0, 0, 1023, 766);
	FillRect (hDC, &rc, GetStockObject (WHITE_BRUSH));
	FillRgn (hDC, rgn, GetStockObject (BLACK_BRUSH));

	SelectObject (hDC, hOldBitmap);
	DeleteDC (hOldDC);

	OpenClipboard (GetDesktopWindow());
	EmptyClipboard ();
	SetClipboardData(CF_BITMAP, hBitmap);
	CloseClipboard ();
//BGNBP
}
#endif
