/*
	File:		NicePWin.cpp

	Contains:	Windows platform implementation for NicePart

	Written by:	Scott Sorensen

	Change History (most recent first):

		 <0>	  8/11/94	SWS		created from NicePMac.cpp
*/

//-------------------------------------------------------------------------
// NicePWin
//
// About this file:
// The file structure for this project is designed to facility cross-platform
// development.  The idea is to use a cross-platform engine to implement the core
// functionality of the product.  The interface is seperated from the engine and
// is diveded into those parts that are cross-platform and those parts that are
// platform specific.  This is the file that contains the platform specific part
// of the OpenDoc code.
//
//-------------------------------------------------------------------------

#ifndef _NICEPART_
#include "NicePart.h"
#endif

#ifndef _FACET_
#include "Facet.h"
#endif

#ifndef _FRAME_
#include "Frame.h"
#endif

#ifndef _SHAPE_
#include "Shape.h"
#endif

#ifndef _FOCUSLIB_
#include "FocusLib.h"
#endif

#ifndef _MENUBAR_
#include "MenuBar.h"
#endif

#ifndef _WINSTAT_
#include "WinStat.h"
#endif

#ifndef _WINDOW_
#include "Window.h"
#endif

#ifndef _CANVAS_
#include "Canvas.h"
#endif

#ifndef _XMPSESSW_
#include "XMPSessW.h"
#endif

#include <string.h>

//-------------------------------------------------------------------------
// BuildPlatMenus
//
// About this method:
// In order to build our menus we first create a popup menu and append the menu
// items.  We then get a copy of the base menu which consists of the file and edit
// menus and add our newly created menu to it.
//
//-------------------------------------------------------------------------

void NicePart::BuildPlatMenus()
{
	fColorMenu.hMenu = CreatePopupMenu();
	strcpy (fColorMenu.strMenutext, "&Color");

	// create the color menu

	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cGray, "&Gray");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cRed, "&Red");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cGreen, "G&reen");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cYellow, "&Yellow");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cBlue, "&Blue");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cMagenta, "&Magenta");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cCyan, "&Cyan");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cWhite, "&White");
	AppendMenu(fColorMenu.hMenu, MF_STRING | MF_ENABLED, mv_cBlack, "&Black");

	// attatch the menu to the end of the base menu

	fMenuBar = fMySession->GetWindowState()->CopyBaseMenuBar();

	fMenuBar->AddMenuLast(m_ColorMenu, fColorMenu, this);

}

//-------------------------------------------------------------------------
// DisposePlatMenus
//
// About this method:
// Destroy the popup menu that was created in the method BuildPlatMenus.
//
//-------------------------------------------------------------------------

void NicePart::DisposePlatMenus()
{
	if(fColorMenu.hMenu)
		DestroyMenu(fColorMenu.hMenu);
}

//--------------------------------------------------------------------
// HandleEvent				-- OpenDoc API method --
//
// About this method:
// All events are passed the the HandleEvent method in an uncooked form.
// This means that the events will have all the information that they
// would have if they were directly retrieve from the event queue.
// The facet pointer may be kXMPNULL for events not based on geometry.
//
//--------------------------------------------------------------------

XMPBoolean NicePart::HandleEvent(XMPEventData event,
								XMPFrame *frame,
								XMPFacet *facet)
{
	XMPBoolean handled;

	switch (event->message)
	{
		case WM_LBUTTONDOWN:
			handled = this->HandleMouseDown(event, frame, facet);
			break;
		case WM_COMMAND:
			handled = this->HandleMenuEvent(event, frame);
			break;
	}

	return handled;
}

//--------------------------------------------------------------------
// HandleMouseDown
//
// About this method:
// When we receive a mouse down event we want to make sure that we have the focus.
//
//--------------------------------------------------------------------

XMPBoolean NicePart::HandleMouseDown(XMPEventData event,
									XMPFrame *frame,
									XMPFacet *facet)
{
XMPUnused(event);
XMPUnused(facet);

	if (!facet->GetWindow()->IsActive())
	{
		facet->GetWindow()->Select();
	}
	else if (!fMyFrameIsActive)
	{
		this->ActivateFrame(frame);
	}

	return kXMPTrue;
}

//--------------------------------------------------------------------
// HandleMenuEvent
//
// About this method:
// The command id for the menu events are retrieved differently for the various
// platforms so we use the method to extract the menu command id before we process
// the menu event.
//
//--------------------------------------------------------------------

XMPBoolean NicePart::HandleMenuEvent(XMPEventData event, XMPFrame *frame)
{
	XMPCommandID 	command;

	command = event->wParam;
	return ProcessMenuEvent(command, frame);
}

//-------------------------------------------------------------------------
// RenderContent
//
// About this method:
// I want to color the entire contents of my frame with the currently chosen color.
// The first thing to do is set up the drawing focus. This is done by using the
// CFocus object. When the CFocus object gets instantiated it saves off the current
// drawing focus and sets up the new drawing focus. When the CFocus's destructor
// gets called it returns us to the original focus. Since the CFocus object gets
// instantiated on the stack it's destructor gets called automatically for us.
// The drawing focus sets up the platform canvas with the correct clipping and a
// transformation.
//
//-------------------------------------------------------------------------

void NicePart::RenderContent(XMPFacet *facet, XMPFrame *frame, XMPShape *invalidShape)
{
	HDC					hdc;
	XMPShape 			*shape;
	XMPPlatformShape	platformShape;
	HBRUSH 				hbr = CreateSolidBrush(RGB(fMyState.contentColor.red,
																fMyState.contentColor.green,
																fMyState.contentColor.blue));

	CFocus f(facet, invalidShape, &hdc);

	// get the bounds of my frame shape
	shape = frame->GetFrameShape();
	platformShape = shape->GetWinRegion();

	// paint the background
	FillRgn(hdc, platformShape, hbr);

	DeleteObject(hbr);
}

//--------------------------------------------------------------------
// UpdateThisFrame
//
// About this method:
// Generate an update event for our document so that the region where our frame
// resides will be redrawn later in our draw method. Frame->Invalidate() requires
// an XMPShape pointer as the parameter.  If we pass NULL, the frame uses its shape
// instead.  Whenever we call Frame->Invalidate() the origin for the platform
// canvas must be set to 0, 0.  I use a facet iterator to get a facet so that I
// can get a platform canvas (HDC) in order to set the window origin.
//
//--------------------------------------------------------------------

void NicePart::UpdateThisFrame(XMPFrame *frame)
{

	HDC hdc;
	XMPFrameFacetIterator *facets;
	XMPFacet *MyFacet;

	facets = frame->CreateFacetIterator();
	MyFacet = (XMPFacet*) facets->First();
	hdc = MyFacet->GetCanvas()->GetPlatformCanvas();
#ifndef WIN32
	SetWindowOrg(hdc,0,0);
#else
	SetWindowOrgEx(hdc,0,0,NULL);
#endif
	frame->Invalidate(NULL);
	MyFacet->GetCanvas()->ReleasePlatformCanvas();
	delete facets;
}


