/*

	File:		DrPlExt.cpp

	Contains:	DrawPalette Extension class

	Written by:	Jeff Hanson
*/

#ifndef _ALTPOINT_
#include "AltPoint.h"
#endif

#ifndef SOM_ODToolSpaceManager_xh
#include <ODTlSpc.xh>
#endif

#ifndef SOM_ODToolSpaceObj_xh
#include <ODTlObj.xh>
#endif

#define DrawPaletteExt_Class_Source
#define VARIABLE_MACROS
#include <DrPlExt.xih>

#define kLeftMargin		4 
#define kTopMargin		4
#define kRightMargin	4 
#define kBottomMargin	4
#define kToolGap		4
#define kToolWidth		25
#define kToolHeight		25

class DrawToolInfo
{
public:
	DrawToolInfo() {	fID = (ODULong)kODNULL;
						fBitmap = kODNULL;
						fState = kDrawToolStateOff;}
	virtual ~DrawToolInfo() { }
	ODULong				fID;
	ODRect				fBounds;
	HBITMAP				fBitmap;
	ODUShort			fState;
};

//==============================================================================
// ToolInfoLink
//==============================================================================

class ToolInfoLink
{
public:
	
	ToolInfoLink*	Next( )	{return fNext;}
	ToolInfoLink*	Prev( ) {return fPrev;}
	DrawToolInfo*	GetToolInfo( ) {return fToolInfo;}

protected:
	ToolInfoLink( DrawToolInfo* aToolInfo, ToolInfoLink *list );
	ToolInfoLink( );
	~ToolInfoLink( );
				
	ToolInfoLink		*fPrev;
	ToolInfoLink		*fNext;
	friend class		ToolInfoList;
	DrawToolInfo*		fToolInfo;
};


//==============================================================================
// ControlList
//==============================================================================

class ToolInfoList : private ToolInfoLink
{
public:
	ToolInfoList( );
	~ToolInfoList( );

	void			Add(DrawToolInfo* aToolInfo);
	void			Remove(DrawToolInfo* aToolInfo);
	ToolInfoLink*	First( ) {return fNext;}
	ToolInfoLink*	Last( ) {return fPrev;}
};

//==============================================================================
// ToolInfoLink
//==============================================================================

ToolInfoLink::ToolInfoLink( ) :
	fToolInfo(kODNULL)
{
	fPrev = this;
	fNext = this;
}

ToolInfoLink::ToolInfoLink( DrawToolInfo* aToolInfo, ToolInfoLink *list ) :
	fToolInfo(aToolInfo),
	fPrev(list),
	fNext(list->fNext)
{
	list->fNext = this;
	fNext->fPrev = this;
}

ToolInfoLink::~ToolInfoLink( )
{
	if( fPrev ) {
		fPrev->fNext = fNext;
	}
	if( fNext ) {
		fNext->fPrev = fPrev;
	}
}


//==============================================================================
// ToolInfoList
//==============================================================================


ToolInfoList::ToolInfoList( ) :
	ToolInfoLink()
{
}


ToolInfoList::~ToolInfoList( )
{
	// Delete all links:
	while( fNext != this ) {
		delete fNext;
	}
}


void ToolInfoList::Add(DrawToolInfo* aToolInfo)
{
	new ToolInfoLink(aToolInfo, this);
}


void ToolInfoList::Remove(DrawToolInfo* aToolInfo)
{
	ToolInfoLink*	link;
	
	for( link = this->First(); link->GetToolInfo() != kODNULL; link = link->Next() ) {
		if(link->GetToolInfo() == aToolInfo) {
			delete link;
			return;
		}
	}
}


//-------------------------------------------------------------------------
// DrawPaletteExt::DrawPaletteExt
//-------------------------------------------------------------------------

SOM_Scope void  SOMLINK somInit(DrawPaletteExt *somSelf)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	DrawPaletteExt_parent_ODExtension_somInit(somSelf);

	_fPart = kODNULL;
	_fInitCompletedProc = kODNULL;
	_fDrawPaletteRect.Set(0, 0, 0, 0);
	_fToolSpaceObj = kODNULL;
	_fTSDomain = kODNoDomain;
	_fPos = kODTSNULL;
	_fToolSpaceWnd = kODTSNULL;
	_fPartFrameShape = kODTSNULL;
	_fSession = kODNULL;
	_fUserInfo = (ODInfoType)kODNULL;
	_fToolInfoList = kODNULL;
	_fToolWidth = IntToFixed(kToolWidth);
	_fToolHeight = IntToFixed(kToolHeight);
	_fFloatRect.Set(IntToFixed(100), IntToFixed(100), IntToFixed(220), IntToFixed(180));
}

//-------------------------------------------------------------------------
// DrawPaletteExt::~DrawPaletteExt
//-------------------------------------------------------------------------

SOM_Scope void  SOMLINK somUninit(DrawPaletteExt *somSelf)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	Environment* ev = somGetGlobalEnvironment();

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		somSelf->RemoveToolInfo(ev, (ODULong)(tiLnk->GetToolInfo()));
	}

	delete _fToolInfoList;

	DrawPaletteExt_parent_ODExtension_somUninit(somSelf);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::InitExtension
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK InitExtension(DrawPaletteExt *somSelf, Environment *ev, 
								ODObject *base)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	if (somSelf->IsInitialized(ev))
		return;
	
	DrawPaletteExt_parent_ODExtension_InitExtension(somSelf, ev, base);
	
	_fPart = base;
	_fToolInfoList = new ToolInfoList;	
}

//-------------------------------------------------------------------------
// DrawPaletteExt::Release
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK Release(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

    DrawPaletteExt_parent_ODExtension_Release(somSelf, ev);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::InstallCompletedProc
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK InstallCompletedProc(DrawPaletteExt *somSelf, Environment *ev, 
								InitCompletedProc anInitCompletedProc)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fInitCompletedProc = anInitCompletedProc;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::CallInitCompleted
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK CallInitCompleted(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	if (_fInitCompletedProc && _fPart) {
		(*_fInitCompletedProc)(_fPart);
	}
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetSize
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK GetSize(DrawPaletteExt *somSelf, Environment *ev, ODRect* anODRect) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	*anODRect = _fDrawPaletteRect;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetSize
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetSize(DrawPaletteExt *somSelf, Environment *ev, ODRect* anODRect) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fDrawPaletteRect = *anODRect;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetFloatRect
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK GetFloatRect(DrawPaletteExt *somSelf, Environment *ev, ODRect* anODRect) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	*anODRect = _fFloatRect;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetFloatRect
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetFloatRect(DrawPaletteExt *somSelf, Environment *ev, ODRect* anODRect) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fFloatRect = *anODRect;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolSpaceObj
//-------------------------------------------------------------------------

SOM_Scope ODToolSpaceObj* SOMLINK GetToolSpaceObj(DrawPaletteExt *somSelf, 
								Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fToolSpaceObj;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolSpaceObj
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolSpaceObj(DrawPaletteExt *somSelf, Environment *ev, 
								ODToolSpaceObj* aToolSpaceObj) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fToolSpaceObj = aToolSpaceObj;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetPos
//-------------------------------------------------------------------------

SOM_Scope ODToolSpacePos SOMLINK GetPos(DrawPaletteExt *somSelf, Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fPos;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetPos
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetPos(DrawPaletteExt *somSelf, Environment *ev, 
								ODToolSpacePos aPos) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fPos = aPos;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolSpaceWnd
//-------------------------------------------------------------------------

SOM_Scope ODPlatformWindow SOMLINK GetToolSpaceWnd(DrawPaletteExt *somSelf, 
								Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fToolSpaceWnd;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolSpaceWnd
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolSpaceWnd(DrawPaletteExt *somSelf, Environment *ev, ODPlatformWindow aToolSpaceWnd) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fToolSpaceWnd = aToolSpaceWnd;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetPartFrameShape
//-------------------------------------------------------------------------

SOM_Scope ODShape* SOMLINK GetPartFrameShape(DrawPaletteExt *somSelf, Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fPartFrameShape;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetPartFrameShape
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetPartFrameShape(DrawPaletteExt *somSelf, Environment *ev, 
								ODShape* aPartFrameShape) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fPartFrameShape = aPartFrameShape;
}
	
//-------------------------------------------------------------------------
// DrawPaletteExt::GetDomain
//-------------------------------------------------------------------------

SOM_Scope ODToolSpaceDomain SOMLINK GetDomain(DrawPaletteExt *somSelf, Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fTSDomain;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetDomain
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetDomain(DrawPaletteExt *somSelf, Environment *ev, 
								ODToolSpaceDomain aTSDomain) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fTSDomain = aTSDomain;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetSession
//-------------------------------------------------------------------------

SOM_Scope ODSession* SOMLINK GetSession(DrawPaletteExt *somSelf, Environment *ev) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fSession;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetSession
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetSession(DrawPaletteExt *somSelf, Environment *ev, ODSession* aSession) 
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fSession = aSession;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetUserInfo
//-------------------------------------------------------------------------

SOM_Scope ODInfoType SOMLINK GetUserInfo(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return _fUserInfo;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetUserInfo
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetUserInfo(DrawPaletteExt *somSelf, Environment *ev, ODInfoType anInfoPtr)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fUserInfo = anInfoPtr;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::AddToolInfo
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK AddToolInfo(DrawPaletteExt *somSelf, Environment *ev, 
				ODULong aToolID, ODRect* bounds, HBITMAP toolBMP, ODUShort state)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	DrawToolInfo*	aToolInfo = new DrawToolInfo;
	aToolInfo->fID = aToolID;
	aToolInfo->fBounds = *bounds;
	aToolInfo->fBitmap = toolBMP;
	aToolInfo->fState = state;
	_fToolInfoList->Add(aToolInfo);
	
	return((ODULong)aToolInfo);	
}

//-------------------------------------------------------------------------
// DrawPaletteExt::RemoveToolInfo
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK RemoveToolInfo(DrawPaletteExt *somSelf, Environment *ev, ODULong aToolInfo)
{ 
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fToolInfoList->Remove((DrawToolInfo*)aToolInfo);
	delete (DrawToolInfo*)aToolInfo;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetFirstTool
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK GetFirstTool(DrawPaletteExt *somSelf, Environment *ev)
{ 
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return((ODULong)(_fToolInfoList->First()->GetToolInfo()));
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetNextTool
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK GetNextTool(DrawPaletteExt *somSelf, Environment *ev, ODULong aTool)
{ 
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		if (tiLnk->GetToolInfo() == (DrawToolInfo*)aTool) {
			return((ODULong)(tiLnk->Next()->GetToolInfo()));
		}
	}

	return((ODULong)kODNULL);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetLastTool
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK GetLastTool(DrawPaletteExt *somSelf, Environment *ev)
{ 
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return((ODULong)(_fToolInfoList->Last()->GetToolInfo()));
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetNextToolBounds
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK GetNextToolBounds(DrawPaletteExt *somSelf, Environment *ev, ODRect* theBounds)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	ODRect	paletteBounds;
	somSelf->GetSize(ev, &paletteBounds);
	ODCoordinate	paletteWidth = (paletteBounds.right - paletteBounds.left);
	ODCoordinate	paletteHeight = (paletteBounds.bottom - paletteBounds.top);

	DrawToolInfo*	botRightToolInfo = _fToolInfoList->First()->GetToolInfo();
	Rect	botRightBoundsRect;
	SetRect(&botRightBoundsRect, kLeftMargin, kTopMargin,
									kLeftMargin + FixedToInt(_fToolWidth), kTopMargin + FixedToInt(_fToolHeight));

	ODBoolean	firstTool = kODTrue;

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		Rect	boundsRect;
		aDrawToolInfo->fBounds.AsWinRect(boundsRect);

		if (boundsRect.top >= botRightBoundsRect.bottom) {
			botRightBoundsRect = boundsRect;
		} else if ((boundsRect.right > botRightBoundsRect.right) && 
			(boundsRect.bottom >= botRightBoundsRect.bottom)) {
			botRightBoundsRect = boundsRect;
		}

		if (!aDrawToolInfo->fBounds.IsEmpty()) {
			firstTool = kODFalse;
		}
	}

	if (firstTool) {
		theBounds->Set(IntToFixed(botRightBoundsRect.left), IntToFixed(botRightBoundsRect.top), 
						IntToFixed(botRightBoundsRect.right), IntToFixed(botRightBoundsRect.bottom));	
	} else {
		if ((botRightBoundsRect.right + kToolGap + FixedToInt(_fToolWidth)) <= (FixedToInt(paletteWidth) - kRightMargin)) {
			theBounds->Set(IntToFixed(botRightBoundsRect.right + kToolGap), IntToFixed(botRightBoundsRect.top), 
							IntToFixed(botRightBoundsRect.right + kToolGap + FixedToInt(_fToolWidth)), 
							IntToFixed(botRightBoundsRect.bottom));	
		} else {
			theBounds->Set(IntToFixed(kLeftMargin), 
							IntToFixed(botRightBoundsRect.bottom + kToolGap), 
							IntToFixed(kLeftMargin + FixedToInt(_fToolWidth)), 
							IntToFixed(botRightBoundsRect.bottom + kToolGap + FixedToInt(_fToolHeight)));	
		}
	}
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolWidth
//-------------------------------------------------------------------------

SOM_Scope ODCoordinate SOMLINK GetToolWidth(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return(_fToolWidth);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolWidth
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolWidth(DrawPaletteExt *somSelf, Environment *ev, ODCoordinate aToolWidth)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fToolWidth = aToolWidth;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolHeight
//-------------------------------------------------------------------------

SOM_Scope ODCoordinate SOMLINK GetToolHeight(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return(_fToolHeight);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolHeight
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolHeight(DrawPaletteExt *somSelf, Environment *ev, ODCoordinate aToolHeight)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	_fToolHeight = aToolHeight;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::CreateTool
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK CreateTool(DrawPaletteExt *somSelf, Environment *ev, 
								ODULong toolID, HBITMAP toolBMP, ODUShort state)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	ODToolSpaceObj*	aToolSpaceObj = somSelf->GetToolSpaceObj(ev);
	if (!aToolSpaceObj) {
		return((ODULong)kODNULL);
	}

	ODPlatformWindow	toolSpaceWnd = aToolSpaceObj->GetPlatformWindow(ev);
	if (!toolSpaceWnd) {
		return((ODULong)kODNULL);
	}
	
	ODRect	ODBoundsRect;	
	somSelf->GetNextToolBounds(ev, &ODBoundsRect);

	return(somSelf->AddToolInfo(ev, toolID, &ODBoundsRect, toolBMP, state));
}

//-------------------------------------------------------------------------
// DrawPaletteExt::DestroyTool
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK DestroyTool(DrawPaletteExt *somSelf, Environment *ev, ODULong toolID)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		if (aDrawToolInfo->fID == toolID) {
			somSelf->RemoveToolInfo(ev, (ODULong)aDrawToolInfo);
			return;
		}
	}
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolState
//-------------------------------------------------------------------------

SOM_Scope ODUShort SOMLINK GetToolState(DrawPaletteExt *somSelf, Environment *ev, 
								ODULong aTool)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return(((DrawToolInfo*)aTool)->fState);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolState
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolState(DrawPaletteExt *somSelf, Environment *ev, 
								ODULong aTool, ODUShort state)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	((DrawToolInfo*)aTool)->fState = state;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::GetToolID
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK GetToolID(DrawPaletteExt *somSelf, Environment *ev, 
								ODULong aTool)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	return(((DrawToolInfo*)aTool)->fID);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::SetToolID
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK SetToolID(DrawPaletteExt *somSelf, Environment *ev, 
								ODULong aTool, ODULong anID)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	((DrawToolInfo*)aTool)->fID = anID;
}

//-------------------------------------------------------------------------
// DrawPaletteExt::PtInTool
//-------------------------------------------------------------------------

SOM_Scope ODULong SOMLINK PtInTool(DrawPaletteExt *somSelf, Environment *ev, ODPoint* aPnt)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		if (aDrawToolInfo->fBounds.Contains(*aPnt)) {
			return((ODULong)aDrawToolInfo);
		}
	}

	return((ODULong)kODNULL);
}

//-------------------------------------------------------------------------
// DrawPaletteExt::UpdateTools
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK UpdateTools(DrawPaletteExt *somSelf, Environment *ev, ODPlatformCanvas aCanvas)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	for (ToolInfoLink *tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		Rect	boundsRect;
		aDrawToolInfo->fBounds.AsWinRect(boundsRect);
							 
		InflateRect(&boundsRect, -1, -1);

		if (aDrawToolInfo->fState == kDrawToolStateOn) {
			FillRect(aCanvas, &boundsRect, (HBRUSH)GetStockObject(DKGRAY_BRUSH));
		} else {
			FillRect(aCanvas, &boundsRect, (HBRUSH)GetStockObject(GRAY_BRUSH));
		}

		InflateRect(&boundsRect, 1, 1);
		
		if (aDrawToolInfo->fBitmap) {
			HBITMAP	savedBMap;
			BITMAP	theBMap;
			HDC		memDC = CreateCompatibleDC(aCanvas);

			GetObject(aDrawToolInfo->fBitmap, sizeof(BITMAP), &theBMap);

			savedBMap = SelectObject(memDC, aDrawToolInfo->fBitmap);

			StretchBlt(aCanvas, boundsRect.left, boundsRect.top,
						(boundsRect.right - boundsRect.left), (boundsRect.bottom - boundsRect.top),
    					memDC, 0, 0, theBMap.bmWidth, theBMap.bmHeight, SRCCOPY);	

			SelectObject(memDC, savedBMap);
			DeleteDC(memDC);
		}
			
		if (aDrawToolInfo->fState == kDrawToolStateOn) {
			FrameRect(aCanvas, &boundsRect, (HBRUSH)GetStockObject(WHITE_BRUSH));
		} else {
			FrameRect(aCanvas, &boundsRect, (HBRUSH)GetStockObject(BLACK_BRUSH));
		}
		
	}
}

//-------------------------------------------------------------------------
// DrawPaletteExt::ArrangeTools
//-------------------------------------------------------------------------

SOM_Scope void SOMLINK ArrangeTools(DrawPaletteExt *somSelf, Environment *ev)
{
	DrawPaletteExtData *somThis = DrawPaletteExtGetData(somSelf);

	ToolInfoLink*	tiLnk;

	// first set all bounds to topleft 

	for (tiLnk = _fToolInfoList->First(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Next()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		aDrawToolInfo->fBounds.Set(0, 0, 0, 0);
	}

	// now, calc new bounds rects

	for (tiLnk = _fToolInfoList->Last(); tiLnk->GetToolInfo() != kODNULL; tiLnk = tiLnk->Prev()) {
		DrawToolInfo*	aDrawToolInfo = tiLnk->GetToolInfo();
		somSelf->GetNextToolBounds(ev, &aDrawToolInfo->fBounds);
	}
}
