/*
	File:		DrawShap.cpp

	Contains:	DrawShape class used with DrawPart

	Written by:	Jeff Hanson	 

*/

#ifndef _ALTPOINT_
#include "AltPoint.h"			// Use C++ savvy ODPoint and ODRect
#endif

#ifndef _ODTYPES_
#include <ODTypes.h>
#endif

#ifndef _ODUTILW_
#include "ODUtilW.h"
#endif

#ifndef _FOCUSLIB_
#include "FocusLib.h"
#endif

#ifndef SOM_ODFacet_xh
#include <Facet.xh>
#endif

#ifndef SOM_ODFrame_xh
#include <Frame.xh>
#endif

#ifndef SOM_ODShape_xh
#include <Shape.xh>
#endif

#ifndef SOM_ODWindow_xh
#include <Window.xh>
#endif

#ifndef SOM_ODTransform_xh
#include <Trnsform.xh>
#endif

#ifndef _DRAWSHAP_
#include "DrawShap.h"
#endif

COLORREF DrawShape::fDefaultFillColor = RGB(255, 255, 255);

//-------------------------------------------------------------------------------------
// StillDown
//-------------------------------------------------------------------------------------

static BOOL StillDown(ODPlatformWindow hwnd)
{
	MSG		msg;

	if (PeekMessage(&msg, hwnd, WM_LBUTTONUP, WM_LBUTTONUP, PM_NOYIELD | PM_REMOVE)) {
		return(FALSE);
	}

	return(TRUE);
}

//==============================================================================
// DrawShape
//==============================================================================


DrawShape::DrawShape()
{
	fFacet = kODNULL;
	fBounds.Set(0, 0, 0, 0);
	fFillColor = DrawShape::fDefaultFillColor;
	fType = kNullShape;
	fRgn = kODNULL;
	fSelected = kODFalse;
	fStartPt.x = 0; fStartPt.y = 0;
	fEndPt.x = 0; fEndPt.y = 0;
}


DrawShape::DrawShape(ODFacet* facet, ODRect* bounds, COLORREF fillColor, DrawShapeType aType)
{
	fFacet = facet;
	fBounds = *bounds;
	fFillColor = fillColor;
	fType = aType;
	fRgn = kODNULL;
	fSelected = kODFalse;
	fStartPt.x = 0; fStartPt.y = 0;
	fEndPt.x = 0; fEndPt.y = 0;
}


DrawShape::~DrawShape()
{
	if (fRgn) {
		DeleteObject((HGDIOBJ)fRgn);
	}
}

ODRgnHandle DrawShape::CreateShapeRgn(DrawShapeType theType, Rect* aBoundsRect, ODPoint* startPt, ODPoint* endPt)
{
	ODRgnHandle	aRgn = kODNULL;

	switch (theType) {
		case kLineShape:
		{
			Point	ptArray[4];
			Point	start1 = startPt->AsWinPoint();
			Point	start2 = startPt->AsWinPoint();
			Point	end1 = endPt->AsWinPoint();
			Point	end2 = endPt->AsWinPoint();
			start1.x -= 2;
			start1.y += 1;
			start2.x += 2;
			start2.y -= 1;
			end1.x -= 2;
			end1.y += 1;
			end2.x += 2;
			end2.y -= 1;
			ptArray[0] = start1;
			ptArray[1] = end1;
			ptArray[2] = end2;
			ptArray[3] = start2;
			aRgn = CreatePolygonRgn(ptArray, 4, WINDING);
		}	
			break;

		case kRectShape:
		case kRectFillShape:
			aRgn = CreateRectRgn(aBoundsRect->left, aBoundsRect->top, 
									aBoundsRect->right, aBoundsRect->bottom);
			break;

		case kRoundRectShape:
		case kRoundRectFillShape:
			aRgn = CreateRoundRectRgn(aBoundsRect->left, aBoundsRect->top, 
									aBoundsRect->right, aBoundsRect->bottom, 15, 15);
			break;

		case kEllipseShape:
		case kEllipseFillShape:
			aRgn = CreateEllipticRgn(aBoundsRect->left, aBoundsRect->top, 
									aBoundsRect->right, aBoundsRect->bottom);
			break;
	}

	return(aRgn);
}

ODRgnHandle DrawShape::CreateBorderRgn(DrawShapeType theType, Rect* aBoundsRect, ODPoint* startPt, ODPoint* endPt)
{
	ODRgnHandle	aRgn = kODNULL;

	switch (theType) {
		case kLineShape:
			aRgn = CreateShapeRgn(theType, aBoundsRect, startPt, endPt);
			break;

		case kRectShape:
		case kRoundRectShape:
		case kEllipseShape:
		case kRectFillShape:
		case kRoundRectFillShape:
		case kEllipseFillShape:
		{
			Rect	bounds1 = *aBoundsRect;	
			Rect	bounds2 = *aBoundsRect;	
			InflateRect(&bounds1, 2, 2); 		// inflate
			InflateRect(&bounds2, -2, -2); 		// inset
			ODRgnHandle	rgn1 = CreateShapeRgn(theType, &bounds1, startPt, endPt);
			if (!rgn1) {
				return(kODNULL);
			}
			ODRgnHandle	rgn2 = CreateShapeRgn(theType, &bounds2, startPt, endPt);
			if (!rgn2) {
				DeleteObject(rgn1);
				return(kODNULL);
			}
			aRgn = CreateRectRgn(0, 0, 0, 0);
			int retVal = CombineRgn(aRgn, rgn1, rgn2, RGN_DIFF);
			if ((retVal == ERROR) || (retVal == NULLREGION)) {
				aRgn = kODNULL;
			}
			DeleteObject(rgn1);
			DeleteObject(rgn2);
		}
			break;
	}

	return(aRgn);
}

void DrawShape::SetBounds(ODRect* newBounds)
{
	fBounds = *newBounds;
	if (fRgn) {
		DeleteObject((HGDIOBJ)fRgn);
	}
	Rect	bounds;
	newBounds->AsWinRect(bounds);
	fRgn = CreateShapeRgn(fType, &bounds, &fStartPt, &fEndPt);
}

ODBoolean DrawShape::DragCreate(Environment* ev, ODFacet* facet, ODPoint* downPnt, DrawShapeType theType)
{
	Point				startPt;
	Point				oldPt;
	Point				newPt;
	ODPlatformWindow	theWnd;

	if (!facet) {
		return(kODFalse);
	}

	if (theType == kNullShape) {
		return(kODFalse);
	}

	theWnd = facet->GetWindow(ev)->GetPlatformWindow(ev);

	ODPlatformWindow	prevCapture = SetCapture(theWnd);
	HDC					hDC;

	hDC = GetDC(theWnd);

	GetCursorPos(&oldPt);
	ScreenToClient(theWnd, &oldPt);
	startPt = oldPt;
	newPt = oldPt;

	int		prevDrawMode = SetROP2(hDC, R2_NOT);
	HBRUSH	theBrush = GetStockObject(GRAY_BRUSH);

	Rect	bounds;
	SetRect(&bounds, startPt.x, startPt.y, startPt.x + 1, startPt.y + 1);
	ODRgnHandle	wndRgn = CreateShapeRgn(theType, &bounds, &fStartPt, &fEndPt);

	if (theType == kLineShape) {
		MoveToEx(hDC, startPt.x, startPt.y, kODNULL);
		LineTo(hDC, oldPt.x, oldPt.y);
	} else {
		FrameRgn(hDC, wndRgn, theBrush, 1, 1);
	}

	while (StillDown(theWnd)) {
		GetCursorPos(&newPt);
		ScreenToClient(theWnd, &newPt);
		if ((newPt.x != oldPt.x) || (newPt.y != oldPt.y)) {
			if (theType == kLineShape) {
				MoveToEx(hDC, startPt.x, startPt.y, kODNULL);
				LineTo(hDC, oldPt.x, oldPt.y);
			} else {
				FrameRgn(hDC, wndRgn, theBrush, 1, 1);		// erase old
			}

			DeleteObject((HGDIOBJ)wndRgn);
			SetRect(&bounds, startPt.x, startPt.y, newPt.x, newPt.y);
			ODPoint	newODPnt(IntToFixed(newPt.x), IntToFixed(newPt.y));
			wndRgn = CreateShapeRgn(theType, &bounds, &fStartPt, &newODPnt);

			if (theType == kLineShape) {
				MoveToEx(hDC, startPt.x, startPt.y, kODNULL);
				LineTo(hDC, newPt.x, newPt.y);
			} else {
				FrameRgn(hDC, wndRgn, theBrush, 1, 1);		// draw new
			}

			oldPt = newPt;
		}
	}

	if (theType == kLineShape) {
		MoveToEx(hDC, startPt.x, startPt.y, kODNULL);
		LineTo(hDC, oldPt.x, oldPt.y);
	} else {
		FrameRgn(hDC, wndRgn, theBrush, 1, 1);		// erase old
	}

	SetROP2(hDC, prevDrawMode);
	SetCapture(prevCapture);
	ReleaseDC(theWnd, hDC);

	Rect	tempRect;
	GetRgnBox(wndRgn, &tempRect);

	ODBoolean	goodShape = kODFalse;
	if (theType == kLineShape) {
		if ((startPt.x < (newPt.x - 1)) || (startPt.x > (newPt.x + 1)) || 
			(startPt.y < (newPt.y - 1)) || (startPt.y > (newPt.y + 1))) {
			goodShape = kODTrue;
		}
	} else {
		if ((tempRect.right > (tempRect.left + 1)) && (tempRect.bottom > (tempRect.top + 1))) {
			goodShape = kODTrue;
		}
	}
		
	if (goodShape) {
		// transform points
		ODPoint	anODPnt1(IntToFixed(startPt.x), IntToFixed(startPt.y));
		ODPoint	anODPnt2(IntToFixed(newPt.x), IntToFixed(newPt.y));
		ODTransform*	wcXForm = facet->GetWindowContentTransform(ev, kODNULL);
		anODPnt1 = wcXForm->InvertPoint(ev, &anODPnt1);
		anODPnt2 = wcXForm->InvertPoint(ev, &anODPnt2);
		wcXForm->Release(ev);
		startPt = anODPnt1.AsWinPoint();
		newPt = anODPnt2.AsWinPoint();
		SetRect(&bounds, startPt.x, startPt.y, newPt.x, newPt.y);
		fStartPt.Set(IntToFixed(startPt.x), IntToFixed(startPt.y));
		fEndPt.Set(IntToFixed(newPt.x), IntToFixed(newPt.y));
		DeleteObject((HGDIOBJ)wndRgn);
		wndRgn = CreateShapeRgn(theType, &bounds, &fStartPt, &fEndPt);

		GetRgnBox(wndRgn, &tempRect);
		fBounds.Set(IntToFixed(tempRect.left), IntToFixed(tempRect.top), 
							IntToFixed(tempRect.right), IntToFixed(tempRect.bottom));
		fFacet = facet;
		fType = theType;
		fRgn = wndRgn;

		return(kODTrue);
	}

	DeleteObject((HGDIOBJ)wndRgn);
	return(kODFalse);
}


void DrawShape::DrawGrowRects(ODPlatformCanvas aCanvas)
{
	Rect	growRectArr[8];

	this->GetGrowRects(growRectArr);

	switch (fType) {
		case kLineShape:
			FillRect(aCanvas, &growRectArr[0], GetStockObject(BLACK_BRUSH));
			FillRect(aCanvas, &growRectArr[1], GetStockObject(BLACK_BRUSH));
			break;

		case kRectShape:
		case kRectFillShape:
		{
			ODSShort	i;
			for (i = 0; i < 8; i++) {
				FillRect(aCanvas, &growRectArr[i], GetStockObject(BLACK_BRUSH));
			}
		}
			break;

		case kRoundRectShape:
		case kRoundRectFillShape:
		case kEllipseShape:
		case kEllipseFillShape:
			FillRect(aCanvas, &growRectArr[1], GetStockObject(BLACK_BRUSH));
			FillRect(aCanvas, &growRectArr[3], GetStockObject(BLACK_BRUSH));
			FillRect(aCanvas, &growRectArr[5], GetStockObject(BLACK_BRUSH));
			FillRect(aCanvas, &growRectArr[7], GetStockObject(BLACK_BRUSH));
			break;
	}
}

void DrawShape::Draw(ODPlatformCanvas aCanvas)
{
	if (fRgn) {
		if (fType == kLineShape) {
			Point	startPt = fStartPt.AsWinPoint();
			Point	endPt = fEndPt.AsWinPoint();
			HPEN	selectPen;
			HGDIOBJ	prevPen;

			if (fSelected) {
				selectPen = CreatePen(PS_SOLID, 3, RGB(128, 128, 128));
				prevPen = SelectObject(aCanvas, (HGDIOBJ)selectPen);
			}
				
			MoveToEx(aCanvas, startPt.x, startPt.y, kODNULL);
			LineTo(aCanvas, endPt.x, endPt.y);

			if (fSelected) {
				this->DrawGrowRects(aCanvas);			
				SelectObject(aCanvas, (HGDIOBJ)prevPen);
				DeleteObject((HGDIOBJ)selectPen);
			}
		} else {
			HBRUSH	fillBrush = CreateSolidBrush(fFillColor);
			if ((fType == kRectFillShape) || (fType == kRoundRectFillShape) || (fType == kEllipseFillShape)) {
				FillRgn(aCanvas, fRgn, fillBrush);
			}
			if (fSelected) {
				FrameRgn(aCanvas, fRgn, GetStockObject(GRAY_BRUSH), 3, 3);
				this->DrawGrowRects(aCanvas);			
			} else {
				FrameRgn(aCanvas, fRgn, GetStockObject(BLACK_BRUSH), 1, 1);
			}
			DeleteObject((HGDIOBJ)fillBrush);
		}
	}
}

void DrawShape::GetGrowRects(Rect* growRectArr)
{
	Rect	winBounds;
	fBounds.AsWinRect(winBounds);

	switch (fType) {
		case kLineShape:
			SetRect(&growRectArr[0], FixedToInt(fStartPt.x) - 1, FixedToInt(fStartPt.y) - 1, 
										FixedToInt(fStartPt.x) + 1, FixedToInt(fStartPt.y) + 1);
			SetRect(&growRectArr[1], FixedToInt(fEndPt.x) - 1, FixedToInt(fEndPt.y) - 1, 
										FixedToInt(fEndPt.x) + 1, FixedToInt(fEndPt.y) + 1);
			break;

		case kRectShape:
		case kRectFillShape:
			SetRect(&growRectArr[0], winBounds.left, winBounds.top, 
									winBounds.left + 3, winBounds.top + 3);
			SetRect(&growRectArr[1], (winBounds.left + ((winBounds.right - winBounds.left) / 2)) - 2, winBounds.top, 
									(winBounds.left + ((winBounds.right - winBounds.left) / 2)) + 2, winBounds.top + 3);
			SetRect(&growRectArr[2], winBounds.right - 3, winBounds.top, 
									winBounds.right, winBounds.top + 3);
			SetRect(&growRectArr[3], winBounds.right - 3, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) - 2, 
									winBounds.right, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) + 2);
			SetRect(&growRectArr[4], winBounds.right - 3, winBounds.bottom - 3, 
									winBounds.right, winBounds.bottom);
			SetRect(&growRectArr[5], (winBounds.left + ((winBounds.right - winBounds.left) / 2)) - 2, winBounds.bottom - 3, 
									(winBounds.left + ((winBounds.right - winBounds.left) / 2)) + 2, winBounds.bottom);
			SetRect(&growRectArr[6], winBounds.left, winBounds.bottom - 3, winBounds.left + 3, winBounds.bottom);
			SetRect(&growRectArr[7], winBounds.left, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) - 2, 
									winBounds.left + 3, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) + 2);
			break;

		case kRoundRectShape:
		case kRoundRectFillShape:
		case kEllipseShape:
		case kEllipseFillShape:
			SetRect(&growRectArr[0], 0, 0, 0, 0);
			SetRect(&growRectArr[1], (winBounds.left + ((winBounds.right - winBounds.left) / 2)) - 2, winBounds.top, 
									(winBounds.left + ((winBounds.right - winBounds.left) / 2)) + 2, winBounds.top + 3);
			SetRect(&growRectArr[2], 0, 0, 0, 0);
			SetRect(&growRectArr[3], winBounds.right - 4, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) - 2, 
									winBounds.right - 1, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) + 2);
			SetRect(&growRectArr[4], 0, 0, 0, 0);
			SetRect(&growRectArr[5], (winBounds.left + ((winBounds.right - winBounds.left) / 2)) - 2, winBounds.bottom - 3, 
									(winBounds.left + ((winBounds.right - winBounds.left) / 2)) + 2, winBounds.bottom);
			SetRect(&growRectArr[6], 0, 0, 0, 0);
			SetRect(&growRectArr[7], winBounds.left, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) - 2, 
									winBounds.left + 3, (winBounds.top + ((winBounds.bottom - winBounds.top) / 2)) + 2);
			break;
	}
}

ODBoolean DrawShape::ContainsPoint(ODPoint* aPnt)
{
	if (fRgn) {
		if (PtInRegion(fRgn, FixedToInt(aPnt->x), FixedToInt(aPnt->y))) {
			return(kODTrue);
		}	
	}

	return(kODFalse);
}


void DrawShape::Invalidate(Environment* ev, ODFrame* frame)
{
	ODShape*	tmpShape = frame->CreateShape(ev);
	tmpShape->SetRectangle(ev, &fBounds);
	frame->Invalidate(ev, tmpShape, kODNULL);
	tmpShape->Release(ev);
}


ODBoolean DrawShape::PtInBorder(ODPoint* aPnt)
{
	if (!this->ContainsPoint(aPnt)) {
		return(kODFalse);
	}

	switch (fType) {
		case kLineShape:
			if (PtInRegion(fRgn, FixedToInt(aPnt->x), FixedToInt(aPnt->y))) {
				return(kODTrue);
			}	
			break;

		case kRectShape:
		case kRoundRectShape:
		case kEllipseShape:
		case kRectFillShape:
		case kRoundRectFillShape:
		case kEllipseFillShape:
		{
			Rect	bounds;
			fBounds.AsWinRect(bounds);
			ODRgnHandle	borderRgn = CreateBorderRgn(fType, &bounds, &fStartPt, &fEndPt);
			if (borderRgn) {
				if (PtInRegion(borderRgn, FixedToInt(aPnt->x), FixedToInt(aPnt->y))) {
					DeleteObject(borderRgn);
					return(kODTrue);
				}	
				DeleteObject(borderRgn);
			}
		}
			break;
	}

	return(kODFalse);
}


ODSShort DrawShape::PtInResizeBox(ODPoint* aPnt)
{
	if (!this->ContainsPoint(aPnt)) {
		return(kODFalse);
	}

	if (!this->PtInBorder(aPnt)) {
		return(kODFalse);
	}

	Rect	growRectArr[8];
	Point	winPnt = aPnt->AsWinPoint();

	switch (fType) {
		case kLineShape:
			this->GetGrowRects(growRectArr);
			if (PtInRect(&growRectArr[0], winPnt)) {
				return(1);
			}	
			if (PtInRect(&growRectArr[1], winPnt)) {
				return(2);
			}	
			break;

		case kRectShape:
		case kRectFillShape:
		{
			this->GetGrowRects(growRectArr);
			ODSShort	i;
			for (i = 0; i < 8; i++) {
				if (PtInRect(&growRectArr[i], winPnt)) {
					return(i + 1);
				}	
			}
		}
			break;

		case kRoundRectShape:
		case kRoundRectFillShape:
		case kEllipseShape:
		case kEllipseFillShape:
			this->GetGrowRects(growRectArr);
			if (PtInRect(&growRectArr[1], winPnt)) {
				return(2);
			}	
			if (PtInRect(&growRectArr[3], winPnt)) {
				return(4);
			}	
			if (PtInRect(&growRectArr[5], winPnt)) {
				return(6);
			}	
			if (PtInRect(&growRectArr[7], winPnt)) {
				return(8);
			}	
			break;
	}

	return(0);
}


void DrawShape::DragMove(Environment* ev, ODFacet* facet, ODPoint* downPnt)
{
	Point				startPt;
	Point				oldPt;
	Point				newPt;
	Point				lineStart;
	Point				lineEnd;
	ODPlatformWindow	theWnd;

	if (!facet) {
		return;
	}

	this->Invalidate(ev, facet->GetFrame(ev));

	theWnd = facet->GetWindow(ev)->GetPlatformWindow(ev);

	ODPlatformWindow	prevCapture = SetCapture(theWnd);
	HDC					hDC;

	CFocus	f(facet, kODNULL, &hDC);

	GetCursorPos(&oldPt);
	ScreenToClient(theWnd, &oldPt);
	newPt = oldPt;
	startPt = oldPt;

	int		prevDrawMode = SetROP2(hDC, R2_NOT);
	HBRUSH	theBrush = GetStockObject(GRAY_BRUSH);

	Rect	bounds;
	GetRgnBox(fRgn, &bounds);

	ODRgnHandle	wndRgn = kODNULL;

	if (fType != kLineShape) {
		wndRgn = CreateShapeRgn(fType, &bounds, &fStartPt, &fEndPt);
	}

	lineStart.x = FixedToInt(fStartPt.x);
	lineStart.y = FixedToInt(fStartPt.y);
	lineEnd.x = FixedToInt(fEndPt.x);
	lineEnd.y = FixedToInt(fEndPt.y);

	ODFacet*	rootFacet = facet->GetWindow(ev)->GetRootFacet(ev);

	Point oldOffset;
	oldOffset.x = 0; oldOffset.y = 0;

	while (StillDown(theWnd)) {
		GetCursorPos(&newPt);
		ScreenToClient(theWnd, &newPt);
		if ((newPt.x != oldPt.x) || (newPt.y != oldPt.y)) {
			if (fType == kLineShape) {
				MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
				LineTo(hDC, lineEnd.x, lineEnd.y);
			} else {
				FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// erase old
			}

			ODPoint	anODPnt1(IntToFixed(newPt.x - startPt.x), IntToFixed(newPt.y - startPt.y));
			ODTransform*	xform = rootFacet->GetExternalTransform(ev, kODNULL);
			anODPnt1 = xform->InvertPoint(ev, &anODPnt1);
			xform->Release(ev);

			Point offset;
			offset.x = FixedToInt(anODPnt1.x);
			offset.y = FixedToInt(anODPnt1.y);
			
			if (fType == kLineShape) {
				lineStart.x = FixedToInt(fStartPt.x) + offset.x;
				lineStart.y = FixedToInt(fStartPt.y) + offset.y;
				lineEnd.x = FixedToInt(fEndPt.x) + offset.x;
				lineEnd.y = FixedToInt(fEndPt.y) + offset.y;
				MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
				LineTo(hDC, lineEnd.x, lineEnd.y);
			} else {
				OffsetRgn(wndRgn, -oldOffset.x, -oldOffset.y);
				OffsetRgn(wndRgn, offset.x, offset.y);
				FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// draw new
			}

			oldPt = newPt;
			oldOffset = offset;
		}
	}

	if (fType == kLineShape) {
		MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
		LineTo(hDC, lineEnd.x, lineEnd.y);
	} else {
		FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// erase old
	}

	SetROP2(hDC, prevDrawMode);
	SetCapture(prevCapture);

	Rect	tempRect;
	GetRgnBox(wndRgn, &tempRect);
	DeleteObject((HGDIOBJ)fRgn);
	if (fType != kLineShape) {
		fBounds.Set(IntToFixed(tempRect.left), IntToFixed(tempRect.top), 
							IntToFixed(tempRect.right), IntToFixed(tempRect.bottom));
		fRgn = wndRgn;
	} else {
		fStartPt.Set(IntToFixed(lineStart.x), IntToFixed(lineStart.y));
		fEndPt.Set(IntToFixed(lineEnd.x), IntToFixed(lineEnd.y));
		fRgn = CreateShapeRgn(fType, &tempRect, &fStartPt, &fEndPt);
		GetRgnBox(fRgn, &tempRect);
		fBounds.Set(IntToFixed(tempRect.left), IntToFixed(tempRect.top), 
						IntToFixed(tempRect.right), IntToFixed(tempRect.bottom));
	}

	this->Invalidate(ev, facet->GetFrame(ev));
}


void DrawShape::DragResize(Environment* ev, ODFacet* facet, ODPoint* downPnt)
{
	Point				startPt;
	Point				oldPt;
	Point				newPt;
	Point				lineStart;
	Point				lineEnd;
	ODPlatformWindow	theWnd;

	if (!facet) {
		return;
	}

	ODSShort	growBoxIdx = this->PtInResizeBox(downPnt);
	if (!growBoxIdx) {
		return;
	}

	this->Invalidate(ev, facet->GetFrame(ev));

	theWnd = facet->GetWindow(ev)->GetPlatformWindow(ev);

	ODPlatformWindow	prevCapture = SetCapture(theWnd);
	HDC					hDC;

	CFocus	f(facet, kODNULL, &hDC);

	GetCursorPos(&oldPt);
	ScreenToClient(theWnd, &oldPt);
	newPt = oldPt;
	startPt = oldPt;

	int		prevDrawMode = SetROP2(hDC, R2_NOT);
	HBRUSH	theBrush = GetStockObject(GRAY_BRUSH);

	Rect	oldBounds;
	Rect	bounds;
	GetRgnBox(fRgn, &bounds);
	oldBounds = bounds;
	
	ODRgnHandle	wndRgn = kODNULL;

	if (fType != kLineShape) {
		wndRgn = CreateShapeRgn(fType, &bounds, &fStartPt, &fEndPt);
	}

	lineStart.x = FixedToInt(fStartPt.x);
	lineStart.y = FixedToInt(fStartPt.y);
	lineEnd.x = FixedToInt(fEndPt.x);
	lineEnd.y = FixedToInt(fEndPt.y);

	ODFacet*	rootFacet = facet->GetWindow(ev)->GetRootFacet(ev);

	Point oldOffset;
	oldOffset.x = 0; oldOffset.y = 0;

	while (StillDown(theWnd)) {
		GetCursorPos(&newPt);
		ScreenToClient(theWnd, &newPt);
		if ((newPt.x != oldPt.x) || (newPt.y != oldPt.y)) {
			if (fType == kLineShape) {
				MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
				LineTo(hDC, lineEnd.x, lineEnd.y);
			} else {
				FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// erase old
			}

			ODPoint	anODPnt1(IntToFixed(newPt.x - startPt.x), IntToFixed(newPt.y - startPt.y));
			ODTransform*	xform = rootFacet->GetExternalTransform(ev, kODNULL);
			anODPnt1 = xform->InvertPoint(ev, &anODPnt1);
			xform->Release(ev);

			Point offset;
			offset.x = FixedToInt(anODPnt1.x);
			offset.y = FixedToInt(anODPnt1.y);
			
			if (fType == kLineShape) {
				if (growBoxIdx == 2) {
					lineEnd.x = FixedToInt(fEndPt.x);
					lineEnd.y = FixedToInt(fEndPt.y);
					lineEnd.x += offset.x;
					lineEnd.y += offset.y;
				} else {
					lineStart.x = FixedToInt(fStartPt.x);
					lineStart.y = FixedToInt(fStartPt.y);
					lineStart.x += offset.x;
					lineStart.y += offset.y;
				}
				MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
				LineTo(hDC, lineEnd.x, lineEnd.y);
			} else {
				if (growBoxIdx == 1) {			// topLeft
					bounds.left = oldBounds.left + offset.x;
					bounds.top = oldBounds.top + offset.y;
				} else if (growBoxIdx == 2) {	// top
					bounds.top = oldBounds.top + offset.y;
				} else if (growBoxIdx == 3) {	// topRight
					bounds.right = oldBounds.right + offset.x;
					bounds.top = oldBounds.top + offset.y;
				} else if (growBoxIdx == 4) {	// right
					bounds.right = oldBounds.right + offset.x;
				} else if (growBoxIdx == 5) {	// bottomRight
					bounds.right = oldBounds.right + offset.x;
					bounds.bottom = oldBounds.bottom + offset.y;
				} else if (growBoxIdx == 6) {	// bottom
					bounds.bottom = oldBounds.bottom + offset.y;
				} else if (growBoxIdx == 7) {	// bottomLeft
					bounds.left = oldBounds.left + offset.x;
					bounds.bottom = oldBounds.bottom + offset.y;
				} else if (growBoxIdx == 8) {	// left
					bounds.left = oldBounds.left + offset.x;
				}

				DeleteObject((HGDIOBJ)wndRgn);
				wndRgn = CreateShapeRgn(fType, &bounds, &fStartPt, &fEndPt);
				FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// draw new
			}

			oldPt = newPt;
			oldOffset = offset;
		}
	}

	if (fType == kLineShape) {
		MoveToEx(hDC, lineStart.x, lineStart.y, kODNULL);
		LineTo(hDC, lineEnd.x, lineEnd.y);
	} else {
		FrameRgn(hDC, wndRgn, theBrush, 2, 2);		// erase old
	}

	SetROP2(hDC, prevDrawMode);
	SetCapture(prevCapture);

	DeleteObject((HGDIOBJ)fRgn);
	if (fType != kLineShape) {
		fRgn = wndRgn;
		fBounds.Set(IntToFixed(bounds.left), IntToFixed(bounds.top), 
						IntToFixed(bounds.right), IntToFixed(bounds.bottom));
	} else {
		fStartPt.Set(IntToFixed(lineStart.x), IntToFixed(lineStart.y));
		fEndPt.Set(IntToFixed(lineEnd.x), IntToFixed(lineEnd.y));
		fRgn = CreateShapeRgn(fType, &bounds, &fStartPt, &fEndPt);
		GetRgnBox(fRgn, &bounds);
		fBounds.Set(IntToFixed(bounds.left), IntToFixed(bounds.top), 
						IntToFixed(bounds.right), IntToFixed(bounds.bottom));
	}

	this->Invalidate(ev, facet->GetFrame(ev));
}


