/* FILE: DragPart.CPP                                                                                           TABSTOP=4
 *
 * DESCRIPTION:
 *   Simple text part that supports drag and drop of text. In the Apple
 *   implementation, this part uses a window control that is a simple text
 *   editor. Since nothing was available, this text editing part is its own
 *   very simple toy text editor. It doesn't attempt to be efficient in things
 *   like text size calculations - it is merely a testcase used in testing
 *   features in OpenDoc for Windows.
 *
 *   This part demonstrates the following features:
 *     - Selection of text within a line
 *     - Selection of font. Uses scalable fonts only, to avoid scaling
 *       difficulties that occur when printing or drawing into OLE metafiles
 *     - Clipboard cut/copy/paste of text and font, allows exchange with
 *       other Windows applications.
 *     - Drag+Drop of text to other OpenDoc parts and OLE2 compliant apps
 *       such as Microsoft Word.
 *     - OLE interoperability. This part works in OLE, and calls the draft's
 *       SetChangedFromPrev method each time its content is changed (ie: every
 *       time the user presses a key) so that the OLE cached presentation is
 *       updated on a regular basis by OpenDoc.
 *
 *   The following functions are yet to be implemented:
 *     - Text wrapping
 *     - Mixing fonts within a line
 *
 * HISTORY:
 *   02/20/95 : Chris Andrew        - Ported to SOM32 Windows version
 *   04/20/94 : Chris Andrew        - Ported to Windows
 *   12/15/93 : Paul Hartenstine    - Original Apple Code
 */
#ifndef _ALTPOINT_
#include "AltPoint.h"                   // Use C++ savvy ODPoint and ODRect
#endif

#ifndef _ODTYPES_
#include "ODTypes.h"
#endif

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _DRWSHARE_
#include "DrwShare.h"
#endif

#define SOM_Module_dragpart_Source
#define VARIABLE_MACROS
#include <DragPart.xih>

#ifndef _ODUTILW_
#include "ODUtilW.h"
#endif

#ifndef SOM_Module_OpenDoc_Foci_defined
#include <Foci.xh>
#endif

#ifndef SOM_ODDragItemIterator_xh
#include "DgItmIt.xh"
#endif

#ifndef SOM_ODFocusSet_xh
#include <FocusSet.xh>
#endif

#ifndef SOM_Module_OpenDoc_Commands_defined
#include <CmdDefs.xh>
#endif

#ifndef SOM_ODArbitrator_xh
#include <Arbitrat.xh>
#endif

#ifndef SOM_ODInfo_xh
#include "Info.xh"
#endif

#ifndef _INFOUTIL_
#include "InfoUtil.h"
#endif

#ifndef _ISOSTRING_
#include "ISOStr.h"
#endif

#ifndef _STDIDSW_
#include "StdIDsW.h"
#endif

#ifndef _ORDCOLL_
#include "OrdColl.h"
#endif

#ifndef SOM_ODDraft_xh
#include <Draft.xh>
#endif

#ifndef SOM_ODDispatcher_xh
#include <Disptch.xh>
#endif

#ifndef _EXCEPT_
#include "Except.h"
#endif

#ifndef _ODMEMORY_
#include "ODMemory.h"
#endif

#ifndef _FOCUSLIB_
#include "FocusLib.h"
#endif

#ifndef SOM_ODFacet_xh
#include <Facet.xh>
#endif

#ifndef SOM_ODFacetIterator_xh
#include "FacetItr.xh"
#endif

#ifndef SOM_ODFrame_xh
#include <Frame.xh>
#endif

#ifndef SOM_ODShape_xh
#include <Shape.xh>
#endif

#ifndef SOM_ODStorageUnit_xh
#include <StorageU.xh>
#endif

#ifndef SOM_Module_OpenDoc_StdProps_defined
#include <StdProps.xh>
#endif

#ifndef SOM_Module_OpenDoc_StdTypes_defined
#include <StdTypes.xh>
#endif

#ifndef SOM_ODStorageUnitView_xh
#include <SUView.xh>
#endif

#ifndef SOM_ODStorageUnitCursor_xh
#include <SUCursor.xh>
#endif

#ifndef SOM_ODDraft_xh
#include <Draft.xh>
#endif

#ifndef SOM_ODDocument_xh
#include <Document.xh>
#endif

#ifndef SOM_ODMenuBar_xh
#include "MenuBar.xh"
#endif

#ifndef SOM_ODWindow_xh
#include <Window.xh>
#endif

#ifndef SOM_ODWindowState_xh
#include <WinStat.xh>
#endif

#ifndef SOM_ODTransform_xh
#include <Trnsform.xh>
#endif

#ifndef SOM_ODFrameFacetIterator_xh
#include "FrFaItr.xh"
#endif

#ifndef SOM_Module_OpenDoc_StdProps_defined
#include <StdProps.xh>
#endif

#ifndef SOM_ODDragAndDrop_xh
#include <DragDrp.xh>
#endif

#define CRAPPO_TEST

#ifndef _ODUTILS_
#include <ODUtils.h>
#endif

#ifndef SOM_ODUndo_xh
#include <Undo.xh>
#endif

#ifndef SOM_ODSession_xh
#include <ODSessn.xh>
#endif

#ifndef SOM_ODClipboard_xh
#include <Clipbd.xh>
#endif

#ifndef SOM_ODLinkSource_xh
#include "LinkSrc.xh"
#endif

#ifndef SOM_ODLink_xh
#include "Link.xh"
#endif

#ifndef SOM_ODLinkSpec_xh
#include "LinkSpec.xh"
#endif

#ifndef SOM_ODLinkManager_xh
#include "LinkMgr.xh"
#endif

#ifndef SOM_odlinkiterator_xh
#include <LinkItr.xh>
#endif

#ifndef SOM_ODLinkSourceIterator_xh
#include <LkSrcItr.xh>
#endif

#ifndef SOM_ODPartWrapper_xh
#include "PartWrap.xh"
#endif

#ifndef SOM_ODCanvas_xh
#include <Canvas.xh>
#endif

#ifndef _T_REGIST_
#include "t_regist.h"
#endif

#ifndef _CONSTDEF_
#include "ConstDf.h"
#endif

#include "stdlib.h"

#pragma segment DragPart

/* Constants
 */
#define kODKindTestDrag                   "Apple:Kind:TestDrag"
const ODPropertyName kODPropText        = "DragPart:Property:Text";

#define kODBogusChangeID                                42

/* dkf: temp hack - note, the kODPropStyl string must be different from the
 *          corresponding property in Mac files, for now
 */
const ODPropertyName kODPropStyl        = "DragPart:Property:WindowsTextStyle";
const ODPropertyName kODPropSourceLink  = "DragPart:Property:SourceLink";
const ODPropertyName kODPropDestLink    = "DragPart:Property:DestLink";
const ODValueType    kDragTextValue     = "DragText:Value";
const ODValueType    kODKindWindowsstyl = "Windows:OSType:Scrap:styl";

/* dkf: the two lines below are a temp hack, needed to transfer
 * files to Mac (for now)
 */
const ODPropertyName kODPropStyl_Appl   	= "DragPart:Property:Style";
const ODValueType kODKindWindowsstyl_Appl 	= "Apple:OSType:Scrap:styl";

/* Functions
 */
BOOL    WaitMouseMoved(HWND);

/* DragPart Command Numbers
 */
#define cDTSampleOne    4001
#define cDTSampleTwo    4002
#define cDTTyping       9999
#define cDTStyling      9998

#define kUndoIt         1
#define kRedoIt         2

#ifdef WIN32
/* dkf: the following is a temp hack so we can write the log font in the same way under 16 bit and 32 bit.
 *      we standardize on the 16 bit version since it was here first
 */
#define LF_FACESIZE_WIN16           32
typedef struct tagLOGFONT_WIN16
{
    short	lfHeight;
    short	lfWidth;
    short   lfEscapement;
    short   lfOrientation;
    short   lfWeight;
    BYTE    lfItalic;
    BYTE    lfUnderline;
    BYTE    lfStrikeOut;
    BYTE    lfCharSet;
    BYTE    lfOutPrecision;
    BYTE    lfClipPrecision;
    BYTE    lfQuality;
    BYTE    lfPitchAndFamily;
    char    lfFaceName[LF_FACESIZE_WIN16];
} LOGFONT_WIN16;
#endif

/* METHOD OVERRIDE: somInit
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK somInit(
	DragPart		*somSelf)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	/* Call our parent's initialization first
	 */
	DragPart_parent_ODPart_somInit(somSelf);

	/* Initialize all our instance variables to known states
	 */
	memset(somThis,0,sizeof(DragPartData));
	_fLastSize               = 12;
	_fFrameShapeChanged      = TRUE;
	_fDstLink.id			 = kODUnknownChange;
	_fxSaved				 = -1;
}

/* METHOD OVERRIDE: somUninit
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK somUninit(
	DragPart 		*somSelf)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	Environment         *ev				= somGetGlobalEnvironment();

	/* Delete our display frames iterator and our memory buffers
	 */
	delete _fDisplayFrames;
	if (_fTextHandle)
	{
		GlobalUnlock(_fTextHandle);
		GlobalFree(_fTextHandle);
	}
	if (_fStylHandle)
	{
		GlobalUnlock(_fStylHandle);
		GlobalFree(_fStylHandle);
	}

	/* Release our menu bars and free our focus set
	 */
	if (_fTextMenu.menu)
		DestroyMenu(_fTextMenu.menu);
	if (_fFontMenu.menu)
		DestroyMenu(_fFontMenu.menu);
	ODReleaseObject(ev,_fMenuBar);
	ODDeleteObject(_fFocusSet);

	/* Release our storage unit and any link sources that we may be using
	 */
	ODReleaseObject(ev,_fTestDragSU);
	if ((_fDstLink).link)
		(_fDstLink).link->Release(ev);

//	delete _fSemtIntf;

	/* Call our parents' somUninit after we have cleaned up
	 */
	DragPart_parent_ODPart_somUninit(somSelf);
}

/* METHOD OVERRIDE: InitPart
 *
 * DESCRIPTION:
 *   Initialize the dragpart's storage unit when it is created
 */
SOM_Scope void SOMLINK InitPart(
	DragPart        *somSelf,
	Environment     *ev,
	ODStorageUnit   *storageUnit,
	ODPartWrapper   *partWrapper)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	
	/* Do standard part initialization
	 */
	if (somSelf->IsInitialized(ev))
		return;
	somSelf->InitPersistentObject(ev, storageUnit);
	somSelf->CommonInit(ev,partWrapper);

	storageUnit->AddProperty(ev,kODPropContents)->AddValue(ev,kODKindTestDrag);
	_fTestDragSU = storageUnit->GetDraft(ev)->CreateStorageUnit(ev);
	_fTestDragSU->AddProperty(ev,kODPropText)->AddValue(ev,kODAppleTEXT);

	/* dkf: temp hack - this prop / value combination for benefit of moving
     *      files to Mac, we set the value to NULL here and forget about it
	 */
	_fTestDragSU->AddProperty(ev,kODPropStyl_Appl)->AddValue(ev,kODKindWindowsstyl_Appl);
	_fTestDragSU->SetValue(ev,0,(ODValue)"");
	
	_fTestDragSU->AddProperty(ev,kODPropStyl)->AddValue(ev,kODKindWindowsstyl);
	_fTestDragSU->AddProperty(ev,kODPropSourceLink)->AddValue(ev,kDragTextValue);
	_fTestDragSU->AddProperty(ev,kODPropDestLink)->AddValue(ev,kDragTextValue);
	_fTestDragSU->AddProperty(ev,kODPropDisplayFrames)->AddValue(ev,kODWeakStorageUnitRefs);

	/* dkf todo: this is a temp hack!
	 *  Apple A6 expects the following property and value in this
	 *  storage unit. In Apple's code, this prop/value is created in
	 *  binding code. We must create the prop/value on a temp basis
	 *  so that our files can be read by the Mac. This hack will go
	 *  away when we implement binding more like Apple does or Apple
	 *  makes the code that reads this prop/value conditional on its
	 *  existence.
	 */
	ODStorageUnit* msu = somSelf->GetStorageUnit(ev);
	msu->AddProperty(ev,kODPropPreferredEditor)->AddValue(ev,kODEditor);
	msu->SetValue(ev,sizeof("appl:dragpart$class"),(ODValue)"appl:dragpart$class");
	
	_fTextHandle = GlobalAlloc(GMEM_ZEROINIT,BUF_SIZE);
	_fTextSize   = BUF_SIZE;
	_fpText      = (char*)GlobalLock(_fTextHandle);
	strcpy(_fpText,"");
	_fStylHandle = GlobalAlloc(GMEM_ZEROINIT,sizeof(LOGFONT));
	_fpStyl      = (LOGFONT *)GlobalLock(_fStylHandle);
	if(_fpStyl)
	{
		GetObject (GetStockObject(SYSTEM_FONT), sizeof(LOGFONT), _fpStyl);
		_fpStyl->lfHeight                = -21;
		_fpStyl->lfWidth                 = 0;
		_fpStyl->lfOutPrecision          = 3;
		_fpStyl->lfClipPrecision         = 2;
		_fpStyl->lfPitchAndFamily        = 34;
		_fpStyl->lfQuality               = 1;
		_fpStyl->lfWeight                = FW_NORMAL;
		strcpy(_fpStyl->lfFaceName,"Arial");
	}

	/* Set the cursor position to the end of the text
	 */
	_fpInsert        = _fpText + strlen(_fpText);
}

/* METHOD OVERRIDE: InitPartFromStorage
 *
 * DESCRIPTION:
 *   Load the dragpart from a pre-existing storage unit
 */
SOM_Scope void SOMLINK InitPartFromStorage(
	DragPart        *somSelf,
	Environment     *ev,
	ODStorageUnit   *storageUnit,
	ODPartWrapper   *partWrapper)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODULong         stylSize;
	ODStorageUnit   *su;
	ODStorageUnitRef aSURef;
	ODFrame         *aFrame;
	ODVolatile(su);
	ODVolatile(aFrame);

	/* Do standard part initialization
	 */
	if (somSelf->IsInitialized(ev))
		return;
	somSelf->InitPersistentObjectFromStorage(ev, storageUnit);
	somSelf->CommonInit(ev,partWrapper);

	su                      = somSelf->GetStorageUnit(ev);
	su->Focus(ev,kODPropContents,kODPosSame,kODKindTestDrag,1,kODPosFirstSib);
	su->GetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
	_fTestDragSU = su->GetDraft(ev)->GetStorageUnit(ev,su->GetIDFromStorageUnitRef(ev,aSURef));
	su           = _fTestDragSU;

	su->Focus(ev,kODPropText, kODPosUndefined, 0, 1, kODPosFirstSib);                       // read in TEXT
	_fTextSize   = (int)su->GetSize(ev);
	_fTextHandle = GlobalAlloc(GMEM_ZEROINIT,_fTextSize);
	_fpText      = (char*)GlobalLock(_fTextHandle);
	su->GetValue(ev,_fTextSize, (ODValue)_fpText);

	/* Set the cursor position to the end of the text
	 */
	_fpInsert    = _fpText + strlen(_fpText);

	/* dkf: total hack: we replace any chr(13) with chr(10), since the MAC
	 *      uses chr(13) for hard return while we use chr(10). Too bad we
	 *      don't wrap worth spit...
	 */
	char *tptr = _fpText;
	char *tendptr = tptr + strlen(_fpText);
	while (tptr < tendptr)
	{
		if (*tptr == 13)
			*tptr = 10;
		tptr++;
	}
	
	/* This property / value combination won't exist in Mac files
	 */
	if (su->Exists(ev,kODPropStyl, kODKindWindowsstyl, 0))
	{
		su->Focus(ev,kODPropStyl, kODPosUndefined, kODKindWindowsstyl, 1, kODPosUndefined);                       // read in styl
#ifdef WIN32
		stylSize = sizeof(LOGFONT);
#else
		stylSize    = su->GetSize();
#endif
		_fStylHandle = GlobalAlloc(GMEM_ZEROINIT,stylSize);
		_fpStyl      = (LOGFONT *)GlobalLock(_fStylHandle);
#ifdef WIN32
		memset(_fpStyl, 0, stylSize);
		LOGFONT_WIN16 lf16;
		memset((char *)&lf16, 0, sizeof(lf16));
		su->GetValue(ev,sizeof(lf16), (ODValue)&lf16);
		_fpStyl->lfHeight = lf16.lfHeight;
		_fpStyl->lfWidth = lf16.lfWidth;
		_fpStyl->lfEscapement = lf16.lfEscapement;
		_fpStyl->lfOrientation = lf16.lfOrientation;
		_fpStyl->lfWeight = lf16.lfWeight;
		_fpStyl->lfItalic = lf16.lfItalic;
		_fpStyl->lfUnderline = lf16.lfUnderline;
		_fpStyl->lfStrikeOut = lf16.lfStrikeOut;
		_fpStyl->lfCharSet = lf16.lfCharSet;
		_fpStyl->lfOutPrecision = lf16.lfOutPrecision;
		_fpStyl->lfClipPrecision = lf16.lfClipPrecision;
		_fpStyl->lfQuality = lf16.lfQuality;
		_fpStyl->lfPitchAndFamily = lf16.lfPitchAndFamily;
		strcpy(_fpStyl->lfFaceName, lf16.lfFaceName);
#else
		su->GetValue(ev,stylSize, (ODValue)_fpStyl);
#endif
	}
	else
	{
		_fStylHandle = GlobalAlloc(GMEM_ZEROINIT,sizeof(LOGFONT));
		_fpStyl      = (LOGFONT *)GlobalLock(_fStylHandle);
		if(_fpStyl)
		{
			GetObject (GetStockObject(SYSTEM_FONT), sizeof(LOGFONT), _fpStyl);
			_fpStyl->lfHeight                = -21;
			_fpStyl->lfWidth                 = 0;
			_fpStyl->lfOutPrecision          = 3;
			_fpStyl->lfClipPrecision         = 2;
			_fpStyl->lfPitchAndFamily        = 34;
			_fpStyl->lfQuality               = 1;
			_fpStyl->lfWeight                = FW_NORMAL;
			strcpy(_fpStyl->lfFaceName,"Arial");
		}
	}

	/* Read in the display frame (there can be only one!)
	 */
	su->Focus(ev,kODPropDisplayFrames, kODPosUndefined, 0, 1, kODPosFirstSib);
	if ( su->GetSize(ev) > 0 )
	{
		su->GetULongValue(ev,sizeof(ODStorageUnitRef), (ODValue)&aSURef);
		TRY
			aFrame = su->GetDraft(ev)->GetFrame(ev,su->GetIDFromStorageUnitRef(ev,aSURef));
			_fDisplayFrames->AddLast((ElementType)aFrame);
			aFrame->SetDroppable(ev,kODTrue);
			_fSession->GetDispatcher(ev)->RegisterIdle(ev,_fPartWrapper,aFrame,20);

			if (aFrame->IsRoot(ev))
				_fNeedsActivating = kODTrue;
		CATCH_ALL
			aFrame = kODNULL;
			su->SetOffset(ev,0);
			su->DeleteValue(ev,sizeof(ODStorageUnitRef));
		ENDTRY
	}

	/* Read in the source and destination links
	 */
//dkf   su = somSelf->GetStorageUnit();
	su->Focus(ev,kODPropSourceLink, kODPosUndefined, kDragTextValue, (ODValueIndex)0, kODPosUndefined);
	if (su->GetSize(ev) > 0)
	{
		TRY
			su->GetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
			_fSrcLink.link  = su->GetDraft(ev)->GetLinkSource(ev,su->GetIDFromStorageUnitRef(ev,aSURef));
			su->GetULongValue(ev,sizeof(kODULong), (ODValue) &_fSrcLink.start);
			su->GetULongValue(ev,sizeof(kODULong), (ODValue) &_fSrcLink.end);
			_fSrcLink.start = _fpText + OFFSETOF(_fSrcLink.start);
			_fSrcLink.end   = _fpText + OFFSETOF(_fSrcLink.end);
		CATCH_ALL
			su->SetOffset(ev,0);
			su->DeleteValue(ev,su->GetSize(ev));
			_fSrcLink.link  = (ODLinkSource*) kODNULL;
			_fSrcLink.start = NULL;
			_fSrcLink.end   = NULL;
		ENDTRY
	}
	su->Focus(ev,kODPropDestLink, kODPosUndefined, kDragTextValue, (ODValueIndex)0, kODPosUndefined);
	if (su->GetSize(ev) > 0)
	{
		TRY
			su->GetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
			_fDstLink.link  = su->GetDraft(ev)->GetLink(ev,su->GetIDFromStorageUnitRef(ev,aSURef), (ODLinkSpec*) kODNULL);
			su->GetULongValue(ev,sizeof(kODULong), (ODValue) &_fDstLink.start);
			su->GetULongValue(ev,sizeof(kODULong), (ODValue) &_fDstLink.end);
			_fDstLink.start = _fpText + OFFSETOF(_fDstLink.start);
			_fDstLink.end   = _fpText + OFFSETOF(_fDstLink.end);
			su->GetULongValue(ev,sizeof(kODULong), (ODValue)&_fDstLink.id);
		CATCH_ALL
			su->SetOffset(ev,0);
			su->DeleteValue(ev,su->GetSize(ev));
			_fDstLink.link  = (ODLink*) kODNULL;
			_fDstLink.start = 0;
			_fDstLink.end   = 0;
			_fDstLink.id	= kODUnknownChange;
		ENDTRY
	}
}

/* METHOD: CommonInit
 *
 * DESCRIPTION:
 *   Common initialization for the dragpart, used by both InitPart and
 *   InitPartFromStorage methods ...
 */
SOM_Scope void SOMLINK CommonInit(
	DragPart        *somSelf,
	Environment 	*ev,
	ODPartWrapper   *partWrapper)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	/* Initialize our commonly used variables
	 */
	_fSession        = somSelf->GetStorageUnit(ev)->GetSession(ev);
	_fPartWrapper    = partWrapper ? partWrapper : (ODPartWrapper*)somSelf;
	_fDisplayFrames  = new OrderedCollection;
	_fSelectionFocus = _fSession->Tokenize(ev,kODSelectionFocus);
	_fMenuFocus      = _fSession->Tokenize(ev,kODMenuFocus);
	_fKeyFocus       = _fSession->Tokenize(ev,kODKeyFocus);
	_fClipboardFocus = _fSession->Tokenize(ev, kODClipboardFocus);

	/* Create the sample text menu
	 */
	_fTextMenu.menu = CreatePopupMenu();
	strcpy(_fTextMenu.strMenu,"&Text");
	AppendMenu(_fTextMenu.menu, MF_STRING | MF_ENABLED, IDM_SAMPLE1,"Sample One");
	AppendMenu(_fTextMenu.menu, MF_STRING | MF_ENABLED, IDM_SAMPLE2,"Sample Two");

	/* Create the font menu
	 */
	_fFontMenu.menu = CreatePopupMenu();
	strcpy(_fFontMenu.strMenu,"&Fo&nt");
	AppendMenu(_fFontMenu.menu, MF_STRING | MF_ENABLED, IDM_FONT,"Set Font...");

	/* Get a copy of the base menu bar and add our own menus into it
	 */
	_fMenuBar = _fSession->GetWindowState(ev)->CopyBaseMenuBar(ev);
	_fMenuBar->AddMenuLast(ev, (ODMenuID)"&Text", &_fTextMenu, _fPartWrapper);
	_fMenuBar->AddMenuLast(ev, (ODMenuID)"Fo&nt", &_fFontMenu, _fPartWrapper);

	/* Create our focus set
	 */
	_fFocusSet = new ODFocusSet();
	_fFocusSet->InitFocusSet(ev);
	_fFocusSet->Add(ev,_fSelectionFocus);
	_fFocusSet->Add(ev,_fMenuFocus);
	_fFocusSet->Add(ev,_fKeyFocus);

//  _fSemtIntf = new ODSemanticInterface(ev,(ODObject*)_fPartWrapper);
//  _fSemtIntf->Initialize(ev);
//  somSelf->InstallObjectAccessors(ev);
}

/* METHOD OVERRIDE: Externalize
 *
 * DESCRIPTION:
 *   Write out our current state to storage
 */
SOM_Scope void SOMLINK Externalize(
	DragPart        *somSelf,
	Environment     *ev)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODStorageUnit   	*su;
	ODULong         	suOffset;
	ODStorageUnitRef 	aSURef;
	ODFrame         	*aFrame;
	LPSTR               offset;

	/* Call our parent method first
	 */
	DragPart_parent_ODPart_Externalize(somSelf,ev);

	/* Get our storage unit
	 */
	su              = _fTestDragSU;
	su->Focus(ev,kODPropText, kODPosUndefined, 0, 1, kODPosFirstSib);
	_fTextSize      = (int)GlobalSize(_fTextHandle);

	/* dkf: total hack: we replace any chr(10) with chr(13) before saving,
	 *                                      since the MAC uses chr(13) for hard return while we
	 *                                      use chr(10).
	 */
	char *tptr              = _fpText;
	char *tendptr   = tptr + strlen(_fpText);
	while (tptr < tendptr)
	{
		if (*tptr == 10)
			*tptr = 13;
		tptr++;
	}

	su->SetValue(ev,_fTextSize, (ODValue)_fpText);

	/* total hack continued: now we reverse the hack above and replace any
	 *                                               chr(13) with chr(10)
	 */
	tptr = _fpText;
	tendptr = tptr + strlen(_fpText);
	while (tptr < tendptr)
	{
		if (*tptr == 13)
			*tptr = 10;
		tptr++;
	}
	
	/* This property / value combination won't exist in Mac files
	 */
	if (su->Exists(ev, kODPropStyl, kODKindWindowsstyl, 0))
	{
		su->Focus(ev, kODPropStyl, kODPosUndefined, kODKindWindowsstyl, 1, kODPosUndefined);
#ifdef WIN32
		LOGFONT_WIN16 lf16;
		memset((char *)&lf16, 0, sizeof(lf16));
		lf16.lfHeight                   = (short)_fpStyl->lfHeight;
		lf16.lfWidth                    = (short)_fpStyl->lfWidth;
		lf16.lfEscapement               = (short)_fpStyl->lfEscapement;
		lf16.lfOrientation              = (short)_fpStyl->lfOrientation;
		lf16.lfWeight                   = (short)_fpStyl->lfWeight;
		lf16.lfItalic                   = _fpStyl->lfItalic;
		lf16.lfUnderline                = _fpStyl->lfUnderline;
		lf16.lfStrikeOut                = _fpStyl->lfStrikeOut;
		lf16.lfCharSet                  = _fpStyl->lfCharSet;
		lf16.lfOutPrecision     = _fpStyl->lfOutPrecision;
		lf16.lfClipPrecision    = _fpStyl->lfClipPrecision;
		lf16.lfQuality                  = _fpStyl->lfQuality;
		lf16.lfPitchAndFamily   = _fpStyl->lfPitchAndFamily;
		strcpy(lf16.lfFaceName, _fpStyl->lfFaceName);
		su->SetValue(ev,sizeof(lf16), (ODValue)&lf16);
#else
		stylSize                        = GlobalSize(_fStylHandle);
		su->SetValue(ev, stylSize,(ODValue)_fpStyl);
#endif
	}

	/* Write out the display frame
	 */
	OrderedCollectionIterator aIter(_fDisplayFrames);
	su->Focus(ev,kODPropDisplayFrames, kODPosUndefined, 0, 1, kODPosFirstSib);
	ODULong oldValueSize = su->GetSize(ev);
	suOffset = 0;
	for ( aFrame = (ODFrame*)aIter.First();
		  aIter.IsNotComplete();
		  aFrame = (ODFrame*)aIter.Next(), suOffset+=sizeof(ODStorageUnitRef)
		)
	{
		aSURef = su->GetWeakStorageUnitRef(ev,aFrame->GetStorageUnit(ev)->GetID(ev));
		su->SetOffset(ev,suOffset);
		su->SetULongValue(ev,sizeof(ODStorageUnitRef), (ODValue)&aSURef);
	}
	if (oldValueSize > suOffset)
		su->DeleteValue(ev,oldValueSize-suOffset);

	/* Write out the source and destination links
	 */
//dkf su = somSelf->GetStorageUnit();
	su->Focus(ev,kODPropSourceLink, kODPosUndefined, kDragTextValue, (ODValueIndex)0, kODPosUndefined);
	if (!_fSrcLink.link)
		su->DeleteValue(ev,su->GetSize(ev));
	else
	{
		_fSrcLink.link->Externalize(ev);
		aSURef = su->GetStrongStorageUnitRef(ev,_fSrcLink.link->GetStorageUnit(ev)->GetID(ev));
		su->SetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
		offset  = (LPSTR)(_fSrcLink.start - _fpText);
		su->SetULongValue(ev,sizeof(kODULong), (ODValue) &offset);
		offset  = (LPSTR)(_fSrcLink.end   - _fpText);
		su->SetULongValue(ev,sizeof(kODULong), (ODValue) &offset);
	}

	su->Focus(ev,kODPropDestLink, kODPosUndefined, kDragTextValue, (ODValueIndex)0, kODPosUndefined);
	if (_fDstLink.link == (ODLink*) kODNULL)
		su->DeleteValue(ev,su->GetSize(ev));
	else
	{
		aSURef = su->GetStrongStorageUnitRef(ev,_fDstLink.link->GetStorageUnit(ev)->GetID(ev));
		su->SetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
		offset  = (LPSTR)(_fDstLink.start - _fpText);
		su->SetULongValue(ev,sizeof(kODULong), (ODValue) &offset);
		offset  = (LPSTR)(_fDstLink.end   - _fpText);
		su->SetULongValue(ev,sizeof(kODULong), (ODValue) &offset);
		su->SetULongValue(ev,sizeof(kODULong),(ODValue)&_fDstLink.id);
	}

	su              = somSelf->GetStorageUnit(ev);
	su->Focus(ev,kODPropContents,kODPosSame,kODKindTestDrag,1,kODPosFirstSib);
	aSURef  = su->GetStrongStorageUnitRef(ev,_fTestDragSU->GetID(ev));
	su->SetULongValue(ev,sizeof(ODStorageUnitRef),&aSURef);
}

/* METHOD OVERRIDE: CloneInto
 *
 * DESCRIPTION:
 *   Clone our content into another draft
 */
SOM_Scope void  SOMLINK CloneInto(
	DragPart		*somSelf,
	Environment 	*ev,
	ODDraftKey 		key,
	ODStorageUnit	*storageUnit,
	ODFrame			*scopeFrame)
{
    DragPartData 	*somThis 	= DragPartGetData(somSelf);
	ODStorageUnit	*su 		= somSelf->GetStorageUnit(ev);

	/* If one of our properties already exists, this object
	 * has been cloned already
	 */
	if ( storageUnit->Exists(ev, kODPropContents, kODNULL, 0) )
		return;

	SOM_CATCH return;

	/* Should clone directly into the argument storage unit, rather
	 * than externalizing first!	
	 */
	somSelf->Externalize(ev);
	
	/* Now that we have externalized, clone ourselves the easy way by
	 * letting our storage unit do it for us. [The proper way of handling
	 * CloneInto is to write all our state data into the new storage unit
	 * directly, rather than doing Externalize and then using su->CloneInto]
	 */
	ODID scopeFrameID = 0;
	if (scopeFrame)
		scopeFrameID = scopeFrame->GetID(ev);
	su->CloneInto(ev, key, storageUnit, scopeFrameID);
	
	/* Also need to clone any links over...
	 */
	ODID clonedID;
	if (_fSrcLink.link)
		clonedID = su->GetDraft(ev)->Clone(ev, key, _fSrcLink.link->GetID(ev), 0, scopeFrameID);
	
	if (_fDstLink.link)
		clonedID = su->GetDraft(ev)->Clone(ev, key, _fDstLink.link->GetID(ev), 0, scopeFrameID);
}

/* METHOD: ExternalizeSelection
 *
 * DESCRIPTION:
 *   Write the selected text out to the given storage unit
 */
SOM_Scope void SOMLINK ExternalizeSelection(
	DragPart                *somSelf,
	Environment             *ev,
	ODStorageUnit   *disu)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	LPSTR           pText;

	/* Allocate a block to hold the selected text
	 */
	pText   = (LPSTR)malloc((int)(_fpselEnd - _fpselStart + 1));
	if (pText)
	{
		/* Write out the selected text, but NULL terminate it first !
		 */
		memcpy(pText,_fpselStart,(int)(_fpselEnd - _fpselStart));
		*(pText + (int)(_fpselEnd-_fpselStart))   = 0;
		disu->AddProperty(ev,kODPropContents);
		disu->AddValue(ev,kODAppleTEXT);
		disu->SetValue(ev,strlen(pText)+1,pText);
		free(pText);
	}

	/* Promise the style information
	 */
	disu->SetPromiseValue(ev,kODKindWindowsstyl,0,0,kODNULL,_fPartWrapper);
}

/* METHOD OVERRIDE: FulfillPromise
 *
 * DESCRIPTION:
 *   Fulfill a promise made earlier during a drag operation or cut to the
 *   system clipboard.
 */
SOM_Scope void SOMLINK FulfillPromise(
	DragPart                *somSelf,
	Environment             *ev,
	ODStorageUnitView       *promiseSUView)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	promiseSUView->SetValue(ev,GlobalSize(_fStylHandle),(ODValue)_fpStyl);
}


/* METHOD OVERRIDE: DragEnter
 *
 * DESCRIPTION:
 *   Only accept drag/drop of text
 */
SOM_Scope ODBoolean SOMLINK DragEnter(
	DragPart    		*somSelf,
	Environment         *ev,
	ODDragItemIterator  *dragInfo,
	ODFacet             *facet,
	ODPoint             *where)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODULong                 attributes;
	ODStorageUnit   *dragSU;
	ODUnused(where);
	ODUnused(facet);
	
	/* Start by assuming that we CAN accept the drop
	 */
	_fbCanAccept            = TRUE;
	_fbCanAcceptLink        = !_fDstLink.link;
	for (dragSU = dragInfo->First(ev); dragSU; dragSU = dragInfo->Next(ev))
	{
		if(!dragSU->Exists(ev,kODPropContents, kODLinkSpec, 0))
			_fbCanAcceptLink        = FALSE;

		if (!dragSU->Exists(ev,kODPropContents, kODAppleTEXT, 0))
		{
			if (!dragSU->Exists(ev,kODPropContents, kODApplehfs, 0))
			{
				// check for Bento file
				_fbCanAccept = FALSE;
			}
			else
			{       ;
				/* LATER: detect what type of files we can accept ?
				 */
			}
		}
	}

	/* Save away the original cursor state
	 */
	if(_fbCanAccept)
	{
		_fbShowCursorTemp       = _fbShowCursor;
		_fbShowCursor           = TRUE;
		_fbInDrag               = TRUE;
	}
	else
		_fbCanAcceptLink                = FALSE;

	attributes = _fSession->GetDragAndDrop(ev)->GetDragAttributes(ev);
	return (attributes & kODdragLink) ? _fbCanAcceptLink : _fbCanAccept;
}

/* METHOD OVERRIDE: DragWithin
 *
 * DESCRIPTION:
 *   Only accept the drag if we know that this is a copy or move operation.
 */
SOM_Scope ODBoolean SOMLINK DragWithin(
	DragPart                *somSelf,
	Environment             *ev,
	ODDragItemIterator      *dragInfo,
	ODFacet                 *facet,
	ODPoint                 *where)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODULong             attributes;
	BOOL                rc;
	Point               localPoint, location;
	ODUnused(dragInfo);
	
	if (!_fbCanAccept)
		return _fbCanAccept;

	/* Get the drag attributes to figure out whether this is a copy or
	 * move operation - don't accept link operations.
	 */
	attributes = _fSession->GetDragAndDrop(ev)->GetDragAttributes(ev);
	rc         =  (attributes & kODdragLink) ? _fbCanAcceptLink : kODTrue;
	if (rc)
	{
		location        = where->AsWinPoint();
		localPoint.x    = location.x * _fxScaleDivisor / _fxScaleMultiplier;
		localPoint.y    = location.y * _fyScaleDivisor / _fyScaleMultiplier;
		_fxSaved    	= -1;
		somSelf->PositionCursor(ev,facet->GetFrame(ev),localPoint.x,localPoint.y,NULL);
	}
	return rc;
}

/* METHOD OVERRIDE: DragLeave
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK DragLeave(
	DragPart                *somSelf,
	Environment             *ev,
	ODFacet                 *facet,
	ODPoint                 *where)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(where);
	ODUnused(facet);

	if (!_fbCanAccept)
		return;

	/* Return the cursor to its original state
	 */
	_fbShowCursor    = _fbShowCursorTemp;
	_fpIdleUpdate    = _fpText;
	_fbInDrag        = FALSE;
	somSelf->UpdateAllViews(ev);
}

/* METHOD OVERRIDE: Drop
 *
 * DESCRIPTION:
 *   Insert text or file(s) that were dropped on us
 */
SOM_Scope ODDropResult SOMLINK Drop(
	DragPart        	*somSelf,
	Environment         *ev,
	ODDragItemIterator  *dropInfo,
	ODFacet             *facet,
	ODPoint             *where)
{
    DragPartData            *somThis        = DragPartGetData(somSelf);
	short                   moveText;
	ODULong                 attributes;
	ODStorageUnit           *dropSU;
	int                     newSize;
	ODUnused(where);

	if (!_fbCanAccept)
		return(kODDropFail);

	/* Return the cursor to its original state
	 */
	_fbShowCursor    = _fbShowCursorTemp;
	_fpIdleUpdate    = _fpText;
	_fbInDrag        = FALSE;

	attributes      = _fSession->GetDragAndDrop(ev)->GetDragAttributes(ev);

	/* If this was a move in the same facet, remove the selected text
	 */
	moveText        = (  (attributes & kODdragIsInSourceFrame)
					  && (attributes & kODdragMove)
					  );
	if (moveText)
		somSelf->DoClear(ev,facet->GetFrame(ev));

	/* Loop through all of the drag items contained in this drag and
     * paste in the text
	 */
	for (dropSU = dropInfo->First(ev); dropSU; dropSU = dropInfo->Next(ev))
	{
		if (attributes & kODdragLink)
		{
			/* For a link, just paste in the link as if we got it from the
			 * clipboard
			 */
			if (dropSU->Exists(ev,kODPropContents,kODLinkSpec,kODPosAll))
				somSelf->PasteLink(ev,dropSU);
		}
		else if (dropSU->Exists(ev,kODPropContents, kODAppleTEXT, kODPosAll))
		{
			/* Now read in the text
			 */
			ODValue                text = kODNULL;
			ODULong                size = 0;
			dropSU->Focus(ev,
						  kODPropContents,
						  kODPosUndefined,
						  kODAppleTEXT,
						  (ODValueIndex)0,
						  kODPosFirstSib);
			size = (int)dropSU->GetSize(ev);
			text = (ODValue)malloc((int)size);
			dropSU->SetOffset(ev,0);
			dropSU->GetValue(ev,size,text);

			/* Grow our text buffer segment larger if necessary to
			 * accommodate the additional character that was typed
			 * in. However, note that the buffer only grows in
			 * discrete increments.
			 */
			if ((int)(strlen(_fpText) + size + 1) >= _fTextSize)
			{
				newSize         = (int)(ROUND_SIZE(strlen(_fpText)+size+1));
				if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
					_fTextSize       = newSize;
			}

			/* Paste the text in
			 */
			size    = strlen((LPSTR)text);
			memmove(_fpInsert + size,_fpInsert,(int)(strlen(_fpInsert)+1));
			memcpy(_fpInsert,text,(int)size);

			/* Force the text to be redrawn from here on down
			 */
			_fpIdleUpdate    = _fpInsert;

			/* Free the text
			 */
			free(text);
		}
	}
	somSelf->UpdateAllViews(ev);
	return moveText ? kODDropMove : kODDropCopy;
}

/* METHOD OVERRIDE: FocusAcquired
 *
 * DESCRIPTION:
 *   Called when we get a given focus
 */
SOM_Scope void SOMLINK FocusAcquired(
	DragPart        *somSelf,
	Environment *ev,
	ODTypeToken focus,
	ODFrame         *ownerFrame)
{
	DragPartData *somThis = DragPartGetData(somSelf);

	if (focus == _fSelectionFocus)
		_fIsActive                 = kODTrue;
	else if (focus == _fMenuFocus)
		somSelf->InstallMenus(ev, ownerFrame);
}

/* METHOD OVERRIDE: FocusLost
 *
 * DESCRIPTION:
 *   Called when we loose a given focus
 */
SOM_Scope void SOMLINK FocusLost(
	DragPart 	*somSelf,
	Environment *ev,
	ODTypeToken focus,
	ODFrame     *ownerFrame)
{
	DragPartData *somThis = DragPartGetData(somSelf);

	if (focus == _fSelectionFocus)
	{
		_fIsActive              = kODFalse;
	}
	else if (focus == _fMenuFocus)
		somSelf->RemoveMenus(ev, ownerFrame);
}

/* METHOD OVERRIDE: FacetAdded
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK FacetAdded(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet)
{
/*  DragPartData        *somThis        = DragPartGetData(somSelf); */
	ODUnused(facet);

}

/* METHOD OVERRIDE: FacetRemoved
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK FacetRemoved(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet)
{
/*  DragPartData        *somThis        = DragPartGetData(somSelf); */
	ODUnused(facet);

}

/* METHOD OVERRIDE: DisplayFrameAdded
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK DisplayFrameAdded(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *newFrame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if (newFrame->GetPart(ev) == _fPartWrapper)
	{
		/* If the first display frame is being added, register
		 * any link destination
		 */
		if (!_fLinksReConnected)
		{
			if (_fDstLink.link != ((ODLink*) kODNULL))
				_fDstLink.link->RegisterDependent(ev, _fPartWrapper,_fDstLink.id);
			_fLinksReConnected = TRUE;
		}

		if (newFrame->IsRoot(ev))
			_fNeedsActivating = kODTrue;

		_fDisplayFrames->AddLast(newFrame);
		newFrame->IncrementRefCount(ev);
		newFrame->SetDroppable(ev,kODTrue);
		_fSession->GetDispatcher(ev)->RegisterIdle(ev,_fPartWrapper,newFrame,20);
	}
}

/* METHOD OVERRIDE: DisplayFrameRemoved
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK DisplayFrameRemoved(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *oldFrame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if (_fDisplayFrames->Contains(oldFrame))
	{
		_fSession->GetDispatcher(ev)->UnregisterIdle(ev,_fPartWrapper,oldFrame);
		somSelf->DeActivateFrame(ev,oldFrame);
		_fDisplayFrames->Remove(oldFrame);
		oldFrame->Release(ev);

		/* If the last display frame was removed, unregister any
		 * link destination
		 */
		if (_fDisplayFrames->Count() == 0)
		{
			if (_fDstLink.link != ((ODLink*) kODNULL))
				_fDstLink.link->UnregisterDependent(ev, _fPartWrapper);
		}
	}
	else
		THROW(kODErrInvalidFrame);
}

/* METHOD OVERRIDE: GeometryChanged
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK GeometryChanged(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet,
	ODBoolean       clipShapeChanged,
	ODBoolean       externalTransformChanged)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(facet);

	if (externalTransformChanged)
	{       ;
//              somSelf->PositionDocumentParts();
//      somSelf->AdjustScrollBar();
//      TEGetHiliteRgn(fDocument->hiliteRgn, fDocument->theTE);
//      _fFrameShapeChanged = FALSE;
	}
}

/* METHOD OVERRIDE: Draw
 *
 * DESCRIPTION:
 *   Draw the text back out to our facet
 */
SOM_Scope void SOMLINK Draw(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet,
	ODShape         *invalidShape)
{
    DragPartData    *somThis        = DragPartGetData(somSelf);
	ODFrame*        displayFrame    = facet->GetFrame(ev);
	HWND            hwnd            = somSelf->GetMyWindow(ev,facet);
	int             Line            = 1;
	BOOL            bIdleErase      = FALSE;
	HFONT           hDrawFont       = CreateFontIndirect(_fpStyl);
	RECT            rect ;
	HFONT           hOldFont;
	HDC             hdcInfo;
	RECT            EraseRect,SelRect;
	char            TempChar;
	int             TextWidth,CharWidth;
	LPSTR           pNextLine, pCurLine;
	BOOL            bselOnThisLine;

	/* Ensure we don't trap
	 */
	if ((_fpInsert - _fpText) > _fTextSize)
		_fpInsert         = _fpText;

	/* If this frame is a current display frame for us, we should draw it
	 */
	if ( _fDisplayFrames->Contains(displayFrame) )
	{
		HDC             hdc;
		ODRect 			bRect;
#ifdef WIN32
		SIZE 			size;
		SIZE 			windowExt, viewportExt;
#endif

		CFocus  f(facet, invalidShape, &hdc);

		/* Figure the scale multipliers currently in use
		 */
		if (GetWindowExtEx(hdc,&windowExt) && GetViewportExtEx(hdc,&viewportExt))
		{
			_fxScaleMultiplier = (int)windowExt.cx;
			_fyScaleMultiplier = (int)windowExt.cy;
			_fxScaleDivisor    = (int)viewportExt.cx;
			_fyScaleDivisor	   = (int)viewportExt.cy;
		}

		/* Calculate the line height if we don't know it...
		 */
		hdcInfo         = GetDC(hwnd);
		if (hdcInfo)
		{
			hOldFont        = SelectObject(hdcInfo, hDrawFont);
#ifndef WIN32
			_fcyLineHeight   = HIWORD(GetTextExtent(hdcInfo,"W",1));
#else
			SIZE size;
			GetTextExtentPoint (hdcInfo, "W", 1, &size);
  			_fcyLineHeight = size.cy * _fyScaleMultiplier / _fyScaleDivisor;
#endif
 			_fcyLineHeight = _fcyLineHeight * _fyScaleDivisor / _fyScaleMultiplier;
			SelectObject(hdcInfo,hOldFont);
			ReleaseDC(hwnd,hdcInfo);
		}

		/* Select the current font
		 */
		hOldFont = SelectObject (hdc, hDrawFont);

		displayFrame->GetFrameShape(ev,kODNULL)->GetBoundingBox(ev,&bRect);
		bRect.AsWinRect( rect );

		/* Wipe out the entire background if we are in normal redraw mode
		 */
		if (_fDrawMode == DRAW_MODE_DEFAULT)
		{
			FillRect(hdc,&rect,GetStockObject(WHITE_BRUSH));
			SetBkColor(hdc,RGB(255,255,255));
		}

		/* Note DrawText() can't be used in a metafile ! Justify the
		 * text ourselves to make sure it looks OK...
		 */
		pNextLine       = _fpText;
		do
		{
			pCurLine        = pNextLine;
			pNextLine       = strchr(pNextLine,'\n');
			if (pNextLine)
				*pNextLine      = '\0';

			/* Is the selection on this line ?
			 */
			bselOnThisLine  =    _fpselStart
							  && (pCurLine <= _fpselStart)
							  && (pNextLine ? (_fpselStart <= pNextLine) : TRUE);

			if (  (_fDrawMode == DRAW_MODE_DEFAULT)
			   || (  (_fDrawMode == DRAW_MODE_LINE)
				  && ((_fpInsert >= pCurLine) && (pNextLine ? (_fpInsert <= pNextLine) : TRUE))
				  )
			   || (  (_fDrawMode == DRAW_MODE_IDLE)
				  && _fpIdleUpdate
				  && (pNextLine ? (_fpIdleUpdate <= pNextLine) : TRUE)
				  )
			   )
			{
				/* Erase this line's background, if necessary
				 */
				CopyRect(&EraseRect,&rect);
				EraseRect.top   = EraseRect.top + ((Line - 1)*_fcyLineHeight);
				if (_fDrawMode == DRAW_MODE_LINE)
				{
					/* Avoid flickering too much by only erasing the text
					 * from the cursor onwards ....
					 */
					if ((_fpInsert > pCurLine) && !bselOnThisLine)
					{
						TempChar                = *(_fpInsert - 1);
						*(_fpInsert - 1) = 0;
#ifndef WIN32
						TextWidth = LOWORD(GetTextExtent(hdc,pCurLine,strlen(pCurLine)));
#else
						GetTextExtentPoint (hdc, pCurLine, strlen(pCurLine), &size);

						TextWidth = size.cx;
#endif

						*(_fpInsert - 1) = TempChar;
						EraseRect.left  = EraseRect.left + TextWidth;
					}
					EraseRect.bottom        = EraseRect.top + _fcyLineHeight;
					FillRect(hdc,&EraseRect,GetStockObject(WHITE_BRUSH));
				}
				else if ((_fDrawMode == DRAW_MODE_IDLE) && !bIdleErase)
				{
					FillRect(hdc,&EraseRect,GetStockObject(WHITE_BRUSH));
					bIdleErase      = TRUE;
				}

				/* Draw the text for this line
				 */
				ExtTextOut(hdc,rect.left,EraseRect.top,
						   ETO_CLIPPED,
						   &rect,
						   pCurLine,
						   lstrlen(pCurLine),
						   NULL);

				/* Draw the cursor if it is on this line
				 */
				if (  _fpInsert
				   && (_fpInsert >= pCurLine)
				   && (pNextLine ? (_fpInsert <= pNextLine) : TRUE)
				   && !facet->GetCanvas(ev)->IsOffscreen(ev)
				   && _fbShowCursor
				   )
				{
					TempChar  = *(_fpInsert);
					*_fpInsert = 0;
#ifndef WIN32
					TextWidth = LOWORD(GetTextExtent(hdc,pCurLine,strlen(pCurLine)));
#else
					GetTextExtentPoint (hdc, pCurLine, strlen(pCurLine), &size);

					TextWidth = size.cx;
#endif
					if (TempChar)
					{
						if (!GetCharWidth(hdc,TempChar,TempChar,&CharWidth))
						{
							ABC     abc;
							GetCharABCWidths(hdc,TempChar,TempChar,&abc);
							CharWidth = abc.abcA + abc.abcB + abc.abcC;
						}
					}
					else
						CharWidth       = 0;
					if(CharWidth < 4)
						CharWidth = 4;
					_fCursorWidth	= CharWidth;
					*_fpInsert = TempChar;
					EraseRect.left   = rect.left + TextWidth;
					EraseRect.right  = EraseRect.left + CharWidth;
					EraseRect.bottom = EraseRect.top  + _fcyLineHeight;
					if(!_fbInDrag)
					{
						/* Replace / Insert cursor
						 */
						if (_fbReplaceMode)
							EraseRect.top = EraseRect.bottom - 4;
						else
							EraseRect.top = EraseRect.bottom - (_fcyLineHeight/2);
					}
					else
					{
						/* Drop target cursor
						 */
						EraseRect.right   = EraseRect.left + 4;
						EraseRect.top     = EraseRect.bottom - _fcyLineHeight + 1;
					}
					OffsetRect(&EraseRect,0,-1);
					InvertRect(hdc,&EraseRect);
				}

				/* Draw the selection if it is on this line
				 */
				if (bselOnThisLine)
				{
					CopyRect(&SelRect,&rect);
					SelRect.top        = SelRect.top  + ((Line-1)*_fcyLineHeight);
					SelRect.bottom = SelRect.top  + _fcyLineHeight;
					SelRect.right  = SelRect.left + _fcxselEnd;
					SelRect.left   = SelRect.left + _fcxselStart;
					InvertRect(hdc,&SelRect);
				}
			}
			if (pNextLine)
			{
				*pNextLine      = '\n';
				pNextLine++;
			}
			Line++;
		} while (pNextLine);

		/* Restore our canvas to its original state
		 */

		SelectObject (hdc, hOldFont);
		DeleteObject (hDrawFont);
	}
	else
	{
		// !!! signal error: invalid frame
	}
}

/* METHOD OVERRIDE: HandleMouseDown
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK HandleMouseDown(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet,
	ODPoint     	*where,
	ODEventData 	*event)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	Point           	xForm, localPoint;
	LPSTR           	pszMouseDown;
	BOOL            	bWasActive      = _fIsActive;
	ODUnused(where);

	if (!facet->GetWindow(ev)->IsActive(ev))
		SetActiveWindow(facet->GetWindow(ev)->GetPlatformWindow(ev));
	else
	{
		somSelf->ActivateFrame(ev,facet->GetFrame(ev));

		xForm              = somSelf->GetAggregateTranslation(ev,facet).AsWinPoint();
		localPoint.x   = LOWORD(event->lParam) - xForm.x;
		localPoint.y   = HIWORD(event->lParam) - xForm.y;

		/* If the mouse moves, this must be selection or dragging: otherwise
		 * it is just a single-click to position the cursor....
		 *   - Bug workaround added: the bWasActive flag is used to prevent
		 *     selection immediately following activation. This avoids a
		 *     visual bug that shows up, and prevents the user from loosing
		 *     his selection each time he reactivates the text part.
		 */
		_fxSaved    	= -1;
		if (  _fbShowCursor
		   && *_fpText
		   && !_fbFontChanged
		   && WaitMouseMoved(facet->GetWindow(ev)->GetPlatformWindow(ev))
		   )
		{
			pszMouseDown    = somSelf->PositionFromPoint(
													  ev,
													  facet->GetFrame(ev),
													  localPoint.x,
													  localPoint.y,
													  NULL);
			if (  _fpselStart
			   && (pszMouseDown >= _fpselStart)
			   && (pszMouseDown <= _fpselEnd)
			   )
				somSelf->DragText(ev,facet,event);
			else if (bWasActive)
				somSelf->SelectText(ev,facet,event);
			else
			{
				_fpselStart   = NULL;
				_fpselEnd     = NULL;
				somSelf->PositionCursor(
									 ev,
									 facet->GetFrame(ev),
									 localPoint.x,
									 localPoint.y,
									 NULL);
			}
		}
		else
		{
			_fpselStart   = NULL;
			_fpselEnd     = NULL;
			somSelf->PositionCursor(
								 ev,
								 facet->GetFrame(ev),
								 localPoint.x,
								 localPoint.y,
								 NULL);
		}
		somSelf->SyncMenuBar(ev,facet->GetFrame(ev));
	}
  /*return kODTrue;*/
}

/* METHOD OVERRIDE: CreateLink
 *
 * DESCRIPTION:
 *   Create a source link to the current selection. If a source link range
 *   currently exists, just use that one - since we don't support multiple
 *   link sources at this time.
 */
SOM_Scope ODLink* SOMLINK CreateLink(
	DragPart        *somSelf,
	Environment     *ev,
	ODPtr           data,
	ODULong         size)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODULong                 textSize;
	ODUnused(data);
	ODUnused(size);
	
	/* Create a new link
	 */
	if(!_fSrcLink.link)
	{
		textSize = _fpselEnd - _fpselStart;
		if (_fpselStart && (textSize > 0))
		{
			_fSrcLink.link = somSelf->GetStorageUnit(ev)->GetDraft(ev)->CreateLinkSource(ev,_fPartWrapper);
			somSelf->LinkChanged(ev,&_fSrcLink);

			/* Remove the selection, so the linked-to text can easily be
			 * typed into (for demo purposes)
			 */
			_fpIdleUpdate   = _fpselStart;
			_fpselStart     = NULL;
			_fpselEnd       = NULL;
		}
	}

	/* Obtain the link from our new link source object
	 */
	ODLink* link = _fSrcLink.link->GetLink(ev);
	link->IncrementRefCount(ev);
	return link;
}

/* METHOD OVERRIDE: LinkChanged
 *
 * DESCRIPTION:
 *   Called by us whenever we want to tell our clients of our source link
 *   that we have changed its value.
 */
SOM_Scope void SOMLINK LinkChanged(
	DragPart        *somSelf,
	Environment     *ev,
	SrceOfLink      *srcLink)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
    ODStorageUnit       *su;
    ODULong             textSize;
	ODLinkKey               key;
	LPSTR                   text;

	/* Don't do anything if the link source doesn't exist...
	 */
    if (_fSrcLink.link)
	{
		/* Add a property and value for the text that we are exporting as
		 * a link-souce
		 */
		if (srcLink->link->Lock(ev,0, &key))
		{
			su = srcLink->link->GetContentStorageUnit(ev,key);
			if (!su->Exists(ev,kODPropContents, (ODValueType) kODNULL, 0))
				su->AddProperty(ev,kODPropContents);
			if (!su->Exists(ev,kODPropContents, kODAppleTEXT, 0))
				su->AddValue(ev,kODAppleTEXT);
			else
			{
				ODULong valueSize;
				su->Focus(ev,kODPropContents, kODPosUndefined,
						  kODAppleTEXT, 0, kODPosUndefined);
				valueSize = su->GetSize(ev);
				su->DeleteValue(ev,valueSize);
			}
			su->Focus(ev,kODPropContents, kODPosUndefined,
					  kODAppleTEXT, 0, kODPosUndefined);
	
			/* Write the current link specification out to the link storage unit
			 */
			textSize        = srcLink->end - srcLink->start;
			text            = new char[textSize+1];
			if (text)
			{
				memcpy(text,srcLink->start,textSize);
				text[textSize]  = 0;
				su->SetValue(ev,textSize+1,text);
				delete text;
			}
	
			/* Tell anyone linked to this link that we just put up some new
			 * content...
			 */
			srcLink->link->Unlock(ev,key);
			srcLink->link->ContentChanged(ev,kODBogusChangeID/* LATER? */,key);
		}
	}
}

/* METHOD OVERRIDE: LinkUpdated
 *
 * DESCRIPTION:
 *   Called on a client part whenever the data being linked to gets updated.
 */
SOM_Scope void SOMLINK LinkUpdated(
	DragPart        *somSelf,
	Environment     *ev,
	ODLink          *updatedLink,
	ODChangeID      id)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODLinkKey           key;
	ODUnused(id);

	/* If the link that was changed is our current destination link, lets
	 * refresh ourselves with the new version of the linked-to data
	 */
	if (updatedLink == _fDstLink.link)
	{
		ODSShort       oldSize   = _fDstLink.end - _fDstLink.start;
		LPSTR          selStart  = _fpselStart;
		LPSTR          selEnd    = _fpselEnd;
		ODStorageUnit  *su;
		ODSShort       newSize, TextLen;
		ODPtr          text;
	
		/* Safety features !
		 */
		if ((_fDstLink.end - _fpText) > _fTextSize)
			_fDstLink.end   = _fpText;
		if ((_fDstLink.start - _fpText) > _fTextSize)
			_fDstLink.start = _fpText;
		if (_fDstLink.end < _fDstLink.start)
			_fDstLink.end   = _fDstLink.start + 1;
	
		/* Delete the existing linked text
		 */
		if (_fDstLink.start)
		{
			TextLen                 = strlen(_fDstLink.end);
			memmove(_fDstLink.start,_fDstLink.end,TextLen+1);
			*(_fDstLink.start + TextLen) = 0;    /* NULL terminate */
			_fpIdleUpdate    = _fDstLink.start;
			if ((_fpInsert > _fDstLink.start) && _fDstLink.start)
				_fpInsert    = _fDstLink.start;
			_fpselStart              = NULL;
			_fpselEnd                = NULL;
		}
		else
			_fDstLink.start = _fpInsert;
	
		/* Insert the new linked text
		 */
		if (_fDstLink.link->Lock(ev,0, &key))
		{
			su              = _fDstLink.link->GetContentStorageUnit(ev,key);
			su->Focus(ev,kODPropContents, kODPosUndefined,
					  kODAppleTEXT, 0, kODPosUndefined);
			newSize = (ODSShort) su->GetSize(ev);
			if (newSize > 0)
			{
				text = new char[newSize];
				su->GetValue(ev,newSize, (ODValue) text);
	
				/* Grow our text buffer if necessary
				 */
				if ((int)(strlen(_fpText) + newSize + 1) >= _fTextSize)
				{
					newSize         = (int)(ROUND_SIZE(strlen(_fpText)+newSize+1));
					if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
						_fTextSize       = newSize;
				}
	
				/* Paste the text in
				 */
				newSize    = strlen((LPSTR)text);
				memmove(_fpInsert + (int)newSize,_fpInsert,(int)(strlen(_fpInsert)+1));
				memcpy(_fpInsert,text,(int)newSize);
	
				/* Force the text to be redrawn from here on down
				 */
				_fpIdleUpdate    = _fpInsert;
	
				delete text;
	
				/* Adjust other links due to this change - which is
				 * special cased for source links only for now and
				 * assumes no overlapping links or recursion problems
				 */
				somSelf->AdjustSrcLinks(ev,
										_fDstLink.start,
										_fDstLink.end,
										_fDstLink.start + newSize);
	
				/* Can we let AdjustDstLinks take care of this one too???
				 */
				_fDstLink.end = _fDstLink.start + newSize;
			}
			_fDstLink.link->Unlock(ev,key);
		}
	
		/* Restore the original selection: Doesn't handle overlapping
		 */
		if (selStart > _fDstLink.start)
			_fpselStart = selStart - (oldSize - newSize);
		if (selEnd > _fDstLink.start)
			_fpselEnd   = selEnd - (oldSize - newSize);
	
		if (!newSize)
		{
			/* Can't call UnregisterDependent inside LinkUpdated
			 * Undo may cause us to re-create this link, anyway
			 */
			if (_fDstLink.link)
				_fDstLink.link->UnregisterDependent(ev,_fPartWrapper);
	
			_fDstLink.link      = (ODLink*) kODNULL;
			_fDstLink.start     = NULL;
			_fDstLink.end       = NULL;
		}
	}
}

/* METHOD: CheckDstLinkDeletion
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK CheckDstLinkDeletion(
	DragPart        *somSelf,
	Environment     *ev,
	LPSTR           start,
	LPSTR           endOld)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if ((start <= _fDstLink.start) && (endOld >= _fDstLink.end))
	{
		/* Destination link has been removed.
		 */
		if (_fDstLink.link)
			_fDstLink.link->UnregisterDependent(ev,_fPartWrapper);
		_fDstLink.link  = (ODLink*)kODNULL;
		_fDstLink.start         = NULL;
		_fDstLink.end   = NULL;
    }
}

/* METHOD: AdjustLinks
 *
 * DESCRIPTION:
 *   Adjust source and destination links following a change to the text
 *   edit records.
 */
SOM_Scope void SOMLINK AdjustLinks(
	DragPart        *somSelf,
	Environment     *ev,
	LPSTR           start,
	LPSTR           endOld,
	LPSTR           endNew)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	somSelf->AdjustDstLinks(ev, start, endOld, endNew);
    somSelf->AdjustSrcLinks(ev, start, endOld, endNew);
}

/* METHOD: AdjustDstLinks
 *
 * DESCRIPTION:
 *   Adjust destination links following a change to the text
 */
SOM_Scope void SOMLINK AdjustDstLinks(
	DragPart        *somSelf,
	Environment     *ev,
	LPSTR           start,
	LPSTR           endOld,
	LPSTR           endNew)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if (endOld > start)
	{
		/* Some content was replaced, so see if the destination link
	 * was deleted
		 */
	somSelf->CheckDstLinkDeletion(ev,start, endOld);
    }

    if (start < _fDstLink.end)
	{
	/* Must be entirely before the destination link
		 */
	if (_fDstLink.link)
		{
	    _fDstLink.start     += (endNew - endOld);
	    _fDstLink.end       += (endNew - endOld);
	}
    }

	if (_fDstLink.start >= _fDstLink.end)
		_fDstLink.end   = _fDstLink.start + 1;
}
			
/* METHOD: AdjustSrcLinks
 *
 * DESCRIPTION:
 *   Adjust source link offsets following a change to the text edit record
 *   Text in the range start..endOld was replaced by the range start..endNew.
 *   Note that this routine calls ContentChanged(); this can cause the
 *   LinkUpdated method of this object to be called, which calls this method.
 *   On entry, _fSrcLink contains old offsets.
 *   On exit, _fSrcLink contains new offsets.
 */
SOM_Scope void SOMLINK AdjustSrcLinks(
	DragPart        *somSelf,
	Environment     *ev,
	LPSTR           start,
	LPSTR           endOld,
	LPSTR           endNew)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODLinkKey               key;

    if (start >= _fSrcLink.end)
	{
		/* Source link not affected
		 */
		return;
    }
    if (  ((start < _fSrcLink.start) && (endOld >= _fSrcLink.end))
	   || ((start <= _fSrcLink.start) && (endOld > _fSrcLink.end))
	   || (  (endNew == start)
	      && (start == _fSrcLink.start)
		  && (endOld == _fSrcLink.end)
		  )
	   )
	{
		/* Fall in here if
		 *  - The text range start..endOld included more than the entire link, or
		 *  - Exactly the link was deleted with no replacement
		 * In either case, delete the link
		 */
		if (_fSrcLink.link && _fSrcLink.link->Lock(ev,0,&key))
		{
			ODStorageUnit* su = _fSrcLink.link->GetContentStorageUnit(ev,key);
			ODULong size;
			su->Focus(ev,kODPropContents,kODPosUndefined,
						  kODAppleTEXT, 0, kODPosUndefined);
			size = su->GetSize(ev);
			su->DeleteValue(ev,size);
			_fSrcLink.link->ContentChanged(ev,kODBogusChangeID/* LATER? */,key);
			_fSrcLink.link->Unlock(ev,key);
			_fSrcLink.link->Release(ev);
			somSelf->RemoveLinkSpec(ev,&_fSrcLink);
		}
		_fSrcLink.link  = (ODLinkSource*)kODNULL;
		_fSrcLink.start         = NULL;
		_fSrcLink.end   = NULL;
		return;
    }
    if ((endOld <= _fSrcLink.start) && (start < _fSrcLink.start))
	{
	/* Changes occurred ahead of the source link, so move it accordingly
	 * (Treat the case of an insertion point at the beginning of the link as
	 * inserting into the link; handled below).
		 */
	_fSrcLink.start         += (endNew - endOld);
	_fSrcLink.end   += (endNew - endOld);
	return;
    }
    if (start < _fSrcLink.start)
	{
	/* Only that portion of link unaffected remains within the link
		 */
	_fSrcLink.start   = start + (endNew - _fpText);
	_fSrcLink.end   += (endNew - endOld);
    }
    else if (endOld > _fSrcLink.end)
	{
	/* Only that portion of link unaffected remains within the link
	 * The replaced text starts in the selection and continues beyond
		 */
	_fSrcLink.end = start;
    }
    else
	{
	/* Replaced text lies entirely within the link, so all new text
	 * is included in the link
		 */
	_fSrcLink.end += (endNew - endOld);
    }
	if(_fSrcLink.start >= _fSrcLink.end)
		_fSrcLink.end = _fSrcLink.start + 1;
    somSelf->LinkChanged(ev,&_fSrcLink);
    somSelf->RemoveLinkSpec(ev,&_fSrcLink);
}

/* METHOD: RemoveLinkSpec
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK RemoveLinkSpec(
	DragPart        *somSelf,
	Environment     *ev,
	SrceOfLink      *srcLink)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
    ODClipboard         * clipboard     = _fSession->GetClipboard(ev);
	ODFrame             *frame          = (ODFrame*)_fDisplayFrames->First();
	ODUnused(srcLink);

	/* LATER: Should keep the frame that last cut or copied to the clipboard
	 */
	if (!frame)
		return;

	ODArbitrator* arbitrator = _fSession->GetArbitrator(ev);
	if (  (frame == arbitrator->GetFocusOwner(ev, _fClipboardFocus))
	   || (arbitrator->RequestFocus(ev, _fClipboardFocus, frame))
	   )
	{
		/* Do we still own the clipboard?
		 */
		ODClipboard* clipboard = _fSession->GetClipboard(ev);
		if ( _fClipboardChangeID == clipboard->GetChangeID(ev) )
		{
			/* Remove the link spec from the clipboard
			 */
			TRY
				ODStorageUnit* clipRootSU = clipboard->GetContentStorageUnit(ev);
				if ( clipRootSU->Exists(ev,kODPropContents/*kODPropLinkSpec*/,
										(ODValueType)kODLinkSpec/*NULL*/,
										0)
				   )
				{
					clipRootSU->Focus(ev,kODPropContents/*kODPropLinkSpec*/,
									  kODPosUndefined,
									  (ODValueType)kODLinkSpec/*NULL*/,
									  0,
									  kODPosUndefined);
					clipRootSU->Remove(ev);
				}
			CATCH_ALL
			ENDTRY
		}
	}
}

/* METHOD: HandleKeyDown
 *
 * DESCRIPTION:
 *   Handle keyboard input of text and cursoring
 */
SOM_Scope void SOMLINK HandleKeyDown(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *focusFrame,
	ODEventData 	*event)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	#define    	NO_TEXT_CHANGE            0xFFFF
	ODBoolean 	handled              	= kODFalse;
	char       	NewChar              	= 0;
	LPSTR      	pInsertOld           	= _fpInsert;
	LPSTR      	pInsertEnd           	= _fpInsert;
	UINT       	cbChanged            	= NO_TEXT_CHANGE;
	LPSTR		pNewCursor,pLineStart;
	char      	TempChar;
	int       	newSize;
	int       	x, y;
	LPSTR      	pPrevCR, pNextCR;

	/* If our parent part didn't handle the event, this is keyboard input
	 */
	if (!handled && focusFrame)
	{
		if (event->message == WM_CHAR)
		{
			/* Handle normal ASCII key strokes - ignoring any control
			 * characters that we will catch as VK_ values...
			 */
			if (event->wParam >= 0x20)
			{
				NewChar = (char)event->wParam;
				_fxSaved	= -1;
			}
		}
		else if (event->message == WM_KEYDOWN)
		{
			/* If this isn't VK_UP or VK_DOWN we must set _fxSaved to -1
			 */
			if ((event->wParam != VK_UP) && (event->wParam != VK_DOWN))
				_fxSaved	= -1;

			/* Handle special virtual keys for cursoring, insert mode and
			 * carriage returns.
			 */
			switch (event->wParam)
			{
				case VK_RETURN:
					NewChar        = '\n';
					_fpUpdateChar  = NULL;
					if(!_fpIdleUpdate || (_fpInsert < _fpIdleUpdate))
						_fpIdleUpdate  = (_fpInsert == _fpText)
											? _fpText
											: _fpInsert - 1;
					break;

				case VK_LEFT:
					if (  (_fpInsert          != _fpText)
					   && (*(_fpInsert-1) != '\n')
					   )
						_fpInsert--;
					break;

				case VK_RIGHT:
					if (  *(_fpInsert+1)
					   && (*(_fpInsert+1) != '\n')
					   )
						_fpInsert++;
					break;
				
				case VK_UP:
				case VK_DOWN:
					/* Calculate the next and previous lines
					 */
					TempChar	= *_fpInsert;
					*_fpInsert	= 'Z';
					pPrevCR 	= _fpInsert;
					while ((pPrevCR > _fpText) && (*pPrevCR != '\n'))
						pPrevCR--;
					pNextCR 	= strchr(_fpInsert,'\n');
					*_fpInsert	= TempChar;

					/* Measure the X offset on this line
					 */
					TempChar    = *_fpInsert;
					if ((event->wParam == VK_DOWN) && (*(pPrevCR+1) == '\n'))
						x	= 0;
					else if ((event->wParam == VK_UP) && (*(pPrevCR-1) == '\n'))
						x	= 0;
					else if (_fxSaved != -1)
						x 	= _fxSaved;
					else	
					{
						TempChar	= *_fpInsert;
						*_fpInsert  = 0;
						x	= somSelf->GetExtent(ev,focusFrame,pPrevCR,0,NULL);
						x	-= _fCursorWidth / 2;
						if (x < 0)
							x	= 0;
						*_fpInsert  = TempChar;
						_fxSaved	= x;
					}

					if (event->wParam == VK_DOWN)
					{
						/* VK_DOWN: move the cursor to beginning of
						 * the next text line
						 */
						if (!pNextCR)
							return /*TRUE*/;
						pNewCursor	= pNextCR;
						if (!*(pNewCursor+1))
							return;
						else
							pNewCursor++;
					}
					else
					{
						/* VK_UP: move the cursor to the beginning
						 * of the previous text line
						 */
						pNewCursor	= pPrevCR;
						if (pNewCursor != _fpText)
						{
							pNewCursor--;
							while ((pNewCursor > _fpText) && (*pNewCursor != '\n'))
								pNewCursor--;
						}
						if (pNewCursor != _fpText)
							pNewCursor++;
					}

					/* Now move the cursor along until the text extent is the
					 * same or more than before...
					 */
					pLineStart	= pNewCursor;
					y			= 0;
					while (y < x)
					{		
  						pNewCursor++;
						if (*pNewCursor == '\n')
						{
							pNewCursor--;
							break;
						}
						if(*(pNewCursor + 1) == 0)
							break;

						TempChar    = *pNewCursor;
						*pNewCursor = 0;
						y           = somSelf->GetExtent(ev,focusFrame,pLineStart,0,NULL);
						*pNewCursor = TempChar;
						if (y >= x)
						{
							pNewCursor--;
							break;
						}
					}
					somSelf->PositionCursor(ev,
										    focusFrame,
											0,
											0,
											pNewCursor);
					break;

				case VK_INSERT:
					_fbReplaceMode   = !_fbReplaceMode;
					break;

				case VK_DELETE:
					if (!(*_fpInsert))
						return /*TRUE*/;
					if (  (*_fpInsert == '\n')
					   || (_fpIdleUpdate && (_fpIdleUpdate > _fpInsert))
					   )
						_fpIdleUpdate    = _fpInsert;
					memmove(_fpInsert,_fpInsert+1,strlen(_fpInsert+1)+1);
					pInsertEnd                      = _fpInsert+1;
					cbChanged                       = 0;
					break;

				case VK_BACK:
					if (_fpInsert == _fpText)
						return /*TRUE*/;
					if (  (*(_fpInsert-1) == '\n')
					   || (_fpIdleUpdate && (_fpIdleUpdate > _fpInsert - 1))
						)
						_fpIdleUpdate    = _fpInsert - 1;
					memmove(_fpInsert-1,_fpInsert,strlen(_fpInsert)+1);
					_fpInsert                       = _fpInsert - 1;
					pInsertEnd                      = _fpInsert+1;
					cbChanged                       = 0;
					break;

				default:
					return /*FALSE*/;
			}
		}

		if (NewChar)
		{
			/* Grow our text buffer segment larger if necessary to
			 * accommodate the additional character that was typed
			 * in. However, note that the buffer only grows in
			 * discrete increments.
			 */
			if ((int)(strlen(_fpText) + 2) >= _fTextSize)
			{
				newSize         = ROUND_SIZE(_fTextSize+10);
				if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
					_fTextSize       = newSize;
			}

			/* Delete the current selection if necessary
			 */
			cbChanged       = 0;
			if (_fpselStart)
			{
				pInsertEnd      = _fpselEnd;
				pInsertOld      = _fpselStart;
				somSelf->DoClear(ev,focusFrame);
			}

			/* Copy or replace the character at the insertion point
			 * for the key that we received the WM_KEYDOWN message for
			 */
			if (_fbReplaceMode)                                                              /* REPLACE */
			{
				if (!*_fpInsert)
					*(_fpInsert+1)   = 0;

				/* Replacing a carriage return causes the following lines
				 * to get shifted upwards !
				 */
				if (*_fpInsert == '\n')
					_fpIdleUpdate    = _fpInsert;
				*_fpInsert++     = NewChar;
			}
			else                                                                                    /* INSERT */
			{
				memmove(_fpInsert+1,_fpInsert,strlen(_fpInsert)+1);
				*_fpInsert++     = NewChar;
				cbChanged++;
			}
		}

		/* Update our source and destination links
		 */
		if (cbChanged != NO_TEXT_CHANGE)
			somSelf->AdjustLinks(ev,pInsertOld,pInsertEnd,pInsertOld+cbChanged);

		/* Update ourselves in all views
		 */
		_fDrawMode       = DRAW_MODE_LINE;
		somSelf->UpdateAllViews(ev);
		_fDrawMode       = DRAW_MODE_DEFAULT;

		/* Specify that we did handle this keyboard event completely
		 */
		handled         = TRUE;

		somSelf->SyncMenuBar(ev,focusFrame);
		somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);
		// OLE: note that SetChangedFromPrev also updates cached image when
		// running in OLE container applications
	}
  /*return handled*/
}

/* METHOD: DoCut
 *
 * DESCRIPTION:
 *   Cut the selected text to the clipboard
 */
SOM_Scope ODBoolean SOMLINK DoCut(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODClipboard     *clipboard      = _fSession->GetClipboard(ev);
	ODStorageUnit   *clipBSU        = kODNULL;

	/* Put the current selection on the clipboard
	 */
	TRY
		clipboard->Clear(ev);
		clipBSU         = clipboard->GetContentStorageUnit(ev);
		somSelf->ExternalizeSelection(ev,clipBSU);
		clipboard->SetPlatformClipboard(ev,NULL);
		_fClipboardChangeID = clipboard->GetChangeID(ev);
	CATCH_ALL
	ENDTRY
		
	/* Cut the text out of the document
	 */
	if (_fpselStart)
	{
		somSelf->AdjustLinks(ev,_fpselStart, _fpselEnd, _fpselStart);
		somSelf->DoClear(ev,frame);
	}
//  somSelf->PushUndoAction(frame,kODCommandCut,delStart,delEnd,text,style,"Cut");

	return kODTrue;
}

/* FUNCTION: RangesOverlap
 *
 * DESCRIPTION:
 *   Returns kODTrue if the two argument text offset ranges overlap
 */
static ODBoolean RangesOverlap(
	LPSTR   start1,
	LPSTR   end1,
	LPSTR   start2,
	LPSTR   end2)
{
	if (start1 >= end2)
		return kODFalse;
	else if (start2 >= end1)
		return kODFalse;
	else if (start1 < start2)
	{
		if (end1 <= start2)
			return kODFalse;
		else
			return kODTrue;
	}
	else
		return kODTrue;
}

/* METHOD: DoCopy
 *
 * DESCRIPTION:
 *   Copy data to the clipboard
 */
SOM_Scope ODBoolean SOMLINK DoCopy(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODClipboard    *clipboard       = _fSession->GetClipboard(ev);
	ODStorageUnit  *clipBSU         = kODNULL;
	ODUnused(frame);

	/* Put the current selection on the clipboard
	 */
	TRY
		clipboard->Clear(ev);
		clipBSU         = clipboard->GetContentStorageUnit(ev);
		somSelf->ExternalizeSelection(ev,clipBSU);

	/* Add a link spec to the clipboard if we don't already have a
	 * source link, and as long as the selection does not include a
	 * link destination.
	 */
		if (!_fSrcLink.link)
	{
			ODBoolean makeLinkSpec;

	    if (_fDstLink.link)
				makeLinkSpec = !RangesOverlap(_fpselStart,
											  _fpselEnd,
											  _fDstLink.start,
											  _fDstLink.end);
			else
				makeLinkSpec  = kODTrue;

	    if (makeLinkSpec)
	    {
		ODDraft *myDraft = somSelf->GetStorageUnit(ev)->GetDraft(ev);
		ODLinkSpec *ls   = myDraft->CreateLinkSpec(ev,_fPartWrapper,kODNULL,0);
		clipBSU->AddProperty(ev,kODPropContents/*kODPropLinkSpec*/);
		ls->WriteLinkSpec(ev,clipBSU);
		delete ls;

		/* Save away the range for the potential link
		 */
		_fSrcLink.start         = _fpselStart;
		_fSrcLink.end   = _fpselEnd;
	    }
		}

		clipboard->SetPlatformClipboard(ev,NULL);
		_fClipboardChangeID = clipboard->GetChangeID(ev);
    CATCH_ALL
    ENDTRY

	return kODTrue;
}

/* METHOD: DoPaste
 *
 * DESCRIPTION:
 *   Paste in data from the clipboard
 */
SOM_Scope ODBoolean SOMLINK DoPaste(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODClipboard     *clipboard      = _fSession->GetClipboard(ev);
	ODStorageUnit   *clipBSU    = NULL;
	LPSTR                   delStart, delEnd;
	int             newSize;

	/* See if type kODAppleText exists in the clipboard storage
	 * unit...
	 */
	TRY
		clipBSU = clipboard->GetContentStorageUnit(ev);
		if (clipBSU->Exists(ev,kODPropContents, kODAppleTEXT, kODPosAll))
		{
			/* Cut the selected text out of the document
			 */
			if (_fpselStart)
			{
				delStart        = _fpselStart;
				delEnd          = _fpselEnd;
				somSelf->DoClear(ev,frame);
			}
			else
			{
				delStart        = _fpInsert;
				delEnd          = _fpInsert;
			}

			/* Now read in the text
			 */
			ODValue                text = kODNULL;
			ODULong                size = 0;
			clipBSU->Focus(ev,
						   kODPropContents,
						   kODPosUndefined,
						   kODAppleTEXT,
						   (ODValueIndex)0,
						   kODPosFirstSib);
			size = (int)clipBSU->GetSize(ev);
			text = (ODValue)malloc((int)size);
			clipBSU->SetOffset(ev,0);
			clipBSU->GetValue(ev,size,text);

			/* Grow our text buffer segment larger if necessary to
			 * accommodate the additional character that was typed
			 * in. However, note that the buffer only grows in
			 * discrete increments.
			 */
			if ((int)(strlen(_fpText) + size + 1) >= _fTextSize)
			{
				newSize         = (int)(ROUND_SIZE(strlen(_fpText)+size+1));
				if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
					_fTextSize       = newSize;
			}

			/* Paste the text in
			 */
			size    = strlen((LPSTR)text);
			memmove(_fpInsert + (int)size,_fpInsert,(int)(strlen(_fpInsert)+1));
			memcpy(_fpInsert,text,(int)size);

			/* Force the text to be redrawn from here on down
			 */
			_fpIdleUpdate    = _fpInsert;

			/* Free the text
			 */
			free(text);

			/* Read in the font - at the moment this isn't useful since
			 * we always only have one font in the text body: but we will
			 * still try to paste in the font info (it will set the font
			 * of ALL the text).
			 */
			if (clipBSU->Exists(ev,kODPropContents, kODKindWindowsstyl, kODPosAll))
			{
				clipBSU->Focus(ev,kODPropContents,
								kODPosUndefined,
								kODKindWindowsstyl,
								(ODValueIndex)0,
								kODPosFirstSib);
				size    = (int)clipBSU->GetSize(ev);
				if (size >= sizeof(LOGFONT))
				{
					clipBSU->SetOffset(ev,0);
					clipBSU->GetValue(ev,sizeof(LOGFONT),(ODValue)_fpStyl);
				}

				/* Redraw ourselves with the new font
				 */
				_fcyLineHeight           = 0;
				_fpIdleUpdate            = NULL;
				somSelf->UpdateAllViews(ev);
			}
	    somSelf->AdjustLinks(ev,delStart, delEnd,_fpText+strlen(_fpText));
		}
	CATCH_ALL
	ENDTRY

	return kODTrue;
}

/* METHOD: DoPasteLink
 *
 * DESCRIPTION:
 *   Create a paste link to the source on the clipboard
 */
SOM_Scope ODBoolean SOMLINK DoPasteLink(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODClipboard     *clipboard      = _fSession->GetClipboard(ev);
	ODUnused(frame);

    TRY
		/* See if there is a link-spec on the clipboard
		 */
	ODStorageUnit   *clipBSU = clipboard->GetContentStorageUnit(ev);
	if (clipBSU->Exists(ev,
						kODPropContents/*kODPropLinkSpec*/,
							(ODValueType)kODNULL,
							0)
		   )
			somSelf->PasteLink(ev,clipBSU);
    CATCH_ALL
    ENDTRY

	return kODTrue;
}
	
/* METHOD: PasteLink
 *
 * DESCRIPTION:
 *   Paste a link as a result of a drag/drop or clipboard operation
 */
SOM_Scope void SOMLINK PasteLink(
	DragPart                *somSelf,
	Environment             *ev,
	ODStorageUnit   *su)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODLinkSpec          *ls             = new ODLinkSpec;
    su->Focus(ev,
		  kODPropContents/*kODPropLinkSpec*/,
			  kODPosUndefined,
			  (ODValueType) kODNULL,
			  0,
			  kODPosUndefined);
    ls->ReadLinkSpec(ev,su);
    TRY
		_fDstLink.link = somSelf->GetStorageUnit(ev)->GetDraft(ev)->GetLink(ev,(ODStorageUnitID)kODNULL, ls);
		if (_fDstLink.link)
		{
			LPSTR delStart;
			LPSTR delEnd;

			if (_fpselStart)
			{
				delStart        = _fpselStart;
				delEnd          = _fpselStart;
			}
			else
			{
				delStart        = _fpInsert;
				delEnd          = _fpInsert;
			}

			/* Delete the existing text and wait for LinkUpdated()
			 * to be called to get the new text
			 */
			_fpselStart             = NULL;
			_fpselEnd               = NULL;

			/* Adjust other links due to this change
			 * Special cased for source links only for now
			 * Assume no overlapping links or recursion problems
			 */
			somSelf->AdjustSrcLinks(ev,delStart, delEnd, delStart);
			_fDstLink.start         = delStart;
			_fDstLink.end   = delStart;

			/* Now register as a dependent of the link
			 */
			_fDstLink.link->RegisterDependent(ev,
											  _fPartWrapper,
											  _fDstLink.id);

			/* Put the insertion point after the text
			 */
			_fpInsert       = _fDstLink.end;
			if(!_fpInsert)
				_fpInsert       = _fpText;
       }
   CATCH_ALL
   ENDTRY
   delete ls;
}

/* METHOD: DoClear
 *
 * DESCRIPTION:
 *   Clear everything from the clipboard
 */
SOM_Scope ODBoolean SOMLINK DoClear(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame* frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	int                     TextLen;
	ODUnused(frame);

	/* Cut the selected text out of the document
	 */
	if (_fpselStart)
	{
		TextLen = strlen(_fpselEnd);
		memmove(_fpselStart,_fpselEnd,TextLen+1);
		*(_fpselStart + TextLen) = 0;    /* NULL terminate */
		_fpIdleUpdate                    = _fpselStart;
		if ((_fpInsert > _fpselStart) && _fpselStart)
			_fpInsert               -= (_fpselEnd - _fpselStart);
		somSelf->AdjustLinks(ev,_fpselStart, _fpselEnd, _fpselStart);
		_fpselStart              = NULL;
		_fpselEnd                = NULL;
	}

	somSelf->UpdateAllViews(ev);
	return kODTrue;
}

/* METHOD: DoSelectAll
 *
 * DESCRIPTION:
 *   Select all the text
 */
SOM_Scope ODBoolean SOMLINK DoSelectAll(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame* frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(frame);

//  _fpInsert                = _fpText;
//  _fpselStart              = _fpText;
//  _fpselEnd                = _fpText + strlen(_fpText);
//  _fpIdleUpdate    = _fpText;
	return kODTrue;
}

/* METHOD: DoSampleOne
 *
 * DESCRIPTION:
 *   Display some sample text (1)
 */
SOM_Scope ODBoolean SOMLINK DoSampleOne(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	char                    *defOne         = "As I sailed into Shadow,\na white bird of my desire\ncame and sat upon my right shoulder,\nand I wrote a note and tied it to \nits leg and sent in on its way.\n";
	int                     newSize;
	ODUnused(frame);

	if (defOne)
	{
		/* Enlarge our text buffer if necessary and then copy in the sample
		 * text
		 */
		newSize = ROUND_SIZE(strlen(defOne)+1);
		if (newSize != _fTextSize)
		{
			if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
				_fTextSize       = newSize;
		}
		strcpy(_fpText,defOne);

		/* Cause all lines in all of our visible facets to get redrawn on
		 * the next idle-time slice that we get from the system.
		 */
		_fpInsert                = _fpText;
		_fpIdleUpdate            = _fpText;
		_fFrameShapeChanged      = TRUE;
		_fDrawMode               = 0;
		_fpUpdateChar            = 0;
		_fcyLineHeight           = 0;
		return kODTrue;
	}
	else
		return kODFalse;
}

/* METHOD: DoSampleTwo
 *
 * DESCRIPTION:
 *   Display some sample text (2)
 */
SOM_Scope ODBoolean SOMLINK DoSampleTwo(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	char                    *defTwo         = "A black bird of my desire came\nand sat on my left shoulder,\nand I wrote a note and tied it to its leg\nand sent it off into the west.\n";
	int                     newSize;
	ODUnused(frame);

	if (defTwo)
	{
		/* Enlarge our text buffer if necessary and then copy in the sample
		 * text
		 */
		newSize = ROUND_SIZE(strlen(defTwo)+1);
		if (newSize != _fTextSize)
		{
			if (somSelf->TextReAlloc(ev,_fTextHandle,newSize,GMEM_ZEROINIT))
				_fTextSize       = newSize;
		}
		strcpy(_fpText,defTwo);

		/* Cause all lines in all of our visible facets to get redrawn on
		 * the next idle-time slice that we get from the system.
		 */
		_fpInsert               = _fpText;
		_fpIdleUpdate           = _fpText;
		_fFrameShapeChanged     = TRUE;
		_fDrawMode              = 0;
		_fpUpdateChar           = 0;
		_fcyLineHeight          = 0;
		return kODTrue;
	}
	else
		return kODFalse;
}

/* METHOD: DoCommand
 *
 * DESCRIPTION:
 *   Handle a menu event
 */
SOM_Scope ODBoolean SOMLINK DoCommand(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame,
	ODCommandID command)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODBoolean               handled     = kODFalse;
	ODFrameFacetIterator* facets= frame->CreateFacetIterator(ev);

	switch (command)
	{
		case kODCommandCut:
			handled = somSelf->DoCut(ev,frame);
			break;

		case kODCommandCopy:
			handled = somSelf->DoCopy(ev,frame);
			break;

		case kODCommandPaste:
			handled = somSelf->DoPaste(ev,frame);
			break;

		case kODCommandPasteAs:
			handled = somSelf->DoPasteLink(ev,frame);
			break;

		case kODCommandClear:
			handled = somSelf->DoClear(ev,frame);
			break;

		case kODCommandSelectAll:
			handled = somSelf->DoSelectAll(ev,frame);
			break;

		/* View this frame as a window
		 */
		case kODCommandViewAsWin:
			somSelf->Open(ev,frame);
			break;

		/* Put in some sample text into our control
		 */
		case IDM_SAMPLE1:
			handled = somSelf->DoSampleOne(ev,frame);
			break;

		case IDM_SAMPLE2:
			handled = somSelf->DoSampleTwo(ev,frame);
			break;

		/* Select a sample font
		 */
		case IDM_FONT:
			somSelf->PickFont(ev,frame,_fpStyl);
			handled = kODTrue;
			break;
	}
	somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);
	// OLE: note that SetChangedFromPrev also updates cached image when
	// running in OLE container applications
	delete facets;
	return handled;
}

/* METHOD: HandleMenuEvent
 *
 * DESCRIPTION:
 *   Process a menu event and resync the menu bar
 */
SOM_Scope ODBoolean SOMLINK HandleMenuEvent(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame,
	ODEventData *event)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODCommandID     command         = event->wParam;
	ODBoolean               handled         = somSelf->DoCommand(ev,frame,command);

	somSelf->SyncMenuBar(ev,frame);
	somSelf->GetStorageUnit(ev)->GetDraft(ev)->SetChangedFromPrev(ev);
	// OLE: note that SetChangedFromPrev also updates cached image when
	// running in OLE container applications
	return handled;
}

//-------------------------------------------------------------------------
// From Undo Protocol
//-------------------------------------------------------------------------

/* METHOD: PushUndoAction
 *
 * DESCRIPTION:
 *   Add an element to the undo stack
 */
SOM_Scope void SOMLINK PushUndoAction(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame,
	ODCommandID command,
	ODULong         start,
	ODULong         end,
	ODPtr           text,
	ODHandle        style,
	char            *mitext)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(frame);
	ODUnused(command);
	ODUnused(start);
	ODUnused(end);
	ODUnused(text);
	ODUnused(text);
	ODUnused(style);
	ODUnused(mitext);

//  // create action object for Undo stack
//  UndoInfo* uaction = (UndoInfo*)ODNewPtr(sizeof(UndoInfo));
//  uaction->displayFrame = frame;
//  uaction->command = command;
//  uaction->start = start;
//  uaction->end = end;
//  uaction->text = text;
//  uaction->style = style;
//
//  // create Undo menu item text
//  ODName* menuText = (ODName*)ODNewPtr(4+strlen(mitext));
//  menuText->theScriptCode = smSystemScript;
//  menuText->theLangCode = smScriptLang;
//  strcpy(mitext, menuText->theText);
//
//  _fSession->GetUndo()->AddActionToHistory(ev,_fPartWrapper, (ODActionData)uaction,
//                                                                                  kODSingleAction, menuText);
//  fUndoCount++;
//
//  frame->IncrementRefCount();
//
//  somSelf->SyncMenuBar(frame);
}
								
/* METHOD OVERRIDE: UndoAction
 *
 * DESCRIPTION:
 *   Undo a particular action - not supported as yet ...
 */
SOM_Scope void SOMLINK UndoAction(
	DragPart        *somSelf,
	Environment     *ev,
	ODActionData 	action)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(action);

//  WASSERTM((fUndoCount == 0), "Undo unexpected, internal count is zero");
//
//  ODBoolean ok = kODFalse;
//  UndoInfo* uaction = (UndoInfo*)action;
//
//  OrderedCollectionIterator dispIter(_fDisplayFrames);
//  for (ODFrame* frame = (ODFrame*)dispIter.First();
//           dispIter.IsNotComplete();
//           frame = (ODFrame*)dispIter.Next())
//  {
//          if (frame == (*uaction).displayFrame) ok = kODTrue;
//  }
//
//  if (ok)
//  {
//      switch ((*uaction).command)
//      {
//          case kODCommandCut:    UndoRedoCut(uaction, kUndoIt); break;
//          case kODCommandPaste:  UndoRedoPaste(uaction, kUndoIt); break;
//          case kODCommandPasteAs:UndoRedoPasteLink(uaction, kUndoIt); break;
//          case kODCommandClear:  UndoRedoClear(uaction, kUndoIt); break;
//          case cDTSampleOne:      UndoRedoSampleOne(uaction, kUndoIt); break;
//          case cDTSampleTwo:      UndoRedoSampleTwo(uaction, kUndoIt); break;
//          case cDTTyping:         UndoRedoTyping(uaction, kUndoIt); break;
//          case cDTStyling:        UndoRedoStyling(uaction, kUndoIt); break;
//          default:
//              break;
//      }
//      fUndoCount--;
//      fRedoCount++;
//  }
//  somSelf->SyncMenuBar(frame);
}

/* METHOD OVERRIDE: RedoAction
 *
 * DESCRIPTION:
 *   Redo a particular action - not supported as yet ...
 */
SOM_Scope void SOMLINK RedoAction(
	DragPart        *somSelf,
	Environment     *ev,
	ODActionData 	action)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(action);

//  WASSERTM((fRedoCount == 0), "Redo unexpected, internal count is zero");
//
//  ODBoolean ok = kODFalse;
//  UndoInfo* uaction = (UndoInfo*)action;
//
//  OrderedCollectionIterator dispIter(_fDisplayFrames);
//  for (ODFrame* frame = (ODFrame*)dispIter.First();
//           dispIter.IsNotComplete();
//           frame = (ODFrame*)dispIter.Next())
//  {
//      if (frame == (*uaction).displayFrame) ok = kODTrue;
//  }
//
//  if (ok)
//  {
//      switch ((*uaction).command)
//      {
//          case kODCommandCut:    UndoRedoCut(uaction, kRedoIt); break;
//          case kODCommandPaste:  UndoRedoPaste(uaction, kRedoIt); break;
//          case kODCommandPasteAs:UndoRedoPasteLink(uaction, kRedoIt); break;
//          case kODCommandClear:  UndoRedoClear(uaction, kRedoIt); break;
//          case cDTSampleOne:      UndoRedoSampleOne(uaction, kRedoIt); break;
//          case cDTSampleTwo:      UndoRedoSampleTwo(uaction, kRedoIt); break;
//          case cDTTyping:         UndoRedoTyping(uaction, kRedoIt); break;
//          case cDTStyling:        UndoRedoStyling(uaction, kRedoIt); break;
//          default:
//                  break;
//      }
//      fUndoCount++;
//      fRedoCount--;
//  }
//  somSelf->SyncMenuBar(frame);
}

/* METHOD OVERRIDE: DisposeActionState
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK DisposeActionState(
	DragPart        *somSelf,
	Environment     *ev,
	ODActionData 	action,
	ODDoneState  	doneState)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(action);
	ODUnused(doneState);

//  WASSERTM((fUndoCount == 0), "Undo unexpected, internal count is zero");
//  WASSERTM((fRedoCount == 0), "Redo unexpected, internal count is zero");
//
//  UndoInfo* uaction = (UndoInfo*)action;
//
//  (*uaction).displayFrame->Release();
//  if ((*uaction).text)
//          ODDisposePtr((Ptr)(*uaction).text);
//  if ((*uaction).style)
//          ODDisposeHandle((Handle)(*uaction).style);
//
//  ODDisposePtr((ODPtr)uaction);
//
//  switch (doneState) {
//      case kODDone:
//      case kODRedone:
//          fUndoCount--; break;
//      case kODUndone:
//          fRedoCount--; break;
//      default:
//              ;
//  }
}

/* METHOD OVERRIDE: WriteActionState
 *
 * DESCRIPTION:
 *   Save our current action state
 */
SOM_Scope void SOMLINK WriteActionState(
	DragPart        	 	*somSelf,
	Environment             *ev,
	ODPtr                   action,
	ODStorageUnitView       *storageUnitView)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(storageUnitView);
	ODUnused(action);

//  UndoInfo* uaction = (UndoInfo*)action;
//
//  ODStorageUnitRef suRef;
//  suRef = storageUnitView->GetWeakStorageUnitRef((*uaction).displayFrame->GetStorageUnit(ev)->GetID(ev));
//
//  storageUnitView->AddProperty(kODUndoInfo);
//  storageUnitView->AddValue(kODFrameRef);
//  storageUnitView->SetValue(sizeof(ODStorageUnitRef), (ODValue)&suRef);
//
//  storageUnitView->AddValue(kODCommandID);
//  storageUnitView->SetValue(sizeof(ODCommandID), (ODValue)&(*uaction).command);
//
//  storageUnitView->AddValue(kODStart);
//  storageUnitView->SetValue(sizeof(ODULong), (ODValue)&(*uaction).start);
//
//  storageUnitView->AddValue(kODEnd);
//  storageUnitView->SetValue(sizeof(ODULong), (ODValue)&(*uaction).end);
//
//  storageUnitView->AddValue(kODAppleTEXT);
//  storageUnitView->SetValue(GetPtrSize((Ptr)(*uaction).text), (ODValue)(*uaction).text);
//
//  HLock((Handle)(*uaction).style);
//  storageUnitView->AddValue(kODApplestyl);
//  storageUnitView->SetValue(GetHandleSize((Handle)(*uaction).style), (ODValue)(*(*uaction).style));
//  HUnlock((Handle)(*uaction).style);
}

/* METHOD OVERRIDE: ReadActionState
 *
 * DESCRIPTION:
 *   Reload our action state
 */
SOM_Scope ODPtr SOMLINK ReadActionState(
	DragPart                *somSelf,
	Environment             *ev,
	ODStorageUnitView       *storageUnitView)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(storageUnitView);

//  ODStorageUnitRef suRef;
//  ODStorageUnit* su = storageUnitView->GetStorageUnit();
//  ODCommandID command;
//  ODULong start, end;
//  ODPtr text;
//  ODULong size;
//  ODHandle style;
//
//  su->Focus(kODUndoInfo, kODPosSame, kODFrameRef, 1, kODPosUndefined);
//  su->SetOffset(0);
//  su->GetValue(sizeof(ODStorageUnitRef), (ODValue)&suRef);
//
//  su->Focus(kODUndoInfo, kODPosSame, kODCommandID, 1, kODPosUndefined);
//  su->SetOffset(0);
//  su->GetValue(sizeof(ODCommandID), (ODValue)&command);
//
//  su->Focus(kODUndoInfo, kODPosSame, kODStart, 1, kODPosUndefined);
//  su->SetOffset(0);
//  su->GetValue(sizeof(ODULong), (ODValue)&start);
//
//  su->Focus(kODUndoInfo, kODPosSame, kODEnd, 1, kODPosUndefined);
//  su->SetOffset(0);
//  su->GetValue(sizeof(ODULong), (ODValue)&end);
//
//  su->Focus(kODUndoInfo, kODPosSame, kODAppleTEXT, 1, kODPosUndefined);
//  su->SetOffset(0);
//  size = su->GetSize();
//  text = ODNewPtr(size);
//  ThrowIfNULL(text);
//  su->GetValue(size, (ODValue)text);
//
//  su->Focus(kODUndoInfo, kODPosSame, kODApplestyl, 1, kODPosUndefined);
//  su->SetOffset(0);
//  size = su->GetSize();
//  style = ODNewHandle(size);
//  ThrowIfNULL(style);
//  HLock((Handle)style);
//  su->GetValue(size, (ODValue)(*style));
//  HUnlock((Handle)style);
//
//  UndoInfo* uaction = (UndoInfo*)ODNewPtr(sizeof(UndoInfo));
//  ThrowIfNULL(uaction);
//
//  (*uaction).displayFrame = su->GetDraft()->GetFrame(su->GetIDFromStorageUnitRef(suRef));
//  (*uaction).command = command;
//  (*uaction).start = start;
//  (*uaction).end = end;
//  (*uaction).text = text;
//
//  return ((ODPtr)uaction);
	return (ODPtr)kODNULL;
}

/* METHOD: InstallMenus
 *
 * DESCRIPTION:
 *   Install our menus as the current menu bar...
 */
SOM_Scope void SOMLINK InstallMenus(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame* frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	/* Display our menu bar
	 */
	if (_fMenuBar && frame)
		_fMenuBar->Display(ev);
	somSelf->SyncMenuBar(ev,frame);

	/* Show the cursor
	 */
	_fbShowCursor    = TRUE;
	_fDrawMode      = DRAW_MODE_LINE;
	somSelf->UpdateAllViews(ev);
	_fDrawMode       = DRAW_MODE_DEFAULT;
}

/* METHOD: RemoveMenus
 *
 * DESCRIPTION:
 */
SOM_Scope void SOMLINK RemoveMenus(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(frame);

	/* Hide the cursor
	 */
	_fbShowCursor    = FALSE;
	_fDrawMode               = DRAW_MODE_LINE;
	somSelf->UpdateAllViews(ev);
	_fDrawMode       = DRAW_MODE_DEFAULT;
}

/* FUNCTION: LinkSpecOnClipboard
 *
 * DESCRIPTION:
 *   Returns TRUE if we have a link spec lurking on the clipboard
 */
ODBoolean LinkSpecOnClipboard(
	ODSession       *session)
{
	Environment             *ev                     = somGetGlobalEnvironment();
    ODClipboard         *clipboard      = session->GetClipboard(ev);

	ODBoolean               result          = kODFalse;
	ODUnused(session);

    TRY
	ODStorageUnit* su = clipboard->GetContentStorageUnit(ev);
	result                     = su->Exists(ev,
									kODPropContents/*kODPropLinkSpec*/,
										(ODValueType)kODLinkSpec,
										0);
    CATCH_ALL
    ENDTRY

	return result;
}

/* METHOD: SyncMenuBar
 *
 * DESCRIPTION:
 *   Make sure that the Edit menu has grayed out items for things that don't
 *   apply right now.
 */
SOM_Scope void SOMLINK SyncMenuBar(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODBoolean               isSelection;
	ODBoolean               canPaste;
	ODBoolean               canPasteLink;
	ODBoolean               canSelectAll;

	/* If no frame was passed in, use the first one from our frame iterator
	 */
	if (!frame)
		frame = (ODFrame*)_fDisplayFrames->First();

	/* Request the clipboard focus when we adjust the "&Edit" menu...
	 */
	ODArbitrator* arbitrator = _fSession->GetArbitrator(ev);
	if (  (frame == arbitrator->GetFocusOwner(ev, _fClipboardFocus))
	   || (arbitrator->RequestFocus(ev, _fClipboardFocus, frame))
	   )
	{
		/* Handle the Undo and Redo items on the edit menu
		 */
		_fMenuBar->EnableCommand(ev,ID_EDIT_UNDO,!!_fUndoCount);
		_fMenuBar->EnableCommand(ev,ID_EDIT_REDO,!!_fRedoCount);
	
		isSelection = _fpselEnd > _fpselStart;
		_fMenuBar->EnableCommand(ev,kODCommandCut, isSelection);
		_fMenuBar->EnableCommand(ev,kODCommandCopy, isSelection);
		_fMenuBar->EnableCommand(ev,kODCommandClear, isSelection);
		
		canPaste = IsClipboardFormatAvailable(CF_TEXT);
		_fMenuBar->EnableCommand(ev,kODCommandPaste, canPaste);

		/* Do not allow pasting a destination link into a source link -
		 * even a potential one!
		 */
		canPasteLink = !_fDstLink.link && LinkSpecOnClipboard(_fSession);
		if (canPasteLink)
		{
			if (_fSrcLink.end > _fSrcLink.start)
				canPasteLink = !RangesOverlap(_fpselStart,_fpselEnd,
											  _fSrcLink.start,_fSrcLink.end);
		}
		_fMenuBar->EnableCommand(ev,kODCommandPasteAs,canPasteLink);
	}

	canSelectAll = FALSE;
	_fMenuBar->EnableCommand(ev,kODCommandSelectAll,canSelectAll);
}

/* METHOD: ActivatingWindow
 *
 * DESCRIPTION:
 *   Window is being activated. Activate our cursor and so on if we are
 *   the active frame.
 */
SOM_Scope void SOMLINK ActivatingWindow(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if (_fNeedsActivating)
		somSelf->ActivateFrame(ev,frame);
}

/* METHOD: DeActivatingWindow
 *
 * DESCRIPTION:
 *   Remove the caret and other hints that we are active - if we are the
 *   active frame within this document.
 */
SOM_Scope void SOMLINK DeActivatingWindow(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	if (frame == _fSession->GetArbitrator(ev)->GetFocusOwner(ev,_fSelectionFocus))
		_fNeedsActivating = kODTrue;
	else
		_fNeedsActivating = kODFalse;
}

/* METHOD: Idle
 *
 * DESCRIPTION:
 *   Handle idle time updating of the text
 */
SOM_Scope void SOMLINK Idle(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(frame);

	if (_fbFontChanged)
	{
		_fpIdleUpdate    = _fpText;
		_fcyLineHeight   = 0;
		_fbFontChanged   = FALSE;
	}

	/* Let all of our facets do their idle time redrawing if necessary ...
	 * this involves repainting from line fnIdleUpdateLine through to
	 * the last visible line of text.
	 */
	if (_fpIdleUpdate)
	{
		/* Tell all our facets to do their idle update drawing
		 */
		_fDrawMode       = DRAW_MODE_IDLE;
		somSelf->UpdateAllViews(ev);
		_fpIdleUpdate = NULL;
		_fDrawMode        = DRAW_MODE_DEFAULT;
	}
}

/* METHOD OVERRIDE: FrameShapeChanged
 *
 * DESCRIPTION:
 *   Invalidate ourselves if we are resized
 */
SOM_Scope void SOMLINK FrameShapeChanged(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	frame->Invalidate(ev,frame->GetUsedShape(ev, kODNULL), kODNULL);

	frame->ChangeUsedShape(ev,kODNULL,kODNULL);
		
//      somSelf->PositionDocumentParts();
//      somSelf->AdjustScrollBar();
//      TEGetHiliteRgn(fDocument->hiliteRgn, fDocument->theTE);
	_fFrameShapeChanged = FALSE;
}

/* METHOD OVERRIDE: Open
 *
 * DESCRIPTION:
 *   Open this part into its own window
 */
SOM_Scope ODID SOMLINK Open(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODWindow                *window         = kODNULL;

	if (frame)
	{
		/* "View as window" case
		 */
		window = _fSession->GetWindowState(ev)->GetWindow(ev,_fWindowID);
		if (window)
			window->Select(ev);
		else
		{
			window = somSelf->DrgCreateWindow(ev,frame);
			_fWindowID = window->GetID(ev);
			window->Open(ev);
			window->Show(ev);
			window->Select(ev);
		}
	}
	else
	{
		/* Text part as the root part case
		 */
		window = somSelf->DrgCreateWindow(ev,frame);
		_fWindowID = window->GetID(ev);
		window->Open(ev);
		window->Show(ev);
		window->Select(ev);
	}
	return window->GetID(ev);
}

/* METHOD OVERRIDE: MouseEnter
 *
 * DESCRIPTION:
 *   Handle the case where the mouse comes into our frame
 */
SOM_Scope void SOMLINK MouseEnter(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet,
	ODPoint         *where)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(facet);
	ODUnused(where);

//  somSelf->AdjustCursor(where);
}
	
/* METHOD OVERRIDE: MouseLeave
 *
 * DESCRIPTION:
 *   Remove the cursor
 */
SOM_Scope void SOMLINK MouseLeave(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet* facet)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(facet);

//  SetCursor(&(ODASLMQDGlobals.arrow));
}

/* METHOD: DragText
 *
 * DESCRIPTION:
 *   Initiate a drag based on the selected text
 */
SOM_Scope void SOMLINK DragText(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet     *facet,
	ODEventData *event)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
    ODBoolean           makeLinkSpec = FALSE;
	ODRgnHandle     dragRgn;
	ODDragAndDrop   *dad;
	ODStorageUnit   *dadsu;
	ODPart          *destPart;
	ODDropResult    dropResult;
	
	/* Nothing selected: no drag !
	 */
	if (!_fpselStart)
		return;

	/* Set up the drag region for the drag outline - we don't create one
	 * and just pass in an empty region instead.
	 */
	dragRgn = CreateRectRgn(0,0,0,0);

	/* Set up the drag and drop storage unit
	 */
	dad = _fSession->GetDragAndDrop(ev);
	dad->Clear(ev);
	dadsu = dad->GetContentStorageUnit(ev);

	/* Write out the text and font information
	 */
	somSelf->ExternalizeSelection(ev,dadsu);

	 /* Add a link spec to the dragdrop if we don't already have a
	  * source link, and as long as the selection does not include a
	  * link destination.
	  */
	if (!_fSrcLink.link)
	 {
		  if (_fDstLink.link)
				 makeLinkSpec = !RangesOverlap(_fpselStart,
											   _fpselEnd,
											   _fDstLink.start,
											   _fDstLink.end);
		else
			makeLinkSpec  = kODTrue;

		  if (makeLinkSpec)
		  {
				ODDraft         *myDraft = somSelf->GetStorageUnit(ev)->GetDraft(ev);
				ODLinkSpec *ls           = myDraft->CreateLinkSpec(ev,_fPartWrapper,kODNULL,0);
				dadsu->AddProperty(ev,kODPropContents/*kODPropLinkSpec*/);
				ls->WriteLinkSpec(ev,dadsu);
	    delete ls;

	    /* Save away the range for the potential link
	     */
	    _fSrcLink.start     = _fpselStart;
	    _fSrcLink.end               = _fpselEnd;
	}
	}

	/* Do the drag
	 */
	dropResult = dad->StartDrag(ev,facet->GetFrame(ev),"RGNH",(ODPtr)&dragRgn,&destPart,event);
	if (dropResult != kODDropFail)
	{
		if (destPart)
		{
			/* The frame was dropped in the same document
			 */
			if (destPart == _fPartWrapper)
			{
				_fpselStart      = NULL;
				_fpselEnd        = NULL;
				somSelf->UpdateAllViews(ev);
			}
			else if (dad->GetDragAttributes(ev) & kODdragMove)
				somSelf->DoClear(ev,facet->GetFrame(ev));
		}
		else
		{
			/* For a move operation, remove the selection from this container
			 */
			if (dad->GetDragAttributes(ev) & kODdragMove)
				somSelf->DoClear(ev,facet->GetFrame(ev));
		}
	}

	/* Delete the link source specification if it was not used
	 */
	if (  (dropResult == kODDropFail)
		|| !(dad->GetDragAttributes(ev) & kODdragLink)
	   )
	{
		_fSrcLink.start = NULL;
		_fSrcLink.end   = NULL;
	}
	DeleteObject(dragRgn);
}

/* METHOD: SelectText
 *
 * DESCRIPTION:
 *   Allow the user to make a selection in the text. This is just a quick
 *   and dirty selection method for testing purposes - not something to
 *   be used for a real part handler !
 */
SOM_Scope void SOMLINK SelectText(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet     *facet,
	ODEventData *event)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	HWND                    hwnd            = facet->GetWindow(ev)->GetPlatformWindow(ev);
	Point                   xForm, initialPoint;
	HDC                     hdc;
	RECT                    rect, rcInitial, rcInvert, rcNew;
	LPSTR                   pNewPos;
	ODRect                  bRect;

	xForm            = somSelf->GetAggregateTranslation(ev,facet).AsWinPoint();
	initialPoint.x   = LOWORD(event->lParam) - xForm.x;
	initialPoint.y   = HIWORD(event->lParam) - xForm.y;

	/* Clear the window of any previous selection
	 */
	_fbShowCursor    = FALSE;
	_fpselStart      = NULL;
	_fpselEnd        = NULL;
	somSelf->UpdateAllViews(ev);

	/* Get the initial position and rectangle
	 */
	_fpselStart              = somSelf->PositionFromPoint(ev,
														  facet->GetFrame(ev),
														  initialPoint.x,
														  initialPoint.y,
														  &rcInitial);
	_fpselEnd                = _fpselStart;

	CFocus  f(facet, NULL, &hdc);

	facet->GetFrame(ev)->GetFrameShape(ev,kODNULL)->GetBoundingBox(ev,&bRect);
	bRect.AsWinRect( rect );

	/* Draw the initial selection
	 */
	CopyRect(&rcInvert,&rcInitial);
	OffsetRect(&rcInvert,rect.left,rect.top);
	InvertRect(hdc,&rcInvert);

	/* Now go into a modal loop tracking the mouse on this line...
	 */
	MSG             msg;
	BOOL    bMoved  = FALSE;
	SetCapture(hwnd);

	_fSelectingText = TRUE;
	_fpIdleUpdate   = NULL;
	do
	{
		GetMessage(&msg,NULL,NULL,NULL);
		TranslateMessage(&msg);    // Translates virtual key codes
		DispatchMessage(&msg);     // Dispatches message to window

		if (  (msg.message == WM_LBUTTONUP)
		   || (msg.message == WM_RBUTTONUP)
		   || (msg.message == WM_CHAR)
		   )
			break;
		else if (msg.message == WM_MOUSEMOVE)
		{
			initialPoint.x  = LOWORD(msg.lParam) - xForm.x;
			initialPoint.y  = HIWORD(msg.lParam) - xForm.y;
			pNewPos                 = somSelf->PositionFromPoint(ev,
																 facet->GetFrame(ev),
																 initialPoint.x,
																 initialPoint.y,
																 &rcNew);
			if (rcNew.top   == rcInitial.top)
			{
				if ((pNewPos >= _fpselStart) && (pNewPos != _fpselEnd))
				{
					/* Re-draw the selection
					 */
					InvertRect(hdc,&rcInvert);
					CopyRect(&rcInvert,&rcInitial);
					rcInvert.right  = rcNew.right;
					OffsetRect(&rcInvert,rect.left,rect.top);
					InvertRect(hdc,&rcInvert);
					_fpselEnd                = pNewPos;
				}
			}
		}
		else if (msg.message == WM_NULL)
			somSelf->Idle(ev,facet->GetFrame(ev));
	} while(!bMoved);
	ReleaseCapture();

	_fSelectingText  = FALSE;
	_fcxselStart     = rcInvert.right;
	_fcxselEnd       = rcInvert.left;
	_fpselEnd++;
	_fpInsert        = _fpselStart;

	/* Show the selection properly
	 */
	_fbShowCursor    = TRUE;
	somSelf->UpdateAllViews(ev);
}

/* METHOD OVERRIDE: HasExtension
 *
 * DESCRIPTION:
 *   Tell the world that we support the "SemanticInterface" extension
 */
SOM_Scope ODBoolean SOMLINK HasExtension(
	DragPart        *somSelf,
	Environment     *ev,
	ODType extensionName)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(extensionName);

//      ODISOStr       semtIntf = "SemanticInterface";
//      return !ODISOStrCompare( (ODISOStr)extensionName, semtIntf ) ;
	return kODFalse;  // added
}

/* METHOD OVERRIDE: GetExtension
 *
 * DESCRIPTION:
 *   Return our "SemanticInterface" extension
 */
SOM_Scope ODExtension* SOMLINK GetExtension(
	DragPart        *somSelf,
	Environment     *ev,
	ODType extensionName)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(extensionName);

//      ODISOStr       semtIntf = "SemanticInterface";
//      if (!ODISOStrCompare( (ODISOStr)extensionName, semtIntf ))
//              return _fSemtIntf;
//      else
	return kODNULL;
}

/* METHOD: InstallObjectAccessors
 *
 * DESCRIPTION
 */
SOM_Scope void SOMLINK InstallObjectAccessors(
	DragPart        *somSelf,
	Environment     *ev)
{
/*  DragPartData        *somThis        = DragPartGetData(somSelf); */
}

/* METHOD: GetSemanticInterface
 *
 * DESCRIPTION:
 */
SOM_Scope ODSemanticInterface* SOMLINK GetSemanticInterface(
	DragPart        *somSelf,
	Environment     *ev)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

//  return _fSemtIntf ;
	return (ODSemanticInterface*)kODNULL;  // added
}

/* METHOD: PickFont
 *
 * DESCRIPTION:
 *   Pick the current display font for the DragPart
 */
SOM_Scope LOGFONT* SOMLINK PickFont(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame     *frame,
	LOGFONT     *pLogFont)
{
    DragPartData                        *somThis        = DragPartGetData(somSelf);
	ODFrameFacetIterator    *fiter          = frame->CreateFacetIterator(ev);
	CHOOSEFONT                          cf;
	HWND                    hwnd, hwndFocus;

	hwnd = fiter->First(ev)->GetWindow(ev)->GetPlatformWindow(ev);
	delete fiter;
										
	/* Use the Windows api to pick a suitable logical font...
	 */
	memset(&cf, 0, sizeof(CHOOSEFONT));     // Set all structure fields to zero.
	cf.lStructSize  = sizeof(CHOOSEFONT);
	cf.hwndOwner    = hwnd;
	cf.lpLogFont    = pLogFont;
	cf.Flags        = CF_SCREENFONTS | CF_EFFECTS | CF_SCALABLEONLY |
					  CF_INITTOLOGFONTSTRUCT;
	cf.rgbColors    = RGB(0, 0, 0);         // black
	cf.nFontType    = SCREEN_FONTTYPE;

	/* This focus-pocus is needed to make sure we don't loose focus when
	 * we show a dialog when embedded within OLE2 containers !
	 */
	hwndFocus               = GetFocus();
	ChooseFont(&cf);
	SetFocus(hwndFocus);

	/* Force all lines to update with the new font and force the draw
	 * routine to recalculate the line height value
	 */
	_fcyLineHeight           = 0;
	_fpIdleUpdate            = _fpText;
	_fbFontChanged           = TRUE;
	return pLogFont;
}

/* METHOD: GetCursorLineNumber
 *
 * DESCRIPTION:
 *   Get the line number for the current cursor position
 */
SOM_Scope int SOMLINK GetCursorLineNumber(
	DragPart        *somSelf,
	Environment     *ev)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	int             LineNo  = 0;
	LPSTR   pCR;

	/* Count the number of carriage returns in the text - stop when we get
	 * to the cursor...
	 */
	pCR     = _fpText;
	do
	{
		pCR     = strchr(pCR,'\n');
		if (pCR)
		{
			if (pCR >= _fpInsert)
				break;
			else
			{
				LineNo++;
				pCR++;
			}
		}
	} while (pCR);

	return LineNo;
}

/* METHOD: GetExtent
 *
 * DESCRIPTION:
 *   Get the text extent of the given piece of text, and the
 *   width of the cursored character (optional).
 */
SOM_Scope int SOMLINK GetExtent(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame     *frame,
	LPSTR       pText,                  /* IN:  Text string to find extent of */
	char        TempChar,               /* IN:  Character to measure the width of */
	int         *pCharWidth)                /* OUT: Width of the character - or NULL */
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	HDC                             hdcInfo;
	HFONT                                   hOldFont;
	HWND                                    hwnd;
	int                             cxText  = 0;
	ODFrameFacetIterator    *facets = frame->CreateFacetIterator(ev);
	ODFacet*               facet    = facets->First(ev);
	HFONT                   hFont   = CreateFontIndirect(_fpStyl);

	delete facets;

	hwnd    = facet->GetWindow(ev)->GetPlatformWindow(ev);
	hdcInfo = GetDC(hwnd);
	if (hdcInfo)
	{
		hOldFont        = SelectObject(hdcInfo, hFont);
#ifndef WIN32
		cxText          = LOWORD(GetTextExtent(hdcInfo,pText,strlen(pText)));
#else
		SIZE size;
		GetTextExtentPoint (hdcInfo, pText, strlen(pText), &size);

		cxText = size.cx;
#endif

		/* Figure the char rectangle, if necessary
		 */
		if(pCharWidth)
		{
			if(TempChar)
			{
				if (!GetCharWidth(hdcInfo,TempChar,TempChar,pCharWidth))
				{
					ABC     abc;
					GetCharABCWidths(hdcInfo,TempChar,TempChar,&abc);
					*pCharWidth = abc.abcA + abc.abcB + abc.abcC;
				}
			}
			else
				*pCharWidth     = 0;
		}
		SelectObject(hdcInfo,hOldFont);
		ReleaseDC(hwnd,hdcInfo);
		DeleteObject(hFont);
	}
	return  cxText;
}

/* METHOD: PositionCursor
 *
 * DESCRIPTION:
 *   Locate the cursor at x,y
 */
SOM_Scope void SOMLINK PositionCursor(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame     	*focusFrame,
	int         	x,
	int         	y,
	LPSTR			pNewCursor)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	LPSTR           	pNewPosition	= pNewCursor;

	/* Get the new cursor position
	 */
	if (!pNewPosition)
	{
		pNewPosition    = somSelf->PositionFromPoint(ev,focusFrame,x,y,NULL);
		_fxSaved    = -1;
	}
	if (pNewPosition != _fpInsert)
	{
		/* Erase the old cursor
		 */
		if(_fbShowCursor)
		{
			_fDrawMode    = DRAW_MODE_LINE;
			_fbShowCursor = FALSE;
			somSelf->UpdateAllViews(ev);
			_fDrawMode    = DRAW_MODE_DEFAULT;
			_fbShowCursor = TRUE;
		}
	
		_fpInsert        = pNewPosition;
	
		/* Draw the new cursor
		 */
		if(_fbShowCursor)
		{
			_fDrawMode    = DRAW_MODE_LINE;
			somSelf->UpdateAllViews(ev);
			_fDrawMode    = DRAW_MODE_DEFAULT;
		}
	}
}

/* METHOD: PositionFromPoint
 *
 * DESCRIPTION:
 *   Get the text position corresponding to x,y
 */
SOM_Scope LPSTR SOMLINK PositionFromPoint(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame     	*focusFrame,
	int         	x,
	int         	y,
	RECT        	*prc)
{
    DragPartData    *somThis    = DragPartGetData(somSelf);
	int     		LineHeight  = _fcyLineHeight ? _fcyLineHeight : 0xFFFF;
	int             LineNo, Extent, CharWidth;
	LPSTR   		pLine, pLinePrev,pEnd;
	char    		TempChar;

	/* First map the coordinates to 72DPI
	 */
	somSelf->MapPoints(ev,&x,&y);

	/* Reposition the cursor:
	 *  - Get the line number by dividing by _fcyLineHeight
	 *  - Get the exact character by binary search
	 */
	LineNo          = y     / LineHeight ;
	pLine           = _fpText;
	pLinePrev       = pLine;
	while (pLine && LineNo)
	{
		pLine           = strchr(pLine,'\n');
		if (pLine)
		{
			LineNo--;
			pLine++;
			pLinePrev       = pLine;
		}
	}
	pEnd            = strchr(pLinePrev,'\n')
						? strchr(pLinePrev,'\n')
						: pLinePrev + strlen(pLinePrev);
	while (pLinePrev < pEnd)
	{
		TempChar        = *pEnd;
		*pEnd           = 0;
		if(prc)
			Extent          = somSelf->GetExtent(ev,focusFrame,pLinePrev,TempChar,&CharWidth);
		else
			Extent          = somSelf->GetExtent(ev,focusFrame,pLinePrev,0,NULL);
		if (Extent < x)
		{
			*pEnd           = TempChar;
			break;
		}
		*pEnd           = TempChar;
		pEnd--;
	}
	if (prc)
	{
		LineNo          = y     / LineHeight;
		prc->left       = Extent;
		prc->top        = LineNo * _fcyLineHeight;
		prc->right      = Extent + CharWidth;
		prc->bottom     = prc->top + _fcyLineHeight;
	}
	return  pEnd;
}

/* METHOD: GetMyWindow
 *
 * DESCRIPTION:
 *   Get the window that we actually live in, even when we are passed a
 *   facet that is offscreen... otherwise we blow up when someone embeds us
 *   in OLE and we are asked to draw on an offscreen canvas, or when the
 *   user tries to print us. We MUST have a window for some of our drawing
 *   calculations, since metafile/offscreen HDCs frequently don't allow us
 *   to do things like measure the length of a piece of text.
 */
SOM_Scope HWND SOMLINK GetMyWindow(
	DragPart        *somSelf,
	Environment     *ev,
	ODFacet         *facet)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	HWND            hwnd    = NULL;
	ODWindow        *window = NULL;

	if(facet)
		window  = facet->GetWindow(ev);
	if(!window)
	{
		/* Tell the facet that has a window available for it ! It
		 * doesn't matter which window - since we are the same size in all
		 * frames and facets in this implementation
		 */
		OrderedCollectionIterator* frames = _fDisplayFrames->CreateIterator();
		for ( ODFrame* frame = (ODFrame*)frames->First();
			  frames->IsNotComplete() && !window;
			  frame = (ODFrame*)frames->Next()
			)
		{
			ODFrameFacetIterator* facets = frame->CreateFacetIterator(ev);
			for ( ODFacet* facet = facets->First(ev);
				  facets->IsNotComplete(ev);
				  facet = facets->Next(ev)
				)
			{
				window  = facet->GetWindow(ev);
				if (window)
					break;
			}
			delete facets;
		}
		delete frames;
	}
	if(window)
		hwnd    = window->GetPlatformWindow(ev);
	else
		hwnd    = NULL;
	return  hwnd;
}

/* METHOD: CreateWindow
 *
 * DESCRIPTION:
 *   Create our own top level window view of this frame, or create the
 *   root OpenDoc frame if we are the root part
 */
SOM_Scope ODWindow* SOMLINK DrgCreateWindow(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *sourceFrame)
{
    DragPartData                *somThis                = DragPartGetData(somSelf);
	ODPlatformWindow    platformWindow  = kODNULL;
	ODWindow            *window         = kODNULL;
	Rect                windRect;
	
//  if (sourceFrame)
//  {
//      RgnHandle framergn = sourceFrame->GetFrameShape()->GetQDRegion();
//      windRect = (**framergn).rgnBBox;
//  }
//  else
		SetRect(&windRect, 4, 40, 204, 100);
	
	platformWindow =  _fSession->GetWindowState(ev)->CreatePlatformWindow(ev,kODFalse,
																		  WS_CHILD | WS_CLIPSIBLINGS);
	window             =  _fSession->GetWindowState(ev)->RegisterWindow(
							ev,
							platformWindow,
							kODNULL,
							(sourceFrame==kODNULL),        // is root
							kODTrue,                       // Is resizable
							kODFalse,                      // Is floating
							kODTrue,                       // should save
							_fPartWrapper,
							(_fSession->Tokenize(ev,kODViewAsFrame)),
							kODNullTypeToken,
							sourceFrame);
	return window;
}

/* METHOD: UpdateAllViews
 *
 * DESCRIPTION:
 *   Simultaneously update all frames we are displayed in
 */
SOM_Scope void SOMLINK UpdateAllViews(
	DragPart        *somSelf,
	Environment     *ev)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	OrderedCollectionIterator* frames = _fDisplayFrames->CreateIterator();
	for ( ODFrame* frame = (ODFrame*)frames->First();
		  frames->IsNotComplete();
		  frame = (ODFrame*)frames->Next()
		)
	{
		ODFrameFacetIterator* facets = frame->CreateFacetIterator(ev);
		for ( ODFacet* facet = facets->First(ev);
			  facets->IsNotComplete(ev);
			  facet = facets->Next(ev)
			)
			somSelf->Draw(ev,facet,kODNULL );
		delete facets;
	}
	delete frames;
}

/* FUNCTION: WaitMouseMoved
 *
 * DESCRIPTION:
 *   Wait till we get a mousebutton up or mouse move event so we know if this
 *   is a selection, dragging or set-the-cursor type of mouse click !
 */
BOOL WaitMouseMoved(
	HWND    hwnd)
{
	MSG             event;
	int				cMoves			= 0;
	BOOL            bMoved          = FALSE;
	DWORD           InitialTick     = GetTickCount();

	/* Don't capture the mouse if we're minimized!
	 */
	if(IsIconic(hwnd))
		return FALSE;

	SetCapture(hwnd);
	do
	{
		GetMessage(&event,NULL,NULL,NULL);
		TranslateMessage(&event);    // Translates virtual key codes
		DispatchMessage(&event);     // Dispatches message to window
	
		if (  (event.message == WM_LBUTTONUP)
			|| (event.message == WM_RBUTTONUP)
			|| (event.message == WM_LBUTTONDOWN)
			|| (event.message == WM_RBUTTONDOWN)
			|| (event.message == WM_CHAR)
			|| (GetTickCount() < InitialTick)
			|| (GetTickCount() > InitialTick+400)
			)
			break;
		else if (event.message == WM_MOUSEMOVE)
		{
			if(cMoves)
				bMoved  = TRUE;
			cMoves++;
		}
	} while(!bMoved);
	ReleaseCapture();

	/* Don't return TRUE if the mouse button is not really down !
	 */
	if (  (GetKeyState(VK_LBUTTON) < 0)
	   || (GetKeyState(VK_RBUTTON) < 0)
	   )
		return  bMoved;
	else
		return  FALSE;
}

/* METHOD: TextReAlloc
 *
 * DESCRIPTION:
 *   Replacement function for GlobalRealloc in our 16bit implementation. In
 *   Win32, memory blocks move when reallocated - instead of being fixed
 *   based segments that grow and shrink as they were in 16bit. So we have to
 *   adjust our instance data pointers appropriately in this one place.
 */
SOM_Scope HGLOBAL SOMLINK TextReAlloc(
	DragPart        *somSelf,
	Environment     *ev,
	HGLOBAL         hglbMem,
	DWORD           cbBytes,
	UINT            fuFlags)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	LPSTR               pTextOld        = _fpText;
	HGLOBAL             hglbNew;

	GlobalUnlock(hglbMem);
	hglbNew = GlobalReAlloc(hglbMem,cbBytes,fuFlags);
	if (hglbNew)
	{
		_fTextHandle    = hglbNew;
		_fpText         = (LPSTR)GlobalLock(_fTextHandle);
		_fTextSize      = cbBytes;
		if (_fSrcLink.start)
			_fSrcLink.start = _fpText + (_fSrcLink.start - pTextOld);
		if (_fSrcLink.end)
			_fSrcLink.end   = _fpText + (_fSrcLink.end   - pTextOld);
		if (_fDstLink.start)
			_fDstLink.start = _fpText + (_fDstLink.start - pTextOld);
		if (_fDstLink.end)
			_fDstLink.end   = _fpText + (_fDstLink.end   - pTextOld);
		if (_fpselStart)
			_fpselStart             = _fpText + (_fpselStart           - pTextOld);
		if (_fpselEnd)
			_fpselEnd               = _fpText + (_fpselEnd     - pTextOld);
		if (_fpInsert)
			_fpInsert               = _fpText + (_fpInsert     - pTextOld);
		if (_fpIdleUpdate)
			_fpIdleUpdate   = _fpText + (_fpIdleUpdate   - pTextOld);
		if (_fpUpdateChar)
			_fpUpdateChar   = _fpText + (_fpUpdateChar   - pTextOld);
	}
	else
		_fpText         = (LPSTR)GlobalLock(hglbMem);

	return  hglbNew;
}

/* METHOD OVERRIDE: AdjustMenus
 *
 * DESCRIPTION:
 *   Adjust the menu bar appropriately
 */
SOM_Scope void SOMLINK AdjustMenus(
	DragPart        *somSelf,
	Environment     *ev,
	ODFrame         *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	/* If this frame is the root frame, enable printing.
	 */
	if (frame->GetContainingFrame(ev) == kODNULL)
	{
		_fMenuBar->EnableCommand(ev,kODCommandPageSetup, kODTrue);
		_fMenuBar->EnableCommand(ev,kODCommandPrint, kODTrue);
	}
	else
	{
		_fMenuBar->EnableCommand(ev,kODCommandPageSetup, kODFalse);
		_fMenuBar->EnableCommand(ev,kODCommandPrint, kODFalse);
	}

	_fMenuBar->EnableCommand(ev,kODCommandGetPartInfo, kODTrue);
	_fMenuBar->EnableCommand(ev,kODCommandViewAsWin, !frame->IsRoot(ev));

	somSelf->SyncMenuBar(ev,frame);
}

// ########################## Stuff from ParentRCR.CPP #######################

/* METHOD OVERRIDE: HandleEvent
 *
 * DESCRIPTION:
 *   Handle all Windows events coming in to this part
 */
SOM_Scope ODBoolean     SOMLINK HandleEvent(
	DragPart                *somSelf,
	Environment             *ev,
	ODEventData     *event,
	ODFrame                 *frame,
	ODFacet                 *facet)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODBoolean               handled         = kODFalse;
	
	switch (event->message)
	{
		case WM_NULL:
			somSelf->Idle(ev,frame);
			break;

		case WM_LBUTTONDOWN:
			{
				ODPoint point(LOWORD(event->lParam), HIWORD(event->lParam));
				somSelf->HandleMouseDown(ev,facet,&point,event);
				handled = kODTrue;
			}
			break;                                                                                                          //OLE2
			
		case WM_COMMAND:
			handled = somSelf->HandleMenuEvent(ev,frame,event);
			break;

		case WM_KEYDOWN:
		case WM_CHAR:
			somSelf->HandleKeyDown(ev,frame, event);
			handled = kODTrue;
			break;
	
		case WM_ACTIVATE:                                                                                               //OLE2
			handled = kODTrue; // actually ignored by dispatcher            //OLE2
			if (event->wParam  != WA_INACTIVE)
				somSelf->ActivatingWindow(ev,frame);
			else
				somSelf->DeActivatingWindow(ev,frame);
			break;
	}
	return handled;
}

/* METHOD: ActivateFrame
 *
 * DESCRIPTION:
 *   Called when a given frame of the part is being activated
 */
SOM_Scope ODBoolean SOMLINK ActivateFrame(
	DragPart                *somSelf,
	Environment             *ev,
	ODFrame                 *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	ODBoolean succeeded = FALSE;
							
	succeeded = _fSession->GetArbitrator(ev)->RequestFocusSet(ev,_fFocusSet,frame);
	if (succeeded)
	{
		somSelf->FocusAcquired(ev, _fSelectionFocus, frame);
		somSelf->FocusAcquired(ev, _fMenuFocus, frame);
		somSelf->FocusAcquired(ev, _fKeyFocus, frame);
	}
	return TRUE;
}

/* METHOD: DeActivateFrame
 *
 * DESCRIPTION:
 *   Called when one of our frames is deactivated
 */
SOM_Scope void SOMLINK DeActivateFrame(
	DragPart                *somSelf,
	Environment             *ev,
	ODFrame                 *frame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

    ODArbitrator* arbitrator = _fSession->GetArbitrator(ev);
									
    arbitrator->RelinquishFocusSet(ev,_fFocusSet,frame);
	somSelf->FocusLost(ev, _fSelectionFocus, frame);
	somSelf->FocusLost(ev, _fMenuFocus, frame);
	somSelf->FocusLost(ev, _fKeyFocus, frame);

	if ( frame == arbitrator->GetFocusOwner(ev, _fClipboardFocus) )
	{
		arbitrator->RelinquishFocus(ev, _fClipboardFocus, frame);
		somSelf->FocusLost(ev, _fClipboardFocus, frame);
	}
}

/* METHOD OVERRIDE: BeginRelinquishFocus
 *
 * DESCRIPTION:
 *   Let the arbitrator know that we want to let our foci be given away
 */
SOM_Scope ODBoolean SOMLINK BeginRelinquishFocus(
	DragPart                *somSelf,
	Environment             *ev,
	ODTypeToken     		focus,
	ODFrame                 *currentFrame,
	ODFrame                 *proposedFrame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(proposedFrame);
	ODUnused(currentFrame);
	ODUnused(focus);

	return  kODTrue;
}

/* METHOD OVERRIDE: CommitRelinquishFocus
 *
 * DESCRIPTION:
 *   Called when the arbitrator steals away one or more of our foci
 */
SOM_Scope void SOMLINK CommitRelinquishFocus(
	DragPart                *somSelf,
	Environment             *ev,
	ODTypeToken     		focus,
	ODFrame                 *currentFrame,
	ODFrame                 *proposedFrame)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);
	ODUnused(proposedFrame);

	somSelf->FocusLost(ev, focus, currentFrame);
}

/* METHOD: GetAggregateTranslation
 *
 * DESCRIPTION:
 */
SOM_Scope ODPoint SOMLINK GetAggregateTranslation(
	DragPart                *somSelf,
	Environment             *ev,
	ODFacet                 *facet)
{
    DragPartData        *somThis        = DragPartGetData(somSelf);

	ODPoint translation(0,0);
	ODTransform* total = facet->GetWindowContentTransform(ev,kODNULL);
	translation = total->TransformPoint(ev,&translation);
	return translation;
}

/* METHOD OVERRIDE: DisplayFrameConnected
 *
 * DESCRIPTION:
 */
SOM_Scope void  SOMLINK DisplayFrameConnected(
	DragPart        *somSelf,
	Environment 	*ev,
	ODFrame         *frame)
{
	DragPartData *somThis = DragPartGetData(somSelf);

	if (frame->GetPart(ev) == _fPartWrapper)
	{
		/* If the first display frame is being added, register
		 * any link destination
		 */
		if (!_fLinksReConnected)
		{
			if (_fDstLink.link != ((ODLink*) kODNULL))
				_fDstLink.link->RegisterDependent(ev, _fPartWrapper,_fDstLink.id);
			_fLinksReConnected = TRUE;
		}
	}
}

/* METHOD OVERRIDE: DisplayFrameClosed
 *
 * DESCRIPTION:
 */
SOM_Scope void  SOMLINK DisplayFrameClosed(
	DragPart        *somSelf,
	Environment 	*ev,
	ODFrame         *frame)
{
    DragPartData *somThis       = DragPartGetData(somSelf);

	if (_fDisplayFrames->Contains(frame))
	{
		_fSession->GetArbitrator(ev)->RelinquishFocusSet(ev, _fFocusSet, frame);
		if ( frame == _fSession->GetArbitrator(ev)->GetFocusOwner(ev, _fClipboardFocus) )
		{
			_fSession->GetArbitrator(ev)->RelinquishFocus(ev, _fClipboardFocus, frame);
			somSelf->FocusLost(ev, _fClipboardFocus, frame);
		}

		_fSession->GetDispatcher(ev)->UnregisterIdle(ev,_fPartWrapper,frame);
		somSelf->DeActivateFrame(ev,frame);
		_fDisplayFrames->Remove(frame);
		frame->Release(ev);
	}
	else
		THROW(kODErrInvalidFrame);
}

/* METHOD: MapPoints
 *
 * DESCRIPTION:
 *   Map points in window coordinates to our current scaling factor
 */
SOM_Scope void SOMLINK MapPoints(
	DragPart	*somSelf,
	Environment	*ev,
	int			*x,
	int			*y)
{
	DragPartData	*somThis	= DragPartGetData(somSelf);

	*x	= (*x * _fxScaleMultiplier) / _fxScaleDivisor;
	*y	= (*y * _fyScaleMultiplier) / _fyScaleDivisor;
}

/*#################### STUB METHODS THAT DO NOTHING BUT ####################*/
/*####################   HAVE TO BE OVERRIDDEN ANYWAY   ####################*/

SOM_Scope void  SOMLINK CanvasUpdated(
	DragPart	*somSelf,
	Environment *ev,
	ODCanvas	*canvas)
{
	ODUnused(canvas);
}

SOM_Scope void SOMLINK HighlightChanged(
	DragPart	*somSelf,
	Environment *ev,
	ODFacet		*facet)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(facet);
}

SOM_Scope ODULong  SOMLINK GetPrintResolution(
	DragPart	*somSelf,
	Environment *ev,
	ODFrame		*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
	return 0;
}

SOM_Scope void  SOMLINK RevealLink(
	DragPart		*somSelf,
	Environment 	*ev,
	ODLinkSource	*linkSource)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(linkSource);
}

SOM_Scope void  SOMLINK EmbeddedFrameChanged(
	DragPart	*somSelf,
	Environment *ev,
	ODFrame		*frame,
	ODChangeID 	change)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(change);
	ODUnused(frame);
}

SOM_Scope void  SOMLINK LinkStatusChanged(
	DragPart	*somSelf,
	Environment *ev,
	ODFrame		*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
}

SOM_Scope void SOMLINK DropCompleted(
	DragPart        *somSelf,
	Environment		*ev,
	ODPart          *destPart,
	ODDropResult    dropResult)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(destPart);
	ODUnused(dropResult);
}	

SOM_Scope void SOMLINK ContainingPartPropertiesChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame,
	ODStorageUnit	*propertyUnit)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
	ODUnused(propertyUnit);
}

SOM_Scope ODStorageUnit* SOMLINK GetContainingPartProperties(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
	return	NULL;
}

SOM_Scope ODBoolean SOMLINK RevealFrame(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*embeddedFrame,
	ODShape			*revealShape)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFrame);
	ODUnused(revealShape);
	return	FALSE;
}

SOM_Scope void SOMLINK EmbeddedFrameSpec(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*embeddedFrame,
	ODObjectSpec	*spec)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFrame);
	ODUnused(spec);
}

SOM_Scope ODEmbeddedFramesIterator* SOMLINK CreateEmbeddedFramesIterator(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);	
	return	NULL;
}

SOM_Scope void SOMLINK AttachSourceFrame(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame,
	ODFrame			*sourceFrame)
{
	ODUnused(somSelf);
	ODUnused(ev);
}

SOM_Scope void SOMLINK ViewTypeChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
}

SOM_Scope void SOMLINK PresentationChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
}

SOM_Scope void SOMLINK SequenceChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*frame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
}

SOM_Scope ODFrame* SOMLINK RequestEmbeddedFrame(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*containingFrame,
	ODFrame			*baseFrame,
	ODShape			*frameShape,
	ODPart			*embedPart,
	ODTypeToken 	viewType,
	ODTypeToken 	presentation,
	ODBoolean 		isOverlaid)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(containingFrame);
	ODUnused(baseFrame);
	ODUnused(frameShape);
	ODUnused(embedPart);
	ODUnused(viewType);
	ODUnused(presentation);
	ODUnused(isOverlaid);
	return	NULL;
}

SOM_Scope void SOMLINK RemoveEmbeddedFrame(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*embeddedFrame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFrame);
}

SOM_Scope ODShape* SOMLINK RequestFrameShape(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*embeddedFrame,
	ODShape			*frameShape)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFrame);
	ODUnused(frameShape);
	return	NULL;
}

SOM_Scope void SOMLINK UsedShapeChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFrame			*embeddedFrame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFrame);
}

SOM_Scope ODShape* SOMLINK AdjustBorderShape(
	DragPart        *somSelf,
	Environment		*ev,
	ODFacet			*embeddedFacet,
	ODShape			*shape)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(embeddedFacet);
	ODUnused(shape);
	return	NULL;
}

SOM_Scope void SOMLINK CanvasChanged(
	DragPart        *somSelf,
	Environment		*ev,
	ODFacet			*facet)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(ev);
}

SOM_Scope void SOMLINK ExternalizeKinds(
	DragPart        *somSelf,
	Environment		*ev,
	ODTypeList		*kindset)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(kindset);
}

SOM_Scope void SOMLINK ChangeKind(
	DragPart        *somSelf,
	Environment		*ev,
	ODType 			kind)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(kind);
}

SOM_Scope ODBoolean SOMLINK HandleEventInEmbedded(
	DragPart        *somSelf,
	Environment		*ev,
	ODEventData		*event,
	ODFrame			*frame,
	ODFacet			*facet,
	ODFrame			*embeddedFrame,
	ODFacet			*embeddedFacet)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(event);
	ODUnused(frame);
	ODUnused(facet);
	ODUnused(embeddedFrame);
	ODUnused(embeddedFacet);
	return	FALSE;
}

SOM_Scope void SOMLINK MouseWithin(
	DragPart        *somSelf,
	Environment		*ev,
	ODFacet			*facet,
	ODPoint			*where)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(facet);
	ODUnused(where);
}

SOM_Scope void  SOMLINK AbortRelinquishFocus(
	DragPart 	*somSelf,
	Environment *ev,
	ODTypeToken focus,
	ODFrame		*ownerFrame,
	ODFrame		*proposedFrame)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(focus);
	ODUnused(ownerFrame);
	ODUnused(proposedFrame);
}

SOM_Scope void SOMLINK WritePartInfo(
	DragPart                *somSelf,
	Environment             *ev,
	ODPtr                   partInfo,
	ODStorageUnitView       *storageUnitView)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(partInfo);
	ODUnused(storageUnitView);
}

SOM_Scope ODPtr SOMLINK ReadPartInfo(
	DragPart                *somSelf,
	Environment             *ev,
	ODFrame                 *frame,
	ODStorageUnitView       *storageUnitView)
{
	ODUnused(somSelf);
	ODUnused(ev);
	ODUnused(frame);
	ODUnused(storageUnitView);
	return	NULL;
}

