//***************************** Module Header ********************************
// Module Name: sqlsniff.c
//
// Sample NT DB-Libary program
//
// Author PeterWil 1994, Microsoft Ltd.
//****************************************************************************
#define DBNTWIN32

// Standard compiler/sql server supplied header files
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <windows.h>
#include <sqlfront.h>
#include <sqldb.h>
#include <srv.h>

// Application specific header files
#include "sqlsniff.h"
#include "initserv.h"
#include "sniffcbs.h"
#include "dialogs.h"
#include "mngeserv.h"

// Could make these user configurable
// Note 45 is the max for dblib
#define MAX_CONNECTIONS "45"
#define MAX_CONNECTION 45

// define the dblogin timeout period (seconds)
#define LOGIN_TIME_OUT 5

// Globals
HANDLE ghModule;
HWND   ghwndMain = NULL;
HWND   hwndClient;
HICON  hiAwake, hiAsleep, hiIn, hiOut, hiWaiting, hiRPC;

HMENU      hMenu, hMenuWindow;
char       szSvr[MAXSVR+1], szPwd[MAXPWD+1], szRegKey[MAXRKEY+1] = "Server";

// Declare the structures that ODS needs to get started
SRV_CONFIG *config;
SRV_SERVER *server;

// The thread for the ODS part of SQLSNIFF to run on
DWORD      lSrvThreadId;

// A lot of other routines make use of the log to control what is logged
LOG        Log;

// defined in sniffcbs
extern     BOOL bAutoScroll;
extern     BOOL bAutoTile;
extern     BOOL bStartMin;

// Flag to start up straight away without a chance to change the config values
BOOL bAutoStart = FALSE;

// The style for the main window
// Done this way so we can modify it on the fly depending on startup options
DWORD MainStyle = WS_OVERLAPPED | WS_CAPTION | WS_BORDER | WS_THICKFRAME |
                    WS_MAXIMIZEBOX | WS_MINIMIZEBOX | WS_CLIPCHILDREN |
                    WS_VISIBLE | WS_SYSMENU | MDIS_ALLCHILDSTYLES;

// Forward declarations for local routines
int WINAPI   MainWndProc(HWND, UINT, DWORD, LONG);
int WINAPI   About(HWND, UINT, DWORD, LONG);
int WINAPI   InitSniffDlgProc(HWND, UINT, DWORD,LONG);
int WINAPI   StartSrv(void);
BOOL         InitializeApp(void);
BOOL         InitializeODSServer(void);

/***************************************************************************\
* main
*
\***************************************************************************/
int WINAPI WinMain(
    HANDLE  hInstance,
    HANDLE  hPrevInstance,
    LPSTR   lpCmdLine,
    int     nShowCmd)
{
    MSG msg;
    LPSTR lpCommandLine;
    
    // Set up as high priority process
    SetPriorityClass(hInstance,HIGH_PRIORITY_CLASS);

    ghModule = GetModuleHandle(NULL);

    // process the command line
    while ( lpCommandLine = strtok(lpCmdLine," \0") )
        {
        // set lpCmdLine to null so the while loop strtok carries on from where it left off
        lpCmdLine = NULL;

        if ( !strcmp(lpCommandLine,"/R") || !strcmp(lpCommandLine,"/r") )
            {
            // set the registry key name
            lpCommandLine = strtok(NULL," \0");
            strncpy(szRegKey,lpCommandLine,MAXRKEY);
            strcat(szRegKey,"\0");
            }
                
        if ( !strcmp(lpCommandLine,"/S") || !strcmp(lpCommandLine,"/s") )
            {
            // set the server name
            lpCommandLine = strtok(NULL," \0");
            strncpy(szSvr,lpCommandLine,MAXSVR);
            strcat(szSvr,"\0");
            }
                
        if ( !strcmp(lpCommandLine,"/A") || !strcmp(lpCommandLine,"/a") )
            {
            // Autostart (no chance to change the config value)
            bAutoStart = TRUE;
            }

        if ( !strcmp(lpCommandLine,"/M") || !strcmp(lpCommandLine,"/m") )
            {
            // Start SQLSNIFF mimimised
            // Only effective if started from a command line
            // File manager and program manager can over-ride this setting
            MainStyle = MainStyle | WS_MINIMIZE;
            }

        if ( !strcmp(lpCommandLine,"/?")  )
            {
            // Display a very little bit of help
            MessageBox(
                ghwndMain,
                "Command line options\n\t/s Server\n\t/r Registry Key\n\t/a auto initialize\nRedirect to another server by appending @<servername> to the userid when\nlogging on from a client application e.g. set username = 'sa@server2'",
                "SQLSNIFF Help",
                MB_OK);
            MessageBox(
                ghwndMain,
                "The sa can control SQLSNIFF remotely by using the sql statements\n'pause sqlsniff', 'restart sqlsniff' and 'shutdown sqlsniff'",
                "SQLSNIFF Help",
                MB_OK);
            // Could exit here if we wanted to
            // return 0;
            }
        }

    // Initialise SQLSNIFF (e.g. set up the classes etc..)
    // If this fails then exit of SQLSNIFF
    if (!InitializeApp()) {
        ERRBOX("SQLSNIFF: InitializeApp failure!");
        return 0;
    }

    // Main message processing loop 
    while (GetMessage(&msg, NULL, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    // THE END
    return 1;
}

/***************************************************************************\
* InitializeApp
*
\***************************************************************************/
BOOL InitializeApp(void)
{
    WNDCLASS wc;
    char *dbver;

    // Load the standard Icons that we'll use
    hiAwake  = LoadIcon(ghModule,MAKEINTRESOURCE(IDI_OPEN));
    hiAsleep = LoadIcon(ghModule,MAKEINTRESOURCE(IDI_SLEEP)); 
    hiIn     = LoadIcon(ghModule,MAKEINTRESOURCE(IDI_IN));
    hiOut    = LoadIcon(ghModule,MAKEINTRESOURCE(IDI_OUT)); 
    hiWaiting= LoadIcon(ghModule,MAKEINTRESOURCE(IDI_WAITING)); 
    hiRPC    = LoadIcon(ghModule,MAKEINTRESOURCE(IDI_RPC)); 

    // Set up the class information for the main window class
    wc.style            = CS_OWNDC;
    wc.lpfnWndProc      = (WNDPROC)MainWndProc;
    wc.cbClsExtra       = 0;
    wc.cbWndExtra       = 0;
    wc.hInstance        = ghModule;
    wc.hIcon            = hiAsleep;
    wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground    = (HBRUSH)(COLOR_WINDOW+1);
    wc.lpszMenuName     = "MainMenu";
    wc.lpszClassName    = "sqlsniffClass";

    // register the main window class, otherwise exit
    if (!RegisterClass(&wc))
        return FALSE;

    // Set up the class information for the child window class
    wc.style            = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc      = (WNDPROC)ClientWndProc;
    wc.cbClsExtra       = 0;
    wc.cbWndExtra       = sizeof(HANDLE); // to hold the CLIENTINFO handle for the MDI
    wc.hInstance        = ghModule;
    wc.hIcon            = NULL;
    wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground    = (HBRUSH)(COLOR_WINDOW+1);
    wc.lpszMenuName     = NULL;
    wc.lpszClassName    = "sqlsniffClientClass";

    // register the child window class, otherwise exit
    if (!RegisterClass(&wc))
        return FALSE;

    // Load the main menu
    hMenu       = LoadMenu(ghModule, "MainMenu");

    // Set up the menu item under which to place the child windows
    hMenuWindow = GetSubMenu(hMenu, WINDOWMENU);

    // Create the main window
    ghwndMain = CreateWindowEx(
        0L,
        (LPSTR)"sqlsniffClass", 
        (LPSTR)"SQLSNIFF",
        MainStyle,
        50, 50, 400, 400, // Arbitrary values
        NULL,       
        hMenu,
        ghModule,
        NULL);

    // if we couldn't create the main window then exit
    if (ghwndMain == NULL)
        return FALSE;

    // Set Default Logging Options
    // Screen options
    Log.MDI_Queries     = FALSE;
    Log.MDI_All         = TRUE;
    // log file options
    Log.File_Queries    = FALSE;
    Log.File_All        = FALSE;

    // If we can't load dblib there's no point going any further
    if ( (dbver = dbinit()) == NULL)
    {
        ERRBOX("Couldn't initialise DBLIB");
        return FALSE;
    }
    // Could display the version of DBLIB
    // else MSGBOX(dbver);


    // If we can't change the max number of dbprocs the default value is always 25
    if (dbsetmaxprocs(MAX_CONNECTION) != SUCCEED)
        ERRBOX("Couldn't change the maximum number of dbprocs\nDefault setting now being used (25 dbprocs)");

    // If we can't change the login timeout to what we want the default one is 60secs
    if (dbsetlogintime(LOGIN_TIME_OUT)!=SUCCEED)
        ERRBOX("Couldn't change the dblib login timeout\nDefault setting is now being used (60 seconds)");

    // set up the dblib error and message handlers
    dberrhandle(RemoteErr);
    dbmsghandle(RemoteMsgs);

    // Set the focus on the main window
    SetFocus(ghwndMain);

    // Are we going to autostart
    if (bAutoStart)
        // If so initialise the ODS Server
        InitializeODSServer();
    else
        // Call the routine that allows the user to initialise the ODS server
        PostMessage(ghwndMain,WM_COMMAND,IDM_INIT_ODS,0L);

    // All finished here
    return TRUE;
}


/***************************************************************************\
* MainWndProc
*
\***************************************************************************/
int WINAPI MainWndProc(
    HWND hwnd,
    UINT message,
    DWORD wParam,
    LONG lParam)
{
    static int         iCount=1;
    CLIENTCREATESTRUCT clientcreate;

    switch (message)
    {
        case WM_CREATE:

        clientcreate.hWindowMenu  = hMenuWindow;
        clientcreate.idFirstChild = 1;

        hwndClient = CreateWindow("MDICLIENT",
                NULL,
                WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE,
                0,0,0,0,
                hwnd,
                NULL,
                ghModule,
                (LPVOID)&clientcreate);
        return 0L;

        case WM_CLOSE:
            if (server != NULL)
            {
                if (MessageBox(hwnd,
                    "Exiting SQLSNIFF in this way will drop all connections without alerting users.\nDo you wish to proceed ?\n\n(the recommended way to shut down SQLSNIFF is to log on to SQLSNIFF as 'sa' and issue the 'shutdown sqlsniff' command).",
                    "Shutdown",
                    MB_OKCANCEL | MB_ICONEXCLAMATION | MB_TASKMODAL)  == IDOK)
                {
                    PostMessage(hwnd,WM_COMMAND,IDM_PAUSE_SERVER,0L);
                    if (MessageBox(hwnd,
                        "SQLSNIFF has been paused (No new connections).\n\nSelect OK to complete shutdown.\n\n(Cancel will leave SQLSNIFF in a paused state).",
                        "Shutdown",
                        MB_OKCANCEL | MB_ICONEXCLAMATION | MB_TASKMODAL) == IDOK)
                        SendMessage(ghwndMain, WM_DESTROY, 0L, 0L);
                }
            } else SendMessage(ghwndMain, WM_DESTROY, 0L, 0L);
            return 0L;

        case WM_DESTROY:
            // End of the app so tidy up dblib
            dbexit();
            // We can now kill the GUI
            PostQuitMessage(0);
        return 0L;

        case WM_SIZE:   
            // when the main window changes size then tile if necessary
            if (bAutoTile) PostMessage(hwnd,WM_COMMAND,IDM_TILE,0L);
            // Need to pass this on to the client
        return DefFrameProc(hwnd,  hwndClient, message, wParam, lParam);


        case WM_COMMAND:
        switch (LOWORD(wParam))
            {
            // Tile the child windows
            case IDM_TILE:
                SendMessage(hwndClient, WM_MDITILE, 0L, 0L);
            return 0L;

            // Cascade the child windows                                        
            case IDM_CASCADE:
                SendMessage(hwndClient, WM_MDICASCADE, 0L, 0L);
            return 0L;

            // Arrange the child Icons
            case IDM_ARRANGE:
                SendMessage(hwndClient, WM_MDIICONARRANGE, 0L, 0L);
            return 0L;

            // Initialise the ODS part
            case IDM_INIT_ODS:
                DialogBox(ghModule,MAKEINTRESOURCE(IDD_INITBOX), ghwndMain, (WNDPROC)InitSniffDlgProc);
            return 0L;

            // No new connections
            case IDM_PAUSE_SERVER:
                // Modify the menu
                ModifyMenu(hMenu,IDM_PAUSE_SERVER  ,MF_GRAYED ,IDM_PAUSE_SERVER,
                    (LPSTR)"&Pause the gateway (No new connections)");
                ModifyMenu(hMenu,IDM_RESTART_SERVER,MF_ENABLED,IDM_RESTART_SERVER,
                    (LPSTR)"&Restart the gateway (Allow new connections)");
                // Pause the server
                ServerSleep(NULL);
            return 0L;

            // Allow new connections
            case IDM_RESTART_SERVER:
                // Modify the menu
                ModifyMenu(hMenu,IDM_RESTART_SERVER,MF_GRAYED ,IDM_RESTART_SERVER,
                    (LPSTR)"&Restart the gateway (Allow new connections)");
                ModifyMenu(hMenu,IDM_PAUSE_SERVER  ,MF_ENABLED,IDM_PAUSE_SERVER,
                    (LPSTR)"&Pause the gateway (No new connections)");
                // Restart the server
                ServerRestart(NULL);
            return 0L;

            // Exit the application
            case IDM_EXIT:
                SendMessage(ghwndMain, WM_CLOSE, 0L, 0L);
            return 0L;

            // Display the about box
            case IDM_ABOUT:
                if (DialogBox(ghModule, "AboutBox", ghwndMain, (WNDPROC)About) == -1)
                    MessageBox(ghwndMain, "SQLSNIFF: About Dialog Creation Error!", "Error", MB_OK);
            return 0L;

            // Toggle the Auto-scrolling of the child windows
            case IDM_AUTOSCROLL:
            if (bAutoScroll)
            {
                ModifyMenu(hMenu,IDM_AUTOSCROLL,MF_UNCHECKED,IDM_AUTOSCROLL,(LPSTR)"Auto-&Scroll");
                bAutoScroll = FALSE;
            } else {
                ModifyMenu(hMenu,IDM_AUTOSCROLL,MF_CHECKED,IDM_AUTOSCROLL,(LPSTR)"Auto-&Scroll");
                bAutoScroll =TRUE;
            }
            return 0L;
            
            // Toggle the Auto-Tiling of the child windows
            case IDM_AUTOTILE:
            if (bAutoTile) 
            {
                bAutoTile = FALSE;
                ModifyMenu(hMenu,IDM_AUTOTILE,MF_UNCHECKED,IDM_AUTOTILE,(LPSTR)"Auto-&Tile");
                ModifyMenu(hMenu,IDM_CASCADE ,MF_ENABLED  ,IDM_CASCADE ,(LPSTR)"&Cascade");
            } else {
                bAutoTile =TRUE;
                ModifyMenu(hMenu,IDM_AUTOTILE,MF_CHECKED,IDM_AUTOTILE,(LPSTR)"Auto-&Tile");
                ModifyMenu(hMenu,IDM_CASCADE ,MF_GRAYED   ,IDM_CASCADE ,(LPSTR)"&Cascade");
                // Tile the windows
                PostMessage(hwnd,WM_COMMAND,IDM_TILE,0L);
            }
            return 0L;

            // Toggle the start mimised for the child windows
            case IDM_STARTMIN:
            if (bStartMin)
            {
                ModifyMenu(hMenu,IDM_STARTMIN,MF_UNCHECKED,IDM_STARTMIN,(LPSTR)"Start &Minimized");
                bStartMin = FALSE;
            } else {
                ModifyMenu(hMenu,IDM_STARTMIN,MF_CHECKED,IDM_STARTMIN,(LPSTR)"Start &Minimized");
                bStartMin = TRUE;
            }
            return 0L;

            // Toggle the Log All to File
            case IDM_LOG_ALL:
            if (Log.File_All)
            {
                ModifyMenu(hMenu,IDM_LOG_ALL, MF_UNCHECKED, IDM_LOG_ALL,
                    (LPSTR)"Log &All to file (Queries, Results, Errors, Attentions)");
                Log.File_All = FALSE;
            } else {
                ModifyMenu(hMenu,IDM_LOG_ALL, MF_CHECKED,   IDM_LOG_ALL,
                    (LPSTR)"Log &All to file (Queries, Results, Errors, Attentions)");
                Log.File_All = TRUE;
            }
            return 0L;

            // Toggle the Log Queries to File
            case IDM_LOG_QUERIES:
            if (Log.File_Queries)
            {
                ModifyMenu(hMenu,IDM_LOG_QUERIES, MF_UNCHECKED,IDM_LOG_QUERIES,
                    (LPSTR)"Log &Queries to file");
                Log.File_Queries = FALSE;
            } else {
                ModifyMenu(hMenu,IDM_LOG_QUERIES, MF_CHECKED,  IDM_LOG_QUERIES,
                    (LPSTR)"Log &Queries to file");
                Log.File_Queries = TRUE;
            }
            return 0L;

            default:
            return DefFrameProc(hwnd,  hwndClient, message, wParam, lParam);
            }

        default:
        return DefFrameProc(hwnd,  hwndClient, message, wParam, lParam);
    }
}


/***************************************************************************\
* About
*
* About dialog proc.
*
/***************************************************************************/
int WINAPI About(HWND hDlg,UINT message,DWORD wParam,LONG lParam)
{
    switch (message)
    {
        case WM_INITDIALOG:
            return TRUE;

        case WM_COMMAND:
            if (wParam == IDOK)
                EndDialog(hDlg, wParam);
            break;
    }

    return FALSE;
}  

//****************************** Function Header ******************************
//
// Function : InitSniffDlgProc
//
// Description: Handles the initialisation dialog box
//
//*****************************************************************************

int WINAPI InitSniffDlgProc(HWND hDlg,UINT message,DWORD wParam,LONG lParam)
{
    switch (message)
    {
        case WM_INITDIALOG:
            // Set the server and regkey values
            SendDlgItemMessage(hDlg,IDD_PIPE,  EM_REPLACESEL,(DWORD)0,(LONG)szRegKey);
            SendDlgItemMessage(hDlg,IDD_SERVER,EM_REPLACESEL,(DWORD)0,(LONG)szSvr);
            // Set Check boxes with current values
            SendDlgItemMessage(hDlg, IDD_MDI_QUERIES, BM_SETCHECK,
                Log.MDI_Queries, (LONG) 0);
            SendDlgItemMessage(hDlg, IDD_MDI_ALL, BM_SETCHECK,
                Log.MDI_All, (LONG) 0);
            SendDlgItemMessage(hDlg, IDD_FILE_QUERIES, BM_SETCHECK,
                Log.File_Queries, (LONG) 0);
            SendDlgItemMessage(hDlg, IDD_FILE_ALL, BM_SETCHECK,
                Log.File_All, (LONG) 0);
            return TRUE;

        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
                case IDD_OK:
                    // Retrieve server name
                    strcpy(szSvr, "");
                    GetDlgItemText(hDlg, IDD_SERVER, szSvr, MAXSVR);
                    strcat(szSvr, "\0");

                    // Retrieve key name
                    strcpy(szRegKey, "");
                    GetDlgItemText(hDlg, IDD_PIPE, szRegKey, MAXRKEY);
                    strcat(szRegKey, "\0");

                    // Retrieve logging options
                    Log.MDI_Queries = SendDlgItemMessage(hDlg,IDD_MDI_QUERIES,
                        BM_GETCHECK, (DWORD) 0, (LONG) 0);
                    Log.MDI_All = SendDlgItemMessage(hDlg,IDD_MDI_ALL,
                        BM_GETCHECK, (DWORD) 0, (LONG) 0);
                    Log.File_Queries = SendDlgItemMessage(hDlg,IDD_FILE_QUERIES,
                        BM_GETCHECK, (DWORD) 0, (LONG) 0);
                    Log.File_All = SendDlgItemMessage(hDlg,IDD_FILE_ALL,
                        BM_GETCHECK, (DWORD) 0, (LONG) 0);

                    // Initilaise the ODS server
                    if ( !InitializeODSServer() )
                        return -1;

                    // End the dialog
                    EndDialog(hDlg, wParam);
                    return 0L;

                case IDD_CANCEL:
                    EndDialog(hDlg, wParam);
                    return 0L;
            }
    }
    return FALSE;
}


//****************************** Function Header ******************************
//
// Function : InitialiseODSServer
//
// Description: Initialises the ODS Server
//
//*****************************************************************************
BOOL InitializeODSServer(void)
{
    char szTmp[256];
    HANDLE hThread;

    // Send the name retrieved to the gateway's DLL module
    SetRemoteServerName(szSvr);

    // Allocate a configuration structure that is used to
    //initialize the Open Data Services application
    config = srv_config_alloc();

    if (config == NULL)
    {
        ERRBOX("Unable to allocate config structure.");
        return FALSE;
    }

    // Allow MAX_CONNECTIONS  connections at a time.
    srv_config(config, (DBINT)SRV_CONNECTIONS, MAX_CONNECTIONS, SRV_NULLTERM);
                    
    // Set the log file.
    srv_config(config, (DBINT)SRV_LOGFILE, "sqlsniff.log", SRV_NULLTERM);

    // Install the Open Data Services error handler.
    srv_errhandle(ChkErr);

    // Initialize the gateway and save the server handle
    // so it can be used in later functions.
    server = srv_init(config, szRegKey, SRV_NULLTERM);

    // Release the config structure
    srv_free(config);

    if (server == NULL)
    {
        ERRBOX("Unable to initialize Gateway.  Check log file");
        return FALSE;
    }

    // When starting the gateway, initialize the remote server 
    // structure. This is done in the InitServer() function.
    // All the other event handlers are also defined in the
    // InitServer() function.
    srv_handle(server, (DBINT)SRV_START, InitServer);

    // Now everything's ready to go with our gateway, so we
    // start it and keep it going until we get a stop request.
    srv_log(server, TRUE, "SQLSNIFF Starting", SRV_NULLTERM);

    // Start the Gateway on a separate thread
    if ( (hThread = CreateThread(NULL, 0,
        (LPTHREAD_START_ROUTINE)StartSrv,
        server,
        (DWORD)NULL,
        &lSrvThreadId)) == NULL)
    {
        ERRBOX("Unable to create server thread.");
        return FALSE;
    }
            
    // modify the control menu options
    ModifyMenu(hMenu,IDM_INIT_ODS      ,MF_GRAYED ,IDM_INIT_ODS,
        (LPSTR)"&Start the gateway");
    ModifyMenu(hMenu,IDM_PAUSE_SERVER  ,MF_ENABLED,IDM_PAUSE_SERVER,
        (LPSTR)"&Pause the gateway (No new connections)");
    ModifyMenu(hMenu,IDM_RESTART_SERVER,MF_GRAYED ,IDM_RESTART_SERVER,
        (LPSTR)"&Restart the gateway (Allow new connections)");
    // modify the log menu options
    ModifyMenu(hMenu,IDM_LOG_QUERIES  ,MF_ENABLED,IDM_LOG_QUERIES,
        (LPSTR)"Log &Queries to file");
    if (Log.File_Queries) ModifyMenu(hMenu,IDM_LOG_QUERIES  ,MF_CHECKED,IDM_LOG_QUERIES,
            (LPSTR)"Log &Queries to file");
    ModifyMenu(hMenu,IDM_LOG_ALL  ,MF_ENABLED,IDM_LOG_ALL,
        (LPSTR)"Log &All to file (Queries, Results, Errors, Attentions)");
    if (Log.File_All) ModifyMenu(hMenu,IDM_LOG_ALL  ,MF_CHECKED,IDM_LOG_ALL,
            (LPSTR)"Log &All to file (Queries, Results, Errors, Attentions)");
                    
    // Change the main title to match the settings
    sprintf(szTmp,"%s %s %s",szSvr,szRegKey,server->connect_name);
    SetWindowText(ghwndMain,szTmp);

    return TRUE;
}

//****************************** Function Header ******************************
//
// Function : StartSrv
//
// Description: Starts the Server
//
//*****************************************************************************
int WINAPI StartSrv(void)
{
    if (srv_run(server) == FAIL)
        MessageBox(ghwndMain,
            "Unable to start the gateway",
            "ODS Error",
            MB_OK | MB_ICONHAND);
    else
    {
        MessageBox(ghwndMain,
            "Gateway has been shut down in response to a remote request.",
            "ODS Message",
            MB_OK | MB_ICONINFORMATION);
    }

    // destroy the window
    PostMessage(ghwndMain,WM_DESTROY,0L,0L);

    return 1;
}

