//****************************** Function Header ******************************
//
// Function : InitServer
//
// Description: Sets up call-back functions etc.
//
// Author PeterWil 1994, Microsoft Ltd.
//*****************************************************************************
#define DBNTWIN32

// Compiler and SQL header files
#include    <windows.h>
#include    <sqlfront.h>
#include    <sqldb.h>
#include    <srv.h>
#include    <stdio.h>
#include    "sqlsniff.h"
#include    "initserv.h"

// Application specific header files
#include    "rpcexec.h"
#include    "exitremo.h"
#include    "langexec.h"
#include    "rowstuff.h"
#include    "initremo.h"
#include    "mngeserv.h"

// Forward declarations for the dblib error/msg handlers
int RemoteMsgs(DBPROCESS  *,DBINT,INT,INT,char *);
int RemoteErr(DBPROCESS *,int,int,int,char *,char *);

// Forward declaration for the Attention handler 
// Used if the user isues a dbcancel
int attn_handler(SRV_PROC *);

// The remote server name of this gateway
extern DBCHAR   *remote_server;

// A global for the client trying to connect which we protect
// by using the following semaphore
extern SRV_PROC *Newsrvproc;
extern HANDLE    InitRemote_SEM;

// We use this semaphore to ensure that the MDI windows are
// initialised before we allow the initremo to continue
extern HANDLE    InitMDI_SEM;

// declare the log
extern LOG Log;

// declare the main window handle
extern HWND ghwndMain;

#pragma check_stack( off )  // turn off stack checking


int InitServer(SRV_SERVER *server)
{
    char log_buffer[LOG_BUF_SIZE];

    // When we get a connection request from a client, we want to
    // call "InitRemote()" to make a connection to the remote
    // server.
    srv_handle(server, (DBINT)SRV_CONNECT, InitRemote);

    // When the client issues a language request, call
    // "LangExecute()" to send the SQL statement to the remote DBMS.
    srv_handle(server, (DBINT)SRV_LANGUAGE, LangExecute);

    // When the client issues an RSP, call "RPCExecute()"
    // to send the RSP to the remote DBMS (the SQL Server).
    srv_handle(server, (DBINT)SRV_RPC, RPCExecute);

    // When a disconnect request is issued, call "ExitRemote()"
    // to close the connection to the remote DBMS.
    srv_handle(server, (DBINT)SRV_DISCONNECT, ExitRemote);

    // Install the handler that will be called when the
    // gateway receives an attention from one of its
    // clients. An attention gets received anytime a client
    // calls dbcancel().
    srv_handle(server, (DBINT)SRV_ATTENTION, attn_handler);

    // Now install the handlers for the control services
    // these are here ready for when we make SQL Sniff a service
    srv_handle(server, (DBINT)SRV_SLEEP  , ServerSleep);
    srv_handle(server, (DBINT)SRV_RESTART, ServerRestart);

    // Now we'll install the message and error handlers for any
    // messages from the remote DBMS.
    dberrhandle(RemoteErr);
    dbmsghandle(RemoteMsgs);

    // Create semaphore objects

    // This one use fo a critical section to protect Newsrvproc
    InitRemote_SEM  = CreateSemaphore(NULL, 1, 1, NULL);
    // This one used by CreateMDI to wait for WM_CREATE to complete
    InitMDI_SEM     = CreateSemaphore(NULL, 1, 1, NULL);

    if (Log.File_Queries || Log.File_All)
    {
        // Log Server information to log file
        sprintf(log_buffer,
            "SQLSNIFF started... Registry Key = %s, Remote Server = %s, Client Connections: %s",
            srv_sfield(server,SRV_SERVERNAME, (int *)NULL),
            remote_server,
            srv_sfield(server, SRV_CONNECTIONS, (int *)NULL));
        srv_log(server, TRUE, log_buffer, SRV_NULLTERM);

    }

    return(SRV_CONTINUE);
}

//****************************** Function Header ******************************
//
// Function : ChkErr
//
// Description: Checks for errors!
//
//*****************************************************************************
int ChkErr(SRV_SERVER *server,
           SRV_PROC   *srvproc,
           int         errornum,
           BYTE        severity,
           BYTE        state,
           int         oserrnum,
           DBCHAR     *errtext,
           int         errtextlen,
           DBCHAR     *oserrtext,
           int         oserrtextlen)
{
    char log_buffer[LOG_BUF_SIZE];

    // Operating system error? Record in the log file
    if (oserrnum != SRV_ENO_OS_ERR)
    {
        sprintf(log_buffer, "SERVER OS ERROR: %d: %s.", oserrnum, oserrtext);
        srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
    }

    // Is this a fatal error for the gateway?
    if (severity >= SRV_FATAL_SERVER )
    {
        sprintf(log_buffer,
            "SERVER: FATAL SERVER ERROR: errornum = %d, severity = %d, state = %d: %s.",
            errornum,
            severity,
            state,
            errtext);

        srv_log(server, TRUE, log_buffer, SRV_NULLTERM);

        MessageBox(ghwndMain,
            log_buffer,
            "Error from Server Process",
            MB_ICONSTOP | MB_OK | MB_TASKMODAL);

        return(SRV_STOP);
    } else {
        // Did the "srvproc" get a fatal error? 
        if (severity >= SRV_FATAL_PROCESS)
        {
            sprintf(log_buffer,
                "SERVER: FATAL CONNECT ERROR: errornum = %d, severity = %d, state = %d: %s.",
                errornum,
                severity,
                state,
                errtext);

            srv_log(server, TRUE, log_buffer, SRV_NULLTERM);

            return(SRV_CANCEL);
        }
    }

    // A non-fatal error or an information message received.
    // We'll pass it through to the client.
    if (srvproc != (SRV_PROC *)NULL && (server != NULL))
        if (severity < 10)
        {   
            // If informational message
            // Want the user to know it's from SQLSniff
            // Otherwise get problems deciding whether the problem
            // is on the gateway or the server
            strcpy(log_buffer,"SQLSNIFF ODS : ");
            strcat(log_buffer,errtext);
            srv_sendmsg(srvproc,
                SRV_MSG_INFO,
                (DBINT)errornum,
                severity,
                0,
                NULL,
                0,
                0,
                log_buffer,
                SRV_NULLTERM);
        } else {
            // Must be an error message
            // Want the user to know it's from SQLSniff
            // Otherwise get problems deciding whether the problem
            // is on the gateway or the server
            strcpy(log_buffer,"SQLSNIFF ODS : ");
            strcat(log_buffer,errtext);
            srv_sendmsg(srvproc,
                SRV_MSG_ERROR,
                (DBINT)errornum,
                severity,
                0,
                NULL,
                0,
                0,
                log_buffer,
                SRV_NULLTERM);
        } else {
            sprintf(log_buffer,
                "GATEWAY ERROR: errornum = %d, severity = %d: %s",
                errornum, severity, errtext);

            srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
        }
    return(SRV_CONTINUE);
}

//****************************** Function Header ******************************
//
// Function : RemoteMsgs
//
// Description: Handles messages from remote DBLIB / Remote SQL Server
//
//*****************************************************************************
int RemoteMsgs(DBPROCESS  *dbproc,
               DBINT       msgno,
               INT  msgstate,
               INT  severity,
               char       *msgtext)
{
    REMOTE_DBMS *remote;
    SRV_PROC    *srvproc;
    SRV_SERVER  *server;
    char        log_buffer[LOG_BUF_SIZE];

    // If a remote DBMS error was received during the remote
    // open, the dbproc is NULL and a message is sent back on the
    // most recent srvproc.
    if (dbproc == (DBPROCESS *)NULL)
        srvproc = Newsrvproc;
    else
        if ((srvproc = (SRV_PROC *)dbgetuserdata(dbproc)) == NULL)
        // An error was received after the dbproc was assigned, but
        // before we were able to associate our srvproc.
        srvproc = Newsrvproc;

    if (severity < 10)
    {
        // An informational message
        // Want the user to know it's from SQLSniff
        // Otherwise get problems deciding whether the problem
        // is on the gateway or the client
        strcpy(log_buffer,"SQLSNIFF DBLIB : ");
        strcat(log_buffer,msgtext);
        srv_sendmsg(srvproc,
            SRV_MSG_INFO,
            msgno,
            (DBTINYINT)severity,
            (DBTINYINT)msgstate,
            NULL,
            0,
            0,
            log_buffer,
            SRV_NULLTERM);
    } else {
        // must be an error message
        // Want the user to know it's from SQLSniff
        // Otherwise get problems deciding whether the problem
        // is on the gateway or the client
        strcpy(log_buffer,"SQLSNIFF DBLIB : ");
        strcat(log_buffer,msgtext);
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            msgno,
            (DBTINYINT)severity,
            (DBTINYINT)msgstate,
            NULL,
            0,
            0,
            log_buffer,
            SRV_NULLTERM);
    }

    remote = (REMOTE_DBMS *)srv_getuserdata(srvproc);

    if (remote && Log.MDI_All && msgno != 5701 && msgno != 5703)
    {
        AddRow(remote->hAllInfo, "DB-Library Message:", colorRed);
        AddRow(remote->hAllInfo, msgtext, colorBlack);
    }

    if (Log.File_All && msgno != 5701 && msgno != 5703)
    {
        server = SRV_GETSERVER(srvproc);
        sprintf(log_buffer,
            "Proc ID: %s DB-Library Message: %s",
            srv_pfield(srvproc, SRV_SPID, (int *)NULL),
            msgtext);
        srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
    }

    return(0);
}

//****************************** Function Header ******************************
//
// Function : RemoteErr
//
// Description: Handles errors from remote SQL Server
//
//*****************************************************************************
int RemoteErr(DBPROCESS  *dbproc,
              int severity,
              int dberr,
              int oserr,
              char  *dberrstr,
              char  *oserrstr)
{
    REMOTE_DBMS *remote;
    SRV_PROC    *srvproc = (SRV_PROC *)NULL;
    SRV_SERVER  *server;
    char         log_buffer[LOG_BUF_SIZE];


    // If the DBLIB process is dead or we get a DBLIB error 10007
    // ("General SQL Server Error:...") then simply ignore it.  The error
    // message has already been sent to the client.
    if (dberr == 10007)
        return(INT_CANCEL);

    // A remote DBMS error may have been issued during the remote
    // open. In this case, the dbproc will be NULL and a message
    // will be sent on the most recent srvproc.
    if (dbproc == (DBPROCESS *)NULL)
        srvproc = Newsrvproc;
    else
        if ((srvproc = (SRV_PROC *)dbgetuserdata(dbproc)) == NULL)
            // An error was issued after the dbproc was assigned but before
            // we were able to associate our srvproc.
            srvproc = Newsrvproc;

    // Send error message to client.
    // Want the user to know it's from SQLSniff
    // Otherwise get problems deciding whether the problem
    // is on the gateway or the client
    strcpy(log_buffer,"SQLSNIFF DBLIB : ");
    strcat(log_buffer,dberrstr);
    srv_sendmsg(srvproc,
        SRV_MSG_ERROR,
        (DBINT)REMOTE_MSG,
        (DBTINYINT)severity,
        (DBTINYINT)0,
        NULL,
        0,
        0,
        log_buffer,
        SRV_NULLTERM);

    remote = (REMOTE_DBMS *)srv_getuserdata(srvproc);

    if (remote && Log.MDI_All)
    {
        AddRow(remote->hAllInfo, "DB-Library Error:", colorRed);
        AddRow(remote->hAllInfo, dberrstr, colorBlack);
    }

    if (Log.File_All)
    {
        server = SRV_GETSERVER(srvproc);
        sprintf(log_buffer,
            "Proc ID: %s DB-Library Error: %s",
            srv_pfield(srvproc, SRV_SPID, (int *)NULL),
            dberrstr);
        srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
    }


    if (oserr != DBNOERR)
    {
    // Want the user to know it's from SQLSniff
    // Otherwise get problems deciding whether the problem
    // is on the gateway or the client
    strcpy(log_buffer,"SQLSNIFF DBLIB OS : ");
    strcat(log_buffer,oserrstr);
    srv_sendmsg(srvproc,
        SRV_MSG_ERROR,
        (DBINT)REMOTE_MSG,
        (DBTINYINT)severity,
        (DBTINYINT)0,
        NULL,
        0,
        0,
        log_buffer,
        SRV_NULLTERM);
    }
    return(INT_CANCEL);
}

//****************************** Function Header ******************************
//
// Function : AttnHandler
//
// Description: Handles "attention" events (dbcancel)
//
//*****************************************************************************
int attn_handler(SRV_PROC  *srvproc)
{
    REMOTE_DBMS *remote;
    SRV_SERVER *server;

    remote = (REMOTE_DBMS *)srv_getuserdata(srvproc);

    if (Log.MDI_All)
    {
        AddRow(remote->hAllInfo, "===> Attention received <===", colorRed);
    }

    if (Log.File_All)
    {
        server = SRV_GETSERVER(srvproc);
        srv_log(server, TRUE, "!!! Attention Received !!!", SRV_NULLTERM);
    }

    return(SRV_CONTINUE);
}

                                         
//****************************** Function Header ******************************
//
// Function : SetRemoteServerName
//
// Description: Sets server name to redirect queries to
//
//*****************************************************************************
void SetRemoteServerName(char *name)
{
    remote_server = name;
}

   
#pragma check_stack()   // set stack checking to its default setting

