//****************************** Function Header ******************************
//
// Function : InitRemote
//
// Description: Connect to remote SQL Server and create an MDI child window
//
// Author PeterWil 1994, Microsoft Ltd.
//*****************************************************************************
#define DBNTWIN32

// Standard compiler and SQL header files
#include    <windows.h>
#include    <sqlfront.h>
#include    <sqldb.h>
#include    <srv.h>
#include    <stdio.h>

// Application specific header files
#include    "sqlsniff.h"
#include    "initremo.h"
#include    "sniffcbs.h"
#include    "rowstuff.h"

extern HWND ghwndMain;
extern LOG  Log;
extern BOOL bAutoTile;

// Used to make sure Newsrvproc is safe
HANDLE    InitRemote_SEM;

DBCHAR   *remote_server = NULL;
SRV_PROC *Newsrvproc;

// The handler used while the server is paused to not allow users to connect
int DoNotInitRemote(SRV_PROC  *srvproc)
{
    // We'll still let the sa log in just in case they want to shutdown the server
    if ( strcmp( srv_pfield(srvproc,SRV_USER,NULL),"sa") == 0 )
        return InitRemote(srvproc);
    else
        // Send a message to the client that
        // the gateway is paused.
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)REMOTE_FAIL,
            (DBTINYINT)14,
            (DBTINYINT)1,
            NULL,
            0,
            0,
            "Login Failed - SQLSNIFF : Gateway is paused. All new logins have been disabled.",
            SRV_NULLTERM);
    return(SRV_DISCONNECT);
}

// The normal handler to allow users to connect
int InitRemote(SRV_PROC  *srvproc)
{
    char          szBuf[250];
    REMOTE_DBMS  *remote;
    SRV_SERVER   *server;
    char          log_buffer[LOG_BUF_SIZE];
    DBCHAR        TempServer[MAXSVR+1],TempUser[MAXUSR+MAXSVR+1],*charpointer;
    HANDLE        hQueryInfo,hAllInfo;
    PCLIENTINFO   pInfo;
    char          szTitle[80];
    HWND          hwndAll,hwndQuery;

    // Set Newsrvproc.  This is used if we get an error on
    // the open from DBLIB.  Since there isn't a dbproc,
    // it is clear which srvproc to send the msg back on when the
    // DBLIB error-handler gets called.
    //
    // First lock out any other threads trying to connect using
    // this same function or we'll get problems with Newsrvproc.
    WaitForSingleObject(InitRemote_SEM, (DWORD)SRV_INDEFINITE_WAIT);

    Newsrvproc = srvproc;

    // Allocate a REMOTE_DBMS information structure.
    remote = (REMOTE_DBMS *)srv_alloc((DBINT)sizeof(*remote));

    // Try to open a connection to the remote DBMS.
    if (remote == NULL)
    {
        // Send a message to the client that
        // the remote connection failed.
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)REMOTE_FAIL,
            (DBTINYINT)14,
            (DBTINYINT)1,
            NULL,
            0,
            0,
            "Login Failed - SQLSNIFF : Unable to allocate memory for REMOTE_DBMS structure. Disconnecting.",
            SRV_NULLTERM);
        // Now allow other threads to enter this function.
        srv_setuserdata(srvproc, (BYTE *)NULL);
        ReleaseSemaphore(InitRemote_SEM, 1, NULL);
        return(SRV_DISCONNECT);
    }

    // Set "bulk insert" mode flag to false.
    remote->bulk_mode = FALSE;

    // Allocate the LOGINREC structure used to make connections to the
    // remote server. Open the connection in the SRV_CONNECT handler.
    remote->login = dblogin();

    if (remote->login == NULL)
    {
        // Send a message to the client that the
        // remote connection failed.
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)REMOTE_FAIL,
            (DBTINYINT)14,
            (DBTINYINT)1,
            NULL,
            0,
            0,
            "Login Failed - SQLSNIFF : dblogin() failed. Disconnecting.",
            SRV_NULLTERM);

        // Deallocate the remote structure and set the user data
        // pointer in "srvproc" to null so that the disconnect
        // handler won't try to disconnect from the remote dbms.
        srv_setuserdata(srvproc, (BYTE *)NULL);
        srv_free(remote);

        // Now allow other threads to enter this function.
        ReleaseSemaphore(InitRemote_SEM, 1, NULL);
        return(SRV_DISCONNECT);
    }
    remote->dbproc = (DBPROCESS *)NULL;

    // If no server name was specified as a parameter to the main program,
    // then assume that the name is comming from the client.
    strncpy(TempServer,(const char *)remote_server,MAXSVR);
    strcat(TempServer,"\0");
    if ( !(*TempServer) )
    {
        strncpy(TempServer,srv_pfield(srvproc, SRV_HOST, (int *)NULL),MAXSVR );
        strcat(TempServer,"\0");
    }

    // Set the user name, password, and application name for the remote DBMS.
    // We'll parse the userid for a @ and if it has one we'll
    // use a different server <userid>@<server name> . 
    strncpy(TempUser,srv_pfield(srvproc, SRV_USER, (int *)NULL),MAXUSR+MAXSVR);
    strcat(TempUser,"\0");
    charpointer = TempUser;
    while ( *charpointer && *charpointer != '@')
        charpointer++;
    if (*charpointer == '@')
        {
        *charpointer = '\0';
        strncpy(TempServer,++charpointer,MAXSVR);
        strcat(TempServer,"\0");
        }

    DBSETLUSER(remote->login, TempUser );
    DBSETLPWD(remote->login, srv_pfield(srvproc, SRV_PWD, (int *)NULL));
    DBSETLAPP(remote->login, srv_pfield(srvproc, SRV_APPLNAME, (int *)NULL));
    DBSETLNATLANG(remote->login, srv_pfield(srvproc, SRV_NATLANG, (int *)NULL));

    // See if client has set Bulk Copy flag
    if (strcmp(srv_pfield(srvproc, SRV_BCPFLAG, (int *)NULL), "TRUE") == 0)
        BCP_SETL(remote->login, TRUE);
    else
        BCP_SETL(remote->login, FALSE);

    // Try to open a connection to the remote DBMS.
    if ((remote->dbproc = dbopen(remote->login, TempServer))
        == (DBPROCESS *)NULL)
    {
        // Send a message to the client that
        // the remote connection failed.
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)REMOTE_FAIL,
            (DBTINYINT)14,
            (DBTINYINT)1,
            NULL,
            0,
            0,
            "Login Failed - SQLSNIFF : dbopen to remote server failed.",
            SRV_NULLTERM);

        // Deallocate the remote structure and set the user data
        // pointer in "srvproc" to null so that the disconnect
        // handler won't try to disconnect from the remote DBMS.
        dbfreelogin(remote->login);
        srv_free(remote);
        srv_setuserdata(srvproc, (BYTE *)NULL);

        // Now allow other threads to enter this function.
        ReleaseSemaphore(InitRemote_SEM, 1, NULL);
        return(SRV_DISCONNECT);
    } else {
        // dbopen succesful
        // Create MDI windows for logging

        // Now allow other threads to enter this function.
        // We've finished with Newsrvproc
        ReleaseSemaphore(InitRemote_SEM, 1, NULL);

        if (Log.MDI_Queries)
        {
            hQueryInfo = LocalAlloc(LHND, (WORD) sizeof(CLIENTINFO));
            if (!hQueryInfo)
            {
                // Couldn't allocate the memory for the info
                ERRBOX("Failed to Allocate CLient Info structure");
                dbclose(remote->dbproc);
                dbfreelogin(remote->login);
                srv_free(remote);
                srv_setuserdata(srvproc, (BYTE *)NULL);
                return(SRV_DISCONNECT);
            } else {
                if ((pInfo = (PCLIENTINFO)LocalLock(hQueryInfo))==NULL)
                {
                     ERRBOX("Failed to Lock hQueryInfo! Can't initialise new session.");
                     // Clean up
                     dbclose(remote->dbproc);
                     dbfreelogin(remote->login);
                     srv_free(remote);
                     srv_setuserdata(srvproc, (BYTE *)NULL);
                     LocalFree(hQueryInfo);
                     return(SRV_DISCONNECT);
                 } else {
                     pInfo->iIconType = IDM_INACTIVE_ICON;
                     // Build window title
                     sprintf(szTitle, "Q Only - %s - %s - %s", 
                         srv_pfield(srvproc, SRV_USER    ,(int *)NULL),
                         srv_pfield(srvproc, SRV_APPLNAME,(int *)NULL),
                         srv_pfield(srvproc, SRV_SPID    ,(int *)NULL));

                     LocalUnlock(hQueryInfo);
                     if ( (hwndQuery=CreateMDI(
                         hQueryInfo, 
                         szTitle,
                         srv_pfield(srvproc, SRV_APPLNAME, (int *)NULL))
                         ))
                     {
                         remote->hQueryInfo = hQueryInfo;
                         // display some info in the window
                         sprintf(szBuf, "Client ID: %s", srv_pfield(srvproc, SRV_SPID, (int *)NULL));
                         AddRow(hQueryInfo, szBuf,colorBlue);
                         sprintf(szBuf, "User name: %s", srv_pfield(srvproc, SRV_USER, (int *)NULL));
                         AddRow(hQueryInfo, szBuf, colorBlue);
                     } else {
                          ERRBOX("Failed to Create window! Can't initialise new session.");
                          // Clean up
                          dbclose(remote->dbproc);
                          dbfreelogin(remote->login);
                          srv_free(remote);
                          srv_setuserdata(srvproc, (BYTE *)NULL);
                          LocalFree(hQueryInfo);
                          return(SRV_DISCONNECT);
                     }
                }
            }
        }


        if (Log.MDI_All)
        {
            hAllInfo = LocalAlloc(LHND, (WORD) sizeof(CLIENTINFO));
            if (!hAllInfo)
            {
                // Couldn't allocate the memory for the info
                ERRBOX("Failed to Allocate CLient Info structure");
                if (Log.MDI_Queries)
                {  
                   PostMessage(hwndQuery,WM_CLOSE,0,0L);
                   LocalFree(hQueryInfo);
                }
                dbclose(remote->dbproc);
                dbfreelogin(remote->login);
                srv_free(remote);
                srv_setuserdata(srvproc, (BYTE *)NULL);
                return(SRV_DISCONNECT);
            } else {
                if ((pInfo = (PCLIENTINFO)LocalLock(hAllInfo))==NULL)
                {
                     ERRBOX("Failed to Lock hAllInfo! Can't initialise new session.");
                     // Clean up
                     dbclose(remote->dbproc);
                     dbfreelogin(remote->login);
                     if (Log.MDI_Queries) 
                     {  
                        PostMessage(hwndQuery,WM_CLOSE,0,0L);
                        LocalFree(hQueryInfo);
                     }
                     srv_free(remote);
                     srv_setuserdata(srvproc, (BYTE *)NULL);
                     LocalFree(hAllInfo);
                     return(SRV_DISCONNECT);
                 } else {
                     pInfo->iIconType = IDM_INACTIVE_ICON;
                     // Build window title
                     sprintf(szTitle, "%s - %s - %s", 
                         srv_pfield(srvproc, SRV_USER    ,(int *)NULL),
                         srv_pfield(srvproc, SRV_APPLNAME,(int *)NULL),
                         srv_pfield(srvproc, SRV_SPID    ,(int *)NULL));

                     LocalUnlock(hAllInfo);
                     if ( (hwndAll=CreateMDI(
                         hAllInfo, 
                         szTitle,
                         srv_pfield(srvproc, SRV_APPLNAME, (int *)NULL))
                         ))
                     {
                         remote->hAllInfo = hAllInfo;
                         // display some info in the window
                         sprintf(szBuf, "Client ID: %s", srv_pfield(srvproc, SRV_SPID, (int *)NULL));
                         AddRow(hAllInfo, szBuf,colorBlue);
                         sprintf(szBuf, "User name: %s", srv_pfield(srvproc, SRV_USER, (int *)NULL));
                         AddRow(hAllInfo, szBuf, colorBlue);
                     } else {
                          ERRBOX("Failed to Create window! Can't initialise new session.");
                          // Clean up
                          dbclose(remote->dbproc);
                          dbfreelogin(remote->login);
                          if (Log.MDI_Queries)
                          {  
                              PostMessage(hwndQuery,WM_CLOSE,0,0L);
                              LocalFree(hQueryInfo);
                           }
                          srv_free(remote);
                          srv_setuserdata(srvproc, (BYTE *)NULL);
                          LocalFree(hAllInfo);
                          return(SRV_DISCONNECT);
                     }
                }
            }
        }

        if (Log.File_Queries || Log.File_All)
        {
            // Display connect info
            sprintf(log_buffer, "Connection: Client ID: %s, User name: %s, Application program name: %s",
                srv_pfield(srvproc, SRV_SPID, (int *)NULL),
                srv_pfield(srvproc, SRV_USER, (int *)NULL),
                srv_pfield(srvproc, SRV_APPLNAME, (int *)NULL));
            server = SRV_GETSERVER(srvproc);
            srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
        }
        // Connection to the remote DBMS successful.  Save
        // remote data structure in the "srvproc" so it will be
        // available to the other handlers. We'll also map the remote
        // DBMS connection to our "srvproc".
        srv_setuserdata(srvproc, (BYTE *)remote);
        dbsetuserdata(remote->dbproc, (SRV_PROC *)srvproc);
    }

    // Tile if necessary
    if (bAutoTile) PostMessage(ghwndMain,WM_COMMAND,IDM_TILE,0L);

    return(SRV_CONTINUE);
}
