//****************************** Function Header ******************************
//
// Function : HandleResults
//
// Description: Pass back results to client and log in MDI client
//      windows and/or log file
//
// Author PeterWil 1994, Microsoft Ltd.
//*****************************************************************************
#define DBNTWIN32

// standard compiler/sql server supplied header files
#include    <windows.h>
#include    <sqlfront.h>
#include    <sqldb.h>
#include    <srv.h>
#include    <stdio.h>
#include    <stdlib.h>

// Application specific header files
#include    "sqlsniff.h"
#include    "hresults.h"
#include    "rowstuff.h"
#include    "sniffcbs.h"

extern LOG Log;
int dbcolntype(DBPROCESS *, int);

#pragma check_stack( off )  // turn off stack checking

int HandleResults(DBPROCESS  *rmtproc, SRV_PROC  *srvproc)
{
    int         cols;
    int         iColumnCounter, iDataCounter, iOldIconType;
    DBINT       rows;
    BOOL        results_sent;
    BOOL        gotcompute;
    int         (*paramarray)[MAXPARAMS];
    RETCODE      returnvalue;
    REMOTE_DBMS *remote;
    SRV_SERVER  *server;
    char        log_buffer[LOG_BUF_SIZE];

    results_sent = FALSE;
    rows = 0L;
    gotcompute = FALSE;
    paramarray = (int (*)[MAXPARAMS])&(((REMOTE_DBMS *)srv_getuserdata(srvproc))->retparams);
    remote  = (REMOTE_DBMS *)srv_getuserdata(srvproc);
                                                         
    // store the old icon message, then use the out icon
    if (Log.MDI_All)      iOldIconType = ChangeIcon(remote->hAllInfo,   IDM_OUT_ICON);
    if (Log.MDI_Queries)  iOldIconType = ChangeIcon(remote->hQueryInfo, IDM_OUT_ICON);

    // Process the results from the remote DBMS.
    // Since a command may consist of multiple commands or a single
    // command that has multiple sets of results, we'll loop through
    // each results set.
    while (TRUE)
    {
        returnvalue = dbresults(rmtproc);
        if (returnvalue == NO_MORE_RESULTS)
            break;

        // Check to see if the client has sent an attention event.  If
        // so, simply discard data from remote server
        if (SRV_GOT_ATTENTION(srvproc)){
            dbcancel(rmtproc);
            continue;
        }

        // If this is the second time through the loop,
        // send a completion message to the client
        // for the previous results sent.
        if (results_sent == TRUE)
        {
            // If there are some COMPUTE rows, send a message
            // to the client that Data Services Library doesn't yet handle them.
            if (gotcompute == TRUE)
            {
                gotcompute = FALSE;
            srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)COMPUTE_ROW,
            (DBTINYINT)0,
            (DBTINYINT)0,
            NULL,
            0,
            0,
            "Data Services library can't handle COMPUTE rows.",
            SRV_NULLTERM);
        }
        // If the previous batch was one that may
        // have returned rows, set the DONE status
        // accordingly.
        if (rows > 0)
        {
            srv_senddone(srvproc, SRV_DONE_MORE | SRV_DONE_COUNT,
                (DBUSMALLINT)0, rows);
        } else
            srv_senddone(srvproc, SRV_DONE_MORE, (DBUSMALLINT)0,
            (DBINT)0);
        }

        // How many data columns are in the row?
        // Non-"select" statements will have 0 columns.
        cols = dbnumcols(rmtproc);

        // Build the row description for the client return.
        for (iColumnCounter = 1; iColumnCounter <= cols; iColumnCounter++)
        {
            // Call "srv_describe()" for each column in the row.
            srv_describe(srvproc,
                iColumnCounter,
                dbcolname(rmtproc, iColumnCounter),
                SRV_NULLTERM,
                (DBINT)dbcolntype(rmtproc, iColumnCounter),
                dbcollen(rmtproc, iColumnCounter),
                (DBINT)dbcolntype(rmtproc, iColumnCounter),
                dbcollen(rmtproc, iColumnCounter),
                (BYTE *)NULL);
        }

        if (Log.MDI_All)
        {
            AddRow(remote->hAllInfo, " ", colorBlue);
            AddRow(remote->hAllInfo, " Result Set: ", colorBlue);
            AddHeader(remote->hAllInfo, rmtproc, colorRed);
        }

        if (Log.File_All)
        {
            server = SRV_GETSERVER(srvproc);
            {
                char *pszHeader;
                ConstructHeader(rmtproc, &pszHeader);
                sprintf(log_buffer,
                    "==> Res  : Client Id: %s - %s",
                    srv_pfield(srvproc, SRV_SPID, (int *)NULL),
                    pszHeader);
                srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
                free(pszHeader);
            }
        }

        // Send each row from the remote DBMS to the client.
        rows = 0;
        while (TRUE)
        {
            returnvalue = dbnextrow(rmtproc);
            if (returnvalue == NO_MORE_ROWS)
                break;

            // Check to see if the client has issued an attention event.
            // If so, discard data from the remote server.
            if (SRV_GOT_ATTENTION(srvproc)){
                dbcancel(rmtproc);
                continue;
            }

            // If it's not a regular row, it's a COMPUTE row.
            // This SQL extension is particular to Sybase
            // TRANSACT-SQL and is not yet supported.
            if (DBROWTYPE(rmtproc) != REG_ROW)
            {
                gotcompute = TRUE;
                continue;
            }  else
                gotcompute = FALSE;

            // The row description is built.  Move the
            // rows from the remote server to the client.

            for (iColumnCounter = 1; iColumnCounter <= cols; iColumnCounter++)
            {
                srv_setcollen(srvproc, iColumnCounter, dbdatlen(rmtproc, iColumnCounter));
                srv_setcoldata(srvproc, iColumnCounter, dbdata(rmtproc, iColumnCounter));
            }

            if (Log.MDI_All)
            {
                AddResultRow(remote->hAllInfo, rmtproc, colorBlack);
            } 

            if (Log.File_All)
            {
                server = SRV_GETSERVER(srvproc);
                {
                    char *pszRow;
                    ConstructRow(rmtproc, &pszRow);
                    sprintf(
                        log_buffer,
                        "Results  : Client Id: %s - %s",
                        srv_pfield(srvproc, SRV_SPID, (int *)NULL),
                        pszRow );
                    srv_log(server, TRUE, log_buffer, SRV_NULLTERM);
                    free(pszRow);
                }
            }

            if (srv_sendrow(srvproc) == SUCCEED)
                rows++;
        }

        // Check to see if any parameter were returned from
        // the remote DBMS.  If so, pass them through to the
        // client.
        for (iDataCounter = 1; iDataCounter <= dbnumrets(rmtproc); iDataCounter++)
        {
            // If the return parameters are a result of
            // an rpc, we used srv_paramset() to set the return
            // value.  If the return parameters are not the
            // result of an rpc, we use srv_returnval().
            if (srv_rpcname(srvproc, NULL) != NULL) {
                //
                // The current rpc may have three parameters, but
                // only the first and third are return parameters.
                // This means that dbnumrets() returns two parameters,
                // not three.  The first call to dbretdata() refers to
                // the rpc's first parameter, and the second call to
                // dbretdata() refers to the rpc's third parameter.
                // To handle this, we map each return parameter to
                // its original parameter so we can later reset the
                // return value of the correct return parameters in
                // HandleResults().
                srv_paramset(srvproc, (*paramarray)[iDataCounter], dbretdata(rmtproc, iDataCounter),
                dbretlen(rmtproc, iDataCounter));
            } else {
                srv_returnval(srvproc,
                    dbretname(rmtproc, iDataCounter),
                    SRV_NULLTERM,
                    SRV_PARAMRETURN,
                    dbrettype(rmtproc, iDataCounter),
                    dbretlen(rmtproc, iDataCounter),
                    dbretlen(rmtproc, iDataCounter),
                    dbretdata(rmtproc, iDataCounter));
            }
        }

        // Check to see if we got a return status code from the
        // remote DBMS.  Pass it through to the client.
        if (dbhasretstat(rmtproc))
            srv_sendstatus(srvproc, dbretstatus(rmtproc));

        // If the command was one where count is meaningful, DBCOUNT()
        // will be >= 0 and we will set rows accordingly.
        //
        if (dbnumcols(rmtproc) > 0)
            rows = DBCOUNT(rmtproc) < 0L ? 0L : rows;
        else
            rows = DBCOUNT(rmtproc);

        // Set flag so that we will send a completion
        // message for the current set of results.
        results_sent = TRUE;
    }

    // If there are some COMPUTE rows, send a message
    // to the client that Open Services Library doesn't handle them yet.
    if (gotcompute == TRUE)
    {
        gotcompute = FALSE;
        srv_sendmsg(srvproc,
            SRV_MSG_ERROR,
            (DBINT)COMPUTE_ROW,
            (DBTINYINT)0,
            (DBTINYINT)0,
            NULL,
            0,
            0,
            "Data Services Library can't yet handle COMPUTE rows.",
            SRV_NULLTERM);
    }

    // Send the final done packet for the execution of the command batch.
    // If the previous batch was one that may
    // have returned rows, set the DONE status
    // accordingly.
    if (rows > 0)
    {
        srv_senddone(srvproc,
            SRV_DONE_COUNT | SRV_DONE_FINAL,
            (DBUSMALLINT)0,
            rows);
    } else {
        srv_senddone(srvproc,
            SRV_DONE_FINAL,
            (DBUSMALLINT)0,
            (DBINT)0);
    }
    // reset the icon
    if (Log.MDI_All)     ChangeIcon(remote->hAllInfo,   iOldIconType);
    if (Log.MDI_Queries) ChangeIcon(remote->hQueryInfo, iOldIconType);

    return(SUCCEED);
}

#pragma check_stack()   // set stack checking to its default setting
