/****************************************************************************
 *
 *  PROGRAM:    WinQuery.c
 *
 *  PURPOSE:    Windows DB-Library Query Application
 *
 *  FUNCTIONS:
 *
 *      WinMain() - calls initialization function, processes message loop
 *      InitApplication() - initializes window data and registers window
 *      InitInstance() - saves instance handle and creates main window
 *      DoBackgroundPart() - do background processing
 *      MainWndProc() - processes messages
 *      InitDblib() - initializes DB-Library
 *      ExitDblib() - exits DB-Library
 *      SendQuery() - sends the query to the server
 *      CancelQuery() -  cancels the current query
 *      ProcessQuery() - processes a small part of the current query
 *      BuildRow() - builds a result row
 *      BuildHeader() - builds the result header
 *      DetermineRowSize() - returns the size of an entire row
 *      DetermineColSize() - returns the size of a column
 *      Connect() - processes messages for "Connect" dialog box
 *      About() - processes messages for "About" dialog box
 *      EditLineCat() - adds a line of text to the edit control
 *      EditStrCat() - appends a string to the end of an edit control
 *      MemFree() - free memory
 *      MemAlloc() - allocate new memory
 *      MemAdd() - reallocate and add more memory
 *      PadCol() - pad with spaces
 *      ErrorHandler() - processes DB-Library errors
 *      MessageHandler() - processes SQL Server messages
 *
 *  COMMENTS:
 *
 *      James McDaniel
 *      SQL Server Support Team
 *
 ****************************************************************************
 */

#include <windows.h>                /* standard Windows include file */
#include <string.h>                 /* standard C string include file */
#define DBMSWIN                     /* needed to define environment */
#include <sqlfront.h>               /* standard dblib include file */
#include <sqldb.h>                  /* standard dblib include file */
#include "winquery.h"               /* specific to this program */
#include "windlg.h"                 /* dialogs include file */

/* GLOBAL VARIABLES */
DBPROCESS *pDbproc = (DBPROCESS *)NULL; /* connection to SQL Server */
char szAppName[] = "WinQuery";          /* application name */
char szDblibVersion[50];                /* DB-Library version string */
FARPROC lpMessageHandler,               /* pointer to MessageHandler */
    lpErrorHandler;                     /* pointer to ErrorHandler */
HANDLE hInst;                           /* current instance */
HANDLE hWnd;                            /* current window */
HANDLE hQuery,                          /* handle to query edit control */
    hResult;                            /* handle to result edit control */
BOOL bQueryPending = FALSE;             /* query pending */
BOOL bCancelQuery = FALSE;              /* cancel query */
int iProcessQueryAction;                /* next query action */


/****************************************************************************
 *
 *  FUNCTION:   WinMain()
 *
 *  PURPOSE:    Calls initialization function, processes message loop
 *
 *  COMMENTS:
 *
 ****************************************************************************
 */

int PASCAL WinMain(
    HANDLE hInstance,
    HANDLE hPrevInstance,
    LPSTR lpCmdLine,
    int nCmdShow)
{
    MSG msg;            /* message */
    HANDLE hAccel;      /* handle to accelerator table */

    if (!hPrevInstance)
        /* Initialize shared things */
        if (!InitApplication(hInstance))
            return (FALSE);

    /* Perform initializations that apply to a specific instance */
    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    hAccel = LoadAccelerators (hInstance, "WinQueryAccelerators");

    while (TRUE)
    {
        if (PeekMessage(&msg,
            NULL,
            0,
            0,
            PM_REMOVE))
        {
            /* Message in queue, process it */
            if (msg.message != WM_QUIT)
            {
                if (!TranslateAccelerator (hWnd, hAccel, &msg))
                {
                    TranslateMessage(&msg); /* Translates virtual key codes */
                    DispatchMessage(&msg);  /* Dispatches message to window */
                }
            }
            else /* WM_QUIT message */
            {
                break; /* while loop */
            }
        }
        else /* no message in queue */
        {
            /* Do background processing if necessary */
            if (!DoBackgroundPart())
            {
                WaitMessage();
            }
        }
    }

    return (msg.wParam);        /* Returns the value from PostQuitMessage */
}

/****************************************************************************
 *
 *  FUNCTION:   InitApplication()
 *
 *  PURPOSE:    Initializes window data and registers window class
 *
 *  COMMENTS:
 *
 ****************************************************************************
 */

BOOL InitApplication(
    HANDLE hInstance)
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the
     * main window.
     */
    wc.style = NULL;                    /* Class style(s) */
    /* Function to retrieve messages for windows of this class */
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;                  /* No per-class extra data */
    wc.cbWndExtra = 0;                  /* No per-window extra data */
    wc.hInstance = hInstance;           /* Application that owns the class */
    wc.hIcon = LoadIcon(hInstance, "WinQueryIcon"); /* icon */
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       /* arrow cursor */
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); /* white background */
    wc.lpszMenuName =  "WinQueryMenu";  /* Name of menu resource in .RC file. */
    wc.lpszClassName = szAppName;       /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */
    return (RegisterClass(&wc));
}

/****************************************************************************
 *
 *  FUNCTION:   InitInstance()
 *
 *  PURPOSE:    Saves instance handle and creates main window
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL InitInstance(
    HANDLE hInstance,
    int nCmdShow)
{
    DWORD dwEditStyle;      /* style bits for edit controls */
    HFONT hFixedFont;       /* handle to stock fixed font */

    /* Save the instance handle in static variable, which will be used in
     * many subsequent calls from this application to Windows.
     */
    hInst = hInstance;

    /* Create a main window for this application instance. */
    hWnd = CreateWindow(szAppName,  /* Class to use */
        "WinQuery",                 /* Text for window title bar */
        WS_OVERLAPPEDWINDOW,        /* Window style */
        CW_USEDEFAULT,              /* Default horizontal position */
        CW_USEDEFAULT,              /* Default vertical position */
        CW_USEDEFAULT,              /* Default width */
        CW_USEDEFAULT,              /* Default height */
        NULL,                       /* Overlapped windows have no parent */
        NULL,                       /* Use the window class menu */
        hInstance,                  /* This instance owns this window */
        NULL);                      /* Pointer not needed */

    /* If window could not be created, return "failure" */
    if (!hWnd)
        return (FALSE);

    /* Set style bit for edit controls */
    dwEditStyle = WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL |
        ES_LEFT | ES_MULTILINE | ES_AUTOHSCROLL | ES_AUTOVSCROLL;

    /* Create query edit window */
    hQuery = CreateWindow("edit",   /* edit control class */
        NULL,                       /* text of control */
        dwEditStyle,                /* style params */
        0, 0, 0, 0,                 /* initial dimensions all 0 */
        hWnd,                       /* parent handle */
        IDE_QUERY,                  /* unique child-window identifier */
        hInst,                      /* instance handle */
        NULL);

    /* Create result edit window */
    hResult = CreateWindow("edit",  /* edit control class */
        NULL,                       /* text of control */
        dwEditStyle,                /* style params */
        0, 0, 0, 0,                 /* initial dimensions all 0 */
        hWnd,                       /* parent handle */
        IDE_RESULT,                 /* unique child-window identifier */
        hInst,                      /* instance handle */
        NULL);

    /* Get stock fixed font, set both edit controls to this fixed font */
    hFixedFont = GetStockObject (ANSI_FIXED_FONT);
    SendMessage(hQuery,
        WM_SETFONT,
        hFixedFont,
        FALSE);
    SendMessage(hResult,
        WM_SETFONT,
        hFixedFont,
        FALSE);

    /* Initialize edit controls window text */
    SetWindowText (hQuery, "");
    SetWindowText (hResult, "");

    SetFocus (hQuery);

    ShowWindow(hWnd, nCmdShow);     /* Show the window */
    UpdateWindow(hWnd);             /* Sends WM_PAINT message */
    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   MainWndProc()
 *
 *  PURPOSE:    Processes messages
 *
 *  MESSAGES:
 *
 *      WM_COMMAND - application menu
 *      WM_CREATE - create window
 *      WM_DESTROY - destroy window
 *      WM_SETFOCUS - window got focus
 *      WM_SIZE - size window
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

LONG FAR PASCAL MainWndProc(
    HWND hWindow,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    FARPROC lpDlgProc;              /* pointer to dialog function */
    int nWidth, nHeight;            /* width and height of window */
    static HANDLE hCurrentFocus;    /* handle of edit control with focus */

    switch (message)
    {
    case WM_COMMAND:    /* message: command from application menu */
        switch (wParam)
        {
        case IDM_CUT:
            /* Cut selected text from edit control */
            SendMessage (hCurrentFocus, WM_CUT, 0, 0L);
            break;

        case IDM_COPY:
            /* Copy selected text from edit control */
            SendMessage (hCurrentFocus, WM_COPY, 0, 0L);
            break;

        case IDM_PASTE:
            /* Paste text from clipboard into edit control */
            SendMessage (hCurrentFocus, WM_PASTE, 0, 0L);
            break;

        case IDM_DELETE:
            /* Delete text from edit control */
            SendMessage (hCurrentFocus, WM_CLEAR, 0, 0L);
            break;

        case IDM_CONNECT:
            lpDlgProc = MakeProcInstance(Connect, hInst);

            DialogBox(hInst,        /* current instance */
                "ConnectBox",       /* resource to use */
                hWindow,            /* parent handle */
                lpDlgProc);         /* Connect() instance address */

            FreeProcInstance(lpDlgProc);
            break;

        case IDM_EXECUTE:
            if (pDbproc != (DBPROCESS *)NULL)
            {
                if (!bQueryPending)
                {
                    SendQuery(pDbproc);
                }
                else /* query pending */
                {
                    ERRBOX (hWnd, "Query still pending.");
                }
            }
            else /* null dbproc */
            {
                ERRBOX(hWnd, "No connection available");
            }
            break;

        case IDE_RESULT:
            switch (HIWORD(lParam))
            {
            case EN_ERRSPACE:
                MSGBOX (hWindow, "Result edit control out of space");
                CancelQuery(pDbproc);
                break;
            case EN_SETFOCUS:
                hCurrentFocus = hResult;
                break;
            }
            break;

        default:
            /* Pass it on if unproccessed */
            return (DefWindowProc(hWindow, message, wParam, lParam));

        }
        break; /* case WM_COMMAND */

    case WM_CREATE:     /* message: window being created */
        hCurrentFocus = hQuery;

        if (!InitDblib())
        {
            ERRBOX(hWindow, "Failed to initialize DB-Library");
        }
        break; /* case WM_CREATE */

    case WM_DESTROY:    /* message: window being destroyed */
        ExitDblib();
        PostQuitMessage(0);
        break; /* case WM_DESTORY */

    case WM_SETFOCUS:   /* message: window now has focus */
        SetFocus (hCurrentFocus);
        break; /* case WM_SETFOCUS */

    case WM_SIZE:       /* message: window resized */
        /* Resize query and result windows to take up half of window */
        nWidth = LOWORD(lParam);
        nHeight = HIWORD(lParam);
        MoveWindow(hQuery,      /* window to move */
            0,                  /* x coord */
            0,                  /* y coord */
            nWidth,             /* width */
            nHeight/2,          /* height */
            TRUE);              /* repaint */

        MoveWindow(hResult,     /* window to move */
            0,                  /* x coord */
            nHeight/2,          /* y coord */
            nWidth,             /* width */
            nHeight/2,          /* height */
            TRUE);              /* repaint */

        break;

    default:
        /* Pass it on if unproccessed */
        return (DefWindowProc(hWindow, message, wParam, lParam));
    }
    return (NULL);
}

/****************************************************************************
 *
 *  FUNCTION:   DoBackgroundPart()
 *
 *  PURPOSE:    Do background processing
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL DoBackgroundPart (void)
{
    BOOL bContinue;     /* continue background processing */

    if (bQueryPending)
    {
        /* Process portion of query */
        bContinue = ProcessQuery(pDbproc);
    }
    else
    {
        bContinue = FALSE;
    }
    return (bContinue);
}

/****************************************************************************
 *
 *  FUNCTION:   InitDblib()
 *
 *  PURPOSE:    Initialize DB-Library
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL InitDblib (void)
{
    DBLOCKLIB();
    /* Initialize DB-Library */
    wsprintf(szDblibVersion, "%s", dbinit());
    if (szDblibVersion == "")
        return (FALSE);

    /* Get procedure instances */
    lpErrorHandler = MakeProcInstance((FARPROC)ErrorHandler, hInst);
    lpMessageHandler = MakeProcInstance((FARPROC)MessageHandler, hInst);
    /* Register handlers with DB-Library */
    dberrhandle(lpErrorHandler);
    dbmsghandle(lpMessageHandler);
    /* Set DB-Library options */
    dbsetlogintime(10);
    dbsettime(100);
    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   ExitDblib()
 *
 *  PURPOSE:    Exit from DB-Library
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL ExitDblib (void)
{
    /* Close connection to SQL Server if open */
    if (pDbproc != (DBPROCESS *)NULL)
    {
        dbclose(pDbproc);
    }
    dbwinexit();
    DBUNLOCKLIB();
    /* Free procedure instances */
    FreeProcInstance(lpErrorHandler);
    FreeProcInstance(lpMessageHandler);
    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   SendQuery()
 *
 *  PURPOSE:    Sends the query to the server
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL SendQuery (
    DBPROCESS *pDbproc)
{
    HANDLE hQueryBuffer;        /* handle to query buffer */
    LPSTR lpQueryBuffer;        /* long pointer to query buffer */
    DWORD dwLines,              /* lines in edit control */
        dwBytesCopied,          /* bytes copied from edit control */
        dwRow,                  /* current line of edit control */
        dwCharIndex,            /* index to char in edit control */
        dwLineLen;              /* length of line in edit control */

    MemAlloc (&hQueryBuffer, &lpQueryBuffer);

    /* get number of lines in query edit control */
    dwLines = SendMessage(hQuery,
        EM_GETLINECOUNT,
        0,
        0);

    for (dwRow=0;dwRow < dwLines;dwRow++)
    {
        /* get index of first char in current line */
        dwCharIndex = SendMessage (hQuery,
            EM_LINEINDEX,
            (WORD)dwRow,
            0);

        /* get length of current line */
        dwLineLen = SendMessage (hQuery,
            EM_LINELENGTH,
            (WORD)dwCharIndex,
            0);

        /* reallocate memory for new line length */
        MemFree (&hQueryBuffer);
        MemAlloc (&hQueryBuffer, &lpQueryBuffer);
        MemAdd (&hQueryBuffer, &lpQueryBuffer, (int)dwLineLen);

        (WORD)*lpQueryBuffer = (WORD)dwLineLen;
        /* copy line from edit control to query buffer */
        dwBytesCopied = SendMessage(hQuery,
            EM_GETLINE,
            (WORD)dwRow,
            (LONG)(lpQueryBuffer));
        lpQueryBuffer[dwBytesCopied] = '\0';

        /* add space, then current line, to query batch */
        dbcmd(pDbproc, " ");
        dbcmd(pDbproc, lpQueryBuffer);
    }

    MemFree (&hQueryBuffer);

    switch (dbsqlsend(pDbproc))
    {
    case SUCCEED:
        bQueryPending = TRUE;
        iProcessQueryAction = DO_DBDATAREADY;
        SetWindowText (hResult, "");
        return (TRUE);
        break;
    case FAIL:
        ERRBOX(hWnd, "Query could not be sent");
        return (FALSE);
        break;
    }
}

/****************************************************************************
 *
 *  FUNCTION:   CancelQuery()
 *
 *  PURPOSE:    Cancels the current query
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL CancelQuery (
    DBPROCESS *pDbproc)
{
    switch (dbcancel(pDbproc))
    {
    case SUCCEED:
        MSGBOX(hWnd, "Query execution canceled");
        bQueryPending = FALSE;
        return (TRUE);
        break;
    case FAIL:
        ERRBOX(hWnd, "Query execution could NOT be canceled");
        return (FALSE);
        break;
    }
}

/****************************************************************************
 *
 *  FUNCTION:   ProcessQuery()
 *
 *  PURPOSE:    Processes a small part of the current query
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL ProcessQuery (
    DBPROCESS *pDbproc)
{
    BOOL bContinue = FALSE;     /* continue processing query*/
    STATUS s;                   /* status from dbnextrow */

    switch (iProcessQueryAction)
    {
    case DO_DBDATAREADY:
        /* Call dbdataready() to see if query has completed */
        if (dbdataready(pDbproc))
        {
            iProcessQueryAction = DO_DBSQLOK;
            bContinue = TRUE;
        }
        else
        {
            bContinue = TRUE;
        }
        break; /* DO_DBDATAREADY */

    case DO_DBSQLOK:
        switch (dbsqlok(pDbproc))
        {
        case SUCCEED:
            /* Query execution succeeded */
            iProcessQueryAction = DO_DBRESULTS;
            bContinue = TRUE;
            break;
        case FAIL:
            ERRBOX (hWnd, "Failed to execute query");
            bContinue = FALSE;
            break;
        }
        break; /* DO_DBDATAREADY */

    case DO_DBRESULTS:
        switch (dbresults(pDbproc))
        {
        case SUCCEED:
            BuildHeader(pDbproc);
            iProcessQueryAction = DO_DBNEXTROW;
            bContinue = TRUE;
            break;
        case NO_MORE_RESULTS:
            MSGBOX(hWnd, "Query execution completed");
            bContinue = FALSE;
            break;
        case FAIL:
            ERRBOX(hWnd, "dbresults() returned FAIL");
            bContinue = FALSE;
            break;
        }
        break; /* DO_DBRESULTS */

    case DO_DBNEXTROW:
        switch (s=dbnextrow(pDbproc))
        {
        case REG_ROW:
            BuildRow(pDbproc, s);
            bContinue = TRUE;
            break;

        case NO_MORE_ROWS:
            EditLineCat (hResult, "");
            iProcessQueryAction = DO_DBRESULTS;
            bContinue = TRUE;
            break;
        case BUF_FULL:
            /* clear buffer */
            bContinue = TRUE;
            break;
        case FAIL:
            ERRBOX(hWnd, "dbnextrow() returned FAIL");
            bContinue = FALSE;
            break;
        default:
            BuildRow(pDbproc, s);
            bContinue = TRUE;
            break;
        }
        break; /* DO_DBNEXTROW */
    }

    bQueryPending = bContinue;

    return (bContinue);
}

/****************************************************************************
 *
 *  FUNCTION:   BuildRow()
 *
 *  PURPOSE:    Builds a result row
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL BuildRow(
    DBPROCESS *pDbproc,
    STATUS s)
{
    HANDLE hRowBuffer;      /* handle to row buffer */
    LPSTR lpRowBuffer,      /* long pointer to row buffer */
        lpPtr;              /* scratch pointer */
    int iNumCols,           /* number of columns in row */
        iColType,           /* column type */
        iCol,               /* current column */
        iColSize,           /* column size */
        iRowSize,           /* size of entire row */
        iConvertLen,        /* length of column data converted to char */
        iComputeID,         /* compute ID for compute row */
        iExtraSpaces;       /* extra spaces needed for column */
    DBINT lColDataLen,      /* length of column data */
        lColMaxLen;         /* max length of column */
    BYTE far *pColData;     /* pointer to column data */

    MemAlloc (&hRowBuffer, &lpRowBuffer);

    /* determine number of regular or compute columns */
    if (s == REG_ROW)
    {
        iNumCols = dbnumcols(pDbproc);
    }
    else
    {
        iComputeID = s;
        iNumCols = dbnumalts(pDbproc, iComputeID);
    }

    lpPtr = lpRowBuffer;

    /* allocate memory for entire result row */
    iRowSize = DetermineRowSize (pDbproc);
    MemAdd (&hRowBuffer, &lpRowBuffer, iRowSize);

    for (iCol=1;iCol<=iNumCols;iCol++)
    {
        if (s == REG_ROW)
        {
            /* Get column type, data length, max data length and pointer
             * to the data for this regular row
             */
            iColType = dbcoltype(pDbproc, iCol);
            lColDataLen = dbdatlen(pDbproc, iCol);
            lColMaxLen = dbcollen(pDbproc, iCol);
            pColData = dbdata(pDbproc, iCol);
        }
        else
        {
            /* Get column type, data length, max data length and pointer
             * to the data for this compute row
             */
            iColType = dbalttype(pDbproc, iComputeID, iCol);
            lColDataLen = dbadlen(pDbproc, iComputeID, iCol);
            lColMaxLen = dbaltlen(pDbproc, iComputeID, iCol);
            pColData = dbadata(pDbproc, iComputeID, iCol);
        }

        iColSize = DetermineColSize (pDbproc, iCol);

        if (lColDataLen)
        {
            /* this column is not NULL, so convert to character */
            lpPtr = lpRowBuffer + lstrlen (lpRowBuffer);

            iConvertLen = dbconvert (pDbproc,
                iColType,
                pColData,
                lColDataLen,
                SQLCHAR,
                lpPtr,
                -1);

            PadCol (lpRowBuffer, ' ', iColSize-iConvertLen);
        }
        else /* NULL column */
        {
            lstrcat (lpRowBuffer, "NULL");
            PadCol (lpRowBuffer, ' ', iColSize-4);
        }
    }

    EditLineCat (hResult, lpRowBuffer);

    MemFree (&hRowBuffer);

    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   BuildHeader()
 *
 *  PURPOSE:    This function builds the string that contains the names
 *              of each column. It does this by finding the print size of
 *              each column, allocating a buffer to hold all column names
 *              plus one space between each column name, then copying that
 *              name into the appropriate location into the buffer.
 *
 *              It also builds an appropriate underline string for the
 *              column names.
 *
 *  COMMENTS:
 *
 ****************************************************************************
 */

BOOL BuildHeader(
    DBPROCESS *pDbproc)
{
    HANDLE hHeaderBuffer,       /* handle to header buffer */
        hUnderlineBuffer;       /* handle to underline buffer */
    LPSTR lpHeaderBuffer,       /* long pointer to header buffer */
        lpUnderlineBuffer,      /* long pointer to underline buffer */
        lpColName,              /* long point to column name */
        lpPtr;                  /* scratch pointer */
    int iCol,                   /* current column */
        iNumCols,               /* number of columns in row */
        iHeaderSize,            /* size of header buffer */
        iColSize,               /* size of column */
        iExtraSpaces;           /* extra spaces needed for column */

    MemAlloc (&hHeaderBuffer, &lpHeaderBuffer);
    MemAlloc (&hUnderlineBuffer, &lpUnderlineBuffer);

    /* Get number of columns */
    iNumCols = dbnumcols(pDbproc);

    /* Get row size and allocate memory for buffer */
    iHeaderSize = DetermineRowSize(pDbproc);
    MemAdd (&hHeaderBuffer, &lpHeaderBuffer, iHeaderSize);
    MemAdd (&hUnderlineBuffer, &lpUnderlineBuffer, iHeaderSize);

    for (iCol=1;
        iCol <= iNumCols;
        iCol++)
    {
        iColSize = DetermineColSize (pDbproc, iCol);

        /* Append column name and extra spaces to row header */
        lpColName = dbcolname(pDbproc,iCol);
        lstrcat (lpHeaderBuffer, lpColName);
        PadCol (lpHeaderBuffer, ' ', iColSize-lstrlen(lpColName));

        /* Append column underline and extra space to underline header */
        PadCol (lpUnderlineBuffer, '-', iColSize);
    }

    EditLineCat (hResult, lpHeaderBuffer);
    EditLineCat (hResult, lpUnderlineBuffer);

    MemFree (&hHeaderBuffer);
    MemFree (&hUnderlineBuffer);

    return(TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   DetermineRowSize()
 *
 *  PURPOSE:    This function is used to determine the size of an entire
 *              row, with one space between each column.
 *
 *  RETURN:     The size of the entire row is returned.
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

int DetermineRowSize(
    DBPROCESS *pDbproc)
{
    int iLength = 0,        /* total length of column or row */
        iNumCols,           /* number of columns */
        iColLen,            /* lePv+PP9F: v
VvvVvvcVFV RPcv
+d(D1 teD*t D0*% =@ t= t= u FPvv^FFD0$<u
+FD0tD*uD0$<@t>GFFFPFPFPD1$<@t:VFt-L(PVF+FPvF+FPFF9F|F D0tD*u +FD0t&F+F;	~L(~ t+	+)ND0 tL(~ t+T+)NL(D0tD0tD*tFFvvFP^Ð%PX> u3۹  SQ%Y[VWPX+tS3ҋ t9O)w!O t
&A G SRZ[GB;sĖt/+ǃ |u	= w33ۣuRZB;sYu'> t;6t# V  nPJ%6	 _^ËQ%3VWPXwȋ>3 tU;
u} t]6L(t<}
 tȚwPPuuuQuMA >A  u&D _^ & & 3 ۏ vCu ~
 uF
wPًNV t t&A &3 urYB9GuuӋRQSN
[YZX«N
ƫFF;>r>3 2 vCtދNV t'  Ë9Gu	 w 3~3G:Gt&A &8A u&&D G[ZYSV3ۋ v9t	t9WuFt^Ë3 v9Gt;Ou;WuP:G
Xu^ vD1$<@uD6+E vVHu+96*u tD1tV vu	GD^_6+=9~uٍ vVHu+CFt= t"= t= t!= t = t ڋ\6  case SQLFLT8:
        iPrintLen = PRFLT8;
        break;
    case SQLDATETIME:
        iPrintLen = PRDATETIME;
        break;
    case SQLMONEY:
        iPrintLen = PRMONEY;
        break;
    /* With VARBINARY, BINARY, and IMAGE data, each byte converts
     * to two hex characters
     */
    case SQLVARBINARY :
    case SQLBINARY:
    case SQLIMAGE:
        iPrintLen = (int) (dbcollen(pDbproc,iCol) * 2);
        break;
    /* Other types are maximum of actual column length */
    default :
        iPrintLen = (int) dbcollen(pDbproc,iCol);
        break;
    }

    /* Get length of column name */
    iNameLen = lstrlen(dbcolname(pDbproc,iCol));

    /* Column length is maximum of column name length or maximum printable
     * column data length
     */
    if(iNameLen > iPrintLen)
    {
        iColLen = iNameLen;
    }
    else
    {
        iColLen = iPrintLen;
    }

    /* Column may contain NULL values */
    if (iColLen < 4)
    {
        iColLen = 4;
    }

    /* Return the column length */
    return (iColLen);
}

/****************************************************************************
 *
 *  FUNCTION:   Connect()
 *
 *  PURPOSE:    Processes messages for "Connect" dialog box
 *
 *  MESSAGES:
 *
 *      WM_INITDIALOG - initialize dialog box
 *      WM_COMMAND    - Input received
 *
 *  COMMENTS:   TRUE must be returned to Windows.
 *
 ***************************************************************************
 */

BOOL FAR PASCAL Connect(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    char szSQLServer[MAX_NAME+1],
        szSQLLogin[MAX_NAME+1],
        szSQLPassword[MAX_NAME+1];
    LOGINREC *pLoginRec;

    switch (message)
    {
    case WM_INITDIALOG:     /* message: initialize dialog box */
        SetDlgItemText(hDlg,
            SQL_LOGIN,
            "sa");
        return (TRUE);
        break;

    case WM_COMMAND:    /* message: received a command */
        switch (wParam)
        {
        case IDOK:
            /* get server name, login, and password */
            GetDlgItemText(hDlg,
                SQL_SERVER,
                szSQLServer,
                MAX_NAME);
            GetDlgItemText(hDlg,
                SQL_LOGIN,
                szSQLLogin,
                MAX_NAME);
            GetDlgItemText(hDlg,
                SQL_PASSWORD,
                szSQLPassword,
                MAX_NAME);

            if(pDbproc != (DBPROCESS *)NULL)
            {
                dbclose(pDbproc);
            }
            pLoginRec = dblogin();
            if (pLoginRec != (LOGINREC *)NULL)
            {
                DBSETLUSER(pLoginRec, szSQLLogin);
                DBSETLPWD(pLoginRec, szSQLPassword);
                DBSETLHOST(pLoginRec, szAppName);
                pDbproc = dbopen(pLoginRec, szSQLServer);
                if (pDbproc != (DBPROCESS *)NULL)
                {
                    MSGBOX(hDlg, "Connected to SQL Server");
                }
                else /* dbopen() failed */
                {
                    ERRBOX(hDlg, "Failed to connect to SQL Server");
                }
            }
            else /* dblogin() failed */
            {
                ERRBOX(hDlg, "Failed to allocate login record");
            }
            dbfreelogin(pLoginRec);

            EndDialog(hDlg, TRUE);
            return (TRUE);
            break;

        case IDCANCEL:
            EndDialog(hDlg, TRUE);
            return (TRUE);
            break;
        }
        break;
    }
    return (FALSE);     /* Didn't process a message */
}

/****************************************************************************
 *
 *  FUNCTION:   About()
 *
 *  PURPOSE:    Processes messages for "About" dialog box
 *
 *  MESSAGES:
 *
 *      WM_INITDIALOG - initialize dialog box
 *      WM_COMMAND    - Input received
 *
 *  COMMENTS:   TRUE must be returned to Windows.
 *
 ***************************************************************************
 */

BOOL FAR PASCAL About(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    switch (message)
    {
    case WM_INITDIALOG:     /* message: initialize dialog box */
        SetDlgItemText(hDlg,
            SQL_DBLIBVERSION,
            szDblibVersion);
        return (TRUE);

    case WM_COMMAND:    /* message: received a command */

        switch (wParam)
        {
        case IDOK:
        case IDCANCEL:
            EndDialog(hDlg, TRUE);
            return (TRUE);
            break;
        }
        break;
    }
    return (FALSE);     /* Didn't process a message */
}

/****************************************************************************
 *
 *  FUNCTION:   EditLineCat()
 *
 *  PURPOSE:    Adds a line of text to an edit control
 *
 *  RETURN:
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL EditLineCat (
    HANDLE hEdit,
    LPSTR lpLine)
{
    EditStrCat (hEdit, lpLine);
    EditStrCat (hEdit, "\r\n");
    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   EditStrCat()
 *
 *  PURPOSE:    Appends a string to the end of an edit control
 *
 *  RETURN:
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL EditStrCat (
    HANDLE hEdit,
    LPSTR lpLine)
{
    WORD wEndIndex;         /* char index of last place in edit control */
    DWORD dwLines,          /* number of lines in edit control */
        dwCharIndex,        /* index of char in edit control */
        dwLen;              /* length of last line */

    /* get number of lines in edit control */
    dwLines = SendMessage (hEdit,
        EM_GETLINECOUNT,
        0,
        0);

    /* get index of first char in last line */
    dwCharIndex = SendMessage (hEdit,
        EM_LINEINDEX,
        (WORD)(dwLines-1),
        0);

    /* get length of last line */
    dwLen = SendMessage (hEdit,
        EM_LINELENGTH,
        (WORD)dwCharIndex,
        0);

    wEndIndex = (WORD)(dwCharIndex + dwLen);

    /* set selection to index at end of edit control text */
    SendMessage (hEdit,
        EM_SETSEL,
        0,
        MAKELONG(wEndIndex,wEndIndex));

    /* replace above selection with string */
    SendMessage (hEdit,
        EM_REPLACESEL,
        0,
        (DWORD)lpLine);

    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   MemFree()
 *
 *  PURPOSE:    Free memory
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL MemFree(
    HANDLE *phBuffer)
{
    GlobalUnlock(*phBuffer);
    GlobalFree(*phBuffer);
    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   MemAlloc()
 *
 *  PURPOSE:    Allocate new memory
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL MemAlloc(
    HANDLE *phBuffer,
    LPSTR *plpBuffer)
{
    HANDLE hNewBuffer;      /* new handle to memory buffer */

    hNewBuffer = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, 1);
    if (hNewBuffer)
    {
        *phBuffer = hNewBuffer;
        *plpBuffer = GlobalLock (hNewBuffer);
        return (TRUE);
    }
    else /* not enough memory to realloc */
    {
        MSGBOX (hWnd, "Memory allocation failed");
        return (FALSE);
    }
}

/****************************************************************************
 *
 *  FUNCTION:   MemAdd()
 *
 *  PURPOSE:    Reallocate and add more memory
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL MemAdd(
    HANDLE *phBuffer,
    LPSTR *plpBuffer,
    int iAddLen)
{
    DWORD iBufferSize;      /* current size of buffer */
    HANDLE hNewBuffer;      /* new handle to memory buffer */

    iBufferSize = GlobalSize(*phBuffer);
    hNewBuffer = GlobalReAlloc (*phBuffer,
        iBufferSize + iAddLen,
        GMEM_MOVEABLE | GMEM_ZEROINIT);
    if (hNewBuffer)
    {
        if (hNewBuffer != *phBuffer)
        {
            /* if handle changed, need to relock */
            GlobalUnlock(*phBuffer);
            *phBuffer = hNewBuffer;
            *plpBuffer = GlobalLock (*phBuffer);
        }
        return (TRUE);
    }
    else /* not enough memory to realloc */
    {
        MSGBOX (hWnd, "Memory allocation failed");
        return (FALSE);
    }
}

/****************************************************************************
 *
 *  FUNCTION:   PadCol()
 *
 *  PURPOSE:    Pad string with extra spaces, then add one space
 *
 *  COMMENTS:
 *
 ***************************************************************************
 */

BOOL PadCol(
    LPSTR lpBuffer,
    char chPadChar,
    int iNumPadChars)
{
    LPSTR lpPtr;        /* scratch pointer */

    lpPtr = lpBuffer + lstrlen(lpBuffer);
    _fmemset (lpPtr, chPadChar, iNumPadChars);
    lpPtr += iNumPadChars;
    *lpPtr = ' ';
    lpPtr++;
    *lpPtr = '\0';

    return (TRUE);
}

/****************************************************************************
 *
 *  FUNCTION:   ErrorHandler()
 *
 *  PURPOSE:    Processes DB-Library errors.
 *
 *  RETURN:     Return continuation code.
 *
 *  COMMENTS:   Must be declared FAR cdecl.  Must return either INT_CANCEL,
 *              INT_CONTINUE, or INT_EXIT to DB-Library.
 *
 ***************************************************************************
 */

int FAR ErrorHandler(
    DBPROCESS *pDbproc,
    DBSMALLINT severity,
    DBSMALLINT errno,
    DBSMALLINT oserr,
    LPSTR dberrstr,
    LPSTR oserrstr)
{
    MessageBox(hWnd,
        dberrstr,
        "DB-Library Error",
        MB_OK | MB_ICONSTOP);

    if (oserr != DBNOERR)       /* operating system error     */
    {
        MessageBox(hWnd,
            oserrstr,
            "Operating System Error",
            MB_OK | MB_ICONSTOP);
    }

    /* this will cause the offending DB-Library function to return FAIL */
    return(INT_CANCEL);
}

/****************************************************************************
 *
 *  FUNCTION:   MessageHandler()
 *
 *  PURPOSE:    Processes SQL Server messages.
 *
 *  RETURN:     Return 0
 *
 *  COMMENTS:   Must be declared FAR cdecl.
 *
 ***************************************************************************
 */

int FAR MessageHandler(
    DBPROCESS *pDbproc,
    DBINT msgno,
    DBSMALLINT msgstate,
    DBSMALLINT severity,
    LPSTR msgtext)
{
    MessageBox(hWnd,
        msgtext,
        "SQL Server Message",
        MB_OK | MB_ICONINFORMATION);
    return(0);
}
