/****************************************************************************

    PROGRAM: WinPipe.c
	     Copyright (C) 1991 Microsoft Corp.

    PURPOSE: Named-Pipe test Windows application

    FUNCTIONS:

	WinMain() - calls initialization function, processes message loop
	TestInit() - initializes window data and registers window
	TestWndProc() - processes messages
	AboutTest() - processes messages for "About" dialog box
	ConnectPipe() - processes input of server name and connects to server

    COMMENTS:

	Windows can have several copies of your application running at the
	same time.  The variable hInst keeps track of which instance this
	application is so that processing will be to the correct window.

	You only need to initialize the application once.  After it is
	initialized, all other copies of the application will use the same
	window class, and do not need to be separately initialized.

   Stephen Dibene
   SQL Server Test Group

****************************************************************************/

#include "windows.h"		    /* required for all Windows applications*/

#define INCL_NETWKSTA
#define INCL_NETERRORS
#include <lan.h>
#include <netcons.h>
#include <nmpipe.h>

#include "winpipe.h"		    /* specific to this program		    */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <share.h>
#include <io.h>
#include <errno.h>

HANDLE hInst;	   		    /* current instance			    */
HWND ghWnd;			    /* global window handle for handlers    */
HWND errhWnd;			    /* global window handle for current error*/

char szMessageText[MAX_MESSAGETEXT + 1];             /* message text */
char szPipeName[MAX_PIPENAME + 1];                   /* pipe name */
int  iIterationCount = 1;                            /* iterations of test */

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

	This will initialize the window class if it is the first time this
	application is run.  It then creates the window, and processes the
	message loop until a PostQuitMessage is received.  It exits the
	application by returning the value passed by the PostQuitMessage.

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;			     /* current instance	     */
HANDLE hPrevInstance;			     /* previous instance	     */
LPSTR lpCmdLine;			     /* command line		     */
int nCmdShow;				     /* show-window type (open/icon) */
{
    HWND hWnd;				     /* window handle		     */
    MSG msg;				     /* message			     */


    if (!hPrevInstance)			/* Has application been initialized? */
	if (!TestInit(hInstance))
	    return (NULL);		/* Exits if unable to initialize     */

    hInst = hInstance;			/* Saves the current instance	     */

    hWnd = CreateWindow("Pipe Test",		  /* window class	     */
	"Named-Pipe Test Windows Application",    /* window name	     */
	WS_OVERLAPPEDWINDOW,			  /* window style	     */
	CW_USEDEFAULT,				  /* x position		     */
	CW_USEDEFAULT,				  /* y position		     */
	CW_USEDEFAULT,				  /* width		     */
	CW_USEDEFAULT,				  /* height		     */
	NULL,					  /* parent handle	     */
	NULL,					  /* menu or child ID	     */
	hInstance,				  /* instance		     */
	NULL);					  /* additional info	     */

    if (!hWnd)					  /* Was the window created? */
	return (NULL);

    ghWnd = hWnd;				  /* set global handle	     */
    errhWnd = hWnd;

    ShowWindow(hWnd, nCmdShow);			  /* Shows the window	     */
    UpdateWindow(hWnd);				  /* Sends WM_PAINT message  */

    while (GetMessage(&msg,	   /* message structure			     */
	    NULL,		   /* handle of window receiving the message */
	    NULL,		   /* lowest message to examine		     */
	    NULL))		   /* highest message to examine	     */
	{
	TranslateMessage(&msg);	   /* Translates virtual key codes	     */
	DispatchMessage(&msg);	   /* Dispatches message to window	     */
    }
    return (msg.wParam);	   /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: TestInit(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

	Sets up a structure to register the window class.  Structure includes
	such information as what function will process messages, what cursor
	and icon to use, etc.


****************************************************************************/

BOOL TestInit(hInstance)
HANDLE hInstance;			       /* current instance	     */
{
    HANDLE hMemory;			       /* handle to allocated memory */
    PWNDCLASS pWndClass;		       /* structure pointer	     */
    BOOL bSuccess;			       /* RegisterClass() result     */

    hMemory = LocalAlloc(LPTR, sizeof(WNDCLASS));
    pWndClass = (PWNDCLASS) LocalLock(hMemory);

    pWndClass->style = NULL; /*CS_HREDRAW | CS_VREDRAW; */
    pWndClass->lpfnWndProc = TestWndProc;
    pWndClass->hInstance = hInstance;
    pWndClass->hIcon = LoadIcon(NULL, IDI_APPLICATION);
    pWndClass->hCursor = LoadCursor(NULL, IDC_ARROW);
    pWndClass->hbrBackground = COLOR_WINDOW + 1;
    pWndClass->lpszMenuName = (LPSTR)"PipeTest";
    pWndClass->lpszClassName = (LPSTR)"Pipe Test";

    bSuccess = RegisterClass(pWndClass);


    LocalUnlock(hMemory);			    /* Unlocks the memory    */
    LocalFree(hMemory);				    /* Returns it to Windows */
    strcpy(szMessageText, "Testing Pipe");          /* Default message text */
    strcpy(szPipeName, "abc");                      /* Default pipe name */
    return (bSuccess);		 /* Returns result of registering the window */
}

/****************************************************************************

    FUNCTION: TestWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

    MESSAGES:

	WM_SYSCOMMAND - system menu (About dialog box)
	WM_CREATE     - create window
	WM_DESTROY    - destroy window
	WM_COMMAND    - application menus (Connect and Select dialog boxes

    COMMENTS:

	To process the ID_ABOUTTEST message, call MakeProcInstance() to get the
	current instance address of the AboutTest() function.  Then call Dialog
	box which will create the box according to the information in your
	SqlTest.rc file and turn control over to the AboutTest() function.	When
	it returns, free the intance address.
	This same action will take place for the menu item Connect.


****************************************************************************/

long FAR PASCAL TestWndProc(hWnd, message, wParam, lParam)
HWND hWnd;				  /* window handle		     */
unsigned message;			  /* type of message		     */
WORD wParam;				  /* additional information	     */
LONG lParam;				  /* additional information	     */
{
    FARPROC lpProcAbout;		  /* pointer to the "About" function */
    FARPROC lpProcConnect;		  /* pointer to ConnectPipe function */
    FARPROC lpProcDefault;		  /* pointer to SetDefaults function */
    HMENU hMenu;			  /* handle to the System menu	     */

    switch (message) {
	case WM_SYSCOMMAND:		/* message: command from system menu */
	    if (wParam == ID_ABOUTTEST) {
		lpProcAbout = MakeProcInstance(AboutTest, hInst);

		DialogBox(hInst,		 /* current instance	     */
		    "ABOUTTEST",		 /* resource to use	     */
		    hWnd,			 /* parent handle	     */
		    lpProcAbout);		 /* About() instance address */

		FreeProcInstance(lpProcAbout);
		break;
	    }

	    else			    /* Lets Windows process it	     */
		return (DefWindowProc(hWnd, message, wParam, lParam));

	case WM_CREATE:			    /* message: window being created */

	    /* Get the handle of the System menu */

	    hMenu = GetSystemMenu(hWnd, FALSE);

            /* Add a separator toe the menu */

            ChangeMenu(hMenu,                         /* menu handle         */
                NULL,                                 /* menu item to change */
                NULL,                                 /* new menu item       */
                NULL,                                 /* menu identifier     */
                MF_APPEND | MF_SEPARATOR);            /* type of change      */


	    /* Add new menu item to the System menu */

	    ChangeMenu(hMenu,			      /* menu handle	     */
		NULL,				      /* menu item to change */
		"A&bout WinPipe...",   	      /* new menu item	     */
		ID_ABOUTTEST,			      /* menu identifier     */
		MF_APPEND | MF_STRING);		      /* type of change	     */

	    break;
	
	case WM_COMMAND :			/* menu selections generate */
						/* the WM_COMMAND message   */	
	    switch(wParam)			/* menu in WORD parameter   */
	    {
		case IDM_CONNECT :		/* connect to server	    */

		    lpProcConnect = MakeProcInstance(ConnectPipe, hInst);
		    DialogBox(hInst,		/* current instance	     */
			"CONNECT",	 	/* resource to use	     */
			hWnd,			/* parent handle	     */
			lpProcConnect);	 	/* ConnectPipe() instance address */

		    FreeProcInstance(lpProcConnect);
		    break;

		case IDM_DEFAULT :		/* set pipename and message  */

		    lpProcDefault = MakeProcInstance(SetDefaults, hInst);
		    DialogBox(hInst,		/* current instance	     */
			"SETDEFAULTS",	 	/* resource to use	     */
			hWnd,			/* parent handle	     */
			lpProcDefault);	 	/* ConnectPipe() instance address */

		    FreeProcInstance(lpProcConnect);
		    break;


                case IDM_QUIT :                 /* exit program */
                    PostQuitMessage(0);
                    break;
	    }
	    break;
	
	case WM_DESTROY:		  /* message: window being destroyed */
	    PostQuitMessage(0);
	    break;

	default:			  /* Passes it on if unproccessed    */
	    return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: AboutTest(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "AboutTest" dialog box

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:

	No initialization is needed for this particular dialog box, but TRUE
	must be returned to Windows.

	Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL AboutTest(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
    switch (message) {
	case WM_INITDIALOG:		   /* message: initialize dialog box */
	    return (TRUE);

	case WM_COMMAND:		      /* message: received a command */
	    if (wParam == IDOK) {	      /* "OK" box selected?	     */
		EndDialog(hDlg, NULL);	      /* Exits the dialog box	     */
		return (TRUE);
	    }
	    break;
    }
    return (FALSE);			      /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: ConnectPipe(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "Connect" dialog box

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:

	No initialization is needed for this particular dialog box, but TRUE
	must be returned to Windows.

        This is where the application attempts to open the named-pipe
        connection.  The user will enter the server name and a message box
        will appear saying that it worked, or giving an error code.

	Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL ConnectPipe(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
    char szServer[31];
    char szConnection[51];
    char szMessageReceived[MAX_MESSAGETEXT + 1];
    char szErrorBuffer[81];
    int iResult, fhPipe, iLoop;

    *szServer = NULL;
    switch (message) {
	case WM_INITDIALOG:		   /* message: initialize dialog box*/
	    SendDlgItemMessage(hDlg,       /* limit input to 30 characters  */
		SERVER,EM_LIMITTEXT,30,0L);
	    return (TRUE);

	case WM_COMMAND:		      /* message: received a command*/
	    errhWnd = hDlg;
	    switch(wParam)
	    {
		case IDOK :		      /* "OK" box selected?	    */
		    iResult = GetDlgItemText(hDlg,SERVER,
			(LPSTR)szServer,
			    MAX_SERVERNAME);  /* get Server name */
		    if(*szServer != NULL) /* was something input	    */
		    {
                       /* named-pipe tests */
                       szServer[iResult] = '\0';
                       strcpy(szConnection, "\\\\");
                       strcat(szConnection, szServer);
                       strcat(szConnection, "\\pipe\\");
                       strcat(szConnection, szPipeName);
                                              /* open the pipe */
                       fhPipe = sopen(szConnection, O_BINARY | O_RDWR,
                                      SH_DENYNO, S_IWRITE | S_IREAD);
                       if (fhPipe == -1)
                       {
                           sprintf(szErrorBuffer, "Failed to open pipe.  Status: %i",errno);
                           MessageBox(hDlg,szErrorBuffer,"ERROR",MB_OK);
                           MessageBox(hDlg,sys_errlist[errno],"Error Text",MB_OK);
                           break;
                       }                      /* set pipe state */
                       iResult = DosSetNmPHandState(fhPipe, NP_WAIT
                                                   | NP_READMODE_MESSAGE);
                       if (iResult != 0)
                           MessageBox(hDlg,"Couldn't set state","ERROR",MB_OK);
                       else
                       {                      /* write/read iteration times */
                           for (iLoop = 1; iLoop <= iIterationCount; iLoop++)
                           {                  /* write to the pipe */
                               iResult = write(fhPipe,szMessageText,strlen(szMessageText)+1);
                               if (iResult == -1)
                               {
                                   MessageBox(hDlg,"Write failed.","ERROR",MB_OK);
                                   break;
                               }
                               else
                               {              /* read from the pipe */
                                   iResult = read(fhPipe,szMessageReceived,sizeof(szMessageReceived));
                                   if (iResult == -1)
                                   {
                                       MessageBox(hDlg,"Read failed.","ERROR",MB_OK);
                                       break;
                                   }
                                   else
                                       szMessageReceived[iResult] = '\0';
                                       if (strcmp(szMessageReceived,szMessageText) != 0)
                                       {
                                           MessageBox(hDlg,szMessageReceived,"ERROR",MB_OK);
                                           break;
                                       }
                                       else
                                           if (iLoop == iIterationCount)
                                               MessageBox(hDlg,szMessageReceived,"Success",MB_OK);
                               }
                           }
                       }                      /* close the pipe */
                       iResult = write(fhPipe,"close",strlen("close")+1);
                       if (iResult == -1)
                           MessageBox(hDlg,"Write failed.","ERROR",MB_OK);
                       close(fhPipe);
                    }
		    EndDialog(hDlg, NULL);	      /* Exits the dialog box	     */
		    return (TRUE);
		    break;
		case IDCANCEL :
		    EndDialog(hDlg, NULL);
		    return(TRUE);
		    break;
		
	    }
	    break;
    }
    return (FALSE);			      /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: SetDefaults(HWND, unsigned, WORD, LONG)

    PURPOSE: Allow user to choose the pipe name and the message text

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:

	No initialization is needed for this particular dialog box, but TRUE
	must be returned to Windows.

        This is where the user is allowed to choose a pipename and the message
        that will be sent and received from MakePipe.

	Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL SetDefaults(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
    int iResult, iCount;
    char szTempString[MAX_MESSAGETEXT];

    switch (message) {
	case WM_INITDIALOG:		   /* message: initialize dialog box*/
	    SendDlgItemMessage(hDlg,       /* limit input to 79 characters  */
		MESSAGETEXT,EM_LIMITTEXT,MAX_MESSAGETEXT,0L);
	    SendDlgItemMessage(hDlg,       /* limit input to 9 characters  */
		PIPENAME,EM_LIMITTEXT,MAX_PIPENAME,0L);
	    SendDlgItemMessage(hDlg,       /* limit input to 5 characters  */
		ITERATIONS,EM_LIMITTEXT,MAX_ITERATIONS,0L);
	    return (TRUE);

	case WM_COMMAND:		      /* message: received a command*/
	    errhWnd = hDlg;
	    switch(wParam)
	    {
		case IDOK :		      /* "OK" box selected?	    */
		    iResult = GetDlgItemText(hDlg,MESSAGETEXT,
			(LPSTR)szTempString,
			    MAX_MESSAGETEXT); /* get message text */
		    if(iResult)                /* was something input	    */
                    {
                        szTempString[iResult] = '\0';
                        strcpy(szMessageText,szTempString);
                    }
                    iResult = GetDlgItemText(hDlg,PIPENAME,
			(LPSTR)szTempString,
			    MAX_PIPENAME);    /* get pipe name */
		    if(iResult)                /* was something input	    */
                    {
                        szTempString[iResult] = '\0';
                        strcpy(szPipeName,szTempString);
                    }
                    iResult = GetDlgItemText(hDlg,ITERATIONS,
			(LPSTR)szTempString,
			    MAX_ITERATIONS);   /* get iteration count */
		    if(iResult)                /* was something input	    */
                    {
                        szTempString[iResult] = '\0';
                        iIterationCount = atoi(szTempString);
                    }
 		    EndDialog(hDlg, NULL);
		    return(TRUE);
		    break;

		case IDCANCEL :
 		    EndDialog(hDlg, NULL);
		    return(TRUE);
		    break;
		
	    }
	    break;
    }
    return (FALSE);			      /* Didn't process a message    */
}
