/*
**
**  PROGRAM:    SQLWINBK.C
**
**  PURPOSE:    Windows DB-Library Query Application, using PeekMessage method
**
**  FUNCTIONS:  See SQLWINBK.H
**
**  Chris Moffatt
**  Technical Resource Group, SQL Server
**
*/

#include <windows.h>                /* standard Windows include file */
#include <string.h>                 /* C string functions include library */
#define DBMSWIN                     /* needed to define environment */
#include <sqlfront.h>               /* standard dblib include file */
#include <sqldb.h>                  /* standard dblib include file */
#include "sqlwinbk.h"               /* specific to this program */
#include "windlg.h"                 /* dialogs include file */

/* GLOBAL VARIABLES */
DBPROCESS *pDbproc = (DBPROCESS *)NULL; /* connection to SQL Server */
char      szAppName[] = "SQLWinBK";     /* application name */
char      szDblibVersion[50];           /* DB-Library version string */
FARPROC   lpMessageHandler,             /* pointer to MessageHandler */
          lpErrorHandler;               /* pointer to ErrorHandler */
HANDLE    hInst;                        /* current instance */
HANDLE    hWnd;                         /* current window */
HANDLE    hQuery,                       /* handle to query edit control */
          hResult;                      /* handle to result edit control */
HANDLE    hQueryBuffer,                 /* handle to query edit buffer */
          hHeaderBuffer,                /* handle to column headings */
          hResultBuffer,                /* handle to result edit buffer */
          hRowBuffer;                   /* handle to current row buffer */
LPSTR     lpQueryBuffer,                /* pointer to query edit buffer */
          lpHeaderBuffer,               /* pointer to column headings */
          lpResultBuffer,               /* pointer to result edit buffer */
          lpRowBuffer;                  /* pointer to row buffer */
BOOL      bQueryPending = FALSE;        /* query pending */
BOOL      bCancelQuery = FALSE;         /* cancel query */
int       iProcessQueryAction;

/*
**
**  FUNCTION:   WinMain()
**
**  PURPOSE:    Calls initialization function, processes message loop
**
*/

int PASCAL WinMain(HANDLE hInstance,
    HANDLE hPrevInstance,
    LPSTR lpCmdLine,
    int nCmdShow)
{
    MSG msg;
    HANDLE hAccel;

    if (!hPrevInstance)
        if (!InitApplication(hInstance))    /* Initialize shared things */
            return (FALSE);

    /* Perform initializations that apply to a specific instance */
    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    hAccel = LoadAccelerators (hInstance, "WinQueryAccelerators");

    while (TRUE)
    {
        if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
        {
            /* message in queue, process it */
            if (msg.message != WM_QUIT)
            {
                if (!TranslateAccelerator (hWnd, hAccel, &msg))
                {
                    TranslateMessage(&msg); /* Translates virtual key codes */
                    DispatchMessage(&msg);  /* Dispatches message to window */
                }
            }
            else
            {
                break;
            }
        }
        else  /* no message in queue */
        {
            /* do background processing if necessary */
            if (!DoBackgroundPart())
            {
                WaitMessage();
            }
        }
    }

    return (msg.wParam);        /* Returns the value from PostQuitMessage */
}

/*
**
**  FUNCTION:   InitApplication()
**
**  PURPOSE:    Initializes window data and registers window class
**
*/
BOOL InitApplication(HANDLE hInstance)
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the */
    /* main window. */
    wc.style = NULL;                    /* Class style(s) */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for */
                                        /* windows of this class */
    wc.cbClsExtra = 0;                  /* No per-class extra data */
    wc.cbWndExtra = 0;                  /* No per-window extra data */
    wc.hInstance = hInstance;           /* Application that owns the class */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);     /* blank default icon */
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       /* arrow cursor */
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); /* white background */
    wc.lpszMenuName =  "WinQueryMenu";  /* Name of menu resource in */
                                        /* .RC file. */
    wc.lpszClassName = szAppName;       /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */
    return (RegisterClass(&wc));
}

/*
**
**  FUNCTION:   InitInstance()
**
**  PURPOSE:    Saves instance handle and creates main window
**
*/
BOOL InitInstance(HANDLE hInstance,int nCmdShow)
{
    DWORD dwEditStyle;

    /* Save the instance handle in static variable, which will be used in */
    /* many subsequent calls from this application to Windows. */
    hInst = hInstance;

    /* Create a main window for this application instance. */
    hWnd = CreateWindow(szAppName,  /* Class to use */
        "SQLWinBK",                 /* Text for window title bar */
        WS_OVERLAPPEDWINDOW,        /* Window style */
        CW_USEDEFAULT,              /* Default horizontal position */
        CW_USEDEFAULT,              /* Default vertical position */
        CW_USEDEFAULT,              /* Default width */
        CW_USEDEFAULT,              /* Default height */
        NULL,                       /* Overlapped windows have no parent */
        NULL,                       /* Use the window class menu */
        hInstance,                  /* This instance owns this window */
        NULL);                      /* Pointer not needed */

    /* If window could not be created, return "failure" */
    if (!hWnd)
        return (FALSE);

    dwEditStyle = WS_CHILD | WS_VISIBLE | WS_BORDER |
        WS_HSCROLL | WS_VSCROLL |
        ES_LEFT | ES_MULTILINE |
        ES_AUTOHSCROLL | ES_AUTOVSCROLL;

    /* create query edit window */
    hQuery = CreateWindow("edit",
        NULL,
        dwEditStyle,    /* style params */
        0, 0, 0, 0,     /* initial dimensions all 0 */
        hWnd,           /* parent handle */
        IDE_QUERY,      /* unique child-window identifier */
        hInst,          /* instance  handle */
        NULL);

    /* create result edit window */
    hResult = CreateWindow("edit",
        NULL,
        dwEditStyle,    /* style params */
        0, 0, 0, 0,     /* initial dimensions all 0 */
        hWnd,           /* parent handle */
        IDE_RESULT,     /* unique child-window identifier */
        hInst,          /* instance  handle */
        NULL);

    SetFocus (hQuery);

    hResultBuffer = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,1);
    hRowBuffer = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,1);

    ShowWindow(hWnd, nCmdShow);     /* Show the window */
    UpdateWindow(hWnd);             /* Sends WM_PAINT message */
    return (TRUE);
}

/*
**
**  FUNCTION:   MainWndProc()
**
**  PURPOSE:    Processes messages
**
*/

LONG FAR PASCAL MainWndProc(
    HWND hWnd,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    FARPROC lpDlgProc;
    int 	nWidth, nHeight;
    static 	HANDLE hCurrentFocus;

    switch (message)
    {
        case WM_COMMAND:    /* message: command from application menu */
            switch (wParam)
            {
                case IDM_CONNECT:
                    lpDlgProc = MakeProcInstance(Connect, hInst);

                    DialogBox(hInst,        /* current instance */
                        "ConnectBox",       /* resource to use */
                        hWnd,               /* parent handle */
                        lpDlgProc);         /* Connect() instance address */

                    FreeProcInstance(lpDlgProc);
                    break;

                case IDM_EXECUTE:
                    if (pDbproc != (DBPROCESS *)NULL)
                    {
                        if (!bQueryPending)
                            SendQuery(pDbproc);
                        else /* query pending */
                            ERRBOX (hWnd, "Query still pending.");
                    }
                    else /* null dbproc */
                        ERRBOX(hWnd, "No connection available");
                    break;

               case IDM_CANCEL:
                    CancelQuery(pDbproc);
                    break;

                case IDM_ABOUT:
                    lpDlgProc = MakeProcInstance(About, hInst);

                    DialogBox(hInst,        /* current instance */
                        "AboutBox",         /* resource to use */
                        hWnd,               /* parent handle   */
                        lpDlgProc);         /* About() instance address */

                    FreeProcInstance(lpDlgProc);
                    break;

                case IDE_QUERY:
                    switch (HIWORD(lParam))
                    {
                        case EN_ERRSPACE:
                            MSGBOX (hWnd, "Query edit control out of space");
                            CancelQuery(pDbproc);
                            break;

                        case EN_SETFOCUS:
                            hCurrentFocus = hQuery;
                            break;
                    }
                    break;

                case IDE_RESULT:
                    switch (HIWORD(lParam))
                    {
                        case EN_ERRSPACE:
                            MSGBOX (hWnd, "Result edit control out of space");
                            CancelQuery(pDbproc);
                            break;

                        case EN_SETFOCUS:
                            hCurrentFocus = hResult;
                            break;
                    }
                    break;

                default:
                    /* Pass it on if unproccessed */
                    return (DefWindowProc(hWnd, message, wParam, lParam));
            }
            break;

        case WM_CREATE:     /* message: window being created */
            hCurrentFocus = hQuery;

        	if (!InitDblib())
        	{
	            ERRBOX(hWnd, "Failed to initialize DB-Library");
        	}
        	break;

    	case WM_DESTROY:    /* message: window being destroyed */
	        ExitDblib();
        	PostQuitMessage(0);
        	break;

    	case WM_SETFOCUS:
	        SetFocus (hCurrentFocus);
        	break;

    	case WM_SIZE:
	        nWidth = LOWORD(lParam);
        	nHeight = HIWORD(lParam);
        	MoveWindow(hQuery,      /* window to move */
	            0,                  /* x coord */
            	0,                  /* y coord */
            	nWidth,             /* width */
            	nHeight/2,          /* height */
            	TRUE);              /* repaint */

        	MoveWindow(hResult,     /* window to move */
	            0,                  /* x coord */
            	nHeight/2,          /* y coord */
            	nWidth,             /* width */
            	nHeight/2,          /* height */
            	TRUE);              /* repaint */

        	break;

    	default:
	        /* Pass it on if unproccessed */
        	return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}

/*
**
**  FUNCTION:   DoBackgroundPart()
**
**  PURPOSE:    Do background processing
**
*/
BOOL DoBackgroundPart (void)
{
    BOOL bContinue;

    if (bQueryPending)
    {
        bContinue = ProcessQuery(pDbproc);
    }
    else
    {
        bContinue = FALSE;
    }
    return (bContinue);
}

/*
**  FUNCTION:   InitDblib()
**
**  PURPOSE:    Initialize DB-Library
*/
BOOL InitDblib (void)
{
    BOOL bSuccess;

    wsprintf(szDblibVersion, "%s", dbinit());
    if (szDblibVersion == "")
    {
        bSuccess = FALSE;
    }
    else
    {
        bSuccess = TRUE;
        /* get procedure instances */
        lpErrorHandler = MakeProcInstance((FARPROC)ErrorHandler, hInst);
        lpMessageHandler = MakeProcInstance((FARPROC)MessageHandler, hInst);
        /* register handlers with DB-Library */
        dberrhandle(lpErrorHandler);
        dbmsghandle(lpMessageHandler);
        /* set DB-Library options */
        dbsetlogintime(10);
        dbsettime(100);
    }
    return (bSuccess);
}

/*
**
**  FUNCTION:   ExitDblib()
**
**  PURPOSE:    Exit from DB-Library
**
*/
BOOL ExitDblib (void)
{
    dbexit();
    dbwinexit();
    FreeProcInstance(lpErrorHandler);
    FreeProcInstance(lpMessageHandler);
    return (TRUE);
}

/*
**
**  FUNCTION:   SendQuery()
**
**  PURPOSE:
*/
BOOL SendQuery (DBPROCESS *pDbproc)
{
    BOOL bSuccess;
    DWORD dwLines, dwLength,dwRow;
    char pLine[256];

    /* Display Processing message */
    GlobalFree(hResultBuffer);
    hResultBuffer = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,1);
    StrCat(&hResultBuffer, (LPSTR)"Processing query asynchronously....");
    StrCat(&hResultBuffer, (LPSTR)"You may do other work, or select CTRL+C to end...\r\n");
    lpResultBuffer = GlobalLock (hResultBuffer);
    SetWindowText (hResult, lpResultBuffer);
    GlobalUnlock (hResultBuffer);

    dwLines = SendMessage(hQuery,EM_GETLINECOUNT,0,0);

    dbcancel(pDbproc);
    for (dwRow=0;dwRow < dwLines;dwRow++)
    {
        dbcmd(pDbproc, " ");
        pLine[0] = 150;
        pLine[1] = 0;
        dwLength = SendMessage(hQuery,
            EM_GETLINE,
            (WORD)dwRow,
            (LONG)((LPSTR)pLine));
        pLine[dwLength] = '\0';
        dbcmd(pDbproc, pLine);
    }

    switch (dbsqlsend(pDbproc))
    {
    case SUCCEED:
        bQueryPending = TRUE;
        iProcessQueryAction = DO_DBDATAREADY;
        GlobalFree (hResultBuffer);
        hResultBuffer = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,1);
        bSuccess = TRUE;
        break;

    case FAIL:
        ERRBOX(hWnd, "Query could not be sent");
        bSuccess = FALSE;
        break;
    }
    return (bSuccess);

}

/*
**
**  FUNCTION:   CancelQuery()
**
**  PURPOSE:    Handles a cancel query request
**
*/
BOOL CancelQuery (DBPROCESS *pDbproc)
{
    BOOL bSuccees;

    switch (dbcancel(pDbproc))
    {
    case SUCCEED:
        MSGBOX(hWnd, "Query execution canceled");
        bQueryPending = FALSE;
        bSuccees = TRUE;
        break;
    case FAIL:
        ERRBOX(hWnd, "Query execution could NOT be canceled");
        bSuccees = FALSE;
        break;
    }
    return (bSuccees);
}

/*
 *
 *  FUNCTION:   ProcessQuery()
 *
 *  PURPOSE:    Asynchronous handling of query
 *
 */
BOOL ProcessQuery (
    DBPROCESS *pDbproc)
{
    BOOL bContinue = FALSE;
    STATUS s;

    switch (iProcessQueryAction)
    {
        case DO_DBDATAREADY:
            if (dbdataready(pDbproc))
            {
                iProcessQueryAction = DO_DBSQLOK;
                bContinue = TRUE;
            }
            else
            {
                bContinue = TRUE;
            }
            break;

        case DO_DBSQLOK:
            switch (dbsqlok(pDbproc))
            {
                case SUCCEED:
                    iProcessQueryAction = DO_DBRESULTS;
                    bContinue = TRUE;
                    break;

                case FAIL:
                    ERRBOX (hWnd, "Failed to execute query");
                    bContinue = FALSE;
                    break;
            }
            break;

        case DO_DBRESULTS:
            switch (dbresults(pDbproc))
            {
                case SUCCEED:
                    /* Retirve header and add to result buffer */
                    ConstructHeader(pDbproc, &hHeaderBuffer);
                    lpHeaderBuffer = GlobalLock (hHeaderBuffer);
                    StrCat(&hResultBuffer,lpHeaderBuffer);
	                StrCat(&hResultBuffer, (LPSTR)"\r\n");
	                StrCat(&hResultBuffer, (LPSTR)"\r\n");
	                GlobalUnlock (hHeaderBuffer);
                    iProcessQueryAction = DO_DBNEXTROW;
                    bContinue = TRUE;
                    break;

                case NO_MORE_RESULTS:
                    lpResultBuffer = GlobalLock (hResultBuffer);
                    SetWindowText (hResult, lpResultBuffer);
                    GlobalUnlock (hResultBuffer);
                    MSGBOX(hWnd, "Query execution completed");
                    bContinue = FALSE;
                    break;

                case FAIL:
                    ERRBOX(hWnd, "dbresults() returned FAIL");
                    bContinue = FALSE;
                    break;
            }
            break;

        case DO_DBNEXTROW:
            switch (s=dbnextrow(pDbproc))
            {
                case REG_ROW:
                    /* Retrieve row and add to result buffer */
                    ConstructRow(pDbproc, &hRowBuffer);
                    lpRowBuffer = GlobalLock (hRowBuffer);
	                StrCat(&hResultBuffer,lpRowBuffer);
	                StrCat(&hResultBuffer, (LPSTR)"\r\n");
	                GlobalUnlock (hRowBuffer);
                    bContinue = TRUE;
                    break;

                case NO_MORE_ROWS:
                    iProcessQueryAction = DO_DBRESULTS;
                    bContinue = TRUE;
                    break;

                case BUF_FULL:
                    /* clear buffer */
                    bContinue = TRUE;
                    break;

                case FAIL:
                    ERRBOX(hWnd, "dbnextrow() returned FAIL");
                    bContinue = FALSE;
                    break;

                default:
                    MSGBOX(hWnd, "dbnextrow() returned a compute row");
                    ConstructRow(pDbproc,&hRowBuffer);
                    lpRowBuffer = GlobalLock (hRowBuffer);
	                StrCat(&hResultBuffer,lpRowBuffer);
	                StrCat(&hResultBuffer, (LPSTR)"\r\n");
	                GlobalUnlock (hRowBuffer);
                    bContinue = TRUE;
                    break;
            }
            break;
    }

    bQueryPending = bContinue;

    return (bContinue);
}

/*
** FUNCTION		ConstructHeader()
**
** PURPOSE		This function builds the string that contains the names
**   			of each column. It does this by finding the print size of
**				each column, allocating a buffer to hold all column names
**				plus one space between each column name, then copying that
**				name into the  appropriate location into the buffer.
**				Finally the two lines are printed.
*/
RETCODE ConstructHeader(DBPROCESS *dbproc, HANDLE *phHeader)
{
    int   i,iCols,iSize, iTabs;		/* counters */
    LPSTR lpHeader;        			/* pointer for seperator buffer */
    LPSTR lpColName;				/* pointer for column names buffer */
    LPSTR lpName;					/* pointer for column name buffer */
	LPSTR lpPtr;     				/* scratch pointer */

	/* Get number of columns so we now how much memory to allo for tabs */
    iTabs = dbnumcols(dbproc);

    /* Get row size and allocate memory for buffer */
    iSize = DetermineRowSize(dbproc,0);
    GlobalFree(*phHeader);
    *phHeader = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,
        (DWORD) (iSize + iTabs + 1));
    if(*phHeader != NULL)
    	lpPtr = lpHeader = GlobalLock(*phHeader);
   	else
   	   	return(FAIL);

    /* Initialize buffer to spaces */
    _fmemset (lpPtr,' ',iSize);

    /* Get number of columns */
    iCols = dbnumcols(dbproc);

    /* Loop on all columns, retrieving column size and name, and adding */
    /* to buffer, spaced with tabs (\t) */
    for(i=1;i<=iCols;i++)
    {
        iSize = DetermineRowSize(dbproc,i);
        lpColName = dbcolname(dbproc,i);
        _fstrncpy(lpPtr,lpColName,lstrlen(lpColName));
        lpPtr+= iSize;
        _fstrncpy(lpPtr, (LPSTR) "\t",1);
        lpPtr += 1;
    }

    /* Finished: null terminate buffer, unlock buffer and return SUCCEED */
    *lpPtr = '\0';
    GlobalUnlock(*phHeader);
    return(SUCCEED);
}


/*
** FUNCTION: 	ConstructRow()
**
** PURPOSE:		This function constructs one row.  dbnextrow() must be
**				called to fetch the row.  This routine could be used to
**				print the current row as many times as wanted, as the
**				current row data is always available until dbnextrow()
**				is called to fetch the next row.  This routine works like
** 				ConstructHeader above, but each column's data is obtained
**				instead of a row name, and converted to a string.  It is then set into the buffer.
**				inserted in the buffer.
*/
RETCODE ConstructRow(DBPROCESS *dbproc, HANDLE *phDataVals)
{
    int i,iCols,iSize,iDatasize, iTabs;	/* counters */
    LPSTR lpDataVals;				    /* data buffer pointer */
    LPSTR lpData;					    /* column data pointer */
    LPSTR lpPtr;					    /* scratch pointer */

     /* Get number of columns so we now how much memory to alloc for tabs */
     iTabs = dbnumcols(dbproc);

    /* Get row size and allocate memory for buffer */
    iSize = DetermineRowSize(dbproc,0);
    GlobalFree(*phDataVals);
    *phDataVals = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,
        (DWORD) (iSize + iTabs + 1));
    if(*phDataVals != NULL)
    	lpPtr = lpDataVals = GlobalLock(*phDataVals);
   	else
   	   	return(FAIL);

    /* Get number of columns */
	iCols = dbnumcols(dbproc);

	/* Loop through all columns, initialize to spaces and then */
	/* retrieve values */
    for(i=1;i<=iCols;i++)
    {
        iSize = DetermineRowSize(dbproc,i);
        _fmemset(lpPtr,' ',iSize);
        lpData = dbdata(dbproc,i);
        if(lpData == (BYTE *)NULL)	/* if NULL, use "NULL" */
        {
            _fstrncpy(lpPtr,(LPSTR) "NULL",4);
            lpPtr += iSize;
            _fstrncpy(lpPtr, (LPSTR) "\t",1);
            lpPtr += 1;
        }
        else	/* else we have data, so convert to char */
        {
            iDatasize = dbconvert(dbproc,
            	dbcoltype(dbproc,i),
            	lpData,
            	dbdatlen(dbproc,i),
                SQLCHAR,
                lpPtr,
                (DBINT)iSize-1);
            lpPtr += iSize;
            _fstrncpy(lpPtr, (LPSTR) "\t",1);
            lpPtr += 1;
        }
    }

    /* Finished: null terminate buffer, unlock buffer and return SUCCEED */
    *lpPtr = '\0';
    GlobalUnlock(*phDataVals);
    return(SUCCEED);
}


/*
** FUNCTION: 	DetermineRowSize()
**
** PURPOSE:		This function returns either the size of all columns in the
**				row, converted to character data (SQLCHAR) with one space
**				between each column, or if col is non-zero, the iLength of
**				the input column converted to string. It is used to build
**				the header strings, and each row of data, and is called to
**				allocate the memory needed for each row, and determine how
** 				much of that space is to be used for each column
**
** 				If 0 is passed in as the 2nd parameter, the size of all the
**				rows combines is calculated
*/
int DetermineRowSize(DBPROCESS *dbproc,int iCol)
{
    int i,iCols;				/* counters */
    int iLength=0;				/* total length of column(row) */
    DBINT ColLength;			/* actual length of column */
    int iNamelength;			/* iLength of name of column */
    int iPrLength;				/* printable iLength */
    LPSTR lpName;				/* pointer to column name */

    /* Get number of columns */
    if(!iCol)
       iCols = dbnumcols(dbproc);

    /* count from 1 to numcols if col is 0, else i will equal col only */
    for(i =  ((iCol) ? iCol : 1);
        i <= ((iCol) ? iCol : iCols);
        i++)
    {
    	/* get column type & determine SQLCHAR converted iLength */
	/* (Values below are found in SQLFRONT.H header file) */
        switch(dbcoltype(dbproc,i))
        {
	    case SQLBIT:
	        iPrLength = PRBIT;
	        break;
	    case SQLINT1:
   	        iPrLength = PRINT1;
	        break;
	    case SQLINT2:
  	        iPrLength = PRINT2;
	        break;
	    case SQLINT4:
	        iPrLength = PRINT4;
	        break;
	    case SQLFLT8:
	        iPrLength = PRFLT8;
	        break;
	    case SQLDATETIME:
	        iPrLength = PRDATETIME;
	        break;
	    case SQLMONEY:
	        iPrLength = PRMONEY;
	        break;
	    /* VARBINARY IMAGE, and BINARY ...convert to 2 times iLength */
        case SQLVARBINARY :
        case SQLBINARY:
        case SQLIMAGE:
        	iPrLength = (int) dbcollen(dbproc,i)*2;
            break;
        /* other types are maximum of actual column iLength */
	    default :
	        iPrLength = (int) dbcollen(dbproc,i);
	        break;
        }

        /* names may be longer than column so use name len if longer of two */
        lpName = dbcolname(dbproc,i);
        iNamelength =  (lpName) ? lstrlen(lpName) : 0;

        /* add one for space between columns */
        if(iPrLength<iNamelength)
           iLength += iNamelength+1;
        else
           iLength += iPrLength+1;
    }

    /* Return the length of the field */
    return iLength;
}


/*
**  FUNCTION:   StrCat()
**
** PURPOSE:		Concatenates a string into a buffer, allocating
**           	memory as it goes.
**
*/

BOOL StrCat(HANDLE *phBuffer,LPSTR lpStr)
{
    BOOL bSuccess;
    int iStrLen;
    DWORD iBufferSize;
    HANDLE hNewBuffer;
    LPSTR lpBuffer;

    /* Determine how much we need to increase buffer size to */
    iStrLen = lstrlen (lpStr);
    iBufferSize = GlobalSize(*phBuffer);

    /* Reallocate memory */
    hNewBuffer = GlobalReAlloc (*phBuffer,
        iBufferSize + iStrLen,
        GMEM_MOVEABLE | GMEM_ZEROINIT);

    if (hNewBuffer)
    {   /* concatenate new data */
        *phBuffer = hNewBuffer;
        lpBuffer = GlobalLock (*phBuffer);
        lstrcat (lpBuffer, lpStr);
        GlobalUnlock(*phBuffer);
        bSuccess = TRUE;
    }
    else /* not enough memory to realloc */
    {
        MSGBOX (hWnd, "Memory allocation failed");
        CancelQuery(pDbproc);
        bSuccess = FALSE;
    }
    return (bSuccess);
}


/*
**
**  FUNCTION:   Connect()
**
**  PURPOSE:    Processes messages for "Connect" dialog box
**
*/
BOOL FAR PASCAL Connect(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    char szSQLServer[MAX_NAME+1],
         szSQLLogin[MAX_NAME+1],
         szSQLPassword[MAX_NAME+1];
    LOGINREC *pLoginRec;

    switch (message)
    {
        case WM_INITDIALOG:     /* message: initialize dialog box */
            return (TRUE);
            break;

        case WM_COMMAND:    /* message: received a command */
            switch (wParam)
            {
                case IDOK:
                    /* get server name, login, and password */
                    GetDlgItemText(hDlg,
                        SQL_SERVER,
                        szSQLServer,
                        MAX_NAME);

                    GetDlgItemText(hDlg,
                        SQL_LOGIN,
                        szSQLLogin,
                        MAX_NAME);

                    GetDlgItemText(hDlg,
                        SQL_PASSWORD,
                        szSQLPassword,
                        MAX_NAME);

                    if(pDbproc != (DBPROCESS *)NULL)
                    {
                        dbclose(pDbproc);
                    }
                    pLoginRec = dblogin();
                    if (pLoginRec != (LOGINREC *)NULL)
                    {
                        DBSETLUSER(pLoginRec, szSQLLogin);
                        DBSETLPWD(pLoginRec, szSQLPassword);
                        DBSETLHOST(pLoginRec, szAppName);
                        pDbproc = dbopen(pLoginRec, szSQLServer);
                        if (pDbproc != (DBPROCESS *)NULL)
                        {
                            MSGBOX(hDlg, "Connected to SQL Server");
                        }
                        else /* dbopen() failed */
                        {
                            ERRBOX(hDlg, "Failed to connect to SQL Server");
                        }
                    }
                    else /* dblogin() failed */
                    {
                        ERRBOX(hDlg, "Failed to allocate login record");
                    }
                    dbfreelogin(pLoginRec);

                    EndDialog(hDlg, TRUE);
                    return (TRUE);
                    break;

                case IDCANCEL:
                    EndDialog(hDlg, TRUE);
                    return (TRUE);
                    break;
            }
            break;
    }
    return (FALSE);     /* Didn't process a message */
}

/*
**  FUNCTION:   About()
**
**  PURPOSE:    Processes messages for "About" dialog box
**
**/

BOOL FAR PASCAL About(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam)
{
    switch (message)
    {
        case WM_INITDIALOG:     /* message: initialize dialog box */
            SetDlgItemText(hDlg, SQL_DBLIBVERSION,szDblibVersion);
            return (TRUE);

        case WM_COMMAND:        /* message: received a command */

            switch (wParam)
            {
                case IDOK:
                case IDCANCEL:
                    EndDialog(hDlg, TRUE);
                    return (TRUE);
                    break;
            }
            break;
    }
    return (FALSE);     /* Didn't process a message */
}

/*
**
**  FUNCTION:   ErrorHandler()
**
**  PURPOSE:    Processes DB-Library errors.
**
*/

int FAR ErrorHandler(
    DBPROCESS *pDbproc,
    DBSMALLINT severity,
    DBSMALLINT errno,
    DBSMALLINT oserr,
    LPSTR dberrstr,
    LPSTR oserrstr)
{
    MessageBox(hWnd,
        dberrstr,
        "DB-Library Error",
        MB_OK | MB_ICONSTOP);

    if (oserr != DBNOERR)       /* operating system error */
    {
        MessageBox(hWnd,
            oserrstr,
            "Operating System Error",
            MB_OK | MB_ICONSTOP);
    }

    /* this will cause the offending DB-Library function to return FAIL */
    return(INT_CANCEL);
}

/*
**
**  FUNCTION:   MessageHandler()
**
**  PURPOSE:    Processes SQL Server messages.
**
*/

int FAR MessageHandler(
    DBPROCESS *pDbproc,
    DBINT msgno,
    DBSMALLINT msgstate,
    DBSMALLINT severity,
    LPSTR msgtext)
{
    /* Filter out database change and language msgs */
    if (msgno != 5701 && msgno != 5703)
        MessageBox(hWnd,
            msgtext,
            "SQL Server Message",
            MB_OK | MB_ICONINFORMATION);
    return(0);
}
