/***********************************************************************
Multi-threaded Named Pipe test program.
Use it similar to the SQL Server MAKEPIPE/READPIPE programs.
There are actually three different programs in this group, currently
called MKPIPE2, RDPIPE2, and RDPIPE

MKPIPE2 is a multi-threaded MAKEPIPE-like,
OS/2-only server program.  Without arguments, it will generate 10
threads, each servicing a Named Pipe.  Pipe Name is always the same,
ABC.  Each pipe is actually an additional instance with the same
name.  A single numeric argument can be used to specify an alternate
number of threads.  The core of the program is a loop that reads data
with DOSREAD and writes it back out with DOSWRITE.

RDPIPE2 is a multi-threaded READPIPE-like,
OS/2-only client program.  Without arguments, it will generate 10
threads, each firing off DosTransactNmPipe to the server.  The format
of the server name is \\servername\pipe\abc, where servername is
the network name of the workstation, pipe is literally 'pipe', and
abc is literally 'abc'.  RDPIPE2 automatically prompts for the name of
the workstation.  If no argument is entered, local OS/2 pipes are
used.  Command line passing of the servername was not implemented
because this program is intended only for interactive use.

RDPIPE is a dual-mode (bound) program for both OS/2 and DOS.	It is
essentially a single-threaded version of RP5.

Each thread number is used as a y-axis offset to position the cursor
at the appropriate screen location for writing status information.
Each pair of _settextposition and printf is defined as a critical
section to prevent the OS/2 scheduler kicking in after the
_settextposition call, but before the printf call.
-- Joe Marler, 9/91

10/91 -- Modified to prompt for delay on how quickly to honor the pipe
request.  This effectively governs the host CPU load on both
client and server, since the clients block waiting for their
request to be honored.

11/91 -- Modified to increase buffer size to 1542 bytes, which is
more typical of what SQL Server uses.  Changed corruption check and
corruption error output to work better.

Joe Marler
Microsoft Product Support Services.

***********************************************************************/

// Includes ----------------------------------------------------------------
#define INCL_DOS
#define INCL_DOSPROCESS
#define INCL_VIO
#define INCL_DOSFILEMGR
#define INCL_DOSNMPIPES
#include  <process.h>
#include  <stddef.h>
#include  <stdio.h>
#include  <stdlib.h>
#include  <os2.h>
#include <netcons.h>
#include <graph.h>



// Prototypes --------------------------------------------------------------
void _far MakePipe(unsigned long ulLimit);
void      main     (int argc, char *argv[]);
long mpdelay=0;			/* Global so all threads can see */

/*************************************************************************
*   Here begins the function that is spun off as seperate threads	 *
*************************************************************************/

void _far MakePipe(unsigned long ulLimit)
{
 #define BUFSZ 1542	    // size of S.S. output buffer

 /****************************** VARIABLES */

  char pipename[80];
  char buf[BUFSZ];

  /* Need next 2 lines for DosPeekNmPipe call */
  AVAILDATA var1;	    //declare structure var1 of type AVAILDATA
  PAVAILDATA pavail;	    //declare pointer to this structure var.


  unsigned char pkbuf[100];
  short pkbytesrd;
  short pkpipst;

  int pipe;
  long counter;
  int threadid;

  unsigned result;
  unsigned bytes;

/******************************* INITIALIZE */

  pavail = &var1;	    //
  counter=0;

  threadid = (int) ulLimit;

  strcpy(pipename,"\\pipe\\abc");



/******************************* Make the Pipe */

  result=DosMakeNmPipe(

	pipename,		  // pipe name

	&pipe,			  // pipe handle

	NP_ACCESS_DUPLEX |	  // open mode
	NP_NOINHERIT	 |
	NP_NOWRITEBEHIND ,

	NP_WAIT 	       |  // pipe mode
	NP_READMODE_MESSAGE    |
	NP_TYPE_MESSAGE        |
	NP_UNLIMITED_INSTANCES ,

	BUFSZ,			  // out buffer

	BUFSZ,			  // in  buffer

	0L);			  // timeout

  if (result)	// result is non zero means it failed
    {
     _settextposition(threadid, 0);
     printf("Thread %d died, Failed to Make Pipe, Status=%d",threadid, result);
     _endthread();
    }


     /************************* CONNECT PIPE */

      DosEnterCritSec();
      _settextposition(threadid, 0);
      printf("Thread %d waiting for Client to Connect...", threadid);
      DosExitCritSec();
      result=DosConnectNmPipe(pipe);

      if (result)		  // result !=0 == fail
	{
	 _settextposition(threadid, 0);
	 printf("Thread %d died, Failed to Connect, Status=%d",threadid, result);
	 _endthread();
	}


     while (1)	  /* Loop forever, reading and writing data */
       {
	DosEnterCritSec();
	_settextposition(threadid, 0);
	printf("Thread %d waiting for client to send...  %ld ", threadid, counter);
	DosExitCritSec();
	counter++;

	/************************* READ PIPE */

	DosEnterCritSec();
	_settextposition(threadid, 0);
	printf("Thread %d reading......................", threadid);
	DosExitCritSec();


	/* Just doing DosPeekNmPipe to test the call */
	result=DosPeekNmPipe(pipe,
			     pkbuf,
			     100,
			     &pkbytesrd,
			     pavail,
			     &pkpipst);


	if (result)	// FAIL  -- result > 0
	  {
	   _settextposition(threadid, 0);
	   printf("Thread %d died, DosPeekNmPipe failed, Status=%d, Bytes Read=%d     ",threadid, result,bytes);
	   _endthread();
	  }

	result=DosRead(
	       pipe,
	       (char far *) buf,
	       sizeof(buf),
	       &bytes);

	if (result)	// FAIL  -- result > 0
	  {
	   _settextposition(threadid, 0);
	   printf("Thread %d died, Read Failed, Status=%d, Bytes Read=%d     ",threadid, result,bytes);
	   _endthread();
	  }

	/* If pipe we are reading from goes away, you don't get an */
	/* error from DosRead().  You get a good result, with 0-bytes */
	/* read.  This is documented behavior.	Following test checks */
	/* for this.  Should probably check for pipe state at this point */
	/* but that's for another time. */
	if (bytes == 0)
	  {
	  _settextposition(threadid, 0);
	  printf("Thread %d died because 0 bytes were read             ", threadid);
	  _endthread();
	  }


	/******* DELAY A WHILE *******/
	DosSleep(mpdelay);



	  /************************* WRITE PIPE */
	DosEnterCritSec();
	_settextposition(threadid, 0);
	printf("Thread %d writing......................", threadid);
	DosExitCritSec();
	result=DosWrite(
			pipe,
			(char far *) buf,
			bytes,
			&bytes);

	if (result)
	  {
	   _settextposition(threadid, 0);
	   printf("Thread %d died, Write Failed, Status=%d     ",threadid, result);
	   _endthread();
	  }


       }  // end while

}//End MakePipe



/**************************************************************************
* Main function merely starts up multiple threads of Makepipe function	  *
**************************************************************************/
void main(int argc, char *argv[])


{

  int i, result;
  int iThreadCnt;

  if (argc < 2) iThreadCnt = 10;
  else		iThreadCnt = atoi(argv[1]);

  printf("\nEnter delay in mS for each MKPIPE2 thread\n");
  printf("to wait before servicing requests.  This will\n");
  printf("govern client and server CPU impact of this test.\n");
  printf("Suggest using 0 for highest testing rate.\n");
  printf("Suggest about 500 for lower CPU impact. ");
  scanf( "%ld", &mpdelay);


  _clearscreen(_GCLEARSCREEN);
  if (DosSetMaxFH(200) != 0)
    {
    printf("\nDosSetMaxFH failed, terminating process");
    DosExit(EXIT_PROCESS, 1);
    }

  for (i=1; i < (iThreadCnt+1); i++)
    {
    if (_beginthread((void (_far *)(void _far *))MakePipe,
                     NULL,
		     4096,
		     (unsigned long) i) < 0) {
      printf("*** Error *** Could not start thread %u\n", i);
      exit(0);
    }
  }
  DosSleep(2000000000L); /* Go to sleep after starting all threads */

  return;
}
