/*
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the
 * terms and conditions of the copyright.
 */

#include <signal.h>
#include <netdb.h>
#include "h/common.h"
#include "h/ip.h"
#include "h/tcp.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"
#include "h/sl.h"
#include "h/terminal.h"
#include <termios.h>

int ttyfd;
int towrite = 0;
struct timeval tt;
int bytesps = 960;
int baud = 9600;
struct ex_list *exec_list = NULL;
struct ex_list *ptyexec_list = NULL;

char *exec_shell;

int do_slowtimo;
int to_write;

struct ifp ifp;

u_long our_addr;
u_long ctl_addr = 0;
u_long special_addr = -1;

void main_loop(void);
int main(int, char **);

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *doconfig = NULL;
	int doredir_x = 0;
	
	fprintf(stderr,"SLiRP %s\n\n",VERSION);
	
	fprintf(stderr,"This program is copyrighted, free software.\n");
	fprintf(stderr,"Please read the file COPYRIGHT that came with the SLiRP\n");
	fprintf(stderr,"package for the terms and conditions of the copyright.\n\n");
	
	/* Initialise everything */
	{
		int i;
	
		for (i = 255; i > 2; i--) {
			close(i);
		}
	}
	/* 
	 * The line below causes the login to become Idle for me,
	 * so we just use fd 0
	 */
/*	ttyfd = open("/dev/tty",O_RDWR,0); */
	ttyfd = 0;
	
	/* Initialise everything */
	so_init();
	sl_init();
	ip_init();
	m_init();

	{
		/* Read .slirprc */
		char buff[256];
		char *ptr;
		 
		if ((ptr = (char *)getenv("HOME")) != NULL) {
			strcpy(buff, ptr);
			strcat(buff, "/.slirprc");
			config(buff);
		}
	}
	
#define ARGINC argv++; argc--; if (argc == 0) goto badopt;

	/* Parse options */
	/* XXX A few more checks wouldn't hurt */
	while(--argc > 0) {
		argv++;
		if (**argv != '-') {
			(*argv)++;
			argc = 0;
			argv++;
			goto badopt;
		}
		(*argv)++;
		if (*(*argv+1) != 0) {
			argc = 0;
			argv++;
			goto badopt;
		}
		switch(**argv) {
		 case 'h':
			fprintf(stderr,"Usage: slirp [-<option> [arg] [-<option> [arg] ...]]\n");
			fprintf(stderr,"Where <option> is one of:\n");
			fprintf(stderr,"    -h           Show usage\n");
			fprintf(stderr,"    -b BAUDRATE  Set the baudrate (default 9600)\n");
			fprintf(stderr,"    -c           Do compression (CSLIP)\n");
			fprintf(stderr,"    -C IP        Only allow packets from IP to use CTL.\n");
			fprintf(stderr,"    -d FILE      Write debugging info to FILE\n");
			fprintf(stderr,"    -e CMD:PORT  Exec CMD (inetd style) on connection to PORT\n");
			fprintf(stderr,"    -f FILE      Read FILE as a configuration file\n");
			fprintf(stderr,"    -m MTU       Set the MTU of the link (default 552)\n");
			fprintf(stderr,"    -p CMD:PORT  Exec CMD (within a pty) on connection to PORT\n");
			fprintf(stderr,"    -s           Print statistics on exit\n");
			fprintf(stderr,"    -X           Redirect a port for X Window System\n");
			exit(0);
		 case 'b':
			ARGINC;
			if ((baud = atoi(*argv)) == 0)
			   baud = 9600;
			break;
		 case 'c':
			ifp.ifp_flags = IF_COMPRESS;
			break;
		 case 'C':
			ARGINC;
			if ((ctl_addr = inet_addr(*argv)) == -1) {
				ctl_addr = 0;
				goto badopt;
			}
			break;
		 case 'd':
			ARGINC;
			debug_init(*argv, 0xf);
			break;
		 case 'e':
			ARGINC;
			{
				char str[256];
				int prt;
				
				if (sscanf(*argv, "%s:%d", str, prt) != 0) {
					fprintf(stderr,"Error: Bad command/port: %s\n", *argv);
					goto badopt;
				}
				if (add_exec(&exec_list, str, ntohs(prt)))
				   fprintf(stderr,"Error: Port allready used\n");
			}
			break;
		 case 'f':
			ARGINC;
			doconfig = *argv;
			break;
		 case 'm':
			ARGINC;
			if ((ifp.ifp_mtu = atoi(*argv)) == 0)
			   goto badopt;
			if (ifp.ifp_mtu < (40+32) || ifp.ifp_mtu > 2048) {
				fprintf(stderr,"Error: Bad MTU, reseting to 552\n");
				ifp.ifp_mtu = 552;
			}
			break;
		 case 'p':
			ARGINC;
			{
				char str[256];
				int prt;
				
				if (sscanf(*argv, "%s:%d", str, prt) != 0)  {
					fprintf(stderr,"Error: Bad command/port: %s\n", *argv);
					goto badopt;
				}
				if (add_exec(&ptyexec_list, str, prt))
				   fprintf(stderr,"Error: Port allready used\n");
			}
			break;
		 case 's':
			dostats = 1;
			break;
		 case 'x':
			ARGINC;
			if ((special_addr = inet_addr(*argv)) == -1) {
				goto badopt;
			}
			break;
		 case 'X':
			doredir_x = 1;
			break;
		 default:
			argc = 0;
			argv++;
badopt:
			(*(argv-1))--;
			fprintf(stderr,"Error: Bad option: ");
			if (argc)
			   fprintf(stderr,"%s %s\n\n", *(argv-1), *argv);
			else
			   fprintf(stderr,"%s\n\n", *(argv-1));
		}
			
	}
	
	
	/* Check wheather to read a given file */
	if (doconfig)
	   config(doconfig);
	
	if (special_addr == -1)
	   special_addr = inet_addr("192.0.2.0");
	
	if (our_addr == 0)
	   getouraddr();
	
	if (our_addr == 0) {
		fprintf(stderr,"Error:  SLiRP Could not determine the address of the this host.\n");
		fprintf(stderr,"        Some programs may not work without knowing this address.\n");
		fprintf(stderr,"        It is recommended you use the \"host address aaa.bbb.ccc.ddd\n\"");
		fprintf(stderr,"        option in your ~/.slirprc config file (where aaa.bbb.ccc.ddd");
		fprintf(stderr,"        is the IP address of the host SLiRP is running on).\n\n");
	} else {
		struct in_addr inaddr;
		
		inaddr.s_addr = our_addr;
		fprintf(stderr,"IP address of SLiRP host: %s\n", inet_ntoa(inaddr));
	}
	
	fprintf(stderr,"You address is 192.0.2.15\n");
	fprintf(stderr,"(or anything else you want)\n\n");
	
	/* Setup ptyexec_list */
	if (exec_shell) {
		add_exec(&ptyexec_list, exec_shell, htons(23));
		free(exec_shell);
		exec_shell = 0;
	} else
		add_exec(&ptyexec_list, "/bin/sh", htons(23));
	
	/* Setup X redirection, if desired */
	if (doredir_x)
		redir_x();
	
	fprintf(stderr,"SLiRP Ready ... (");
	switch(ifp.ifp_flags) {
	 case IF_COMPRESS:
		fprintf(stderr,"talking CSLIP");
		break;
	 case IF_AUTOCOMP:
		fprintf(stderr,"autodetect SLIP/CSLIP");
		break;
	 case IF_NOCOMPRESS:
		fprintf(stderr,"talking SLIP");
		break;
	}
	fprintf(stderr,", MTU %d, %d baud)\n", ifp.ifp_mtu, baud);
	
	updtime();
	
	/* Init a few things XXX */
	last_slowtimo = curtime;
	time_fasttimo = 0;
	lastime = curtime;
	towrite = bytesps;
	
	term_save(ttyfd);
	
	/*
	 * The minimum packet length for compressed TCP packets
	 * is 3 bytes, 40 bytes for an uncompressed TCP packet.
	 */
	if (ifp.ifp_flags == IF_COMPRESS) {
		if (term_raw(ttyfd, 3, 2, baud) < 0) {
			fprintf(stderr,"ERROR: term_raw: %d\n", strerror(errno));
			exit(1);
		}
	} else {
		if (term_raw(ttyfd, 40, 2, baud) < 0) {
			fprintf(stderr,"ERROR: term_raw: %d\n", strerror(errno));
			exit(1);
		}
	}
	
	/*
	 * Set bytes we can write per second
	 * XXX This is wrong, but for some reason it works the best for me
	 * (should be something like baud / (8 + stop bits) or similar,
	 *  but this seems to write to the modem too fast)
	 */
	bytesps = baud / 10;

	/* Main_loop init */
	signal(SIGCHLD, SIG_IGN);
	signal(SIGHUP, slirp_exit);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT, slirp_exit);
	signal(SIGQUIT, slirp_exit);
	signal(SIGTERM, slirp_exit);
	signal(SIGBUS, SIG_IGN);
	
	main_loop();
	
	/* NOTREACHED */
	term_restore(ttyfd);
}

#define CONN_CANFSEND(so) (((so)->so_state & (SS_FCANTSENDMORE|SS_ISFCONNECTED)) == SS_ISFCONNECTED)
#define CONN_CANFRCV(so) (((so)->so_state & (SS_FCANTRCVMORE|SS_ISFCONNECTED)) == SS_ISFCONNECTED)
#define UPD_NFDS(x) if (nfds < (x)) nfds = (x)

fd_set writefds, readfds, xfds;

void
main_loop(void) 
{
	struct socket *so, *so_next;
	struct timeval timeout;
	int ret, nfds;

while(1) {

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&xfds);
	
	/*
	 * Allways set modem for reading
	 */
	FD_SET(ttyfd, &readfds);
	nfds = ttyfd;
	
	/* 
	 * *_slowtimo needs calling if there are IP fragments
	 * in the fragment queue, or there are TCP connections active
	 */
	do_slowtimo = ((tcb.so_next != &tcb) ||
		       ((struct ipasfrag *)&ipq != (struct ipasfrag *)ipq.next));
	/*
	 * First, TCP sockets
	 * XXX Should somehow limit the max number of packets queued
	 * for each session.  This is done by the TCP window, but with a low timeout
	 * value, error-prone links hence lots of retransmits, and some greedy UDP sessions
	 * can make TCP queue too many packets
	 */
	for (so = tcb.so_next; so != &tcb; so = so_next) {
		so_next = so->so_next;
		
		/*
		 * See if we need a tcp_fasttimo (the above sockets never will)
		 */
		if (time_fasttimo == 0 && so->so_tcpcb->t_flags & TF_DELACK)
			time_fasttimo = curtime; /* Flag when we want a fasttimo */
		
		/*
		 * NOFDREF can include still connecting to local-host,
		 * newly socreated() sockets etc. Don't want to select these.
		 */
		if (so->so_state & SS_NOFDREF)
			continue;
		
                /*
                 * Set for reading sockets which are accepting
                 */
		if (so->so_state & SS_FACCEPTCONN) {
			FD_SET(so->s,&readfds);
			UPD_NFDS(so->s);
			continue;
		}
		
		/*
		 * Set for writing sockets which are connecting
		 */
		if (so->so_state & SS_ISFCONNECTING) {
			FD_SET(so->s,&writefds);
			UPD_NFDS(so->s);
			continue;
		}
		
		/*
		 * Set for writing if we are connected, can send more, and
		 * we have something to send
		 */
		if (CONN_CANFSEND(so) && so->so_rcv.sb_cc) {
			FD_SET(so->s, &writefds);
			UPD_NFDS(so->s);
		}

		/*
		 * Set for reading (and urgent data) if we are connected, can
		 * receive more, and we have room for it XXX /2 ?
		 */
		if (CONN_CANFRCV(so) && (so->so_snd.sb_cc < (so->so_snd.sb_datalen/2))) {
			FD_SET(so->s, &readfds);
			FD_SET(so->s, &xfds);
			UPD_NFDS(so->s);
		}
	}

	/*
	 * UDP sockets
	 */
	for (so = udb.so_next; so != &udb; so = so_next) {
		so_next = so->so_next;

		/*
		 * See if it's timed out
		 */
		if (so->so_expire) {
			if (so->so_expire <= curtime) {
				udp_detach(so);
				continue;
			} else
				do_slowtimo = 1; /* Let socket expire */
			
		}
		
		/*
		 * When UDP packets are received from over the
		 * link, they're sendto()'d straight away, so
		 * no need for seting for writing
		 * Limit the number of packets queued by this session
		 * to 4.  Note that even though we try and limit this
		 * to 4 packets, the session could have more queued
		 * if the packets needed to be fragmented
		 * XXX <= 4 ???
		 */
		if ((so->so_state & SS_ISFCONNECTED) && so->so_queued <= 4) {
			FD_SET(so->s, &readfds);
			UPD_NFDS(so->s);
		}
	}

	/*
	 * Setup timout to use minimum CPU usage, especially when idle
	 */
	updtime();
	
	/* 
	 * First, see the timeout needed by *timo
	 * XXX this is ugly...
	 */
	timeout.tv_sec = 0;
	timeout.tv_usec = -1;
	/*
	 * If a slowtimo is needed, set timeout to 500ms from the last
	 * slow timeout. If a fast timeout is needed, set timeout within
	 * 200ms of when it was requested.
	 */
	if (do_slowtimo) {
		/* XXX */
		timeout.tv_usec = ((500 - (curtime - last_slowtimo)) * 1000) + 10000;
		if (timeout.tv_usec < 0)
			timeout.tv_usec = 0;
		
		/* Can only fasttimo if we also slowtimo */
		if (time_fasttimo)  {
			tmp_time = (200 - (curtime - time_fasttimo)) * 1000;
			if (tmp_time < 0)
				tmp_time = 0;
			
			/* Choose the smallest of the 2 */
			if (tmp_time < timeout.tv_usec)
				timeout.tv_usec = tmp_time;
		}
	}
	
	/*
	 * If there's something queued, timeout instantly if we can write
	 * to the modem, otherwise timeout when we can write to the modem
	 */
/*	if ((if_fastq.ifq_next == &if_fastq) && (if_batchq.ifq_next == &if_batchq)) */
	if (if_queued == 0)
		tmp_time = -1; /* Flag to block forever */
	else  {
		if (towrite >= 0)
			tmp_time = 0;
		else {
			tmp_time = (((-towrite + 1) * 1000000) / bytesps);
			
			if (tmp_time > 500000)
				tmp_time = 500000;
			else if (tmp_time < 100000)
				tmp_time = 100000;
		}
	}
	
	/*
	 * Take the minimum of the above calculated timeouts
	 */
	if ((timeout.tv_usec < 0) || (tmp_time > 0 && tmp_time < timeout.tv_usec))
		timeout.tv_usec = tmp_time;
	
#ifdef DEBUG
	debug_misc(dfd," towrite = %d, timeout.tv_usec = %u",
		towrite, timeout.tv_usec);
	if (time_fasttimo) debug_misc(dfd,", need fasttimo");
	fflush_misc(dfd);
#endif
	
	/*
	 * Do the real select call
	 */
/*	do { */
		if (timeout.tv_usec == -1)
        	        ret = select(nfds+1, &readfds, &writefds, &xfds, (struct timeval *)NULL);
		else
			ret = select(nfds+1, &readfds, &writefds, &xfds, &timeout);
/*	} while (ret < 0 && errno == EAGAIN); */ /* XXX Needed? */
	
	/* See if we err'ed XXX Shouldn't happen */
/*	if (ret < 0) */
	if (ret < 0 && errno != EAGAIN)
	   slirp_exit(1);
	
	/* Update time */
	updtime();
	
#ifdef DEBUG
	debug_misc(dfd," towrite = %d\n", towrite);
#endif
	
	/*
	 * See if anything has timed out 
	 */
	
	if (time_fasttimo && ((curtime - time_fasttimo) >= 199)) {
		tcp_fasttimo();
		time_fasttimo = 0;
	}
	
	if (do_slowtimo && ((curtime - last_slowtimo) >= 499)) {
		ip_slowtimo();
		tcp_slowtimo();
		last_slowtimo = curtime;
	}
	
	/* 
	 * Check if the modem's ready
	 */
	if (FD_ISSET(ttyfd,&readfds))
		(*(ifp.ifp_input))();

	/*
	 * Check TCP sockets
	 */
	for (so = tcb.so_next; so != &tcb; so = so_next) {
		so_next = so->so_next;
		
		/*
		 * FD_ISSET is meaningless on these sockets
		 * (and they can crash the program)
		 */
		if (so->so_state & SS_NOFDREF)
			continue;
		
		/*
		 * Check for URG data
		 * This will soread as well, so no need to
		 * test for readfds below if this succeeds
		 */
		if (FD_ISSET(so->s, &xfds))
			sorecvoob(so);
		/*
		 * Check sockets for reading
		 */
		else if (FD_ISSET(so->s, &readfds)) {
			/*
			 * Check for incoming connections
			 */
			if (so->so_state & SS_FACCEPTCONN) {
				tcp_connect(so);
				continue;
			} /* else */
			ret = soread(so);
			
			/* Output it if we read something */
			if (ret > 0)
			   tcp_output(sototcpcb(so));
		}

		/*
		 * Check sockets for writing
		 */
		if (FD_ISSET(so->s,&writefds)) {
			/*
			 * Check for non-blocking, still-connecting sockets
			 */
			if (so->so_state & SS_ISFCONNECTING) {
				/* Connected */
				so->so_state &= ~SS_ISFCONNECTING;

				ret = write(so->s, &ret, 0);
				if (ret < 0) {
					if ( errno == EAGAIN)
						continue;
					/* else failed */
					so->so_state = SS_NOFDREF;
				} else
					so->so_state &= ~SS_ISFCONNECTING;

				/*
				 * Continue tcp_input
				 */
				tcp_input((struct mbuf *)NULL, sizeof(struct ip), so);
				/* continue; */
			} else
				ret = sowrite(so);
				/*
				 * XXX If we wrote something (a lot), there 
				 * could be a need for a window update.
				 * In the worst case, the remote will send
				 * a window probe to get things going again
				 */
		}
		
		/*
		 * Check if the non-blocking, still-conncting socket
		 * is still OK
		 */
		if (so->so_state & SS_ISFCONNECTING) {
			ret = read(so->s, (char *)&ret, 0);

			if (ret < 0) {
				if (errno == EAGAIN)
				   continue; /* Still connecting, continue */
				
				/* else failed */
				so->so_state = SS_NOFDREF;
				
				/* tcp_input will take care of it */
				tcp_input((struct mbuf *)NULL, sizeof(struct ip), so);
			} else {
                                ret = write(so->s, &ret, 0);
                                if (ret < 0) { 
                                        if ( errno == EAGAIN)   
                                                continue;
                                        /* else failed */
                                        so->so_state = SS_NOFDREF;
                                } else
                                        so->so_state &= ~SS_ISFCONNECTING;

				tcp_input((struct mbuf *)NULL, sizeof(struct ip), so);
			}
		}
	}

	/*
	 * Now UDP sockets.
	 * Incoming packets are sent straight away, they're not buffered.
	 * Incoming UDP data isn't buffered either.
	 */
	for (so = udb.so_next; so != &udb; so = so_next) {
		so_next = so->so_next;
		
		if (FD_ISSET(so->s,&readfds))
			sorecvfrom(so);
	}

        /*
	 * See if we can start outputing
         */
	if (((if_fastq.ifq_next != &if_fastq) || (if_batchq.ifq_next != &if_batchq))
	    && (towrite >= 0))
		(*(ifp.ifp_start))();

} /* while(1) { */
}
