
{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit Report;

interface

uses SysUtils, WinTypes, WinProcs, Classes, Controls, Forms,
  DDEMan, Dsgnintf;

const
  ctDBase = 2;
  ctExcel = 3;
  ctParadox = 4;
  ctAscii = 5;
  ctSqlServer = 6;
  ctOracle = 7;
  ctDB2 = 8;
  ctNetSQL = 9;
  ctSybase = 10;
  ctBtrieve = 11;
  ctGupta = 12;
  ctIngres = 13;
  ctWatcom = 14;
  ctOcelot = 15;
  ctTeraData = 16;
  ctDB2Gupta = 17;
  ctAS400 = 18;
  ctUnify = 19;
  ctQry = 20;
  ctMinNative = 2;
  ctMaxNative = 20;
  ctODBCDBase = 40;
  ctODBCExcel = 41;
  ctODBCParadox = 42;
  ctODBCSqlServer = 43;
  ctODBCOracle = 44;
  ctODBCDB2 = 45;
  ctODBCNetSql = 46;
  ctODBCSybase = 47;
  ctODBCBtrieve = 48;
  ctODBCGupta = 49;
  ctODBCIngres = 50;
  ctODBCDB2Gupta = 51;
  ctODBCTeraData = 52;
  ctODBCAS400 = 53;
  ctODBCDWatcom = 54;
  ctODBCDefault = 55;
  ctODBCUnify = 56;
  ctMinODBC = 40;
  ctMaxODBC = 56;
  ctIDAPIStandard = 60;
  ctIDAPIParadox = 61;
  ctIDAPIDBase = 62;
  ctIDAPIAscii = 63;
  ctIDAPIOracle = 64;
  ctIDAPISybase = 65;
  ctIDAPINovSql = 66;
  ctIDAPIInterbase = 67;
  ctIDAPIIBMEE = 68;
  ctIDAPIDB2 = 69;
  ctIDAPIInformix = 70;
  ctMinIDAPI = 60;
  ctMaxIDAPI = 70;

type
  EReportError = class(Exception);

  TReport = class(TComponent)
  private
    FDdeClient: TDdeClientConv;
    FReportName: PString;
    FReportDir: PString;
    FNumCopies: Word;
    FStartPage: Word;
    FEndPage: Word;
    FMaxRecords: Word;
    FRunTime: Boolean;
    FStartedApp: Boolean;
    FAutoUnload: Boolean;
    FInitialValues: TStrings;
    FLoaded: Boolean;
    FProcessing: Boolean;
    FVersionMajor: Integer;
    FVersionMinor: Integer;
    FReportHandle: HWND;
    FBusy: Boolean;
    FPreview: Boolean;
    function GetInitialValues: TStrings;
    function GetReportHandle: HWND;
    function GetReportDir: string;
    function GetReportName: string;
    procedure RunApp;
    procedure SetReportDir(Value: string);
    procedure SetReportName(Value: string);
    function StartApplication: Boolean;
    function ReportActive: Boolean;
    procedure RunReport;
    procedure SetInitialValues(Value: TStrings);
    function UseRunTime: Boolean;
    function Wait: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CloseApplication(ShowDialogs: Boolean): Boolean;
    function CloseReport(ShowDialogs: Boolean): Boolean;
    function Connect(ServerType: Word; const ServerName,
      UserName, Password, DatabaseName:string): Boolean;
    function Print: Boolean;
    function RecalcReport: Boolean;
    procedure Run;
    function RunMacro(Macro: PChar): Boolean;
    function SetVariable(const Name, Value: string): Boolean;
    function SetVariableLines(const Name: string; Value: TStrings): Boolean;
    property ReportHandle: HWND read FReportHandle;
    property Busy: Boolean read FBusy;
    property VersionMajor: Integer read FVersionMajor;
    property VersionMinor: Integer read FVersionMinor;
  published
    property ReportName: string read GetReportName write SetReportName;
    property ReportDir: string read GetReportDir write SetReportDir;
    property PrintCopies: Word read FNumCopies write FNumCopies default 1;
    property StartPage: Word read FStartPage write FStartPage default 1;
    property EndPage: Word read FEndPage write FEndPage default 9999;
    property MaxRecords: Word read FMaxRecords write FMaxRecords default 0;
    property AutoUnload: Boolean read FAutoUnload write FAutoUnload default False;
    property InitialValues: TStrings read GetInitialValues write SetInitialValues;
    property Preview: Boolean read FPreview write FPreview default False;
  end;

  TReportEditor = class(TComponentEditor)
  private
    procedure Edit; override;
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

  TReportDirProperty = class(TPropertyEditor)
  public
    function GetValue: string; override;
    procedure SetValue(const Value: string); override;
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

  TReportNameProperty = class(TPropertyEditor)
  public
    function GetValue: string; override;
    procedure SetValue(const Value: string); override;
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

implementation

uses DBConsts, FileCtrl, Dialogs, IniFiles, LibHelp;

const
  DesignName = 'ReportSmith';
  RunName = 'RS_RUNTIME';
  TopicName = 'Command';
  ReportClassName = 'Swap:Main';
  DesignExeName = 'RptSmith';
  RunExeName = 'RS_Run';

procedure RaiseError(const Message: string);
begin
  raise EReportError.Create(Message);
end;

constructor TReport.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDdeClient := TDdeClientConv.Create(nil);
  FDdeClient.ConnectMode := ddeAutomatic;
  PrintCopies := 1;
  StartPage := 1;
  EndPage := 9999;
  MaxRecords := 0;
  FInitialValues := TStringList.Create;
  FReportDir := NullStr;
  FReportName := NullStr;
end;

destructor TReport.Destroy;
begin
  if FRunTime and FStartedApp then CloseApplication(True);
  FDdeClient.Free;
  FInitialValues.Free;
  DisposeStr(FReportName);
  DisposeStr(FReportDir);
  inherited Destroy;
end;

procedure TReport.SetInitialValues(Value: TStrings);
begin
  FInitialValues.Assign(Value);
end;

function TReport.GetReportName: string;
begin
  Result := FReportName^;
end;

procedure TReport.SetReportName(Value: string);
begin
  AssignStr(FReportName, Value);
end;

function TReport.GetReportDir: string;
begin
  Result := FReportDir^;
end;

procedure TReport.SetReportDir(Value: string);
begin
  AssignStr(FReportDir, Value);
end;

function TReport.GetInitialValues: TStrings;
begin
  Result := FInitialValues;
end;

function TReport.SetVariable(const Name, Value: string): Boolean;
var
  Temp: array[0..255] of char;
begin
  Result := RunMacro(StrPCopy(Temp, FmtLoadStr(SSetVar, [Name, Value])));
end;

function TReport.SetVariableLines(const Name: string; Value: TStrings): Boolean;
var
  Buffer, StrEnd: PChar;
  BufLen: Word;
  I, L, Count: Integer;
  SetFmt: string;
  Temp: array[0..255] of char;
begin
  SetFmt := FmtLoadStr(SSetVarLines, [Name]);
  BufLen := Length(SetFmt) + 2;
  for I := 0 to Value.Count - 1 do
  begin
    L := Ord(Value[I][0]) + 1;
    if L > 65520 - BufLen then Break;
    Inc(BufLen, L);
  end;
  Buffer := AllocMem(BufLen);
  try
    StrPCopy(Buffer, SetFmt);
    StrEnd := Buffer + Length(SetFmt);
    Count := Value.Count - 1;
    for I := 0 to Count do
    begin
      StrPCopy(Temp, Value[I]);
      StrEnd := StrECopy(StrEnd, Temp);
      if I <> Count then StrEnd := StrECopy(StrEnd, ' ');
    end;
    Buffer[StrLen(Buffer)] := '"';
    RunMacro(Buffer);
  finally
    FreeMem(Buffer, BufLen);
  end;
end;

function TReport.RecalcReport: Boolean;
var
  Temp: array[0..255] of char;
begin
  Result := RunMacro(StrPCopy(Temp, LoadStr(SRecalc)));
end;

function TReport.ReportActive: Boolean;
begin
  Result := (ReportHandle <> 0) and (FDdeClient.Conv <> 0);
end;

function TReport.UseRunTime: Boolean;
begin
  Result := not (csDesigning in ComponentState);
end;

function TReport.Print: Boolean;
var
  Temp: array[0..255] of char;
begin
  Result := RunMacro(StrPCopy(Temp, FmtLoadStr(SPrintReport, [FStartPage, FEndPage, PrintCopies])));
end;

function TReport.StartApplication: Boolean;
var
  TempExeName: array[0..255] of char;
  ShowMode: Word;
  IniFile: TIniFile;
  IniFileName: string;
  ExeName: string;
  Section: string;
  ExePath: string;
begin
  if FRunTime then
  begin
    ExeName := RunExeName;
    IniFileName := LoadStr(SRptRunTimeIniFile);
    Section := LoadStr(SRptRunTimeSection);
  end 
  else begin
    ExeName := DesignExeName;
    IniFileName := LoadStr(SRptDesignTimeIniFile);
    Section := LoadStr(SRptDesignTimeSection);
  end;
  IniFile := TIniFile.Create(IniFileName);
  try
    ExePath := IniFile.ReadString(Section, LoadStr(SRptPath), '');
    if (ExePath <> '') and (ExePath[Length(ExePath)] <> '\') then
      ExePath := ExePath + '\';
    ExeName := ExePath + ExeName;
    StrPCopy(TempExeName, ExeName);
    if Preview then ShowMode := SW_SHOWNORMAL
    else ShowMode := SW_SHOWMINNOACTIVE;
    Result := WinExec(@TempExeName, ShowMode) >= 32;
    FStartedApp := Result;
  finally
    IniFile.Free;
  end;
end;

function TReport.CloseReport(ShowDialogs: Boolean): Boolean;
var
  Temp: array[0..255] of char;
begin
  if ReportActive then
    Result := RunMacro(StrPCopy(Temp, FmtLoadStr(SCloseReport, [Ord(ShowDialogs)])))
  else Result := True;
end;

function TReport.Connect(ServerType: Word; const ServerName,
  UserName, Password, DatabaseName: string): Boolean;
var
  Temp: array[0..255] of char;
begin
  if ((ServerType >= ctMinNative) and (ServerType <= ctMaxNative)) or
    ((ServerType >= ctMinODBC) and (ServerType <= ctMaxODBC)) or
    ((ServerType >= ctMinIDAPI) and (ServerType <= ctMaxIDAPI)) then
    Result := RunMacro(StrPCopy(Temp, FmtLoadStr(SConnect, [ServerType, ServerName,
      UserName, Password, DatabaseName])))
  else RaiseError(LoadStr(SInvalidServer));
end;

function TReport.CloseApplication(ShowDialogs: Boolean): Boolean;
var
  Temp: array[0..255] of char;
begin
  if ReportActive then
  begin
    Result := RunMacro(StrPCopy(Temp, FmtLoadStr(SCloseApp, [Ord(ShowDialogs)])));
    if Result then
    begin
      FStartedApp := False;
      FReportHandle := 0;
    end;
  end
  else Result := True;
end;

function TReport.GetReportHandle: HWND;
var
  TempWinName: array[0..255] of char;
  TempAppName: array[0..255] of char;
begin
  if FRunTime then StrCopy(TempAppName, RunName)
  else StrPCopy(TempAppName, DesignName);
  StrPCopy(TempWinName, ReportClassName);
  Result := FindWindow(@TempWinName, @TempAppName);
end;

function TReport.RunMacro(Macro: PChar): Boolean;
var
  Msg: TMsg;
begin
  Result := False;
  if (Macro = nil) or Busy then Exit;
  FBusy := True;
  try
    RunApp;
    if (ReportHandle <> 0) and (not FProcessing) then
    begin
      repeat
        if PeekMessage(Msg, 0, 0, 0, PM_NOREMOVE) then
        begin
          with Application do
          begin
            HandleMessage;
            if Terminated then Exit;
          end;
        end;
        FProcessing := not FDdeClient.ExecuteMacro(Macro, True);
      until not FProcessing;
      Result := True;
    end;
  finally
    if not FProcessing then FBusy := False;
  end;
end;

function TReport.Wait: Boolean;
begin
  repeat
    Application.HandleMessage;
    if ReportHandle = 0 then FReportHandle := GetReportHandle;
  until Application.Terminated or (ReportHandle <> 0);
  Result := ReportHandle <> 0;
end;

procedure TReport.RunApp;
var
  AppName: string;
  I: Integer;
  Data: PChar;
  Version: Word;
begin
  if not ReportActive then
  begin
    FRunTime := UseRunTime;
    FReportHandle := GetReportHandle;
    if ReportHandle = 0 then
    begin
      if not StartApplication then
      begin
        if FRunTime then raise Exception.Create(LoadStr(SRunLoadFailed))
        else raise Exception.Create(LoadStr(SDesignLoadFailed));
      end;
      if not Wait then Exit;
    end;
    if FRunTime then AppName := RunName
    else AppName := DesignName;
    FDdeClient.SetLink(AppName, 'System');
    Data := FDdeClient.RequestData('ByteVersion');
    if Data <> nil then
      try
        Version := Word(Pointer(Data)^);
        FVersionMajor := HiByte(Version);
        FVersionMinor := LoByte(Version);
      finally
        StrDispose(Data);
      end;
    if VersionMajor = 0 then
    begin
      if FStartedApp then CloseApplication(False);
      raise Exception.Create(LoadStr(SIncorrectVersion));
    end;
    FDdeClient.SetLink(AppName, TopicName);
  end;
end;

procedure TReport.Run;
begin
  RunReport;
  if not Busy and FRunTime and FStartedApp and
    AutoUnload and not Preview then CloseApplication(True);
end;

procedure TReport.RunReport;
var
  Path, FileName: string;
  Temp: array[0..255] of char;
  Buffer, StrEnd: PChar;
  BufLen: Word;
  I, L, Count: Integer;
begin
  if RunMacro(StrPCopy(Temp, FmtLoadStr(SRecordLimit, [FMaxRecords]))) then
  begin
    Path := ReportDir;
    if (Path <> EmptyStr) and (Path[Length(Path)] <> '\') then
      Path := Path + '\';
    FileName := ReportName;
    if (FileName <> '') and (Pos('.', FileName) = 0) then
      FileName := FileName + '.rpt';
    if FileName <> '' then
    begin
      FileName := FmtLoadStr(SLoadReport, [Path, FileName]);
      BufLen := Length(FileName) + 2;
      for I := 0 to FInitialValues.Count - 1 do
      begin
        L := Ord(FInitialValues[I][0]) + 2;
        if L > 65520 - BufLen then Break;
        Inc(BufLen, L);
      end;
      Buffer := AllocMem(BufLen);
      try
        StrPCopy(Buffer, FileName);
        StrEnd := Buffer + Length(FileName);
        Count := FInitialValues.Count - 1;
        for I := 0 to Count do
        begin
          StrPCopy(Temp, FInitialValues[I]);
          StrEnd := StrECopy(StrEnd, Temp);
          if (I <> Count) and (Pos('>', FInitialValues[I]) > 0) then
            StrEnd := StrECopy(StrEnd, ', ');
        end;
        Buffer[StrLen(Buffer)] := '"';
        RunMacro(Buffer);
      finally
        FreeMem(Buffer, BufLen);
      end;
      if FRunTime and not Preview then Print;
    end;
  end;
end;

{ TReportEditor }

procedure TReportEditor.Edit;
begin
  TReport(Component).Run;
end;

procedure TReportEditor.ExecuteVerb(Index: Integer);
begin
  if Index = 0 then Edit;
end;

function TReportEditor.GetVerb(Index: Integer): string;
begin
  Result := LoadStr(SReportVerb);
end;

function TReportEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;

{ TReportDirProperty }

function TReportDirProperty.GetValue: string;
begin
  Result := (GetComponent(0) as TReport).ReportDir;
end;

procedure TReportDirProperty.SetValue(const Value: string);
begin
  (GetComponent(0) as TReport).ReportDir := Value;
end;

function TReportDirProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paMultiSelect];
end;

procedure TReportDirProperty.Edit;
var
  FilePath: TFileName;
begin
  FilePath := '';
  if SelectDirectory(FilePath, [], hcDSelectReportDir) then
  begin
    if FilePath[Length(FilePath)] <> '\' then FilePath := FilePath + '\';
    (GetComponent(0) as TReport).ReportDir := LowerCase(FilePath);
  end;
end;

{ TReportNameProperty }

function TReportNameProperty.GetValue: string;
begin
  Result := (GetComponent(0) as TReport).ReportName;
end;

procedure TReportNameProperty.SetValue(const Value: string);
begin
  (GetComponent(0) as TReport).ReportName := Value;
end;

function TReportNameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paMultiSelect];
end;

procedure TReportNameProperty.Edit;
var
  Dialog: TOpenDialog;
  FilePath: string;
begin
  Dialog := TOpenDialog.Create(nil);
  try
    with Dialog do
    begin
      DefaultExt := 'rpt';
      Filter := LoadStr(SReportFilter);
      if Execute then
      begin
        with GetComponent(0) as TReport do
        begin
          FileName := LowerCase(FileName);
          FilePath := ExtractFilePath(FileName);
          if ReportDir = '' then
          begin
            ReportDir := FilePath;
            ReportName := ExtractFileName(FileName);
          end
          else if ReportDir = FilePath then
            ReportName := ExtractFileName(FileName)
          else ReportName := FileName;
        end;
      end;
    end;
  finally
    Dialog.Free;
  end;
end;

end.


