{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit IniFiles;

interface

uses WinTypes, Classes;

type
  TIniFile = class(TObject)
  private
    FFileName: PChar;
    function GetFileName: string;
  public
    constructor Create(const FileName: string);
    destructor Destroy; override;
    function ReadString(const Section, Ident, Default: string): string;
    procedure WriteString(const Section, Ident, Value: String);
    function ReadInteger(const Section, Ident: string;
      Default: Longint): Longint;
    procedure WriteInteger(const Section, Ident: string; Value: Longint);
    function ReadBool(const Section, Ident: string;
      Default: Boolean): Boolean;
    procedure WriteBool(const Section, Ident: string; Value: Boolean);
    procedure ReadSection(const Section: string; Strings: TStrings);
    procedure ReadSectionValues(const Section: string; Strings: TStrings);
    procedure EraseSection(const Section: string);
    property FileName: string read GetFileName;
  end;

implementation

uses WinProcs, SysUtils, Consts;

constructor TIniFile.Create(const FileName: string);
begin
  FFileName := StrPCopy(StrAlloc(Length(FileName) + 1), FileName);
end;

destructor TIniFile.Destroy;
begin
  StrDispose(FFileName);
end;

function TIniFile.GetFileName: string;
begin
  Result := StrPas(FFileName);
end;

function TIniFile.ReadString(const Section, Ident, Default: string): string;
var
  Len: Integer;
  CSection: array[0..127] of Char;
  CIdent: array[0..127] of Char;
  CDefault: array[0..255] of Char;
begin
  Len := GetPrivateProfileString(
    StrPLCopy(CSection, Section, SizeOf(CSection) - 1),
    StrPLCopy(CIdent, Ident, SizeOf(CIdent) - 1),
    StrPCopy(CDefault, Default),
    @Result, 256, FFileName);
  Move(Result[0], Result[1], Len);
  Result[0] := Char(Len);
end;

procedure TIniFile.WriteString(const Section, Ident, Value: string);
var
  CSection: array[0..127] of Char;
  CIdent: array[0..127] of Char;
  CValue: array[0..255] of Char;
begin
  if not WritePrivateProfileString(
    StrPLCopy(CSection, Section, SizeOf(CSection) - 1),
    StrPLCopy(CIdent, Ident, SizeOf(CIdent) - 1),
    StrPCopy(CValue, Value), FFileName) then
      raise Exception.Create(FmtLoadStr(SIniFileWriteError, [FileName]));
end;

function TIniFile.ReadInteger(const Section, Ident: string;
  Default: Longint): Longint;
var
  IStr: string;
begin
  IStr := ReadString(Section, Ident, '');
  if CompareText(Copy(IStr, 1, 2), '0x') = 0 then
    IStr := '$' + Copy(IStr, 3, 255);
  Result := StrToIntDef(IStr, Default);
end;

procedure TIniFile.WriteInteger(const Section, Ident: string; Value: Longint);
begin
  WriteString(Section, Ident, IntToStr(Value));
end;

function TIniFile.ReadBool(const Section, Ident: string;
  Default: Boolean): Boolean;
begin
  Result := ReadInteger(Section, Ident, Ord(Default)) <> 0;
end;

procedure TIniFile.WriteBool(const Section, Ident: string; Value: Boolean);
const
  Values: array[Boolean] of string[1] = ('0', '1');
begin
  WriteString(Section, Ident, Values[Value]);
end;

procedure TIniFile.ReadSection(const Section: string; Strings: TStrings);
const
  BufSize = 8192;
var
  Buffer, P: PChar;
  Count: Integer;
  CSection: array[0..127] of Char;
begin
  GetMem(Buffer, BufSize);
  try
    Count := GetPrivateProfileString(StrPLCopy(CSection, Section,
      SizeOf(CSection) - 1), nil, nil, Buffer, BufSize, FFileName);
    P := Buffer;
    if Count > 0 then
      while P[0] <> #0 do
      begin
        Strings.Add(StrPas(P));
        Inc(P, StrLen(P) + 1);
      end;
  finally
    FreeMem(Buffer, BufSize);
  end;
end;

procedure TIniFile.ReadSectionValues(const Section: string; Strings: TStrings);
var
  KeyList: TStringList;
  I: Integer;
begin
  KeyList := TStringList.Create;
  try
    ReadSection(Section, KeyList);
    for I := 0 to Keylist.Count - 1 do
      Strings.Values[KeyList[I]] := ReadString(Section, KeyList[I], '');
  finally
    KeyList.Free;
  end;
end;

procedure TIniFile.EraseSection(const Section: string);
var
  CSection: array[0..127] of Char;
begin
  if not WritePrivateProfileString(
    StrPLCopy(CSection, Section, SizeOf(CSection) - 1),
    nil, nil, FFileName) then
      raise Exception.Create(FmtLoadStr(SIniFileWriteError, [FileName]));
end;

end.
