
{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit MaskProp;

interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, StdCtrls,
  ExtCtrls, Buttons, DsgnIntf, Mask, SysUtils, Dialogs;

type
{ TMaskProperty
  Property editor the Masked Edit Mask property   }

  TMaskProperty = class(TStringProperty)
  public
    function  GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;


type
  TMaskForm = class(TForm)
    InputMask: TEdit;
    Label1: TLabel;
    ListBox1: TListBox;
    Label2: TLabel;
    Label3: TLabel;
    TestEdit: TMaskEdit;
    Label4: TLabel;
    Blanks: TEdit;
    Bevel1: TBevel;
    OKButton: TBitBtn;
    CancelButton: TBitBtn;
    HelpButton: TBitBtn;
    SaveMaskCheck: TCheckBox;
    Masks: TButton;
    OpenDialog1: TOpenDialog;
    procedure BlanksChange(Sender: TObject);
    procedure InputMaskChange(Sender: TObject);
    procedure ListDrawItem(Control: TWinControl;
          Index: Integer; Rect: TRect; State: TOwnerDrawState);
    procedure ListBoxSelect(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure MasksClick(Sender: TObject);
  private
    FInEditChange: Boolean;
    function AddBlanks(Value: String): String;
    procedure LoadMaskList(const FileName: string);
  protected
    function GetListMaskValue(Index: Integer): string;
    function GetMaskValue(const Value: string): string;
    procedure Loaded; override;
  public
  end;

var
  MaskForm: TMaskForm;

implementation

uses IniFiles, LibHelp, LibConst;

{$R *.DFM}

{ TMaskProperty }

function TMaskProperty.GetAttributes: TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] - [paSubProperties];
end;


procedure TMaskProperty.Edit;
var
  Frm : TMaskForm;
  Str : String;
begin
  Frm := TMaskForm.Create (Application);
  try
    Frm.InputMask.Text := GetValue;
    Frm.TestEdit.EditMask := Frm.InputMask.Text;
    Frm.OpenDialog1.InitialDir := PrivateDirectory;  

    if Frm.ShowModal = idOK then
    begin
      Str := Frm.InputMask.Text;
      if (Length (Str) <> 0) and (MaskGetFldSeparator (Str) < 0) then 
      begin
        Frm.BlanksChange (Frm);
        Str := Frm.InputMask.Text;
      end;
      if GetValue <> Str then
        SetValue (Str);
    end;
  finally
    Frm.Free;
  end;
end;

{-----------}

procedure TMaskForm.BlanksChange(Sender: TObject);
Var
  Str : String;
  Pos : Integer;
  BlankChar : Char;
  SaveChar : Char;
begin
  Str := InputMask.Text;
  if (Length (Str) <> 0) and not FInEditChange then
    InputMask.Text := AddBlanks (Str);
end;

function TMaskForm.AddBlanks(Value: String): String;
Var
  Pos : Integer;
  BlankChar : Char;
  SaveChar : Char;
begin
  Result := Value;
  if Length (Result) <> 0 then
  begin
    BlankChar := MaskGetMaskBlank (Value);
    SaveChar  := '1';
    if Length (Blanks.Text) > 0 then
      BlankChar := Blanks.Text[1];
    if not SaveMaskCheck.Checked then
      SaveChar := MaskNoSave;

    Pos := MaskGetFldSeparator (Result);
    if Pos > 0 then
      Result[0] := Char (Pos - 1);
    Result := Result + MaskFieldSeparator + SaveChar 
               + MaskFieldSeparator + BlankChar;
  end;
end;

procedure TMaskForm.InputMaskChange(Sender: TObject);
var
  BlankChar: string;
begin
  TestEdit.Text := EmptyStr;
  TestEdit.EditMask := InputMask.Text;
  if TestEdit.EditMask <> GetListMaskValue(ListBox1.ItemIndex) then
    ListBox1.ItemIndex := -1;
  FInEditChange := True;
  SaveMaskCheck.Checked := MaskGetMaskSave(TestEdit.EditMask);
  BlankChar := MaskGetMaskBlank (TestEdit.EditMask);
  Blanks.Text := Format ('%s', [BlankChar]);
  FInEditChange := False;
end;

procedure TMaskForm.ListDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  R: TRect;
  CString: array[0..80] of Char;
  Str, EditMask : String;
  X, Temp, StrPos : Integer;
begin
  if (Index >= ListBox1.Items.Count) then Exit;
  with ListBox1.Canvas do
  begin
    Str := ListBox1.Items[Index];
    StrPos := Pos ('|', Str);
    if StrPos = 0 then Exit;
    Temp := Integer (Str[0]);
    Str[0] := Char (StrPos - 1);
    StrPCopy(CString, Str);
    Str[0] := Char (Temp);

    R := Rect;
    X := R.Right;
    R.Right := R.Right div 2;
    ExtTextOut(Handle, R.Left + 1, R.Top + 1, ETO_CLIPPED or ETO_OPAQUE, @R,
      CString, StrPos - 1, nil);
    MoveTo(R.Right, R.Top);
    LineTo(R.Right, R.Bottom);

    Str := Copy(Str, StrPos + 2, Length (Str) - StrPos - 2);
    StrPos := Pos ('|', Str);
    if StrPos = 0 then Exit;
    Temp := Integer (Str[0]);
    Str[0] := Char (StrPos - 2);
    EditMask := GetListMaskValue(Index);
    EditMask[Length(EditMask) - 2] := '0';
    EditMask := FormatMaskText(EditMask, Str);
    StrPCopy(CString, EditMask);
    Str[0] := Char (Temp);

    R.Left := R.Right + 1;
    R.Right := X;
    ExtTextOut(Handle, R.Left + 1, R.Top + 1, ETO_CLIPPED or ETO_OPAQUE, @R,
      CString, Length(EditMask), nil);
  end;
end;

procedure TMaskForm.ListBoxSelect(Sender: TObject);
begin
  if ListBox1.ItemIndex >= 0 then
    InputMask.Text := GetListMaskValue(ListBox1.ItemIndex);
end;

function TMaskForm.GetListMaskValue(Index: Integer): string;
begin
  if Index >= 0 then Result := GetMaskValue(ListBox1.Items[Index])
  else Result := '';
end;

function TMaskForm.GetMaskValue(const Value: string): string;
var
  StrPos: Integer;
begin
  Result := '';
  StrPos := Pos ('|', Value);
  if StrPos = 0 then Exit;
  Result := Value;
  Result := Copy(Result, StrPos + 1, Length (Result) - StrPos);
  StrPos := Pos ('|', Result);
  if StrPos = 0 then 
  begin
    Result := '';
    Exit;
  end;
  if Result[StrPos+1] = ' ' then
    Inc (StrPos);
  Result := Copy(Result, StrPos + 1, Length (Result) - StrPos);
end;

procedure TMaskForm.FormCreate(Sender: TObject);
begin
  HelpContext := hcDInputMaskEditor;
end;

procedure TMaskForm.MasksClick(Sender: TObject);
var
  AppIniFile: TIniFile;
begin
  OpenDialog1.HelpContext := hcDOpenMaskFile;
  if OpenDialog1.Execute then
  begin
    LoadMaskList(OpenDialog1.FileName);
    try
      AppIniFile := TIniFile.Create(LoadStr(SIniFileName));
      AppIniFile.WriteString (LoadStr(SIniEditorsName), 
        LoadStr(SINIEditMaskName), OpenDialog1.FileName);
    finally
      AppIniFile.Free;
    end;
  end;
end;

procedure TMaskForm.Loaded;
var
  AppIniFile: TIniFile;
  Value: String;
begin
  inherited Loaded;
  try
    AppIniFile := TIniFile.Create(LoadStr(SIniFileName));
    Value := AppIniFile.ReadString (LoadStr(SIniEditorsName), 
      LoadStr(SIniEditMaskName), '');
    LoadMaskList(Value);
  finally
    AppIniFile.Free;
  end;
end;

procedure TMaskForm.LoadMaskList(const FileName: string);
var
  I: Integer;
  Items: TStringList;
begin
  if (Length(FileName) > 0) and FileExists(FileName) then
  begin
    try
      Items := TStringList.Create;
      Items.LoadFromFile(FileName);
      I := 0;
      while I < (Items.Count - 1) do
      begin
        if (Length(Items[I]) = 0) then Items.Delete(I)
        else if (GetMaskValue(Items[I]) = '') then
          raise EInvalidOperation.Create(LoadStr(SInvalidSampleMask))
        else Inc(I);
        if I >= Items.Count then break;
      end;
      ListBox1.Items := Items;
    finally
      Items.Free;
    end;
  end;
end;

end.

