#include "ns.h"

/*
 * This code implements a simple extension to the NaviServer. For each
 * connection, it logs any Referer data that comes in in the header data. The
 * file is written such that it can be copied directly into the database
 * periodically, i.e., the columns are separated by tabs.
 * 
 * The [LogRefer] section of the nsd.ini file can contain two keys:
 * 
 * File=<name of log file>
 * Db=<0 or 1, indicating logging should be done to the database>.
 * 
 * Note that logging to the database will slow the NaviServer down.  This
 * may or may not be noticable depending on your connection rate.
 * 
 * To use this module, compile and link it with the included Makefile and add
 * the following entry to the [Modules] section of the nsd.ini file:
 * 
 * Load=referlog.so(LogReferInit)
 * 
 * Possible extensions: Log more information (e.g., authuser, client IP
 * address), perhaps letting NaviServer administrators pick the fields.
 *
 */


static void     LogRefer(Ns_OpContext context, Ns_Conn *conn);
static void     CloseReferLog(Ns_OpContext context);
static int      logindb;


/*
 * LogReferInit - The module initalization routine.  The log file is openend
 * and/or the database table created for refer logging.  The LogRefer
 * procedure is registered as a trace.  If a log file was opened, the
 * CloseReferLog procedure is registered for shutdown.
 */
void DllExport
LogReferInit(void)
{
    FILE           *logfp;
    char           *logfile;
    int             log;

    log = 0;
    logfp = NULL;


    /*
     * Open the a refer log if logging is to be done to a file.  If a log
     * file is opened, CloseReferLog is registered to close it at shutdown
     * time.
     */
    logfile = Ns_ConfigGet("LogRefer", "File");
    if (logfile != NULL) {
	Ns_DString      ds;

	Ns_DStringInit(&ds);

	if (!Ns_PathIsAbsolute(logfile)) {
	    Ns_DStringVarAppend(&ds, Ns_InfoHome(NULL), "/log/", logfile, NULL);
	    logfile = ds.string;
	}
	logfp = fopen(logfile, "a");
	if (logfp == NULL) {
	    Ns_Fatal("Could not open refer log file \"%s\".", logfile);
	}
	Ns_RegisterShutdown((Ns_Callback *) CloseReferLog, (Ns_OpContext) logfp);
	log = 1;
	Ns_DStringFree(&ds);
    }

    /*
     * Check if logging should be done in the database, creating the referlog
     * table if it doesn't already exist.
     */
    logindb = 0;
    Ns_ConfigGetInt("LogRefer", "Db", &logindb);
    if (logindb == 1) {
	if (Ns_DbGetTableInfo("referlog") == NULL) {
	    Ns_DbHandle    *handle;

	    handle = Ns_ServerDbHandle();

	    if (Ns_DbDML(handle, "create table referlog ("
		    "log_method text,"
		    "log_url text,"
		    "log_referer text,"
		    "log_time text,"
		    "log_status text);"
		    "insert into ns_tables (table_name, table_description)"
		    "values ('referlog','Log of referrals');") != NS_OK) {
		Ns_Fatal("Could not create table for refer logging");
	    }
	}
	log = 1;
    }
    if (log == 1) {
	Ns_RegisterTrace((Ns_TraceProc *) LogRefer, (Ns_OpContext) logfp);
	Ns_Log(Notice, "Refer logging initialized");
    }
}


/*
 * CloseReferLog - Close the refer log at shutdown time.
 */
static void
CloseReferLog(Ns_OpContext context)
{
    FILE           *logfp;

    logfp = (FILE *) context;
    fclose(logfp);
    Ns_Log(Notice, "Refer log closed");
}



/*
 * LogRefer - The trace proceudre to log referer headers to a file and/or
 * database table. The open log FILE * is passed as the trace context to
 * determine if file logging is required and the logindb file scope variable
 * is checked to determine if database logging is required.
 */
static void
LogRefer(Ns_OpContext context, Ns_Conn *conn)
{
    Ns_Set         *headers;
    char           *refer;
    Ns_DbHandle    *handle;
    FILE           *logfp;

    headers = conn->headers;
    if (headers == NULL) {
	return;
    }
    refer = Ns_SetGet(headers, "REFERER");
    if (refer == NULL) {
	return;
    }
    if (logindb == 1) {
	Ns_DString      sql;

	Ns_DStringInit(&sql);

	handle = Ns_ServerDbHandle();

	Ns_DStringVarAppend(&sql, "insert into referlog "
	    "(log_method, log_url, log_referer, log_time, log_status) "
	    "values ('",
	    conn->request->method, "','",
	    conn->request->url, "','",
	    refer, "','",
	    Ns_HttpTime(NULL), "','", NULL);
	Ns_DStringPrintf(&sql, "%d');", conn->returnStatus);
	if (Ns_DbDML(handle, sql.string) != NS_OK) {
	    Ns_Log(Error, "Could not log Referer in database");
	}
	Ns_DStringFree(&sql);
    }
    logfp = (FILE *) context;
    if (logfp != NULL) {
	fprintf(logfp, "%s\t%s\t%s\t%s\t%d\n",
	    conn->request->method,
	    conn->request->url,
	    refer,
	    Ns_HttpTime(NULL),
	    conn->returnStatus);
	fflush(logfp);
    }
}
