// File: GFMTutor.prg
// Notes: GrumpFish Menu Tutorial routines

#include "gfm.ch"
#include "gfmtutor.ch"
#include "fileio.ch"

#trans fPos( <n> ) => fSeek( <n>, 0, 1 ) // Current file position

STATIC aTopicIndex, cTopicFile := "GFMTUTOR.DOC"

///////////////////
///////////////////
//
//	Purpose:
//		Index topics for a tutorial file
//
//	Syntax:
//		TopicIndex( [ <cFile> ] ) -> aIndex
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		cFile     	Tutorial file. (Defaults to cTopicFile)
//
//	Returns:
//		An array containing the file offsets for every tutorial topic
//
//	Examples:
//		TopicIndex( "GFMTUTOR.DOC" )
//		TUTOR INDEX GFMTUTOR.DOC
//
//	Files:
//		
//
//	Description:
//		Creates an index of topics for a tutorial file.  The index contains
//		the title of the topic and its file offset.
//
//	Notes:
//		
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicFetch()
//
//	Include files:
//		gfm.ch
//		gfmtutor.ch
//		fileio.ch
//
//	Revisions:
//		01/17/92	08:49:18	1.0	Original version
//
///////////////////
///////////////////
FUNCTION TopicIndex( cFile )
	LOCAL cTopic, nTopicPos, nTopicStart := Len( TUTOR_TOPIC ) + 1
	LOCAL nHandle
	DEFAULT cFile TO cTopicFile								// Default file name
	aTopicIndex := {}												// Clear topic index
	cTopicFile := cFile											// Save the file name

	IF ( nHandle := fOpen( cTopicFile, FO_READ + FO_SHARED ) ) > -1
		@ MaxRow(), 0 SAY "Creating Topic Index for " + cTopicFile
		DO WHILE fLocate( nHandle, TUTOR_TOPIC, 'U' ) > -1	// Load index
			nTopicPos	:= fPos( nHandle )
			cTopic 		:= SubStr( fReadLine( nHandle ), nTopicStart )
			aAdd( aTopicIndex, { cTopic, nTopicPos } )
		ENDDO
		@ MaxRow(), 0
		fClose( nHandle )
	ENDIF

RETURN aTopicIndex

//////////////////
//////////////////
//
//	Purpose:
//		Current tutorial file name
//
//	Syntax:
//		TutorFile() -> cFileName
//
//	No arguments specified
//
//	Returns:
//		The name of the current tutorial file
//
//	Examples:
//		? "You are learning from ", TutorFile()
//
//	Files:
//		
//
//	Description:
//		Provides access to the name of the current tutorial file
//
//	Notes:
//		
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicIndex()
//
//	Revisions:
//		01/17/92	08:55:35	1.0	Original version
//
//////////////////
//////////////////
FUNCTION TutorFile
RETURN cTopicFile

///////////////////////
///////////////////////
//
//	Purpose:
//		Locates a topic in the index
//
//	Syntax:
//		TopicAt( <cTopic> ) -> nPosition
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		cTopic    	Topic to locate
//
//	Returns:
//		The position of the topic in the index, or 0 if not found
//
//	Examples:
//		IF TopicAt( "What's New" )
//			TUTOR "What's New"
//		ENDIF
//
//	Files:
//		
//
//	Description:
//		Scans the topic index for the title of a topic.  If it is found, it
//		returns its position.  Otherwise, it returns 0.
//
//	Notes:
//		
//		FUNCTION TopicAt
//		is only visible in its own source file
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicFetch() TopicShow()
//
//	Include files:
//		gfm.ch
//		gfmtutor.ch
//		fileio.ch
//
//	Revisions:
//		01/17/92	08:57:03	1.0	Original version
//
///////////////////////
///////////////////////
STATIC FUNCTION TopicAt( cTopic )
	cTopic := Upper( cTopic )
RETURN aScan( aTopicIndex, { |a| Upper( TopicTitle( a ) ) == cTopic } )

//////////////////////////
//////////////////////////
//
//	Purpose:
//		Retrieve a topic's text
//
//	Syntax:
//		TopicFetch( <cTopic> ) -> cText
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		cTopic    	Topic to fetch text for
//
//	Returns:
//		The text associated with a tutorial topic
//
//	Examples:
//
//
//	Files:
//		
//
//	Description:
//		Reads the text for a specified topic from the tutorial file and
//		returns it.
//
//	Notes:
//		
//		FUNCTION TopicFetch
//		is only visible in its own source file
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicShow() TopicAt()
//
//	Include files:
//		gfm.ch
//		gfmtutor.ch
//		fileio.ch
//
//	Revisions:
//		01/17/92	08:58:51	1.0	Original version
//
//////////////////////////
//////////////////////////
STATIC FUNCTION TopicFetch( cTopic )
	LOCAL nTopic := TopicAt( cTopic )
	LOCAL cTopicText, nTopicStart, nTopicStop
	IF Full( nTopic )
		nTopicStart := TopicOffset( aTopicIndex[ nTopic ] ) + Len( TUTOR_TOPIC )
		IF nTopic < Len( aTopicIndex )
			nTopicStop := TopicOffset( aTopicIndex[ nTopic + 1 ] ) - 3
		ENDIF
		cTopicText := FileRead( cTopicFile, nTopicStart, nTopicStop )
	ENDIF
RETURN cTopicText

//////////////////
//////////////////
//
//	Purpose:
//		Display a tutorial topic
//
//	Syntax:
//		TopicShow( <cTopic>, [ <nTop> ], [ <nLeft> ], [ <nBottom> ], 
//						[ <nRight> ] ) -> lShown
//
//	Formal Arguments: (5)
//		Name       	Description
//			
//		cTopic    	Topic to display
//		nTop      	Top row of display window. (Defaults to scaled top row)
//		nLeft     	Left column of display window. (Defaults to 0)
//		nBottom   	Bottom row of display window. 
//						(Defaults to scaled bottom row)
//		nRight    	Right column of display window. (Defaults to MaxCol())
//
//	Returns:
//		.T. if a topic was found and shown, .F. otherwise
//
//	Examples:
//		TopicShow( "What's New" )
//
//		TUTOR ON "What's New"
//
//	Files:
//		
//
//	Description:
//		Loads and displays the text for a topic provided the topic is found.
//
//	Notes:
//		
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicFetch() TopicAt()
//
//	Include files:
//		gfm.ch
//		gfmtutor.ch
//		fileio.ch
//
//	Revisions:
//		01/17/92	09:00:57	1.0	Original version
//
//////////////////
//////////////////
FUNCTION TopicShow( cTopic, nTop, nLeft, nBottom, nRight )
	LOCAL lShown := FALSE, nLines
	LOCAL cTopicText := TopicFetch( cTopic )

	IF Full( cTopicText )
		nLines := ChrCount( Chr(13), cTopicText )
		DEFAULT nTop TO Max( 1, MaxRow()/2 - nLines/2 - 1 ), ;
					nLeft TO 0, ;
					nBottom TO Min( MaxRow() - 1, nTop + nLines + 1 ), ;
					nRight TO MaxCol()
		
		w_Shade( nTop, nLeft, nBottom, nRight, Frame(1), ;
					"GrumpFish Menu Tutorial", "Esc=exit" )
		MemoEdit( cTopicText, ++nTop, ++nLeft, --nBottom, --nRight, FALSE,, ;
					 TUTOR_MARGIN, TUTOR_TABSIZE )
		w_Pop()
		lShown := TRUE
	ELSE
		w_Error( "", 0, cTopic, "Tutorial Topic not found!", "Press a key" )
	ENDIF
RETURN lShown

///////////////////
///////////////////
//
//	Purpose:
//		Menu of tutor topics
//
//	Syntax:
//		TopicPick()
//
//	No arguments specified
//
//	Examples:
//		TopicPick()
//
//	Files:
//		
//
//	Description:
//		Provides a pick list of the topics contained in the current tutorial
//		file.  If the user selects a topic, it is then displayed.
//
//	Notes:
//		
//
//	Category:
//		Tutorial
//
//	See Also:
//		TopicShow()
//
//	Revisions:
//		01/17/92	09:09:34	1.0	Original version
//
///////////////////
///////////////////
PROCEDURE TopicPick
	LOCAL nTopic := PopChoice( aColumn( aTopicIndex, 1 ), "Tutorialial Topic" )
	IF Full( nTopic )
		TUTOR ON TopicTitle( aTopicIndex[ nTopic ] )
	ENDIF
RETURN
