#include "gfm.ch"
#include "inkey.ch"
#include "setcurs.ch"

**************
**************
**
**	Purpose:	Display the current screen of prompts
**
**	Syntax:	_d123_( <nR1>, <aPrompts>, <aCols>, <i> )
**
**	Parameters:
**
**			Name		Description
**				
**			nR1		Row to display on
**			nC1		Column to begin display on
**			nC2		Column to end display on
**			aPrompts	Array of prompts
**			aCols		Array FOR each column
**			i			Current element
**
**		Notes:	This FUNCTION will RETURN the rightmost value of the current
**					window of prompts that <i> is contained in. <i> is the leftmost
**					value FOR the current window - the only time this FUNCTION is
**					called is when the menu windows change.
**					Furthermore, it will display all of the prompts including
**					that of the rightmost value's.
**					Internal to m_123().
**
**	Returns:	the value furthest to the right in this menu screen
**
**	Example:	_d123_( m->row, prompts, aCols, m->i )
**
** Category:  Menu Handling
**
**		Files:	None.
**
**************
**************
STATIC FUNCTION _d123_( nR1, nC1, nC2, aPrompts, aCols, nPos, alToggles, ;
								bSecure )
	LOCAL nPrompts := Len( aPrompts ), nCol, nRun, nCols := nC2 - nC1 + 1
	nCol := aCols[ nPos ]
	@ nR1, nC1 clear to nR1, nC2

	FOR nRun := nPos to nPrompts
		IF ! Eval( bSecure, alToggles, nRun )
			SetUns()
		ENDIF
		@ nR1, aCols[ nRun ] say SubStr( trim( aPrompts[ nRun ] ), 1, nCols )
		IF ! Eval( bSecure, alToggles, nRun )
			SetStd()
		ENDIF
		IF nRun < nPrompts	// Test FOR the NEXT screen
			IF aCols[ nRun + 1 ] <= aCols[ nRun ]
				EXIT
			ENDIF
		ENDIF
	NEXT

	IF nRun < nPrompts
		@ nR1, nC2 say '>'
	ELSEIF nPos > 1
		@ nR1, nC1 say '<'
	ENDIF
RETURN nRun

***************
***************
**
**	Purpose:	Find the menu screen that a prompt is contained in
**
**	Syntax:	_f123_( <aCols>, <nLeft>, <nC1> )
**
**	Params:
**
**			Name			Description
**				
**			aCols		Column array
**			nLeft		Leftmost value
**			nC1			Start column
**
**		Notes:	This FUNCTION will RETURN the leftmost value of the current
**						window of prompts that <nLeft> is contained in.	Internal to
**						m_123().
**
**	Returns:	the leftmost value in that window
**
**	Example:	_f123_( aCols, nLeft, _mnC1 )
**
**
** Category:  Menu Handling
**
**		Files:	None.
**
***************
***************
STATIC FUNCTION _f123_( aCols, nLeft, nC1 )
	nC1++
	DO WHILE aCols[ nLeft ] # nC1 .and. nLeft > 1
		nLeft--
	ENDDO
RETURN nLeft


***************
***************
**
**	Purpose:	Pops-up a help screen for m_123() menus
**
**	Syntax:	_h123_( <keys> )
**
**	Params:
**
**			Name		T Description
**				 
**			keys		C A string of the valid hot keys
**
**		Notes:	activates when [F1] is pressed, then waits FOR another key
**						press.	Internal to m_123().
**						This FUNCTION will display the valid keys FOR the current
**						menu prompts.
**
**	Returns:	.t.
**
**	Example:	_h123_( "abcde" )
**
** Category:  Menu Handling
**
**		Files:	None.
**
***************
***************
STATIC FUNCTION _h123_( cKeys )
	w_error( setcolor(), 0, ;
	"Key            Action																										~" + ;
	"	 ~" + ;
	"[Home]         First item in menu~" + ;
	"[Left Arrow]	 Previous item in menu (or WRAP to last when WRAP is on)~" + ;
	"[Right Arrow]	 Next item in menu (or WRAP to first when WRAP is on)~" + ;
	"[End]          Last item in menu~" + ;
	"[^Left Arrow]	 Previous menu screen~" + ;
	"[^Right Arrow] Next menu screen~~" + ;
	"Also, any of the keys listed below will position and optionally execute the~" + ;
	"option beginning with that letter:~'"+cKeys+'"', ;
	"Valid keys FOR this 1-2-3 style menu:" , ;
	"Press a key to return" )
RETURN .t.


**************
**************
**
**	Purpose:	123-style menus from an achoice() - like call
**
**	Syntax:	m_123( <aPrompts> [, <aMessages> [, <nR1> [, <nC1> [, <nC2>
**						[, <nOption> [, <lExecute> [, <nSpacing> ] ] ] ] ] ] ] )
**
**	Parameters:
**
**			Name		Description
**				
**			aPrompts	Array of prompts
**			aMessages Array of messages corresponding to prompts.	Defaults
**						to none being displayed.
**			nR1		Row to display menu on.	Defaults to 0.
**			nC1		Column to start menu on.	Defaults to 0.
**			nC2		Last column to put menu in.	Defaults to maxcol().
**			nOption	Default starting position in menu.	Defaults to 1.
**			lExecute	Toggle FOR just highlighting menu option when pressed,
**						or executing.	Defaults to .t.
**			nSpacing	Spacing between prompts.	Defaults to 1.
**
**		Notes:	Emulates a Lotus 1-2-3 style menu, only better.
**
**	Returns:	the selected array element
**
**	Example:	m_123( choices, messages )
**
** Category:  Menu Handling
**
** See Also:	chrcount() setstatus() maxcol()
**
**		Files:	None.
**
**************
**************
FUNCTION m_123( aPrompts, aMessages, nR1, nC1, nC2, nOption, lExecute, ;
					nSpacing, alToggles )
	LOCAL nPrompts, nTmp, lShowMess, nWidth, nRight, aCols := {}
	LOCAL nKey := 0, nForce := -1
	LOCAL nResult	:= 0
	LOCAL lRefresh	:= .t.
	LOCAL lWrap		:= Set( _SET_WRAP )		// Get Wrap setting
	LOCAL nCursor	:= SetCursor()				// Save cursor settings
	LOCAL nMessRow := Set( _SET_MESSAGE )	// Get message line setting
	LOCAL cHotKeys	:= ''
	LOCAL nPos		:= 0
	LOCAL nLeft		:= 1
	LOCAL bSecure, bMessage

	IF GFMForceNext() > -1
		nForce := GFMForce()
		IF nForce == 0
			RETURN nForce
		ELSE
			nOption := nForce
			KeyStuff( Chr( K_ENTER ) )
		ENDIF
	ENDIF

	IF aPrompts == NIL .or. ValType( aPrompts ) # 'A' .or. Len( aPrompts ) == 0
		RETURN 0
	ENDIF

	nPrompts	:= Len( aPrompts )
	lShowMess := ( Full( aMessages ) .AND. ValType( aMessages ) == 'A' )
	nWidth	:= nC1 + 1

	DEFAULT nSpacing TO 1, ;
				lExecute TO .T., ;
				nR1 TO 0, ;
				nC1 TO 0, ;
				nC2 TO MaxCol(), ;
				nOption TO 1, ;
				alToggles TO .T.

	IF GFMSet( _GFM_STYLE ) $ 'S1B'
		bMessage := {|t,r,c|SetPos(r,c),DispOut( PadR( t, nC2 - nC1 ) ) }
	ELSE
		bMessage := {|t,r,c|SetPos(r,0),DispOut( PadC( t, MaxCol() + 1 ) ) }
	ENDIF

	IF ValType( alToggles ) == 'A'	// Enabling security levels
		bSecure := { | al, n | al[ n ] }
		IF aScan( alToggles, { | a | a  } ) == 0	// No options enabled
			RETURN 0
		ENDIF
	ELSE
		bSecure := {||.t.}
	ENDIF

	IF nOption < 1
		nOption := 1
	ELSEIF nOption > nPrompts
		nOption := nPrompts
	ENDIF

	FOR nPos := 1 to nPrompts
		cHotKeys	+= m_hotkey( aPrompts[ nPos ] )
		nTmp		:= len( trim( aPrompts[ nPos ] ) )
		IF nTmp + nWidth > nC2 - 1 THEN nWidth := nC1 + 1
		aAdd( aCols, nWidth )
		nWidth += nTmp + nSpacing
	NEXT nPos
	nPos	:= nOption
	nLeft	:= _f123_( aCols, nPos, nC1 )

	SetCursor( SC_NONE )

	DO WHILE nKey # K_ENTER .AND. nKey # K_ESC

		IF lRefresh
			nRight	:= _d123_( nR1, nC1, nC2, aPrompts, aCols, nLeft, ;
									  alToggles, bSecure  )
			lRefresh := .f.
		ENDIF

		IF Eval( bSecure, alToggles, nPos )

			setenh()
			@ nR1, aCols[ nPos ] SAY Trim( aPrompts[ nPos ] )
			setstd()
			SetPos( nR1, aCols[ nPos ] )

			IF lShowMess	// Center msg when appropriate!
				nTmp := aMessages[ nPos ]
				IF Type( aMessages[ nPos ] ) == 'C'
					nTmp := &( aMessages[ nPos ] )
				ELSE
					nTmp := aMessages[ nPos ]
				ENDIF
				Eval( bMessage, nTmp, nMessRow, nC1 )
			ENDIF

			IF nForce == -1
				nKey := GFMGetKey( 0 )
				IF GFMForceNext() > -1
					nForce := nResult := nPos := GFMForce()
					lRefresh := .T.
					EXIT
				ENDIF
			ELSE
				nKey := K_ENTER
			ENDIF
			@ nR1, aCols[ nPos ] say Trim( aPrompts[ nPos ] )

		ELSEIF Empty( nKey )

			IF ( nPos += GFMSet( _GFM_DIRECTION ) ) > nPrompts
				nPos := 1
			ELSEIF nPos < 1
				nPos := nPrompts
			ENDIF

		ENDIF

		SetPos( nR1, aCols[ nPos ] )

		IF GFMSet( _GFM_ACTIVE ) .AND. Full( nKey )
			// Check for GrumpFish Menu Stuff
			IF nKey == GFMSet( _GFM_CFGKEY ) .and. GFMSet( _GFM_LEVEL ) == 0
				GFMSet( _GFM_CFGHIT, .t. )
				EXIT
			ENDIF
			IF nKey == GFMSet( _GFM_UTILKEY ) .AND. nKey # 0
				GFMUtils( ProcName(), ProcLine() )
				nKey := 0
			ENDIF
			IF nKey == GFMSet( _GFM_NOTEKEY ) .AND. nKey # 0
				GFMNotes( ProcName(), ProcLine() )
				nKey := 0
			ENDIF
		ENDIF

		DO CASE
		CASE nKey == NIL
			nKey := NIL
		CASE GFMSet( _GFM_SIZABLE ) .AND. KeyTest('S') 	// Resize window
			GFMSet( _GFM_RESIZE, .T. )
			EXIT
		CASE nKey == K_RIGHT
			IF nPos < nPrompts
				IF ( ++nPos > nRight )
					nLeft := nPos
					lRefresh := .t.
				ENDIF
			ELSEIF lWrap
				nPos	:= nLeft := 1
				lRefresh := .t.
			ENDIF
		CASE nKey == K_CTRL_RIGHT
			IF nRight < nPrompts
				nPos := nLeft := nRight + 1
				lRefresh := .t.
			ELSEIF lWrap
				nPos	:= nLeft := 1
				lRefresh := .t.
			ELSE
				nPos := nLeft
			ENDIF

		CASE nKey == K_LEFT
			IF nPos > 1
				IF --nPos < nLeft
					nLeft := _f123_( aCols, nPos, nC1 )
					lRefresh := .t.
				ENDIF
			ELSEIF lWrap
				nPos	:= nPrompts
				nLeft := _f123_( aCols, nPos, nC1 )
				lRefresh := .t.
			ENDIF
		CASE nKey == K_CTRL_LEFT
			IF nLeft > 1
				nPos	:= nLeft - 1
				nLeft := _f123_( aCols, nPos, nC1 )
				lRefresh := .t.
				nPos := nLeft
			ELSEIF lWrap
				nPos	:= nPrompts
				nLeft := _f123_( aCols, nPos, nC1 )
				lRefresh := .t.
				nPos	:= nLeft
			ELSE
				nPos := 1
			ENDIF
		CASE nKey == K_HOME
			nPos	:= nLeft := 1
			lRefresh := .t.
		CASE nKey == K_END
			nPos	:= nPrompts
			nLeft := _f123_( aCols, nPos, nC1 )
			lRefresh := .t.
		CASE nKey == K_ENTER ; nResult := nPos
		CASE nKey == K_ESC ; nResult := 0
		CASE nKey > 31 // Check for a key press

			IF ( nResult := fat( nTmp := Upper( chr( nKey ) ), cHotKeys, nPos ) ) == 0
				nResult := at( nTmp, cHotKeys )	// current key not found at end
			ENDIF
			IF nResult # 0
				nPos := nResult
				IF ( lExecute .and. ChrCount( nTmp, cHotKeys ) == 1 )
					EXIT
				ENDIF
				IF ! ( nLeft <= nPos .and. nPos <= nRight )
					nLeft := _f123_( aCols, nPos, nC1 )
					lRefresh := .t.
				ENDIF
			ENDIF
		ENDCASE

	ENDDO

	IF nForce > -1 THEN nResult := nPos := nForce

	IF nResult > 0
		IF ! ( nLeft <= nResult .and. nResult <= nRight )	// Redisplay?
			nLeft := _f123_( aCols, nResult, nC1 )
			_d123_( nR1, nC1, nC2, aPrompts, aCols, nLeft, alToggles, bSecure )
		ENDIF
		setenh()
		@ nR1, aCols[ nResult ] SAY Trim( aPrompts[ nResult ] )
		setstd()
		SetPos( nR1, aCols[ nResult ] )
	ENDIF
	SetCursor( nCursor ) // Restore the cursor settings

RETURN nResult

FUNCTION m_hotkey( cText )
	LOCAL cChar := GetUpChr( cText, 1 )
	IF Len( cChar ) == 0
		RETURN Upper( SubStr( LTrim( cText ), 1, 1 ) )
	ENDIF
RETURN cChar

// End of 123 style menus
#define POPFRAME  frame(3)

//////////////////
//////////////////
//
//  Purpose:
//    Provide a pop-up version of aChoice()
//
//  Syntax:
//    popchoice( <aChoices>, [ <cTitle> ], [ <cFooter> ], [ <nTop> ], [
//               <nLeft> ], <nBottom>, [ <nRight> ], [ <aToggles> ], [ <cUdf> ],
//               [ <nCurrent> ], [ <nRelRow> ] )
//
//  Formal Arguments: (11)
//    Name        Description
//     
//    aChoices    Array of choices
//    cTitle      Title of box. (Defaults to "Choices")
//    cFooter     Footer for box. (Defaults to "  Esc")
//    nTop        Top row of box. (Defaults to -1)
//    nLeft       Left column of box. (Defaults to -1)
//    nBottom     Bottom row of box
//    nRight      Right column of box. (Defaults to -1)
//    aToggles    Array of logicals for making options selectable, or .T.
//                (Defaults to .t.)
//    cUdf        UDF to call from achoice().  (Defaults to no UDF)
//    nCurrent    Default element to highlight.  (Defaults to 1)
//    nRelRow     Row in window to position item at.  (Defaults to 1)
//
//  Returns:
//    The choice number picked, or 0 if no selection was made
//
//  Examples:
//    if ( nChoice := popchoice( { 'Screen', 'Printer', 'File' }, ;
//        'Output to' ) ) # 0
//      if nChoice == 3
//        cFileName := ask4string( "Output file", 64 )
//      endif
//      .
//      .
//    endif
//
//  Files:
//    None.
//
//  Description:
//    Windowed routine for aChoice(), to provide quick access to a pick list
//    for a one-dimensional array, without having to worry about drawing
//    boxes, saving the screen, etc.
//
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    PickFile() PickField() PickDBF() PickNTX()
//
//  Include files:
//    jfk.ch
//
//  Revisions:
//    11/19/90  14:23:35  1.0 Original version
//
//////////////////
//////////////////
FUNCTION PopChoice( aChoices, cTitle, cFooter, nTop, nLeft, nBottom, nRight, aToggles, ;
                    cUdf, nCurrent, nRelRow )
   LOCAL nMaxWid, nReturn
   IF Empty( aChoices )
     RETURN 0
   ENDIF

   DEFAULT cTitle to "Choices", cFooter to "  Esc", nTop to -1, ;
         nLeft TO -1, nBottom TO -1, nRight TO -1, aToggles TO .t.

   IF nCurrent == NIL .or. nCurrent < 1
     nCurrent := 1
   ENDIF
   IF nRelRow == NIL .or. nRelRow < 0
     nRelRow := 0
   ENDIF

   IF nTop == -1 .or. nLeft == -1 .or. nBottom == -1 .or. nRight == -1
     // Re-sizing window
     w_scale( len( aChoices ), w_TextDim( aMaxStrLen( aChoices ), ;
             cTitle, cFooter ), @nTop, @nLeft, @nBottom, @nRight )
   ENDIF

   w_shade( nTop, nLeft, nBottom, nRight, POPFRAME, cTitle, cFooter )
   nReturn := aChoice( nTop + 1, nLeft + 1, nBottom - 1, nRight - 1, aChoices, ;
               aToggles,, nCurrent, nRelRow )
   w_Pop()
RETURN nReturn

#include "achoice.ch"
#include "set.ch"

******************
******************
**
**  Purpose:  Achoice()-like function that allows for multiple options
**
**   Syntax:  gchoice( <aPrompts>, <aMessages>, <nR1>, <nC1>, <nR2>, <nC2>
**              [, <nOption> [, <aToggles> [, <lTagging> ] ] ] )
**
**   Params:
**      Name      Description
**        
**      aPrompts  array of prompts
**      aMessages array of messages corresponding to prompts
**      nR1       Top row of box
**      nC1       Left column of box
**      nR2       Bottom row of box
**      nC2       Right column of box
**      nOption   # of current option
**      aToggles  Logical array for which menu options are enabled, or .T.
**      lTagging  Can multiple items be selected?
**
**    Notes:  does several things that the achoice() udf assumes:
**
**            o builds HotKeys() list (first-character keys to execute on).
**              This list is case insensitive
**            o sets Tagging()
**
**  Returns:  the Nth value chosen, or 0 (zero) if [Esc] is pressed
**
**  Example:  gchoice( prompts, messages, 4, 4, 10, 50, 1, .t., .t. )
**
** Category:  Menu Handling
**
******************
******************
FUNCTION GChoice
   MEMVAR lShowMess, cHotKeys, nMark, nForce
   LOCAL nPrompt, nPrompts, nChoice
   PARAMETERS aPrompts, aMessages, nR1, nC1, nR2, nC2, nOption, aToggles, lTagging
   PRIVATE lShowMess, cHotKeys, nForce := -1

	IF GFMForceNext() > -1
		nForce := GFMForce()
		IF nForce == 0
			RETURN nForce
		ELSE
			MEMVAR->nOption := nForce
			KeyStuff( Chr( K_ENTER ) )
		ENDIF
	ENDIF

   IF MEMVAR->aPrompts == NIL .or. MEMVAR->nR1 == NIL .or. MEMVAR->nC1 == NIL .or. ;
      MEMVAR->nR2 == NIL .or. MEMVAR->nC2 == NIL
     RETURN 0
   ENDIF

   IF MEMVAR->nOption == NIL .or. MEMVAR->nOption < 1
     MEMVAR->nOption := 1
   ELSEIF MEMVAR->nOption > len( MEMVAR->aPrompts )
     MEMVAR->nOption := len( MEMVAR->aPrompts )
   ENDIF

   DEFAULT MEMVAR->lTagging TO .F.

   IF MEMVAR->nR2 > maxrow() - 2  // rescale lower boundary
     MEMVAR->nR2 := maxrow() - 2
   ENDIF

   IF ( MEMVAR->lShowMess := ( MEMVAR->aMessages # NIL .and. ;
       valtype( MEMVAR->aMessages ) == 'A' ) )
     ShowGFMMess( MEMVAR->lShowMess, MEMVAR->aMessages, MEMVAR->nOption )
   ENDIF

   MEMVAR->cHotKeys := ""
   aEval( MEMVAR->aPrompts, { | cPrompt | MEMVAR->cHotKeys += ;
          upper( substr( cPrompt, 1, 1 ) ) } )

   MEMVAR->nMark := len( MEMVAR->aPrompts[ 1 ] )

	IF nForce # -1 THEN KeyStuff( Chr( K_ENTER ) ) // Go into exception
   DO WHILE .T.  // Loop in case tagging is being performed

     nChoice := aChoice( MEMVAR->nR1, MEMVAR->nC1, MEMVAR->nR2, MEMVAR->nC2, ;
                         MEMVAR->aPrompts, MEMVAR->aToggles, "_gmenu_", ;
                         MEMVAR->nOption, MEMVAR->nOption )
     IF nChoice == 0 .OR. LastKey() == 13
       EXIT
     ELSE
       MEMVAR->nOption := nChoice
     ENDIF

   ENDDO

	IF GFMForceNext() > -1 .AND. nForce == -1 THEN nChoice := GFMForce()
	IF nChoice # 0 THEN gfattr( Row(), MEMVAR->nC1, Row(), MEMVAR->nC2, ;
										attr2val( strextract( SetColor(), ',', 2 ) ) )
   ShowGFMMess( MEMVAR->lShowMess, MEMVAR->aMessages, nChoice )

RETURN nChoice


****************
****************
**
**  Purpose:  UDF for gchoice(), called from Achoice()
**
**   Syntax:  _gmenu_( <nStatus>, <nElement>, <nRow> )
**
**   Parameters:
**
**      Name      Description
**        
**      nStatus   status of achoice()
**      nElement  current element that is high-lighted
**      nRow      current row offset
**
**    Notes:  o assumes cHotKeys is set correctly.
**            o assumes nMark and lTagging are set correctly
**            o assumes that lShowMess is set correctly
**            o assumes that an array called "aMessages" is defined if
**              lShowMess is true.
**
**  Returns:  an action to achoice()
**
**  Example:
**
** Category:  Menu Handling
**
**    Files:  None.
**
****************
****************
FUNCTION _gmenu_( nStatus, nElement, nRow )
   LOCAL lWrap := Set( _SET_WRAP ), nKey, nReturn := AC_CONT, cTemp, nPos

   DO CASE
	CASE GFMSet( _GFM_SIZABLE ) .AND. KeyTest('S') 	// Resize window
		GFMSet( _GFM_RESIZE, .T. )
		nReturn := AC_ABORT
   CASE nStatus == AC_IDLE; ShowGFMMess( MEMVAR->lShowMess, MEMVAR->aMessages, nElement )

   CASE nStatus == AC_HITTOP
      IF lWrap
         KeyStuff( chr( 30 ) )   // ^PgDn to go to last record
      ENDIF

   CASE nStatus == AC_HITBOTTOM  // went past bottom
      IF lWrap
         KeyStuff( chr( 31 ) )   // ^PgUp to go to first record
      ENDIF

   CASE nStatus == AC_EXCEPT  // keystroke exception

		DO CASE
      CASE ( cTemp := Upper( Chr( nKey := GFMGetKey(,LastKey()) ) ) ) $ MEMVAR->cHotKeys
        nReturn := AC_GOTO	// move to next occurence of the key in cHotKeys
        IF ChrCount( cTemp, MEMVAR->cHotKeys ) == 1
          KeyStuff( chr( 13 ) )	// only one occurence - execute it!
        ENDIF
		CASE nKey == GFMSet( _GFM_NOTEKEY ) .AND. nKey # 0
			GFMNotes( ProcName(4), ProcLine(4) )
      CASE nKey == K_ESC .OR. GFMForceNext() > -1 .AND. MEMVAR->nForce == -1
        nReturn := AC_ABORT  // abort selection
      CASE nKey == K_ENTER ; nReturn := AC_SELECT  // make selection
      CASE nKey == K_CTRL_T .or. nKey == K_CTRL_U .and. MEMVAR->lTagging  // ^T or ^U pressed
        cTemp := IIF( nKey == K_CTRL_T, '', ' ' )
        aEval( MEMVAR->aPrompts, ;
               { | c, n | MEMVAR->aPrompts[ n ] := Stuff( c, MEMVAR->nMark, 1, cTemp ) } )
        nReturn := AC_SELECT
      CASE nKey == 32 .AND. MEMVAR->lTagging
        MEMVAR->aPrompts[ nElement ] := Stuff( MEMVAR->aPrompts[ nElement ], ;
                                        MEMVAR->nMark, 1, if( '' $ ;
                                        MEMVAR->aPrompts[ nElement ], ' ', '' ) )
        nReturn := AC_SELECT
      ENDCASE

   CASE nStatus == AC_NOITEM  // no item selectable

      @ Set( _SET_MESSAGE ), 0
      @ Set( _SET_MESSAGE ), 0 say "No prompt is active!  Press a key."
      DO WHILE NextKey() == 0; ENDDO
      nReturn := AC_ABORT

   ENDCASE

RETURN nReturn

STATIC PROCEDURE ShowGFMMess( lShowMess, aMessages, nElement )
   LOCAL cTemp
   IF lShowMess == NIL .or. ! lShowMess
		RETURN
   ENDIF

   IF lShowMess .and. nElement # 0 .and. nElement <= len( aMessages )
		// display current message - macroed for evaluation purposes
		IF type( aMessages[ nElement ] ) == 'C'
			cTemp := &( aMessages[ nElement ] )
		ELSE
			cTemp := aMessages[ nElement ]
		ENDIF
		@ Set( _SET_MESSAGE ), 0 SAY PadC( cTemp, MaxCol() + 1 )
   ENDIF
RETURN
